/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/
#include <math.h>
#include <string.h>
#include <cpl.h>

#include "moo_resp.h"
#include "moo_utils.h"
#include "moo_pfits.h"
#include "moo_scilist.h"
#include "moo_sci.h"
/*----------------------------------------------------------------------------*/
/**
 * @defgroup moo_resp RESP format
 * @ingroup moo_data
 * This module provides functions to create, use, and destroy a @em moo_resp
 *
 * Functionality include:
 *
 * @par Synopsis:
 * @code
 *   #include "moo_resp.h"
 * @endcode
 */
/*----------------------------------------------------------------------------*/
/**@{*/

/*-----------------------------------------------------------------------------
                              Function codes
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Create a new moo_resp
  @return   1 newly allocated moo_resp or NULL in case of an error

  The returned object must be deallocated using moo_resp_delete().

 */
/*----------------------------------------------------------------------------*/
moo_resp *
moo_resp_new(void)
{
    moo_resp *res = cpl_calloc(1, sizeof(moo_resp));
    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief assign table in moo_resp
  @param self moo_resp
  @param type the detcor type
  @param t the table

Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/

cpl_error_code
moo_resp_set_table(moo_resp *self, moo_detector_type type, cpl_table *t)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);

    cpl_error_code status = CPL_ERROR_NONE;
    cpl_errorstate prev_state = cpl_errorstate_get();

    if (t != NULL) {
        switch (type) {
            case MOO_TYPE_RI:
                self->ri = t;
                break;
            case MOO_TYPE_YJ:
                self->yj = t;
                break;
            case MOO_TYPE_H:
                self->h = t;
                break;
        }
    }
    if (!cpl_errorstate_is_equal(prev_state)) {
        status = cpl_error_get_code();
        cpl_errorstate_set(prev_state);
    }
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Get a reponse table from RESP
  @param self moo_resp structure
  @param type type of detector
  @return   cpl table

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_table *
moo_resp_get_table(moo_resp *self, moo_detector_type type)
{
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);

    switch (type) {
        case MOO_TYPE_RI:
            return self->ri;
            break;
        case MOO_TYPE_YJ:
            return self->yj;
            break;
        case MOO_TYPE_H:
            return self->h;
            break;
        default:
            return NULL;
    }
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Prepare RESP for SCI frame
  @param    scilist the SCI list from where we computed the rsponse
  @param    badpix_level level of bad pixel
  @return   create table for response

 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 *

 */
/*----------------------------------------------------------------------------*/
moo_resp *
moo_resp_create(moo_scilist *scilist, int badpix_level)
{
    moo_resp *result = NULL;

    cpl_ensure(scilist != NULL, CPL_ERROR_NULL_INPUT, NULL);

    int size = moo_scilist_get_size(scilist);

    result = moo_resp_new();
    result->primary_header = cpl_propertylist_new();
    result->response_table = cpl_table_new(size);

    cpl_table_new_column(result->response_table, MOO_RESP_TABLE_INDEXRBN,
                         CPL_TYPE_INT);
    cpl_table_new_column(result->response_table, MOO_RESP_TABLE_SPECTRO,
                         CPL_TYPE_INT);
    cpl_table_new_column(result->response_table, MOO_RESP_TABLE_RESPONSE,
                         CPL_TYPE_STRING);

    for (int il = 0; il < size; il++) {
        cpl_array *sel = NULL;
        char *response_colname = NULL;
        moo_sci *sci = moo_scilist_get(scilist, il);
        moo_target_table *target_table = moo_sci_get_target_table(sci);
        if (target_table->table != NULL) {
            const char *targname = moo_target_table_get_std_name(target_table);
            int idtarget =
                moo_target_table_find_target(sci->target_table, targname);
            cpl_table_select_all(sci->target_table->table);
            cpl_table_and_selected_int(sci->target_table->table,
                                       MOO_TARGET_TABLE_INDEXTARG, CPL_EQUAL_TO,
                                       idtarget);
            sel = cpl_table_where_selected(sci->target_table->table);
            int idx = cpl_array_get_cplsize(sel, 0, NULL);
            int idrbn = cpl_table_get_int(sci->target_table->table,
                                          MOO_TARGET_TABLE_INDEXRBN, idx, NULL);
            const int spectro =
                cpl_table_get_int(sci->target_table->table,
                                  MOO_TARGET_TABLE_SPECTRO, idx, NULL);
            cpl_table_set_int(result->response_table, MOO_RESP_TABLE_INDEXRBN,
                              il, idrbn);
            response_colname = cpl_sprintf("%s_%d", MOO_RESP_RESPONSE, il);
            cpl_table_set_int(result->response_table, MOO_RESP_TABLE_SPECTRO,
                              il, spectro);
            cpl_table_set_string(result->response_table,
                                 MOO_RESP_TABLE_RESPONSE, il, response_colname);
            cpl_array_delete(sel);
            cpl_free(response_colname);
        }
    }

    for (int i = 0; i < 3; i++) {
        cpl_table *table = NULL;

        for (int il = 0; il < size; il++) {
            const char *response_colname =
                cpl_table_get_string(result->response_table,
                                     MOO_RESP_TABLE_RESPONSE, il);
            moo_sci *sci = moo_scilist_get(scilist, il);
            moo_target_table *target_table = moo_sci_get_target_table(sci);
            if (target_table->table != NULL) {
                const char *targname =
                    moo_target_table_get_std_name(target_table);
                int idtarget =
                    moo_target_table_find_target(sci->target_table, targname);

                moo_sci_single *sci_single =
                    moo_sci_load_single(sci, i, badpix_level);

                if (sci_single != NULL) {
                    cpl_propertylist *header =
                        moo_sci_single_get_header(sci_single);

                    double crpix1 = moo_pfits_get_crpix1(header);
                    double crval1 = moo_pfits_get_crval1(header);
                    double cd1_1 = moo_pfits_get_cd1_1(header);

                    hdrl_image *himg = moo_sci_single_get_image(sci_single);
                    cpl_image *img = hdrl_image_get_image(himg);
                    int nx = cpl_image_get_size_x(img);

                    if (il == 0) {
                        table = cpl_table_new(nx);
                        cpl_table_new_column(table, MOO_RESP_WAVE,
                                             CPL_TYPE_DOUBLE);
                        for (int j = 1; j <= nx; j++) {
                            double w = (j - crpix1) * cd1_1 + crval1;
                            cpl_table_set_double(table, MOO_RESP_WAVE, j - 1,
                                                 w);
                        }
                    }
                    cpl_table_new_column(table, response_colname,
                                         CPL_TYPE_DOUBLE);
#if MOO_DEBUG_RESP
                    {
                        char *flux_colname =
                            cpl_sprintf("%s_%s", response_colname,
                                        MOO_RESP_SPECFLUX);
                        char *fluxfilter_colname =
                            cpl_sprintf("%s_%s", response_colname,
                                        MOO_RESP_SPECFLUXFILTER);
                        char *flx_colname =
                            cpl_sprintf("%s_%s", response_colname,
                                        MOO_RESP_FLX);
                        char *atm_colname =
                            cpl_sprintf("%s_%s", response_colname,
                                        MOO_RESP_ATM);


                        cpl_table_new_column(table, flux_colname,
                                             CPL_TYPE_DOUBLE);
                        cpl_table_new_column(table, fluxfilter_colname,
                                             CPL_TYPE_DOUBLE);
                        cpl_table_new_column(table, flx_colname,
                                             CPL_TYPE_DOUBLE);
                        cpl_table_new_column(table, atm_colname,
                                             CPL_TYPE_DOUBLE);

                        cpl_free(flux_colname);
                        cpl_free(fluxfilter_colname);
                        cpl_free(flx_colname);
                        cpl_free(atm_colname);
                    }
#endif
                    for (int j = 1; j <= nx; j++) {
                        int rej;
                        double f = cpl_image_get(img, j, idtarget, &rej);
                        if (rej != 0) {
                            f = NAN;
                        }

                        cpl_table_set_double(table, response_colname, j - 1, f);
#if MOO_DEBUG_RESP
                        {
                            char *flux_colname =
                                cpl_sprintf("%s_%s", response_colname,
                                            MOO_RESP_SPECFLUX);
                            cpl_table_set_double(table, flux_colname, j - 1, f);
                            cpl_free(flux_colname);
                        }
#endif
                    }
                }
            }
        }
        moo_resp_set_table(result, i, table);
    }
    return result;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Delete a moo_resp
  @param    self    moo_resp to delete
  @return   void

  If the moo_resp @em self is @c NULL, nothing is done and no error is set.

 */
/*----------------------------------------------------------------------------*/

void
moo_resp_delete(moo_resp *self)
{
    if (self != NULL) {
        if (self->primary_header != NULL) {
            cpl_propertylist_delete(self->primary_header);
        }
        if (self->ri != NULL) {
            cpl_table_delete(self->ri);
        }
        if (self->yj != NULL) {
            cpl_table_delete(self->yj);
        }
        if (self->h != NULL) {
            cpl_table_delete(self->h);
        }
        if (self->response_table != NULL) {
            cpl_table_delete(self->response_table);
        }
        cpl_free(self);
    }
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Save a moo_resp to a FITS file
  @param    self          moo_resp to write to disk or NULL
  @param    filename      Name of the file to write
  @return   CPL_ERROR_NONE or the relevant _cpl_error_code_ on error

  This function saves a moo_resp to a FITS file, using cfitsio.
  Only not NULL extensions are written.
*/
/*----------------------------------------------------------------------------*/
void
moo_resp_save(moo_resp *self, const char *filename)
{
    if (self != NULL) {
        cpl_propertylist_save(self->primary_header, filename, CPL_IO_CREATE);
        if (self->ri != NULL) {
            cpl_propertylist *h = cpl_propertylist_new();
            cpl_propertylist_append_string(h, MOO_PFITS_EXTNAME,
                                           MOO_RESP_EXT_RI);
            cpl_table_save(self->ri, h, h, filename, CPL_IO_EXTEND);
            cpl_propertylist_delete(h);
        }
        if (self->yj != NULL) {
            cpl_propertylist *h = cpl_propertylist_new();
            cpl_propertylist_append_string(h, MOO_PFITS_EXTNAME,
                                           MOO_RESP_EXT_YJ);
            cpl_table_save(self->yj, h, h, filename, CPL_IO_EXTEND);
            cpl_propertylist_delete(h);
        }
        if (self->h != NULL) {
            cpl_propertylist *h = cpl_propertylist_new();
            cpl_propertylist_append_string(h, MOO_PFITS_EXTNAME,
                                           MOO_RESP_EXT_H);
            cpl_table_save(self->h, h, h, filename, CPL_IO_EXTEND);
            cpl_propertylist_delete(h);
        }

        if (self->response_table != NULL) {
            cpl_propertylist *h = cpl_propertylist_new();
            cpl_propertylist_append_string(h, MOO_PFITS_EXTNAME,
                                           MOO_RESP_EXT_RESPONSE_TABLE);
            cpl_table_save(self->response_table, h, h, filename, CPL_IO_EXTEND);
            cpl_propertylist_delete(h);
        }
    }
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Load a RESP frame and create a moo_resp
  @return   1 newly allocated moo_resp or NULL in case of an error

  The returned object must be deallocated using moo_resp_delete().
  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
moo_resp *
moo_resp_load(const cpl_frame *frame)
{
    cpl_ensure(frame != NULL, CPL_ERROR_NULL_INPUT, NULL);

    const char *filename = cpl_frame_get_filename(frame);
    cpl_ensure(filename != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_errorstate prev_state = cpl_errorstate_get();

    moo_resp *res = moo_resp_new();

    res->primary_header = cpl_propertylist_load(filename, 0);

    cpl_size qnum = cpl_fits_find_extension(filename, MOO_RESP_EXT_RI);
    if (qnum > 0) {
        res->ri = cpl_table_load(filename, qnum, 0);
    }
    qnum = cpl_fits_find_extension(filename, MOO_RESP_EXT_YJ);
    if (qnum > 0) {
        res->yj = cpl_table_load(filename, qnum, 0);
    }

    qnum = cpl_fits_find_extension(filename, MOO_RESP_EXT_H);
    if (qnum > 0) {
        res->h = cpl_table_load(filename, qnum, 0);
    }

    qnum = cpl_fits_find_extension(filename, MOO_RESP_EXT_RESPONSE_TABLE);
    if (qnum > 0) {
        res->response_table = cpl_table_load(filename, qnum, 0);
    }

    if (!cpl_errorstate_is_equal(prev_state)) {
        cpl_msg_debug(__func__, "Error in loading response : %d",
                      cpl_error_get_code());
        moo_resp_delete(res);
        res = NULL;
        cpl_errorstate_set(prev_state);
    }
    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Get the name of the response colname for a given indexrbn
  @param    self the response
  @param    idrbn the indexrbn of the spectra
  @param    fspectro the SPECTRO (1, 2)
  @return   CPL_ERROR_NONE or the relevant _cpl_error_code_ on error

*/
/*----------------------------------------------------------------------------*/

const char *
moo_resp_get_colname(moo_resp *self, int idrbn, int fspectro)
{
    const char *colname = NULL;
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(self->response_table != NULL, CPL_ERROR_NULL_INPUT, NULL);
    int nrow = cpl_table_get_nrow(self->response_table);
    int min_diff = 1024;

    for (int i = 0; i < nrow; i++) {
        int n = cpl_table_get_int(self->response_table, MOO_RESP_TABLE_INDEXRBN,
                                  i, NULL);
        const char *ncolname = cpl_table_get_string(self->response_table,
                                                    MOO_RESP_TABLE_RESPONSE, i);
        const int spectro = cpl_table_get_int(self->response_table,
                                              MOO_RESP_TABLE_SPECTRO, i, NULL);
        if (spectro == fspectro) {
            int diff = (int)fabs(idrbn - n);

            if (diff == 0) {
                colname = ncolname;
                break;
            }
            else if (min_diff > diff) {
                colname = ncolname;
                min_diff = diff;
            }
        }
    }

    return colname;
}
/**@}*/
