/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/
#include <math.h>
#include <string.h>
#include <cpl.h>

#include "moo_s1d.h"
#include "moo_fits.h"
#include "moo_pfits.h"
#include "moo_badpix.h"
#include "moo_utils.h"
#include "moo_dfs.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup moo_s1d S1D format
 * @ingroup moo_data
 * This module provides functions to create, use, and destroy a @em moo_s1d
 *
 * Functionality include:
 *
 * @par Synopsis:
 * @code
 *   #include "moo_create_s1d.h"
 * @endcode
 */


/*----------------------------------------------------------------------------*/
/**@{*/

/*-----------------------------------------------------------------------------
                              Function codes
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Create a new moo_cube
  @return   1 newly allocated moo_cube or NULL in case of an error

  The returned object must be deallocated using moo_cube_delete().

 */
/*----------------------------------------------------------------------------*/
moo_s1d *
moo_s1d_new(void)
{
    moo_s1d *res = cpl_calloc(1, sizeof(moo_s1d));
    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Delete a moo_s1d
  @param    self    moo_s1d to delete
  @return   void

  If the moo_map @em self is @c NULL, nothing is done and no error is set.

 */
/*----------------------------------------------------------------------------*/

void
moo_s1d_delete(moo_s1d *self)
{
    if (self != NULL) {
        if (self->primary_header != NULL) {
            cpl_propertylist_delete(self->primary_header);
        }
        if (self->header != NULL) {
            cpl_propertylist_delete(self->header);
        }
        if (self->data != NULL) {
            cpl_table_delete(self->data);
        }

        cpl_free(self->filename);
        cpl_free(self);
    }
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Save a moo_s1d to a FITS file
  @param    self moo_s1d to write to disk or NULL
  @return   CPL_ERROR_NONE or the relevant _cpl_error_code_ on error

  This function saves a moo_s1d to a FITS file, using cfitsio.
  Only not NULL extensions are written.
*/
/*----------------------------------------------------------------------------*/
void
moo_s1d_save(moo_s1d *self)
{
    if (self != NULL) {
        cpl_table_save(self->data, self->primary_header, self->header,
                       self->filename, CPL_IO_CREATE);
    }
}


/**@}*/
