/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/
#include <math.h>
#include <cpl.h>
#include "moo_pfits.h"
#include "moo_fits.h"
#include "moo_qc.h"
#include "moo_saturate_map.h"
/*----------------------------------------------------------------------------*/
/**
 * @defgroup moo_saturate_map SATURATE_MAP format
 * @ingroup moo_data
 * This module provides functions to create, use, and destroy a @em moo_saturate_map
 *
 * Functionality include:
 *
 * @par Synopsis:
 * @code
 *   #include "moo_saturate_map.h"
 * @endcode
 */


/*----------------------------------------------------------------------------*/

/**@{*/

/*-----------------------------------------------------------------------------
                              Function codes
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Create a new moo_saturate_map
  @return   1 newly allocated moo_saturate_map or NULL in case of an error

  The returned object must be deallocated using moo_saturate_map_delete().

 */
/*----------------------------------------------------------------------------*/
moo_saturate_map *
moo_saturate_map_new(void)
{
    moo_saturate_map *res = cpl_calloc(1, sizeof(moo_saturate_map));
    return res;
}

moo_saturate_map *
moo_saturate_map_load(const cpl_frame *mapframe)
{
    cpl_ensure(mapframe != NULL, CPL_ERROR_NULL_INPUT, NULL);
    const char *filename = cpl_frame_get_filename(mapframe);
    cpl_ensure(filename != NULL, CPL_ERROR_NULL_INPUT, NULL);

    cpl_errorstate prev_state = cpl_errorstate_get();
    moo_saturate_map *res = moo_saturate_map_new();

    for (int i = 0; i < 2; i++) {
        for (int j = 0; j < 3; j++) {
            const char *extname = moo_detector_get_extname(j, i + 1);
            char *name = cpl_sprintf("%s_%s", MOO_SATURATE_MAP_INDEX, extname);
            cpl_propertylist *header = NULL;
            header = moo_fits_load_extension_header(filename, NULL, name);
            res->data_header[i * 3 + j] = header;

            if (header != NULL) {
                int naxis = moo_pfits_get_naxis(header);
                if (naxis == 2) {
                    res->data[i * 3 + j] =
                        moo_fits_load_extension_image(filename,
                                                      MOO_SATURATE_MAP_INDEX,
                                                      extname, CPL_TYPE_INT);
                    res->flux[i * 3 + j] =
                        moo_fits_load_extension_image(filename,
                                                      MOO_SATURATE_MAP_FLUX,
                                                      extname, CPL_TYPE_DOUBLE);
                    res->err[i * 3 + j] =
                        moo_fits_load_extension_image(filename,
                                                      MOO_SATURATE_MAP_ERR,
                                                      extname, CPL_TYPE_DOUBLE);
                    res->exptime[i * 3 + j] =
                        moo_fits_load_extension_image(filename,
                                                      MOO_SATURATE_MAP_TIME,
                                                      extname, CPL_TYPE_DOUBLE);
                }
            }
            cpl_free(name);
        }
    }
    if (!cpl_errorstate_is_equal(prev_state)) {
        moo_saturate_map_delete(res);
        res = NULL;
    }

    return res;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    set saturate map data for relevant extension
  @param self
  @param type the detector type
  @param ntas number of TAS
  @param data the map data
  @param flux the map flux
  @param err the map err
  @param exptime the map exptime
  @param header the map header
  @return   error code

 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_saturate_map_set_data(moo_saturate_map *self,
                          moo_detector_type type,
                          int ntas,
                          cpl_image *data,
                          cpl_image *flux,
                          cpl_image *err,
                          cpl_image *exptime,
                          cpl_propertylist *header)
{
    cpl_ensure_code(ntas >= 1 && ntas <= 2, CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(header != NULL, CPL_ERROR_NULL_INPUT);

    self->data[(ntas - 1) * 3 + type] = data;
    self->flux[(ntas - 1) * 3 + type] = flux;
    self->err[(ntas - 1) * 3 + type] = err;
    self->exptime[(ntas - 1) * 3 + type] = exptime;
    self->data_header[(ntas - 1) * 3 + type] = header;

    return CPL_ERROR_NONE;
}
/*----------------------------------------------------------------------------*/
/**
  @brief Add CPL_IMAGE extension to MAP filename and update moo_saurate_map structure
  @param self moo_bpm structure to update
  @param data CPL_IMAGE extension to store
  @param flux CPL_IMAGE flux extension to store
  @param err  CPL_IMAGE error extension to store
  @param exptime CPL_IMAGE extension to store
  @param type type of detector
  @param ntas number of spectrographs

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_saturate_map_add_data(moo_saturate_map *self,
                          cpl_image *data,
                          cpl_image *flux,
                          cpl_image *err,
                          cpl_image *exptime,
                          moo_detector_type type,
                          int ntas)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(self->filename != NULL, CPL_ERROR_NULL_INPUT);
    cpl_error_code status = CPL_ERROR_NONE;
    cpl_errorstate prev_state = cpl_errorstate_get();
    const char *extname = moo_detector_get_extname(type, ntas);
    cpl_propertylist *header = cpl_propertylist_new();
    char *name = cpl_sprintf("%s_%s", MOO_SATURATE_MAP_INDEX, extname);
    cpl_propertylist_append_string(header, MOO_PFITS_EXTNAME, name);
    cpl_free(name);
    if (data != NULL) {
        cpl_image_save(data, self->filename, CPL_TYPE_INT, header,
                       CPL_IO_EXTEND);
    }
    else {
        cpl_propertylist_save(header, self->filename, CPL_IO_EXTEND);
    }
    cpl_propertylist *tmp_header = cpl_propertylist_new();
    name = cpl_sprintf("%s_%s", MOO_SATURATE_MAP_FLUX, extname);
    cpl_propertylist_append_string(header, MOO_PFITS_EXTNAME, name);
    cpl_free(name);
    if (flux != NULL) {
        cpl_image_save(flux, self->filename, CPL_TYPE_DOUBLE, header,
                       CPL_IO_EXTEND);
    }
    else {
        cpl_propertylist_save(header, self->filename, CPL_IO_EXTEND);
    }
    cpl_propertylist_delete(tmp_header);

    tmp_header = cpl_propertylist_new();
    name = cpl_sprintf("%s_%s", MOO_SATURATE_MAP_ERR, extname);
    cpl_propertylist_append_string(header, MOO_PFITS_EXTNAME, name);
    cpl_free(name);
    if (err != NULL) {
        cpl_image_save(err, self->filename, CPL_TYPE_DOUBLE, header,
                       CPL_IO_EXTEND);
    }
    else {
        cpl_propertylist_save(header, self->filename, CPL_IO_EXTEND);
    }
    cpl_propertylist_delete(tmp_header);

    tmp_header = cpl_propertylist_new();
    name = cpl_sprintf("%s_%s", MOO_SATURATE_MAP_TIME, extname);
    cpl_propertylist_append_string(header, MOO_PFITS_EXTNAME, name);
    cpl_free(name);
    if (exptime != NULL) {
        cpl_image_save(exptime, self->filename, CPL_TYPE_DOUBLE, header,
                       CPL_IO_EXTEND);
    }
    else {
        cpl_propertylist_save(header, self->filename, CPL_IO_EXTEND);
    }
    cpl_propertylist_delete(tmp_header);

    moo_saturate_map_set_data(self, type, ntas, data, flux, err, exptime,
                              header);

    if (!cpl_errorstate_is_equal(prev_state)) {
        cpl_msg_error(__func__, "Error for adding data to %s %s (%s)",
                      self->filename, extname,
                      cpl_error_get_message_default(cpl_error_get_code()));
        status = cpl_error_get_code();
    }
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Save a moo_saturate_map to a FITS file
  @param    self moo_saturate_map to write to disk or NULL
  @param    filename Name of the file to write
  @return   CPL_ERROR_NONE or the relevant _cpl_error_code_ on error

  This function saves a moo_bpm to a FITS file, using cfitsio.
  Only not NULL extensions are written.
*/
/*----------------------------------------------------------------------------*/
void
moo_saturate_map_save(moo_saturate_map *self, const char *filename)
{
    if (self != NULL) {
        cpl_propertylist_save(self->primary_header, filename, CPL_IO_CREATE);
        for (int i = 0; i < 2; i++) {
            for (int j = 0; j < 3; j++) {
                const char *extname = moo_detector_get_extname(j, i + 1);
                moo_fits_write_extension_image(self->data[i * 3 + j], filename,
                                               MOO_SATURATE_MAP_INDEX, extname,
                                               CPL_TYPE_INT,
                                               self->data_header[i * 3 + j]);
                moo_fits_write_extension_image(self->flux[i * 3 + j], filename,
                                               MOO_SATURATE_MAP_FLUX, extname,
                                               CPL_TYPE_DOUBLE, NULL);
                moo_fits_write_extension_image(self->err[i * 3 + j], filename,
                                               MOO_SATURATE_MAP_ERR, extname,
                                               CPL_TYPE_DOUBLE, NULL);
                moo_fits_write_extension_image(self->exptime[i * 3 + j],
                                               filename, MOO_SATURATE_MAP_TIME,
                                               extname, CPL_TYPE_DOUBLE, NULL);
            }
        }
    }
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Delete a moo_map_saturate
  @param    self    moo_saturate_map to delete
  @return   void

  If the moo_saturate_map @em self is @c NULL, nothing is done and no error is set.

 */
/*----------------------------------------------------------------------------*/

void
moo_saturate_map_delete(moo_saturate_map *self)
{
    if (self != NULL) {
        int i;
        if (self->primary_header != NULL) {
            cpl_propertylist_delete(self->primary_header);
        }
        for (i = 0; i < 6; i++) {
            if (self->data[i] != NULL) {
                cpl_image_delete(self->data[i]);
            }
            if (self->flux[i] != NULL) {
                cpl_image_delete(self->flux[i]);
            }
            if (self->err[i] != NULL) {
                cpl_image_delete(self->err[i]);
            }
            if (self->exptime[i] != NULL) {
                cpl_image_delete(self->exptime[i]);
            }
            if (self->data_header[i] != NULL) {
                cpl_propertylist_delete(self->data_header[i]);
            }
        }
        if (self->filename != NULL) {
            cpl_free(self->filename);
        }
        cpl_free(self);
    }
}
/**@}*/
