/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/
#include <math.h>
#include <cpl.h>
#include "moo_pfits.h"
#include "moo_fits.h"
#include "moo_sci.h"
#include "moo_badpix.h"
#include "moo_target_table.h"
/*----------------------------------------------------------------------------*/
/**
 * @defgroup moo_sci SCI format
 * @ingroup moo_data
 * This module provides functions to create, use, and destroy a @em moo_sci
 *
 * Functionality include:
 *
 * @par Synopsis:
 * @code
 *   #include "moo_sci.h"
 * @endcode
 */

/*----------------------------------------------------------------------------*/

/**@{*/

/*-----------------------------------------------------------------------------
                              Function codes
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Create a new moo_sci
  @return   1 newly allocated moo_sci or NULL in case of an error

  The returned object must be deallocated using moo_sci_delete().

 */
/*----------------------------------------------------------------------------*/
moo_sci *
moo_sci_new(void)
{
    moo_sci *res = cpl_calloc(1, sizeof(moo_sci));
    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Delete a moo_sci
  @param    self    moo_sci to delete
  @return   void

  If the moo_rbn @em self is @c NULL, nothing is done and no error is set.

 */
/*----------------------------------------------------------------------------*/

void
moo_sci_delete(moo_sci *self)
{
    if (self != NULL) {
        if (self->primary_header != NULL) {
            cpl_propertylist_delete(self->primary_header);
        }

        if (self->ri != NULL) {
            moo_sci_single_delete(self->ri);
        }
        if (self->yj != NULL) {
            moo_sci_single_delete(self->yj);
        }
        if (self->h != NULL) {
            moo_sci_single_delete(self->h);
        }

        if (self->target_table != NULL) {
            moo_target_table_delete(self->target_table);
        }
        cpl_free(self);
    }
}

cpl_error_code
moo_sci_load(moo_sci *self, unsigned int level)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);

    for (int i = 0; i < 3; i++) {
        moo_sci_load_single(self, i, level);
    }
    moo_sci_get_target_table(self);

    return CPL_ERROR_NONE;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Add target table to SCI file and update moo_sci structure
  @param self moo_sci structure to update
  @param table target table to store on SCI file
  @return the relevant error code or CPL_ERROR_NONE

Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_sci_add_target_table(moo_sci *self, moo_target_table *table)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(self->filename != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(table != NULL, CPL_ERROR_NULL_INPUT);

    cpl_error_code status = CPL_ERROR_NONE;
    cpl_errorstate prev_state = cpl_errorstate_get();
    self->target_table = table;

    cpl_propertylist *h = cpl_propertylist_new();

    // FIXME: Use cpl_propertylist_update family of functions, rather than
    //        cpl_propertylist_append. This avoids unintentional keyword
    //        duplication!
    cpl_propertylist_append_string(h, MOO_PFITS_EXTNAME,
                                   MOO_TARGET_TABLE_EXTNAME);
    cpl_table_save(self->target_table->table, h, h, self->filename,
                   CPL_IO_EXTEND);

    cpl_propertylist_delete(h);

    if (!cpl_errorstate_is_equal(prev_state)) {
        cpl_msg_error(__func__, "Error for adding target table to %s (%d)",
                      self->filename, cpl_error_get_code());
        status = cpl_error_get_code();
        cpl_errorstate_set(prev_state);
    }
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief assign moo_sci_single structure in moo_sci structure
  @param self moo_sci structure to update
  @param single moo_sci_single structure

Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_sci_set_single(moo_sci *self, moo_sci_single *single)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);

    cpl_error_code status = CPL_ERROR_NONE;
    cpl_errorstate prev_state = cpl_errorstate_get();

    if (single != NULL) {
        switch (single->type) {
            case MOO_TYPE_RI:
                self->ri = single;
                break;
            case MOO_TYPE_YJ:
                self->yj = single;
                break;
            case MOO_TYPE_H:
                self->h = single;
                break;
        }
    }
    if (!cpl_errorstate_is_equal(prev_state)) {
        cpl_msg_error("moo_sci", "Error for adding SCI_SINGLE to file %s (%d)",
                      self->filename, cpl_error_get_code());
        status = cpl_error_get_code();
        cpl_errorstate_set(prev_state);
    }
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief Add SCI_SINGLE extension to SCI file and update moo_sci structure
  @param self moo_sci structure to update
  @param type the detector type (RI, YJ, H)
  @param single moo_sci_single structure
  @return the relevant error code or CPL_ERROR_NONE

Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_sci_add_single(moo_sci *self,
                   moo_detector_type type,
                   moo_sci_single *single)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(self->filename != NULL, CPL_ERROR_NULL_INPUT);
    cpl_error_code status = CPL_ERROR_NONE;
    cpl_errorstate prev_state = cpl_errorstate_get();

    moo_sci_set_single(self, single);

    moo_sci_single_save(single, self->filename, type);
    if (!cpl_errorstate_is_equal(prev_state)) {
        cpl_msg_error("moo_sci", "Error for adding single to %s (%s)",
                      self->filename,
                      cpl_error_get_message_default(cpl_error_get_code()));
        status = cpl_error_get_code();
    }
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Create a new empty SCI filename
  @return   1 newly allocated moo_sci or NULL in case of an error

The returned object must be deallocated using moo_sci_delete().

Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
moo_sci *
moo_sci_create(const cpl_frame *frame)
{
    const char *filename = NULL;
    cpl_ensure(frame != NULL, CPL_ERROR_NULL_INPUT, NULL);

    cpl_errorstate prev_state = cpl_errorstate_get();
    moo_sci *sci = moo_sci_new();

    filename = cpl_frame_get_filename(frame);
    sci->filename = filename;
    sci->primary_header = cpl_propertylist_load(filename, 0);
    sci->ri = moo_sci_single_create(filename, MOO_TYPE_RI);
    sci->yj = moo_sci_single_create(filename, MOO_TYPE_YJ);
    sci->h = moo_sci_single_create(filename, MOO_TYPE_H);
    if (!cpl_errorstate_is_equal(prev_state)) {
        cpl_msg_info("moo_sci", "sci create using filename %s : %d", filename,
                     cpl_error_get_code());
        cpl_errorstate_set(prev_state);
    }
    return sci;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Get the target table of SCI file
  @param    self moo_sci object to read
  @return   the TARGET_TABLE object or NULL

The returned object must be deallocated using moo_sci_delete().

Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
moo_target_table *
moo_sci_get_target_table(moo_sci *self)
{
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    if (self->target_table == NULL && self->filename != NULL) {
        cpl_size qnum =
            cpl_fits_find_extension(self->filename, MOO_TARGET_TABLE_EXTNAME);
        if (qnum > 0) {
            self->target_table = moo_target_table_new();

            cpl_errorstate prev_state = cpl_errorstate_get();
            self->target_table->table = cpl_table_load(self->filename, qnum, 0);

            if (!cpl_errorstate_is_equal(prev_state) &&
                CPL_ERROR_DATA_NOT_FOUND == cpl_error_get_code()) {
                cpl_errorstate_set(prev_state);
            }
            self->target_table->primary_header =
                cpl_propertylist_load(self->filename, qnum);
        }
    }
    return self->target_table;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_sci
  @brief    Load the type part in SCI and return it
  @param    self    the SCI
  @param    type    the type of extension to load
  @param    level   the mask error level (equal or greater than 0)

  @return  the moo_sci_single corresponding to the type or NULL

  Possible error code :
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 *
 */
/*----------------------------------------------------------------------------*/
moo_sci_single *
moo_sci_load_single(moo_sci *self, moo_detector_type type, int level)
{
    moo_sci_single *res = NULL;

    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);

    res = moo_sci_get_single(self, type);

    moo_sci_single_load(res, level);

    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup moo_sci
  @brief    Get the type part in SCI and return it
  @param    self    the SCI
  @param    type    the type of extension to get

  @return  the moo_sci_single corresponding to the type or NULL

  Possible error code :
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL

 */
/*----------------------------------------------------------------------------*/
moo_sci_single *
moo_sci_get_single(moo_sci *self, moo_detector_type type)
{
    moo_sci_single *result = NULL;

    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    switch (type) {
        case MOO_TYPE_RI:
            result = self->ri;
            break;
        case MOO_TYPE_YJ:
            result = self->yj;
            break;
        case MOO_TYPE_H:
            result = self->h;
            break;
        default:
            result = NULL;
            break;
    }
    return result;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Save a moo_sci to a FITS file
  @param    self          moo_sci to write to disk or NULL
  @param    filename      Name of the file to write
  @return   CPL_ERROR_NONE or the relevant _cpl_error_code_ on error

  This function saves a moo_sci to a FITS file, using cfitsio.
  Only not NULL extensions are written.
*/
/*----------------------------------------------------------------------------*/
void
moo_sci_save(moo_sci *self, const char *filename)
{
    if (self != NULL) {
        cpl_propertylist_save(self->primary_header, filename, CPL_IO_CREATE);
        moo_sci_single_save(self->ri, filename, MOO_TYPE_RI);
        moo_sci_single_save(self->yj, filename, MOO_TYPE_YJ);
        moo_sci_single_save(self->h, filename, MOO_TYPE_H);

        if (self->target_table != NULL) {
            cpl_propertylist *h = cpl_propertylist_new();
            cpl_propertylist_append_string(h, MOO_PFITS_EXTNAME,
                                           MOO_TARGET_TABLE_EXTNAME);

            cpl_table_save(self->target_table->table, h, h, filename,
                           CPL_IO_EXTEND);
            cpl_propertylist_delete(h);
        }
    }
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Compute SNR for all targets in SCI
  @param    self [SCI] file
  @param    skylines SKY_LINE_LIST
  @return   CPL_ERROR_NONE or the relevant _cpl_error_code_ on error

*/
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_sci_compute_snr(moo_sci *self, moo_sky_lines_list *skylines)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(skylines != NULL, CPL_ERROR_NULL_INPUT);

    unsigned int badpix_level =
        MOO_BADPIX_OUTSIDE_DATA_RANGE | MOO_BADPIX_CALIB_DEFECT;

    const char *snrs[] = { MOO_TARGET_TABLE_MEDSNRRI_SCI,
                           MOO_TARGET_TABLE_MEDSNRYJ_SCI,
                           MOO_TARGET_TABLE_MEDSNRH_SCI };

    moo_target_table *target_table = moo_sci_get_target_table(self);

    cpl_table *table = target_table->table;
    int nrow = cpl_table_get_nrow(table);
    for (int t = 0; t < nrow; t++) {
        int targ_idx =
            cpl_table_get_int(table, MOO_TARGET_TABLE_INDEXTARG, t, NULL);
        for (int i = 0; i < 3; i++) {
            moo_sci_single *single = moo_sci_load_single(self, i, badpix_level);
            if (single != NULL) {
                double snr =
                    moo_sci_single_compute_snr(single, targ_idx, skylines);
                cpl_table_set_double(table, snrs[i], t, snr);
            }
        }
    }

    return CPL_ERROR_NONE;
}
/**@}*/
