/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#include "math.h"
#include <cpl.h>
#include "moo_skycorr.h"
#include "sc_skycorr.h"
#include "hdrl_types.h"
#include "moo_params.h"
#include "moo_pfits.h"
#include "moo_utils.h"
/*----------------------------------------------------------------------------*/
/**
 * @defgroup moons_skycorr     Skycorr functions
 */
/*----------------------------------------------------------------------------*/

/**@{*/

static cpl_table *
_moo_spectrum_to_table(hdrl_image *himg,
                       int index,
                       double crpix1,
                       double crval1,
                       double cd1_1,
                       cpl_parameterlist *parlist)
{
    cpl_table *result = NULL;
    cpl_parameter *p;
    char errtxt[SC_MAXLEN];
    double meanlam = 0.;

    cpl_ensure(himg != NULL, CPL_ERROR_NULL_INPUT, NULL);

    int nrow = hdrl_image_get_size_x(himg);

    result = cpl_table_new(nrow);

    /* Get wavelength unit conversion factor */
    p = cpl_parameterlist_find(parlist, "wlgtomicron");
    double wlgtomicron = cpl_parameter_get_double(p);

    cpl_table_new_column(result, "lambda", CPL_TYPE_DOUBLE);
    cpl_table_new_column(result, "flux", CPL_TYPE_DOUBLE);
    cpl_table_new_column(result, "dflux", CPL_TYPE_DOUBLE);
    cpl_table_new_column(result, "mask", CPL_TYPE_INT);
    cpl_table_new_column(result, "weight", CPL_TYPE_DOUBLE);

    for (int i = 1; i <= nrow; i++) {
        int rej;

        double wave = crval1 + (i - crpix1) * cd1_1;
        wave = wave * wlgtomicron;
        hdrl_value val = hdrl_image_get_pixel(himg, i, index, &rej);

        int rejected = rej != 0 || isnan(val.data);
        cpl_table_set(result, "lambda", i - 1, wave);
        cpl_table_set(result, "flux", i - 1, val.data);
        cpl_table_set(result, "dflux", i - 1, val.error);
        cpl_table_set(result, "mask", i - 1, !rejected);
        if (val.error <= 0. || rejected) {
            cpl_table_set(result, "weight", i - 1, 0.);
            cpl_table_set(result, "flux", i - 1, 0.);
            cpl_table_set(result, "dflux", i - 1, 0.);
        }
        else {
            cpl_table_set(result, "weight", i - 1, 1. / val.error);
        }
    }

    if (cpl_table_get_nrow(result) == 0) {
        sprintf(errtxt, "%s: cpl_table *spec", SC_ERROR_NDA_TXT);
        cpl_error_set_message(cpl_func, SC_ERROR_NDA, "%s", errtxt);
        return NULL;
    }

    /* All weights = 0? */
    if (cpl_table_get_column_max(result, "weight") == 0) {
        sprintf(errtxt, "%s: cpl_table *spec (all weights = 0)",
                SC_ERROR_IOV_TXT);
        cpl_error_set_message(cpl_func, SC_ERROR_IOV, "%s", errtxt);
        return NULL;
    }

    /* Set mean wavelength */
    meanlam = 0.5 * (cpl_table_get(result, "lambda", 0, NULL) +
                     cpl_table_get(result, "lambda",
                                   cpl_table_get_nrow(result) - 1, NULL));
    p = cpl_parameterlist_find(parlist, "meanlam");
    cpl_parameter_set_double(p, meanlam);


    return result;
}

static cpl_error_code
__mo_sc_readspec_header(cpl_parameterlist *parlist, cpl_propertylist *header)
{
    /*!
     * Reads keywords from a FITS file header and puts the read values in the
     * general CPL parameter list. The presence of keywords for date (MJD or
     * date in years), time (UTC in s), and telescope altitude angle (in deg)
     * is required. The keyword names are included in the general parameter
     * list. The default names are MJD_OBS, TM-START, and ESO TEL ALT. If an
     * expected keyword cannot be found in the FITS header, an error message
     * will be written to the CPL error structure. The values of the required
     * parameters are also part of the general parameter list. If a value is
     * set manually, this value is taken instead of the corresponding FITS
     * keyword content.
     *
     * \b INPUT:
     * \param parlist   general CPL parameter list
     * \param filename  path and name of FITS file
     *
     * \b OUTPUT:
     * \param parlist   general CPL parameter list with FITS header values
     *
     * \b ERRORS:
     * - File opening failed
     * - Unexpected file structure
     */

    cpl_parameter *p;
    cpl_property *prop;
    cpl_type type;
    char key[SC_NKEY][SC_LENLINE + 1] = { "", "", "" }, errtxt[SC_MAXLEN];
    int i = 0, nerr = 0;
    double val = 0.;

    /* Parameter names related to FITS keywords */
    char keypar[SC_NKEY][SC_LENLINE + 1] = { "date_key", "time_key",
                                             "telalt_key" };
    char valpar[SC_NKEY][SC_LENLINE + 1] = { "date_val", "time_val",
                                             "telalt_val" };

    for (i = 0; i < SC_NKEY; i++) {
        /* Get value from parameter list */
        p = cpl_parameterlist_find(parlist, valpar[i]);
        val = cpl_parameter_get_double(p);

        /* Get name of required FITS keyword */
        p = cpl_parameterlist_find(parlist, keypar[i]);
        strncpy(key[i], cpl_parameter_get_string(p), SC_LENLINE + 1);

        /* Write info message */
        //cpl_msg_info(cpl_func, "Read keyword %s", key[i]);

        /* Get FITS keyword */
        prop = cpl_propertylist_get_property(header, key[i]);

        if (prop == NULL && val == -1.) {
            /* Set error message in the case of missing keyword */
            nerr++;
            sprintf(errtxt, "%s: (keyword %s not found)", SC_ERROR_UFS_TXT,
                    key[i]);
            cpl_error_set_message(cpl_func, SC_ERROR_UFS, "%s", errtxt);
        }
        else {
            /* Take FITS keyword only if parameter value was not given
               manually */
            if (val == -1.) {
                /* Check for type */
                type = cpl_property_get_type(prop);
                if (type == CPL_TYPE_DOUBLE) {
                    val = cpl_property_get_double(prop);
                }
                else if (type == CPL_TYPE_FLOAT) {
                    val = (double)cpl_property_get_float(prop);
                }
                else if (type == CPL_TYPE_INT) {
                    val = (double)cpl_property_get_int(prop);
                }
                else {
                    nerr++;
                    sprintf(errtxt, "%s: (non-numerical keyword %s)",
                            SC_ERROR_UFS_TXT, key[i]);
                    cpl_error_set_message(cpl_func, SC_ERROR_UFS, "%s", errtxt);
                    continue;
                }
            }

            /* MJD -> date in years (if required; save MJD before) */
            if (i == 0 && val > 3000.) {
                p = cpl_parameterlist_find(parlist, "mjd");
                cpl_parameter_set_double(p, val);
                val = sc_basic_mjd2fracyear(val);
            }

            /* Write new value into parameter list */
            p = cpl_parameterlist_find(parlist, valpar[i]);
            cpl_parameter_set_double(p, val);
        }
    }

    /* Return SC_ERROR_UFS in the case of keyword mismatch */
    if (nerr > 0) {
        return SC_ERROR_UFS;
    }

    return CPL_ERROR_NONE;
}

static cpl_error_code
_moo_sc_lines_readlist(cpl_table *groups, const char *linetabfile)
{
    /*!
     * Reads airglow line data from ASCII file in the sysdata/ folder and
     * writes them into a CPL table. Each line of the file must consist of
     * vacuum wavelength (1st), unextincted flux (2nd), atmospheric
     * transmission at zenith (3rd), variability class (4rd), roto-vibrational
     * system (5th), line group A number (6th), and line group B number (7th).
     * Header lines are allowed if they are marked by #.
     *
     *
     * \b OUTPUT:
     * \param groups   CPL table with airglow line information
     *
     * \b ERRORS:
     * - File opening failed
     * - Unexpected file structure
     */
    FILE *stream;
    char errtxt[SC_MAXLEN], str[SC_LENLINE + 2];
    cpl_boolean isoutrange = CPL_FALSE;
    int nhead = 0, nrec = 0, feat = 0, syst = 0, agroup = 0, bgroup = 0;
    int ncol0 = 0, ncolmin = 7, i = 0, ncol = 0;
    double lam = 0., flux = 0., trans = 0.;

    /* Create CPL table columns */
    cpl_table_new_column(groups, "lambda", CPL_TYPE_DOUBLE);
    cpl_table_new_column(groups, "flux", CPL_TYPE_DOUBLE);
    cpl_table_new_column(groups, "trans", CPL_TYPE_DOUBLE);
    cpl_table_new_column(groups, "feat", CPL_TYPE_INT);
    cpl_table_new_column(groups, "system", CPL_TYPE_INT);
    cpl_table_new_column(groups, "groupA", CPL_TYPE_INT);
    cpl_table_new_column(groups, "groupB", CPL_TYPE_INT);

    if ((stream = fopen(linetabfile, "r")) == NULL) {
        sprintf(errtxt, "%s: %s", SC_ERROR_FOF_TXT, linetabfile);
        return cpl_error_set_message(cpl_func, SC_ERROR_FOF, "%s", errtxt);
    }

    /* Write info message */
    //cpl_msg_info(cpl_func, "Read line list %s", linetabfile);

    /* Find number of header and data lines */
    while (fgets(str, SC_LENLINE + 2, stream) != NULL) {
        if (str[0] == '#') {
            nhead++;
            if (nrec != 0) {
                fclose(stream);
                sprintf(errtxt, "%s: %s (comment line in data part)",
                        SC_ERROR_UFS_TXT, linetabfile);
                return cpl_error_set_message(cpl_func, SC_ERROR_UFS, "%s",
                                             errtxt);
            }
        }
        else if (isdigit(str[0]) || isspace(str[0]) || str[0] == '-') {
            if (nrec == 0) {
                /* Number of values per line */
                ncol0 = sscanf(str, "%le %le %le %d %d %d %d", &lam, &flux,
                               &trans, &feat, &syst, &agroup, &bgroup);
                if (ncol0 == 0) {
                    fclose(stream);
                    sprintf(errtxt, "%s: %s (empty line)", SC_ERROR_UFS_TXT,
                            linetabfile);
                    return cpl_error_set_message(cpl_func, SC_ERROR_UFS, "%s",
                                                 errtxt);
                }
                else if (ncol0 < ncolmin) {
                    fclose(stream);
                    sprintf(errtxt, "%s: %s (too low number of columns)",
                            SC_ERROR_UFS_TXT, linetabfile);
                    return cpl_error_set_message(cpl_func, SC_ERROR_UFS, "%s",
                                                 errtxt);
                }
            }
            nrec++;
        }
        else {
            fclose(stream);
            sprintf(errtxt, "%s: %s (unexpected first character at line)",
                    SC_ERROR_UFS_TXT, linetabfile);
            return cpl_error_set_message(cpl_func, SC_ERROR_UFS, "%s", errtxt);
        }
    }
    rewind(stream);

    /* No data points */
    if (nrec == 0) {
        fclose(stream);
        sprintf(errtxt, "%s: %s (no data)", SC_ERROR_UFS_TXT, linetabfile);
        return cpl_error_set_message(cpl_func, SC_ERROR_UFS, "%s", errtxt);
    }

    /* Resize output table */
    cpl_table_set_size(groups, nrec);

    /* Skip header */
    for (i = 0; i < nhead; i++) {
        if (fgets(str, SC_LENLINE + 2, stream)) {
        };
    }

    /* Read line data from file and write it to CPL table */

    for (i = 0; i < nrec; i++) {
        ncol = fscanf(stream, "%le %le %le %d %d %d %d", &lam, &flux, &trans,
                      &feat, &syst, &agroup, &bgroup);

        if (ncol != ncol0) {
            cpl_table_set_size(groups, 0);
            fclose(stream);
            sprintf(errtxt, "%s: %s (unexpected number of values at line)",
                    SC_ERROR_UFS_TXT, linetabfile);
            return cpl_error_set_message(cpl_func, SC_ERROR_UFS, "%s", errtxt);
        }

        cpl_table_set(groups, "lambda", i, lam);
        cpl_table_set(groups, "flux", i, flux);
        cpl_table_set(groups, "trans", i, trans);
        cpl_table_set(groups, "feat", i, feat);
        cpl_table_set(groups, "system", i, syst);
        cpl_table_set(groups, "groupA", i, agroup);
        if (syst > 0 && bgroup == 0) {
            /* Negative numbers for molecular lines without B group number */
            cpl_table_set(groups, "groupB", i, -syst);
        }
        else {
            cpl_table_set(groups, "groupB", i, bgroup);
        }

        if (flux < 0.) {
            flux = 0.;
            isoutrange = CPL_TRUE;
        }

        if (trans < 0.) {
            trans = 0.;
            isoutrange = CPL_TRUE;
        }

        if (trans > 1.) {
            trans = 1.;
            isoutrange = CPL_TRUE;
        }

        if (feat < 1) {
            feat = 1;
            isoutrange = CPL_TRUE;
        }

        if (agroup < 0) {
            agroup = 0;
            isoutrange = CPL_TRUE;
        }

        if (bgroup < 0) {
            bgroup = 0;
            isoutrange = CPL_TRUE;
        }
    }

    fclose(stream);

    if (isoutrange == CPL_TRUE) {
        cpl_msg_warning(cpl_func,
                        "%s: Input value(s) out of range -> "
                        "Take lowest/highest allowed value(s)",
                        linetabfile);
    }

    return CPL_ERROR_NONE;
}

static cpl_error_code
_moo_sc_lines_readsolflux(cpl_parameterlist *parlist, const char *soldatfile)
{
    /*!
     * Reads solar radio flux for given year and month from an ASCII file in
     * the sysdata/ folder. If the file source is set to 'WEB' (instead of
     * 'LOCAL'), the local file is substituted by the one from the Canadian
     * reference web site before the reading is started. If the programme
     * fails to find the correct flux value, the \e solflux parameter is set
     * to -1. If no file source is given, the average value of the solar
     * cycles 19 to 23, i.e. 130 sfu, is assumed.
     *
     * \b INPUT:
     * \param parlist  input CPL parameter list
     *
     * \b OUTPUT:
     * \param parlist  CPL parameter list with updated solar radio flux
     *
     * \b ERRORS:
     * - File opening failed
     * - Unexpected file structure
     */

    FILE *stream;
    cpl_parameter *p;
    char soldatsource[SC_MAXLEN];
    char errtxt[SC_MAXLEN], str[SC_LENLINE + 2];
    int year = 0, month = 0, i = 0, y = 0, m = 0, ncol = 0, ncolmin = 5;
    double obsflux = 0., adjflux = 0., absflux = 0., solflux = -1.;

    /* Get year and month from parameter list */
    p = cpl_parameterlist_find(parlist, "year");
    year = cpl_parameter_get_int(p);
    p = cpl_parameterlist_find(parlist, "month");
    month = cpl_parameter_get_int(p);

    /* Set solar radio flux to -1 to indicate if month is not found */
    p = cpl_parameterlist_find(parlist, "solflux");
    cpl_parameter_set_double(p, -1.);

    /* Get source of the solar radio flux data from parameter list */
    p = cpl_parameterlist_find(parlist, "soldatsource");
    strncpy(soldatsource, cpl_parameter_get_string(p), SC_MAXLEN);

    /* Take average solar radio flux if no data source */
    if ((strcmp(soldatsource, "LOCAL") != 0 &&
         strcmp(soldatsource, "WEB") != 0) ||
        (strcmp(soldatsource, "WEB") == 0 &&
         ((year == 1947 && month == 1) || year < 1947))) {
        p = cpl_parameterlist_find(parlist, "solflux");
        cpl_parameter_set_double(p, 130.);
        /*cpl_msg_warning(cpl_func, "No source for solar radio flux available"
                        " -> Take average of 130 sfu");*/
        return CPL_ERROR_NONE;
    }

    /* Write info message */
    /*cpl_msg_info(cpl_func, "Get solar radio flux for %d/%d from %s", month,
                 year, soldatfile);*/

    /* Open file with sky line data */
    if ((stream = fopen(soldatfile, "r")) == NULL) {
        sprintf(errtxt, "%s: %s", SC_ERROR_FOF_TXT, soldatfile);
        return cpl_error_set_message(cpl_func, SC_ERROR_FOF, "%s", errtxt);
    }

    /* Skip header */
    for (i = 0; i < 2; i++) {
        if (fgets(str, SC_LENLINE + 2, stream)) {
        };
    }

    /* Read file and find solar radio flux for given year and month */

    while (fgets(str, SC_LENLINE + 2, stream) != NULL) {
        if (isdigit(str[0]) || isspace(str[0])) {
            /* Read values */
            ncol = sscanf(str, "%d %d %le %le %le", &y, &m, &obsflux, &adjflux,
                          &absflux);

            /* Check number of values per line */
            if (ncol == 0) {
                fclose(stream);
                sprintf(errtxt, "%s: %s (empty line)", SC_ERROR_UFS_TXT,
                        soldatfile);
                return cpl_error_set_message(cpl_func, SC_ERROR_UFS, "%s",
                                             errtxt);
            }
            else if (ncol < ncolmin) {
                fclose(stream);
                sprintf(errtxt, "%s: %s (too low number of columns)",
                        SC_ERROR_UFS_TXT, soldatfile);
                return cpl_error_set_message(cpl_func, SC_ERROR_UFS, "%s",
                                             errtxt);
            }

            /* Check validity of values */
            if (y < 1947 || m < 1 || m > 12 || obsflux < 0.) {
                fclose(stream);
                sprintf(errtxt, "%s: %s (invalid value(s))", SC_ERROR_UFS_TXT,
                        soldatfile);
                return cpl_error_set_message(cpl_func, SC_ERROR_UFS, "%s",
                                             errtxt);
            }

            /* Find solar radio flux for given year and month */
            if (y == year && m == month) {
                solflux = obsflux;
                break;
            }
        }
        else {
            /* No digit or space at the beginning of the line */
            fclose(stream);
            sprintf(errtxt, "%s: %s (unexpected first character at line)",
                    SC_ERROR_UFS_TXT, soldatfile);
            return cpl_error_set_message(cpl_func, SC_ERROR_UFS, "%s", errtxt);
        }
    }

    fclose(stream);

    /* If the required month is not in the file on the web server, take the
       last entry */
    if (strcmp(soldatsource, "WEB") == 0) {
        if ((month != 1 && m < month - 1 && y == year) ||
            (month != 1 && y < year) ||
            (month == 1 && m < 12 && y == year - 1) ||
            (month == 1 && y < year - 1)) {
            /* Warning message if last file entry is outdated */
            cpl_msg_warning(cpl_func, "Solar radio fluxes until %d/%d only", m,
                            y);
        }
        if (obsflux != solflux) {
            solflux = obsflux;
        }
    }

    /* Write solar radio flux into parameter list */
    p = cpl_parameterlist_find(parlist, "solflux");
    cpl_parameter_set_double(p, floor(solflux + 0.5));

    return CPL_ERROR_NONE;
}

static cpl_error_code
_moo_sc_lines_getmodelbins(cpl_parameterlist *parlist,
                           const cpl_frame *solflux_frame)
{
    /*!
     * Gets season and time bins for airglow model from observing date in
     * years and UT observing time in s and writes them into the general
     * parameter list. Moreover, the solar radio flux is obtained from year
     * and month which are also taken from observing date. The flux value is
     * read from an ASCII file in the data folder. If the required year and
     * month cannot be found, the local file is updated by a one downloaded
     * from the Canadian reference web server. The most recent entry is taken,
     * if the required month is not in the list. If the procedure fails, an
     * average value of 130 sfu is written into the general parameter list.
     * The entire procedure for obtaining the solar radio flux is not carried
     * out if a positive value for the \e solflux parameter is already in the
     * parameter list.
     *
     * \b INPUT:
     * \param parlist  input CPL parameter list
     *
     * \b OUTPUT:
     * \param parlist  parameter list with sky model bins
     *
     * \b ERRORS:
     * - Ivalid object value(s)
     * - see ::sc_lines_readsolflux
     */
    cpl_error_code status = CPL_ERROR_NONE;
    cpl_parameter *p, *q;
    char errtxt[SC_MAXLEN];
    int year = 0, month = 0, day = 0, hh = 0, mm = 0;
    int season = 0, timebin = 0;
    double ss = 0., fracyear = 0., ut = 0., nlen = 0.;
    double tlim1 = 0., tlim2 = 0.;
    double nstart[6] = { 0.98, 0.48, -0.44, -0.57, -0.23, 0.29 };
    double nend[6] = { 8.50, 9.29, 9.77, 10.07, 9.61, 8.60 };

    /* Get date in years and UT time in s from parameter list */
    p = cpl_parameterlist_find(parlist, "date_val");
    fracyear = cpl_parameter_get_double(p);
    p = cpl_parameterlist_find(parlist, "time_val");
    ut = cpl_parameter_get_double(p) / 3600.;

    /* Check for existence of date and time information */
    if (fracyear < 0 || ut < 0) {
        sprintf(errtxt,
                "%s: cpl_parameterlist *parlist (invalid date_val "
                "and/or time_val)",
                SC_ERROR_IOV_TXT);
        return cpl_error_set_message(cpl_func, SC_ERROR_IOV, "%s", errtxt);
    }

    /* Derive season bin */
    sc_basic_fracyear2date(&year, &month, &day, &hh, &mm, &ss, &fracyear);
    if (month == 12) {
        season = 1;
    }
    else {
        season = floor((float)month / 2) + 1;
    }

    /* Derive time bin */
    if (ut > 16.) {
        ut -= 24.;
    }
    nlen = nend[season - 1] - nstart[season - 1];
    tlim1 = nstart[season - 1] + nlen / 3;
    tlim2 = nend[season - 1] - nlen / 3;
    if (ut < tlim1) {
        timebin = 1;
    }
    else if (ut >= tlim2) {
        timebin = 3;
    }
    else {
        timebin = 2;
    }

    /* Write season and time bins into parameter list */
    p = cpl_parameter_new_value("season", CPL_TYPE_INT, "", "", season);
    cpl_parameterlist_append(parlist, p);
    p = cpl_parameter_new_value("timebin", CPL_TYPE_INT, "", "", timebin);
    cpl_parameterlist_append(parlist, p);

    /* Write year and month into parameter list */
    p = cpl_parameter_new_value("year", CPL_TYPE_INT, "", "", year);
    cpl_parameterlist_append(parlist, p);
    p = cpl_parameter_new_value("month", CPL_TYPE_INT, "", "", month);
    cpl_parameterlist_append(parlist, p);

    /* Take solar radio flux from parameter list if given */
    q = cpl_parameterlist_find(parlist, "solflux");

    if (cpl_parameter_get_double(q) >= 0) {
        /*cpl_msg_info(cpl_func, "SOLFLUX = %g sfu",
                     cpl_parameter_get_double(q));*/
        return CPL_ERROR_NONE;
    }

    /* Get solar radio flux for year and month from file in sysdata/ */
    p = cpl_parameterlist_find(parlist, "soldatsource");
    cpl_parameter_set_string(p, "LOCAL");
    const char *solflux_filename = cpl_frame_get_filename(solflux_frame);
    status = _moo_sc_lines_readsolflux(parlist, solflux_filename);

    /* If reading fails, take flux from file on web server */
    q = cpl_parameterlist_find(parlist, "solflux");

    /* If the procedure fails, take average value of 130 sfu */
    if (cpl_parameter_get_double(q) < 0) {
        cpl_parameter_set_string(p, "NONE");
        _moo_sc_lines_readsolflux(parlist, NULL);
    }

    /* Write info message */
    q = cpl_parameterlist_find(parlist, "solflux");
    //cpl_msg_info(cpl_func, "SOLFLUX = %g sfu", cpl_parameter_get_double(q));

    return status;
}

static cpl_error_code
_moo_sc_lines_readvarpar(cpl_array *varpar,
                         cpl_parameterlist *parlist,
                         const char *vardatfile)
{
    /*!
     * Reads data related to airglow scaling from an ASCII file in the
     * sysdata/ folder and fills a CPL array with feature-related correction
     * factors. The resulting factors are related to the standard strengths of
     * airglow emission features of the upper atmosphere given by Hanuschik
     * (2003) and Rousselot et al. (2000). The scaling factors depend on the
     * emission layer width (related to airmass), the monthly-averaged solar
     * flux in sfu, the season of the year, and the time of the day.
     *
     * \b INPUT:
     * \param parlist  input CPL parameter list
     *
     * \b OUTPUT:
     * \param varpar   correction factors for variability classes as CPL array
     *
     * \b ERRORS:
     * - File opening failed
     * - Unexpected file structure
     * - Invalid object value(s)
     */

    FILE *stream;
    cpl_parameter *p;
    scpar x[SC_MAXPAR], m[SC_MAXPAR];
    char errtxt[SC_MAXLEN];
    int n = SC_MAXPAR, nfeat = 0, nseason = 0, ntime = 0, season = 0;
    int timebin = 0, i = 0, j = 0;
    double alt = 0., solflux = 0., height = 0., scale = 0., cons = 0.;
    double slope = 0., mean = 0., z = 0., cvr = 0., csol = 0.;

    if ((stream = fopen(vardatfile, "r")) == NULL) {
        sprintf(errtxt, "%s: %s", SC_ERROR_FOF_TXT, vardatfile);
        return cpl_error_set_message(cpl_func, SC_ERROR_FOF, "%s", errtxt);
    }

    /* Write info message */
    //cpl_msg_info(cpl_func, "Read line variability model file %s", vardatfile);

    /* Read airglow scaling parameter file */

    /* Read numbers for data set size */
    sc_basic_readline(stream, x, &n);
    nfeat = x[0].i;
    sc_basic_readline(stream, x, &n);
    nseason = x[0].i;
    ntime = x[1].i;
    //nbin = (nseason + 1) * (ntime + 1);

    /* Return NULL vectors in the case of one or more zero for the data set
       size */
    if (nfeat == 0 || nseason == 0 || ntime == 0) {
        fclose(stream);
        sprintf(errtxt, "%s: %s (nfeat == 0 || nseason == 0 || ntime == 0)",
                SC_ERROR_UFS_TXT, vardatfile);
        return cpl_error_set_message(cpl_func, SC_ERROR_UFS, "%s", errtxt);
    }

    /* Get and check airglow model parameters */
    p = cpl_parameterlist_find(parlist, "telalt_val");
    alt = cpl_parameter_get_double(p);
    p = cpl_parameterlist_find(parlist, "solflux");
    solflux = cpl_parameter_get_double(p);
    p = cpl_parameterlist_find(parlist, "season");
    season = cpl_parameter_get_int(p);
    p = cpl_parameterlist_find(parlist, "timebin");
    timebin = cpl_parameter_get_int(p);
    if (alt < 0. || alt > 90. || solflux < 0. || season < 0 ||
        season > nseason || timebin < 0 || timebin > ntime) {
        fclose(stream);
        sprintf(errtxt,
                "%s: cpl_parameterlist *parlist (telalt, solflux, "
                "season, and/or timebin out of range) ",
                SC_ERROR_IOV_TXT);
        return cpl_error_set_message(cpl_func, SC_ERROR_IOV, "%s", errtxt);
    }

    /* Set size of varpar array */
    cpl_array_set_size(varpar, nfeat);

    /* Read data for each feature, extract bin data for selected time, and
       fill CPL array with feature-specific flux correction factors */

    for (i = 0; i < nfeat; i++) {
        /* Time-independent parameters */
        sc_basic_readline(stream, x, &n);  // skip molmass
        sc_basic_readline(stream, x, &n);  // skip temp
        sc_basic_readline(stream, x, &n);
        height = x[0].d;
        sc_basic_readline(stream, x, &n);
        scale = x[0].d;
        sc_basic_readline(stream, x, &n);
        cons = x[0].d;
        slope = x[1].d;

        /* Mean value for selected time */
        for (j = 0; j < ntime + 1; j++) {
            sc_basic_readline(stream, m, &n);
            if (j == timebin) {
                mean = m[season].d;
            }
        }

        /* Standard deviation for selected time (skipped) */
        for (j = 0; j < ntime + 1; j++) {
            sc_basic_readline(stream, m, &n);
        }

        /* Emission line brightness depending on airmass and layer height
           (van Rhijn 1921) */
        z = (90. - alt) * CPL_MATH_RAD_DEG;
        cvr = 1 / sqrt(1 - pow((SC_ERAD / (SC_ERAD + height)) * sin(z), 2));

        /* Influence of solar radio flux [sfu] */
        csol = slope * solflux + cons;

        /* Set feature-specific correction factors */
        cpl_array_set(varpar, i, scale * cvr * csol * mean);
    }

    /* Close file */
    fclose(stream);

    return CPL_ERROR_NONE;
}

static cpl_error_code
_moo_sc_lines(cpl_table *groups,
              cpl_parameterlist *parlist,
              const cpl_frame *solflux_frame,
              const cpl_frame *airglow_group_frame,
              const cpl_frame *airglow_var_frame)
{
    /*!
     * \callgraph
     *
     * Reads airglow line list from an ASCII file and modifies the line
     * fluxes depending on variability class, period of the night, season,
     * solar activity, and zenith distance. Air wavelengths are provided if
     * required.
     *
     * \b INPUT:
     * \param parlist  input CPL parameter list
     *
     * \b OUTPUT:
     * \param groups   CPL table with airglow line information
     * \param parlist  parameter list complemented by parameters related to
     *                 airglow model
     *
     * \b ERRORS:
     * - see subroutines
     */

    cpl_error_code status = CPL_ERROR_NONE;
    cpl_array *varpar;

    const char *airglow_group_filename =
        cpl_frame_get_filename(airglow_group_frame);
    const char *airglow_var_filename =
        cpl_frame_get_filename(airglow_var_frame);

    /* Read line groups from ASCII file */
    if ((status = _moo_sc_lines_readlist(groups, airglow_group_filename)) !=
        CPL_ERROR_NONE) {
        return status;
    }
    /* Get season and time bins for airglow model from time of observation */
    if ((status = _moo_sc_lines_getmodelbins(parlist, solflux_frame)) !=
        CPL_ERROR_NONE) {
        return status;
    }

    /* Read scaling factors of line variability classes from ASCII file */
    varpar = cpl_array_new(0, CPL_TYPE_DOUBLE);
    if ((status =
             _moo_sc_lines_readvarpar(varpar, parlist, airglow_var_filename)) !=
        CPL_ERROR_NONE) {
        cpl_array_delete(varpar);
        return status;
    }

    /* Modify line fluxes by multiplying factors depending on the variability
       class and the airmass */

    sc_lines_scalelines(groups, varpar, parlist);


    /* Convert vacuum to air wavelengths in line group list if required */
    if ((status = sc_lines_vactoair(groups, parlist)) != CPL_ERROR_NONE) {
        cpl_array_delete(varpar);
        return status;
    }

    /* Free allocated memory */
    cpl_array_delete(varpar);

    return CPL_ERROR_NONE;
}

static cpl_error_code
_moo_table_to_spectrum(cpl_table *scispec,
                       hdrl_image *himg,
                       cpl_image *sky,
                       int index)
{
    /*!
     * Fills CPL table with the results of SKYCORR saved as a FITS table.
     * Converts micron into initial wavelength units if necessary.
     *
     * \b INPUT:
     * \param results  empty CPL table
     * \param parlist  general CPL parameter list
     *
     * \b OUTPUT:
     * \param results  CPL table with results of SKYCORR
     *
     * \b ERRORS:
     * - File opening failed
     */
    cpl_ensure_code(himg != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(sky != NULL, CPL_ERROR_NULL_INPUT);

    int nrow = hdrl_image_get_size_x(himg);

    for (int i = 1; i <= nrow; i++) {
        int mask = cpl_table_get_int(scispec, "mask", i - 1, NULL);
        double flux = NAN;
        double err = NAN;
        double sky_flux = cpl_table_get_double(scispec, "mflux", i - 1, NULL);
        cpl_image_set(sky, i, index, sky_flux);

        if (mask == 1) {
            flux = cpl_table_get_double(scispec, "scflux", i - 1, NULL);
            err = cpl_table_get_double(scispec, "scdflux", i - 1, NULL);
        }
        hdrl_value val;
        val.data = flux;
        val.error = err;
        hdrl_image_set_pixel(himg, i, index, val);
    }

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
  @brief call the skycorr subtract sky function
  @param rbn hdrl image of science spectrum to subtract from sky
  @param sky  hdrl image of sky to subtract
  @param indextarg the index o the target 
  @param indexrbn  the index of OBJ in rbn
  @param crpix1  the crpix1 keyword
  @param crval1  the crval1 keyword
  @param cd1_1  the cd1_1 keyword
  @param header the RBN header where to read keyword
  @param solflux_frame the monthly averages of solar radio flux at 10.7 cm frame
  @param airglow_group_frame the airglow line list frame (use in subsky)
  @param airglow_var_frame the airglow scaling parameters frame (use in subsky)
  @param params the skycorr parameters
  @param sci the result sky subtracted image
  @param sci_sky the associated sky subtracted
  @return the relevant error code or CPL_ERROR_NONE
*
* Possible error code :
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL or list size == 0
*
*/
cpl_error_code
moo_skycorr(hdrl_image *rbn,
            hdrl_image *sky,
            int indextarg,
            int indexrbn,
            double crpix1,
            double crval1,
            double cd1_1,
            cpl_propertylist *header,
            const cpl_frame *solflux_frame,
            const cpl_frame *airglow_group_frame,
            const cpl_frame *airglow_var_frame,
            moo_skycorr_params *params,
            hdrl_image *sci,
            cpl_image *sci_sky)
{
    cpl_error_code status = CPL_ERROR_NONE;
    char errtxt[SC_MAXLEN];
    cpl_table *scispec = NULL;
    cpl_table *skyspec = NULL;
    cpl_table *groups = NULL;
    cpl_table *skylinetab = NULL;
    cpl_table *scilinetab = NULL;
    cpl_parameterlist *parlist = NULL;
    cpl_parameter *p = NULL;

    cpl_ensure_code(header != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(airglow_var_frame != NULL, CPL_ERROR_NULL_INPUT);
    double ltol = params->ltol;
    double min_line_dist_fac = params->min_line_dist_fac;
    double min_line_flux_fac = params->min_line_flux_fac;

    double fluxlim = params->fluxlim;

    double ftol = params->ftol;
    double xtol = params->xtol;
    double wtol = params->wtol;
    int cheby_max = params->cheby_max;
    int cheby_min = params->cheby_min;
    double cheby_const = params->cheby_const;
    int rebintype = params->rebintype;
    double weightlim = params->weightlim;
    double siglim = params->siglim;
    double fitlim = params->fitlim;

    cpl_msg_info("test", "call SC with indextarg %d indexrbn %d", indextarg,
                 indexrbn);
    parlist = cpl_parameterlist_new();

    p = cpl_parameter_new_value("col_dflux", CPL_TYPE_STRING, "", "", "dflux");
    cpl_parameterlist_append(parlist, p);

    p = cpl_parameter_new_value("vac_air", CPL_TYPE_STRING, "", "", "air");
    cpl_parameterlist_append(parlist, p);
    p = cpl_parameter_new_value("wlgtomicron", CPL_TYPE_DOUBLE, "", "", 1e-03);
    cpl_parameterlist_append(parlist, p);
    p = cpl_parameter_new_value("date_key", CPL_TYPE_STRING, "", "", "MJD-OBS");
    cpl_parameterlist_append(parlist, p);
    p = cpl_parameter_new_value("date_val", CPL_TYPE_DOUBLE, "", "", -1.);
    cpl_parameterlist_append(parlist, p);
    p = cpl_parameter_new_value("mjd", CPL_TYPE_DOUBLE, "", "", -1.);
    cpl_parameterlist_append(parlist, p);
    p = cpl_parameter_new_value("time_key", CPL_TYPE_STRING, "", "",
                                MOO_PFITS_UTC);
    cpl_parameterlist_append(parlist, p);
    p = cpl_parameter_new_value("time_val", CPL_TYPE_DOUBLE, "", "", -1.);
    cpl_parameterlist_append(parlist, p);
    p = cpl_parameter_new_value("telalt_key", CPL_TYPE_STRING, "", "",
                                "ESO TEL ALT");
    cpl_parameterlist_append(parlist, p);
    p = cpl_parameter_new_value("telalt_val", CPL_TYPE_DOUBLE, "", "", -1.);
    cpl_parameterlist_append(parlist, p);

    /* Required input data */
    p = cpl_parameter_new_value("linetabname", CPL_TYPE_STRING, "", "",
                                "skylinegroups");
    cpl_parameterlist_append(parlist, p);
    p = cpl_parameter_new_value("vardatname", CPL_TYPE_STRING, "", "",
                                "sky_featvar");
    cpl_parameterlist_append(parlist, p);
    p = cpl_parameter_new_value(
        "soldaturl", CPL_TYPE_STRING, "", "",
        "ftp.geolab.nrcan.gc.ca/data/solar_flux/monthly_averages");
    cpl_parameterlist_append(parlist, p);
    p = cpl_parameter_new_value("soldatname", CPL_TYPE_STRING, "", "",
                                "solflux_monthly_average.txt");
    cpl_parameterlist_append(parlist, p);
    p = cpl_parameter_new_value("soldatsource", CPL_TYPE_STRING, "", "",
                                "NONE");
    cpl_parameterlist_append(parlist, p);
    p = cpl_parameter_new_value("solflux", CPL_TYPE_DOUBLE, "", "", -1.);
    cpl_parameterlist_append(parlist, p);

    /* Line identification */
    p = cpl_parameter_new_value("fwhm", CPL_TYPE_DOUBLE, "", "", MOO_SC_FWHM);
    cpl_parameterlist_append(parlist, p);
    p = cpl_parameter_new_range("varfwhm", CPL_TYPE_INT, "", "", MOO_SC_VARFWHM,
                                0, 1);
    cpl_parameterlist_append(parlist, p);
    p = cpl_parameter_new_value("meanlam", CPL_TYPE_DOUBLE, "", "", 1.);
    cpl_parameterlist_append(parlist, p);
    p = cpl_parameter_new_value("ltol", CPL_TYPE_DOUBLE, "", "", ltol);
    cpl_parameterlist_append(parlist, p);
    p = cpl_parameter_new_value("min_line_dist_fac", CPL_TYPE_DOUBLE, "", "",
                                min_line_dist_fac);
    cpl_parameterlist_append(parlist, p);
    p = cpl_parameter_new_value("min_line_flux_fac", CPL_TYPE_DOUBLE, "", "",
                                min_line_flux_fac);
    cpl_parameterlist_append(parlist, p);
    p = cpl_parameter_new_value("fluxlim", CPL_TYPE_DOUBLE, "", "", fluxlim);
    cpl_parameterlist_append(parlist, p);
    p = cpl_parameter_new_value("iteration", CPL_TYPE_INT, "", "", 0);
    cpl_parameterlist_append(parlist, p);

    /* Fitting of sky lines */
    p = cpl_parameter_new_value("ftol", CPL_TYPE_DOUBLE, "", "", ftol);
    cpl_parameterlist_append(parlist, p);
    p = cpl_parameter_new_value("xtol", CPL_TYPE_DOUBLE, "", "", xtol);
    cpl_parameterlist_append(parlist, p);
    p = cpl_parameter_new_value("wtol", CPL_TYPE_DOUBLE, "", "", wtol);
    cpl_parameterlist_append(parlist, p);
    p = cpl_parameter_new_value("cheby_max", CPL_TYPE_INT, "", "", cheby_max);
    cpl_parameterlist_append(parlist, p);
    p = cpl_parameter_new_value("cheby_min", CPL_TYPE_INT, "", "", cheby_min);
    cpl_parameterlist_append(parlist, p);
    p = cpl_parameter_new_value("cheby_const", CPL_TYPE_DOUBLE, "", "",
                                cheby_const);
    cpl_parameterlist_append(parlist, p);
    p = cpl_parameter_new_value("rebintype", CPL_TYPE_INT, "", "", rebintype);
    cpl_parameterlist_append(parlist, p);
    p = cpl_parameter_new_value("weightlim", CPL_TYPE_DOUBLE, "", "",
                                weightlim);
    cpl_parameterlist_append(parlist, p);
    p = cpl_parameter_new_value("siglim", CPL_TYPE_DOUBLE, "", "", siglim);
    cpl_parameterlist_append(parlist, p);
    p = cpl_parameter_new_value("fitlim", CPL_TYPE_DOUBLE, "", "", fitlim);
    cpl_parameterlist_append(parlist, p);

    /* Plotting */
    p = cpl_parameter_new_value("plot_type", CPL_TYPE_STRING, "", "", "N");
    cpl_parameterlist_append(parlist, p);

    scispec =
        _moo_spectrum_to_table(rbn, indexrbn, crpix1, crval1, cd1_1, parlist);

    skyspec = _moo_spectrum_to_table(sky, 1, crpix1, crval1, cd1_1, parlist);

    moo_try_check(status = __mo_sc_readspec_header(parlist, header),
                  "error initialising SKYCORR");
    groups = cpl_table_new(0);
    _moo_sc_lines(groups, parlist, solflux_frame, airglow_group_frame,
                  airglow_var_frame);

    scilinetab = cpl_table_new(0);
    sc_specdiss_init_linetab(scilinetab);
    /* Subtraction of continuum from science and sky spectrum and estimation
    of line FWHM in sky spectrum */

    /* Prepare science spectrum */
    //cpl_msg_info(cpl_func, "Science spectrum:");
    sc_skycorr_subcont(scispec, scilinetab, parlist, groups);

    if ((status = cpl_error_get_code()) != CPL_ERROR_NONE) {
        sprintf(errtxt,
                "%s: error while separating lines and continuum of "
                "science spectrum",
                SC_ERROR_EIS_TXT);
        cpl_error_set_message(cpl_func, SC_ERROR_EIS, "%s", errtxt);
    }

    /* Prepare sky spectrum */
    skylinetab = cpl_table_new(0);
    sc_specdiss_init_linetab(skylinetab);

    sc_skycorr_subcont(skyspec, skylinetab, parlist, groups);

    if ((status = cpl_error_get_code()) != CPL_ERROR_NONE) {
        sprintf(errtxt,
                "%s: error while separating lines and continuum of "
                "sky spectrum",
                SC_ERROR_EIS_TXT);
        cpl_error_set_message(cpl_func, SC_ERROR_EIS, "%s", errtxt);
    }

    /* CMPFIT-based fitting procedure to adapt the sky line spectrum to the
       science line spectrum */
    //cpl_table_save(scispec,NULL,NULL,"moo_scipec.fits",CPL_IO_CREATE);
    //cpl_table_save(skyspec,NULL,NULL,"moo_skyspec.fits",CPL_IO_CREATE);
    sc_skycorr_fit(scispec, skyspec, groups, parlist);

    //cpl_table_save(skyspec,NULL,NULL,"moo_skyspec.fits",CPL_IO_CREATE);
    /* Perform sky subtraction */
    sc_skycorr_subsky(scispec);
//cpl_table_save(scispec,NULL,NULL,"moo_skyspec_subsky.fits",CPL_IO_CREATE);
#if MOO_DEBUG_SKYCORR
    {
        char *namesci = cpl_sprintf("SKYCORR_SCI_%d.fits", indextarg);
        cpl_table_save(scispec, header, NULL, namesci, CPL_IO_CREATE);
        cpl_free(namesci);

        char *namesky = cpl_sprintf("SKYCORR_SKY_%d.fits", indextarg);
        cpl_table_save(skyspec, header, NULL, namesky, CPL_IO_CREATE);
        cpl_free(namesky);
    }
#endif
    _moo_table_to_spectrum(scispec, sci, sci_sky, indextarg);

moo_try_cleanup:
    cpl_table_delete(skylinetab);
    cpl_table_delete(scilinetab);
    cpl_table_delete(scispec);
    cpl_table_delete(skyspec);
    cpl_table_delete(groups);
    cpl_parameterlist_delete(parlist);

    return status;
}
/**@}*/
