/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/
#include <math.h>
#include <cpl.h>
#include "moo_spectral_format.h"
/*----------------------------------------------------------------------------*/
/**
 * @defgroup moo_spectral_format SPECTRAL_FORMAT format
 * @ingroup moo_data
 * This module provides functions to create, use, and destroy a @em moo_spectral_format
 *
 * Functionality include:
 *
 * @par Synopsis:
 * @code
 *   #include "moo_spectral_format.h"
 * @endcode
 */


/*----------------------------------------------------------------------------*/

/**@{*/

/*-----------------------------------------------------------------------------
                              Function codes
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Create a new moo_spectral_format
  @return   1 newly allocated moo_spectral_format or NULL in case of an error

  The returned object must be deallocated using moo_ext_delete().

 */
/*----------------------------------------------------------------------------*/
moo_spectral_format *
moo_spectral_format_new(void)
{
    moo_spectral_format *res = cpl_calloc(1, sizeof(moo_spectral_format));
    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Create a new moo_spectral_format_info
  @return   1 newly allocated moo_spectral_format_info or NULL in case of an error

  The returned object must be deallocated using moo_ext_delete().

 */
/*----------------------------------------------------------------------------*/
moo_spectral_format_info *
moo_spectral_format_info_new(void)
{
    moo_spectral_format_info *res =
        cpl_calloc(1, sizeof(moo_spectral_format_info));
    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Load a SPECTRAL FORMAT frame and create a moo_spectral_format
  @param frame the SPECTRAL_FORMAT frame
  @param m the mode of instrument
  @return   1 newly allocated moo_spectral_format or NULL in case of an error

  The returned object must be deallocated using moo_ext_delete().
  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
moo_spectral_format *
moo_spectral_format_load(const cpl_frame *frame, moo_mode_type m)
{
    cpl_ensure(frame != NULL, CPL_ERROR_NULL_INPUT, NULL);
    const char *filename = cpl_frame_get_filename(frame);
    cpl_ensure(filename != NULL, CPL_ERROR_NULL_INPUT, NULL);

    cpl_errorstate prev_state = cpl_errorstate_get();
    moo_spectral_format *res = moo_spectral_format_new();

    cpl_table *full_table = cpl_table_load(filename, 1, 0);
    const char *mode = moo_mode_get_name(m);
    res->mode = m;
    cpl_table_select_all(full_table);
    cpl_table_and_selected_string(full_table, MODE_COLNAME, CPL_EQUAL_TO, mode);
    res->table = cpl_table_extract_selected(full_table);
    cpl_table_delete(full_table);

    if (!cpl_errorstate_is_equal(prev_state)) {
        cpl_errorstate_set(prev_state);
        cpl_msg_error("moo_spectral_format",
                      "Error in loading spectral_format %s", filename);
        moo_spectral_format_delete(res);
        res = NULL;
    }
    return res;
}

static moo_spectral_format_info *
_moo_spectral_format_get(moo_spectral_format *self, int nrow)
{
    moo_spectral_format_info *info = NULL;
    int flag;
    double wmin = cpl_table_get_double(self->table, WMIN_COLNAME, nrow, &flag);
    double wmax = cpl_table_get_double(self->table, WMAX_COLNAME, nrow, &flag);
    double resolution =
        cpl_table_get_double(self->table, RES_COLNAME, nrow, &flag);
    double dispmin =
        cpl_table_get_double(self->table, DISPMIN_COLNAME, nrow, &flag);
    double dispmax =
        cpl_table_get_double(self->table, DISPMAX_COLNAME, nrow, &flag);
    int dir = cpl_table_get_int(self->table, DIRECTION_COLNAME, nrow, &flag);
    info = moo_spectral_format_info_new();
    info->dispmin = dispmin;
    info->dispmax = dispmax;
    info->resolution = resolution;
    info->wmin = wmin;
    info->wmax = wmax;
    info->direction = dir;
    return info;
}
/*----------------------------------------------------------------------------*/
/**
  @brief Get the spectral format information for a given ARM
  @param self the SPECTRAL_FORMAT
  @param type the wanted detector type
  @param ntas the number of wanted detector
  @return   1 newly allocated moo_spectral_format_info or NULL in case of an error

 */
/*----------------------------------------------------------------------------*/
moo_spectral_format_info *
moo_spectral_format_get(moo_spectral_format *self,
                        moo_detector_type type,
                        int ntas)
{
    moo_spectral_format_info *info = NULL;

    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(ntas >= 1 && ntas <= 2, CPL_ERROR_ILLEGAL_INPUT, NULL);

    char *arm = cpl_sprintf("%s%d", moo_detector_get_name(type), ntas);
    cpl_table_select_all(self->table);
    cpl_table_and_selected_string(self->table, ARM_COLNAME, CPL_EQUAL_TO, arm);
    cpl_array *idx_array = cpl_table_where_selected(self->table);
    double idx = cpl_array_get(idx_array, 0, NULL);

    info = _moo_spectral_format_get(self, idx);

    cpl_array_delete(idx_array);

    cpl_free(arm);
    return info;
}

/*----------------------------------------------------------------------------*/
/**
  @brief Get the spectral format wave range for a given detector
  @param self the SPECTRAL_FORMAT
  @param type the wanted detector type
  @param min [out] the minimum of wavelength range
  @param max [out] the maximum of wavelength range
  @return error code

 */
/*----------------------------------------------------------------------------*/

cpl_error_code
moo_spectral_format_get_wave_range(moo_spectral_format *self,
                                   moo_detector_type type,
                                   double *min,
                                   double *max)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(min != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(max != NULL, CPL_ERROR_NULL_INPUT);

    moo_spectral_format_info *info = moo_spectral_format_get(self, type, 1);
    *min = info->wmin;
    *max = info->wmax;

    moo_spectral_format_info_delete(info);

    return cpl_error_get_code();
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Delete a moo_spectral_format
  @param    self    moo_spectral_format to delete
  @return   void

  If the moo_spectral_format @em self is @c NULL, nothing is done and no error is set.

 */
/*----------------------------------------------------------------------------*/
void
moo_spectral_format_delete(moo_spectral_format *self)
{
    if (self != NULL) {
        if (self->table != NULL) {
            cpl_table_delete(self->table);
        }
        cpl_free(self);
    }
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Delete a moo_spectral_format_info
  @param    self    moo_spectral_format_info to delete
  @return   void

  If the moo_spectral_format_info @em self is @c NULL, nothing is done and no error is set.

 */
/*----------------------------------------------------------------------------*/
void
moo_spectral_format_info_delete(moo_spectral_format_info *self)
{
    if (self != NULL) {
        cpl_free(self);
    }
}
/**@}*/
