/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/
#include <math.h>
#include <string.h>
#include <cpl.h>

#include "moo_target_table.h"
#include "moo_fibres_table.h"
#include "moo_rbn_single.h"
#include "moo_utils.h"
/*----------------------------------------------------------------------------*/
/**
 * @defgroup moo_target_table  TARGET table functions
 *
 */
/*----------------------------------------------------------------------------*/
/**@{*/

/*-----------------------------------------------------------------------------
                              Function codes
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Get the name of target mode
  @param m the target mode

  @return target mode name

 - - -
 _Error code_:
  - None
 */
/*----------------------------------------------------------------------------*/
const char *
moo_target_table_mode_get_name(moo_target_table_mode m)
{
    const char *res[] = { "STARE", "STARENOD", "XSWITCH" };
    int idx = m;
    return res[idx];
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Create a new moo_target_table
  @return   1 newly allocated moo_target_table or NULL in case of an error

  The returned object must be deallocated using moo_target_table_delete().

 */
/*----------------------------------------------------------------------------*/
moo_target_table *
moo_target_table_new(void)
{
    moo_target_table *res = cpl_calloc(1, sizeof(moo_target_table));

    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Duplicate an existing target_table
  @return   1 newly allocated moo_target_table or NULL in case of an error

  The returned object must be deallocated using moo_target_table_delete().

 */
/*----------------------------------------------------------------------------*/
moo_target_table *
moo_target_table_duplicate(moo_target_table *self)
{
    moo_target_table *res = NULL;
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);

    res = moo_target_table_new();
    res->primary_header = cpl_propertylist_duplicate(self->primary_header);
    res->table = cpl_table_duplicate(self->table);

    return res;
}

const char *
moo_target_table_get_mode_name(moo_target_table_mode mode)
{
    switch (mode) {
        case MOO_MODE_STARE:
            return "STARE";
        case MOO_MODE_STARENOD:
            return "STARENOD";
        case MOO_MODE_XSWITCH:
            return "XSWITCH";
        default:
            return "UNKKNOWN";
    }
}

moo_target_table *
moo_create_empty_target_table(void)
{
    moo_target_table *result = NULL;
    cpl_table *table = NULL;

    cpl_errorstate prestate = cpl_errorstate_get();
    result = moo_target_table_new();
    table = cpl_table_new(0);

    moo_try_check(cpl_table_new_column(table, MOO_TARGET_TABLE_INDEXTARG,
                                       CPL_TYPE_INT),
                  " ");
    moo_try_check(cpl_table_new_column(table, MOO_TARGET_TABLE_TARGNAME,
                                       CPL_TYPE_STRING),
                  " ");
    moo_try_check(cpl_table_new_column(table, MOO_TARGET_TABLE_TARGALPHA,
                                       CPL_TYPE_DOUBLE),
                  " ");
    moo_try_check(cpl_table_new_column(table, MOO_TARGET_TABLE_TARGDELTA,
                                       CPL_TYPE_DOUBLE),
                  " ");
    moo_try_check(cpl_table_new_column(table, MOO_TARGET_TABLE_TYPE,
                                       CPL_TYPE_STRING),
                  " ");
    moo_try_check(cpl_table_new_column(table, MOO_TARGET_TABLE_FIBRE,
                                       CPL_TYPE_STRING),
                  " ");
    moo_try_check(cpl_table_new_column(table, MOO_TARGET_TABLE_INDEX,
                                       CPL_TYPE_INT),
                  " ");
    cpl_table_new_column(table, MOO_TARGET_TABLE_PAIREDFIBRE, CPL_TYPE_INT);
    moo_try_check(cpl_table_new_column(table, MOO_TARGET_TABLE_MAG,
                                       CPL_TYPE_DOUBLE),
                  " ");
    moo_try_check(cpl_table_new_column(table, MOO_TARGET_TABLE_INDEXRBN,
                                       CPL_TYPE_INT),
                  " ");
    cpl_table_new_column(table, MOO_TARGET_TABLE_PAIREDINDEXRBN, CPL_TYPE_INT);
    cpl_table_new_column(table, MOO_TARGET_TABLE_EXPTIMERI, CPL_TYPE_DOUBLE);
    cpl_table_new_column(table, MOO_TARGET_TABLE_EXPTIMEYJ, CPL_TYPE_DOUBLE);
    cpl_table_new_column(table, MOO_TARGET_TABLE_EXPTIMEH, CPL_TYPE_DOUBLE);
    cpl_table_new_column(table, MOO_TARGET_TABLE_OBSTYPE, CPL_TYPE_STRING);
    cpl_table_new_column(table, MOO_TARGET_TABLE_SUBSKYMODE, CPL_TYPE_STRING);
    cpl_table_new_column(table, MOO_TARGET_TABLE_MEDSNRRI_SCI, CPL_TYPE_DOUBLE);
    cpl_table_new_column(table, MOO_TARGET_TABLE_MEDSNRYJ_SCI, CPL_TYPE_DOUBLE);
    cpl_table_new_column(table, MOO_TARGET_TABLE_MEDSNRH_SCI, CPL_TYPE_DOUBLE);
    cpl_table_new_column(table, MOO_TARGET_TABLE_TRANSRI, CPL_TYPE_FLOAT);
    cpl_table_new_column(table, MOO_TARGET_TABLE_TRANSYJ, CPL_TYPE_FLOAT);
    cpl_table_new_column(table, MOO_TARGET_TABLE_TRANSH, CPL_TYPE_FLOAT);
    cpl_table_new_column(table, MOO_TARGET_TABLE_NBSKY, CPL_TYPE_INT);
    cpl_table_new_column(table, MOO_TARGET_TABLE_MAX_DIST_SLIT, CPL_TYPE_INT);
    cpl_table_new_column(table, MOO_TARGET_TABLE_MAX_DIST_SKY, CPL_TYPE_FLOAT);
    cpl_table_new_column(table, MOO_TARGET_TABLE_MEDSNRRI_EXT, CPL_TYPE_FLOAT);
    cpl_table_new_column(table, MOO_TARGET_TABLE_MEDSNRYJ_EXT, CPL_TYPE_FLOAT);
    cpl_table_new_column(table, MOO_TARGET_TABLE_MEDSNRH_EXT, CPL_TYPE_FLOAT);
    cpl_table_new_column(table, MOO_TARGET_TABLE_DERSNRRI_EXT, CPL_TYPE_FLOAT);
    cpl_table_new_column(table, MOO_TARGET_TABLE_DERSNRYJ_EXT, CPL_TYPE_FLOAT);
    cpl_table_new_column(table, MOO_TARGET_TABLE_DERSNRH_EXT, CPL_TYPE_FLOAT);
    cpl_table_new_column(table, MOO_TARGET_TABLE_MEDSNRRI_RBN, CPL_TYPE_FLOAT);
    cpl_table_new_column(table, MOO_TARGET_TABLE_MEDSNRYJ_RBN, CPL_TYPE_FLOAT);
    cpl_table_new_column(table, MOO_TARGET_TABLE_MEDSNRH_RBN, CPL_TYPE_FLOAT);
    cpl_table_new_column(table, MOO_TARGET_TABLE_PTRANSRI, CPL_TYPE_FLOAT);
    cpl_table_new_column(table, MOO_TARGET_TABLE_PTRANSYJ, CPL_TYPE_FLOAT);
    cpl_table_new_column(table, MOO_TARGET_TABLE_PTRANSH, CPL_TYPE_FLOAT);
    cpl_table_new_column(table, MOO_TARGET_TABLE_PNBSKY, CPL_TYPE_INT);
    cpl_table_new_column(table, MOO_TARGET_TABLE_PMAX_DIST_SLIT, CPL_TYPE_INT);
    cpl_table_new_column(table, MOO_TARGET_TABLE_PMAX_DIST_SKY, CPL_TYPE_FLOAT);
    cpl_table_new_column(table, MOO_TARGET_TABLE_PMEDSNRRI_EXT, CPL_TYPE_FLOAT);
    cpl_table_new_column(table, MOO_TARGET_TABLE_PMEDSNRYJ_EXT, CPL_TYPE_FLOAT);
    cpl_table_new_column(table, MOO_TARGET_TABLE_PMEDSNRH_EXT, CPL_TYPE_FLOAT);
    cpl_table_new_column(table, MOO_TARGET_TABLE_PDERSNRRI_EXT, CPL_TYPE_FLOAT);
    cpl_table_new_column(table, MOO_TARGET_TABLE_PDERSNRYJ_EXT, CPL_TYPE_FLOAT);
    cpl_table_new_column(table, MOO_TARGET_TABLE_PDERSNRH_EXT, CPL_TYPE_FLOAT);
    cpl_table_new_column(table, MOO_TARGET_TABLE_PMEDSNRRI_RBN, CPL_TYPE_FLOAT);
    cpl_table_new_column(table, MOO_TARGET_TABLE_PMEDSNRYJ_RBN, CPL_TYPE_FLOAT);
    cpl_table_new_column(table, MOO_TARGET_TABLE_PMEDSNRH_RBN, CPL_TYPE_FLOAT);


    result->table = table;
    result->primary_header = cpl_propertylist_new();

moo_try_cleanup:
    if (!cpl_errorstate_is_equal(prestate)) {
        moo_target_table_delete(result);
        result = NULL;
    }
    return result;
}

static moo_target_table *
_moo_create_target_table(moo_rbn *rbn, moo_target_table_params *params)
{
    moo_target_table *result = NULL;
    cpl_table *table = NULL;
    cpl_ensure(rbn != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_table *fibre_table = moo_rbn_get_fibre_table(rbn);
    cpl_ensure(fibre_table, CPL_ERROR_NULL_INPUT, NULL);

    cpl_errorstate prestate = cpl_errorstate_get();
    int nrow = cpl_table_get_nrow(fibre_table);
    result = moo_target_table_new();
    table = cpl_table_new(nrow);
    moo_try_check(cpl_table_new_column(table, MOO_TARGET_TABLE_INDEXTARG,
                                       CPL_TYPE_INT),
                  " ");
    moo_try_check(cpl_table_duplicate_column(table, MOO_TARGET_TABLE_TARGNAME,
                                             fibre_table,
                                             MOO_FIBRES_TABLE_TARGNAME),
                  " ");
    moo_try_check(cpl_table_duplicate_column(table, MOO_TARGET_TABLE_TARGALPHA,
                                             fibre_table,
                                             MOO_FIBRES_TABLE_TARGALPHA),
                  " ");
    moo_try_check(cpl_table_duplicate_column(table, MOO_TARGET_TABLE_TARGDELTA,
                                             fibre_table,
                                             MOO_FIBRES_TABLE_TARGDELTA),
                  " ");
    moo_try_check(cpl_table_duplicate_column(table, MOO_TARGET_TABLE_TYPE,
                                             fibre_table,
                                             MOO_FIBRES_TABLE_TYPE),
                  " ");
    moo_try_check(cpl_table_duplicate_column(table, MOO_TARGET_TABLE_FIBRE,
                                             fibre_table,
                                             MOO_FIBRES_TABLE_FIBRE),
                  " ");
    moo_try_check(cpl_table_duplicate_column(table, MOO_TARGET_TABLE_SPECTRO,
                                             fibre_table,
                                             MOO_FIBRES_TABLE_SPECTRO),
                  " ");
    moo_try_check(cpl_table_duplicate_column(table, MOO_TARGET_TABLE_INDEX,
                                             fibre_table,
                                             MOO_FIBRES_TABLE_INDEX),
                  " ");
    moo_try_check(cpl_table_duplicate_column(table,
                                             MOO_TARGET_TABLE_PAIREDFIBRE,
                                             fibre_table,
                                             MOO_FIBRES_TABLE_PAIREDFIBRE),
                  " ");
    moo_try_check(cpl_table_duplicate_column(table, MOO_TARGET_TABLE_MAG,
                                             fibre_table, MOO_FIBRES_TABLE_MAG),
                  " ");
    moo_try_check(cpl_table_duplicate_column(table, MOO_TARGET_TABLE_INDEXRBN,
                                             fibre_table,
                                             MOO_FIBRES_TABLE_INDEXRBN),
                  " ");
    cpl_table_new_column(table, MOO_TARGET_TABLE_PAIREDINDEXRBN, CPL_TYPE_INT);
    cpl_table_new_column(table, MOO_TARGET_TABLE_EXPTIMERI, CPL_TYPE_DOUBLE);
    cpl_table_new_column(table, MOO_TARGET_TABLE_EXPTIMEYJ, CPL_TYPE_DOUBLE);
    cpl_table_new_column(table, MOO_TARGET_TABLE_EXPTIMEH, CPL_TYPE_DOUBLE);
    cpl_table_new_column(table, MOO_TARGET_TABLE_OBSTYPE, CPL_TYPE_STRING);
    cpl_table_new_column(table, MOO_TARGET_TABLE_SUBSKYMODE, CPL_TYPE_STRING);
    cpl_table_new_column(table, MOO_TARGET_TABLE_MEDSNRRI_SCI, CPL_TYPE_DOUBLE);
    cpl_table_new_column(table, MOO_TARGET_TABLE_MEDSNRYJ_SCI, CPL_TYPE_DOUBLE);
    cpl_table_new_column(table, MOO_TARGET_TABLE_MEDSNRH_SCI, CPL_TYPE_DOUBLE);
    moo_try_check(cpl_table_duplicate_column(table, MOO_TARGET_TABLE_TRANSRI,
                                             fibre_table,
                                             MOO_FIBRES_TABLE_TRANS_RI),
                  " ");
    moo_try_check(cpl_table_duplicate_column(table, MOO_TARGET_TABLE_TRANSYJ,
                                             fibre_table,
                                             MOO_FIBRES_TABLE_TRANS_YJ),
                  " ");
    moo_try_check(cpl_table_duplicate_column(table, MOO_TARGET_TABLE_TRANSH,
                                             fibre_table,
                                             MOO_FIBRES_TABLE_TRANS_H),
                  " ");
    cpl_table_new_column(table, MOO_TARGET_TABLE_NBSKY, CPL_TYPE_INT);
    cpl_table_new_column(table, MOO_TARGET_TABLE_MAX_DIST_SLIT, CPL_TYPE_INT);
    cpl_table_new_column(table, MOO_TARGET_TABLE_MAX_DIST_SKY, CPL_TYPE_FLOAT);
    moo_try_check(cpl_table_duplicate_column(table,
                                             MOO_TARGET_TABLE_MEDSNRRI_EXT,
                                             fibre_table,
                                             MOO_FIBRES_TABLE_MEDSNR_RI_EXT),
                  " ");
    moo_try_check(cpl_table_duplicate_column(table,
                                             MOO_TARGET_TABLE_MEDSNRYJ_EXT,
                                             fibre_table,
                                             MOO_FIBRES_TABLE_MEDSNR_YJ_EXT),
                  " ");
    moo_try_check(cpl_table_duplicate_column(table,
                                             MOO_TARGET_TABLE_MEDSNRH_EXT,
                                             fibre_table,
                                             MOO_FIBRES_TABLE_MEDSNR_H_EXT),
                  " ");
    moo_try_check(cpl_table_duplicate_column(table,
                                             MOO_TARGET_TABLE_DERSNRRI_EXT,
                                             fibre_table,
                                             MOO_FIBRES_TABLE_DERSNR_RI_EXT),
                  " ");
    moo_try_check(cpl_table_duplicate_column(table,
                                             MOO_TARGET_TABLE_DERSNRYJ_EXT,
                                             fibre_table,
                                             MOO_FIBRES_TABLE_DERSNR_YJ_EXT),
                  " ");
    moo_try_check(cpl_table_duplicate_column(table,
                                             MOO_TARGET_TABLE_DERSNRH_EXT,
                                             fibre_table,
                                             MOO_FIBRES_TABLE_DERSNR_H_EXT),
                  " ");
    moo_try_check(cpl_table_duplicate_column(table,
                                             MOO_TARGET_TABLE_MEDSNRRI_RBN,
                                             fibre_table,
                                             MOO_FIBRES_TABLE_MEDSNR_RI_RBN),
                  " ");
    moo_try_check(cpl_table_duplicate_column(table,
                                             MOO_TARGET_TABLE_MEDSNRYJ_RBN,
                                             fibre_table,
                                             MOO_FIBRES_TABLE_MEDSNR_YJ_RBN),
                  " ");
    moo_try_check(cpl_table_duplicate_column(table,
                                             MOO_TARGET_TABLE_MEDSNRH_RBN,
                                             fibre_table,
                                             MOO_FIBRES_TABLE_MEDSNR_H_RBN),
                  " ");

    const int *healtht =
        cpl_table_get_data_int_const(fibre_table, MOO_FIBRES_TABLE_HEALTH);
    const char **typet =
        cpl_table_get_data_string_const(fibre_table, MOO_FIBRES_TABLE_TYPE);
    const int *indexrbnt =
        cpl_table_get_data_int_const(fibre_table, MOO_FIBRES_TABLE_INDEXRBN);
    cpl_table_select_all(table);

    for (int i = 0; i < nrow; i++) {
        int health = healtht[i];
        const char *type = typet[i];
        int indexrbn = indexrbnt[i];

        if (health != 1 || indexrbn == 0) {
            cpl_table_unselect_row(table, i);
        }
        else if (strcmp(type, MOO_FIBRES_TABLE_TYPE_SKYCONTAM) == 0 ||
                 strcmp(type, MOO_FIBRES_TABLE_TYPE_BLANK) == 0) {
            cpl_table_unselect_row(table, i);
        }
        else if (params->nosky == 1 &&
                 strcmp(type, MOO_FIBRES_TABLE_TYPE_SKY) == 0) {
            cpl_table_unselect_row(table, i);
        }
        else {
            cpl_table_set_int(table, MOO_TARGET_TABLE_PAIREDINDEXRBN, i, -1);
            cpl_table_set_double(table, MOO_TARGET_TABLE_EXPTIMERI, i, NAN);
            cpl_table_set_double(table, MOO_TARGET_TABLE_EXPTIMEYJ, i, NAN);
            cpl_table_set_double(table, MOO_TARGET_TABLE_EXPTIMEH, i, NAN);
            cpl_table_set_double(table, MOO_TARGET_TABLE_MEDSNRRI_SCI, i, NAN);
            cpl_table_set_double(table, MOO_TARGET_TABLE_MEDSNRYJ_SCI, i, NAN);
            cpl_table_set_double(table, MOO_TARGET_TABLE_MEDSNRH_SCI, i, NAN);
            cpl_table_set_int(table, MOO_TARGET_TABLE_NBSKY, i, -1);
            cpl_table_set_int(table, MOO_TARGET_TABLE_MAX_DIST_SLIT, i, -1);
            cpl_table_set_float(table, MOO_TARGET_TABLE_MAX_DIST_SKY, i, NAN);
        }
    }
    cpl_table *good = cpl_table_extract_selected(table);
    result->table = good;
    result->primary_header = cpl_propertylist_new();

moo_try_cleanup:
    cpl_table_delete(table);
    if (!cpl_errorstate_is_equal(prestate)) {
        moo_target_table_delete(result);
        result = NULL;
    }
    return result;
}


static moo_target_table *
_moo_create_target_table_stare(moo_rbn *rbn, moo_target_table_params *params)
{
    moo_target_table *result = NULL;

    cpl_ensure(rbn != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_table *fibre_table = moo_rbn_get_fibre_table(rbn);
    cpl_ensure(fibre_table, CPL_ERROR_NULL_INPUT, NULL);

    cpl_errorstate prestate = cpl_errorstate_get();
    moo_try_check(result = _moo_create_target_table(rbn, params), " ");
    cpl_table *table = result->table;
    int nrow = cpl_table_get_nrow(table);
    char **obstype = cpl_table_get_data_string(table, MOO_TARGET_TABLE_OBSTYPE);
    const char **types =
        cpl_table_get_data_string_const(table, MOO_TARGET_TABLE_TYPE);

    for (int i = 0; i < nrow; i++) {
        const char *type = types[i];
        if (strcmp(type, MOO_FIBRES_TABLE_TYPE_SKY) == 0) {
            obstype[i] = cpl_strdup(MOO_OBSTYPE_SKYSTARE);
        }
        else {
            obstype[i] = cpl_strdup(MOO_OBSTYPE_OBSSTARE);
        }
        cpl_table_set_int(table, MOO_TARGET_TABLE_PAIREDFIBRE, i, -1);
        cpl_table_set_int(table, MOO_TARGET_TABLE_PAIREDINDEXRBN, i, -1);
    }

moo_try_cleanup:
    if (!cpl_errorstate_is_equal(prestate)) {
        moo_target_table_delete(result);
        result = NULL;
    }
    return result;
}

cpl_error_code
moo_target_table_set_snr(moo_target_table *self,
                         int idx,
                         cpl_table *fibre_table,
                         int ft_idx)
{
    cpl_table *target = self->table;
    float snrriext =
        cpl_table_get_float(fibre_table, MOO_FIBRES_TABLE_MEDSNR_RI_EXT, ft_idx,
                            NULL);
    float dsnrriext =
        cpl_table_get_float(fibre_table, MOO_FIBRES_TABLE_DERSNR_RI_EXT, ft_idx,
                            NULL);
    float snryjext =
        cpl_table_get_float(fibre_table, MOO_FIBRES_TABLE_MEDSNR_YJ_EXT, ft_idx,
                            NULL);
    float dsnryjext =
        cpl_table_get_float(fibre_table, MOO_FIBRES_TABLE_DERSNR_YJ_EXT, ft_idx,
                            NULL);
    float snrhext =
        cpl_table_get_float(fibre_table, MOO_FIBRES_TABLE_MEDSNR_H_EXT, ft_idx,
                            NULL);
    float dsnrhext =
        cpl_table_get_float(fibre_table, MOO_FIBRES_TABLE_DERSNR_H_EXT, ft_idx,
                            NULL);
    float snrrirbn =
        cpl_table_get_float(fibre_table, MOO_FIBRES_TABLE_MEDSNR_RI_RBN, ft_idx,
                            NULL);
    float snryjrbn =
        cpl_table_get_float(fibre_table, MOO_FIBRES_TABLE_MEDSNR_YJ_RBN, ft_idx,
                            NULL);
    float snrhrbn =
        cpl_table_get_float(fibre_table, MOO_FIBRES_TABLE_MEDSNR_H_RBN, ft_idx,
                            NULL);

    cpl_table_set_float(target, MOO_TARGET_TABLE_MEDSNRRI_EXT, idx, snrriext);
    cpl_table_set_float(target, MOO_TARGET_TABLE_DERSNRRI_EXT, idx, dsnrriext);
    cpl_table_set_float(target, MOO_TARGET_TABLE_MEDSNRYJ_EXT, idx, snryjext);
    cpl_table_set_float(target, MOO_TARGET_TABLE_DERSNRYJ_EXT, idx, dsnryjext);
    cpl_table_set_float(target, MOO_TARGET_TABLE_MEDSNRH_EXT, idx, snrhext);
    cpl_table_set_float(target, MOO_TARGET_TABLE_DERSNRH_EXT, idx, dsnrhext);

    cpl_table_set_float(target, MOO_TARGET_TABLE_MEDSNRRI_RBN, idx, snrrirbn);
    cpl_table_set_float(target, MOO_TARGET_TABLE_MEDSNRYJ_RBN, idx, snryjrbn);
    cpl_table_set_float(target, MOO_TARGET_TABLE_MEDSNRH_RBN, idx, snrhrbn);

    return CPL_ERROR_NONE;
}

cpl_error_code
moo_target_table_copy_row(moo_target_table *self,
                          int idx,
                          moo_target_table *tocopy,
                          int ft_idx)
{
    cpl_table *target = self->table;
    cpl_table *origin = tocopy->table;
    int nbsky = cpl_table_get_int(origin, MOO_TARGET_TABLE_NBSKY, ft_idx, NULL);
    int maxdistslit =
        cpl_table_get_int(origin, MOO_TARGET_TABLE_MAX_DIST_SLIT, ft_idx, NULL);
    float maxdistsky =
        cpl_table_get_float(origin, MOO_TARGET_TABLE_MAX_DIST_SKY, ft_idx,
                            NULL);
    int pnbsky =
        cpl_table_get_int(origin, MOO_TARGET_TABLE_PNBSKY, ft_idx, NULL);
    int pmaxdistslit =
        cpl_table_get_int(origin, MOO_TARGET_TABLE_PMAX_DIST_SLIT, ft_idx,
                          NULL);
    float pmaxdistsky =
        cpl_table_get_float(origin, MOO_TARGET_TABLE_PMAX_DIST_SKY, ft_idx,
                            NULL);
    float snrriext = cpl_table_get_float(origin, MOO_TARGET_TABLE_MEDSNRRI_EXT,
                                         ft_idx, NULL);
    float dsnrriext = cpl_table_get_float(origin, MOO_TARGET_TABLE_DERSNRRI_EXT,
                                          ft_idx, NULL);
    float snryjext = cpl_table_get_float(origin, MOO_TARGET_TABLE_MEDSNRYJ_EXT,
                                         ft_idx, NULL);
    float dsnryjext = cpl_table_get_float(origin, MOO_TARGET_TABLE_DERSNRYJ_EXT,
                                          ft_idx, NULL);
    float snrhext =
        cpl_table_get_float(origin, MOO_TARGET_TABLE_MEDSNRH_EXT, ft_idx, NULL);
    float dsnrhext =
        cpl_table_get_float(origin, MOO_TARGET_TABLE_DERSNRH_EXT, ft_idx, NULL);
    float snrrirbn = cpl_table_get_float(origin, MOO_TARGET_TABLE_MEDSNRRI_RBN,
                                         ft_idx, NULL);
    float snryjrbn = cpl_table_get_float(origin, MOO_TARGET_TABLE_MEDSNRYJ_RBN,
                                         ft_idx, NULL);
    float snrhrbn =
        cpl_table_get_float(origin, MOO_TARGET_TABLE_MEDSNRH_RBN, ft_idx, NULL);

    float psnrriext =
        cpl_table_get_float(origin, MOO_TARGET_TABLE_PMEDSNRRI_EXT, ft_idx,
                            NULL);
    float pdsnrriext =
        cpl_table_get_float(origin, MOO_TARGET_TABLE_PDERSNRRI_EXT, ft_idx,
                            NULL);
    float psnryjext =
        cpl_table_get_float(origin, MOO_TARGET_TABLE_PMEDSNRYJ_EXT, ft_idx,
                            NULL);
    float pdsnryjext =
        cpl_table_get_float(origin, MOO_TARGET_TABLE_PDERSNRYJ_EXT, ft_idx,
                            NULL);
    float psnrhext = cpl_table_get_float(origin, MOO_TARGET_TABLE_PMEDSNRH_EXT,
                                         ft_idx, NULL);
    float pdsnrhext = cpl_table_get_float(origin, MOO_TARGET_TABLE_PDERSNRH_EXT,
                                          ft_idx, NULL);
    float psnrrirbn =
        cpl_table_get_float(origin, MOO_TARGET_TABLE_PMEDSNRRI_RBN, ft_idx,
                            NULL);
    float psnryjrbn =
        cpl_table_get_float(origin, MOO_TARGET_TABLE_PMEDSNRYJ_RBN, ft_idx,
                            NULL);
    float psnrhrbn = cpl_table_get_float(origin, MOO_TARGET_TABLE_PMEDSNRH_RBN,
                                         ft_idx, NULL);

    cpl_table_set_float(target, MOO_TARGET_TABLE_MEDSNRRI_EXT, idx, snrriext);
    cpl_table_set_float(target, MOO_TARGET_TABLE_DERSNRRI_EXT, idx, dsnrriext);
    cpl_table_set_float(target, MOO_TARGET_TABLE_MEDSNRYJ_EXT, idx, snryjext);
    cpl_table_set_float(target, MOO_TARGET_TABLE_DERSNRYJ_EXT, idx, dsnryjext);
    cpl_table_set_float(target, MOO_TARGET_TABLE_MEDSNRH_EXT, idx, snrhext);
    cpl_table_set_float(target, MOO_TARGET_TABLE_DERSNRH_EXT, idx, dsnrhext);

    cpl_table_set_float(target, MOO_TARGET_TABLE_MEDSNRRI_RBN, idx, snrrirbn);
    cpl_table_set_float(target, MOO_TARGET_TABLE_MEDSNRYJ_RBN, idx, snryjrbn);
    cpl_table_set_float(target, MOO_TARGET_TABLE_MEDSNRH_RBN, idx, snrhrbn);

    cpl_table_set_float(target, MOO_TARGET_TABLE_PMEDSNRRI_EXT, idx, psnrriext);
    cpl_table_set_float(target, MOO_TARGET_TABLE_PDERSNRRI_EXT, idx,
                        pdsnrriext);
    cpl_table_set_float(target, MOO_TARGET_TABLE_PMEDSNRYJ_EXT, idx, psnryjext);
    cpl_table_set_float(target, MOO_TARGET_TABLE_PDERSNRYJ_EXT, idx,
                        pdsnryjext);
    cpl_table_set_float(target, MOO_TARGET_TABLE_PMEDSNRH_EXT, idx, psnrhext);
    cpl_table_set_float(target, MOO_TARGET_TABLE_PDERSNRH_EXT, idx, pdsnrhext);

    cpl_table_set_float(target, MOO_TARGET_TABLE_PMEDSNRRI_RBN, idx, psnrrirbn);
    cpl_table_set_float(target, MOO_TARGET_TABLE_PMEDSNRYJ_RBN, idx, psnryjrbn);
    cpl_table_set_float(target, MOO_TARGET_TABLE_PMEDSNRH_RBN, idx, psnrhrbn);

    cpl_table_set_int(target, MOO_TARGET_TABLE_NBSKY, idx, nbsky);
    cpl_table_set_int(target, MOO_TARGET_TABLE_MAX_DIST_SLIT, idx, maxdistslit);
    cpl_table_set_float(target, MOO_TARGET_TABLE_MAX_DIST_SKY, idx, maxdistsky);

    cpl_table_set_int(target, MOO_TARGET_TABLE_PNBSKY, idx, pnbsky);
    cpl_table_set_int(target, MOO_TARGET_TABLE_PMAX_DIST_SLIT, idx,
                      pmaxdistslit);
    cpl_table_set_float(target, MOO_TARGET_TABLE_PMAX_DIST_SKY, idx,
                        pmaxdistsky);
    return CPL_ERROR_NONE;
}

cpl_error_code
moo_target_table_set_ft_psnr(moo_target_table *self,
                             int idx,
                             cpl_table *fibre_table,
                             int ft_idx)
{
    cpl_table *target = self->table;
    float snrriext =
        cpl_table_get_float(fibre_table, MOO_FIBRES_TABLE_MEDSNR_RI_EXT, ft_idx,
                            NULL);
    float dsnrriext =
        cpl_table_get_float(fibre_table, MOO_FIBRES_TABLE_DERSNR_RI_EXT, ft_idx,
                            NULL);
    float snryjext =
        cpl_table_get_float(fibre_table, MOO_FIBRES_TABLE_MEDSNR_YJ_EXT, ft_idx,
                            NULL);
    float dsnryjext =
        cpl_table_get_float(fibre_table, MOO_FIBRES_TABLE_DERSNR_YJ_EXT, ft_idx,
                            NULL);
    float snrhext =
        cpl_table_get_float(fibre_table, MOO_FIBRES_TABLE_MEDSNR_H_EXT, ft_idx,
                            NULL);
    float dsnrhext =
        cpl_table_get_float(fibre_table, MOO_FIBRES_TABLE_DERSNR_H_EXT, ft_idx,
                            NULL);
    float snrrirbn =
        cpl_table_get_float(fibre_table, MOO_FIBRES_TABLE_MEDSNR_RI_RBN, ft_idx,
                            NULL);
    float snryjrbn =
        cpl_table_get_float(fibre_table, MOO_FIBRES_TABLE_MEDSNR_YJ_RBN, ft_idx,
                            NULL);
    float snrhrbn =
        cpl_table_get_float(fibre_table, MOO_FIBRES_TABLE_MEDSNR_H_RBN, ft_idx,
                            NULL);

    cpl_table_set_float(target, MOO_TARGET_TABLE_PMEDSNRRI_EXT, idx, snrriext);
    cpl_table_set_float(target, MOO_TARGET_TABLE_PDERSNRRI_EXT, idx, dsnrriext);
    cpl_table_set_float(target, MOO_TARGET_TABLE_PMEDSNRYJ_EXT, idx, snryjext);
    cpl_table_set_float(target, MOO_TARGET_TABLE_PDERSNRYJ_EXT, idx, dsnryjext);
    cpl_table_set_float(target, MOO_TARGET_TABLE_PMEDSNRH_EXT, idx, snrhext);
    cpl_table_set_float(target, MOO_TARGET_TABLE_PDERSNRH_EXT, idx, dsnrhext);
    cpl_table_set_float(target, MOO_TARGET_TABLE_PMEDSNRRI_RBN, idx, snrrirbn);
    cpl_table_set_float(target, MOO_TARGET_TABLE_PMEDSNRYJ_RBN, idx, snryjrbn);
    cpl_table_set_float(target, MOO_TARGET_TABLE_PMEDSNRH_RBN, idx, snrhrbn);

    return CPL_ERROR_NONE;
}

cpl_error_code
moo_target_table_set_psnr(moo_target_table *self,
                          int idx,
                          moo_target_table *torigin,
                          int ft_idx)
{
    cpl_table *target = self->table;
    cpl_table *origin = torigin->table;
    moo_target_table_set_ft_psnr(self, idx, origin, ft_idx);

    int pnbsky =
        cpl_table_get_int(origin, MOO_TARGET_TABLE_NBSKY, ft_idx, NULL);
    int pmaxdistslit =
        cpl_table_get_int(origin, MOO_TARGET_TABLE_MAX_DIST_SLIT, ft_idx, NULL);
    float pmaxdistsky =
        cpl_table_get_float(origin, MOO_TARGET_TABLE_MAX_DIST_SKY, ft_idx,
                            NULL);
    cpl_table_set_int(target, MOO_TARGET_TABLE_PNBSKY, idx, pnbsky);
    cpl_table_set_int(target, MOO_TARGET_TABLE_PMAX_DIST_SLIT, idx,
                      pmaxdistslit);
    cpl_table_set_float(target, MOO_TARGET_TABLE_PMAX_DIST_SKY, idx,
                        pmaxdistsky);

    return CPL_ERROR_NONE;
}

static cpl_error_code
_moo_target_table_add_from_ft(cpl_table *target,
                              cpl_table *fibre_table,
                              cpl_table *pfibre_table,
                              int ft_idx,
                              int nrow,
                              const char *obstype)
{
    cpl_table_set_size(target, nrow);

    const char *type =
        cpl_table_get_string(fibre_table, MOO_FIBRES_TABLE_TYPE, ft_idx);
    const char *targname =
        cpl_table_get_string(fibre_table, MOO_FIBRES_TABLE_TARGNAME, ft_idx);
    double targalpha =
        cpl_table_get_double(fibre_table, MOO_FIBRES_TABLE_TARGALPHA, ft_idx,
                             NULL);
    double targdelta =
        cpl_table_get_double(fibre_table, MOO_FIBRES_TABLE_TARGDELTA, ft_idx,
                             NULL);
    const char *fibre =
        cpl_table_get_string(fibre_table, MOO_FIBRES_TABLE_FIBRE, ft_idx);
    int index =
        cpl_table_get_int(fibre_table, MOO_FIBRES_TABLE_INDEX, ft_idx, NULL);
    int pfibre = cpl_table_get_int(fibre_table, MOO_FIBRES_TABLE_PAIREDFIBRE,
                                   ft_idx, NULL);
    int indexrbn =
        cpl_table_get_int(fibre_table, MOO_FIBRES_TABLE_INDEXRBN, ft_idx, NULL);
    int pindexrbn = -1;
    float ptransri = NAN;
    float ptransyj = NAN;
    float ptransh = NAN;
    if (pfibre >= 0) {
        int health = cpl_table_get_int(pfibre_table, MOO_FIBRES_TABLE_HEALTH,
                                       pfibre, NULL);
        if (health > 0) {
            pindexrbn =
                cpl_table_get_int(pfibre_table, MOO_FIBRES_TABLE_INDEXRBN,
                                  pfibre, NULL);
            ptransri =
                cpl_table_get_float(pfibre_table, MOO_FIBRES_TABLE_TRANS_RI,
                                    pfibre, NULL);
            ptransyj =
                cpl_table_get_float(pfibre_table, MOO_FIBRES_TABLE_TRANS_YJ,
                                    pfibre, NULL);
            ptransh =
                cpl_table_get_float(pfibre_table, MOO_FIBRES_TABLE_TRANS_H,
                                    pfibre, NULL);
            ptransri =
                cpl_table_get_float(pfibre_table, MOO_FIBRES_TABLE_TRANS_RI,
                                    pfibre, NULL);
        }
    }
    double mag =
        cpl_table_get_double(fibre_table, MOO_FIBRES_TABLE_MAG, ft_idx, NULL);
    float transri = cpl_table_get_float(fibre_table, MOO_FIBRES_TABLE_TRANS_RI,
                                        ft_idx, NULL);
    float transyj = cpl_table_get_float(fibre_table, MOO_FIBRES_TABLE_TRANS_YJ,
                                        ft_idx, NULL);
    float transh = cpl_table_get_float(fibre_table, MOO_FIBRES_TABLE_TRANS_H,
                                       ft_idx, NULL);

    cpl_table_set_int(target, MOO_TARGET_TABLE_INDEXTARG, nrow - 1, -1);
    cpl_table_set_int(target, MOO_TARGET_TABLE_NBSKY, nrow - 1, -1);
    cpl_table_set_int(target, MOO_TARGET_TABLE_MAX_DIST_SLIT, nrow - 1, -1);
    cpl_table_set_float(target, MOO_TARGET_TABLE_MAX_DIST_SKY, nrow - 1, NAN);
    cpl_table_set_int(target, MOO_TARGET_TABLE_PNBSKY, nrow - 1, -1);
    cpl_table_set_int(target, MOO_TARGET_TABLE_PMAX_DIST_SLIT, nrow - 1, -1);
    cpl_table_set_float(target, MOO_TARGET_TABLE_PMAX_DIST_SKY, nrow - 1, NAN);

    cpl_table_set_string(target, MOO_TARGET_TABLE_TARGNAME, nrow - 1, targname);
    cpl_table_set_double(target, MOO_TARGET_TABLE_TARGALPHA, nrow - 1,
                         targalpha);
    cpl_table_set_double(target, MOO_TARGET_TABLE_TARGDELTA, nrow - 1,
                         targdelta);
    cpl_table_set_double(target, MOO_TARGET_TABLE_MAG, nrow - 1, mag);
    cpl_table_set_string(target, MOO_TARGET_TABLE_FIBRE, nrow - 1, fibre);
    cpl_table_set_int(target, MOO_TARGET_TABLE_INDEX, nrow - 1, index);
    cpl_table_set_int(target, MOO_TARGET_TABLE_PAIREDFIBRE, nrow - 1, pfibre);
    cpl_table_set_string(target, MOO_TARGET_TABLE_TYPE, nrow - 1, type);
    cpl_table_set_string(target, MOO_TARGET_TABLE_OBSTYPE, nrow - 1, obstype);
    cpl_table_set_int(target, MOO_TARGET_TABLE_INDEXRBN, nrow - 1, indexrbn);
    cpl_table_set_int(target, MOO_TARGET_TABLE_PAIREDINDEXRBN, nrow - 1,
                      pindexrbn);
    cpl_table_set_float(target, MOO_TARGET_TABLE_TRANSRI, nrow - 1, transri);
    cpl_table_set_float(target, MOO_TARGET_TABLE_TRANSYJ, nrow - 1, transyj);
    cpl_table_set_float(target, MOO_TARGET_TABLE_TRANSH, nrow - 1, transh);
    cpl_table_set_float(target, MOO_TARGET_TABLE_PTRANSRI, nrow - 1, ptransri);
    cpl_table_set_float(target, MOO_TARGET_TABLE_PTRANSYJ, nrow - 1, ptransyj);
    cpl_table_set_float(target, MOO_TARGET_TABLE_PTRANSH, nrow - 1, ptransh);

    cpl_table_set_float(target, MOO_TARGET_TABLE_PMEDSNRRI_EXT, nrow - 1, NAN);
    cpl_table_set_float(target, MOO_TARGET_TABLE_PDERSNRRI_EXT, nrow - 1, NAN);
    cpl_table_set_float(target, MOO_TARGET_TABLE_PMEDSNRYJ_EXT, nrow - 1, NAN);
    cpl_table_set_float(target, MOO_TARGET_TABLE_PDERSNRYJ_EXT, nrow - 1, NAN);
    cpl_table_set_float(target, MOO_TARGET_TABLE_PMEDSNRH_EXT, nrow - 1, NAN);
    cpl_table_set_float(target, MOO_TARGET_TABLE_PDERSNRH_EXT, nrow - 1, NAN);
    cpl_table_set_float(target, MOO_TARGET_TABLE_PMEDSNRRI_RBN, nrow - 1, NAN);
    cpl_table_set_float(target, MOO_TARGET_TABLE_PMEDSNRYJ_RBN, nrow - 1, NAN);
    cpl_table_set_float(target, MOO_TARGET_TABLE_PMEDSNRH_RBN, nrow - 1, NAN);
    return CPL_ERROR_NONE;
}
static moo_target_table *
_moo_create_target_table_xswitch(moo_rbn *rbnA,
                                 moo_rbn *rbnB,
                                 moo_target_table_params *params)
{
    moo_target_table *result = NULL;

    cpl_ensure(rbnA != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(rbnB != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_table *fibre_tableA = moo_rbn_get_fibre_table(rbnA);
    cpl_table *fibre_tableB = moo_rbn_get_fibre_table(rbnB);

    cpl_ensure(fibre_tableA, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(fibre_tableB, CPL_ERROR_NULL_INPUT, NULL);

    moo_fibres_table_by_index(fibre_tableA);
    moo_fibres_table_by_index(fibre_tableB);

    cpl_errorstate prestate = cpl_errorstate_get();
    moo_try_check(result = moo_create_empty_target_table(), " ");
    cpl_table *table = result->table;

    int nrow = cpl_table_get_nrow(table);

    int *hAt = cpl_table_get_data_int(fibre_tableA, MOO_FIBRES_TABLE_HEALTH);
    int *indexrbnAt =
        cpl_table_get_data_int(fibre_tableA, MOO_FIBRES_TABLE_INDEXRBN);
    int *pfAt =
        cpl_table_get_data_int(fibre_tableA, MOO_FIBRES_TABLE_PAIREDFIBRE);
    char **typeAt =
        cpl_table_get_data_string(fibre_tableA, MOO_FIBRES_TABLE_TYPE);
    int frow = cpl_table_get_nrow(fibre_tableA);

    int *pfBt =
        cpl_table_get_data_int(fibre_tableB, MOO_FIBRES_TABLE_PAIREDFIBRE);
    char **typeBt =
        cpl_table_get_data_string(fibre_tableB, MOO_FIBRES_TABLE_TYPE);

    int *fdone = cpl_calloc(sizeof(int), frow);

    for (int i = 0; i < frow; i++) {
        const char *obstype = NULL;
        int hA = hAt[i];
        int indexrbnA = indexrbnAt[i];
        char *typeA = typeAt[i];
        char *typeB = typeBt[i];
        int pfA = pfAt[i];
        int pfB = pfBt[i];

        if (fdone[i] == 0 && hA != 0 && indexrbnA > 0) {
            if (strcmp(typeA, MOO_FIBRES_TABLE_TYPE_SCIENCE) == 0) {
                fdone[i] = 1;
                if (pfA >= 0 && hAt[pfA] != 0) {
                    fdone[pfA] = 1;
                    char *ptypeA = typeAt[pfA];
                    if (strcmp(ptypeA, MOO_FIBRES_TABLE_TYPE_SKY) == 0) {
                        if (strcmp(typeB, MOO_FIBRES_TABLE_TYPE_SKY) == 0) {
                            obstype = MOO_OBSTYPE_OBSSWITCH;
                        }
                        else {
                            obstype = MOO_OBSTYPE_OBSPNODSTARE;
                        }
                    }
                    else {
                        if (strcmp(typeB, MOO_FIBRES_TABLE_TYPE_SKY) == 0) {
                            obstype = MOO_OBSTYPE_OBSNODSTARE;
                        }
                        else {
                            obstype = MOO_OBSTYPE_OBSSWITCHSTARE;
                        }
                    }
                }
                else {
                    if (strcmp(typeB, MOO_FIBRES_TABLE_TYPE_SKY) == 0) {
                        obstype = MOO_OBSTYPE_OBSNOD;
                    }
                    else {
                        obstype = MOO_OBSTYPE_OBSSTARE;
                    }
                }
                nrow++;
                _moo_target_table_add_from_ft(table, fibre_tableA, fibre_tableB,
                                              i, nrow, obstype);
            }
            else if (strcmp(typeB, MOO_FIBRES_TABLE_TYPE_SCIENCE) == 0) {
                fdone[i] = 1;
                if (pfB < 0 || hAt[pfB] == 0) {
                    if (strcmp(typeA, MOO_FIBRES_TABLE_TYPE_SKY) == 0) {
                        obstype = MOO_OBSTYPE_OBSPNOD;
                    }
                    else {
                        obstype = MOO_OBSTYPE_OBSPSTARE;
                    }
                    nrow++;
                    _moo_target_table_add_from_ft(table, fibre_tableB,
                                                  fibre_tableA, i, nrow,
                                                  obstype);
                }
            }
            else if (strcmp(typeA, MOO_FIBRES_TABLE_TYPE_SKY) == 0 &&
                     strcmp(typeB, MOO_FIBRES_TABLE_TYPE_SKY) == 0 &&
                     params->nosky == 0) {
                obstype = MOO_OBSTYPE_SKYRESNOD;
                nrow++;
                _moo_target_table_add_from_ft(table, fibre_tableA, fibre_tableB,
                                              i, nrow, obstype);
            }
            else if (strcmp(typeA, MOO_FIBRES_TABLE_TYPE_SKY) == 0 &&
                     strcmp(typeB, MOO_FIBRES_TABLE_TYPE_SKYCONTAM) == 0 &&
                     params->nosky == 0) {
                obstype = MOO_OBSTYPE_SKYRESSTARE;
                nrow++;
                _moo_target_table_add_from_ft(table, fibre_tableA, fibre_tableB,
                                              i, nrow, obstype);
            }
            else if (strcmp(typeA, MOO_FIBRES_TABLE_TYPE_SKYCONTAM) == 0 &&
                     strcmp(typeB, MOO_FIBRES_TABLE_TYPE_SKY) == 0 &&
                     params->nosky == 0) {
                obstype = MOO_OBSTYPE_SKYRESPSTARE;
                nrow++;
                _moo_target_table_add_from_ft(table, fibre_tableB, fibre_tableA,
                                              i, nrow, obstype);
            }
        }
    }
    cpl_free(fdone);
moo_try_cleanup:
    if (!cpl_errorstate_is_equal(prestate)) {
        moo_target_table_delete(result);
        result = NULL;
    }
    return result;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Prepare target table for SCI frames using their attached fibre
  @param    rbnA the RBN object
  @param    rbnB the RBN (object B) or null
  @param    mode the TARGET_TABLE mode
  @param   params the target_table parameters
  @return   target_table or NULL

 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 *

 */
/*----------------------------------------------------------------------------*/
moo_target_table *
moo_create_target_table(moo_rbn *rbnA,
                        moo_rbn *rbnB,
                        moo_target_table_mode mode,
                        moo_target_table_params *params)
{
    cpl_ensure(rbnA != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(params != NULL, CPL_ERROR_NULL_INPUT, NULL);
    moo_target_table *result = NULL;

    cpl_msg_info(__func__, "Create target table for mode %s",
                 moo_target_table_get_mode_name(mode));

    if (mode == MOO_MODE_STARE) {
        result = _moo_create_target_table_stare(rbnA, params);
    }
    else if (mode == MOO_MODE_XSWITCH || mode == MOO_MODE_STARENOD) {
        result = _moo_create_target_table_xswitch(rbnA, rbnB, params);
    }

    return result;
}


cpl_error_code
moo_target_table_save(moo_target_table *self, const char *filename)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(filename != NULL, CPL_ERROR_NULL_INPUT);

    self->filename = filename;
    cpl_error_code status = cpl_table_save(self->table, self->primary_header,
                                           NULL, filename, CPL_IO_CREATE);

    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Delete a moo_target_table
  @param    self    moo_target_table to delete
  @return   void

  If the moo_target_table @em self is @c NULL, nothing is done and no error is set.

 */
/*----------------------------------------------------------------------------*/

void
moo_target_table_delete(moo_target_table *self)
{
    if (self != NULL) {
        if (self->table != NULL) {
            cpl_table_delete(self->table);
        }
        if (self->primary_header != NULL) {
            cpl_propertylist_delete(self->primary_header);
        }
        cpl_free(self);
    }
}

/*----------------------------------------------------------------------------*/
/**
  @brief    get the index of the sky in the target table sort by indexspec
  @param self the target table
  @param sky_rbn the _RBN_ sky 
  @return   the array index

 */
/*----------------------------------------------------------------------------*/

cpl_array *
moo_target_table_select_sky_index(moo_target_table *self, moo_rbn *sky_rbn)
{
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);

    cpl_table *table = self->table;
    cpl_propertylist *order = cpl_propertylist_new();
    cpl_propertylist_append_bool(order, MOO_TARGET_TABLE_INDEXRBN, CPL_FALSE);
    cpl_table_sort(table, order);
    cpl_propertylist_delete(order);

    if (sky_rbn == NULL) {
        cpl_table_unselect_all(table);
        cpl_table_or_selected_string(table, MOO_TARGET_TABLE_OBSTYPE,
                                     CPL_EQUAL_TO, MOO_OBSTYPE_SKYSTARE);
    }
    else {
        if (cpl_table_has_column(table, MOO_TARGET_TABLE_TYPENOD)) {
            cpl_table_select_all(table);
            cpl_table_and_selected_string(table, MOO_TARGET_TABLE_TYPENOD,
                                          CPL_NOT_EQUAL_TO,
                                          MOO_FIBRES_TABLE_TYPE_SKYCONTAM);
        }
        else {
            table = moo_rbn_get_fibre_table(sky_rbn);
            moo_fibres_table_by_indexrbn(table);
            cpl_table_unselect_all(table);
            cpl_table_or_selected_string(table, MOO_FIBRES_TABLE_TYPE,
                                         CPL_EQUAL_TO,
                                         MOO_FIBRES_TABLE_TYPE_SKY);
            cpl_table_and_selected_string(table, MOO_FIBRES_TABLE_TYPE,
                                          CPL_NOT_EQUAL_TO,
                                          MOO_FIBRES_TABLE_TYPE_SKYCONTAM);
            cpl_table_and_selected_int(table, MOO_FIBRES_TABLE_HEALTH,
                                       CPL_GREATER_THAN, 0);
            cpl_table_and_selected_int(table, MOO_FIBRES_TABLE_INDEXRBN,
                                       CPL_GREATER_THAN, 0);
        }
    }

    cpl_array *res = cpl_table_where_selected(table);

    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    select STARE object in trget table
  @param self the target table
  @return   the error_code or CPL_ERROR_NONE

 */
/*----------------------------------------------------------------------------*/

cpl_error_code
moo_target_table_select_stare(moo_target_table *self)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(self->table != NULL, CPL_ERROR_NULL_INPUT);

    cpl_table_unselect_all(self->table);
    cpl_table_or_selected_string(self->table, MOO_TARGET_TABLE_OBSTYPE,
                                 CPL_EQUAL_TO, MOO_OBSTYPE_OBSSTARE);
    cpl_table_or_selected_string(self->table, MOO_TARGET_TABLE_OBSTYPE,
                                 CPL_EQUAL_TO, MOO_OBSTYPE_SKYSTARE);
    cpl_table_or_selected_string(self->table, MOO_TARGET_TABLE_OBSTYPE,
                                 CPL_EQUAL_TO, MOO_OBSTYPE_SKYRESSTARE);
    cpl_table_or_selected_string(self->table, MOO_TARGET_TABLE_OBSTYPE,
                                 CPL_EQUAL_TO, MOO_OBSTYPE_OBSPNODSTARE);
    cpl_table_or_selected_string(self->table, MOO_TARGET_TABLE_OBSTYPE,
                                 CPL_EQUAL_TO, MOO_OBSTYPE_OBSSWITCHSTARE);
    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    select PSTARE object in trget table
  @param self the target table
  @return   the error_code or CPL_ERROR_NONE

 */
/*----------------------------------------------------------------------------*/

cpl_error_code
moo_target_table_select_pstare(moo_target_table *self)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(self->table != NULL, CPL_ERROR_NULL_INPUT);

    cpl_table_unselect_all(self->table);
    cpl_table_or_selected_string(self->table, MOO_TARGET_TABLE_OBSTYPE,
                                 CPL_EQUAL_TO, MOO_OBSTYPE_OBSPSTARE);
    cpl_table_or_selected_string(self->table, MOO_TARGET_TABLE_OBSTYPE,
                                 CPL_EQUAL_TO, MOO_OBSTYPE_SKYRESPSTARE);
    cpl_table_or_selected_string(self->table, MOO_TARGET_TABLE_OBSTYPE,
                                 CPL_EQUAL_TO, MOO_OBSTYPE_OBSNODSTARE);
    cpl_table_or_selected_string(self->table, MOO_TARGET_TABLE_OBSTYPE,
                                 CPL_EQUAL_TO, MOO_OBSTYPE_OBSSWITCHSTARE);
    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    select STARE object in trget table
  @param self the target table
  @return   the error_code or CPL_ERROR_NONE

 */
/*----------------------------------------------------------------------------*/

cpl_error_code
moo_target_table_select_nod(moo_target_table *self)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(self->table != NULL, CPL_ERROR_NULL_INPUT);

    cpl_table_unselect_all(self->table);
    cpl_table_or_selected_string(self->table, MOO_TARGET_TABLE_OBSTYPE,
                                 CPL_EQUAL_TO, MOO_OBSTYPE_OBSNOD);
    cpl_table_or_selected_string(self->table, MOO_TARGET_TABLE_OBSTYPE,
                                 CPL_EQUAL_TO, MOO_OBSTYPE_OBSSWITCH);
    cpl_table_or_selected_string(self->table, MOO_TARGET_TABLE_OBSTYPE,
                                 CPL_EQUAL_TO, MOO_OBSTYPE_SKYRESNOD);
    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    select STARE object in trget table
  @param self the target table
  @return   the error_code or CPL_ERROR_NONE

 */
/*----------------------------------------------------------------------------*/

cpl_error_code
moo_target_table_select_pnod(moo_target_table *self)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(self->table != NULL, CPL_ERROR_NULL_INPUT);

    cpl_table_unselect_all(self->table);
    cpl_table_or_selected_string(self->table, MOO_TARGET_TABLE_OBSTYPE,
                                 CPL_EQUAL_TO, MOO_OBSTYPE_OBSPNOD);
    cpl_table_or_selected_string(self->table, MOO_TARGET_TABLE_OBSTYPE,
                                 CPL_EQUAL_TO, MOO_OBSTYPE_OBSSWITCH);
    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    get the index of the object in the target table sort by indexrbn
  @param self the target table
  @return   the array index

 */
/*----------------------------------------------------------------------------*/

int
moo_target_table_xswitch_paired(moo_target_table *self)
{
    cpl_error_code status = CPL_ERROR_NONE;
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);

    const char *transname[] = { MOO_TARGET_TABLE_TRANSRI,
                                MOO_TARGET_TABLE_TRANSYJ,
                                MOO_TARGET_TABLE_TRANSH };
    const char *ptransname[] = { MOO_TARGET_TABLE_PTRANSRI,
                                 MOO_TARGET_TABLE_PTRANSYJ,
                                 MOO_TARGET_TABLE_PTRANSH };

    cpl_table *table = self->table;
    int nrow = cpl_table_get_nrow(table);
    for (int i = 0; i < nrow; i++) {
        int indexrbn =
            cpl_table_get_int(table, MOO_TARGET_TABLE_INDEXRBN, i, NULL);
        int pindexrbn =
            cpl_table_get_int(table, MOO_TARGET_TABLE_PAIREDINDEXRBN, i, NULL);
        int index = cpl_table_get_int(table, MOO_TARGET_TABLE_INDEX, i, NULL);
        int pindex =
            cpl_table_get_int(table, MOO_TARGET_TABLE_PAIREDFIBRE, i, NULL);

        if (pindex != -1) {
            cpl_table_set_int(table, MOO_TARGET_TABLE_INDEX, i, pindex);
            cpl_table_set_int(table, MOO_TARGET_TABLE_PAIREDFIBRE, i, index);
        }
        if (pindexrbn != -1) {
            cpl_table_set_int(table, MOO_TARGET_TABLE_INDEXRBN, i, pindexrbn);
            cpl_table_set_int(table, MOO_TARGET_TABLE_PAIREDINDEXRBN, i,
                              indexrbn);
        }
        for (int j = 0; j < 3; j++) {
            float trans = cpl_table_get_float(table, transname[j], i, NULL);
            float ptrans = cpl_table_get_float(table, ptransname[j], i, NULL);
            cpl_table_set_float(table, transname[j], i, ptrans);
            cpl_table_set_float(table, ptransname[j], i, trans);
        }
    }

    return status;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    get the index of the object in the target table sort by indexrbn
  @param self the target table
  @param ispaired if TRUE select the index of fibres in paired mode
  @return   the array index

 */
/*----------------------------------------------------------------------------*/

cpl_array *
moo_target_table_select_obj_index(moo_target_table *self, int ispaired)
{
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_propertylist *order = cpl_propertylist_new();
    cpl_propertylist_append_bool(order, MOO_TARGET_TABLE_INDEXRBN, CPL_FALSE);
    cpl_table_sort(self->table, order);
    cpl_propertylist_delete(order);

    cpl_table_unselect_all(self->table);
    if (ispaired) {
        cpl_table_or_selected_string(self->table, MOO_TARGET_TABLE_OBSTYPE,
                                     CPL_EQUAL_TO, MOO_OBSTYPE_OBSSWITCHSTARE);
        cpl_table_or_selected_string(self->table, MOO_TARGET_TABLE_OBSTYPE,
                                     CPL_EQUAL_TO, MOO_OBSTYPE_OBSNODSTARE);
        cpl_table_or_selected_string(self->table, MOO_TARGET_TABLE_OBSTYPE,
                                     CPL_EQUAL_TO, MOO_OBSTYPE_SKYRESPSTARE);
    }
    else {
        cpl_table_or_selected_string(self->table, MOO_TARGET_TABLE_OBSTYPE,
                                     CPL_EQUAL_TO, MOO_OBSTYPE_OBSSTARE);
        cpl_table_or_selected_string(self->table, MOO_TARGET_TABLE_OBSTYPE,
                                     CPL_EQUAL_TO, MOO_OBSTYPE_SKYSTARE);
        cpl_table_or_selected_string(self->table, MOO_TARGET_TABLE_OBSTYPE,
                                     CPL_EQUAL_TO, MOO_OBSTYPE_SKYRESSTARE);
        cpl_table_or_selected_string(self->table, MOO_TARGET_TABLE_OBSTYPE,
                                     CPL_EQUAL_TO, MOO_OBSTYPE_OBSSWITCHSTARE);
        cpl_table_or_selected_string(self->table, MOO_TARGET_TABLE_OBSTYPE,
                                     CPL_EQUAL_TO, MOO_OBSTYPE_OBSPNODSTARE);
        cpl_table_or_selected_string(self->table, MOO_TARGET_TABLE_OBSTYPE,
                                     CPL_EQUAL_TO, MOO_OBSTYPE_OBSPSTARE);
    }

    cpl_array *res = cpl_table_where_selected(self->table);

    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Load a TARGET_TABLE frame and create a moo_target_table
  @param frame the TARGET_TABLE frame
  @return   1 newly allocated moo_spectral_format or NULL in case of an error

  The returned object must be deallocated using moo_target_table_delete().
 *
  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
moo_target_table *
moo_target_table_load(const cpl_frame *frame)
{
    cpl_ensure(frame != NULL, CPL_ERROR_NULL_INPUT, NULL);
    const char *filename = cpl_frame_get_filename(frame);
    cpl_ensure(filename != NULL, CPL_ERROR_NULL_INPUT, NULL);

    cpl_errorstate prev_state = cpl_errorstate_get();
    moo_target_table *res = moo_target_table_new();

    res->filename = filename;
    res->table = cpl_table_load(filename, 1, 0);
    res->primary_header = cpl_propertylist_load(filename, 0);

    if (!cpl_errorstate_is_equal(prev_state)) {
        cpl_msg_error("moo_target_table", "can't load TARGET_TABLE %s",
                      filename);
        moo_target_table_delete(res);
        res = NULL;
        cpl_errorstate_set(prev_state);
    }
    return res;
}

static void
_moo_merge_row(cpl_table *a, cpl_size rowa, cpl_table *b, cpl_size rowb)
{
    int indexspecA =
        cpl_table_get_int(a, MOO_TARGET_TABLE_INDEXRBN, rowa, NULL);
    int indexspecB =
        cpl_table_get_int(b, MOO_TARGET_TABLE_INDEXRBN, rowb, NULL);

    if (indexspecA != indexspecB) {
        cpl_table_set_invalid(a, MOO_TARGET_TABLE_INDEXRBN, rowa);
        cpl_table_set_int(a, MOO_TARGET_TABLE_INDEXRBN, rowa, 0);
    }

    int pindexspecA =
        cpl_table_get_int(a, MOO_TARGET_TABLE_PAIREDINDEXRBN, rowa, NULL);
    int pindexspecB =
        cpl_table_get_int(b, MOO_TARGET_TABLE_PAIREDINDEXRBN, rowb, NULL);

    if (pindexspecA != pindexspecB) {
        cpl_table_set_invalid(a, MOO_TARGET_TABLE_PAIREDINDEXRBN, rowa);
        cpl_table_set_int(a, MOO_TARGET_TABLE_PAIREDINDEXRBN, rowa, 0);
    }
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Merge two TARGET_TABLE
  @param self the merge result TARGET_TABLE
  @param tomerge the TARGET_TABLE to merge
  @return   error code or CPL_ERROR_NONE


  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_target_table_merge(moo_target_table *self, moo_target_table *tomerge)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(tomerge != NULL, CPL_ERROR_NULL_INPUT);

    cpl_table *res_table = self->table;

    cpl_table *mtable = tomerge->table;
    const char **mtargnames =
        cpl_table_get_data_string_const(mtable, MOO_TARGET_TABLE_TARGNAME);

    int nrow = cpl_table_get_nrow(mtable);


    for (int j = 0; j < nrow; j++) {
        const char *mtargname = mtargnames[j];

        cpl_table_unselect_all(res_table);
        int nb = moo_table_or_selected_sequal_string(res_table,
                                                     MOO_TARGET_TABLE_TARGNAME,
                                                     mtargname);
        if (nb == 0) {
            /* add it */
            int rrow = cpl_table_get_nrow(res_table);
            cpl_table_unselect_all(mtable);
            cpl_table_or_selected_window(mtable, j, 1);
            cpl_table *sel = cpl_table_extract_selected(mtable);
            cpl_table_set_int(sel, MOO_TARGET_TABLE_INDEXTARG, 0, rrow + 1);
            cpl_table_insert(res_table, sel, rrow);
            cpl_table_delete(sel);
        }
        else if (nb == 1) {
            /* merge it */
            cpl_array *indexes = cpl_table_where_selected(res_table);
            int idx = cpl_array_get_cplsize(indexes, 0, NULL);
            _moo_merge_row(res_table, idx, mtable, j);
            cpl_array_delete(indexes);
        }
        else {
            cpl_error_set_message(
                "moo_target_table", CPL_ERROR_ILLEGAL_INPUT,
                "Error in merging %s : multiple objects (%d) found", mtargname,
                nb);
            return CPL_ERROR_ILLEGAL_INPUT;
        }
    }
    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
  @brief find target targindex in TARGET_TABLE
  @param self the TARGET_TABLE
  @param targname the name of TARGET_TABLE to find
  @return the indextarg or -1


  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  _ CPL_ERROR_ILLEGAL_INPUT if table is NULL
 */
/*----------------------------------------------------------------------------*/
int
moo_target_table_find_target(moo_target_table *self, const char *targname)
{
    int index = -1;
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, index);
    cpl_ensure(targname != NULL, CPL_ERROR_NULL_INPUT, index);
    cpl_ensure(self->table != NULL, CPL_ERROR_ILLEGAL_INPUT, index);

    cpl_table_unselect_all(self->table);
    int nb = moo_table_or_selected_sequal_string(self->table,
                                                 MOO_TARGET_TABLE_TARGNAME,
                                                 targname);

    const int *indextarg =
        cpl_table_get_data_int_const(self->table, MOO_TARGET_TABLE_INDEXTARG);

    if (nb == 1) {
        cpl_array *indexes = cpl_table_where_selected(self->table);
        int idx = cpl_array_get_cplsize(indexes, 0, NULL);
        index = indextarg[idx];
        cpl_array_delete(indexes);
    }
    return index;
}


/*----------------------------------------------------------------------------*/
/**
  @brief remove columns for coadd in target_table
  @param self the TARGET_TABLE
  @return the error code
*/
/*----------------------------------------------------------------------------*/
cpl_error_code
moo_target_table_remove_coadd_cols(moo_target_table *self)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
    cpl_table *table = self->table;
    if (table != NULL) {
        cpl_errorstate prev_state = cpl_errorstate_get();
        cpl_table_erase_column(table, MOO_TARGET_TABLE_OBSTYPE);
        cpl_table_erase_column(table, MOO_TARGET_TABLE_MEDSNRRI_EXT);
        cpl_table_erase_column(table, MOO_TARGET_TABLE_MEDSNRYJ_EXT);
        cpl_table_erase_column(table, MOO_TARGET_TABLE_MEDSNRH_EXT);

        cpl_table_erase_column(table, MOO_TARGET_TABLE_DERSNRRI_EXT);
        cpl_table_erase_column(table, MOO_TARGET_TABLE_DERSNRYJ_EXT);
        cpl_table_erase_column(table, MOO_TARGET_TABLE_DERSNRH_EXT);
        cpl_table_erase_column(table, MOO_TARGET_TABLE_MEDSNRRI_RBN);
        cpl_table_erase_column(table, MOO_TARGET_TABLE_MEDSNRYJ_RBN);
        cpl_table_erase_column(table, MOO_TARGET_TABLE_MEDSNRH_RBN);
        cpl_table_erase_column(table, MOO_TARGET_TABLE_PMEDSNRRI_EXT);
        cpl_table_erase_column(table, MOO_TARGET_TABLE_PMEDSNRYJ_EXT);
        cpl_table_erase_column(table, MOO_TARGET_TABLE_PMEDSNRH_EXT);
        cpl_table_erase_column(table, MOO_TARGET_TABLE_PDERSNRRI_EXT);
        cpl_table_erase_column(table, MOO_TARGET_TABLE_PDERSNRYJ_EXT);
        cpl_table_erase_column(table, MOO_TARGET_TABLE_PDERSNRH_EXT);

        cpl_table_erase_column(table, MOO_TARGET_TABLE_PMEDSNRRI_RBN);
        cpl_table_erase_column(table, MOO_TARGET_TABLE_PMEDSNRYJ_RBN);
        cpl_table_erase_column(table, MOO_TARGET_TABLE_PMEDSNRH_RBN);

        cpl_errorstate_set(prev_state);
        int nrow = cpl_table_get_nrow(table);
        for (int i = 0; i < nrow; i++) {
            cpl_table_set(table, MOO_TARGET_TABLE_MEDSNRRI_SCI, i, NAN);
            cpl_table_set(table, MOO_TARGET_TABLE_MEDSNRYJ_SCI, i, NAN);
            cpl_table_set(table, MOO_TARGET_TABLE_MEDSNRH_SCI, i, NAN);
        }
    }
    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
  @brief find target indexspec in TARGET_TABLE
  @param self the TARGET_TABLE
  @param targname the name of TARGET_TABLE to find
  @return the indexspec or -1


  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  _ CPL_ERROR_ILLEGAL_INPUT if table is NULL
 */
/*----------------------------------------------------------------------------*/
int
moo_target_table_find_index(moo_target_table *self, const char *targname)
{
    int index = -1;
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, index);
    cpl_ensure(targname != NULL, CPL_ERROR_NULL_INPUT, index);
    cpl_ensure(self->table != NULL, CPL_ERROR_ILLEGAL_INPUT, index);

    cpl_table_unselect_all(self->table);
    int nb = moo_table_or_selected_sequal_string(self->table,
                                                 MOO_TARGET_TABLE_TARGNAME,
                                                 targname);


    if (nb == 1) {
        cpl_array *indexes = cpl_table_where_selected(self->table);
        int idx = cpl_array_get_cplsize(indexes, 0, NULL);
        index = idx;
        cpl_array_delete(indexes);
    }
    return index;
}

const char *
moo_target_table_get_std_name(moo_target_table *self)
{
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);

    return moo_fibres_table_get_std_name(self->table);
}

cpl_error_code
moo_target_table_set_exptimes(moo_target_table *self,
                              double *exptimeA,
                              double *exptimeB)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(exptimeA != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(exptimeB != NULL, CPL_ERROR_NULL_INPUT);

    int nrow = cpl_table_get_nrow(self->table);
    char **obstypet =
        cpl_table_get_data_string(self->table, MOO_TARGET_TABLE_OBSTYPE);

    for (int i = 0; i < nrow; i++) {
        const char *obstype = obstypet[i];
        for (int j = 0; j < 3; j++) {
            double exptime = exptimeA[j];

            if (strcmp(obstype, MOO_OBSTYPE_OBSSWITCH) == 0) {
                exptime += exptimeB[j];
            }
            moo_target_table_set_exptime(self, i, j, exptime);
        }
    }
    return CPL_ERROR_NONE;
}

cpl_error_code
moo_target_table_set_exptime(moo_target_table *self,
                             int i,
                             int j,
                             double exptime)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);

    const char *exptimecols[] = { MOO_TARGET_TABLE_EXPTIMERI,
                                  MOO_TARGET_TABLE_EXPTIMEYJ,
                                  MOO_TARGET_TABLE_EXPTIMEH };

    cpl_table_set_double(self->table, exptimecols[j], i, exptime);

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
  @brief rget the exptime of a target
  @param self the TARGET_TABLE
  @param i the index of the TARGET_TABLE
  @param j the band
  @return the error code
*/
/*----------------------------------------------------------------------------*/

double
moo_target_table_get_exptime(moo_target_table *self, int i, int j)
{
    double res = 0.0;

    const char *exptimecols[] = { MOO_TARGET_TABLE_EXPTIMERI,
                                  MOO_TARGET_TABLE_EXPTIMEYJ,
                                  MOO_TARGET_TABLE_EXPTIMEH };

    res = cpl_table_get_double(self->table, exptimecols[j], i, NULL);

    return res;
}

int
moo_is_obstype_stare(const char *obstype)
{
    return (strcmp(obstype, MOO_OBSTYPE_SKYSTARE) == 0 ||
            strcmp(obstype, MOO_OBSTYPE_OBSSTARE) == 0 ||
            strcmp(obstype, MOO_OBSTYPE_OBSPSTARE) == 0 ||
            strcmp(obstype, MOO_OBSTYPE_OBSNODSTARE) == 0 ||
            strcmp(obstype, MOO_OBSTYPE_OBSPNODSTARE) == 0 ||
            strcmp(obstype, MOO_OBSTYPE_OBSSWITCHSTARE) == 0 ||
            strcmp(obstype, MOO_OBSTYPE_SKYRESSTARE) == 0 ||
            strcmp(obstype, MOO_OBSTYPE_SKYRESPSTARE) == 0);
}

/**@}*/
