/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/
#include <math.h>
#include <string.h>
#include <cpl.h>

#include "moo_telluric.h"
#include "moo_utils.h"
#include "moo_pfits.h"
#include "moo_fits.h"
#include "moo_scilist.h"
#include "moo_sci.h"
/*----------------------------------------------------------------------------*/
/**
 * @defgroup moo_telluric TELLURIC format
 * @ingroup moo_data
 * This module provides functions to create, use, and destroy a @em moo_telluric
 *
 * Functionality include:
 *
 * @par Synopsis:
 * @code
 *   #include "moo_telluric.h"
 * @endcode
 */
/*----------------------------------------------------------------------------*/
/**@{*/

/*-----------------------------------------------------------------------------
                              Function codes
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Create a new moo_telluric
  @return   1 newly allocated moo_telluric or NULL in case of an error

  The returned object must be deallocated using moo_telluric_delete().

 */
/*----------------------------------------------------------------------------*/
moo_telluric *
moo_telluric_new(void)
{
    moo_telluric *res = cpl_calloc(1, sizeof(moo_telluric));
    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief assign image in moo_telluric
  @param self moo_telluric
  @param type the detector type
  @param im the image
  @return the relevant error code or CPl_ERROR_NONE

Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/

cpl_error_code
moo_telluric_set_image(moo_telluric *self,
                       moo_detector_type type,
                       cpl_image *im)
{
    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);

    cpl_error_code status = CPL_ERROR_NONE;

    if (type >= 0 && type < 3) {
        self->data[type] = im;
    }
    else {
        status = CPL_ERROR_ILLEGAL_INPUT;
    }
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Get a telluric image from TELLURIC
  @param self moo_telluric structure
  @param type type of detector
  @return   cpl table

  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
cpl_image *
moo_telluric_get_image(moo_telluric *self, moo_detector_type type)
{
    cpl_image *result = NULL;
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    if (type >= 0 && type < 3) {
        result = self->data[type];
    }
    return result;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Prepare TELLURIC for SCI frame
  @param    scilist the SCI list from where we computed the rsponse
  @param    badpix_level the bad pixel level
  @return   create table for telluric

 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 *

 */
/*----------------------------------------------------------------------------*/
moo_telluric *
moo_telluric_create(moo_scilist *scilist, int badpix_level)
{
    moo_telluric *result = NULL;

    cpl_ensure(scilist != NULL, CPL_ERROR_NULL_INPUT, NULL);

    int size = moo_scilist_get_size(scilist);

    result = moo_telluric_new();
    result->primary_header = cpl_propertylist_new();
    result->telluric_table = cpl_table_new(size * 3);

    cpl_table_new_column(result->telluric_table, MOO_TELLURIC_TABLE_INDEXRBN,
                         CPL_TYPE_INT);
    cpl_table_new_column(result->telluric_table, MOO_TELLURIC_TABLE_SPECTRO,
                         CPL_TYPE_INT);
    cpl_table_new_column(result->telluric_table, MOO_TELLURIC_TABLE_BAND,
                         CPL_TYPE_STRING);
    cpl_table_new_column(result->telluric_table, MOO_TELLURIC_TABLE_TELLURIC,
                         CPL_TYPE_INT);

    const char *bandcolnames[] = { MOO_TELLURIC_EXT_RI, MOO_TELLURIC_EXT_YJ,
                                   MOO_TELLURIC_EXT_H };
    for (int il = 0; il < size; il++) {
        cpl_array *sel = NULL;
        moo_sci *sci = moo_scilist_get(scilist, il);
        moo_target_table *target_table = moo_sci_get_target_table(sci);
        if (target_table->table != NULL) {
            const char *targname = moo_target_table_get_std_name(target_table);
            int idtarget =
                moo_target_table_find_target(sci->target_table, targname);
            cpl_table_select_all(sci->target_table->table);
            cpl_table_and_selected_int(sci->target_table->table,
                                       MOO_TARGET_TABLE_INDEXTARG, CPL_EQUAL_TO,
                                       idtarget);
            sel = cpl_table_where_selected(sci->target_table->table);
            int idx = cpl_array_get_cplsize(sel, 0, NULL);
            int idrbn = cpl_table_get_int(sci->target_table->table,
                                          MOO_TARGET_TABLE_INDEXRBN, idx, NULL);
            const int spectro =
                cpl_table_get_int(sci->target_table->table,
                                  MOO_TARGET_TABLE_SPECTRO, idx, NULL);
            for (int type = 0; type < 3; type++) {
                const char *band = bandcolnames[type];

                cpl_table_set_int(result->telluric_table,
                                  MOO_TELLURIC_TABLE_INDEXRBN, il * 3 + type,
                                  idrbn);
                cpl_table_set_int(result->telluric_table,
                                  MOO_TELLURIC_TABLE_SPECTRO, il * 3 + type,
                                  spectro);
                cpl_table_set_string(result->telluric_table,
                                     MOO_TELLURIC_TABLE_BAND, il * 3 + type,
                                     band);
                cpl_table_set_int(result->telluric_table,
                                  MOO_TELLURIC_TABLE_TELLURIC, il * 3 + type,
                                  il + 1);
            }
            cpl_array_delete(sel);
        }
    }

    for (int i = 0; i < 3; i++) {
        cpl_image *im = NULL;
        int ny = 0;
        int nx = 0;
        for (int il = 0; il < size; il++) {
            moo_sci *sci = moo_scilist_get(scilist, il);
            moo_target_table *target_table = moo_sci_get_target_table(sci);

            if (target_table->table != NULL) {
                moo_sci_single *sci_single =
                    moo_sci_load_single(sci, i, badpix_level);
                if (sci_single != NULL) {
                    cpl_propertylist *sci_header =
                        moo_sci_single_get_header(sci_single);

                    if (il == 0) {
                        cpl_propertylist *header = cpl_propertylist_new();
                        cpl_propertylist_copy_property(header, sci_header,
                                                       MOO_PFITS_CD1_1);
                        cpl_propertylist_copy_property(header, sci_header,
                                                       MOO_PFITS_CRVAL1);
                        cpl_propertylist_copy_property(header, sci_header,
                                                       MOO_PFITS_CRPIX1);
                        cpl_propertylist_copy_property(header, sci_header,
                                                       MOO_PFITS_CUNIT1);
                        result->data_header[i] = header;
                    }
                    hdrl_image *himg = moo_sci_single_get_image(sci_single);
                    nx = hdrl_image_get_size_x(himg);
                    ny++;
                }
            }
        }
        if (ny > 0) {
            im = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);
            moo_telluric_set_image(result, i, im);
        }
    }

    return result;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Prepare TELLURIC for SCI frame
  @param    rbn the RBN from where we computed the rsponse
  @return   create table for telluric

 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 *

 */
/*----------------------------------------------------------------------------*/
moo_telluric *
moo_telluric_create_from_rbn(moo_rbn *rbn)
{
    moo_telluric *result = NULL;

    cpl_ensure(rbn != NULL, CPL_ERROR_NULL_INPUT, NULL);

    cpl_table *fibre_table = moo_rbn_get_fibre_table(rbn);
    int size = cpl_table_get_nrow(fibre_table);
    result = moo_telluric_new();
    result->primary_header = cpl_propertylist_new();
    result->telluric_table = cpl_table_new(size);

    cpl_table_new_column(result->telluric_table, MOO_TELLURIC_TABLE_INDEXRBN,
                         CPL_TYPE_INT);
    cpl_table_new_column(result->telluric_table, MOO_TELLURIC_TABLE_SPECTRO,
                         CPL_TYPE_STRING);
    cpl_table_new_column(result->telluric_table, MOO_TELLURIC_TABLE_TELLURIC,
                         CPL_TYPE_INT);

    for (int i = 0; i < size; i++) {
        int idrbn =
            cpl_table_get_int(fibre_table, MOO_TARGET_TABLE_INDEXRBN, i, NULL);
        const int spectro =
            cpl_table_get_int(fibre_table, MOO_TARGET_TABLE_SPECTRO, i, NULL);
        cpl_table_set_int(result->telluric_table, MOO_TELLURIC_TABLE_INDEXRBN,
                          i, idrbn);
        cpl_table_set_int(result->telluric_table, MOO_TELLURIC_TABLE_SPECTRO, i,
                          spectro);
        cpl_table_set_int(result->telluric_table, MOO_TELLURIC_TABLE_TELLURIC,
                          i, idrbn);
    }

    for (int i = 0; i < 3; i++) {
        int ny = 0;
        int nx = 0;

        moo_rbn_single *rbn_single =
            moo_rbn_load_single(rbn, i, MOONS_FLAG_GOOD);
        if (rbn_single != NULL) {
            cpl_propertylist *rbn_header =
                moo_rbn_single_get_header(rbn_single);
            cpl_propertylist *header = cpl_propertylist_new();
            cpl_propertylist_copy_property(header, rbn_header, MOO_PFITS_CD1_1);
            cpl_propertylist_copy_property(header, rbn_header,
                                           MOO_PFITS_CRVAL1);
            cpl_propertylist_copy_property(header, rbn_header,
                                           MOO_PFITS_CRPIX1);
            cpl_propertylist_copy_property(header, rbn_header,
                                           MOO_PFITS_CUNIT1);
            result->data_header[i] = header;

            hdrl_image *himg = moo_rbn_single_get_image(rbn_single);
            nx = hdrl_image_get_size_x(himg);
            ny = hdrl_image_get_size_y(himg);

            cpl_image *im = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);
            moo_telluric_set_image(result, i, im);
        }
    }

    return result;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Delete a moo_telluric
  @param    self    moo_telluric to delete
  @return   void

  If the moo_telluric @em self is @c NULL, nothing is done and no error is set.

 */
/*----------------------------------------------------------------------------*/

void
moo_telluric_delete(moo_telluric *self)
{
    if (self != NULL) {
        if (self->primary_header != NULL) {
            cpl_propertylist_delete(self->primary_header);
        }
        for (int i = 0; i < 3; i++) {
            if (self->data[i] != NULL) {
                cpl_image_delete(self->data[i]);
            }
            if (self->data_header[i] != NULL) {
                cpl_propertylist_delete(self->data_header[i]);
            }
        }
        if (self->telluric_table != NULL) {
            cpl_table_delete(self->telluric_table);
        }
        cpl_free(self);
    }
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Save a moo_telluric to a FITS file
  @param    self          moo_telluric to write to disk or NULL
  @param    filename      Name of the file to write
  @return   CPL_ERROR_NONE or the relevant _cpl_error_code_ on error

  This function saves a moo_telluric to a FITS file, using cfitsio.
  Only not NULL extensions are written.
*/
/*----------------------------------------------------------------------------*/
void
moo_telluric_save(moo_telluric *self, const char *filename)
{
    if (self != NULL) {
        cpl_propertylist_save(self->primary_header, filename, CPL_IO_CREATE);
        const char *extnames[] = { MOO_TELLURIC_EXT_RI, MOO_TELLURIC_EXT_YJ,
                                   MOO_TELLURIC_EXT_H };

        for (int i = 0; i < 3; i++) {
            cpl_propertylist *h = cpl_propertylist_new();
            cpl_image *im = self->data[i];
            const char *extname = extnames[i];
            moo_fits_write_extension_image(im, filename, NULL, extname,
                                           CPL_TYPE_FLOAT,
                                           self->data_header[i]);
            cpl_propertylist_delete(h);
        }

        if (self->telluric_table != NULL) {
            cpl_propertylist *h = cpl_propertylist_new();
            cpl_propertylist_append_string(h, MOO_PFITS_EXTNAME,
                                           MOO_TELLURIC_EXT_STANDARD_TABLE);
            cpl_table_save(self->telluric_table, h, h, filename, CPL_IO_EXTEND);
            cpl_propertylist_delete(h);
        }
    }
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Load a TELLURIC frame and create a moo_telluric
  @return   1 newly allocated moo_telluric or NULL in case of an error

  The returned object must be deallocated using moo_telluric_delete().
  Possible _cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
moo_telluric *
moo_telluric_load(const cpl_frame *frame)
{
    cpl_ensure(frame != NULL, CPL_ERROR_NULL_INPUT, NULL);

    const char *filename = cpl_frame_get_filename(frame);
    cpl_ensure(filename != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_errorstate prev_state = cpl_errorstate_get();

    moo_telluric *res = moo_telluric_new();

    res->primary_header = cpl_propertylist_load(filename, 0);
    const char *extnames[] = { MOO_TELLURIC_EXT_RI, MOO_TELLURIC_EXT_YJ,
                               MOO_TELLURIC_EXT_H };

    for (int i = 0; i < 3; i++) {
        const char *extname = extnames[i];
        cpl_propertylist *header = NULL;
        header = moo_fits_load_extension_header(filename, NULL, extname);
        res->data_header[i] = header;

        if (header != NULL) {
            int naxis = moo_pfits_get_naxis(header);
            if (naxis == 2) {
                res->data[i] =
                    moo_fits_load_extension_image(filename, NULL, extname,
                                                  CPL_TYPE_DOUBLE);
            }
        }
    }
    cpl_size qnum =
        cpl_fits_find_extension(filename, MOO_TELLURIC_EXT_STANDARD_TABLE);
    if (qnum > 0) {
        res->telluric_table = cpl_table_load(filename, qnum, 0);
    }

    if (!cpl_errorstate_is_equal(prev_state)) {
        cpl_msg_debug(__func__, "Error in loading telluric : %d",
                      cpl_error_get_code());
        moo_telluric_delete(res);
        res = NULL;
        cpl_errorstate_set(prev_state);
    }
    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Get the index of the telluric for a given indexrbn
  @param    self the telluric
  @param    idrbn the indexrbn of the spectra
  @param    fspectro the SPECTRO (1, 2)
  @param    fband the band (Ri, YJ, H)
  @return   the wanted index or -1

*/
/*----------------------------------------------------------------------------*/

int
moo_telluric_get_index(moo_telluric *self,
                       int idrbn,
                       int fspectro,
                       const char *fband)
{
    int index = -1;
    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, -1);
    cpl_ensure(self->telluric_table != NULL, CPL_ERROR_NULL_INPUT, -1);
    int nrow = cpl_table_get_nrow(self->telluric_table);
    int min_diff = 1024;

    for (int i = 0; i < nrow; i++) {
        int n = cpl_table_get_int(self->telluric_table,
                                  MOO_TELLURIC_TABLE_INDEXRBN, i, NULL);
        const int nindex =
            cpl_table_get_int(self->telluric_table, MOO_TELLURIC_TABLE_TELLURIC,
                              i, NULL);
        const int spectro =
            cpl_table_get_int(self->telluric_table, MOO_TELLURIC_TABLE_SPECTRO,
                              i, NULL);
        const char *band = cpl_table_get_string(self->telluric_table,
                                                MOO_TELLURIC_TABLE_BAND, i);
        if ((spectro == fspectro) && (strcmp(band, fband) == 0)) {
            int diff = (int)fabs(idrbn - n);

            if (diff == 0) {
                index = nindex;
                break;
            }
            else if (min_diff > diff) {
                index = nindex;
                min_diff = diff;
            }
        }
    }

    return index;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Get the band of the telluric  
  @param    type the detector type
  @return   the band or NULL

*/
/*----------------------------------------------------------------------------*/
const char *
moo_telluric_get_band(moo_detector_type type)
{
    const char *res[] = { MOO_TELLURIC_EXT_RI, MOO_TELLURIC_EXT_YJ,
                          MOO_TELLURIC_EXT_H };
    int idx = type;
    return res[idx];
}
/**@}*/
