/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/
#include <math.h>
#include <string.h>
#include <cpl.h>
#include <hdrl.h>
#include "moo_params.h"
#include "moo_single.h"
#include "moo_badpix.h"
#include "moo_ext.h"
#include "moo_ext_single.h"
#include "moo_pfits.h"
#include "moo_map.h"
#include "moo_utils.h"
#include "moo_qc.h"
#include "moo_wavesol.h"
#include "moo_fibres_table.h"
#include "moo_line_table.h"
#ifdef _OPENMP
#include <omp.h>
#endif
/*----------------------------------------------------------------------------*/
/**
 * @defgroup moons_drl  Moons data reduction
 *
 */
/*----------------------------------------------------------------------------*/
/**@{*/

/*-----------------------------------------------------------------------------
                              Function codes
 -----------------------------------------------------------------------------*/

static moo_tcheby2d_polynomial *
_moo_wavesol_fit2d(moo_loc_single *loc,
                   cpl_image *res,
                   moo_spectral_format_info *info,
                   int degx,
                   int degy,
                   int niter,
                   double max_frac,
                   double kappa,
                   int *nb_sigclip,
                   int slit_idx,
                   cpl_table *line_table,
                   const char *xcolname,
                   const char *gfit_colname)
{
    moo_tcheby2d_polynomial *result = NULL;
    cpl_array *seltab = NULL;
    cpl_vector *vx = NULL;
    cpl_vector *vy = NULL;
    cpl_vector *vl = NULL;

    cpl_ensure(loc != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(info != NULL, CPL_ERROR_NULL_INPUT, NULL);

    cpl_table_select_all(line_table);
    cpl_table_and_selected_int(line_table, MOO_LINE_TABLE_SLITLET, CPL_EQUAL_TO,
                               slit_idx);
    cpl_size line_table_size =
        cpl_table_and_selected_int(line_table, MOO_LINE_TABLE_MFITTED,
                                   CPL_EQUAL_TO, 1);

    int min_size = (degx + 1) * (degy + 1) + 1;
    moo_try_assure(line_table_size >= min_size, CPL_ERROR_ILLEGAL_INPUT,
                   "Not enough lines %lld (expected at least %d)",
                   line_table_size, min_size);

    seltab = cpl_table_where_selected(line_table);

    double *xpos = cpl_table_get_data_double(line_table, xcolname);
    double *wave = cpl_table_get_data_double(line_table, MOO_LINE_TABLE_WAVE);
    double *ypos = cpl_table_get_data_double(line_table, MOO_LINE_TABLE_YLOC);
    double *fwhm = cpl_table_get_data_double(line_table, MOO_LINE_TABLE_FWHM);
    int *indexext = cpl_table_get_data_int(line_table, MOO_LINE_TABLE_INDEXEXT);
    int *fitted = cpl_table_get_data_int(line_table, MOO_LINE_TABLE_MFITTED);
    int nx = cpl_image_get_size_x(res);
    cpl_image *ycentroids = moo_loc_single_get_f_centroids(loc);
    cpl_image_reject_value(ycentroids, CPL_VALUE_NAN);
    double ny = cpl_image_get_max(ycentroids);
    vx = cpl_vector_new(line_table_size);
    vl = cpl_vector_new(line_table_size);
    vy = cpl_vector_new(line_table_size);

    for (int i = 0; i < line_table_size; i++) {
        int idx = cpl_array_get_cplsize(seltab, i, NULL);
        cpl_vector_set(vx, i, xpos[idx]);
        cpl_vector_set(vy, i, ypos[idx]);
        cpl_vector_set(vl, i, wave[idx]);
    }

    result = moo_tcheby2d_polynomial_fit(vx, degx, 1., (double)nx, vy, degy, 1.,
                                         ny, vl, info->wmin, info->wmax);
    int direction = info->direction;
    int total_rejected = 0;
    int size = line_table_size;
    int i;
    for (i = 1; i <= niter; i++) {
        cpl_vector *vdiff = cpl_vector_new(size);
        for (int j = 0; j < size; j++) {
            int idx = cpl_array_get_cplsize(seltab, j, NULL);
            double x = xpos[idx];
            double y = ypos[idx];
            double l = wave[idx];
            double t = moo_tcheby2d_polynomial_eval(result, x, y);
            double diff = l - t;

            cpl_vector_set(vdiff, j, diff);
        }

        double stdev = cpl_vector_get_stdev(vdiff);
        int nb_rejected = 0;

        for (int j = 0; j < size; j++) {
            double diff = cpl_vector_get(vdiff, j);
            int idx = cpl_array_get_cplsize(seltab, j, NULL);
            if (fabs(diff) >= kappa * stdev) {
                fitted[idx] = 2;
                nb_rejected++;
            }
        }
        cpl_vector_delete(vdiff);

        if (nb_rejected == size) {
            break;
        }

        if (nb_rejected == 0) {
            break;
        }

        double frac =
            (double)(nb_rejected + total_rejected) / (double)line_table_size;

        if (frac > max_frac) {
            break;
        }
        total_rejected += nb_rejected;
        moo_tcheby2d_polynomial_delete(result);
        cpl_array_delete(seltab);

        cpl_table_select_all(line_table);
        cpl_table_and_selected_int(line_table, MOO_LINE_TABLE_SLITLET,
                                   CPL_EQUAL_TO, slit_idx);
        size = cpl_table_and_selected_int(line_table, MOO_LINE_TABLE_MFITTED,
                                          CPL_EQUAL_TO, 2);

        seltab = cpl_table_where_selected(line_table);
        for (int j = 0; j < size; j++) {
            int idx = cpl_array_get_cplsize(seltab, j, NULL);
            fitted[idx] = 0;
        }
        cpl_array_delete(seltab);

        cpl_table_select_all(line_table);
        cpl_table_and_selected_int(line_table, MOO_LINE_TABLE_SLITLET,
                                   CPL_EQUAL_TO, slit_idx);
        size = cpl_table_and_selected_int(line_table, MOO_LINE_TABLE_MFITTED,
                                          CPL_EQUAL_TO, 1);
        seltab = cpl_table_where_selected(line_table);

        cpl_vector_delete(vx);
        cpl_vector_delete(vy);
        cpl_vector_delete(vl);

        vx = cpl_vector_new(size);
        vl = cpl_vector_new(size);
        vy = cpl_vector_new(size);

        for (int j = 0; j < size; j++) {
            cpl_size idx = cpl_array_get_cplsize(seltab, j, NULL);
            cpl_vector_set(vx, j, xpos[idx]);
            cpl_vector_set(vy, j, ypos[idx]);
            cpl_vector_set(vl, j, wave[idx]);
        }

        result = moo_tcheby2d_polynomial_fit(vx, degx, 1, nx, vy, degy, 1, ny,
                                             vl, info->wmin, info->wmax);
    }

    *nb_sigclip = size;
    cpl_array_delete(seltab);
    cpl_msg_debug(__func__, "Fit2d filtering data from %lld to %d",
                  line_table_size, size);

    cpl_table_select_all(line_table);
    cpl_table_and_selected_int(line_table, MOO_LINE_TABLE_SLITLET, CPL_EQUAL_TO,
                               slit_idx);
    line_table_size =
        cpl_table_and_selected_int(line_table, gfit_colname, CPL_EQUAL_TO, 1);
    seltab = cpl_table_where_selected(line_table);

    for (i = 0; i < line_table_size; i++) {
        int rej;

        int idx = cpl_array_get_cplsize(seltab, i, NULL);
        double x = xpos[idx];
        double y = ypos[idx];
        int iy = indexext[idx];

        double wavefit = moo_tcheby2d_polynomial_eval(result, x, y);

        cpl_table_set_double(line_table, MOO_LINE_TABLE_WAVEFIT, idx, wavefit);

        double fw = fwhm[idx] / 2.;

        double x1 = x - fw;
        double x2 = x + fw;

        if (x1 >= 1 && x2 >= 1 && x1 <= nx && x2 <= nx) {
            double y1 = moo_loc_single_eval_f_centroids(loc, x1, iy, &rej);
            double y2 = moo_loc_single_eval_f_centroids(loc, x2, iy, &rej);

            double wavefit1 = moo_tcheby2d_polynomial_eval(result, x1, y1);
            double wavefit2 = moo_tcheby2d_polynomial_eval(result, x2, y2);

            double resolution = wavefit / (wavefit2 - wavefit1) * direction;

            cpl_table_set_double(line_table, MOO_LINE_TABLE_RESOLUTION, idx,
                                 resolution);
        }
        else {
            cpl_table_set_double(line_table, MOO_LINE_TABLE_RESOLUTION, idx,
                                 NAN);
        }
    }

moo_try_cleanup:
    cpl_array_delete(seltab);
    cpl_vector_delete(vx);
    cpl_vector_delete(vy);
    cpl_vector_delete(vl);

    return result;
}

static cpl_error_code
_moo_wavesol_fit1d(moo_loc_single *loc,
                   int degree,
                   int numfib,
                   cpl_image *res,
                   int niter,
                   double max_frac,
                   double kappa,
                   int *nb_sigclip,
                   cpl_table *result_table,
                   int indexext_idx,
                   moo_detector_type type,
                   cpl_table *line_table,
                   const char *xcolname)
{
    cpl_table_select_all(line_table);
    cpl_table_and_selected_int(line_table, MOO_LINE_TABLE_INDEXEXT,
                               CPL_EQUAL_TO, numfib);
    cpl_size line_table_size =
        cpl_table_and_selected_int(line_table, MOO_LINE_TABLE_MFITTED,
                                   CPL_EQUAL_TO, 1);

    if (line_table_size > degree) {
        int total_rejected = 0;
        moo_tcheby_polynomial *poly = NULL;

        cpl_image *ycentroids = moo_loc_single_get_f_centroids(loc);
        cpl_image_reject_value(ycentroids, CPL_VALUE_NAN);
        int nx = cpl_image_get_size_x(res);

        cpl_array *seltab = cpl_table_where_selected(line_table);

        double *xpos = cpl_table_get_data_double(line_table, xcolname);
        double *wave =
            cpl_table_get_data_double(line_table, MOO_LINE_TABLE_WAVE);
        double *fwhm =
            cpl_table_get_data_double(line_table, MOO_LINE_TABLE_FWHM);
        int *fitted =
            cpl_table_get_data_int(line_table, MOO_LINE_TABLE_MFITTED);

        cpl_bivector *data = cpl_bivector_new(line_table_size);
        cpl_vector *vx = cpl_bivector_get_x(data);
        cpl_vector *vy = cpl_bivector_get_y(data);

        for (int i = 0; i < line_table_size; i++) {
            int idx = cpl_array_get_cplsize(seltab, i, NULL);
            double x = xpos[idx];
            double w = wave[idx];
            cpl_vector_set(vx, i, x);
            cpl_vector_set(vy, i, w);
        }
        poly = moo_tcheby_polynomial_fit(data, degree, 1, nx);
        cpl_size size = line_table_size;

        for (int i = 1; i <= niter; i++) {
            cpl_msg_info(__func__,
                         "#%d iter %d / %d with %" CPL_SIZE_FORMAT
                         " data (total rejected %d)",
                         numfib, i, niter, size, total_rejected);
            cpl_vector *vdiff = cpl_vector_new(size);

            for (int j = 0; j < size; j++) {
                double x = cpl_vector_get(vx, j);
                double y = cpl_vector_get(vy, j);
                double t = moo_tcheby_polynomial_eval(poly, x);
                double diff = y - t;
                cpl_vector_set(vdiff, j, diff);
            }

            double stdev = cpl_vector_get_stdev(vdiff);
            int nb_rejected = 0;

            for (int j = 0; j < size; j++) {
                double diff = cpl_vector_get(vdiff, j);
                if (fabs(diff) >= kappa * stdev) {
                    int idx = cpl_array_get_cplsize(seltab, j, NULL);
                    fitted[idx] = 0;
                    nb_rejected++;
                }
            }
            cpl_vector_delete(vdiff);

            if (nb_rejected == size) {
                cpl_msg_info("test", "#%d all rejected", numfib);
                break;
            }

            if (nb_rejected == 0) {
                cpl_msg_info("test", "#%d no rejected", numfib);
                break;
            }
            double frac = (double)(nb_rejected + total_rejected) /
                          (double)line_table_size;

            if (frac > max_frac) {
                cpl_msg_info(__func__, "max frac reached (%f > %f)", frac,
                             max_frac);
                break;
            }
            total_rejected += nb_rejected;
            moo_tcheby_polynomial_delete(poly);
            cpl_array_delete(seltab);

            cpl_table_select_all(line_table);
            cpl_table_and_selected_int(line_table, MOO_LINE_TABLE_INDEXEXT,
                                       CPL_EQUAL_TO, numfib);
            size =
                cpl_table_and_selected_int(line_table, MOO_LINE_TABLE_MFITTED,
                                           CPL_EQUAL_TO, 1);
            seltab = cpl_table_where_selected(line_table);

            cpl_bivector_delete(data);
            data = cpl_bivector_new(size);
            vx = cpl_bivector_get_x(data);
            vy = cpl_bivector_get_y(data);
            for (int j = 0; j < size; j++) {
                int idx = cpl_array_get_cplsize(seltab, j, NULL);
                cpl_vector_set(vx, j, xpos[idx]);
                cpl_vector_set(vy, j, wave[idx]);
            }
            poly = moo_tcheby_polynomial_fit(data, degree, 1, nx);
        }
        cpl_array_delete(seltab);
        cpl_table_select_all(line_table);
        cpl_table_and_selected_int(line_table, MOO_LINE_TABLE_INDEXEXT,
                                   CPL_EQUAL_TO, numfib);
        size = cpl_table_and_selected_int(line_table, MOO_LINE_TABLE_MFITTED,
                                          CPL_EQUAL_TO, 1);

        int nb_points = size;
        *nb_sigclip += nb_points;

        cpl_table_select_all(line_table);
        line_table_size =
            cpl_table_and_selected_int(line_table, MOO_LINE_TABLE_INDEXEXT,
                                       CPL_EQUAL_TO, numfib);
        seltab = cpl_table_where_selected(line_table);

        for (int i = 0; i < line_table_size; i++) {
            int idx = cpl_array_get_cplsize(seltab, i, NULL);
            double x = xpos[idx];
            double wavefit = moo_tcheby_polynomial_eval(poly, x);
            cpl_table_set_double(line_table, MOO_LINE_TABLE_WAVEFIT, idx,
                                 wavefit);

            double fw = fwhm[idx] / 2.;

            double x1 = x - fw;
            double x2 = x + fw;
            if (x1 >= 1 && x2 >= 1) {
                double wavefit1 = moo_tcheby_polynomial_eval(poly, x1);
                double wavefit2 = moo_tcheby_polynomial_eval(poly, x2);
                double resolution = wavefit / (wavefit2 - wavefit1);
                cpl_table_set_double(line_table, MOO_LINE_TABLE_RESOLUTION, idx,
                                     resolution);
            }
            else {
                cpl_table_set_double(line_table, MOO_LINE_TABLE_RESOLUTION, i,
                                     NAN);
            }
        }

        for (int i = 1; i <= nx; i++) {
            int rej;

            cpl_image_get(ycentroids, i, numfib, &rej);

            if (rej == 1) {
                cpl_image_set(res, i, numfib, NAN);
            }
            else {
                double t = moo_tcheby_polynomial_eval(poly, i);
                cpl_image_set(res, i, numfib, t);
            }
        }

        moo_fibres_table_set_rejectline(result_table, type, indexext_idx,
                                        total_rejected);

        moo_tcheby_polynomial_delete(poly);
        cpl_bivector_delete(data);
        cpl_array_delete(seltab);
    }
    return cpl_error_get_code();
}

static cpl_vector *
_compute_slitlet_shift(cpl_table *slitlet_shift,
                       cpl_table *catalog_lines,
                       int slitlet_idx)
{
    cpl_vector *vxdiff = NULL;

    cpl_ensure(catalog_lines != NULL, CPL_ERROR_NULL_INPUT, NULL);

    const double *waves =
        cpl_table_get_data_double_const(catalog_lines, MOO_ARCLINE_LIST_WAVE);
    int nb_catalog_lines = cpl_table_get_nrow(catalog_lines);

    vxdiff = cpl_vector_new(nb_catalog_lines);

    if (slitlet_shift != NULL) {
        int min = cpl_table_get_column_min(slitlet_shift,
                                           MOO_SLITLET_SHIFT_TABLE_SLITLET);
        int max = cpl_table_get_column_max(slitlet_shift,
                                           MOO_SLITLET_SHIFT_TABLE_SLITLET);
        double center = (min + max) / 2.;
        for (int i = 0; i < nb_catalog_lines; i++) {
            double waveref = waves[i];
            cpl_table_select_all(slitlet_shift);
            cpl_table_and_selected_int(slitlet_shift,
                                       MOO_SLITLET_SHIFT_TABLE_SLITLET,
                                       CPL_EQUAL_TO, slitlet_idx);
            cpl_table_and_selected_double(slitlet_shift,
                                          MOO_SLITLET_SHIFT_TABLE_WAVE,
                                          CPL_EQUAL_TO, waveref);
            cpl_array *sel = cpl_table_where_selected(slitlet_shift);
            int slitletshift_idx = cpl_array_get(sel, 0, NULL);
            double xdiff = cpl_table_get_double(slitlet_shift,
                                                MOO_SLITLET_SHIFT_TABLE_XDIFF,
                                                slitletshift_idx, NULL);
            cpl_array_delete(sel);

            if (isnan(xdiff)) {
                cpl_table_select_all(slitlet_shift);
                int next_slit = slitlet_idx + 1;
                if (slitlet_idx > center) {
                    next_slit = slitlet_idx - 1;
                }
                cpl_table_and_selected_int(slitlet_shift,
                                           MOO_SLITLET_SHIFT_TABLE_SLITLET,
                                           CPL_EQUAL_TO, next_slit);
                cpl_table_and_selected_double(slitlet_shift,
                                              MOO_SLITLET_SHIFT_TABLE_WAVE,
                                              CPL_EQUAL_TO, waveref);
                sel = cpl_table_where_selected(slitlet_shift);
                slitletshift_idx = cpl_array_get(sel, 0, NULL);
                cpl_array_delete(sel);
                xdiff = cpl_table_get_double(slitlet_shift,
                                             MOO_SLITLET_SHIFT_TABLE_XDIFF,
                                             slitletshift_idx, NULL);
                if (isnan(xdiff)) {
                    xdiff = 0;
                }
            }
            cpl_vector_set(vxdiff, i, xdiff);
        }
    }
    else {
        for (int i = 0; i < nb_catalog_lines; i++) {
            cpl_vector_set(vxdiff, i, 0.0);
        }
    }
    return vxdiff;
}

static cpl_table *
_moo_wavesol_fibre_using_guess(cpl_image *data,
                               cpl_image *errs,
                               cpl_image *guess,
                               cpl_table *slitlet_shift,
                               int numfib,
                               cpl_table *catalog_lines,
                               int winhsize,
                               double sigma_min,
                               double sigma_max,
                               int slitlet_idx,
                               int indexext,
                               const char *extname,
                               const int spectro,
                               cpl_table **lline_table,
                               int *ldetected,
                               int *lfailed,
                               int *lfitted)
{
    cpl_table *detected_table = NULL;
    cpl_table *ref_table = NULL;
    cpl_vector *sol_w = NULL;
    cpl_vector *sol_wx = NULL;
    cpl_bivector *sol = NULL;

    cpl_ensure(data != NULL, CPL_ERROR_NULL_INPUT, NULL);

    int nx = cpl_image_get_size_x(data);
    ref_table = cpl_table_new(nx);
    cpl_table_new_column(ref_table, MOO_WAVESOL_REFTABLE_WAVE, CPL_TYPE_DOUBLE);
    cpl_table_new_column(ref_table, MOO_WAVESOL_REFTABLE_X, CPL_TYPE_DOUBLE);

    cpl_vector *vect = cpl_vector_new_from_image_row(data, numfib);
    cpl_vector *verr = cpl_vector_new_from_image_row(errs, numfib);
    cpl_vector *vsol = cpl_vector_new_from_image_row(guess, numfib);

    for (int i = 0; i < nx; i++) {
        double w = (float)cpl_vector_get(vsol, i);
        int x = i + 1;
        if (!isnan(w)) {
            cpl_table_set(ref_table, MOO_WAVESOL_REFTABLE_WAVE, i, w);
            cpl_table_set(ref_table, MOO_WAVESOL_REFTABLE_X, i, x);
        }
        else {
            cpl_table_set_invalid(ref_table, MOO_WAVESOL_REFTABLE_WAVE, i);
        }
    }
    cpl_vector_delete(vsol);

    if (!cpl_table_has_valid(ref_table, MOO_WAVESOL_REFTABLE_WAVE)) {
        cpl_msg_info("moo_wavesol", "#%d invalid guess data", numfib);
        cpl_vector_delete(verr);
        cpl_vector_delete(vect);
        cpl_table_delete(ref_table);
        return detected_table;
    }

    cpl_table_erase_invalid(ref_table);
    int ref_size = cpl_table_get_nrow(ref_table);
    double *refd_w =
        cpl_table_get_data_double(ref_table, MOO_WAVESOL_REFTABLE_WAVE);
    double *refd_wx =
        cpl_table_get_data_double(ref_table, MOO_WAVESOL_REFTABLE_X);

    sol_w = cpl_vector_wrap(ref_size, refd_w);
    sol_wx = cpl_vector_wrap(ref_size, refd_wx);
    sol = cpl_bivector_wrap_vectors(sol_w, sol_wx);

    double wmin =
        cpl_table_get_column_min(ref_table, MOO_WAVESOL_REFTABLE_WAVE);
    double wmax =
        cpl_table_get_column_max(ref_table, MOO_WAVESOL_REFTABLE_WAVE);

    cpl_bivector *solx = cpl_bivector_new(ref_size);
    cpl_vector *solx_x = cpl_bivector_get_x(solx);
    cpl_vector *solx_y = cpl_bivector_get_y(solx);

    for (int i = 0; i < ref_size; i++) {
        double xv = cpl_vector_get(sol_wx, i);
        cpl_vector_set(solx_x, i, xv);
        double yv = cpl_vector_get(sol_w, i);
        cpl_vector_set(solx_y, i, yv);
    }

    cpl_bivector_sort(sol, sol, CPL_SORT_ASCENDING, CPL_SORT_BY_X);

    const double *waves =
        cpl_table_get_data_double_const(catalog_lines, MOO_ARCLINE_LIST_WAVE);
    int nb_catalog_lines = cpl_table_get_nrow(catalog_lines);
    int nb_failed = 0;
    int nb_detected = 0;
    detected_table = moo_detected_table_new(nb_catalog_lines, 0);
    cpl_vector *vxdiff =
        _compute_slitlet_shift(slitlet_shift, catalog_lines, slitlet_idx);

    for (int i = 0; i < nb_catalog_lines; i++) {
        double waveref = waves[i];
        cpl_table_set_int(detected_table, MOO_LINE_TABLE_INDEXEXT, i, indexext);
        cpl_table_set_int(detected_table, MOO_LINE_TABLE_SLITLET, i,
                          slitlet_idx);
        cpl_table_set_double(detected_table, MOO_LINE_TABLE_WAVE, i, waveref);
        double xdiff = cpl_vector_get(vxdiff, i);

        cpl_table_set_double(detected_table, MOO_LINE_TABLE_XDIFF, i, xdiff);
        cpl_table_set_int(detected_table, MOO_DETECTED_TABLE_NOT_IN_RANGE, i,
                          1);
        cpl_table_set_double(detected_table, MOO_LINE_TABLE_INTENSITY, i, NAN);
        cpl_table_set_int(detected_table, MOO_LINE_TABLE_GFITTED, i, 0);
        cpl_table_set_double(detected_table, MOO_LINE_TABLE_XGAUSS, i, NAN);
        cpl_table_set_double(detected_table, MOO_LINE_TABLE_XBARY, i, NAN);
        cpl_table_set_double(detected_table, MOO_LINE_TABLE_FWHM, i, NAN);
        cpl_table_set_double(detected_table, MOO_LINE_TABLE_AMPLITUDE, i, NAN);
        cpl_table_set_double(detected_table, MOO_LINE_TABLE_BACKGROUND, i, NAN);
        cpl_table_set_double(detected_table, MOO_LINE_TABLE_FIT_FLUX, i, NAN);
        cpl_table_set_double(detected_table, MOO_LINE_TABLE_FIT_ERR, i, NAN);
        cpl_table_set_double(detected_table, MOO_LINE_TABLE_FIT_CHI2, i, NAN);
        cpl_table_set_double(detected_table, MOO_LINE_TABLE_FIT_MSE, i, NAN);
        cpl_table_set_double(detected_table, MOO_DETECTED_TABLE_WIN_CEN, i,
                             NAN);
        cpl_table_set_double(detected_table, MOO_DETECTED_TABLE_WIN_MIN, i,
                             NAN);
        cpl_table_set_double(detected_table, MOO_DETECTED_TABLE_WIN_MAX, i,
                             NAN);
        cpl_table_set_int(detected_table, MOO_LINE_TABLE_FILTERED, i, -1);
        cpl_table_set_double(detected_table, MOO_DETECTED_TABLE_WAVETH, i, NAN);
    }

    cpl_vector_delete(vxdiff);

    cpl_table_select_all(detected_table);
    cpl_table_and_selected_double(detected_table, MOO_LINE_TABLE_WAVE,
                                  CPL_NOT_LESS_THAN, wmin);
    int nb_lines =
        cpl_table_and_selected_double(detected_table, MOO_LINE_TABLE_WAVE,
                                      CPL_NOT_GREATER_THAN, wmax);

    cpl_array *seltab = cpl_table_where_selected(detected_table);
    nb_failed = nb_catalog_lines - nb_lines;

    if (nb_lines > 0) {
        cpl_vector *nwavepos_l = cpl_vector_new(nb_lines);
        for (int i = 0; i < nb_lines; i++) {
            int idx = cpl_array_get_cplsize(seltab, i, NULL);
            double w = cpl_table_get_double(detected_table, MOO_LINE_TABLE_WAVE,
                                            idx, NULL);
            cpl_table_set_int(detected_table, MOO_DETECTED_TABLE_NOT_IN_RANGE,
                              idx, 0);
            cpl_vector_set(nwavepos_l, i, w);
        }

        cpl_vector_sort(nwavepos_l, CPL_SORT_ASCENDING);
        cpl_vector *nwavepos_x = cpl_vector_new(nb_lines);
        cpl_bivector *nwavepos =
            cpl_bivector_wrap_vectors(nwavepos_l, nwavepos_x);
        cpl_bivector_interpolate_linear(nwavepos, sol);

        for (int i = 0; i < nb_lines; i++) {
            double xguess = (float)cpl_vector_get(nwavepos_x, i);

            int idx = cpl_array_get_cplsize(seltab, i, NULL);

            double xdiff =
                cpl_table_get_double(detected_table, MOO_LINE_TABLE_XDIFF, idx,
                                     NULL);
            double win_cen = round(xguess - xdiff);
            int size = winhsize * 2 + 1;
            double win_min = win_cen - winhsize;
            double win_max = win_cen + winhsize;
            if (win_min <= 1) {
                win_min = 1;
            }
            if (win_max >= nx) {
                win_max = nx;
            }
            size = win_max - win_min + 1;
            cpl_table_set_double(detected_table, MOO_LINE_TABLE_XGUESS, idx,
                                 xguess);
            cpl_table_set_double(detected_table, MOO_DETECTED_TABLE_WIN_MIN,
                                 idx, win_min);
            cpl_table_set_double(detected_table, MOO_DETECTED_TABLE_WIN_MAX,
                                 idx, win_max);
            cpl_table_set_double(detected_table, MOO_DETECTED_TABLE_WIN_CEN,
                                 idx, win_cen);

            cpl_bivector *points = cpl_bivector_new(size);
            cpl_vector *x = cpl_bivector_get_x(points);
            cpl_vector *y = cpl_bivector_get_y(points);
            cpl_vector *yerr = cpl_vector_new(size);
            nb_detected++;
            for (int pos = win_min; pos <= win_max; pos++) {
                cpl_vector_set(x, pos - win_min, pos);
                cpl_vector_set(y, pos - win_min,
                               (float)cpl_vector_get(vect, pos - 1));
                cpl_vector_set(yerr, pos - win_min,
                               (float)cpl_vector_get(verr, pos - 1));
            }
            double intensity = cpl_vector_get_max(y);
            cpl_table_set_double(detected_table, MOO_LINE_TABLE_INTENSITY, idx,
                                 intensity);

            double centroid = NAN, background = NAN, area = NAN, sigma = NAN,
                   mse = 0, chi2 = 0;
            cpl_errorstate prev_state = cpl_errorstate_get();
            cpl_vector_fit_gaussian(x, NULL, y, NULL, CPL_FIT_ALL, &centroid,
                                    &sigma, &area, &background, &mse, NULL,
                                    NULL);
            cpl_vector_delete(yerr);

            if (centroid >= win_min && centroid <= win_max &&
                cpl_errorstate_is_equal(prev_state)) {
                int rej;
                cpl_table_set_int(detected_table, MOO_LINE_TABLE_GFITTED, idx,
                                  1);
                double fit_flux =
                    cpl_image_get(data, round(centroid), numfib, &rej);
                double fit_err =
                    cpl_image_get(errs, round(centroid), numfib, &rej);

                cpl_table_set_double(detected_table,
                                     MOO_DETECTED_TABLE_FIT_FLUX, idx,
                                     fit_flux);
                cpl_table_set_double(detected_table, MOO_DETECTED_TABLE_FIT_ERR,
                                     idx, fit_err);

                cpl_table_set_double(detected_table,
                                     MOO_DETECTED_TABLE_FIT_CHI2, idx, chi2);
                cpl_table_set_double(detected_table, MOO_DETECTED_TABLE_FIT_MSE,
                                     idx, mse);

                cpl_bivector *fout = cpl_bivector_new(1);
                cpl_vector *fout_x = cpl_bivector_get_x(fout);
                cpl_vector *fout_y = cpl_bivector_get_y(fout);
                cpl_vector_set(fout_x, 0, centroid);

                cpl_bivector_interpolate_linear(fout, solx);
                double waveth = cpl_vector_get(fout_y, 0);

                cpl_bivector_delete(fout);
                cpl_table_set_double(detected_table, MOO_LINE_TABLE_XGAUSS, idx,
                                     centroid);
                cpl_table_set_double(detected_table, MOO_DETECTED_TABLE_WAVETH,
                                     idx, waveth);

                double amplitude = area / sqrt(CPL_MATH_2PI * sigma * sigma);
                double fwhm = CPL_MATH_FWHM_SIG * sigma;
                cpl_table_set_double(detected_table, MOO_LINE_TABLE_FWHM, idx,
                                     fwhm);
                cpl_table_set_double(detected_table, MOO_LINE_TABLE_AMPLITUDE,
                                     idx, amplitude);
                cpl_table_set_double(detected_table, MOO_LINE_TABLE_BACKGROUND,
                                     idx, background);
                double width_bary = 0;
                double x_bary = 0;
                moo_barycenter_fit(points, &x_bary, &width_bary);
                cpl_table_set_double(detected_table, MOO_LINE_TABLE_XBARY, idx,
                                     x_bary);
                if (sigma >= sigma_min && sigma <= sigma_max) {
                    cpl_table_set_int(detected_table, MOO_LINE_TABLE_FILTERED,
                                      idx, 0);
                }
                else if (sigma < sigma_min) {
                    cpl_table_set_int(detected_table, MOO_LINE_TABLE_FILTERED,
                                      idx, 1);
                }
                else {
                    cpl_table_set_int(detected_table, MOO_LINE_TABLE_FILTERED,
                                      idx, 2);
                }
            }
            else {
                nb_failed++;
                cpl_errorstate_set(prev_state);
            }
            cpl_bivector_delete(points);
        }
        cpl_array_delete(seltab);

        cpl_table_select_all(detected_table);
        int nb_fitted =
            cpl_table_and_selected_int(detected_table, MOO_LINE_TABLE_GFITTED,
                                       CPL_EQUAL_TO, 1);

        seltab = cpl_table_where_selected(detected_table);
        *ldetected = nb_detected;
        *lfailed = nb_failed;
        *lfitted = nb_fitted;

        cpl_msg_debug("moo_wavesol", "s%d #%d lines %d fitted %d", slitlet_idx,
                      numfib, nb_catalog_lines, nb_fitted);

        *lline_table = moo_line_table_new(0);
        cpl_table_set_size(*lline_table, nb_fitted);

        for (int i = 0; i < nb_fitted; i++) {
            int idx = cpl_array_get_cplsize(seltab, i, NULL);
            double d_waveref =
                cpl_table_get_double(detected_table, MOO_LINE_TABLE_WAVE, idx,
                                     NULL);
            double xgauss =
                cpl_table_get_double(detected_table, MOO_LINE_TABLE_XGAUSS, idx,
                                     NULL);
            double xbary =
                cpl_table_get_double(detected_table, MOO_LINE_TABLE_XBARY, idx,
                                     NULL);
            double fwhm = cpl_table_get_double(detected_table,
                                               MOO_LINE_TABLE_FWHM, idx, NULL);
            double intensity =
                cpl_table_get_double(detected_table, MOO_LINE_TABLE_INTENSITY,
                                     idx, NULL);
            double amplitude =
                cpl_table_get_double(detected_table, MOO_LINE_TABLE_AMPLITUDE,
                                     idx, NULL);
            double background =
                cpl_table_get_double(detected_table, MOO_LINE_TABLE_BACKGROUND,
                                     idx, NULL);
            double fit_flux =
                cpl_table_get_double(detected_table, MOO_LINE_TABLE_FIT_FLUX,
                                     idx, NULL);
            double fit_err =
                cpl_table_get_double(detected_table, MOO_LINE_TABLE_FIT_ERR,
                                     idx, NULL);
            double fit_chi2 =
                cpl_table_get_double(detected_table, MOO_LINE_TABLE_FIT_CHI2,
                                     idx, NULL);
            double fit_mse =
                cpl_table_get_double(detected_table, MOO_LINE_TABLE_FIT_MSE,
                                     idx, NULL);

            double xguess =
                cpl_table_get_double(detected_table, MOO_LINE_TABLE_XGUESS, idx,
                                     NULL);
            double waveth =
                cpl_table_get_double(detected_table, MOO_DETECTED_TABLE_WAVETH,
                                     idx, NULL);
            int detected = cpl_table_get_int(detected_table,
                                             MOO_LINE_TABLE_GFITTED, idx, NULL);
            int filtered =
                cpl_table_get_int(detected_table, MOO_LINE_TABLE_FILTERED, idx,
                                  NULL);
            int mfitted = 0;
            if (filtered == 0) {
                mfitted = 1;
            }
            cpl_table_set_int(*lline_table, MOO_LINE_TABLE_INDEXEXT, i,
                              indexext);
            cpl_table_set_int(*lline_table, MOO_LINE_TABLE_SLITLET, i,
                              slitlet_idx);
            cpl_table_set_double(*lline_table, MOO_LINE_TABLE_WAVE, i,
                                 d_waveref);
            cpl_table_set_double(*lline_table, MOO_LINE_TABLE_FLUX, i, NAN);
            cpl_table_set_string(*lline_table, MOO_LINE_TABLE_NAME, i, "");
            cpl_table_set_int(*lline_table, MOO_LINE_TABLE_INDEXRBN, i, -1);
            cpl_table_set_double(*lline_table, MOO_LINE_TABLE_YLOC, i, NAN);
            cpl_table_set_int(*lline_table, MOO_LINE_TABLE_GFITTED, i,
                              detected);
            cpl_table_set_double(*lline_table, MOO_LINE_TABLE_INTENSITY, i,
                                 intensity);
            cpl_table_set_double(*lline_table, MOO_LINE_TABLE_XGUESS, i,
                                 xguess);
            cpl_table_set_double(*lline_table, MOO_LINE_TABLE_XGAUSS, i,
                                 xgauss);
            cpl_table_set_double(*lline_table, MOO_LINE_TABLE_XDIFF, i,
                                 xguess - xgauss);
            cpl_table_set_double(*lline_table, MOO_LINE_TABLE_WAVEDIFF, i,
                                 d_waveref - waveth);
            cpl_table_set_double(*lline_table, MOO_LINE_TABLE_XBARY, i, xbary);
            cpl_table_set_double(*lline_table, MOO_LINE_TABLE_FWHM, i, fwhm);
            cpl_table_set_double(*lline_table, MOO_LINE_TABLE_AMPLITUDE, i,
                                 amplitude);
            cpl_table_set_double(*lline_table, MOO_LINE_TABLE_BACKGROUND, i,
                                 background);
            cpl_table_set_double(*lline_table, MOO_LINE_TABLE_FIT_FLUX, i,
                                 fit_flux);
            cpl_table_set_double(*lline_table, MOO_LINE_TABLE_FIT_ERR, i,
                                 fit_err);
            cpl_table_set_double(*lline_table, MOO_LINE_TABLE_FIT_CHI2, i,
                                 fit_chi2);
            cpl_table_set_double(*lline_table, MOO_LINE_TABLE_FIT_MSE, i,
                                 fit_mse);
            cpl_table_set_int(*lline_table, MOO_LINE_TABLE_FILTERED, i,
                              filtered);
            cpl_table_set_int(*lline_table, MOO_LINE_TABLE_MFITTED, i, mfitted);
            cpl_table_set_double(*lline_table, MOO_LINE_TABLE_WAVEFIT, i, NAN);
            cpl_table_set_double(*lline_table, MOO_LINE_TABLE_RESOLUTION, i,
                                 NAN);
            cpl_table_set_string(*lline_table, MOO_LINE_TABLE_DETECTOR, i,
                                 extname);
            cpl_table_set_int(*lline_table, MOO_LINE_TABLE_SPECTRO, i, spectro);
        }

        cpl_array_delete(seltab);
        cpl_bivector_unwrap_vectors(nwavepos);
        cpl_vector_delete(nwavepos_l);
        cpl_vector_delete(nwavepos_x);
    }
    else {
        cpl_msg_info("moo_wavesol", "#%d No lines in solution range (%f,%f)",
                     numfib, wmin, wmax);
    }
    cpl_bivector_delete(solx);

    cpl_bivector_unwrap_vectors(sol);
    cpl_vector_unwrap(sol_w);
    cpl_vector_unwrap(sol_wx);

    cpl_table_delete(ref_table);

    cpl_vector_delete(verr);
    cpl_vector_delete(vect);
    return detected_table;
}

static cpl_table *
_moo_do_ppm_match(const double *waves,
                  int nb_lines,
                  cpl_table *detected_table,
                  int numfib,
                  double min_disp,
                  double max_disp,
                  double tolerance,
                  int direction,
                  const char *prefix)
{
    cpl_bivector *res = NULL;
    cpl_table *ppm_table = NULL;

    cpl_vector *clines = cpl_vector_new(nb_lines);
    for (int i = 0; i < nb_lines; i++) {
        double wa = waves[i] * direction;
        cpl_vector_set(clines, i, wa * 10);
    }
    cpl_table_select_all(detected_table);
    cpl_size size =
        cpl_table_and_selected_int(detected_table, MOO_LINE_TABLE_FILTERED,
                                   CPL_EQUAL_TO, 0);
    if (size > 0) {
        cpl_array *seltab = cpl_table_where_selected(detected_table);
        cpl_vector *npeaks = cpl_vector_new(size);
        for (int i = 0; i < size; i++) {
            int idx = cpl_array_get_cplsize(seltab, i, NULL);
            double xpos =
                cpl_table_get_double(detected_table, MOO_LINE_TABLE_XGAUSS, idx,
                                     NULL);
            cpl_vector_set(npeaks, i, xpos);
        }
        cpl_array_delete(seltab);

        cpl_vector_sort(npeaks, CPL_SORT_ASCENDING);
#if MOO_DEBUG_WAVESOL_MATCHING
        {
            char *name =
                cpl_sprintf("%s_ppmmatch_peaks_%d.txt", prefix, numfib);
            FILE *test = fopen(name, "w");
            cpl_vector_dump(npeaks, test);
            fclose(test);
            cpl_free(name);
        }
#endif
        cpl_vector_sort(clines, CPL_SORT_ASCENDING);

#if MOO_DEBUG_WAVESOL_MATCHING
        {
            char *name =
                cpl_sprintf("%s_ppmmatch_lines_%d.txt", prefix, numfib);
            FILE *test = fopen(name, "w");
            cpl_vector_dump(clines, test);
            fclose(test);
            cpl_free(name);
        }
#endif

        res = cpl_ppm_match_positions(npeaks, clines, min_disp * 10,
                                      max_disp * 10, tolerance, NULL, NULL);
        if (res != NULL) {
            int rsize = cpl_bivector_get_size(res);
            ppm_table = cpl_table_new(rsize);
            cpl_table_new_column(ppm_table, MOO_PPM_TABLE_WAVE,
                                 CPL_TYPE_DOUBLE);
            cpl_table_new_column(ppm_table, MOO_PPM_TABLE_X, CPL_TYPE_DOUBLE);
            cpl_table_new_column(ppm_table, MOO_PPM_TABLE_INDEXEXT,
                                 CPL_TYPE_INT);
            cpl_table_new_column(ppm_table, MOO_PPM_TABLE_DISP,
                                 CPL_TYPE_DOUBLE);

            cpl_vector *x = cpl_bivector_get_x(res);
            cpl_vector *w = cpl_bivector_get_y(res);

            for (int i = 0; i < rsize; i++) {
                double x1 = cpl_vector_get(x, i);
                double w1 = cpl_vector_get(w, i) / 10. * direction;
                cpl_table_set(ppm_table, MOO_PPM_TABLE_WAVE, i, w1);
                cpl_table_set(ppm_table, MOO_PPM_TABLE_X, i, x1);
                cpl_table_set_int(ppm_table, MOO_PPM_TABLE_INDEXEXT, i, numfib);
            }
            for (int i = 0; i < rsize - 1; i++) {
                double x1 = cpl_vector_get(x, i);
                double x2 = cpl_vector_get(x, i + 1);
                double w1 = cpl_vector_get(w, i) / 10 * direction;
                double w2 = cpl_vector_get(w, i + 1) / 10 * direction;
                double wdiff = (w2 - w1) * direction;
                double xdiff = x2 - x1;
                cpl_table_unselect_all(ppm_table);
                cpl_table_or_selected_double(ppm_table, MOO_PPM_TABLE_WAVE,
                                             CPL_EQUAL_TO, w1);
                int nb =
                    cpl_table_or_selected_double(ppm_table, MOO_PPM_TABLE_WAVE,
                                                 CPL_EQUAL_TO, w2);
                double disp = wdiff / xdiff;
                cpl_array *sel = cpl_table_where_selected(ppm_table);
                for (int j = 0; j < nb; j++) {
                    int idx = cpl_array_get_cplsize(sel, j, NULL);
                    int rej;
                    double old =
                        cpl_table_get(ppm_table, MOO_PPM_TABLE_DISP, idx, &rej);
                    if (rej == 0) {
                        disp = (disp + old) / 2.;
                    }
                    cpl_table_set(ppm_table, MOO_PPM_TABLE_DISP, idx, disp);
                }
                cpl_array_delete(sel);
            }
#if MOO_DEBUG_WAVESOL_MATCHING
            {
                char *name =
                    cpl_sprintf("%s_ppmmatch_res_%d.fits", prefix, numfib);
                cpl_table_save(ppm_table, NULL, NULL, name, CPL_IO_CREATE);
                cpl_free(name);
            }
#endif
        }
        cpl_vector_delete(npeaks);
    }
    cpl_vector_delete(clines);
    cpl_bivector_delete(res);
    return ppm_table;
}

static cpl_table *
_moo_wavesol_fibre(cpl_image *data,
                   cpl_image *errs,
                   int numfib,
                   int nb_peaks_lines,
                   int winhsize,
                   double sigma_min,
                   double sigma_max,
                   double ref_snr,
                   cpl_table *catalog_lines,
                   double min_disp,
                   double max_disp,
                   double tolerance,
                   int direction,
                   const char *prefix,
                   int degpoly,
                   int slitlet_idx,
                   int indexext_idx,
                   const char *extname,
                   const int spectro,
                   cpl_table **lline_table,
                   int *ldetected,
                   int *lfailed,
                   int *lfitted,
                   int *lmatched)
{
    cpl_table *detected_table = NULL;

    cpl_errorstate prestate = cpl_errorstate_get();
    cpl_image *idata = NULL;
    cpl_image *ierr = NULL;

    cpl_ensure(data != NULL, CPL_ERROR_NULL_INPUT, NULL);

    int nx = cpl_image_get_size_x(data);
    moo_try_check(idata = cpl_image_extract(data, 1, numfib, nx, numfib), " ");
    moo_try_check(ierr = cpl_image_extract(errs, 1, numfib, nx, numfib), " ");
    cpl_mask *imask = cpl_image_get_bpm(idata);

    const double *waves =
        cpl_table_get_data_double_const(catalog_lines, MOO_ARCLINE_LIST_WAVE);
    int nb_catalog_lines = cpl_table_get_nrow(catalog_lines);

    int nb_detected = 0;
    int nb_fitted = 0;
    int nb_failed = 0;

    int current_detected = 0;
    detected_table = moo_detected_table_new(current_detected, 1);

    while (nb_fitted < nb_peaks_lines) {
        int rej;
        cpl_size x, y;
        int nb_bad = cpl_mask_count(imask);
        if (nb_bad == nx) {
            break;
        }
        cpl_table_set_size(detected_table, nb_detected + 1);
        cpl_image_get_maxpos(idata, &x, &y);
        double intensity = cpl_image_get(idata, x, y, &rej);
        cpl_table_set_int(detected_table, MOO_LINE_TABLE_INDEXEXT, nb_detected,
                          numfib);
        cpl_table_set_int(detected_table, MOO_LINE_TABLE_SLITLET, nb_detected,
                          slitlet_idx);
        cpl_table_set_int(detected_table, MOO_DETECTED_TABLE_NGOOD, nb_detected,
                          0);
        cpl_table_set_int(detected_table, MOO_DETECTED_TABLE_BADFIT,
                          nb_detected, -1);
        cpl_table_set_int(detected_table, MOO_LINE_TABLE_FILTERED, nb_detected,
                          -1);
        int min = x - winhsize;
        if (min < 1) {
            min = 1;
        }
        int max = x + winhsize;
        if (max >= nx) {
            max = nx;
        }

        cpl_table_set_int(detected_table, MOO_DETECTED_TABLE_MIN, nb_detected,
                          min);
        cpl_table_set_int(detected_table, MOO_DETECTED_TABLE_MAX, nb_detected,
                          max);
        cpl_table_set_double(detected_table, MOO_DETECTED_TABLE_WIN_CEN,
                             nb_detected, x);

        cpl_vector *xv = NULL;
        cpl_vector *yv = NULL;
        cpl_vector *yerr = NULL;
        cpl_matrix *covariance = NULL;

        int size = max - min + 1;

        xv = cpl_vector_new(size);
        yv = cpl_vector_new(size);
        yerr = cpl_vector_new(size);

        int ngood = 0;
        nb_bad = cpl_mask_count(imask);

        for (int i = min; i <= max; i++) {
            double flux = cpl_image_get(idata, i, 1, &rej);
            if (rej == CPL_BINARY_0) {
                double flux_err = cpl_image_get(ierr, i, 1, &rej);
                cpl_vector_set(xv, ngood, i);
                cpl_vector_set(yv, ngood, (float)flux);
                cpl_vector_set(yerr, ngood, (float)flux_err);
                ngood++;
            }
        }

        cpl_table_set_int(detected_table, MOO_DETECTED_TABLE_NGOOD, nb_detected,
                          ngood);

        if (ngood > 0) {
            cpl_vector_set_size(xv, ngood);
            cpl_vector_set_size(yv, ngood);
            cpl_vector_set_size(yerr, ngood);
            cpl_table_set_double(detected_table, MOO_LINE_TABLE_INTENSITY,
                                 nb_detected, intensity);
            double centroid = NAN, background = NAN, area = NAN, sigma = NAN,
                   mse = NAN, chi2 = NAN;
            cpl_errorstate prev_state = cpl_errorstate_get();

            cpl_vector_fit_gaussian(xv, NULL, yv, yerr, CPL_FIT_ALL, &centroid,
                                    &sigma, &area, &background, &mse, &chi2,
                                    NULL);

            if (cpl_errorstate_is_equal(prev_state)) {
                double fit_flux =
                    cpl_image_get(idata, round(centroid), 1, &rej);
                double fit_err = cpl_image_get(ierr, round(centroid), 1, &rej);

                cpl_table_set_double(detected_table,
                                     MOO_DETECTED_TABLE_FIT_FLUX, nb_detected,
                                     fit_flux);
                cpl_table_set_double(detected_table, MOO_DETECTED_TABLE_FIT_ERR,
                                     nb_detected, fit_err);
                cpl_table_set_int(detected_table, MOO_DETECTED_TABLE_BADFIT,
                                  nb_detected, 0);
                cpl_table_set_double(detected_table, MOO_DETECTED_TABLE_FIT_MSE,
                                     nb_detected, mse);
                cpl_table_set_double(detected_table,
                                     MOO_DETECTED_TABLE_FIT_CHI2, nb_detected,
                                     chi2);
                cpl_table_set_double(detected_table,
                                     MOO_DETECTED_TABLE_FIT_BACKGROUND,
                                     nb_detected, background);
                cpl_table_set_double(detected_table, MOO_LINE_TABLE_XGAUSS,
                                     nb_detected, centroid);
                double amplitude = area / sqrt(CPL_MATH_2PI * sigma * sigma);
                double fwhm = CPL_MATH_FWHM_SIG * sigma;
                double fit_snr = amplitude / fit_err;
                cpl_table_set_double(detected_table, MOO_LINE_TABLE_FWHM,
                                     nb_detected, fwhm);
                cpl_table_set_double(detected_table, MOO_LINE_TABLE_AMPLITUDE,
                                     nb_detected, amplitude);
                cpl_table_set_double(detected_table, MOO_LINE_TABLE_BACKGROUND,
                                     nb_detected, background);
                double width_bary = 0;
                double x_bary = 0;

                cpl_bivector *fitdata = cpl_bivector_wrap_vectors(xv, yv);
                moo_barycenter_fit(fitdata, &x_bary, &width_bary);
                cpl_bivector_unwrap_vectors(fitdata);
                cpl_table_set_double(detected_table, MOO_LINE_TABLE_XBARY,
                                     nb_detected, x_bary);

                if (sigma >= sigma_min && sigma <= sigma_max) {
                    if (fit_snr >= ref_snr && fit_flux > 0) {
                        cpl_table_set_int(detected_table,
                                          MOO_LINE_TABLE_FILTERED, nb_detected,
                                          0);
                        nb_fitted++;
                    }
                    else {
                        cpl_table_set_int(detected_table,
                                          MOO_LINE_TABLE_FILTERED, nb_detected,
                                          3);
                    }
                }
                else if (sigma < sigma_min) {
                    cpl_table_set_int(detected_table, MOO_LINE_TABLE_FILTERED,
                                      nb_detected, 1);
                }
                else {
                    cpl_table_set_int(detected_table, MOO_LINE_TABLE_FILTERED,
                                      nb_detected, 2);
                }
            }
            else {
                cpl_table_set_int(detected_table, MOO_DETECTED_TABLE_BADFIT,
                                  nb_detected, 1);
                cpl_errorstate_set(prev_state);
            }
        }
        cpl_matrix_delete(covariance);
        cpl_vector_delete(xv);
        cpl_vector_delete(yv);
        cpl_vector_delete(yerr);

        for (int i = min; i <= max; i++) {
            cpl_mask_set(imask, i, 1, CPL_BINARY_1);
        }
        nb_detected++;
    }
    cpl_table *ppm_table =
        _moo_do_ppm_match(waves, nb_catalog_lines, detected_table, numfib,
                          min_disp, max_disp, tolerance, direction, prefix);
    int nb_matched = 0;
    int ppm_table_nrow = 0;

    if (ppm_table != NULL) {
        ppm_table_nrow = cpl_table_get_nrow(ppm_table);
    }

    if (ppm_table_nrow <= 1) {
        cpl_msg_error("moo_wavesol",
                      "#%d-%d Pattern matching failed with in input %d peaks "
                      "fitted (%d peaks detected)",
                      slitlet_idx, numfib, nb_fitted, nb_detected);
    }
    else {
        nb_matched = ppm_table_nrow;
        *lline_table = moo_line_table_new(1);
        cpl_table_set_size(*lline_table, nb_catalog_lines);

        for (int i = 0; i < nb_catalog_lines; i++) {
            double nm = waves[i];
            cpl_table_set_double(*lline_table, MOO_LINE_TABLE_WAVE, i, nm);
            cpl_table_set_int(*lline_table, MOO_LINE_TABLE_INDEXEXT, i,
                              indexext_idx);
            cpl_table_set_int(*lline_table, MOO_LINE_TABLE_SLITLET, i,
                              slitlet_idx);
            cpl_table_set_int(*lline_table, MOO_LINE_TABLE_INDEXRBN, i, -1);
            cpl_table_set_double(*lline_table, MOO_LINE_TABLE_XGAUSS, i, NAN);
            cpl_table_set_string(*lline_table, MOO_LINE_TABLE_DETECTOR, i,
                                 extname);
            cpl_table_set_int(*lline_table, MOO_LINE_TABLE_SPECTRO, i, spectro);
            cpl_table_set_double(*lline_table, MOO_LINE_TABLE_INTENSITY, i,
                                 NAN);
            cpl_table_set_double(*lline_table, MOO_LINE_TABLE_XBARY, i, NAN);
            cpl_table_set_double(*lline_table, MOO_LINE_TABLE_AMPLITUDE, i,
                                 NAN);
            cpl_table_set_double(*lline_table, MOO_LINE_TABLE_BACKGROUND, i,
                                 NAN);
            cpl_table_set_double(*lline_table, MOO_LINE_TABLE_FWHM, i, NAN);
            cpl_table_set_int(*lline_table, MOO_LINE_TABLE_MATCHED, i, 0);
            cpl_table_set_int(*lline_table, MOO_LINE_TABLE_MFITTED, i, 0);
            cpl_table_set_double(*lline_table, MOO_LINE_TABLE_WAVEFIT, i, NAN);
            cpl_table_set_double(*lline_table, MOO_LINE_TABLE_RESOLUTION, i,
                                 NAN);
            cpl_table_set_double(*lline_table, MOO_LINE_TABLE_YLOC, i, NAN);
            cpl_table_set_double(*lline_table, MOO_LINE_TABLE_FLUX, i, NAN);
            cpl_table_set_string(*lline_table, MOO_LINE_TABLE_NAME, i, "");
        }

        for (int i = 0; i < nb_matched; i++) {
            double nm = cpl_table_get(ppm_table, MOO_PPM_TABLE_WAVE, i, NULL);
            double px = cpl_table_get(ppm_table, MOO_PPM_TABLE_X, i, NULL);
            double disp = cpl_table_get(ppm_table, MOO_PPM_TABLE_DISP, i, NULL);

            cpl_table_select_all(*lline_table);
            /* due to operation on ppm wave can do EQUAL_TO operation on wave */
            int size =
                cpl_table_and_selected_double(*lline_table, MOO_LINE_TABLE_WAVE,
                                              CPL_LESS_THAN, nm + 1E-5);
            size =
                cpl_table_and_selected_double(*lline_table, MOO_LINE_TABLE_WAVE,
                                              CPL_GREATER_THAN, nm - 1E-5);
            if (size == 1) {
                cpl_array *lline_indexes =
                    cpl_table_where_selected(*lline_table);
                cpl_size lline_idx =
                    cpl_array_get_cplsize(lline_indexes, 0, NULL);
                cpl_array_delete(lline_indexes);

                cpl_table_or_selected_double(detected_table,
                                             MOO_LINE_TABLE_XGAUSS,
                                             CPL_EQUAL_TO, px);
                cpl_table_set_int(*lline_table, MOO_LINE_TABLE_INDEXEXT,
                                  lline_idx, slitlet_idx);
                cpl_table_set_int(*lline_table, MOO_LINE_TABLE_INDEXEXT,
                                  lline_idx, indexext_idx);
                cpl_table_set_int(*lline_table, MOO_LINE_TABLE_INDEXRBN,
                                  lline_idx, -1);
                cpl_table_set_double(*lline_table, MOO_LINE_TABLE_XGAUSS,
                                     lline_idx, px);
                cpl_table_set_double(*lline_table, MOO_LINE_TABLE_DISP,
                                     lline_idx, disp);
                cpl_table_set_string(*lline_table, MOO_LINE_TABLE_DETECTOR,
                                     lline_idx, extname);
                cpl_table_set_int(*lline_table, MOO_LINE_TABLE_SPECTRO,
                                  lline_idx, spectro);
                cpl_table_unselect_all(detected_table);
                cpl_size detected_nb =
                    cpl_table_or_selected_double(detected_table,
                                                 MOO_LINE_TABLE_XGAUSS,
                                                 CPL_EQUAL_TO, px);
                if (detected_nb == 1) {
                    cpl_array *detected_indexes =
                        cpl_table_where_selected(detected_table);
                    cpl_size detected_idx =
                        cpl_array_get_cplsize(detected_indexes, 0, NULL);
                    double intensity =
                        cpl_table_get_double(detected_table,
                                             MOO_LINE_TABLE_INTENSITY,
                                             detected_idx, NULL);
                    double xbary = cpl_table_get_double(detected_table,
                                                        MOO_LINE_TABLE_XBARY,
                                                        detected_idx, NULL);
                    double amplitude =
                        cpl_table_get_double(detected_table,
                                             MOO_LINE_TABLE_AMPLITUDE,
                                             detected_idx, NULL);
                    double background =
                        cpl_table_get_double(detected_table,
                                             MOO_LINE_TABLE_BACKGROUND,
                                             detected_idx, NULL);
                    double fit_flux =
                        cpl_table_get_double(detected_table,
                                             MOO_LINE_TABLE_FIT_FLUX,
                                             detected_idx, NULL);
                    double fit_err =
                        cpl_table_get_double(detected_table,
                                             MOO_LINE_TABLE_FIT_ERR,
                                             detected_idx, NULL);
                    double fit_chi2 =
                        cpl_table_get_double(detected_table,
                                             MOO_LINE_TABLE_FIT_CHI2,
                                             detected_idx, NULL);
                    double fit_mse =
                        cpl_table_get_double(detected_table,
                                             MOO_LINE_TABLE_FIT_MSE,
                                             detected_idx, NULL);
                    double fwhm = cpl_table_get_double(detected_table,
                                                       MOO_LINE_TABLE_FWHM,
                                                       detected_idx, NULL);
                    cpl_table_set_double(*lline_table, MOO_LINE_TABLE_INTENSITY,
                                         lline_idx, intensity);
                    cpl_table_set_double(*lline_table, MOO_LINE_TABLE_XBARY,
                                         lline_idx, xbary);
                    cpl_table_set_double(*lline_table, MOO_LINE_TABLE_AMPLITUDE,
                                         lline_idx, amplitude);
                    cpl_table_set_double(*lline_table,
                                         MOO_LINE_TABLE_BACKGROUND, lline_idx,
                                         background);
                    cpl_table_set_double(*lline_table, MOO_LINE_TABLE_FWHM,
                                         lline_idx, fwhm);

                    cpl_table_set_double(*lline_table, MOO_LINE_TABLE_FIT_FLUX,
                                         lline_idx, fit_flux);
                    cpl_table_set_double(*lline_table, MOO_LINE_TABLE_FIT_ERR,
                                         lline_idx, fit_err);
                    cpl_table_set_double(*lline_table, MOO_LINE_TABLE_FIT_CHI2,
                                         lline_idx, fit_chi2);
                    cpl_table_set_double(*lline_table, MOO_LINE_TABLE_FIT_MSE,
                                         lline_idx, fit_mse);
                    cpl_table_set_int(*lline_table, MOO_LINE_TABLE_MATCHED,
                                      lline_idx, 1);
                    cpl_table_set_int(*lline_table, MOO_LINE_TABLE_MFITTED,
                                      lline_idx, 1);
                    cpl_array_delete(detected_indexes);
                }
            }
            else {
                cpl_msg_error("test", "the line %f is not find in LINE_TABLE",
                              nm);
            }
        }

        if (nb_matched <= degpoly) {
            cpl_msg_error("moo_wavesol",
                          "#%d-%d Not enough matched lines (%d) to fit "
                          "polynomial degree %d",
                          slitlet_idx, numfib, nb_matched, degpoly);
        }
        else {
            cpl_msg_debug("moo_wavesol",
                          "#%d-%d, %d peaks detected, %d fitted, %d matched / "
                          "%d catlines",
                          slitlet_idx, numfib, nb_detected, nb_fitted,
                          nb_matched, nb_catalog_lines);
        }
    }
    cpl_table_delete(ppm_table);
    *ldetected = nb_detected;
    *lfitted = nb_fitted;
    *lfailed = nb_failed;
    *lmatched = nb_matched;

moo_try_cleanup:
    cpl_image_delete(idata);
    cpl_image_delete(ierr);

    // dump error from the state
    if (!cpl_errorstate_is_equal(prestate)) {
        cpl_msg_error(__func__, "#%d Error in wavesol pattern matching",
                      numfib);
        cpl_errorstate_dump(prestate, CPL_FALSE, cpl_errorstate_dump_one);
        // Recover from the error(s) (Reset to prestate))
        cpl_errorstate_set(prestate);
    }

    return detected_table;
}

static cpl_error_code
_moo_wavecal_qc(cpl_propertylist *header, int nb_sigclip, cpl_table *line_table)
{
    cpl_table *extract_table = NULL;
    cpl_array *sel = NULL;
    cpl_vector *waveres = NULL;
    cpl_ensure_code(header != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(line_table != NULL, CPL_ERROR_NULL_INPUT);

    moo_qc_set_wavecal_clipline(header, nb_sigclip);

    cpl_table_select_all(line_table);
    cpl_table_and_selected_int(line_table, MOO_LINE_TABLE_MFITTED, CPL_EQUAL_TO,
                               1);
    extract_table = cpl_table_extract_selected(line_table);
    double med_resol =
        cpl_table_get_column_median(extract_table, MOO_LINE_TABLE_RESOLUTION);
    cpl_table_delete(extract_table);
    extract_table = NULL;
    moo_qc_set_wavecal_resol_med(header, med_resol);

    cpl_table_select_all(line_table);
    if (cpl_table_has_column(line_table, MOO_LINE_TABLE_FILTERED)) {
        cpl_table_and_selected_int(line_table, MOO_LINE_TABLE_FILTERED,
                                   CPL_EQUAL_TO, 0);
    }
    sel = cpl_table_where_selected(line_table);
    int size = cpl_array_get_size(sel);
    waveres = cpl_vector_new(size);
    for (int i = 0; i < size; i++) {
        int idx = cpl_array_get_cplsize(sel, i, NULL);
        double wave =
            cpl_table_get_double(line_table, MOO_LINE_TABLE_WAVE, idx, NULL);
        double wavefit =
            cpl_table_get_double(line_table, MOO_LINE_TABLE_WAVEFIT, idx, NULL);
        double res = wave - wavefit;
        cpl_vector_set(waveres, i, res);
    }
    double residwave_med = cpl_vector_get_median(waveres);
    double residwave_std = cpl_vector_get_stdev(waveres);
    moo_qc_set_wavecal_residwave_med(header, residwave_med);
    moo_qc_set_wavecal_residwave_std(header, residwave_std);

    cpl_table_delete(extract_table);
    cpl_vector_delete(waveres);
    cpl_array_delete(sel);
    return CPL_ERROR_NONE;
}

static cpl_image *
_moo_wavesol_single(moo_ext_single *ext,
                    moo_loc_single *loc,
                    cpl_array *indexes,
                    const int *health,
                    cpl_table *result_table,
                    cpl_table *catalog_lines,
                    double min_disp,
                    double max_disp,
                    double tolerance,
                    int direction,
                    cpl_image *guess,
                    cpl_propertylist *guess_header,
                    moo_map *wmap,
                    int nb_peaks_lines,
                    int winhsize,
                    int linefit_recentre,
                    int degx,
                    int degy,
                    moo_spectral_format_info *info,
                    double sigma_min,
                    double sigma_max,
                    double ref_snr,
                    cpl_propertylist *header,
                    moo_wavesol_params *params,
                    cpl_table **single_line_table)

{
    cpl_image *res = NULL;
    cpl_table **all_detected = NULL;
    cpl_table **all_fitted = NULL;
    cpl_ensure(ext != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(loc != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(health != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(params != NULL, CPL_ERROR_NULL_INPUT, NULL);

    const char *model = params->model;
    const char *linefit_method = params->linefit_method;
    const char *control = params->control;
    int niter = params->clip_niter;
    double max_frac = params->clip_frac;
    double kappa = params->clip_kappa;

    cpl_table *line_table = moo_line_table_new(guess == NULL);

    cpl_errorstate prestate = cpl_errorstate_get();
    moo_detector_type type = ext->type;

    cpl_image *data = moo_ext_single_get_data(ext);
    cpl_image *errs = moo_ext_single_get_errs(ext);

    int nb_fibres = cpl_image_get_size_y(data);
    cpl_size nb_index = cpl_array_get_size(indexes);

    cpl_ensure(nb_index == nb_fibres, CPL_ERROR_ILLEGAL_INPUT, NULL);

    int nx = cpl_image_get_size_x(data);

    res = cpl_image_new(nx, nb_fibres, CPL_TYPE_DOUBLE);

    all_detected = cpl_calloc(nb_fibres, sizeof(cpl_table *));
    all_fitted = cpl_calloc(nb_fibres, sizeof(cpl_table *));

    cpl_propertylist_append_double(header, MOO_PFITS_CRPIX1, MOO_MAP_CRPIX1);
    cpl_propertylist_append_double(header, MOO_PFITS_CRPIX2, MOO_MAP_CRPIX2);
    cpl_propertylist_append_double(header, MOO_PFITS_CRVAL1, MOO_MAP_CRVAL1);
    cpl_propertylist_append_double(header, MOO_PFITS_CRVAL2, MOO_MAP_CRVAL2);
    cpl_propertylist_append_double(header, MOO_PFITS_CD1_1, MOO_MAP_CD1_1);
    cpl_propertylist_append_double(header, MOO_PFITS_CD1_2, 0.);
    cpl_propertylist_append_double(header, MOO_PFITS_CD2_1, 0.);
    cpl_propertylist_append_double(header, MOO_PFITS_CD2_2, MOO_MAP_CD2_2);
    cpl_propertylist_append_string(header, MOO_PFITS_CTYPE1, MOO_MAP_CTYPE1);
    cpl_propertylist_append_string(header, MOO_PFITS_CTYPE2, MOO_MAP_CTYPE2);
    cpl_propertylist_append_string(header, MOO_PFITS_BUNIT, MOO_MAP_BUNIT);
    int catline = cpl_table_get_nrow(catalog_lines);
    const char *extname = ext->extname;
    int spectro = moo_detector_get_spectro(ext->ntas);

    int detectline = 0;
    int failedfit = 0;
    int fitline = 0;
    int matchline = 0;
    int validfibre = 0;
    const int *slitlet =
        cpl_table_get_data_int_const(result_table, MOO_FIBRES_TABLE_SLITLET);
    moo_try_assure(slitlet != NULL, CPL_ERROR_NULL_INPUT,
                   "No slitlet column in fibre table");

    const int *indexext =
        cpl_table_get_data_int_const(result_table, MOO_FIBRES_TABLE_INDEXEXT);
    if (guess != NULL) {
        cpl_msg_info("moo_wavesol", "Fit centroid lines using method %s",
                     linefit_method);
#ifdef _OPENMP
#pragma omp parallel default(none)                                       \
    shared(nb_fibres, indexes, health, result_table, type, all_detected, \
               all_fitted, extname, data, errs, guess, catalog_lines,    \
               winhsize, degx, sigma_min, sigma_max, linefit_method,     \
               line_table, indexext, slitlet) firstprivate(spectro)      \
    reduction(+ : detectline, failedfit, fitline, validfibre)
        {
#pragma omp for
#endif
            for (int i = 1; i <= nb_fibres; i++) {
                //  for(int i=20; i<=20; i++){
                int idx = cpl_array_get_cplsize(indexes, i - 1, NULL);
                int indexext_idx = indexext[idx];
                int slitlet_idx = slitlet[idx];
                int h = health[idx];
                if (h == 1) {
                    int ldetected = 0;
                    int lfailed = 0;
                    int lfitted = 0;
                    all_detected[i - 1] = _moo_wavesol_fibre_using_guess(
                        data, errs, guess, NULL, i, catalog_lines, winhsize,
                        sigma_min, sigma_max, slitlet_idx, indexext_idx,
                        extname, spectro, &all_fitted[i - 1], &ldetected,
                        &lfailed, &lfitted);
                    moo_fibres_table_set_detectline(result_table, type, idx,
                                                    ldetected);
                    moo_fibres_table_set_failedfit(result_table, type, idx,
                                                   lfailed);
                    moo_fibres_table_set_fitline(result_table, type, idx,
                                                 lfitted);
                    moo_fibres_table_set_rejectline(result_table, type, idx,
                                                    lfitted);
                    detectline += ldetected;
                    failedfit += lfailed;
                    fitline += lfitted;
                    validfibre++;
                }
                else {
                    all_detected[i - 1] = NULL;
                    all_fitted[i - 1] = NULL;
                }
            }
#ifdef _OPENMP
        }
#endif
        matchline = fitline;
    }
    else {
#ifdef _OPENMP
#pragma omp parallel default(none)                                            \
    shared(nb_fibres, indexes, health, result_table, type, all_detected,      \
               all_fitted, extname, data, errs, catalog_lines, min_disp,      \
               max_disp, tolerance, direction, nb_peaks_lines, winhsize, ext, \
               degx, sigma_min, sigma_max, ref_snr, indexext, slitlet)        \
    firstprivate(spectro)                                                     \
    reduction(+ : detectline, failedfit, fitline, matchline, validfibre)
        {
#pragma omp for
#endif
            for (int i = 1; i <= nb_fibres; i++) {
                int idx = cpl_array_get_cplsize(indexes, i - 1, NULL);
                int h = health[idx];
                int ldetected = 0;
                int lfailed = 0;
                int lfitted = 0;
                int lmatched = 0;
                int indexext_idx = indexext[idx];

                int slitlet_idx = slitlet[idx];
                if (h == 1) {
                    all_detected[i - 1] =
                        _moo_wavesol_fibre(data, errs, i, nb_peaks_lines,
                                           winhsize, sigma_min, sigma_max,
                                           ref_snr, catalog_lines, min_disp,
                                           max_disp, tolerance, direction,
                                           ext->extname, degx, slitlet_idx,
                                           indexext_idx, extname, spectro,
                                           &all_fitted[i - 1], &ldetected,
                                           &lfailed, &lfitted, &lmatched);

                    detectline += ldetected;
                    failedfit += lfailed;
                    fitline += lfitted;
                    matchline += lmatched;
                    validfibre++;
                }
                else {
                    cpl_msg_info("moo_wavesol",
                                 "Broken fibre indexext %d, slitlet %d",
                                 indexext_idx, slitlet_idx);

                    all_detected[i - 1] = NULL;
                }
                moo_fibres_table_set_detectline(result_table, type, idx,
                                                ldetected);
                moo_fibres_table_set_failedfit(result_table, type, idx,
                                               lfailed);

                moo_fibres_table_set_fitline(result_table, type, idx, lfitted);
                moo_fibres_table_set_matchline(result_table, type, idx,
                                               lmatched);
                moo_fibres_table_set_rejectline(result_table, type, idx,
                                                lmatched);
            }
#ifdef _OPENMP
        }
#endif
    }


#if MOO_DEBUG_WAVESOL_DETECTED
    {
        cpl_table *test = moo_detected_table_new(0, guess == NULL);
        for (int i = 0; i < nb_fibres; i++) {
            cpl_table *t = all_detected[i];
            if (t != NULL) {
                moo_table_append(test, t);
            }
        }
        char *name = NULL;

        if (guess == NULL) {
            name = cpl_sprintf("%s_ppm_detected.fits", ext->extname);
        }
        else {
            name = cpl_sprintf("%s_refit_lfwhs%d_detected.fits", ext->extname,
                               winhsize);
        }

        cpl_table_save(test, NULL, NULL, name, CPL_IO_CREATE);
        cpl_free(name);
        cpl_table_delete(test);
    }
#endif
    for (int i = 0; i < nb_fibres; i++) {
        cpl_table *t = all_fitted[i];

        if (t != NULL) {
            moo_table_append(line_table, t);
        }
    }
    moo_qc_set_wavecal_catline(header, catline);
    moo_qc_set_wavecal_detectline(header, detectline);
    moo_qc_set_wavecal_failedfit(header, failedfit);
    moo_qc_set_wavecal_fitline(header, fitline);
    moo_qc_set_wavecal_matchline(header, matchline);
    moo_qc_set_wavecal_validfibre(header, validfibre);

    int nb_sigclip = 0;

    cpl_table_erase_invalid(line_table);
    int nb_line_table = cpl_table_get_nrow(line_table);
    cpl_msg_info(__func__, "line table size : %d fitline %d matchline %d",
                 nb_line_table, fitline, matchline);
    const char *line_table_xfit = MOO_LINE_TABLE_XGAUSS;
    if (guess != NULL &&
        (strcmp(linefit_method, MOO_WAVESOL_LINEFIT_BARYCENTER) == 0)) {
        line_table_xfit = MOO_LINE_TABLE_XBARY;
    }

    for (int i = 0; i < nb_line_table; i++) {
        int rej;

        double x = cpl_table_get_double(line_table, line_table_xfit, i, NULL);

        int indexext_idx =
            cpl_table_get_int(line_table, MOO_LINE_TABLE_INDEXEXT, i, NULL);

        if (!isnan(x)) {
            double yc =
                moo_loc_single_eval_f_centroids(loc, x, indexext_idx, &rej);

            cpl_table_set_double(line_table, MOO_LINE_TABLE_YLOC, i, yc);
        }

        double nm =
            cpl_table_get_double(line_table, MOO_LINE_TABLE_WAVE, i, NULL);

        cpl_table_unselect_all(catalog_lines);
        cpl_table_or_selected_double(catalog_lines, MOO_LINE_TABLE_WAVE,
                                     CPL_GREATER_THAN, nm - 1E-5);
        cpl_size cat_nb =
            cpl_table_and_selected_double(catalog_lines, MOO_LINE_TABLE_WAVE,
                                          CPL_LESS_THAN, nm + 1E-5);

        if (cat_nb == 1) {
            cpl_array *cat_indexes = cpl_table_where_selected(catalog_lines);
            cpl_size cat_idx = cpl_array_get_cplsize(cat_indexes, 0, NULL);
            cpl_array_delete(cat_indexes);
            double cat_flux =
                cpl_table_get_double(catalog_lines, MOO_LINE_TABLE_FLUX,
                                     cat_idx, NULL);
            const char *cat_name =
                cpl_table_get_string(catalog_lines, MOO_LINE_TABLE_NAME,
                                     cat_idx);
            cpl_table_set_string(line_table, MOO_LINE_TABLE_NAME, i, cat_name);
            cpl_table_set_double(line_table, MOO_LINE_TABLE_FLUX, i, cat_flux);
        }
    }

#if MOO_DEBUG_WAVESOL_DETECTED
    {
        char *name = NULL;
        if (guess == NULL) {
            name = cpl_sprintf("%s_ppm_lines.fits", ext->extname);
        }
        else {
            name = cpl_sprintf("%s_refit_lfwhs%d_lines.fits", ext->extname,
                               winhsize);
        }
        cpl_table_save(line_table, NULL, NULL, name, CPL_IO_CREATE);
        cpl_free(name);
    }
#endif
    if (cpl_table_has_column(line_table, MOO_LINE_TABLE_DISP)) {
        double disp_min =
            cpl_table_get_column_min(line_table, MOO_LINE_TABLE_DISP);
        double disp_max =
            cpl_table_get_column_max(line_table, MOO_LINE_TABLE_DISP);
        moo_qc_set_wavecal_disp_min(header, disp_min);
        moo_qc_set_wavecal_disp_max(header, disp_max);
    }
    else if (guess_header != NULL) {
        cpl_propertylist_copy_property(header, guess_header,
                                       MOONS_QC_WAVECAL_DISP_MIN);
        cpl_propertylist_copy_property(header, guess_header,
                                       MOONS_QC_WAVECAL_DISP_MAX);
    }

    if (guess != NULL && linefit_recentre) {
        cpl_msg_info("moo_wavesol", "RECENTRE : compute slitlet shift with %d",
                     catline);
        int i = 1;
        int slit_idx = -1;
        cpl_table *slitlet_table_global = moo_slitlet_shift_table_new(0);

        while (i <= nb_fibres) {
            int idx = cpl_array_get_cplsize(indexes, i - 1, NULL);

            if (slit_idx != slitlet[idx]) {
                cpl_table *slitlet_table = moo_slitlet_shift_table_new(catline);
                slit_idx = slitlet[idx];

                for (int j = 0; j < catline; j++) {
                    double w =
                        cpl_table_get_double(catalog_lines,
                                             MOO_ARCLINE_LIST_WAVE, j, NULL);
                    cpl_table_set_int(slitlet_table,
                                      MOO_SLITLET_SHIFT_TABLE_SLITLET, j,
                                      slit_idx);
                    cpl_table_set_double(slitlet_table,
                                         MOO_SLITLET_SHIFT_TABLE_WAVE, j, w);
                    cpl_table_select_all(line_table);
                    cpl_table_and_selected_int(line_table,
                                               MOO_LINE_TABLE_SLITLET,
                                               CPL_EQUAL_TO, slit_idx);
                    cpl_table_and_selected_int(line_table,
                                               MOO_LINE_TABLE_FILTERED,
                                               CPL_EQUAL_TO, 0);
                    int nblines =
                        cpl_table_and_selected_double(line_table,
                                                      MOO_LINE_TABLE_WAVE,
                                                      CPL_EQUAL_TO, w);
                    double median = NAN;
                    if (nblines > 0) {
                        cpl_table *sel = cpl_table_extract_selected(line_table);
                        median =
                            cpl_table_get_column_median(sel,
                                                        MOO_LINE_TABLE_XDIFF);
                        cpl_table_delete(sel);
                    }
                    cpl_table_set_double(slitlet_table,
                                         MOO_SLITLET_SHIFT_TABLE_XDIFF, j,
                                         median);
                }
                moo_table_append(slitlet_table_global, slitlet_table);
                cpl_table_delete(slitlet_table);
            }
            i++;
        }

        int min_slitlet =
            cpl_table_get_column_min(slitlet_table_global,
                                     MOO_SLITLET_SHIFT_TABLE_SLITLET);
        int max_slitlet =
            cpl_table_get_column_max(slitlet_table_global,
                                     MOO_SLITLET_SHIFT_TABLE_SLITLET);
        double central_slitlet = (min_slitlet + max_slitlet) / 2.;
        int lo_central_slitlet = floor(central_slitlet) - 1;
        int up_central_slitlet = ceil(central_slitlet) + 1;

        cpl_vector *vx = cpl_vector_new(catline);
        cpl_vector *vy = cpl_vector_new(catline);
        int vsize = 0;

        for (int j = 0; j < catline; j++) {
            double w = cpl_table_get_double(catalog_lines,
                                            MOO_ARCLINE_LIST_WAVE, j, NULL);
            cpl_table_select_all(line_table);
            cpl_table_and_selected_int(line_table, MOO_LINE_TABLE_SLITLET,
                                       CPL_GREATER_THAN, lo_central_slitlet);
            cpl_table_and_selected_int(line_table, MOO_LINE_TABLE_SLITLET,
                                       CPL_LESS_THAN, up_central_slitlet);
            cpl_table_and_selected_int(line_table, MOO_LINE_TABLE_FILTERED,
                                       CPL_EQUAL_TO, 0);
            int nblines =
                cpl_table_and_selected_double(line_table, MOO_LINE_TABLE_WAVE,
                                              CPL_EQUAL_TO, w);
            if (nblines > 0) {
                cpl_table *sel = cpl_table_extract_selected(line_table);
                double median =
                    cpl_table_get_column_median(sel, MOO_LINE_TABLE_XDIFF);
                cpl_table_delete(sel);
                cpl_vector_set(vx, vsize, w);
                cpl_vector_set(vy, vsize, median);
                vsize++;
            }
        }
        cpl_vector_set_size(vx, vsize);
        cpl_vector_set_size(vy, vsize);
        double wmin = cpl_vector_get(vx, 0);
        double wmin_xdiff = cpl_vector_get(vy, 0);
        double wmax = cpl_vector_get(vx, vsize - 1);
        double wmax_xdiff = cpl_vector_get(vy, vsize - 1);
        cpl_bivector *fref = cpl_bivector_wrap_vectors(vx, vy);
        cpl_bivector *fout = cpl_bivector_new(1);
        for (int j = 0; j < catline; j++) {
            double w = cpl_table_get_double(catalog_lines,
                                            MOO_ARCLINE_LIST_WAVE, j, NULL);
            double median;
            if (w < wmin) {
                median = wmin_xdiff;
            }
            else if (w > wmax) {
                median = wmax_xdiff;
            }
            else {
                cpl_vector *fout_x = cpl_bivector_get_x(fout);
                cpl_vector_set(fout_x, 0, w);
                cpl_vector *fout_y = cpl_bivector_get_y(fout);
                cpl_bivector_interpolate_linear(fout, fref);
                median = cpl_vector_get(fout_y, 0);
            }
            cpl_table_select_all(slitlet_table_global);
            cpl_table_and_selected_int(slitlet_table_global,
                                       MOO_SLITLET_SHIFT_TABLE_SLITLET,
                                       CPL_EQUAL_TO, lo_central_slitlet + 1);
            cpl_table_and_selected_double(slitlet_table_global,
                                          MOO_LINE_TABLE_WAVE, CPL_EQUAL_TO, w);
            cpl_array *sel = cpl_table_where_selected(slitlet_table_global);
            int idx = cpl_array_get_cplsize(sel, 0, NULL);
            cpl_array_delete(sel);
            cpl_table_set_double(slitlet_table_global,
                                 MOO_SLITLET_SHIFT_TABLE_XDIFF, idx, median);

            cpl_table_select_all(slitlet_table_global);
            cpl_table_and_selected_int(slitlet_table_global,
                                       MOO_SLITLET_SHIFT_TABLE_SLITLET,
                                       CPL_EQUAL_TO, up_central_slitlet - 1);
            cpl_table_and_selected_double(slitlet_table_global,
                                          MOO_LINE_TABLE_WAVE, CPL_EQUAL_TO, w);
            sel = cpl_table_where_selected(slitlet_table_global);
            idx = cpl_array_get_cplsize(sel, 0, NULL);
            cpl_array_delete(sel);
            cpl_table_set_double(slitlet_table_global,
                                 MOO_SLITLET_SHIFT_TABLE_XDIFF, idx, median);
        }
        cpl_bivector_delete(fout);
        cpl_bivector_unwrap_vectors(fref);
        cpl_vector_delete(vx);
        cpl_vector_delete(vy);
#if MOO_DEBUG_WAVESOL_DETECTED
        {
            char *name = cpl_sprintf("%s_refit%d_slitlet.fits", ext->extname,
                                     params->isrefit);
            cpl_table_save(slitlet_table_global, NULL, NULL, name,
                           CPL_IO_CREATE);
            cpl_free(name);
        }
#endif

        cpl_table *detected_global_step2 =
            moo_detected_table_new(0, guess == NULL);
        cpl_table *lines_global_step2 = moo_line_table_new(0);

        for (int s = lo_central_slitlet + 1; s >= min_slitlet; s--) {
            cpl_table_select_all(result_table);
            int nb = cpl_table_and_selected_int(result_table,
                                                MOO_FIBRES_TABLE_SLITLET,
                                                CPL_EQUAL_TO, s);
            nb = cpl_table_and_selected_int(result_table,
                                            MOO_FIBRES_TABLE_HEALTH,
                                            CPL_NOT_EQUAL_TO, 0);
            nb = cpl_table_and_selected_int(result_table,
                                            MOO_FIBRES_TABLE_SPECTRO,
                                            CPL_EQUAL_TO, spectro);
            cpl_msg_info("moo_wavesol", "do slitlet %d : MOONS%d %d", s,
                         spectro, nb);
            cpl_array *slitlet_sel = cpl_table_where_selected(result_table);
            for (int isel = 0; isel < nb; isel++) {
                int idx = cpl_array_get_cplsize(slitlet_sel, isel, NULL);
                int indexext_idx =
                    cpl_table_get_int(result_table, MOO_FIBRES_TABLE_INDEXEXT,
                                      idx, NULL);
                int slitlet_idx =
                    cpl_table_get_int(result_table, MOO_FIBRES_TABLE_SLITLET,
                                      idx, NULL);
                int ldetected = 0;
                int lfailed = 0;
                int lfitted = 0;

                cpl_table *tfitted = NULL;
                cpl_table *tdetected = _moo_wavesol_fibre_using_guess(
                    data, errs, guess, slitlet_table_global, indexext_idx,
                    catalog_lines, winhsize, sigma_min, sigma_max, slitlet_idx,
                    indexext_idx, extname, spectro, &tfitted, &ldetected,
                    &lfailed, &lfitted);
                moo_table_append(detected_global_step2, tdetected);
                moo_table_append(lines_global_step2, tfitted);
                cpl_table_delete(tdetected);
                cpl_table_delete(tfitted);
                /*
          moo_fibres_table_set_detectline(result_table,type,idx,ldetected);
          moo_fibres_table_set_failedfit(result_table,type,idx,lfailed);
          moo_fibres_table_set_fitline(result_table,type,idx,lfitted);
          moo_fibres_table_set_rejectline(result_table,type,idx,lfitted);
          detectline +=ldetected;
          failedfit += lfailed;
          fitline += lfitted;
          */
            }
            cpl_array_delete(slitlet_sel);

            for (int j = 0; j < catline; j++) {
                double w = cpl_table_get_double(catalog_lines,
                                                MOO_ARCLINE_LIST_WAVE, j, NULL);
                cpl_table_select_all(lines_global_step2);
                int nblines = cpl_table_and_selected_int(lines_global_step2,
                                                         MOO_LINE_TABLE_SLITLET,
                                                         CPL_EQUAL_TO, s);

                nblines = cpl_table_and_selected_int(lines_global_step2,
                                                     MOO_LINE_TABLE_FILTERED,
                                                     CPL_EQUAL_TO, 0);
                nblines = cpl_table_and_selected_double(lines_global_step2,
                                                        MOO_LINE_TABLE_WAVE,
                                                        CPL_EQUAL_TO, w);
                double median = NAN;
                if (nblines > 0) {
                    cpl_table *sel =
                        cpl_table_extract_selected(lines_global_step2);
                    median =
                        cpl_table_get_column_median(sel, MOO_LINE_TABLE_XDIFF);
                    cpl_table_delete(sel);
                }
                cpl_table_select_all(slitlet_table_global);
                cpl_table_and_selected_int(slitlet_table_global,
                                           MOO_SLITLET_SHIFT_TABLE_SLITLET,
                                           CPL_EQUAL_TO, s);
                cpl_table_and_selected_double(slitlet_table_global,
                                              MOO_LINE_TABLE_WAVE, CPL_EQUAL_TO,
                                              w);
                cpl_array *sel = cpl_table_where_selected(slitlet_table_global);
                int idx = cpl_array_get_cplsize(sel, 0, NULL);
                cpl_array_delete(sel);
                cpl_table_set_double(slitlet_table_global,
                                     MOO_SLITLET_SHIFT_TABLE_XDIFF, idx,
                                     median);
            }
        }
        for (int s = up_central_slitlet - 1; s <= max_slitlet; s++) {
            cpl_table_select_all(result_table);
            int nb = cpl_table_and_selected_int(result_table,
                                                MOO_FIBRES_TABLE_SLITLET,
                                                CPL_EQUAL_TO, s);
            nb = cpl_table_and_selected_int(result_table,
                                            MOO_FIBRES_TABLE_HEALTH,
                                            CPL_NOT_EQUAL_TO, 0);
            nb = cpl_table_and_selected_int(result_table,
                                            MOO_FIBRES_TABLE_SPECTRO,
                                            CPL_EQUAL_TO, spectro);
            cpl_array *slitlet_sel = cpl_table_where_selected(result_table);
            cpl_msg_info("moo_wavesol", "do slitlet %d : MOONS%d %d", s,
                         spectro, nb);
            for (int isel = 0; isel < nb; isel++) {
                int idx = cpl_array_get_cplsize(slitlet_sel, isel, NULL);
                int indexext_idx =
                    cpl_table_get_int(result_table, MOO_FIBRES_TABLE_INDEXEXT,
                                      idx, NULL);
                int slitlet_idx =
                    cpl_table_get_int(result_table, MOO_FIBRES_TABLE_SLITLET,
                                      idx, NULL);
                int ldetected = 0;
                int lfailed = 0;
                int lfitted = 0;

                cpl_table *tfitted = NULL;
                cpl_table *tdetected = _moo_wavesol_fibre_using_guess(
                    data, errs, guess, slitlet_table_global, indexext_idx,
                    catalog_lines, winhsize, sigma_min, sigma_max, slitlet_idx,
                    indexext_idx, extname, spectro, &tfitted, &ldetected,
                    &lfailed, &lfitted);
                moo_table_append(detected_global_step2, tdetected);
                moo_table_append(lines_global_step2, tfitted);
                cpl_table_delete(tdetected);
                cpl_table_delete(tfitted);
                /*
          moo_fibres_table_set_detectline(result_table,type,idx,ldetected);
          moo_fibres_table_set_failedfit(result_table,type,idx,lfailed);
          moo_fibres_table_set_fitline(result_table,type,idx,lfitted);
          moo_fibres_table_set_rejectline(result_table,type,idx,lfitted);
          detectline +=ldetected;
          failedfit += lfailed;
          fitline += lfitted;
          */
            }
            cpl_array_delete(slitlet_sel);
            for (int j = 0; j < catline; j++) {
                double w = cpl_table_get_double(catalog_lines,
                                                MOO_ARCLINE_LIST_WAVE, j, NULL);
                cpl_table_select_all(lines_global_step2);
                cpl_table_and_selected_int(lines_global_step2,
                                           MOO_LINE_TABLE_SLITLET, CPL_EQUAL_TO,
                                           s);
                cpl_table_and_selected_int(lines_global_step2,
                                           MOO_LINE_TABLE_FILTERED,
                                           CPL_EQUAL_TO, 0);
                int nblines = cpl_table_and_selected_double(lines_global_step2,
                                                            MOO_LINE_TABLE_WAVE,
                                                            CPL_EQUAL_TO, w);
                double median = NAN;
                if (nblines > 0) {
                    cpl_table *sel =
                        cpl_table_extract_selected(lines_global_step2);
                    median =
                        cpl_table_get_column_median(sel, MOO_LINE_TABLE_XDIFF);
                    cpl_table_delete(sel);
                }
                cpl_table_select_all(slitlet_table_global);
                cpl_table_and_selected_int(slitlet_table_global,
                                           MOO_SLITLET_SHIFT_TABLE_SLITLET,
                                           CPL_EQUAL_TO, s);
                cpl_table_and_selected_double(slitlet_table_global,
                                              MOO_LINE_TABLE_WAVE, CPL_EQUAL_TO,
                                              w);
                cpl_array *sel = cpl_table_where_selected(slitlet_table_global);
                int idx = cpl_array_get_cplsize(sel, 0, NULL);
                cpl_array_delete(sel);
                cpl_table_set_double(slitlet_table_global,
                                     MOO_SLITLET_SHIFT_TABLE_XDIFF, idx,
                                     median);
            }
        }
#if MOO_DEBUG_WAVESOL_DETECTED
        {
            char *name = cpl_sprintf("%s_slitlet_step2.fits", ext->extname);
            cpl_table_save(slitlet_table_global, NULL, NULL, name,
                           CPL_IO_CREATE);
            cpl_free(name);
        }
        {
            char *name = cpl_sprintf("%s_refit%d_slitlet_step2.fits",
                                     ext->extname, params->isrefit);
            cpl_table_save(detected_global_step2, NULL, NULL, name,
                           CPL_IO_CREATE);
            cpl_free(name);
        }
        {
            char *name = cpl_sprintf("%s_lines_step2.fits", ext->extname);
            cpl_table_save(lines_global_step2, NULL, NULL, name, CPL_IO_CREATE);
            cpl_free(name);
        }
#endif
        cpl_table_delete(line_table);
        line_table = lines_global_step2;
        nb_line_table = cpl_table_get_nrow(line_table);

        for (i = 0; i < nb_line_table; i++) {
            int rej;
            double x =
                cpl_table_get_double(line_table, line_table_xfit, i, NULL);
            int indexext_idx =
                cpl_table_get_int(line_table, MOO_LINE_TABLE_INDEXEXT, i, NULL);
            if (!isnan(x)) {
                double yc =
                    moo_loc_single_eval_f_centroids(loc, x, indexext_idx, &rej);
                cpl_table_set_double(line_table, MOO_LINE_TABLE_YLOC, i, yc);
            }
            double nm =
                cpl_table_get_double(line_table, MOO_LINE_TABLE_WAVE, i, NULL);
            cpl_table_unselect_all(catalog_lines);
            cpl_table_or_selected_double(catalog_lines, MOO_LINE_TABLE_WAVE,
                                         CPL_GREATER_THAN, nm - 1E-5);
            cpl_size cat_nb =
                cpl_table_and_selected_double(catalog_lines,
                                              MOO_LINE_TABLE_WAVE,
                                              CPL_LESS_THAN, nm + 1E-5);

            if (cat_nb == 1) {
                cpl_array *cat_indexes =
                    cpl_table_where_selected(catalog_lines);
                cpl_size cat_idx = cpl_array_get_cplsize(cat_indexes, 0, NULL);
                cpl_array_delete(cat_indexes);
                double cat_flux =
                    cpl_table_get_double(catalog_lines, MOO_LINE_TABLE_FLUX,
                                         cat_idx, NULL);
                const char *cat_name =
                    cpl_table_get_string(catalog_lines, MOO_LINE_TABLE_NAME,
                                         cat_idx);
                cpl_table_set_string(line_table, MOO_LINE_TABLE_NAME, i,
                                     cat_name);
                cpl_table_set_double(line_table, MOO_LINE_TABLE_FLUX, i,
                                     cat_flux);
            }
        }
        cpl_table_delete(slitlet_table_global);
        cpl_table_delete(detected_global_step2);

        for (int f = 1; f <= nb_fibres; f++) {
            int idx = cpl_array_get_cplsize(indexes, f - 1, NULL);
            int h = health[idx];
            int indexext_idx = indexext[idx];
            if (h == 1) {
                cpl_table_select_all(line_table);
                cpl_table_and_selected_int(line_table, MOO_LINE_TABLE_INDEXEXT,
                                           CPL_EQUAL_TO, indexext_idx);
                cpl_size nb =
                    cpl_table_and_selected_int(line_table,
                                               MOO_LINE_TABLE_FILTERED,
                                               CPL_EQUAL_TO, 0);
                if (nb > 0) {
                    cpl_table *sel = cpl_table_extract_selected(line_table);
                    double xdiff =
                        cpl_table_get_column_median(sel, MOO_LINE_TABLE_XDIFF);
                    cpl_table_delete(sel);
                    moo_fibres_table_set_xdiff(result_table, type, idx, xdiff);
                }
            }
        }
    }

    if ((strcmp(control, MOO_WAVESOL_CONTROL_CHECK) == 0) ||
        (strcmp(control, MOO_WAVESOL_CONTROL_UPDATE) == 0)) {
        cpl_table *all_guess_line_table = moo_map_get_line_table(wmap);
        if (!cpl_table_has_column(all_guess_line_table, MOO_LINE_TABLE_XDIFF)) {
            cpl_table_new_column(all_guess_line_table, MOO_LINE_TABLE_XDIFF,
                                 CPL_TYPE_DOUBLE);
        }
        if (!cpl_table_has_column(all_guess_line_table,
                                  MOO_LINE_TABLE_WAVEDIFF)) {
            cpl_table_new_column(all_guess_line_table, MOO_LINE_TABLE_WAVEDIFF,
                                 CPL_TYPE_DOUBLE);
        }
        cpl_table_select_all(all_guess_line_table);
        cpl_table_and_selected_string(all_guess_line_table,
                                      MOO_LINE_TABLE_DETECTOR, CPL_EQUAL_TO,
                                      ext->extname);

        cpl_table *guess_line_table =
            cpl_table_extract_selected(all_guess_line_table);
        for (int i = 1; i <= nb_fibres; i++) {
            int idx = cpl_array_get_cplsize(indexes, i - 1, NULL);
            int h = health[idx];
            int indexext_idx = indexext[idx];
            if (h == 1) {
                int rej;
                double xdiff =
                    moo_fibres_table_get_xdiff(result_table, type, idx);
                cpl_table_select_all(guess_line_table);
                cpl_table_and_selected_int(guess_line_table,
                                           MOO_LINE_TABLE_INDEXEXT,
                                           CPL_EQUAL_TO, indexext_idx);
                cpl_size nb =
                    cpl_table_and_selected_int(guess_line_table,
                                               MOO_LINE_TABLE_FILTERED,
                                               CPL_EQUAL_TO, 0);
                if (nb > 0) {
                    cpl_array *sel = cpl_table_where_selected(guess_line_table);
                    for (int j = 0; j < nb; j++) {
                        int sel_idx = cpl_array_get_cplsize(sel, j, NULL);
                        double xgauss =
                            cpl_table_get_double(guess_line_table,
                                                 MOO_LINE_TABLE_XGAUSS, sel_idx,
                                                 &rej);
                        cpl_table_set_double(guess_line_table,
                                             MOO_LINE_TABLE_XGAUSS, sel_idx,
                                             xgauss - xdiff);
                    }
                    cpl_array_delete(sel);
                }
            }
        }
        moo_table_append(line_table, guess_line_table);
        cpl_table_select_all(line_table);
        cpl_size nb = cpl_table_and_selected_int(guess_line_table,
                                                 MOO_LINE_TABLE_FILTERED,
                                                 CPL_EQUAL_TO, 0);
        if (nb > 0) {
            cpl_array *sel = cpl_table_where_selected(line_table);
            for (int j = 0; j < nb; j++) {
                int sel_idx = cpl_array_get_cplsize(sel, j, NULL);
                cpl_table_set_int(line_table, MOO_LINE_TABLE_MFITTED, sel_idx,
                                  0);
            }
            cpl_array_delete(sel);
        }
        cpl_table_delete(guess_line_table);
    }

    const char *line_table_gfit = MOO_LINE_TABLE_MATCHED;
    if (guess != NULL) {
        line_table_gfit = MOO_LINE_TABLE_GFITTED;
    }

    if (strcmp(model, MOO_WAVESOL_MODEL_1D) == 0) {
        cpl_msg_info("moo_wavesol", "Do Polynomial %s fit with degree %d",
                     model, degx);
        for (int i = 1; i <= nb_fibres; i++) {
            int idx = cpl_array_get_cplsize(indexes, i - 1, NULL);
            int h = health[idx];
            if (h == 1) {
                _moo_wavesol_fit1d(loc, degx, i, res, niter, max_frac, kappa,
                                   &nb_sigclip, result_table, idx, ext->type,
                                   line_table, line_table_xfit);
            }
            else {
                for (int x = 1; x <= nx; x++) {
                    cpl_image_set(res, x, i, NAN);
                }
            }
        }
    }
    else {
        cpl_msg_info("moo_wavesol",
                     "Do Polynomial %s fit with X degree %d and Y degree %d",
                     model, degx, degy);
        int i = 1;
        int slit_idx = -1;
        moo_tcheby2d_polynomial *sol = NULL;
        while (i <= nb_fibres) {
            int idx = cpl_array_get_cplsize(indexes, i - 1, NULL);
            int h = health[idx];
            if (h == 1) {
                if (slit_idx != slitlet[idx]) {
                    moo_tcheby2d_polynomial_delete(sol);
                    slit_idx = slitlet[idx];
                    cpl_msg_debug("moo_wavesol", "Do slitlet %d :", slit_idx);
                    cpl_errorstate prestate_fit2d = cpl_errorstate_get();
                    sol = _moo_wavesol_fit2d(loc, res, info, degx, degy, niter,
                                             max_frac, kappa, &nb_sigclip,
                                             slit_idx, line_table,
                                             line_table_xfit, line_table_gfit);
                    if (!cpl_errorstate_is_equal(prestate_fit2d)) {
                        moo_tcheby2d_polynomial_delete(sol);
                        sol = NULL;
                        cpl_errorstate_dump(prestate_fit2d, CPL_FALSE,
                                            cpl_errorstate_dump_one);
                        cpl_errorstate_set(prestate_fit2d);
                    }
                }

                cpl_image *ycentroids = moo_loc_single_get_f_centroids(loc);
                for (int x = 1; x <= nx; x++) {
                    int rej;

                    double y =
                        cpl_image_get(ycentroids, x, indexext[idx], &rej);

                    if (rej == 1) {
                        cpl_image_set(res, x, i, NAN);
                    }
                    else {
                        double v = NAN;
                        if (sol != NULL) {
                            v = moo_tcheby2d_polynomial_eval(sol, x, y);
                        }
                        cpl_image_set(res, x, i, v);
                    }
                }
            }
            else {
                for (int x = 1; x <= nx; x++) {
                    cpl_image_set(res, x, i, NAN);
                }
            }
            i++;
        }
        moo_tcheby2d_polynomial_delete(sol);

        for (i = 1; i < nb_fibres; i++) {
            cpl_table_select_all(line_table);
            cpl_table_and_selected_int(line_table, MOO_LINE_TABLE_MFITTED,
                                       CPL_EQUAL_TO, 0);
            cpl_size nblines =
                cpl_table_and_selected_int(line_table, MOO_LINE_TABLE_INDEXEXT,
                                           CPL_EQUAL_TO, i);
            int idx = cpl_array_get_cplsize(indexes, i - 1, NULL);
            moo_fibres_table_set_rejectline(result_table, type, idx, nblines);
        }
    }
    // QC compute
    _moo_wavecal_qc(header, nb_sigclip, line_table);

    *single_line_table = line_table;

moo_try_cleanup:
    if (all_detected != NULL) {
        for (int i = 0; i < nb_fibres; i++) {
            cpl_table_delete(all_detected[i]);
        }
        cpl_free(all_detected);
    }
    if (all_fitted != NULL) {
        for (int i = 0; i < nb_fibres; i++) {
            cpl_table_delete(all_fitted[i]);
        }
        cpl_free(all_fitted);
    }
    if (!cpl_errorstate_is_equal(prestate)) {
        cpl_image_delete(res);
        res = NULL;
        cpl_msg_error(__func__, "Error in wavesol for file %s", ext->filename);
        cpl_errorstate_dump(prestate, CPL_FALSE, cpl_errorstate_dump_one);
        // Recover from the error(s) (Reset to prestate))
        cpl_errorstate_set(prestate);
    }
    return res;
}

static cpl_table *
_arclines_get_caliblines(cpl_table *table,
                         moo_detector_type type,
                         int isGuess,
                         int isrefit)
{
    cpl_table *result = NULL;
    int size = 0;

    cpl_errorstate prestate = cpl_errorstate_get();

    cpl_ensure(table != NULL, CPL_ERROR_NULL_INPUT, NULL);

    const char *typename = moo_detector_get_name(type);
    const char *wavecaltype = MOO_ARCLINE_LIST_USEFINAL;
    if (isGuess) {
        wavecaltype = MOO_ARCLINE_LIST_USEGUESS;
    }
    else if (isrefit) {
        wavecaltype = MOO_ARCLINE_LIST_USEGUESS_REFIT;
    }

    cpl_table_select_all(table);
    moo_try_check(size = cpl_table_and_selected_string(table,
                                                       MOO_ARCLINE_LIST_BAND,
                                                       CPL_EQUAL_TO, typename),
                  " ");
    moo_try_check(size = cpl_table_and_selected_int(table, wavecaltype,
                                                    CPL_EQUAL_TO, 1),
                  " ");

    if (size == 0) {
        cpl_msg_warning(__func__, "ARC_LINE_LIST %s no arclines found",
                        typename);
    }
    else {
        result = cpl_table_extract_selected(table);
    }
moo_try_cleanup:
    if (!cpl_errorstate_is_equal(prestate)) {
        cpl_table_delete(result);
        result = NULL;
    }
    return result;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Computes the wavelength solution from 1D extracted arc spectra
  @param    arc_ext _EXT_ Extracted 1D arc spectra
  @param    lineCat Line list frame
  @param    sformat the SPECTRAL FORMAT
  @param    loc _LOC_ localization trace
  @param    wmap the WAVE MAP GUESS if a preliminary solution is alrezady computed
  @param    params the wavesol parameters
  @return   the _EXT_ spectra

 * This function computes the wavelength solution. It first determine the (X,Y)
 * positions of a set of arc lines given as input, and then creates a wavelength
 * map for each detector extensions in the input frame.

 * _Bad pixels flags_:
  - None

 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
static moo_map *
_moo_wavesol_refit(moo_ext *arc_ext,
                   const char *lineCatname,
                   cpl_table *lines,
                   moo_spectral_format *sformat,
                   moo_loc *loc,
                   moo_map *wmap,
                   moo_wavesol_params *params)
{
    moo_map *result = NULL;
    cpl_array *indexes = NULL;
    int *wavemap_degx = NULL;
    int wavemap_degy = 0;
    int *line_winhsize = NULL;
    cpl_image *guess_tab[6];
    cpl_propertylist *guess_headers[6];

    cpl_errorstate prestate = cpl_errorstate_get();
    cpl_ensure(arc_ext != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(lineCatname != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(sformat != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(loc != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(params != NULL, CPL_ERROR_NULL_INPUT, NULL);

    unsigned int badpix_level =
        MOO_BADPIX_COSMETIC | MOO_BADPIX_HOT | MOO_BADPIX_OUTSIDE_DATA_RANGE;

    cpl_table *fibres_table = moo_ext_get_fibre_table(arc_ext);
    cpl_ensure(fibres_table != NULL, CPL_ERROR_ILLEGAL_INPUT, NULL);

    result = moo_map_new();
    result->primary_header = cpl_propertylist_new();
    moo_try_check(cpl_propertylist_append_string(result->primary_header,
                                                 MOO_PFITS_PRO_WAVESOL_MODEL,
                                                 params->model),
                  " ");
    char *degx_tag = NULL;
    degx_tag = cpl_sprintf("%d,%d,%d", params->wavemap_degx[0],
                           params->wavemap_degx[1], params->wavemap_degx[2]);
    moo_try_check(cpl_propertylist_append_string(result->primary_header,
                                                 MOO_PFITS_PRO_WAVESOL_DEGX,
                                                 degx_tag),
                  " ");
    cpl_free(degx_tag);
    if (strcmp(params->model, MOO_WAVESOL_MODEL_2D) == 0) {
        moo_try_check(cpl_propertylist_append_int(result->primary_header,
                                                  MOO_PFITS_PRO_WAVESOL_DEGY,
                                                  params->wavemap_degy),
                      " ");
    }
    cpl_table *result_table = cpl_table_duplicate(fibres_table);

    moo_map_set_fibre_table(result, result_table);
    moo_fibres_table_erase_extract_cols(result_table);
    int isGuess = (wmap == NULL);
    cpl_table *global_line_table = moo_line_table_new(isGuess);
    moo_map_set_line_table(result, global_line_table);

    if (wmap != NULL) {
        if (params->isrefit) {
        }
        wavemap_degx = params->wavemap_degx;
        wavemap_degy = params->wavemap_degy;
        line_winhsize = params->linefit_winhsize;
        cpl_errorstate stateA = cpl_errorstate_get();
        const char *guess_model =
            moo_pfits_get_pro_wavesol_model(wmap->primary_header);
        cpl_errorstate_set(stateA);
        moo_try_check(moo_fibres_table_add_wavecal_cols(result_table,
                                                        wmap->fibre_table,
                                                        guess_model),
                      " ");
        if (guess_model != NULL) {
            const char *degx =
                moo_pfits_get_pro_wavesol_degx(wmap->primary_header);
            char *tag = NULL;
            if (strcmp(guess_model, MOO_WAVESOL_MODEL_2D) == 0) {
                int degy = moo_pfits_get_pro_wavesol_degy(wmap->primary_header);
                tag = cpl_sprintf("%s_x%s_y%d", guess_model, degx, degy);
            }
            else {
                tag = cpl_sprintf("%s_x%s", guess_model, degx);
            }
            cpl_msg_info(__func__, "Compute wave solution using %s guess", tag);
            cpl_propertylist_append_string(result->primary_header,
                                           MOO_PFITS_PRO_WAVEMAP_GUESS, tag);
            cpl_free(tag);
        }

        for (int i = 0; i < 6; i++) {
            guess_tab[i] = wmap->data[i];
            guess_headers[i] = wmap->data_header[i];
        }
    }
    else {
        params->isrefit++;
        wavemap_degx = params->ppm_wavemap_degx;
        wavemap_degy = params->ppm_wavemap_degy;
        line_winhsize = params->linedetect_winhsize;

        moo_fibres_table_add_wavecalguess_cols(result_table);
        cpl_msg_info(__func__, "Compute first wave solution");
        for (int i = 0; i < 6; i++) {
            guess_tab[i] = NULL;
            guess_headers[i] = NULL;
        }
    }

    cpl_msg_indent_more();

    for (int i = 1; i <= 2; i++) {
        cpl_table_unselect_all(result_table);
        indexes = moo_fibres_table_get_spectro_indexext(result_table, i);
        const int *health =
            cpl_table_get_data_int_const(result_table, MOO_FIBRES_TABLE_HEALTH);

        if (health != NULL) {
            for (int j = 0; j < 3; j++) {
                int idx = (i - 1) * 3 + j;
                double tolerance = params->tolerance[idx];
                int winhsize = line_winhsize[idx];
                int degx = wavemap_degx[idx];
                int linefit_recentre = params->linefit_recentre[idx];

                int linedetect_nlines = params->linedetect_nlines[idx];
                double sigma_min = params->fwhm_min[idx] * CPL_MATH_SIG_FWHM;
                double sigma_max = params->fwhm_max[idx] * CPL_MATH_SIG_FWHM;
                double min_snr = params->min_snr[idx];
                cpl_image *guess = guess_tab[(i - 1) * 3 + j];
                cpl_propertylist *guess_header = guess_headers[(i - 1) * 3 + j];

                moo_ext_single *ext_single =
                    moo_ext_load_single(arc_ext, j, i, badpix_level);
                moo_loc_single *loc_single = moo_loc_get_single(loc, j, i);
                cpl_image *wave_single = NULL;

                if (ext_single != NULL && loc_single != NULL) {
                    cpl_table *sellines = NULL;
                    cpl_propertylist *header = NULL;
                    cpl_msg_info(
                        __func__,
                        "Compute wave solution for extension %s, winhsize:%d "
                        "fwhm:[%f,%f] min_snr:%f recentre:%d",
                        moo_detector_get_extname(j, i), winhsize,
                        params->fwhm_min[idx], params->fwhm_max[idx], min_snr,
                        linefit_recentre);
                    moo_try_check(sellines =
                                      _arclines_get_caliblines(lines, j,
                                                               isGuess,
                                                               params->isrefit),
                                  "Error for Line table %s", lineCatname);
                    if (sellines != NULL) {
                        moo_spectral_format_info *info =
                            moo_spectral_format_get(sformat, j, i);
                        header = cpl_propertylist_new();
                        cpl_table *line_table = NULL;

                        wave_single =
                            _moo_wavesol_single(ext_single, loc_single, indexes,
                                                health, result_table, sellines,
                                                info->dispmin, info->dispmax,
                                                tolerance, info->direction,
                                                guess, guess_header, wmap,
                                                linedetect_nlines, winhsize,
                                                linefit_recentre, degx,
                                                wavemap_degy, info, sigma_min,
                                                sigma_max, min_snr, header,
                                                params, &line_table);
                        moo_table_append(result->line_table, line_table);
                        cpl_table_delete(line_table);
                        moo_spectral_format_info_delete(info);
                        cpl_table_delete(sellines);
                    }
                    else if (guess != NULL) {
                        header = cpl_propertylist_new();
                        wave_single = cpl_image_cast(guess, CPL_TYPE_DOUBLE);
                    }
                    moo_map_set_data(result, j, i, wave_single, header);
                }
            }
        }

        cpl_array_delete(indexes);
        indexes = NULL;
    }

    cpl_msg_indent_less();

moo_try_cleanup:
    if (!cpl_errorstate_is_equal(prestate)) {
        moo_map_delete(result);
        cpl_array_delete(indexes);
        result = NULL;
        cpl_msg_error(__func__, "Error in wavesolution for EXT %s LOC %s",
                      arc_ext->filename, loc->filename);
        cpl_errorstate_dump(prestate, CPL_FALSE, cpl_errorstate_dump_one);
        // Recover from the error(s) (Reset to prestate))
        cpl_errorstate_set(prestate);
    }
    return result;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Computes the wavelength solution from 1D extracted arc spectra
  @param    arc_ext _EXT_ Extracted 1D arc spectra
  @param    lineCatname Name of the Line list file
  @param    sformat the SPECTRAL FORMAT
  @param    loc _LOC_ localization trace
  @param    wmap the WAVE MAP GUESS if a preliminary solution is alrezady computed
  @param    params the wavesol parameters
  @return   the _EXT_ spectra

 * This function computes the wavelength solution. It first determine the (X,Y)
 * positions of a set of arc lines given as input, and then creates a wavelength
 * map for each detector extensions in the input frame.

 * _Bad pixels flags_:
  - None

 - - -
 _Error code_:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
moo_map *
moo_wavesol(moo_ext *arc_ext,
            const char *lineCatname,
            moo_spectral_format *sformat,
            moo_loc *loc,
            moo_map *wmap,
            moo_wavesol_params *params)
{
    moo_map *result = NULL;
    cpl_table *lines = NULL;
    int dowavesol = 1;

    cpl_ensure(lineCatname != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(sformat != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(params != NULL, CPL_ERROR_NULL_INPUT, NULL);

    if (sformat->mode == MOO_MODE_LR) {
        lines = cpl_table_load(lineCatname, 1, 0);
    }
    else {
        lines = cpl_table_load(lineCatname, 2, 0);
    }

    /* check i refit is needed */
    if (wmap != NULL) {
        if (params->isrefit > 0) {
            dowavesol = 0;
            const char *refit_colname = MOO_ARCLINE_LIST_USEGUESS;

            if (params->isrefit == 1) {
                refit_colname = MOO_ARCLINE_LIST_USEGUESS_REFIT;
            }
            if (cpl_table_has_column(lines, refit_colname)) {
                cpl_table_select_all(lines);
                int size =
                    cpl_table_and_selected_int(lines,
                                               MOO_ARCLINE_LIST_USEGUESS_REFIT,
                                               CPL_EQUAL_TO, 1);
                if (size > 0) {
                    cpl_msg_info("moo_wavesol", "Try doing a refit using %s",
                                 refit_colname);
                    dowavesol = 1;
                }
            }
        }
    }

    if (dowavesol) {
        result = _moo_wavesol_refit(arc_ext, lineCatname, lines, sformat, loc,
                                    wmap, params);
        /* check result */
        moo_map_check(result, sformat);
    }

    cpl_table_delete(lines);

    return result;
}
/**@}*/
