/*
 * This file is part of the HDRL
 * Copyright (C) 2013,2014 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                    Includes
 -----------------------------------------------------------------------------*/
#include <cpl.h>

#include <stdlib.h>
#include <stdio.h>

#include "moo_det.h"
#include "moo_apply_flat.h"
#include "moo_params.h"

char *
path_from_filename(const char *name)
{
    return cpl_sprintf("%s/%s", DATA_DIR, name);
}

char *
result_from_filename(const char *name)
{
    return cpl_sprintf("%s/%s", RESULT_DIR, name);
}
/*----------------------------------------------------------------------------*/
/**
 * @defgroup drl_functions apply flat test
        Testing of functions used in moo_appy_flat
 */
/*----------------------------------------------------------------------------*/

void
_moo_apply_flat_lr_test(void)
{
    char *filename = path_from_filename("SCIENCE_EXTSPECTRA_OFFSET0_LR_0.fits");
    char *ffname = path_from_filename("FF_EXTSPECTRA_OFFSET0_LR.fits");
    char *f2fname = path_from_filename("F2F_OFFSET0_LR.fits");
    cpl_frame *frame = cpl_frame_new();
    cpl_frame_set_filename(frame, filename);
    cpl_frame_set_tag(frame, "TAG");
    cpl_frame_set_group(frame, CPL_FRAME_GROUP_PRODUCT);
    moo_ext *sci_ext = moo_ext_create(frame);
    cpl_frame *ff_frame = cpl_frame_new();
    cpl_frame_set_filename(ff_frame, ffname);
    cpl_frame_set_tag(ff_frame, "TAG");
    cpl_frame_set_group(ff_frame, CPL_FRAME_GROUP_PRODUCT);
    moo_ext *ff_ext = moo_ext_create(ff_frame);

    cpl_frame *f2f_frame = cpl_frame_new();
    cpl_frame_set_filename(f2f_frame, f2fname);
    cpl_frame_set_tag(f2f_frame, "TAG");
    cpl_frame_set_group(f2f_frame, CPL_FRAME_GROUP_PRODUCT);
    moo_f2f *f2f = moo_f2f_load(f2f_frame);

    moo_apply_flat(sci_ext, ff_ext, f2f);
    char *res = result_from_filename("EXT_APPLYFLAT_LR.fits");
    moo_ext_save(sci_ext, res);
    cpl_free(res);

    moo_ext_delete(sci_ext);
    moo_ext_delete(ff_ext);
    moo_f2f_delete(f2f);

    cpl_frame_delete(frame);
    cpl_frame_delete(ff_frame);
    cpl_frame_delete(f2f_frame);

    cpl_free(filename);
    cpl_free(ffname);
    cpl_free(f2fname);
}

void
_moo_apply_flat_hr_test(void)
{
    char *filename =
        path_from_filename("XSWITCH_OBJECTA_EXTSPECTRA_OFFSET0_HR_0.fits");
    char *ffname = path_from_filename("FF_EXTSPECTRA_OFFSET0_HR.fits");
    char *f2fname = path_from_filename("F2F_OFFSET0_HR.fits");
    cpl_frame *frame = cpl_frame_new();
    cpl_frame_set_filename(frame, filename);
    cpl_frame_set_tag(frame, "TAG");
    cpl_frame_set_group(frame, CPL_FRAME_GROUP_PRODUCT);
    moo_ext *sci_ext = moo_ext_create(frame);

    cpl_frame *ff_frame = cpl_frame_new();
    cpl_frame_set_filename(ff_frame, ffname);
    cpl_frame_set_tag(ff_frame, "TAG");
    cpl_frame_set_group(ff_frame, CPL_FRAME_GROUP_PRODUCT);

    moo_ext *ff_ext = moo_ext_create(ff_frame);

    cpl_frame *f2f_frame = cpl_frame_new();
    cpl_frame_set_filename(f2f_frame, f2fname);
    cpl_frame_set_tag(f2f_frame, "TAG");
    cpl_frame_set_group(f2f_frame, CPL_FRAME_GROUP_PRODUCT);

    moo_f2f *f2f = NULL;
    f2f = moo_f2f_load(f2f_frame);

    moo_apply_flat(sci_ext, ff_ext, f2f);

    char *res = result_from_filename("EXT_APPLYFLAT_HR.fits");
    moo_ext_save(sci_ext, res);
    cpl_free(res);

    moo_ext_delete(sci_ext);
    moo_ext_delete(ff_ext);
    moo_f2f_delete(f2f);

    cpl_frame_delete(frame);
    cpl_frame_delete(ff_frame);
    cpl_frame_delete(f2f_frame);

    cpl_free(filename);
    cpl_free(ffname);
    cpl_free(f2fname);
}

void
_moo_apply_flat_hr_std_test(void)
{
    char *filename = path_from_filename("STDFLUX_EXTSPECTRA_OFFSET1_HR_0.fits");
    char *ffname = path_from_filename("FF_EXTSPECTRA_OFFSET1_HR.fits");
    char *f2fname = path_from_filename("F2F_OFFSET1_HR.fits");
    cpl_frame *frame = cpl_frame_new();
    cpl_frame_set_filename(frame, filename);
    cpl_frame_set_tag(frame, "TAG");
    cpl_frame_set_group(frame, CPL_FRAME_GROUP_PRODUCT);
    moo_ext *sci_ext = moo_ext_create(frame);

    cpl_frame *ff_frame = cpl_frame_new();
    cpl_frame_set_filename(ff_frame, ffname);
    cpl_frame_set_tag(ff_frame, "TAG");
    cpl_frame_set_group(ff_frame, CPL_FRAME_GROUP_PRODUCT);

    moo_ext *ff_ext = moo_ext_create(ff_frame);

    cpl_frame *f2f_frame = cpl_frame_new();
    cpl_frame_set_filename(f2f_frame, f2fname);
    cpl_frame_set_tag(f2f_frame, "TAG");
    cpl_frame_set_group(f2f_frame, CPL_FRAME_GROUP_PRODUCT);

    moo_f2f *f2f = NULL;
    f2f = moo_f2f_load(f2f_frame);

    moo_apply_flat(sci_ext, ff_ext, f2f);

    char *res = result_from_filename("EXT_APPLYFLAT_OFFSET1_HR.fits");
    moo_ext_save(sci_ext, res);
    cpl_free(res);

    moo_ext_delete(sci_ext);
    moo_ext_delete(ff_ext);
    moo_f2f_delete(f2f);

    cpl_frame_delete(frame);
    cpl_frame_delete(ff_frame);
    cpl_frame_delete(f2f_frame);

    cpl_free(filename);
    cpl_free(ffname);
    cpl_free(f2fname);
}

void
_moo_apply_flat_offset1_hr_starenod2(void)
{
    char *filename =
        path_from_filename("STARENOD_OBJ_EXTSPECTRA_OFFSET1_HR_0.fits");
    char *ffname = path_from_filename("FF_EXTSPECTRA_OFFSET1_HR.fits");
    char *f2fname = path_from_filename("F2F_OFFSET1_HR.fits");
    cpl_frame *frame = cpl_frame_new();
    cpl_frame_set_filename(frame, filename);
    cpl_frame_set_tag(frame, "TAG");
    cpl_frame_set_group(frame, CPL_FRAME_GROUP_PRODUCT);
    moo_ext *sci_ext = moo_ext_create(frame);

    cpl_frame *ff_frame = cpl_frame_new();
    cpl_frame_set_filename(ff_frame, ffname);
    cpl_frame_set_tag(ff_frame, "TAG");
    cpl_frame_set_group(ff_frame, CPL_FRAME_GROUP_PRODUCT);

    moo_ext *ff_ext = moo_ext_create(ff_frame);

    cpl_frame *f2f_frame = cpl_frame_new();
    cpl_frame_set_filename(f2f_frame, f2fname);
    cpl_frame_set_tag(f2f_frame, "TAG");
    cpl_frame_set_group(f2f_frame, CPL_FRAME_GROUP_PRODUCT);

    moo_f2f *f2f = NULL;
    f2f = moo_f2f_load(f2f_frame);

    moo_apply_flat(sci_ext, ff_ext, f2f);

    char *res = result_from_filename("STARENOD_EXTFFSPECTRA_OFFSET1_HR_0.fits");
    moo_ext_save(sci_ext, res);
    cpl_free(res);

    moo_ext_delete(sci_ext);
    moo_ext_delete(ff_ext);
    moo_f2f_delete(f2f);

    cpl_frame_delete(frame);
    cpl_frame_delete(ff_frame);
    cpl_frame_delete(f2f_frame);

    cpl_free(filename);
    cpl_free(ffname);
    cpl_free(f2fname);
}

/*----------------------------------------------------------------------------*/
/**
 @brief   Unit tests
 **/
/*----------------------------------------------------------------------------*/
int
main(void)
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_INFO);
    _moo_apply_flat_offset1_hr_starenod2();
    return cpl_test_end(0);
}
