/*
 * This file is part of the HDRL
 * Copyright (C) 2013,2014 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                    Includes
 -----------------------------------------------------------------------------*/
#include <cpl.h>

#include <stdlib.h>
#include <stdio.h>
#include "moo_badpix.h"
#include "moo_det.h"
#include "moo_spectral_format.h"
#include "moo_map.h"
#include "moo_rbn.h"
#include "moo_f2f.h"
#include "moo_sub_sky_stare.h"
#include "moo_combine_pair.h"
#include "moo_params.h"

#ifdef _OPENMP
#include <omp.h>
#endif
/*----------------------------------------------------------------------------*/
/**
 * @defgroup drl_functions localise test
        Testing of functions used in moo_localise
 */
/*----------------------------------------------------------------------------*/

char *
path_from_filename(const char *name)
{
    return cpl_sprintf("%s/%s", DATA_DIR, name);
}

cpl_frame *
create_frame(const char *name)
{
    char *filename = path_from_filename(name);

    cpl_frame *frame = cpl_frame_new();
    cpl_frame_set_filename(frame, filename);
    cpl_free(filename);
    cpl_frame_set_tag(frame, "TAG");
    cpl_frame_set_group(frame, CPL_FRAME_GROUP_PRODUCT);

    return frame;
}

static void
_moo_combine_pair_xswitch(void)
{
    moo_rbn *nod_rbn = NULL;
    moo_rbn *oa_rbn = NULL;
    moo_rbn *ob_rbn = NULL;
    moo_sci *sci = NULL;
    moo_f2f *f2f = NULL;
    moo_target_table *target_table = NULL;

    cpl_frame *nod_rbn_frame =
        create_frame("SCIENCE_XSWITCH_RBNSPECTRA_NOD_OFFSET1_HR_0.fits");
    cpl_frame *obja_rbn_frame =
        create_frame("SCIENCE_XSWITCH_RBNSPECTRA_A_OFFSET1_HR_0.fits");
    cpl_frame *objb_rbn_frame =
        create_frame("SCIENCE_XSWITCH_RBNSPECTRA_B_OFFSET1_HR_0.fits");
    cpl_frame *target_table_frame =
        create_frame("XSWITCH_TARGET_TABLE_OFFSET1_HR_0.fits");
    cpl_frame *f2f_frame = create_frame("F2F_TABLE_OFFSET1_HR.fits");

    moo_sub_sky_stare_params *sub_sky_stare_params =
        moo_sub_sky_stare_params_new();
    moo_combine_pair_params *combine_pair_params =
        moo_combine_pair_params_new();

    nod_rbn = moo_rbn_create(nod_rbn_frame);

    ob_rbn = moo_rbn_create(objb_rbn_frame);

    oa_rbn = moo_rbn_create(obja_rbn_frame);

    target_table = moo_target_table_load(target_table_frame);
    f2f = moo_f2f_load(f2f_frame);

    sub_sky_stare_params->method = MOO_SUB_SKY_STARE_METHOD_SIMPLE;
    sub_sky_stare_params->radius_sky = 1.5;
    sub_sky_stare_params->step_r = 0.15;
    sub_sky_stare_params->min_sky = 5;
    sub_sky_stare_params->min_trans = 0.;
    sub_sky_stare_params->maxdistslit = 20;

    combine_pair_params->optimal = 1;

    cpl_msg_info("", "Use target table %s",
                 cpl_frame_get_filename(target_table_frame));
    sci =
        moo_combine_pair(nod_rbn, ob_rbn, oa_rbn, target_table, f2f, NULL, NULL,
                         NULL, combine_pair_params, sub_sky_stare_params,
                         "SCI_XSWITCH.fits", NULL, NULL);

    moo_sci_delete(sci);
    moo_rbn_delete(nod_rbn);
    moo_rbn_delete(oa_rbn);
    moo_rbn_delete(ob_rbn);
    moo_target_table_delete(target_table);
    moo_f2f_delete(f2f);

    moo_sub_sky_stare_params_delete(sub_sky_stare_params);
    moo_combine_pair_params_delete(combine_pair_params);
    cpl_frame_delete(nod_rbn_frame);
    cpl_frame_delete(obja_rbn_frame);
    cpl_frame_delete(objb_rbn_frame);
    cpl_frame_delete(target_table_frame);
    cpl_frame_delete(f2f_frame);
}


static void
_moo_combine_pair_starenod(void)
{
    moo_rbn *nod_rbn = NULL;
    moo_rbn *oa_rbn = NULL;
    moo_rbn *ob_rbn = NULL;
    moo_sci *sci = NULL;
    moo_f2f *f2f = NULL;

    cpl_frame *nod_rbn_frame =
        create_frame("STARENOD_NOD_RBNSPECTRA_OFFSET0_LR_0.fits");
    cpl_frame *oa_rbn_frame =
        create_frame("STARENOD_OBJ_RBNSPECTRA_OFFSET0_LR_0.fits");
    cpl_frame *ob_rbn_frame =
        create_frame("STARENOD_SKY_RBNSPECTRA_OFFSET0_LR_0.fits");
    cpl_frame *target_table_frame =
        create_frame("STARENOD_NOD_TARGET_TABLE_OFFSET0_LR_0.fits");
    cpl_frame *f2f_frame = create_frame("F2F_OFFSET0_LR.fits");

    moo_sub_sky_stare_params *sub_sky_stare_params =
        moo_sub_sky_stare_params_new();
    moo_combine_pair_params *combine_pair_params =
        moo_combine_pair_params_new();

    nod_rbn = moo_rbn_create(nod_rbn_frame);

    ob_rbn = moo_rbn_create(ob_rbn_frame);

    oa_rbn = moo_rbn_create(oa_rbn_frame);

    moo_target_table *target_table = moo_target_table_load(target_table_frame);
    f2f = moo_f2f_load(f2f_frame);

    sub_sky_stare_params->method = MOO_SUB_SKY_STARE_METHOD_SIMPLE;
    sub_sky_stare_params->radius_sky = 6;
    sub_sky_stare_params->step_r = 1;
    sub_sky_stare_params->min_sky = 5;
    sub_sky_stare_params->min_trans = 0.;
    sub_sky_stare_params->maxdistslit = 20;

    combine_pair_params->optimal = 1;

    cpl_msg_info("", "Use target table %s",
                 cpl_frame_get_filename(target_table_frame));
    sci = moo_combine_pair(nod_rbn, ob_rbn, oa_rbn, target_table, f2f, NULL,
                           NULL, NULL, combine_pair_params,
                           sub_sky_stare_params, "SCI_NOD.fits", NULL, NULL);

    moo_sci_delete(sci);
    moo_rbn_delete(nod_rbn);
    moo_rbn_delete(oa_rbn);
    moo_rbn_delete(ob_rbn);
    moo_target_table_delete(target_table);
    moo_f2f_delete(f2f);

    moo_sub_sky_stare_params_delete(sub_sky_stare_params);
    moo_combine_pair_params_delete(combine_pair_params);
    cpl_frame_delete(nod_rbn_frame);
    cpl_frame_delete(oa_rbn_frame);
    cpl_frame_delete(ob_rbn_frame);
    cpl_frame_delete(target_table_frame);
    cpl_frame_delete(f2f_frame);
}

static void
_moo_create_sci2d_starenod(void)
{
    moo_rbn *nod_rbn = NULL;
    moo_rbn *sky_rbn = NULL;
    moo_sci *sci = NULL;

    cpl_frame *nod_rbn_frame =
        create_frame("STARENOD_NOD_RBNSPECTRA_OFFSET1_HR.fits");
    cpl_frame *sky_rbn_frame =
        create_frame("STARENOD_SKY_RBNSPECTRA_OFFSET1_HR_0.fits");
    cpl_frame *target_table_frame =
        create_frame("STARENOD_TARGET_TABLE_OFFSET1_HR.fits");

    nod_rbn = moo_rbn_create(nod_rbn_frame);
    sky_rbn = moo_rbn_create(sky_rbn_frame);

    moo_target_table *target_table = moo_target_table_load(target_table_frame);
    cpl_msg_info("", "Use target table %s",
                 cpl_frame_get_filename(target_table_frame));
    sci = moo_create_sci2d(nod_rbn, sky_rbn, NULL, target_table,
                           MOO_SCI2D_NOT_PAIRED, "SCI2D.fits");

    moo_sci_delete(sci);
    moo_rbn_delete(nod_rbn);
    moo_rbn_delete(sky_rbn);
    moo_target_table_delete(target_table);

    cpl_frame_delete(nod_rbn_frame);
    cpl_frame_delete(sky_rbn_frame);
    cpl_frame_delete(target_table_frame);
}

static void
_moo_create_sci2d_xswitch(void)
{
    moo_rbn *nod_rbn = NULL;
    moo_rbn *obja_rbn = NULL;
    moo_rbn *objb_rbn = NULL;
    moo_sci *sci2d = NULL;
    moo_sci *psci2d = NULL;

    cpl_frame *nod_rbn_frame =
        create_frame("xswitch2/XSWITCH_NOD_RBNSPECTRA_OFFSET0_HR.fits");
    cpl_frame *obja_rbn_frame =
        create_frame("xswitch2/XSWITCH_OBJECTA_RBNSPECTRA_OFFSET0_HR_0.fits");
    cpl_frame *objb_rbn_frame =
        create_frame("xswitch2/XSWITCH_OBJECTB_RBNSPECTRA_OFFSET0_HR_0.fits");
    cpl_frame *target_table_frame =
        create_frame("xswitch2/XSWITCH_TARGET_TABLE_OFFSET0_HR.fits");
    cpl_frame *f2f_frame = create_frame("xswitch2/F2F_OFFSET0_HR.fits");

    nod_rbn = moo_rbn_create(nod_rbn_frame);
    obja_rbn = moo_rbn_create(obja_rbn_frame);
    objb_rbn = moo_rbn_create(objb_rbn_frame);

    moo_target_table *target_table = moo_target_table_load(target_table_frame);
    cpl_msg_info("", "Use target table %s",
                 cpl_frame_get_filename(target_table_frame));

    psci2d = moo_create_sci2d(nod_rbn, objb_rbn, obja_rbn, target_table,
                              MOO_SCI2D_PAIRED, "SCI2D_PAIRED.fits");
    sci2d = moo_create_sci2d(nod_rbn, objb_rbn, obja_rbn, target_table,
                             MOO_SCI2D_NOT_PAIRED, "SCI2D.fits");

    cpl_msg_info("test", "1D compute");
    moo_sub_sky_stare_params *params = moo_sub_sky_stare_params_new();

    moo_f2f *f2f = NULL;
    moo_sci *sci1d = NULL;
    moo_sci *psci1d = NULL;
    f2f = moo_f2f_load(f2f_frame);

    params->method = MOO_SUB_SKY_STARE_METHOD_SIMPLE;
    params->radius_sky = 180;
    params->step_r = 0.15;
    params->min_sky = 5;
    params->maxdistslit = 20;

    sci1d = moo_sub_sky_stare(obja_rbn, target_table, objb_rbn, f2f, NULL, NULL,
                              NULL, params, "SCI1D.fits", 0);
    psci1d = moo_sub_sky_stare(obja_rbn, target_table, objb_rbn, f2f, NULL,
                               NULL, NULL, params, "SCI1D_PAIRED.fits", 1);

    moo_sci *cpair_sci = NULL;

    cpair_sci = moo_combine_pair_sci(target_table, sci1d, psci1d, sci2d, psci2d,
                                     "SCI_COMBINED.fits");

    moo_sci_delete(cpair_sci);
    moo_sci_delete(sci1d);
    moo_sci_delete(psci1d);
    moo_f2f_delete(f2f);
    moo_sub_sky_stare_params_delete(params);
    cpl_frame_delete(f2f_frame);

    moo_sci_delete(sci2d);
    moo_sci_delete(psci2d);
    moo_rbn_delete(nod_rbn);
    moo_rbn_delete(obja_rbn);
    moo_rbn_delete(objb_rbn);
    moo_target_table_delete(target_table);

    cpl_frame_delete(nod_rbn_frame);
    cpl_frame_delete(obja_rbn_frame);
    cpl_frame_delete(objb_rbn_frame);
    cpl_frame_delete(target_table_frame);
}

static void
_moo_combine_pair_sci(void)
{
    moo_sci *result = NULL;
    moo_sci *sci1d = NULL;
    moo_sci *psci1d = NULL;
    moo_sci *sci2d = NULL;
    moo_sci *psci2d = NULL;

    cpl_frame *sci1d_frame = create_frame("XSWITCH_SCI1D_OFFSET0_HR.fits");
    cpl_frame *psci1d_frame =
        create_frame("XSWITCH_SCI1D_PAIRED_OFFSET0_HR.fits");
    cpl_frame *sci2d_frame = create_frame("XSWITCH_NOD_SCI2D_OFFSET0_HR.fits");
    cpl_frame *psci2d_frame =
        create_frame("XSWITCH_NOD_PAIRED_SCI2D_OFFSET0_HR.fits");
    cpl_frame *target_table_frame =
        create_frame("XSWITCH_TARGET_TABLE_OFFSET0_HR.fits");
    moo_target_table *target_table = moo_target_table_load(target_table_frame);
    sci1d = moo_sci_create(sci1d_frame);
    psci1d = moo_sci_create(psci1d_frame);
    sci2d = moo_sci_create(sci2d_frame);
    psci2d = moo_sci_create(psci2d_frame);

    result = moo_combine_pair_sci(target_table, sci1d, psci1d, sci2d, psci2d,
                                  "CPAIR_SCI.fits");

    moo_sci_delete(sci1d);
    moo_sci_delete(psci1d);
    moo_sci_delete(sci2d);
    moo_sci_delete(psci2d);
    moo_sci_delete(result);
    moo_target_table_delete(target_table);

    cpl_frame_delete(sci1d_frame);
    cpl_frame_delete(psci1d_frame);
    cpl_frame_delete(sci2d_frame);
    cpl_frame_delete(psci2d_frame);
    cpl_frame_delete(target_table_frame);
}
/*----------------------------------------------------------------------------*/
/**
 @brief   Unit tests of moo_combine_pair
 **/
/*----------------------------------------------------------------------------*/
int
main(void)
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_INFO);
    //_moo_combine_pair_starenod();
    //_moo_create_sci2d_xswitch();
    _moo_create_sci2d_starenod();
    //_moo_combine_pair_xswitch();
    //_moo_combine_pair_sci();
    return cpl_test_end(0);
}
