/*
 * This file is part of the HDRL
 * Copyright (C) 2013,2014 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                    Includes
 -----------------------------------------------------------------------------*/
#include <cpl.h>

#include <stdlib.h>
#include <stdio.h>
#include "moo_badpix.h"
#include "moo_drl.h"
#include "moo_params.h"

#ifdef _OPENMP
#include <omp.h>
#endif
/*----------------------------------------------------------------------------*/
/**
 * @defgroup drl_functions localise test
        Testing of functions used in moo_localise
 */
/*----------------------------------------------------------------------------*/

char *
path_from_filename(const char *name)
{
    return cpl_sprintf("%s/%s", DATA_DIR, name);
}

char *
result_from_filename(const char *name)
{
    return cpl_sprintf("%s/%s", RESULT_DIR, name);
}

cpl_frame *
create_frame(const char *name)
{
    char *filename = path_from_filename(name);

    cpl_frame *frame = cpl_frame_new();
    cpl_frame_set_filename(frame, filename);
    cpl_free(filename);
    cpl_frame_set_tag(frame, "TAG");
    cpl_frame_set_group(frame, CPL_FRAME_GROUP_PRODUCT);

    return frame;
}

static void
_moo_compute_noise(void)
{
    cpl_frame *dark0_frame = create_frame("DARK_CORRECTBIAS_0.fits");
    cpl_frame *dark1_frame = create_frame("DARK_CORRECTBIAS_1.fits");
    cpl_frame *dark2_frame = create_frame("DARK_CORRECTBIAS_2.fits");
    cpl_frame *darkhot_frame = create_frame("DARK_COMPUTEHOT.fits");
    cpl_frame *mask0_frame = create_frame("MASK_0.fits");
    cpl_frame *mask1_frame = create_frame("MASK_1.fits");
    cpl_frame *mask2_frame = create_frame("MASK_2.fits");

    cpl_frameset *set = cpl_frameset_new();
    cpl_frameset_insert(set, dark0_frame);
    cpl_frameset_insert(set, dark1_frame);
    cpl_frameset_insert(set, dark2_frame);

    cpl_frameset *mset = cpl_frameset_new();
    cpl_frameset_insert(mset, mask0_frame);
    cpl_frameset_insert(mset, mask1_frame);
    cpl_frameset_insert(mset, mask2_frame);

    moo_det *darkhot = moo_det_create(darkhot_frame);
    moo_detlist *list = moo_detlist_create(set);
    moo_masklist *mlist = NULL;
    moo_nos_params *params = moo_nos_params_new();
    params->clip_kappa = 5;
    params->clip_niter = 2;
    params->clip_diff = 0.1;
    params->clip_frac = 0.1;
    moo_compute_noise_map(darkhot, list, mlist, params);

    char *resname = result_from_filename("DET_DARK_NOISE.fits");
    moo_det_save(darkhot, resname);
    cpl_free(resname);

    moo_nos_params_delete(params);
    moo_detlist_delete(list);
    moo_masklist_delete(mlist);
    moo_det_delete(darkhot);
    cpl_frameset_delete(set);
    cpl_frameset_delete(mset);
    cpl_frame_delete(darkhot_frame);
}

/*----------------------------------------------------------------------------*/
/**
 @brief   Unit tests of hdrl_image module
 **/
/*----------------------------------------------------------------------------*/
int
main(void)
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_INFO);
    _moo_compute_noise();
    return cpl_test_end(0);
}
