/*
 * This file is part of the HDRL
 * Copyright (C) 2013,2014 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                    Includes
 -----------------------------------------------------------------------------*/
#include <cpl.h>
#include <stdlib.h>
#include <stdio.h>
#include "moo_map.h"
#include "moo_ext.h"
#include "moo_sci.h"
#include "moo_sky_lines_list.h"

#ifdef _OPENMP
#include <omp.h>
#endif
/*----------------------------------------------------------------------------*/
/**
 * @defgroup drl_functions compute snr tests
        Testing of functions used to compute snr
 */
/*----------------------------------------------------------------------------*/

char *
path_from_filename(const char *name)
{
    return cpl_sprintf("%s/%s", DATA_DIR, name);
}

cpl_frame *
create_frame(const char *name)
{
    char *filename = path_from_filename(name);

    cpl_frame *frame = cpl_frame_new();
    cpl_frame_set_filename(frame, filename);
    cpl_free(filename);
    cpl_frame_set_tag(frame, "TAG");
    cpl_frame_set_group(frame, CPL_FRAME_GROUP_PRODUCT);

    return frame;
}


static void
_moo_compute_ext_snr(void)
{
    moo_ext *ext = NULL;
    moo_map *wmap = NULL;
    moo_sky_lines_list *skylines = NULL;
    moo_spectral_format *sformat = NULL;

    cpl_frame *ext_frame = create_frame("NOD_EXT.fits");
    cpl_frame *wmap_frame = create_frame("WAVE_MAP_OFFSET0_HR.fits");
    cpl_frame *skylines_frame = create_frame("SKY_LINE_LIST.fits");
    cpl_frame *sformat_frame = create_frame("sformat.fits");

    ext = moo_ext_create(ext_frame);
    wmap = moo_map_load(wmap_frame);
    skylines = moo_sky_lines_list_load(skylines_frame);
    sformat = moo_spectral_format_load(sformat_frame, MOO_MODE_HR);

    moo_ext_compute_snr(ext, wmap, sformat, skylines);
    moo_ext_save(ext, "EXT_SNR.fits");
    moo_ext_delete(ext);
    moo_map_delete(wmap);
    moo_sky_lines_list_delete(skylines);
    moo_spectral_format_delete(sformat);

    cpl_frame_delete(ext_frame);
    cpl_frame_delete(sformat_frame);
    cpl_frame_delete(wmap_frame);
    cpl_frame_delete(skylines_frame);
}

static void
_moo_compute_ext_snr_stdtell(void)
{
    moo_ext *ext = NULL;
    moo_map *wmap = NULL;
    moo_sky_lines_list *skylines = NULL;
    moo_spectral_format *sformat = NULL;

    cpl_frame *ext_frame = create_frame("STDTELL_EXTSPECTRA_OFFSET1_HR_0.fits");
    cpl_frame *wmap_frame = create_frame("WAVE_MAP_TRUE_OFFSET1_HR.fits");
    cpl_frame *skylines_frame = create_frame("SKY_LINE_LISTnewclean.fits");
    cpl_frame *sformat_frame = create_frame("spectralfmt.fits");

    ext = moo_ext_create(ext_frame);
    wmap = moo_map_load(wmap_frame);
    skylines = moo_sky_lines_list_load(skylines_frame);
    sformat = moo_spectral_format_load(sformat_frame, MOO_MODE_HR);

    moo_ext_compute_snr(ext, wmap, sformat, skylines);
    moo_ext_save(ext, "STDTELL_EXT_SNR.fits");
    moo_ext_delete(ext);
    moo_map_delete(wmap);
    moo_sky_lines_list_delete(skylines);
    moo_spectral_format_delete(sformat);

    cpl_frame_delete(ext_frame);
    cpl_frame_delete(sformat_frame);
    cpl_frame_delete(wmap_frame);
    cpl_frame_delete(skylines_frame);
}

static void
_moo_compute_rbn_snr(void)
{
    moo_rbn *rbn = NULL;
    moo_sky_lines_list *skylines = NULL;

    cpl_frame *rbn_frame = create_frame("NOD_RBN.fits");
    cpl_frame *skylines_frame = create_frame("SKY_LINE_LIST.fits");

    rbn = moo_rbn_create(rbn_frame);
    skylines = moo_sky_lines_list_load(skylines_frame);

    moo_rbn_compute_snr(rbn, skylines);
    moo_rbn_save(rbn, "RBN_SNR.fits");
    moo_rbn_delete(rbn);
    moo_sky_lines_list_delete(skylines);
    cpl_frame_delete(rbn_frame);
    cpl_frame_delete(skylines_frame);
}

static void
_moo_compute_sci_snr(void)
{
    moo_sci *sci = NULL;
    moo_sky_lines_list *skylines = NULL;

    cpl_frame *sci_frame =
        create_frame("SCIENCE_STARENOD_COMBINED_SPECTRA_HR.fits");
    cpl_frame *skylines_frame = create_frame("SKY_LINE_LIST.fits");

    sci = moo_sci_create(sci_frame);
    moo_sci_single_delete(sci->h);
    sci->h = NULL;
    skylines = moo_sky_lines_list_load(skylines_frame);

    moo_sci_compute_snr(sci, skylines);
    //moo_rbn_save(rbn, "RBN_SNR.fits");
    moo_sci_delete(sci);
    moo_sky_lines_list_delete(skylines);
    cpl_frame_delete(sci_frame);
    cpl_frame_delete(skylines_frame);
}
/*----------------------------------------------------------------------------*/
/**
 @brief   Unit tests of moo_combine_pair
 **/
/*----------------------------------------------------------------------------*/
int
main(void)
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_INFO);
    _moo_compute_sci_snr();
    return cpl_test_end(0);
}
