/*
 * This file is part of the HDRL
 * Copyright (C) 2013,2014 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                    Includes
 -----------------------------------------------------------------------------*/
#include <cpl.h>

#include <stdlib.h>
#include <stdio.h>

#include "moo_rbn.h"
#include "moo_target_table.h"

#ifdef _OPENMP
#include <omp.h>
#endif
/*----------------------------------------------------------------------------*/
/**
 * @defgroup drl_functions localise test
        Testing of functions used in moo_localise
 */
/*----------------------------------------------------------------------------*/

char *
path_from_filename(const char *name)
{
    return cpl_sprintf("%s/%s", DATA_DIR, name);
}

char *
result_from_filename(const char *name)
{
    return cpl_sprintf("%s/%s", RESULT_DIR, name);
}

cpl_frame *
create_frame(const char *name)
{
    char *filename = path_from_filename(name);

    cpl_frame *frame = cpl_frame_new();
    cpl_frame_set_filename(frame, filename);
    cpl_free(filename);
    cpl_frame_set_tag(frame, "TAG");
    cpl_frame_set_group(frame, CPL_FRAME_GROUP_PRODUCT);

    return frame;
}

static void
_moo_create_target_table_stare(void)
{
    cpl_frame *rbn_frame = create_frame("SCIENCE_RBNSPECTRA_0.fits");

    moo_rbn *rbn = moo_rbn_create(rbn_frame);
    moo_target_table_params *params = moo_target_table_params_new();
    params->nosky = CPL_FALSE;
    moo_target_table *target_table =
        moo_create_target_table(rbn, NULL, MOO_MODE_STARE, params);
    char *resname = result_from_filename("TARGET_TABLE_STARE.fits");
    moo_target_table_save(target_table, resname);
    moo_target_table_delete(target_table);
    moo_target_table_params_delete(params);
    moo_rbn_delete(rbn);
    cpl_frame_delete(rbn_frame);
    cpl_free(resname);
}

static void
_moo_create_target_table_starenod(void)
{
    cpl_frame *rbn_frame =
        create_frame("STARENOD_NOD_RBNSPECTRA_OFFSET0_LR_0.fits");
    cpl_frame *sky_frame =
        create_frame("STARENOD_SKY_RBNSPECTRA_OFFSET0_LR_0.fits");

    moo_rbn *rbn = moo_rbn_create(rbn_frame);
    moo_rbn *sky_rbn = moo_rbn_create(sky_frame);
    moo_target_table_params *params = moo_target_table_params_new();
    params->nosky = CPL_FALSE;
    moo_target_table *target_table =
        moo_create_target_table(rbn, sky_rbn, MOO_MODE_STARENOD, params);
    char *resname = result_from_filename("TARGET_TABLE_STARENOD.fits");

    moo_target_table_save(target_table, resname);
    moo_target_table_delete(target_table);
    moo_target_table_params_delete(params);
    moo_rbn_delete(rbn);
    moo_rbn_delete(sky_rbn);

    cpl_frame_delete(sky_frame);
    cpl_frame_delete(rbn_frame);
    cpl_free(resname);
}

static void
_moo_create_target_table_xswitch(void)
{
    cpl_frame *rbnA_frame =
        create_frame("XSWITCH_OBJECTA_RBNSPECTRA_OFFSET1_HR_0.fits");
    cpl_frame *rbnB_frame =
        create_frame("XSWITCH_OBJECTB_RBNSPECTRA_OFFSET1_HR_0.fits");
    moo_rbn *rbnA = moo_rbn_create(rbnA_frame);
    moo_rbn *rbnB = moo_rbn_create(rbnB_frame);
    moo_target_table_params *params = moo_target_table_params_new();
    params->nosky = CPL_FALSE;
    moo_target_table *target_table =
        moo_create_target_table(rbnA, rbnB, MOO_MODE_XSWITCH, params);
    char *resname = result_from_filename("TARGET_TABLE_XSWITCH.fits");
    moo_target_table_save(target_table, resname);
    moo_target_table_delete(target_table);

    moo_target_table_params_delete(params);
    moo_rbn_delete(rbnA);
    cpl_frame_delete(rbnA_frame);
    moo_rbn_delete(rbnB);
    cpl_frame_delete(rbnB_frame);
    cpl_free(resname);
}

/*----------------------------------------------------------------------------*/
/**
 @brief   Unit tests of hdrl_image module
 **/
/*----------------------------------------------------------------------------*/
int
main(void)
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_INFO);
    _moo_create_target_table_xswitch();
    return cpl_test_end(0);
}
