/*
 * This file is part of the HDRL
 * Copyright (C) 2013,2014 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                    Includes
 -----------------------------------------------------------------------------*/

#include "moo_det.h"
#include "moo_detector.h"
#include "moo_badpix.h"
#include "moo_dfs.h"
#include "moo_testbed_utils.h"
#include <moo_drl.h>

#include <cpl.h>
#include <stdlib.h>
#include <stdio.h>
#include <fitsio2.h>
/*----------------------------------------------------------------------------*/
/**
 * @defgroup DET-test
        Testing of functions dealing with DET
 */
/*----------------------------------------------------------------------------*/
static char *
_moo_path_from_filename(const char *name)
{
    return cpl_sprintf("%s/%s", DATA_DIR, name);
}

static void
_moo_det_test(void)
{
    moo_det *det = NULL;

    char *filename = _moo_path_from_filename("FLAT_REMOVECRH_OFFSET1_HR.fits");
    cpl_frame *flatframe = NULL;
    det = moo_det_create(flatframe);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    cpl_test_null(det);

    flatframe = cpl_frame_new();
    cpl_frame_set_filename(flatframe, filename);
    cpl_frame_set_tag(flatframe, MOONS_TAG_FLAT);
    cpl_frame_set_group(flatframe, CPL_FRAME_GROUP_PRODUCT);

    det = moo_det_create(flatframe);
    cpl_test_nonnull(det);
    /*
    single = moo_single_new(MOO_TYPE_RI,1);                
    cpl_test_error(CPL_ERROR_NONE);    
    
    moo_single_dump(single, stdout);    
    cpl_test_error(CPL_ERROR_NONE);    
    
    cpl_test_null(moo_single_get_image(single));    
    cpl_test_error(CPL_ERROR_NONE);       
    
    moo_single_delete(single);
    cpl_test_error(CPL_ERROR_NONE);
   
    single = moo_single_create(NULL,MOO_TYPE_RI,1);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    cpl_test_null(single);
    
    single = moo_single_create("det.fits",MOO_TYPE_RI,3);
    cpl_test_error(CPL_ERROR_ILLEGAL_INPUT);
    cpl_test_null(single);
        
    // - create a DET file
    const char* bands[3] = {"RI"};    
    moo_testbed_create_raw( 10, bands, 1, 2, "raw.fits", 1, 1,
            0.9, 5, 150);
    cpl_frame* raw1 = cpl_frame_new();
    cpl_frame_set_filename(raw1, "raw.fits");
    cpl_frame_set_tag(raw1,"BIAS");
    cpl_frame_set_group(raw1,CPL_FRAME_GROUP_RAW);  

    moo_det* det1 = moo_prepare(raw1, NULL, NULL, NULL);
    cpl_frame_delete(raw1);

    moo_det_save(det1,"det.fits");
  
    cpl_image* test = cpl_image_new(10,10,CPL_TYPE_INT);
    cpl_image_set(test,5,5,100);
    
    
    fitsfile* fptr = NULL;
    int status=0;
    const char* name = "QUAL_RI_1";
    int hdunum;
    int bitpix;
    long fpixel[] = {1,1};    
    fits_open_file( &fptr, "det.fits", READWRITE, &status);    
    fits_get_num_hdus(fptr, &hdunum, &status);    
    fits_movnam_hdu(fptr, IMAGE_HDU, (char*)name,0, &status);    
    void* test_data = cpl_image_get_data(test);
    fits_get_img_type(fptr, &bitpix, &status);    
    fits_write_pix(fptr,  TINT, fpixel,
               100, (void*)test_data, &status);    
    fits_close_file(fptr, &status);
        
    cpl_image_delete(test);
    moo_det_delete(det1);
    
    single = moo_single_create("det.fits",MOO_TYPE_RI,1);
    cpl_test_nonnull(single);
    hdrl_image* img = moo_single_get_image(single);    
    cpl_test_null(img);
    
    moo_single_load(single, MOO_BADPIX_GOOD);
    img = moo_single_get_image(single);    
    cpl_test_nonnull(img);
    
    hdrl_value v = hdrl_image_get_pixel(img,1,1,NULL);
    moo_single_sum(single,single);
    hdrl_value v2 = hdrl_image_get_pixel(img,1,1,NULL);
    cpl_test_rel(2*v.data, v2.data, 0.001);
    moo_single_sub(single,single);
    hdrl_value v3 = hdrl_image_get_pixel(img,1,1,NULL);
    cpl_test_rel(v3.data, 0, 0.001);
    cpl_image* qual = moo_single_get_qual(single);
    cpl_test_nonnull(qual);
    int* data = cpl_image_get_data_int(qual);
    cpl_test_rel(data[0],0,0.0001);
    int nx = cpl_image_get_size_x(qual);
    int ny = cpl_image_get_size_y(qual);
    cpl_mask* mask = cpl_mask_new(nx,ny);
    cpl_mask_set(mask,1,1,1);
    moo_single_apply_mask(single,mask, 2);        
    cpl_test_eq(data[0],2);
    moo_single_apply_mask(single,mask, 4);        
    cpl_test_eq(data[0],6);
    cpl_mask_delete(mask);
    moo_single_free(single);
    img = moo_single_get_image(single);        
    cpl_test_null(img);
    qual = moo_single_get_qual(single);
    cpl_test_null(qual);
    */
    moo_det_delete(det);
    cpl_frame_delete(flatframe);
    cpl_free(filename);
}


/*----------------------------------------------------------------------------*/
/**
 @brief   Unit tests of hdrl_image module
 **/
/*----------------------------------------------------------------------------*/
int
main(void)
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_INFO);

    _moo_det_test();

    return cpl_test_end(0);
}
