/*
 * This file is part of the HDRL
 * Copyright (C) 2013,2014 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                    Includes
 -----------------------------------------------------------------------------*/
#include <cpl.h>

#include <stdlib.h>
#include <stdio.h>

#include "moo_det.h"
#include "moo_localise.h"
#include "moo_params.h"

#ifdef _OPENMP
#include <omp.h>
#endif
/*----------------------------------------------------------------------------*/
/**
 * @defgroup drl_functions localise test
        Testing of functions used in moo_localise
 */
/*----------------------------------------------------------------------------*/

char *
path_from_filename(const char *name)
{
    return cpl_sprintf("%s/%s", DATA_DIR, name);
}

char *
result_from_filename(const char *name)
{
    return cpl_sprintf("%s/%s", RESULT_DIR, name);
}

void
_moo_localise_create_guess_lr(void)
{
    moo_loc *loc = NULL;
    char *filename = path_from_filename("MEDFLAT_OFFSET1_LR.fits");

    cpl_frame *frame = cpl_frame_new();
    cpl_frame_set_filename(frame, filename);
    cpl_frame_set_tag(frame, "TAG");
    cpl_frame_set_group(frame, CPL_FRAME_GROUP_PRODUCT);

    moo_localise_params *params = moo_localise_params_new();
    params->centralpos = 2048;
    params->centralwidth = 30;
    params->backg_method = MOO_LOCALISE_BACKG_METHOD_RUNNINGMIN;
    params->backg_winhsize[0] = 3;
    params->backg_winhsize[1] = 3;
    params->backg_winhsize[2] = 3;
    params->polydeg[0] = 9;
    params->polydeg[1] = 8;
    params->polydeg[2] = 10;
    params->polydeg[0] = 9;
    params->polydeg[1] = 8;
    params->polydeg[2] = 10;

    params->wdiff_lim = 1.0;
    params->ydiff_lim = 0.5;

    params->relativethresh = 0.1;
    params->ref_snr[0] = 20;
    params->xgap_max = 100;
    params->loc_xlim_hwin = 30;
    params->loc_xlim_fracmin = 0.3;
    params->keep_points = 1;
    params->goodptsfrac_min[0] = 0.4;
    params->goodptsfrac_min[1] = 0.4;
    params->goodptsfrac_min[2] = 0.4;

    moo_det *det = moo_det_create(frame);

    char *resname = result_from_filename("FF_TRACE_GUESS_OFFSET1_LR.fits");
    loc = moo_localise(det, NULL, params, resname);
    cpl_free(resname);

    moo_loc_delete(loc);
    moo_det_delete(det);
    moo_localise_params_delete(params);
    cpl_frame_delete(frame);
    cpl_free(filename);
}

void
_moo_localise_create_final_lr(void)
{
    moo_loc *loc = NULL;
    char *filename = path_from_filename("MEDFLAT_OFFSET0_LR.fits");
    char *locname = path_from_filename("FF_TRACE_GUESS_OFFSET0_LR.fits");

    cpl_frame *frame = cpl_frame_new();
    cpl_frame_set_filename(frame, filename);
    cpl_frame_set_tag(frame, "TAG");
    cpl_frame_set_group(frame, CPL_FRAME_GROUP_PRODUCT);

    cpl_frame *guess = cpl_frame_new();
    cpl_frame_set_filename(guess, locname);
    cpl_frame_set_tag(guess, "TAG");
    cpl_frame_set_group(guess, CPL_FRAME_GROUP_PRODUCT);

    moo_localise_params *params = moo_localise_params_new();

    params->centralpos = 2048;
    params->centralwidth = 25;
    params->backg_method = MOO_LOCALISE_BACKG_METHOD_RUNNINGMIN;
    params->backg_winhsize[0] = 3;
    params->backg_winhsize[1] = 3;
    params->backg_winhsize[2] = 3;
    params->polydeg[0] = 9;
    params->polydeg[1] = 8;
    params->polydeg[2] = 10;
    params->polydeg[0] = 9;
    params->polydeg[1] = 8;
    params->polydeg[2] = 10;

    params->wdiff_lim = 1.0;
    params->ydiff_lim = 0.5;
    params->relativethresh = 0.1;
    params->ref_snr[0] = 40;
    params->xgap_max = 100;
    params->loc_xlim_hwin = 30;
    params->loc_xlim_fracmin = 0.1;
    params->keep_points = 1;
    params->goodptsfrac_min[0] = 0.4;
    params->goodptsfrac_min[1] = 0.3;
    params->goodptsfrac_min[2] = 0.4;
    params->method = MOO_LOCALISE_METHOD_BARYCENTER;

    moo_det *det = moo_det_create(frame);
    char *resname = result_from_filename("FF_TRACE_OFFSET0_LR.fits");

    loc = moo_localise(det, guess, params, resname);

    cpl_free(resname);
    moo_loc_delete(loc);
    moo_det_delete(det);
    moo_localise_params_delete(params);
    cpl_frame_delete(frame);
    cpl_frame_delete(guess);
    cpl_free(filename);
    cpl_free(locname);
}

void
_moo_localise_create_final_hr(void)
{
    moo_loc *loc = NULL;
    char *filename = path_from_filename("FLAT_REMOVECRH_OFFSET1_LR.fits");
    char *locname = path_from_filename("FF_TRACE_GUESS_OFFSET1_LR.fits");

    cpl_frame *frame = cpl_frame_new();
    cpl_frame_set_filename(frame, filename);
    cpl_frame_set_tag(frame, "TAG");
    cpl_frame_set_group(frame, CPL_FRAME_GROUP_PRODUCT);

    cpl_frame *guess = cpl_frame_new();
    cpl_frame_set_filename(guess, locname);
    cpl_frame_set_tag(guess, "TAG");
    cpl_frame_set_group(guess, CPL_FRAME_GROUP_PRODUCT);

    moo_localise_params *params = moo_localise_params_new();
    params->centralpos = 2048;
    params->centralwidth = 30;
    params->backg_method = MOO_LOCALISE_BACKG_METHOD_RUNNINGMIN;
    params->backg_winhsize[0] = 3;
    params->backg_winhsize[1] = 3;
    params->backg_winhsize[2] = 3;
    params->polydeg[0] = 9;
    params->polydeg[1] = 8;
    params->polydeg[2] = 10;
    params->polydeg[0] = 9;
    params->polydeg[1] = 8;
    params->polydeg[2] = 10;

    params->wdiff_lim = 1.0;
    params->ydiff_lim = 0.5;
    params->relativethresh = 0.1;
    params->ref_snr[0] = 20;
    params->xgap_max = 100;
    params->loc_xlim_hwin = 30;
    params->loc_xlim_fracmin = 0.1;
    params->keep_points = 1;
    params->goodptsfrac_min[0] = 0.4;
    params->goodptsfrac_min[1] = 0.4;
    params->goodptsfrac_min[2] = 0.4;
    params->method = MOO_LOCALISE_METHOD_BARYCENTER;
    //params->method = MOO_LOCALISE_METHOD_GAUSSIAN;

    moo_det *det = moo_det_create(frame);
    char *resname = result_from_filename("FF_TRACE_OFFSET1_HR.fits");

    loc = moo_localise(det, guess, params, resname);

    cpl_free(resname);
    moo_loc_delete(loc);
    moo_det_delete(det);
    moo_localise_params_delete(params);
    cpl_frame_delete(frame);
    cpl_frame_delete(guess);
    cpl_free(filename);
    cpl_free(locname);
}

void
_moo_localise_create_guess_hr(void)
{
    char *filename = path_from_filename("MEDFLAT_OFFSET0_hr.fits");

    cpl_frame *frame = cpl_frame_new();
    cpl_frame_set_filename(frame, filename);
    cpl_frame_set_tag(frame, "TAG");
    cpl_frame_set_group(frame, CPL_FRAME_GROUP_PRODUCT);

    moo_localise_params *params = moo_localise_params_new();
    params->centralpos = 2048;
    params->centralwidth = 30;

    params->backg_method = MOO_LOCALISE_BACKG_METHOD_RUNNINGMIN;
    params->backg_winhsize[0] = 3;

    params->polydeg[0] = 9;
    params->polydeg[1] = 8;
    params->polydeg[2] = 10;
    params->polydeg[3] = 9;
    params->polydeg[4] = 8;
    params->polydeg[5] = 10;

    params->keep_points = 0;
    params->wdiff_lim = 1.0;
    params->ydiff_lim = 0.5;
    params->relativethresh = 0.1;
    params->ref_snr[0] = 20;

    moo_det *det = moo_det_create(frame);
    moo_single_delete(det->ri[0]);
    det->ri[0] = NULL;
    // moo_single_delete(det->yj[0]);
    // det->yj[0]=NULL;
    moo_single_delete(det->h[0]);
    det->h[0] = NULL;
    moo_single_delete(det->ri[1]);
    det->ri[1] = NULL;
    moo_single_delete(det->yj[1]);
    det->yj[1] = NULL;
    moo_single_delete(det->h[1]);
    det->h[1] = NULL;

    char *resname = result_from_filename("FF_TRACE_GUESS_hr.fits");
    moo_loc *loc = moo_localise(det, NULL, params, resname);
    cpl_free(resname);

    moo_loc_delete(loc);
    moo_det_delete(det);
    moo_localise_params_delete(params);
    cpl_frame_delete(frame);
    cpl_free(filename);
}

void
_moo_localise_create_guess_offset1_hr(void)
{
    moo_loc *loc = NULL;

    char *filename = path_from_filename("FLAT_REMOVECRH_OFFSET1_HR.fits");

    cpl_frame *frame = cpl_frame_new();
    cpl_frame_set_filename(frame, filename);
    cpl_frame_set_tag(frame, "TAG");
    cpl_frame_set_group(frame, CPL_FRAME_GROUP_PRODUCT);

    moo_localise_params *params = moo_localise_params_new();

    params->centralpos = 2048;
    params->centralwidth = 30;
    params->backg_method = MOO_LOCALISE_BACKG_METHOD_RUNNINGMIN;

    params->backg_winhsize[0] = 3;
    params->backg_polydeg[0] = 3;
    params->backg_clip_kappalow[0] = 3;
    params->backg_clip_kappaup[0] = 1;
    params->backg_clip_niter[0] = 4;
    params->goodptsfrac_min[0] = 0.4;
    params->ref_snr[0] = 40.;
    params->polydeg[0] = 9;
    params->detect_niter[0] = 10;
    params->relativethresh = 0.1;

    params->backg_winhsize[2] = 3;
    params->backg_winhsize[5] = 3;
    params->backg_polydeg[2] = 5;
    params->backg_polydeg[5] = 5;
    params->backg_clip_kappalow[2] = 3;
    params->backg_clip_kappalow[5] = 3;
    params->backg_clip_kappaup[2] = 1;
    params->backg_clip_kappaup[5] = 1;
    params->backg_clip_niter[2] = 4;
    params->backg_clip_niter[5] = 4;
    params->goodptsfrac_min[2] = 0.4;
    params->goodptsfrac_min[5] = 0.4;

    params->xgap_max = 100;
    params->loc_xlim_hwin = 30;
    params->loc_xlim_fracmin = 0.2;
    params->ref_snr[2] = 40.;
    params->ref_snr[5] = 40.;

    params->method = MOO_LOCALISE_METHOD_BARYCENTER;
    params->wdiff_lim = 1.;
    params->ydiff_lim = 0.5;
    params->polydeg[2] = 10;
    params->keep_points = 1;

    moo_det *det = moo_det_create(frame);
    /*moo_single_delete(det->ri[0]);
    det->ri[0] = NULL;*/
    moo_single_delete(det->yj[0]);
    det->yj[0] = NULL;
    moo_single_delete(det->h[0]);
    det->h[0] = NULL;

    moo_single_delete(det->ri[1]);
    det->ri[1] = NULL;
    moo_single_delete(det->yj[1]);
    det->yj[1] = NULL;
    moo_single_delete(det->h[1]);
    det->h[1] = NULL;

    char *resname = result_from_filename("FF_TRACE_GUESS_OFFSET1_HR.fits");
    loc = moo_localise(det, NULL, params, resname);

    cpl_free(resname);

    moo_loc_delete(loc);
    moo_det_delete(det);
    moo_localise_params_delete(params);
    cpl_frame_delete(frame);
    cpl_free(filename);
}

void
_moo_localise_create_final_offset1_hr(void)
{
    moo_loc *loc = NULL;

    char *filename = path_from_filename("FLAT_REMOVECRH_OFFSET1_HR.fits");

    cpl_frame *frame = cpl_frame_new();
    cpl_frame_set_filename(frame, filename);
    cpl_frame_set_tag(frame, "TAG");
    cpl_frame_set_group(frame, CPL_FRAME_GROUP_PRODUCT);

    char *guess_filename = path_from_filename("FF_TRACE_GUESS_OFFSET1_HR.fits");

    cpl_frame *guess_frame = cpl_frame_new();
    cpl_frame_set_filename(guess_frame, guess_filename);
    cpl_frame_set_tag(guess_frame, "TAG");
    cpl_frame_set_group(guess_frame, CPL_FRAME_GROUP_PRODUCT);

    moo_localise_params *params = moo_localise_params_new();

    params->centralpos = 2048;
    params->centralwidth = 30;
    params->backg_method = MOO_LOCALISE_BACKG_METHOD_RUNNINGMIN;

    params->backg_winhsize[0] = 3;
    params->backg_polydeg[0] = 3;
    params->backg_clip_kappalow[0] = 3;
    params->backg_clip_kappaup[0] = 1;
    params->backg_clip_niter[0] = 4;
    params->goodptsfrac_min[0] = 0.4;
    params->ref_snr[0] = 40.;
    params->polydeg[0] = 9;
    params->detect_niter[0] = 10;
    params->relativethresh = 0.1;

    params->backg_winhsize[2] = 3;
    params->backg_winhsize[5] = 3;
    params->backg_polydeg[2] = 5;
    params->backg_polydeg[5] = 5;
    params->backg_clip_kappalow[2] = 3;
    params->backg_clip_kappalow[5] = 3;
    params->backg_clip_kappaup[2] = 1;
    params->backg_clip_kappaup[5] = 1;
    params->backg_clip_niter[2] = 4;
    params->backg_clip_niter[5] = 4;
    params->goodptsfrac_min[2] = 0.4;
    params->goodptsfrac_min[5] = 0.4;

    params->xgap_max = 100;
    params->loc_xlim_hwin = 30;
    params->loc_xlim_fracmin = 0.2;
    params->ref_snr[2] = 40.;
    params->ref_snr[5] = 40.;

    params->method = MOO_LOCALISE_METHOD_BARYCENTER;
    params->wdiff_lim = 1.;
    params->ydiff_lim = 0.5;
    params->polydeg[2] = 10;
    params->keep_points = 1;

    moo_det *det = moo_det_create(frame);
    /*moo_single_delete(det->ri[0]);
    det->ri[0] = NULL;*/
    moo_single_delete(det->yj[0]);
    det->yj[0] = NULL;
    moo_single_delete(det->h[0]);
    det->h[0] = NULL;

    moo_single_delete(det->ri[1]);
    det->ri[1] = NULL;
    moo_single_delete(det->yj[1]);
    det->yj[1] = NULL;
    moo_single_delete(det->h[1]);
    det->h[1] = NULL;

    char *resname = result_from_filename("FF_TRACE_OFFSET1_HR.fits");
    loc = moo_localise(det, guess_frame, params, resname);

    cpl_free(resname);

    moo_loc_delete(loc);
    moo_det_delete(det);
    moo_localise_params_delete(params);

    cpl_frame_delete(guess_frame);
    cpl_free(guess_filename);

    cpl_frame_delete(frame);
    cpl_free(filename);
}

void
_moo_localise_create_guess_offset2_hr(void)
{
    moo_loc *loc = NULL;

    char *filename = path_from_filename("FLAT_REMOVECRH_OFFSET2_HR.fits");

    cpl_frame *frame = cpl_frame_new();
    cpl_frame_set_filename(frame, filename);
    cpl_frame_set_tag(frame, "TAG");
    cpl_frame_set_group(frame, CPL_FRAME_GROUP_PRODUCT);

    moo_localise_params *params = moo_localise_params_new();

    params->centralpos = 2048;
    params->centralwidth = 30;
    params->backg_method = MOO_LOCALISE_BACKG_METHOD_RUNNINGMIN;

    params->backg_winhsize[0] = 3;
    params->backg_polydeg[0] = 3;
    params->backg_clip_kappalow[0] = 3;
    params->backg_clip_kappaup[0] = 1;
    params->backg_clip_niter[0] = 4;
    params->goodptsfrac_min[0] = 0.4;
    params->ref_snr[0] = 40.;
    params->polydeg[0] = 9;
    params->detect_niter[0] = 10;
    params->relativethresh = 0.1;

    params->backg_winhsize[2] = 3;
    params->backg_polydeg[2] = 5;
    params->backg_clip_kappalow[2] = 3;
    params->backg_clip_kappaup[2] = 1;
    params->backg_clip_niter[2] = 4;
    params->ref_snr[2] = 40.;

    params->backg_polydeg[5] = 5;
    params->backg_winhsize[5] = 3;
    params->backg_clip_kappalow[5] = 3;
    params->backg_clip_kappaup[5] = 1;
    params->backg_clip_niter[5] = 4;
    params->goodptsfrac_min[2] = 0.4;
    params->goodptsfrac_min[5] = 0.4;

    params->xgap_max = 100;
    params->loc_xlim_hwin = 30;
    params->loc_xlim_fracmin = 0.2;
    params->ref_snr[5] = 40.;

    params->method = MOO_LOCALISE_METHOD_BARYCENTER;
    params->wdiff_lim = 1.;
    params->ydiff_lim = 0.5;
    params->polydeg[2] = 10;
    params->keep_points = 1;

    moo_det *det = moo_det_create(frame);
    moo_single_delete(det->ri[0]);
    det->ri[0] = NULL;
    moo_single_delete(det->yj[0]);
    det->yj[0] = NULL;
    /*moo_single_delete(det->h[0]);
    det->h[0] = NULL;
    */
    moo_single_delete(det->ri[1]);
    det->ri[1] = NULL;
    moo_single_delete(det->yj[1]);
    det->yj[1] = NULL;
    moo_single_delete(det->h[1]);
    det->h[1] = NULL;

    char *resname = result_from_filename("FF_TRACE_GUESS_OFFSET2_HR.fits");
    loc = moo_localise(det, NULL, params, resname);

    cpl_free(resname);

    moo_loc_delete(loc);
    moo_det_delete(det);
    moo_localise_params_delete(params);
    cpl_frame_delete(frame);
    cpl_free(filename);
}

/*----------------------------------------------------------------------------*/
/**
 @brief   Unit tests of hdrl_image module
 **/
/*----------------------------------------------------------------------------*/
int
main(void)
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_INFO);
    //_moo_localise_create_guess_lr();
    //_moo_localise_create_guess_offset1_hr();
    //_moo_localise_create_final_offset1_hr();
    _moo_localise_create_guess_offset2_hr();

    return cpl_test_end(0);
}
