/*
 * This file is part of the HDRL
 * Copyright (C) 2013,2014 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                    Includes
 -----------------------------------------------------------------------------*/
#include <cpl.h>

#include <stdlib.h>
#include <stdio.h>
#include "moo_badpix.h"
#include "moo_det.h"
#include "moo_spectral_format.h"
#include "moo_map.h"
#include "moo_rbn.h"
#include "moo_rebin.h"
#include "moo_params.h"

#ifdef _OPENMP
#include <omp.h>
#endif
/*----------------------------------------------------------------------------*/
/**
 * @defgroup drl_functions localise test
        Testing of functions used in moo_localise
 */
/*----------------------------------------------------------------------------*/

char *
path_from_filename(const char *name)
{
    return cpl_sprintf("%s/%s", DATA_DIR, name);
}

cpl_frame *
create_frame(const char *name)
{
    char *filename = path_from_filename(name);

    cpl_frame *frame = cpl_frame_new();
    cpl_frame_set_filename(frame, filename);
    cpl_free(filename);
    cpl_frame_set_tag(frame, "TAG");
    cpl_frame_set_group(frame, CPL_FRAME_GROUP_PRODUCT);

    return frame;
}

void
_moo_rebin_sum_hr(void)
{
    cpl_frame *ext_frame = create_frame("ARC_EXTSPECTRA_OFFSET0_HR.fits");
    //cpl_frame* ext_frame = create_frame("SCIENCE_EXTSPECTRA_OFFSET0_LR_0.fits");
    cpl_frame *wmap_frame = create_frame("WAVE_MAP_TRUE_HR.fits");
    cpl_frame *spec_frame = create_frame("spectralfmt2.fits");
    moo_rbn *rbn = NULL;

    moo_rebin_params *params = moo_rebin_params_new();


    moo_ext *ext = moo_ext_create(ext_frame);
    /*moo_ext_single_delete(ext->yj[0]);
    ext->yj[0] = NULL;
    moo_ext_single_delete(ext->h[0]);
    ext->h[0] = NULL;

    moo_ext_single_delete(ext->ri[1]);
    ext->ri[1] = NULL;
    moo_ext_single_delete(ext->yj[1]);
    ext->yj[1] = NULL;
    moo_ext_single_delete(ext->h[1]);
    ext->h[1] = NULL;
    */
    moo_spectral_format *sformat =
        moo_spectral_format_load(spec_frame, MOO_MODE_HR);
    moo_map *wmap = moo_map_load(wmap_frame);

    params->step[0] = 0.1;
    params->step[1] = 0.2;
    params->step[2] = 0.02;
    params->method = MOO_REBIN_METHOD_INTEGRATE;
    rbn = moo_rebin(ext, wmap, sformat, params, "REBIN_HR.fits");
    /*
    img = hdrl_image_get_image(rbn->ri->image);
    mask = cpl_image_get_bpm(img);
    moo_badpix_to_mask(rbn->ri->qual, mask, bp);
    flux = cpl_image_get_flux(img);
    cpl_msg_info("test","flux %.2f",flux);
    moo_rbn_delete(rbn);

    params->step[0] = 0.2;
    params->step[1] = 0.2;
    params->step[2] = 0.2;
    params->method = MOO_REBIN_METHOD_SUM;
    rbn = moo_rebin(ext, wmap, sformat, params, "REBIN.fits");

    img = hdrl_image_get_image(rbn->ri->image);
    mask = cpl_image_get_bpm(img);
    moo_badpix_to_mask(rbn->ri->qual, mask, bp);
    flux = cpl_image_get_flux(img);
    cpl_msg_info("test","flux %.2f",flux);
    moo_rbn_delete(rbn);

    params->step[0] = 1;
    params->step[1] = 0.2;
    params->step[2] = 0.2;
    params->method = MOO_REBIN_METHOD_SUM;
    rbn = moo_rebin(ext, wmap, sformat, params, "REBIN.fits");

    img = hdrl_image_get_image(rbn->ri->image);
    mask = cpl_image_get_bpm(img);
    moo_badpix_to_mask(rbn->ri->qual, mask, bp);
    flux = cpl_image_get_flux(img);
    cpl_msg_info("test","flux %.2f",flux);

     */
    moo_rbn_delete(rbn);

    moo_ext_delete(ext);
    moo_map_delete(wmap);
    moo_spectral_format_delete(sformat);
    moo_rebin_params_delete(params);

    cpl_frame_delete(ext_frame);
    cpl_frame_delete(wmap_frame);
    cpl_frame_delete(spec_frame);
}


void
_moo_rebin_integrate_hr(void)
{
    cpl_frame *ext_frame =
        create_frame("XSWITCH_OBJECTA_EXTFFSPECTRA_OFFSET1_HR_0.fits");
    cpl_frame *wmap_frame = create_frame("WAVE_MAP_TRUE_OFFSET1_HR.fits");
    cpl_frame *spec_frame = create_frame("spectralfmt.fits");
    moo_rbn *rbn = NULL;
    moo_rebin_params *params = moo_rebin_params_new();

    moo_ext *ext = moo_ext_create(ext_frame);

    moo_spectral_format *sformat =
        moo_spectral_format_load(spec_frame, MOO_MODE_HR);
    moo_map *wmap = moo_map_load(wmap_frame);

    params->step[0] = 0.05;
    params->step[1] = 0.05;
    params->step[2] = 0.05;
    params->method = MOO_REBIN_METHOD_INTEGRATE;
    params->conserv_flux = 0;
    rbn = moo_rebin(ext, wmap, sformat, params, "REBIN_INTEGRATE.fits");
    moo_rbn_save(rbn, "REBIN_INTEGRATE.fits");
    moo_rbn_delete(rbn);

    moo_ext_delete(ext);
    moo_map_delete(wmap);
    moo_spectral_format_delete(sformat);
    moo_rebin_params_delete(params);

    cpl_frame_delete(ext_frame);
    cpl_frame_delete(wmap_frame);
    cpl_frame_delete(spec_frame);
}

void
_moo_rebin_interp_offset1_hr(void)
{
    cpl_frame *ext_frame =
        create_frame("STARENOD_OBJ_EXTSPECTRA_OFFSET1_HR_0.fits");
    cpl_frame *wmap_frame = create_frame("WAVE_MAP_OFFSET1_HR.fits");
    cpl_frame *spec_frame = create_frame("spectralfmt.fits");
    moo_rbn *rbn = NULL;
    moo_rebin_params *params = moo_rebin_params_new();

    moo_ext *ext = moo_ext_create(ext_frame);

    moo_ext_single_delete(ext->ri[0]);
    ext->ri[0] = NULL;
    moo_ext_single_delete(ext->ri[1]);
    ext->ri[1] = NULL;

    moo_ext_single_delete(ext->yj[0]);
    ext->yj[0] = NULL;
    moo_ext_single_delete(ext->yj[1]);
    ext->yj[1] = NULL;

    /*moo_ext_single_delete(ext->h[0]);
    ext->h[0] = NULL;
    */
    moo_ext_single_delete(ext->h[1]);
    ext->h[1] = NULL;

    moo_spectral_format *sformat =
        moo_spectral_format_load(spec_frame, MOO_MODE_HR);
    moo_map *wmap = moo_map_load(wmap_frame);

    params->step[0] = 0.02;
    params->step[1] = 0.05;
    params->step[2] = 0.02;
    params->conserv_flux = 0;
    params->method = MOO_REBIN_METHOD_INTERPOLATE;
    rbn = moo_rebin(ext, wmap, sformat, params, "REBIN_INTERPOLATE.fits");

    moo_rbn_delete(rbn);

    moo_ext_delete(ext);
    moo_map_delete(wmap);
    moo_spectral_format_delete(sformat);
    moo_rebin_params_delete(params);

    cpl_frame_delete(ext_frame);
    cpl_frame_delete(wmap_frame);
    cpl_frame_delete(spec_frame);
}

void
_moo_rebin_hr_testflux(void)
{
    cpl_frame *ext_frame = create_frame("SCIENCE_EXT_SUM_OFFSET0_HR.fits");
    cpl_frame *wmap_frame = create_frame("WAVE_MAP_OFFSET0_HR.fits");
    cpl_frame *spec_frame = create_frame("spectralfmt.fits");
    cpl_image *img = NULL;
    cpl_mask *mask = NULL;
    double flux;
    moo_rbn *rbn = NULL;
    int bp = MOO_BADPIX_CALIB_DEFECT | MOO_BADPIX_OUTSIDE_DATA_RANGE;
    moo_rebin_params *params = moo_rebin_params_new();

    moo_ext *ext = moo_ext_create(ext_frame);

    moo_ext_single_delete(ext->ri[0]);
    ext->ri[0] = NULL;
    moo_ext_single_delete(ext->yj[0]);
    ext->yj[0] = NULL;
    //moo_ext_single_delete(ext->h[0]);
    //ext->h[0] = NULL;

    moo_ext_single_delete(ext->ri[1]);
    ext->ri[1] = NULL;
    moo_ext_single_delete(ext->yj[1]);
    ext->yj[1] = NULL;
    moo_ext_single_delete(ext->h[1]);
    ext->h[1] = NULL;
    moo_spectral_format *sformat =
        moo_spectral_format_load(spec_frame, MOO_MODE_HR);
    moo_map *wmap = moo_map_load(wmap_frame);

    double step[] = { 0.001, 0.01, 0.05, 0.1, 0.2, 0.5 };
    const char *methods[] = { MOO_REBIN_METHOD_INTERPOLATE,
                              MOO_REBIN_METHOD_INTEGRATE };
    params->step[0] = 0.01;
    params->step[1] = 0.01;
    params->step[2] = 0.01;
    params->conserv_flux = 1;
    params->method = MOO_REBIN_METHOD_INTERPOLATE;
    for (int j = 0; j < 2; j++) {
        params->method = methods[j];
        for (int i = 0; i < 6; i++) {
            params->step[2] = step[i];
            rbn = moo_rebin(ext, wmap, sformat, params, "REBIN.fits");

            img = hdrl_image_get_image(rbn->h->image);
            mask = cpl_image_get_bpm(img);
            moo_badpix_to_mask(rbn->h->qual, mask, bp);
            flux = cpl_image_get_flux(img);
            cpl_msg_info("test", "%s step %f flux %.2f", methods[j], step[i],
                         flux);
            moo_rbn_delete(rbn);
        }
    }
    moo_ext_delete(ext);
    moo_map_delete(wmap);
    moo_spectral_format_delete(sformat);
    moo_rebin_params_delete(params);

    cpl_frame_delete(ext_frame);
    cpl_frame_delete(wmap_frame);
    cpl_frame_delete(spec_frame);
}

void
_moo_rebin_interp_lr(void)
{
    cpl_frame *ext_frame = create_frame("ARC_EXTSPECTRA_OFFSET0_LR.fits");
    cpl_frame *wmap_frame = create_frame("WAVE_MAP_OFFSET0_LR.fits");
    cpl_frame *spec_frame = create_frame("spectralfmt.fits");

    moo_rbn *rbn = NULL;

    moo_rebin_params *params = moo_rebin_params_new();

    params->step[0] = 0.05;
    params->step[1] = 0.05;
    params->step[2] = 0.05;
    params->method = MOO_REBIN_METHOD_INTERPOLATE;

    moo_ext *ext = moo_ext_create(ext_frame);
    moo_spectral_format *sformat =
        moo_spectral_format_load(spec_frame, MOO_MODE_LR);
    moo_map *wmap = moo_map_load(wmap_frame);

    rbn = moo_rebin(ext, wmap, sformat, params, "REBIN_LR.fits");

    moo_rbn_delete(rbn);
    moo_ext_delete(ext);
    moo_map_delete(wmap);
    moo_spectral_format_delete(sformat);
    moo_rebin_params_delete(params);

    cpl_frame_delete(ext_frame);
    cpl_frame_delete(wmap_frame);
    cpl_frame_delete(spec_frame);
}

/*----------------------------------------------------------------------------*/
/**
 @brief   Unit tests of hdrl_image module
 **/
/*----------------------------------------------------------------------------*/
int
main(void)
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_INFO);
    //_moo_rebin_interp();
    _moo_rebin_interp_offset1_hr();
    //_moo_rebin_integrate_hr();
    //_moo_rebin_hr_testflux();
    return cpl_test_end(0);
}
