/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
//#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#include <stdlib.h>
#include <string.h>

#include <cpl.h>
#include <hdrl.h>
#include "moo_single.h"
#include "moo_det.h"
#include "moo_detlist.h"
#include "moo_masklist.h"

#include <moo_drl.h>
#include <moo_dfs.h>
#include "moo_testbed_utils.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup moons_prepare_test  Unit test of moons_prepare
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/


/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of moons_dfs_set_groups
 */
/*----------------------------------------------------------------------------*/

char *
path_from_filename(const char *name)
{
    return cpl_sprintf("%s/%s", DATA_DIR, name);
}

char *
result_from_filename(const char *name)
{
    return cpl_sprintf("%s/%s", RESULT_DIR, name);
}

static void
_moons_test_removeCRH_mean()
{
    char *filename1 = path_from_filename("FLAT_CORRECTDARK_OFFSET1_LR_0.fits");
    char *filename2 = path_from_filename("FLAT_CORRECTDARK_OFFSET1_LR_1.fits");
    char *filename3 = path_from_filename("FLAT_CORRECTDARK_OFFSET1_LR_2.fits");

    cpl_frame *frame1 = cpl_frame_new();
    cpl_frame_set_filename(frame1, filename1);
    cpl_frame_set_tag(frame1, "TAG");
    cpl_frame_set_group(frame1, CPL_FRAME_GROUP_PRODUCT);

    cpl_frame *frame2 = cpl_frame_new();
    cpl_frame_set_filename(frame2, filename2);
    cpl_frame_set_tag(frame2, "TAG");
    cpl_frame_set_group(frame2, CPL_FRAME_GROUP_PRODUCT);

    cpl_frame *frame3 = cpl_frame_new();
    cpl_frame_set_filename(frame3, filename3);
    cpl_frame_set_tag(frame3, "TAG");
    cpl_frame_set_group(frame3, CPL_FRAME_GROUP_PRODUCT);

    cpl_frameset *frameset = cpl_frameset_new();
    cpl_frameset_insert(frameset, frame1);
    cpl_frameset_insert(frameset, frame2);
    cpl_frameset_insert(frameset, frame3);

    int nbraw = cpl_frameset_get_size(frameset);
    // 3- Create DET list
    moo_detlist *detlist = moo_detlist_create(frameset);
    moo_masklist *cosmicslist = moo_masklist_create(nbraw);

    // 4- call remove CRH
    moo_crh_params *params = moo_crh_params_new();
    params->method = MOO_CRH_METHOD_MEAN;
    moo_det *res = moo_remove_CRH(detlist, cosmicslist, params);

    char *resname = result_from_filename("DET_CRH_mean.fits");
    moo_det_save(res, resname);
    moo_det_delete(res);
    cpl_free(resname);

    // 5- Free memory
    cpl_free(filename1);
    cpl_free(filename2);
    cpl_free(filename3);
    moo_crh_params_delete(params);
    moo_detlist_delete(detlist);
    moo_masklist_delete(cosmicslist);
    cpl_frameset_delete(frameset);

    return;
}

static void
_moons_test_removeCRH_median()
{
    char *filename1 = path_from_filename("FLAT_CORRECTDARK_OFFSET1_LR_0.fits");
    char *filename2 = path_from_filename("FLAT_CORRECTDARK_OFFSET1_LR_1.fits");
    char *filename3 = path_from_filename("FLAT_CORRECTDARK_OFFSET1_LR_2.fits");

    cpl_frame *frame1 = cpl_frame_new();
    cpl_frame_set_filename(frame1, filename1);
    cpl_frame_set_tag(frame1, "TAG");
    cpl_frame_set_group(frame1, CPL_FRAME_GROUP_PRODUCT);

    cpl_frame *frame2 = cpl_frame_new();
    cpl_frame_set_filename(frame2, filename2);
    cpl_frame_set_tag(frame2, "TAG");
    cpl_frame_set_group(frame2, CPL_FRAME_GROUP_PRODUCT);

    cpl_frame *frame3 = cpl_frame_new();
    cpl_frame_set_filename(frame3, filename3);
    cpl_frame_set_tag(frame3, "TAG");
    cpl_frame_set_group(frame3, CPL_FRAME_GROUP_PRODUCT);

    cpl_frameset *frameset = cpl_frameset_new();
    cpl_frameset_insert(frameset, frame1);
    cpl_frameset_insert(frameset, frame2);
    cpl_frameset_insert(frameset, frame3);

    int nbraw = cpl_frameset_get_size(frameset);
    // 3- Create DET list
    moo_detlist *detlist = moo_detlist_create(frameset);
    moo_masklist *cosmicslist = moo_masklist_create(nbraw);

    // 4- call remove CRH
    moo_crh_params *params = moo_crh_params_new();
    params->method = MOO_CRH_METHOD_MEDIAN;
    moo_det *res = moo_remove_CRH(detlist, cosmicslist, params);

    char *resname = result_from_filename("DET_CRH_median.fits");
    moo_det_save(res, resname);
    moo_det_delete(res);
    cpl_free(resname);


    // 5- Free memory
    cpl_free(filename1);
    cpl_free(filename2);
    cpl_free(filename3);
    moo_crh_params_delete(params);
    moo_detlist_delete(detlist);
    moo_masklist_delete(cosmicslist);
    cpl_frameset_delete(frameset);

    return;
}

static void
_moons_test_removeCRH_median2()
{
    char *filename1 = path_from_filename("XSWITCH_NOD_DET_OFFSET0_HR_0.fits");

    cpl_frame *frame1 = cpl_frame_new();
    cpl_frame_set_filename(frame1, filename1);
    cpl_frame_set_tag(frame1, "TAG");
    cpl_frame_set_group(frame1, CPL_FRAME_GROUP_PRODUCT);


    cpl_frameset *frameset = cpl_frameset_new();
    cpl_frameset_insert(frameset, frame1);

    int nbraw = cpl_frameset_get_size(frameset);
    // 3- Create DET list
    moo_detlist *detlist = moo_detlist_create(frameset);

    // 4- call remove CRH
    moo_crh_params *params = moo_crh_params_new();
    params->method = MOO_CRH_METHOD_MEDIAN;
    moo_det *res = moo_remove_CRH(detlist, NULL, params);

    char *resname = result_from_filename("DET_CRH_median.fits");
    moo_det_save(res, resname);
    moo_det_delete(res);
    cpl_free(resname);


    // 5- Free memory
    cpl_free(filename1);
    moo_crh_params_delete(params);
    moo_detlist_delete(detlist);
    cpl_frameset_delete(frameset);

    return;
}


static void
_moons_test_removeCRH_sigclip()
{
    char *filename1 = path_from_filename("DET_FLAT_DARKCORR_0.fits");
    char *filename2 = path_from_filename("DET_FLAT_DARKCORR_1.fits");
    char *filename3 = path_from_filename("DET_FLAT_DARKCORR_2.fits");

    cpl_frame *frame1 = cpl_frame_new();
    cpl_frame_set_filename(frame1, filename1);
    cpl_frame_set_tag(frame1, "TAG");
    cpl_frame_set_group(frame1, CPL_FRAME_GROUP_PRODUCT);

    cpl_frame *frame2 = cpl_frame_new();
    cpl_frame_set_filename(frame2, filename2);
    cpl_frame_set_tag(frame2, "TAG");
    cpl_frame_set_group(frame2, CPL_FRAME_GROUP_PRODUCT);

    cpl_frame *frame3 = cpl_frame_new();
    cpl_frame_set_filename(frame3, filename3);
    cpl_frame_set_tag(frame3, "TAG");
    cpl_frame_set_group(frame3, CPL_FRAME_GROUP_PRODUCT);

    cpl_frameset *frameset = cpl_frameset_new();
    cpl_frameset_insert(frameset, frame1);
    cpl_frameset_insert(frameset, frame2);
    cpl_frameset_insert(frameset, frame3);

    int nbraw = cpl_frameset_get_size(frameset);
    // 3- Create DET list
    moo_detlist *detlist = moo_detlist_create(frameset);
    moo_masklist *cosmicslist = moo_masklist_create(nbraw);

    // 4- call remove CRH
    moo_crh_params *params = moo_crh_params_new();
    params->method = MOO_CRH_METHOD_SIGCLIP;
    params->kappa = 5.0;
    params->niter = 5;
    moo_det *res = moo_remove_CRH(detlist, cosmicslist, params);
    for (int k = 0; k < nbraw; k++) {
        moo_mask *mmask = moo_masklist_get(cosmicslist, k);
        char *mask_name = cpl_sprintf("MASK_%d.fits", k);
        char *fname = result_from_filename(mask_name);
        moo_mask_save(mmask, fname);
        cpl_free(mask_name);
        cpl_free(fname);
    }

    char *resname = result_from_filename("DET_CRH_sigclip.fits");
    moo_det_save(res, resname);
    moo_det_delete(res);
    cpl_free(resname);

    // 5- Free memory
    cpl_free(filename1);
    cpl_free(filename2);
    cpl_free(filename3);
    moo_crh_params_delete(params);
    moo_detlist_delete(detlist);
    moo_masklist_delete(cosmicslist);
    cpl_frameset_delete(frameset);

    return;
}

static void
_moons_test_removeCRH_sigclip_vis()
{
    char *filename1 = path_from_filename("2/DET_DARK_0.fits");
    char *filename2 = path_from_filename("2/DET_DARK_1.fits");
    char *filename3 = path_from_filename("2/DET_DARK_2.fits");

    cpl_frame *frame1 = cpl_frame_new();
    cpl_frame_set_filename(frame1, filename1);
    cpl_frame_set_tag(frame1, "TAG");
    cpl_frame_set_group(frame1, CPL_FRAME_GROUP_PRODUCT);

    cpl_frame *frame2 = cpl_frame_new();
    cpl_frame_set_filename(frame2, filename2);
    cpl_frame_set_tag(frame2, "TAG");
    cpl_frame_set_group(frame2, CPL_FRAME_GROUP_PRODUCT);

    cpl_frame *frame3 = cpl_frame_new();
    cpl_frame_set_filename(frame3, filename3);
    cpl_frame_set_tag(frame3, "TAG");
    cpl_frame_set_group(frame3, CPL_FRAME_GROUP_PRODUCT);

    cpl_frameset *frameset = cpl_frameset_new();
    cpl_frameset_insert(frameset, frame1);
    cpl_frameset_insert(frameset, frame2);
    cpl_frameset_insert(frameset, frame3);

    // 3- Create DET list
    moo_detlist *detlist = moo_detlist_create(frameset);
    moo_masklist *cosmicslist = NULL;

    // 4- call remove CRH
    moo_crh_params *params = moo_crh_params_new();
    params->method = MOO_CRH_METHOD_SIGCLIP;
    params->kappa = 5.0;
    params->niter = 5;
    moo_det *res = moo_remove_CRH(detlist, cosmicslist, params);

    char *resname = result_from_filename("DET_CRH_sigclip.fits");
    moo_det_save(res, resname);
    moo_det_delete(res);
    cpl_free(resname);

    // 5- Free memory
    cpl_free(filename1);
    cpl_free(filename2);
    cpl_free(filename3);
    moo_crh_params_delete(params);
    moo_detlist_delete(detlist);
    cpl_frameset_delete(frameset);

    return;
}

static void
_moons_test_removeCRH_median_qual()
{
    char *filename1 = path_from_filename("DET_FLAT_DARKCORR_0.fits");
    char *filename2 = path_from_filename("DET_FLAT_DARKCORR_1.fits");
    char *filename3 = path_from_filename("DET_FLAT_DARKCORR_2.fits");

    cpl_frame *frame1 = cpl_frame_new();
    cpl_frame_set_filename(frame1, filename1);
    cpl_frame_set_tag(frame1, "TAG");
    cpl_frame_set_group(frame1, CPL_FRAME_GROUP_PRODUCT);

    cpl_frame *frame2 = cpl_frame_new();
    cpl_frame_set_filename(frame2, filename2);
    cpl_frame_set_tag(frame2, "TAG");
    cpl_frame_set_group(frame2, CPL_FRAME_GROUP_PRODUCT);

    cpl_frame *frame3 = cpl_frame_new();
    cpl_frame_set_filename(frame3, filename3);
    cpl_frame_set_tag(frame3, "TAG");
    cpl_frame_set_group(frame3, CPL_FRAME_GROUP_PRODUCT);

    cpl_frameset *frameset = cpl_frameset_new();
    cpl_frameset_insert(frameset, frame1);
    cpl_frameset_insert(frameset, frame2);
    cpl_frameset_insert(frameset, frame3);

    // 3- Create DET list
    moo_detlist *detlist = moo_detlist_create(frameset);
    moo_masklist *cosmicslist = NULL;

    // 4- call remove CRH
    moo_crh_params *params = moo_crh_params_new();
    params->method = MOO_CRH_METHOD_SIGCLIP;
    params->kappa = 5.0;
    params->niter = 5;
    moo_det *res = NULL;
    char *resname = NULL;

    res = moo_remove_CRH(detlist, cosmicslist, params);

    resname = result_from_filename("DET_CRH_median.fits");
    moo_det_save(res, resname);

    moo_det_delete(res);
    cpl_free(resname);

    // 5- Free memory
    cpl_free(filename1);
    cpl_free(filename2);
    cpl_free(filename3);
    moo_crh_params_delete(params);
    moo_detlist_delete(detlist);
    cpl_frameset_delete(frameset);

    return;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit tests of moons_dfs module
 */
/*----------------------------------------------------------------------------*/

int
main()
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_INFO);
    _moons_test_removeCRH_median();

    return cpl_test_end(0);
}

/**@}*/
