/*
 * This file is part of the HDRL
 * Copyright (C) 2013,2014 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                    Includes
 -----------------------------------------------------------------------------*/
#include <cpl.h>

#include <stdlib.h>
#include <stdio.h>
#include "moo_badpix.h"
#include "moo_det.h"
#include "moo_subtract_nod.h"
#include "moo_params.h"

#ifdef _OPENMP
#include <omp.h>
#endif
/*----------------------------------------------------------------------------*/
/**
 * @defgroup drl_functions localise test
        Testing of functions used in moo_localise
 */
/*----------------------------------------------------------------------------*/

char *
path_from_filename(const char *name)
{
    return cpl_sprintf("%s/%s", DATA_DIR, name);
}

char *
result_from_filename(const char *name)
{
    return cpl_sprintf("%s/%s", RESULT_DIR, name);
}

cpl_frame *
create_frame(const char *name)
{
    char *filename = path_from_filename(name);

    cpl_frame *frame = cpl_frame_new();
    cpl_frame_set_filename(frame, filename);
    cpl_free(filename);
    cpl_frame_set_tag(frame, "TAG");
    cpl_frame_set_group(frame, CPL_FRAME_GROUP_PRODUCT);

    return frame;
}

void
_moo_subtract_nod(void)
{
    cpl_frame *obj_frame =
        create_frame("STARENOD_OBJ_CORRECTDARK_OFFSET1_HR_0.fits");
    cpl_frame *sky_frame =
        create_frame("STARENOD_SKY_CORRECTDARK_OFFSET1_HR_0.fits");

    char *resname =
        result_from_filename("STARENOD_SUBTRACTNOD_OFFSET1_HR_0.fits");

    moo_det *obj = moo_det_create(obj_frame);
    moo_det *sky = moo_det_create(sky_frame);
    moo_det_load(obj, MOO_BADPIX_GOOD);
    moo_det_load(sky, MOO_BADPIX_GOOD);
    moo_det *nod = NULL;

    nod = moo_subtract_nod(obj, sky, resname);

    moo_det_save(nod, resname);

    cpl_free(resname);
    moo_det_delete(nod);
    moo_det_delete(obj);
    moo_det_delete(sky);
    cpl_frame_delete(obj_frame);
    cpl_frame_delete(sky_frame);
}


/*----------------------------------------------------------------------------*/
/**
 @brief   Unit tests of subtract_nod module
 **/
/*----------------------------------------------------------------------------*/
int
main(void)
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_INFO);
    _moo_subtract_nod();
    return cpl_test_end(0);
}
