/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#include "math.h"
#include <string.h>
#include <cpl.h>

#include "moo_testbed_utils.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup moons_tests     Tests helper functions
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*----------------------------------------------------------------------------*/
/**
  @brief Create a moons raw file
  @param npix size of the image: npix×npix
  @param bands observed bands: 'RI','YJ' and 'H'
  @param nbbands number of observed bands
  @param ntas number of spectrographs
  @param output filename of the output
  @param gain gain value
  @param ron read-out-noise value
  @param exptime exposure time value
  @param val value of the image pixels
  @return   The FITS file

  The function returns a pointer to the statically allocated license string.
  This string should not be modified using the returned pointer.
 */
/*----------------------------------------------------------------------------*/
void
moo_testbed_create_raw(int npix,
                       const char *bands[],
                       int nbband,
                       int ntas,
                       const char *output,
                       double gain,
                       double ron,
                       double exptime,
                       double offset,
                       double val)
{
    cpl_propertylist *primary_header = cpl_propertylist_new();
    cpl_propertylist_append_float(primary_header, "ESO INS SLIT OFFSET",
                                  offset);

    cpl_propertylist_save(primary_header, output, CPL_IO_CREATE);
    cpl_propertylist_delete(primary_header);

    int i, j;
    for (i = 0; i < ntas; i++) {
        for (j = 0; j < nbband; j++) {
            char *extname = cpl_sprintf("%s_%d", bands[j], i + 1);

            cpl_propertylist *img_header = cpl_propertylist_new();
            cpl_propertylist_append_string(img_header, "EXTNAME", extname);
            cpl_propertylist_append_float(img_header, "EXPTIME", exptime);
            cpl_propertylist_append_int(img_header, "ESO DET OUTPUTS", 1);
            cpl_propertylist_append_float(img_header, "ESO DET OUT1 GAIN",
                                          gain);
            cpl_propertylist_append_float(img_header, "ESO DET OUT1 RON", ron);
            cpl_propertylist_append_int(img_header, "ESO DET OUT1 PRSCX", 0);
            cpl_propertylist_append_int(img_header, "ESO DET OUT1 PRSCY", 0);
            cpl_propertylist_append_int(img_header, "ESO DET OUT1 OVSCX", 0);
            cpl_propertylist_append_int(img_header, "ESO DET OUT1 OVSCY", 0);
            cpl_propertylist_append_int(img_header, "ESO DET OUT1 X", 1);
            cpl_propertylist_append_int(img_header, "ESO DET OUT1 Y", 1);
            cpl_propertylist_append_int(img_header, "ESO DET CHIP NX", npix);
            cpl_propertylist_append_int(img_header, "ESO DET CHIP NY", npix);
            cpl_propertylist_append_int(img_header, "ESO DET OUT1 NX", npix);
            cpl_propertylist_append_int(img_header, "ESO DET OUT1 NY", npix);
            cpl_propertylist_append_float(img_header, "CRPIX1", 1);
            cpl_propertylist_append_float(img_header, "CRPIX2", 1);
            cpl_propertylist_append_float(img_header, "CRVAL1", 1);
            cpl_propertylist_append_float(img_header, "CRVAL2", 1);
            cpl_propertylist_append_float(img_header, "CDELT1", 1);
            cpl_propertylist_append_float(img_header, "CD1_1", 1);
            cpl_propertylist_append_float(img_header, "CDELT2", 1);
            cpl_propertylist_append_float(img_header, "CD2_2", 1);
            cpl_propertylist_append_string(img_header, "CTYPE1", "PIXELS");
            cpl_propertylist_append_string(img_header, "CTYPE2", "PIXELS");
            cpl_propertylist_append_bool(img_header, "ESO DET LIVE", CPL_TRUE);
            cpl_propertylist_append_bool(img_header, "ESO DET CHIP LIVE",
                                         CPL_TRUE);
            if (npix == 0) {
                cpl_propertylist_save(img_header, output, CPL_IO_EXTEND);
            }
            else {
                cpl_type type;
                if (strcmp(bands[j], "RI") == 0) {
                    type = CPL_TYPE_INT;
                }
                else {
                    type = CPL_TYPE_FLOAT;
                }
                cpl_image *im = cpl_image_new(npix, npix, type);
                cpl_image_fill_window(im, 1, 1, npix, npix, val);
                cpl_image_save(im, output, type, img_header, CPL_IO_EXTEND);
                cpl_image_delete(im);
            }
            cpl_propertylist_delete(img_header);
            cpl_free(extname);
        }
    }
}

/*----------------------------------------------------------------------------*/
/**
  @brief Create a BPMAP file
  @param npix size of the image: npix×npix
  @param bands observed bands: 'RI','YJ' and 'H'
  @param nbbands number of observed bands
  @param ntas number of spectrographs
  @param output filename of the output
  @param badpix bad pixels position
  @param nbbadpix number of badpix
  @return   The FITS file

  The function returns a pointer to the statically allocated license string.
  This string should not be modified using the returned pointer.
 */
/*----------------------------------------------------------------------------*/
void
moo_testbed_create_bpmap(int npix,
                         const char *bands[],
                         int nbband,
                         int ntas,
                         const char *output,
                         int (*badpix)[][2],
                         int nbbadpix)
{
    cpl_propertylist *primary_header = cpl_propertylist_new();
    cpl_propertylist_save(primary_header, output, CPL_IO_CREATE);
    cpl_propertylist_delete(primary_header);

    int i, j, k;
    for (i = 0; i < ntas; i++) {
        for (j = 0; j < nbband; j++) {
            char *extname = cpl_sprintf("QUAL_%s_%d", bands[j], i + 1);

            cpl_propertylist *img_header = cpl_propertylist_new();
            cpl_propertylist_append_string(img_header, "EXTNAME", extname);

            cpl_image *im = cpl_image_new(npix, npix, CPL_TYPE_INT);

            for (k = 0; k < nbbadpix; k++) {
                int x = (*badpix)[i][0];
                int y = (*badpix)[i][0];
                cpl_image_set(im, x, y, 1);
            }
            cpl_image_save(im, output, CPL_TYPE_INT, img_header, CPL_IO_EXTEND);
            cpl_image_delete(im);

            cpl_propertylist_delete(img_header);
            cpl_free(extname);
        }
    }
}

cpl_frameset *
moo_testbed_frameset_load(const char *sof)
{
    cpl_frameset *set = cpl_frameset_new();
    FILE *file = NULL;
    file = fopen(sof, "r");
    char name[80];
    char tag[30];
    while (EOF != fscanf(file, "%s %s\n", name, tag)) {
        cpl_frame *frame = cpl_frame_new();
        cpl_frame_set_filename(frame, name);
        cpl_frame_set_tag(frame, tag);
        cpl_frameset_insert(set, frame);
    }

    fclose(file);
    return set;
}

/**@}*/
