/*
 * This file is part of the HDRL
 * Copyright (C) 2013,2014 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                    Includes
 -----------------------------------------------------------------------------*/
#include <cpl.h>

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "moo_det.h"
#include "moo_utils.h"
#include "moo_testbed_utils.h"
/*----------------------------------------------------------------------------*/
/**
 * @defgroup hdrl_imagelist_io-test 
        Testing of functions working on hdrl_imagelist
 */
/*----------------------------------------------------------------------------*/


static void _moo_fit_barycenter_test(void)
{   
    cpl_vector* pos = cpl_vector_new(5);
    cpl_vector* fluxes = cpl_vector_new(5);
    double center;
    double width;
    
    cpl_vector_set(pos, 0,1.18);
    cpl_vector_set(pos, 1,2);
    cpl_vector_set(pos, 2,3);
    cpl_vector_set(pos, 3,4);
    cpl_vector_set(pos, 4,4.37);
   
    cpl_vector_set(fluxes, 0, 20.0);
    cpl_vector_set(fluxes, 1,70.0);
    cpl_vector_set(fluxes, 2,81.54);
    cpl_vector_set(fluxes, 3,18.32);
    cpl_vector_set(fluxes, 4, 27.15);
    
    cpl_bivector* points = cpl_bivector_wrap_vectors(pos,fluxes);
    moo_barycenter_fit(points, &center, &width);
    
    cpl_test_rel(width,3.19, MOO_TESTS_DOUBLE_PRECISION);
    cpl_test_rel(center,2.765520, MOO_TESTS_DOUBLE_PRECISION);
    
    cpl_bivector_unwrap_vectors(points);
    cpl_vector_delete(pos);
    cpl_vector_delete(fluxes);
}

static void _moo_fit_barycenter_test2(void)
{   
    cpl_vector* pos = cpl_vector_new(6);
    cpl_vector* fluxes = cpl_vector_new(6);
    double center;
    double width;
    
    cpl_vector_set(pos, 0,0.75);
    cpl_vector_set(pos, 1,1.25);
    cpl_vector_set(pos, 2,1.75);    
    cpl_vector_set(pos, 3,2.25);
    cpl_vector_set(pos, 4,2.75);
    cpl_vector_set(pos, 5,3.25);
    
    cpl_vector_set(fluxes, 0, 0);
    cpl_vector_set(fluxes, 1,0);
    cpl_vector_set(fluxes, 2,1);    
    cpl_vector_set(fluxes, 3,1);    
    cpl_vector_set(fluxes, 4,2);    
    cpl_vector_set(fluxes, 5,2);    
    
    cpl_bivector* points = cpl_bivector_wrap_vectors(pos,fluxes);
    moo_barycenter_fit(points, &center, &width);
    
    cpl_msg_info("test","center at %f",center);
    cpl_test_rel(center,2.66666666, MOO_TESTS_DOUBLE_PRECISION);
    
    cpl_bivector_unwrap_vectors(points);
    cpl_vector_delete(pos);
    cpl_vector_delete(fluxes);
}

static void _moo_fit_linear_test(void)
{   
    cpl_bivector* fref = cpl_bivector_new(5);
    cpl_vector* fref_errs = cpl_vector_new(5);
    int fref_qual[5];
    cpl_bivector* fout = cpl_bivector_new(5);
    cpl_vector* fout_errs = cpl_vector_new(5);
    double errs[] = {0.1, 0.1118033988749895, 0.1, 0.07615773105863909, 0.1};
    int fout_qual[5];
    
    cpl_vector* fref_x = cpl_bivector_get_x(fref);
    cpl_vector* fref_y = cpl_bivector_get_y(fref);
    for(int i=0; i<5;i++){
        cpl_vector_set(fref_x,i,i);
    }
    for(int i=0; i<5;i++){
        cpl_vector_set(fref_y,i,i*10);
    }
    for(int i=0; i<5;i++){
        cpl_vector_set(fref_errs,i,0.1);
    }
    cpl_vector_set(fref_errs,2,0.2);
    for(int i=0; i<5;i++){
        fref_qual[i]=i;
    }
    
    
    cpl_vector* fout_x = cpl_bivector_get_x(fout);
    cpl_vector_set(fout_x,0,0);
    for(int i=1; i<4;i++){
        cpl_vector_set(fout_x,i,0.4+i+0.1*i);
    }
    cpl_vector_set(fout_x,4,4);
    moo_interpolate_linear(fout, fout_errs, fout_qual, 
            fref, fref_errs, fref_qual);
    
    for(int i=0;i<5;i++){
        double v = cpl_vector_get(fout_errs,i);
        cpl_test_rel(v,errs[i], MOO_TESTS_DOUBLE_PRECISION);
    }
    cpl_vector_delete(fref_errs);
    cpl_bivector_delete(fref);
    cpl_vector_delete(fout_errs);
    cpl_bivector_delete(fout);
}

static void _moo_find_treshold_limits_test(void)
{   
    cpl_vector* pos = cpl_vector_new(5);
    cpl_vector* fluxes = cpl_vector_new(5);
    
    cpl_vector_set(pos, 0,10);
    cpl_vector_set(pos, 1,11);
    cpl_vector_set(pos, 2,12);
    cpl_vector_set(pos, 3,13);
    cpl_vector_set(pos, 4,14);
   
    cpl_vector_set(fluxes, 0, 20.0);
    cpl_vector_set(fluxes, 1,30.0);
    cpl_vector_set(fluxes, 2,81.54);
    cpl_vector_set(fluxes, 3,35.3);
    cpl_vector_set(fluxes, 4, 17.15);
    
    cpl_bivector* points = cpl_bivector_wrap_vectors(pos,fluxes);
        
    double xmin;
    double xmax;
    moo_find_threshold_limits(points,29,&xmin,&xmax);
    cpl_test_rel(xmin,10.9,MOO_TESTS_DOUBLE_PRECISION);    
    cpl_test_rel(xmax,13.347107,MOO_TESTS_DOUBLE_PRECISION);            
    
    cpl_bivector_unwrap_vectors(points);
    cpl_vector_delete(pos);
    cpl_vector_delete(fluxes);
}

static void _moo_savgol_filter_test(void)
{   
    cpl_vector* data = cpl_vector_new(10);    
    
    cpl_vector_set(data, 0,2);
    cpl_vector_set(data, 1,2);
    cpl_vector_set(data, 2,5);
    cpl_vector_set(data, 3,2);
    cpl_vector_set(data, 4,1);
    cpl_vector_set(data, 5,NAN);
    cpl_vector_set(data, 6,0);    
    cpl_vector_set(data, 7,1);
    cpl_vector_set(data, 8,4);        
    cpl_vector_set(data, 9,9);
        
        
    int polyorder=3;
    int window_length=5;
    cpl_vector_dump(data,stdout);
    cpl_vector* res = moo_savgol_filter(data,window_length,polyorder);        
    cpl_msg_info("test","degpoly %d windowsize %d",polyorder,window_length);
    cpl_vector_dump(res,stdout);
    cpl_vector_delete(res);
    cpl_vector_delete(data);    
}

static void _moo_frameset_test(void)
{
    cpl_frameset* orig = cpl_frameset_new();
    cpl_frame* a = cpl_frame_new();
    cpl_frame_set_tag(a,"A");
    cpl_frame* b = cpl_frame_new();
    cpl_frame_set_tag(b,"B");
    
    cpl_frameset_insert(orig,a);
    cpl_frameset_insert(orig,b);
    
    for(int i=0; i< cpl_frameset_get_size(orig); i++){
        cpl_frame* f = cpl_frameset_get_position(orig,i);
        const char* tag = cpl_frame_get_tag(f);
        if ( strcmp(tag,"A")==0){
            cpl_frame_set_tag(f,"NEW_A");
        }
        
    }    
    cpl_msg_info("test","a %s",cpl_frame_get_tag(a));
    cpl_frame* t = cpl_frameset_get_position(orig,0);
    cpl_msg_info("test","t %s",cpl_frame_get_tag(t));
    cpl_frameset_delete(orig);
}
/*----------------------------------------------------------------------------*/
/**
 @brief   Unit tests of hdrl_image module
 **/
/*----------------------------------------------------------------------------*/

// good pixel (0)
#define BADPIX_GOOD 0x0 
// Electronic pickup noise (8)
#define BADPIX_NOISY 0x8
// Cosmic (unremoved) (32)
#define BADPIX_COSMIC_UNREMOVED 0x20
//Calibration file defect (128)
#define BADPIX_CALIB_DEFECT 0x80
// Hot pixel (256)
#define BADPIX_HOT 0x100
// Permanent camera defect (8192)
#define BADPIX_COSMETIC 0x2000
//  Non linear response pixel (32768)
#define BADPIX_NON_LINEAR 0x8000
//Outside data range (2^31)
#define BADPIX_OUTSIDE_DATA_RANGE 0x80000000
int main(void)
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_INFO);
    
    /*unsigned int bzero = 2147483648;
    
    cpl_image* test = cpl_image_new(3,3, CPL_TYPE_INT);
    
    cpl_image_set(test,1,1,(int)(BADPIX_GOOD));        
    cpl_image_set(test,1,2,(int)(BADPIX_NOISY));
    cpl_image_set(test,1,3,(int)(BADPIX_COSMIC_UNREMOVED));
    cpl_image_set(test,2,1,(int)(BADPIX_CALIB_DEFECT));
    cpl_image_set(test,2,2,(int)(BADPIX_HOT));
    cpl_image_set(test,2,3,(int)(BADPIX_COSMETIC));    
    cpl_image_set(test,3,1,(int)(BADPIX_NON_LINEAR));
    cpl_image_set(test,3,2,(int)(BADPIX_OUTSIDE_DATA_RANGE));        
    unsigned int t = BADPIX_OUTSIDE_DATA_RANGE | BADPIX_NOISY;
    cpl_msg_info("test","bef 3_3 %u",t);
    cpl_image_set(test,3,3,(int)(t));
        
    cpl_propertylist* h = cpl_propertylist_new();
    cpl_propertylist_append_double(h,"BZERO",bzero);
    
    cpl_image_save(test,"test.fits",CPL_TYPE_INT,h,CPL_IO_CREATE);    
    cpl_image_delete(test);
    
    test = cpl_image_load("test.fits",CPL_TYPE_INT,0,0);
    int rej;
    cpl_msg_info("test","test 1_1 %lf",cpl_image_get(test,1,1,&rej));
    cpl_msg_info("test","test 1_2 %lf",cpl_image_get(test,1,2,&rej));
    cpl_msg_info("test","test 1_3 %lf",cpl_image_get(test,1,3,&rej));
    
    cpl_msg_info("test","test 2_1 %lf",cpl_image_get(test,2,1,&rej));
    cpl_msg_info("test","test 2_2 %lf",cpl_image_get(test,2,2,&rej));
    cpl_msg_info("test","test 2_3 %lf",cpl_image_get(test,2,3,&rej));
    
    cpl_msg_info("test","test 3_1 %lf",cpl_image_get(test,3,1,&rej));
    cpl_msg_info("test","test 3_2 %u",(unsigned int)cpl_image_get(test,3,2,&rej));
    cpl_msg_info("test","test 3_3 %u",(unsigned int)cpl_image_get(test,3,3,&rej));   
    
    cpl_image_delete(test);
    cpl_propertylist_delete(h);
     * */
    _moo_fit_barycenter_test();
    /*_moo_find_treshold_limits_test();
    _moo_fit_linear_test();
    _moo_frameset_test();*/
    _moo_savgol_filter_test();
    return cpl_test_end(0);
}
