/*
 * This file is part of the HDRL
 * Copyright (C) 2013,2014 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                    Includes
 -----------------------------------------------------------------------------*/
#include <cpl.h>

#include <stdlib.h>
#include <stdio.h>

#include "moo_det.h"
#include "moo_spectral_format.h"
#include "moo_map.h"
#include "moo_badpix.h"
#include "moo_wavesol.h"
#include "moo_params.h"

#ifdef _OPENMP
#include <omp.h>
#endif
/*----------------------------------------------------------------------------*/
/**
 * @defgroup drl_functions localise test
        Testing of functions used in moo_localise
 */
/*----------------------------------------------------------------------------*/

char *
result_from_filename(const char *name)
{
    return cpl_sprintf("%s/%s", RESULT_DIR, name);
}

char *
path_from_filename(const char *name)
{
    return cpl_sprintf("%s/%s", DATA_DIR, name);
}

cpl_frame *
create_frame(const char *name)
{
    char *filename = path_from_filename(name);

    cpl_frame *frame = cpl_frame_new();
    cpl_frame_set_filename(frame, filename);
    cpl_free(filename);
    cpl_frame_set_tag(frame, "TAG");
    cpl_frame_set_group(frame, CPL_FRAME_GROUP_PRODUCT);

    return frame;
}

void
_moo_wavesol_lr_guess(void)
{
    cpl_frame *ext_frame = create_frame("ARC_EXTSPECTRA_OFFSET1_LR.fits");
    cpl_frame *cat_frame = create_frame("Ln11_C4_4.fits");
    const char *cat_name = cpl_frame_get_filename(cat_frame);
    cpl_frame *loc_frame = create_frame("FF_TRACE_OFFSET1_LR.fits");
    cpl_frame *spec_frame = create_frame("spectralfmt.fits");
    moo_wavesol_params *params = moo_wavesol_params_new();

    params->linedetect_winhsize[0] = 2;

    params->fwhm_min[0] = 2.2;
    params->fwhm_min[1] = 2.;
    params->fwhm_min[2] = 1.9;
    params->fwhm_min[3] = 2.6;
    params->fwhm_min[4] = 2.;
    params->fwhm_min[5] = 1.9;

    params->fwhm_max[0] = 3.4;
    params->fwhm_max[1] = 2.6;
    params->fwhm_max[2] = 2.7;
    params->fwhm_max[3] = 3.6;
    params->fwhm_max[4] = 2.6;
    params->fwhm_max[5] = 2.7;

    params->min_snr[0] = 25;
    params->tolerance[0] = 0.007;
    params->tolerance[1] = 0.009;
    params->tolerance[2] = 0.007;
    params->tolerance[3] = 0.007;
    params->tolerance[4] = 0.009;
    params->tolerance[5] = 0.007;

    params->linefit_winhsize[0] = 5;
    params->linefit_winhsize[1] = 3;
    params->linefit_winhsize[2] = 3;
    params->linefit_winhsize[3] = 3;
    params->linefit_winhsize[4] = 3;
    params->linefit_winhsize[5] = 3;

    params->linefit_method = MOO_WAVESOL_LINEFIT_GAUSSIAN;

    params->model = MOO_WAVESOL_MODEL_2D;

    params->ppm_wavemap_degx[0] = 3;
    params->ppm_wavemap_degy = 1;

    params->wavemap_degx[0] = 3;
    params->wavemap_degx[1] = 3;
    params->wavemap_degx[2] = 3;
    params->wavemap_degx[3] = 3;
    params->wavemap_degx[4] = 3;
    params->wavemap_degy = 1;

    params->clip_kappa = 2;
    params->clip_frac = 0.7;
    params->clip_niter = 10;

    params->linedetect_nlines[0] = 100;
    params->linedetect_nlines[1] = 100;
    params->linedetect_nlines[2] = 100;
    params->linedetect_nlines[3] = 100;
    params->linedetect_nlines[4] = 100;
    params->linedetect_nlines[5] = 100;

    params->control = MOO_WAVESOL_CONTROL_NONE;

    moo_loc *loc = moo_loc_load(loc_frame);
    moo_ext *ext = moo_ext_create(ext_frame);

    moo_spectral_format *sformat =
        moo_spectral_format_load(spec_frame, MOO_MODE_LR);
    moo_map *wave = moo_wavesol(ext, cat_name, sformat, loc, NULL, params);

    char *resname = result_from_filename("WAVE_MAP_GUESS_OFFSET1_LR.fits");
    moo_map_save(wave, resname);
    cpl_free(resname);

    moo_map_delete(wave);

    moo_spectral_format_delete(sformat);
    moo_loc_delete(loc);
    moo_ext_delete(ext);
    moo_wavesol_params_delete(params);

    cpl_frame_delete(ext_frame);
    cpl_frame_delete(cat_frame);
    cpl_frame_delete(loc_frame);
    cpl_frame_delete(spec_frame);
}

void
_moo_wavesol_hr_guess_refit(void)
{
    moo_map *wmap_guess = NULL;
    cpl_frame *ext_frame = create_frame("ARC_EXTSPECTRA_OFFSET1_HR.fits");
    cpl_frame *cat_frame = create_frame("Ln11_C4_4.fits");
    const char *cat_name = cpl_frame_get_filename(cat_frame);
    cpl_frame *loc_frame = create_frame("FF_TRACE_OFFSET1_HR.fits");
    cpl_frame *spec_frame = create_frame("spectralfmt.fits");
    moo_wavesol_params *params = moo_wavesol_params_new();
    cpl_frame *wmap_frame = create_frame("WAVE_MAP_PPM_GUESS_OFFSET1_HR.fits");

    params->fwhm_min[0] = 2.3;
    params->fwhm_min[1] = 2.;
    params->fwhm_min[2] = 2;
    params->fwhm_min[3] = 2.6;
    params->fwhm_min[4] = 2.;
    params->fwhm_min[5] = 1.9;

    params->fwhm_max[0] = 3.6;
    params->fwhm_max[1] = 2.6;
    params->fwhm_max[2] = 3.2;
    params->fwhm_max[3] = 3.6;
    params->fwhm_max[4] = 2.6;
    params->fwhm_max[5] = 2.7;

    params->tolerance[0] = 0.007;
    params->tolerance[1] = 0.009;
    params->tolerance[2] = 0.007;
    params->tolerance[3] = 0.007;
    params->tolerance[4] = 0.009;
    params->tolerance[5] = 0.007;

    params->min_snr[0] = 100;

    params->linefit_winhsize[0] = 25;
    params->linefit_winhsize[1] = 3;
    params->linefit_winhsize[2] = 3;
    params->linefit_winhsize[3] = 3;
    params->linefit_winhsize[4] = 3;
    params->linefit_winhsize[5] = 3;

    params->linefit_recentre[0] = 1;
    params->control = MOO_WAVESOL_CONTROL_NONE;

    params->linefit_method = MOO_WAVESOL_LINEFIT_GAUSSIAN;

    params->model = MOO_WAVESOL_MODEL_2D;

    params->wavemap_degx[0] = 3;
    params->wavemap_degx[1] = 3;
    params->wavemap_degx[2] = 3;
    params->wavemap_degx[3] = 3;
    params->wavemap_degx[4] = 3;
    params->wavemap_degy = 1;
    params->ppm_wavemap_degx[0] = 3;
    params->ppm_wavemap_degy = 1;

    params->clip_kappa = 2;
    params->clip_frac = 0.7;
    params->clip_niter = 10;

    params->linedetect_winhsize[0] = 5;

    params->linedetect_nlines[0] = 100;
    params->linedetect_nlines[1] = 100;
    params->linedetect_nlines[2] = 100;
    params->linedetect_nlines[3] = 100;
    params->linedetect_nlines[4] = 100;
    params->linedetect_nlines[5] = 100;

    moo_loc *loc = moo_loc_load(loc_frame);
    moo_ext *ext = moo_ext_create(ext_frame);

    /*moo_ext_single_delete(ext->ri[0]);
    ext->ri[0] = NULL;*/
    moo_ext_single_delete(ext->yj[0]);
    ext->yj[0] = NULL;

    moo_ext_single_delete(ext->h[0]);
    ext->h[0] = NULL;

    moo_ext_single_delete(ext->ri[1]);
    ext->ri[1] = NULL;
    moo_ext_single_delete(ext->yj[1]);
    ext->yj[1] = NULL;
    moo_ext_single_delete(ext->h[1]);
    ext->h[1] = NULL;
    moo_spectral_format *sformat =
        moo_spectral_format_load(spec_frame, MOO_MODE_HR);
    wmap_guess = moo_map_load(wmap_frame);
    moo_map *wave =
        moo_wavesol(ext, cat_name, sformat, loc, wmap_guess, params);

    char *resname = result_from_filename("WAVE_MAP_GUESS_2D.fits");
    moo_map_save(wave, resname);
    cpl_free(resname);

    moo_map_delete(wmap_guess);
    moo_map_delete(wave);

    moo_spectral_format_delete(sformat);
    moo_loc_delete(loc);
    moo_ext_delete(ext);
    moo_wavesol_params_delete(params);

    cpl_frame_delete(ext_frame);
    cpl_frame_delete(cat_frame);
    cpl_frame_delete(loc_frame);
    cpl_frame_delete(spec_frame);
    cpl_frame_delete(wmap_frame);
}

void
_moo_wavesol_hr_ppm(void)
{
    moo_map *wmap_guess = NULL;
    cpl_frame *ext_frame = create_frame("ARC_EXTSPECTRA_OFFSET1_HR.fits");
    cpl_frame *cat_frame = create_frame("Ln10clean_tot.fits");
    const char *arc_line_list_name = cpl_frame_get_filename(cat_frame);
    cpl_frame *loc_frame = create_frame("FF_TRACE_OFFSET1_HR.fits");
    cpl_frame *sformat_frame = create_frame("spectralfmt.fits");

    moo_wavesol_params *wavesol_params = moo_wavesol_params_new();
    wavesol_params->clip_frac = 0.7;
    wavesol_params->clip_kappa = 2;
    wavesol_params->clip_niter = 10;
    wavesol_params->control = MOO_WAVESOL_CONTROL_NONE;
    wavesol_params->fwhm_min[0] = 2.6;
    wavesol_params->fwhm_max[0] = 3.6;
    wavesol_params->isrefit = 0;
    wavesol_params->linedetect_nlines[0] = 80;
    wavesol_params->linedetect_winhsize[0] = 3;
    wavesol_params->linefit_method = MOO_WAVESOL_LINEFIT_GAUSSIAN;
    wavesol_params->linefit_recentre[0] = 0;
    wavesol_params->linefit_winhsize[0] = 15;
    wavesol_params->min_snr[0] = 4.;
    wavesol_params->model = MOO_WAVESOL_MODEL_2D;
    wavesol_params->ppm_wavemap_degx[0] = 3;
    wavesol_params->ppm_wavemap_degy = 1;
    wavesol_params->ppm_wavemap_degx[0] = 3;
    wavesol_params->ppm_wavemap_degy = 1;
    wavesol_params->tolerance[0] = 0.007;
    wavesol_params->wavemap_degx[0] = 3;
    wavesol_params->wavemap_degy = 1;

    moo_loc *loc = moo_loc_load(loc_frame);

    moo_ext *ext = moo_ext_create(ext_frame);
    moo_spectral_format *sformat =
        moo_spectral_format_load(sformat_frame, MOO_MODE_HR);

    cpl_msg_info("test", "REGDEBUG ext:%s", ext->filename);
    cpl_msg_info("test", "REGDEBUG arc:%s", arc_line_list_name);
    cpl_msg_info("test", "REGDEBUG sformat:%s",
                 cpl_frame_get_filename(sformat_frame));
    cpl_msg_info("test", "REGDEBUG loc:%s", loc->filename);
    cpl_msg_info("test", "REGDEBUG clip_frac:%f", wavesol_params->clip_frac);
    cpl_msg_info("test", "REGDEBUG clip_kappa:%f", wavesol_params->clip_kappa);
    cpl_msg_info("test", "REGDEBUG clip_niter:%d", wavesol_params->clip_niter);
    cpl_msg_info("test", "REGDEBUG control:%s", wavesol_params->control);
    cpl_msg_info("test", "REGDEBUG fwhm:%f,%f", wavesol_params->fwhm_min[0],
                 wavesol_params->fwhm_max[0]);
    cpl_msg_info("test", "REGDEBUG is_refit:%d", wavesol_params->isrefit);
    cpl_msg_info("test", "REGDEBUG ld_nlines:%d",
                 wavesol_params->linedetect_nlines[0]);
    cpl_msg_info("test", "REGDEBUG ld_whsize:%d",
                 wavesol_params->linedetect_winhsize[0]);
    cpl_msg_info("test", "REGDEBUG lf_method:%s",
                 wavesol_params->linefit_method);
    cpl_msg_info("test", "REGDEBUG lf_recentre:%d",
                 wavesol_params->linefit_recentre[0]);
    cpl_msg_info("test", "REGDEBUG lf_whsize:%d",
                 wavesol_params->linefit_winhsize[0]);
    cpl_msg_info("test", "REGDEBUG min_snr:%f", wavesol_params->min_snr[0]);
    cpl_msg_info("test", "REGDEBUG model:%s", wavesol_params->model);
    cpl_msg_info("test", "REGDEBUG ppm_wm_dx:%d",
                 wavesol_params->ppm_wavemap_degx[0]);
    cpl_msg_info("test", "REGDEBUG ppm_wm_dy:%d",
                 wavesol_params->ppm_wavemap_degy);
    cpl_msg_info("test", "REGDEBUG tolerance:%f", wavesol_params->tolerance[0]);
    cpl_msg_info("test", "REGDEBUG wm_dx:%d", wavesol_params->wavemap_degx[0]);
    cpl_msg_info("test", "REGDEBUG wm_dy:%d", wavesol_params->wavemap_degy);
    wmap_guess = moo_wavesol(ext, arc_line_list_name, sformat, loc, NULL,
                             wavesol_params);
    moo_map_save(wmap_guess, "WAVE_MAP_PPM_GUESS_OFFSET0_HR.fits");

    moo_map_delete(wmap_guess);
    moo_spectral_format_delete(sformat);
    moo_loc_delete(loc);
    moo_ext_delete(ext);
    moo_wavesol_params_delete(wavesol_params);

    cpl_frame_delete(cat_frame);
    cpl_frame_delete(ext_frame);
    cpl_frame_delete(loc_frame);
    cpl_frame_delete(sformat_frame);
}

void
_moo_wavesol_hr_refit(void)
{
    moo_map *wmap_guess = NULL;
    moo_map *wmap_refit = NULL;
    cpl_frame *ext_frame = create_frame("refit/ARC_EXTSPECTRA_OFFSET0_HR.fits");
    cpl_frame *cat_frame = create_frame("refit/Ln10clean_tot.fits");
    const char *arc_line_list_name = cpl_frame_get_filename(cat_frame);
    cpl_frame *loc_frame = create_frame("refit/FF_TRACE_OFFSET0_HR.fits");
    cpl_frame *sformat_frame = create_frame("refit/spectralfmt.fits");
    cpl_frame *wmap_frame =
        create_frame("refit/WAVE_MAP_PPM_GUESS_OFFSET0_HR.fits");

    moo_wavesol_params *wavesol_params = moo_wavesol_params_new();
    wavesol_params->clip_frac = 0.7;
    wavesol_params->clip_kappa = 2;
    wavesol_params->clip_niter = 10;
    wavesol_params->control = MOO_WAVESOL_CONTROL_NONE;
    wavesol_params->fwhm_min[0] = 2.6;
    wavesol_params->fwhm_max[0] = 3.6;
    wavesol_params->isrefit = 1;
    wavesol_params->linedetect_nlines[0] = 80;
    wavesol_params->linedetect_winhsize[0] = 3;
    wavesol_params->linefit_method = MOO_WAVESOL_LINEFIT_GAUSSIAN;
    wavesol_params->linefit_recentre[0] = 0;
    wavesol_params->linefit_winhsize[0] = 15;
    wavesol_params->min_snr[0] = 4.;
    wavesol_params->model = MOO_WAVESOL_MODEL_2D;
    wavesol_params->ppm_wavemap_degx[0] = 3;
    wavesol_params->ppm_wavemap_degy = 1;
    wavesol_params->tolerance[0] = 0.007;
    wavesol_params->wavemap_degx[0] = 3;
    wavesol_params->wavemap_degy = 1;

    moo_loc *loc = moo_loc_load(loc_frame);

    moo_ext *ext = moo_ext_create(ext_frame);
    moo_spectral_format *sformat =
        moo_spectral_format_load(sformat_frame, MOO_MODE_HR);
    wmap_guess = moo_map_load(wmap_frame);

    cpl_msg_info("test", "REGDEBUG ext:%s", ext->filename);
    cpl_msg_info("test", "REGDEBUG arc:%s", arc_line_list_name);
    cpl_msg_info("test", "REGDEBUG sformat:%s",
                 cpl_frame_get_filename(sformat_frame));
    cpl_msg_info("test", "REGDEBUG wmap:%s", wmap_guess->filename);

    cpl_msg_info("test", "REGDEBUG loc:%s", loc->filename);
    cpl_msg_info("test", "REGDEBUG clip_frac:%f", wavesol_params->clip_frac);
    cpl_msg_info("test", "REGDEBUG clip_kappa:%f", wavesol_params->clip_kappa);
    cpl_msg_info("test", "REGDEBUG clip_niter:%d", wavesol_params->clip_niter);
    cpl_msg_info("test", "REGDEBUG control:%s", wavesol_params->control);
    cpl_msg_info("test", "REGDEBUG fwhm:%f,%f", wavesol_params->fwhm_min[0],
                 wavesol_params->fwhm_max[0]);
    cpl_msg_info("test", "REGDEBUG is_refit:%d", wavesol_params->isrefit);
    cpl_msg_info("test", "REGDEBUG ld_nlines:%d",
                 wavesol_params->linedetect_nlines[0]);
    cpl_msg_info("test", "REGDEBUG ld_whsize:%d",
                 wavesol_params->linedetect_winhsize[0]);
    cpl_msg_info("test", "REGDEBUG lf_method:%s",
                 wavesol_params->linefit_method);
    cpl_msg_info("test", "REGDEBUG lf_recentre:%d",
                 wavesol_params->linefit_recentre[0]);
    cpl_msg_info("test", "REGDEBUG lf_whsize:%d",
                 wavesol_params->linefit_winhsize[0]);
    cpl_msg_info("test", "REGDEBUG min_snr:%f", wavesol_params->min_snr[0]);
    cpl_msg_info("test", "REGDEBUG model:%s", wavesol_params->model);
    cpl_msg_info("test", "REGDEBUG ppm_wm_dx:%d",
                 wavesol_params->ppm_wavemap_degx[0]);
    cpl_msg_info("test", "REGDEBUG ppm_wm_dy:%d",
                 wavesol_params->ppm_wavemap_degy);
    cpl_msg_info("test", "REGDEBUG tolerance:%f", wavesol_params->tolerance[0]);
    cpl_msg_info("test", "REGDEBUG wm_dx:%d", wavesol_params->wavemap_degx[0]);
    cpl_msg_info("test", "REGDEBUG wm_dy:%d", wavesol_params->wavemap_degy);

    wmap_refit = moo_wavesol(ext, arc_line_list_name, sformat, loc, wmap_guess,
                             wavesol_params);
    moo_map_save(wmap_refit, "WAVE_MAP_REFIT_GUESS_OFFSET0_HR.fits");

    moo_map_delete(wmap_refit);
    moo_map_delete(wmap_guess);
    moo_spectral_format_delete(sformat);
    moo_loc_delete(loc);
    moo_ext_delete(ext);
    moo_wavesol_params_delete(wavesol_params);
    cpl_frame_delete(wmap_frame);
    cpl_frame_delete(cat_frame);
    cpl_frame_delete(ext_frame);
    cpl_frame_delete(loc_frame);
    cpl_frame_delete(sformat_frame);
}

void
_moo_wavesol(void)
{
    moo_map *wave = NULL;
    cpl_frame *ext_frame = create_frame("ARC_EXTSPECTRA_OFFSET0_LR.fits");
    cpl_frame *cat_frame = create_frame("Ln07clean.fits");
    const char *cat_name = cpl_frame_get_filename(cat_frame);
    cpl_frame *loc_frame = create_frame("FF_TRACE_OFFSET0_LR.fits");
    cpl_frame *spec_frame = create_frame("spectralfmt.fits");
    cpl_frame *wmap_frame = create_frame("WAVE_MAP_GUESS_OFFSET0_LR.fits");

    moo_wavesol_params *params = moo_wavesol_params_new();

    params->linefit_winhsize[0] = 3;
    params->linefit_winhsize[1] = 3;
    params->linefit_winhsize[2] = 3;

    params->fwhm_min[0] = 2.6;
    params->fwhm_min[1] = 2.;
    params->fwhm_min[2] = 1.9;
    params->fwhm_min[3] = 2.6;
    params->fwhm_min[4] = 2.;
    params->fwhm_min[5] = 1.9;
    params->fwhm_max[0] = 3.6;
    params->fwhm_max[1] = 2.6;
    params->fwhm_max[2] = 2.7;
    params->fwhm_max[3] = 3.6;
    params->fwhm_max[4] = 2.6;
    params->fwhm_max[5] = 2.7;

    params->linefit_method = MOO_WAVESOL_LINEFIT_GAUSSIAN;

    params->clip_niter = 10;
    params->clip_frac = 0.7;
    params->clip_kappa = 2;

    params->wavemap_degx[0] = 3;
    params->wavemap_degx[1] = 3;
    params->wavemap_degx[2] = 3;
    params->wavemap_degx[3] = 3;
    params->wavemap_degx[4] = 3;
    params->wavemap_degy = 2;

    params->model = MOO_WAVESOL_MODEL_1D;

    moo_loc *loc = moo_loc_load(loc_frame);

    moo_ext *ext = moo_ext_create(ext_frame);
    moo_ext_single_delete(ext->yj[0]);
    ext->yj[0] = NULL;
    moo_ext_single_delete(ext->h[0]);
    ext->h[0] = NULL;
    moo_ext_single_delete(ext->ri[1]);
    ext->ri[1] = NULL;
    moo_ext_single_delete(ext->yj[1]);
    ext->yj[1] = NULL;
    moo_ext_single_delete(ext->h[1]);
    ext->h[1] = NULL;

    moo_spectral_format *sformat =
        moo_spectral_format_load(spec_frame, MOO_MODE_LR);
    moo_map *wmap_guess = moo_map_load(wmap_frame);

    wave = moo_wavesol(ext, cat_name, sformat, loc, wmap_guess, params);

    moo_map_save(wave, "WAVE_MAP_OFFSET0_HR.fits");

    moo_map_delete(wave);
    moo_spectral_format_delete(sformat);
    moo_map_delete(wmap_guess);
    moo_loc_delete(loc);
    moo_ext_delete(ext);
    moo_wavesol_params_delete(params);

    cpl_frame_delete(wmap_frame);
    cpl_frame_delete(cat_frame);
    cpl_frame_delete(ext_frame);
    cpl_frame_delete(loc_frame);
    cpl_frame_delete(spec_frame);
}

void
_moo_wavesol_lr_final(void)
{
    moo_map *wave = NULL;
    cpl_frame *ext_frame = create_frame("ryj/ARC_EXTSPECTRA_OFFSET0_LR.fits");
    cpl_frame *cat_frame = create_frame("ryj/Ln10cleanC3_2.fits");
    const char *cat_name = cpl_frame_get_filename(cat_frame);
    cpl_frame *loc_frame = create_frame("ryj/FF_TRACE_OFFSET0_LR.fits");
    cpl_frame *spec_frame = create_frame("ryj/spectralfmtC3.fits");
    cpl_frame *wmap_frame = create_frame("ryj/WAVE_MAP_GUESS_OFFSET0_LR.fits");

    moo_wavesol_params *params = moo_wavesol_params_new();

    params->min_snr[1] = 8.;
    params->control = MOO_WAVESOL_CONTROL_NONE;

    params->linefit_winhsize[0] = 3;
    params->linefit_winhsize[1] = 3;
    params->linefit_winhsize[2] = 3;

    params->fwhm_min[0] = 2.6;
    params->fwhm_min[1] = 2.;
    params->fwhm_min[2] = 1.9;
    params->fwhm_min[3] = 2.6;
    params->fwhm_min[4] = 2.;
    params->fwhm_min[5] = 1.9;

    params->fwhm_max[0] = 3.6;
    params->fwhm_max[1] = 2.6;
    params->fwhm_max[2] = 2.7;
    params->fwhm_max[3] = 3.6;
    params->fwhm_max[4] = 2.6;
    params->fwhm_max[5] = 2.7;

    params->linefit_method = MOO_WAVESOL_LINEFIT_GAUSSIAN;

    params->clip_niter = 10;
    params->clip_frac = 0.7;
    params->clip_kappa = 2;

    params->wavemap_degx[0] = 3;
    params->wavemap_degx[1] = 4;
    params->wavemap_degx[2] = 3;
    params->wavemap_degx[3] = 3;
    params->wavemap_degx[4] = 3;
    params->wavemap_degy = 1;

    params->model = MOO_WAVESOL_MODEL_2D;

    moo_loc *loc = moo_loc_load(loc_frame);
    moo_ext *ext = moo_ext_create(ext_frame);

    moo_spectral_format *sformat =
        moo_spectral_format_load(spec_frame, MOO_MODE_LR);
    moo_map *wmap_guess = moo_map_load(wmap_frame);

    wave = moo_wavesol(ext, cat_name, sformat, loc, wmap_guess, params);
    moo_map_save(wave, "WAVE_MAP_OFFSET0_LR.fits");

    moo_map_delete(wave);
    moo_spectral_format_delete(sformat);
    moo_map_delete(wmap_guess);
    moo_loc_delete(loc);
    moo_ext_delete(ext);
    moo_wavesol_params_delete(params);

    cpl_frame_delete(wmap_frame);
    cpl_frame_delete(cat_frame);
    cpl_frame_delete(ext_frame);
    cpl_frame_delete(loc_frame);
    cpl_frame_delete(spec_frame);
}

/*----------------------------------------------------------------------------*/
/**
 @brief   Unit tests of hdrl_image module
 **/
/*----------------------------------------------------------------------------*/
int
main(void)
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_INFO);
    _moo_wavesol_hr_ppm();
    //_moo_wavesol_lr_final();
    //_moo_wavesol_hr_refit();
    //_moo_wavesol();
    return cpl_test_end(0);
}
