/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#include "moo_utils.h"
#include "moo_badpix.h"
#include "moo_pfits.h"
#include "moo_dfs.h"
#include "moo_raw.h"
#include "moo_params.h"
#include "moo_detector.h"
#include "moo_drl.h"
#include "moo_products.h"
#include "moo_compute_linearity.h"
#include <cpl.h>
#include <hdrl.h>
#include <math.h>
#include <string.h>
#include <irplib_utils.h>
#include <irplib_plugin.h>

#define RECIPE_NAME "moons_linear"
/*-----------------------------------------------------------------------------
                              Plugin registration
 -----------------------------------------------------------------------------*/

int cpl_plugin_get_info(cpl_pluginlist *list);

/*-----------------------------------------------------------------------------
                            Private function prototypes
 -----------------------------------------------------------------------------*/

static int _moons_linear_create(cpl_plugin *plugin);
static int _moons_linear_exec(cpl_plugin *plugin);
static int _moons_linear_destroy(cpl_plugin *plugin);
static int
_moons_linear(cpl_frameset *frameset, const cpl_parameterlist *parlist);
/*-----------------------------------------------------------------------------
                            Static variables
 -----------------------------------------------------------------------------*/

static const char *const _moons_linear_description =
    "INPUT FRAMES\n"
    "NOTE: m refers to frames with different exposure time with m >= 3\n"
    "NOTE: n refers to frames with same exposure time with n >= 1\n"
    "  * RawList1 n1 x m1 files                   (RAW) with tag LINEARITY : "
    "linearity files at offset 0\n"
    "  * RawList2 n2 x m2 files                   (RAW) with tag LINEARITY : "
    "linearity files at offset 1\n"
    "  * [OPTIONAL] trace1  1 file                (LOC) with tag "
    "FF_TRACE_GUESS : "
    "guess trace for offset 0\n"
    "  * [OPTIONAL] trace2  1 file                (LOC) with tag "
    "FF_TRACE_GUESS : "
    "guess trace for offset 1\n"
    "  * [OPTIONAL] ReferenceBadPixMask 1 file    (QUA) with tag BP_MAP_RP : "
    "cosmetic bad pixel map\n"
    "PRODUCTS\n"
    "  * BP_MAP_NL.fits                           (QUA) with tag BP_MAP_NL : "
    "the linearity bad pixel mask\n"
    "  * LINEARITY_COEFF_CUBE_OFFSET[offset].fits (3D) with tag "
    "LINEARITY_COEFF_CUBE\n";

/*-----------------------------------------------------------------------------
                                Function code
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Build the list of available plugins, for this module.
  @param    list    the plugin list
  @return   0 if everything is ok, 1 otherwise
  @note     Only this function is exported

  Create the recipe instance and make it available to the application using the
  interface.
 */
/*----------------------------------------------------------------------------*/

int
cpl_plugin_get_info(cpl_pluginlist *list)
{
    cpl_recipe *recipe = cpl_calloc(1, sizeof *recipe);
    cpl_plugin *plugin = &recipe->interface;

    if (cpl_plugin_init(
            plugin, CPL_PLUGIN_API, MOONS_BINARY_VERSION,
            CPL_PLUGIN_TYPE_RECIPE, "moons_linear",
            "Produces a bad pixel mask with a series of linearity exposures",
            _moons_linear_description, "Regis Haigron", PACKAGE_BUGREPORT,
            moo_get_license(), _moons_linear_create, _moons_linear_exec,
            _moons_linear_destroy)) {
        cpl_msg_error(cpl_func, "Plugin initialization failed");
        (void)cpl_error_set_where(cpl_func);
        return 1;
    }

    if (cpl_pluginlist_append(list, plugin)) {
        cpl_msg_error(cpl_func, "Error adding plugin to list");
        (void)cpl_error_set_where(cpl_func);
        return 1;
    }

    return 0;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Setup the recipe options
  @param    plugin  the plugin
  @return   0 if everything is ok

  Defining the command-line/configuration parameters for the recipe.
 */
/*----------------------------------------------------------------------------*/

static int
_moons_linear_create(cpl_plugin *plugin)
{
    cpl_recipe *recipe;

    /* Do not create the recipe if an error code is already set */
    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        cpl_msg_error(cpl_func, "%s():%d: An error is already set: %s",
                      cpl_func, __LINE__, cpl_error_get_where());
        return (int)cpl_error_get_code();
    }

    if (plugin == NULL) {
        cpl_msg_error(cpl_func, "Null plugin");
        cpl_ensure_code(0, (int)CPL_ERROR_NULL_INPUT);
    }

    /* Verify plugin type */
    if (cpl_plugin_get_type(plugin) != CPL_PLUGIN_TYPE_RECIPE) {
        cpl_msg_error(cpl_func, "Plugin is not a recipe");
        cpl_ensure_code(0, (int)CPL_ERROR_TYPE_MISMATCH);
    }

    /* Get the recipe */
    recipe = (cpl_recipe *)plugin;

    /* Create the parameters list in the cpl_recipe object */
    recipe->parameters = cpl_parameterlist_new();
    if (recipe->parameters == NULL) {
        cpl_msg_error(cpl_func, "Parameter list allocation failed");
        cpl_ensure_code(0, (int)CPL_ERROR_ILLEGAL_OUTPUT);
    }

    moo_params *params = moo_params_new("moons", "moons_linear");
    moo_params_add_keep_temp(params, recipe->parameters);
    moo_params_add_prepare(params, recipe->parameters);
    moo_params_add_linear(params, recipe->parameters);
    moo_params_delete(params);

    return 0;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Execute the plugin instance given by the interface
  @param    plugin  the plugin
  @return   0 if everything is ok
 */
/*----------------------------------------------------------------------------*/

static int
_moons_linear_exec(cpl_plugin *plugin)
{
    cpl_recipe *recipe;
    int recipe_status;
    cpl_errorstate initial_errorstate = cpl_errorstate_get();

    /* Return immediately if an error code is already set */
    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        cpl_msg_error(cpl_func, "%s():%d: An error is already set: %s",
                      cpl_func, __LINE__, cpl_error_get_where());
        return (int)cpl_error_get_code();
    }

    if (plugin == NULL) {
        cpl_msg_error(cpl_func, "Null plugin");
        cpl_ensure_code(0, (int)CPL_ERROR_NULL_INPUT);
    }

    /* Verify plugin type */
    if (cpl_plugin_get_type(plugin) != CPL_PLUGIN_TYPE_RECIPE) {
        cpl_msg_error(cpl_func, "Plugin is not a recipe");
        cpl_ensure_code(0, (int)CPL_ERROR_TYPE_MISMATCH);
    }

    /* Get the recipe */
    recipe = (cpl_recipe *)plugin;

    /* Verify parameter and frame lists */
    if (recipe->parameters == NULL) {
        cpl_msg_error(cpl_func, "Recipe invoked with NULL parameter list");
        cpl_ensure_code(0, (int)CPL_ERROR_NULL_INPUT);
    }
    if (recipe->frames == NULL) {
        cpl_msg_error(cpl_func, "Recipe invoked with NULL frame set");
        cpl_ensure_code(0, (int)CPL_ERROR_NULL_INPUT);
    }

    /* Invoke the recipe */
    recipe_status = _moons_linear(recipe->frames, recipe->parameters);

    /* Ensure DFS-compliance of the products */
    if (cpl_dfs_update_product_header(recipe->frames)) {
        if (!recipe_status)
            recipe_status = (int)cpl_error_get_code();
    }

    if (!cpl_errorstate_is_equal(initial_errorstate)) {
        /* Dump the error history since recipe execution start.
           At this point the recipe cannot recover from the error */
        cpl_errorstate_dump(initial_errorstate, CPL_FALSE, NULL);
    }

    return recipe_status;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Destroy what has been created by the 'create' function
  @param    plugin  the plugin
  @return   0 if everything is ok
 */
/*----------------------------------------------------------------------------*/

static int
_moons_linear_destroy(cpl_plugin *plugin)
{
    cpl_recipe *recipe;

    if (plugin == NULL) {
        cpl_msg_error(cpl_func, "Null plugin");
        cpl_ensure_code(0, (int)CPL_ERROR_NULL_INPUT);
    }

    /* Verify plugin type */
    if (cpl_plugin_get_type(plugin) != CPL_PLUGIN_TYPE_RECIPE) {
        cpl_msg_error(cpl_func, "Plugin is not a recipe");
        cpl_ensure_code(0, (int)CPL_ERROR_TYPE_MISMATCH);
    }

    /* Get the recipe */
    recipe = (cpl_recipe *)plugin;

    cpl_parameterlist_delete(recipe->parameters);

    return 0;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Interpret the command line options and execute the data processing
  @param    frameset   the frames list
  @param    parlist    the parameters list
  @return   0 if everything is ok
 */
/*----------------------------------------------------------------------------*/
static int
_moons_compare_offset(const cpl_frame *a, const cpl_frame *b)
{
    int res = 0;
    const char *filenameA = cpl_frame_get_filename(a);
    cpl_propertylist *lista = cpl_propertylist_load(filenameA, 0);

    double oA = moo_pfits_get_slit_offset(lista);
    const char *filenameB = cpl_frame_get_filename(b);
    cpl_propertylist *listb = cpl_propertylist_load(filenameB, 0);

    double oB = moo_pfits_get_slit_offset(listb);
    if (fabs(oA - oB) < 1E-12) {
        res = 1;
    }

    cpl_propertylist_delete(lista);
    cpl_propertylist_delete(listb);
    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Interpret the command line options and execute the data processing
  @param    frameset   the frames list
  @param    parlist    the parameters list
  @return   0 if everything is ok
 */
/*----------------------------------------------------------------------------*/
static int
_moons_compare_exptime(const cpl_frame *a, const cpl_frame *b)
{
    int res = 0;

    const char *filenameA = cpl_frame_get_filename(a);
    const char *filenameB = cpl_frame_get_filename(b);

    cpl_propertylist *lista = cpl_propertylist_load(filenameA, 0);
    if (cpl_propertylist_has(lista, MOO_PFITS_EXPTIME)) {
        double exptimeA = moo_pfits_get_exptime(lista);
        cpl_propertylist *listb = cpl_propertylist_load(filenameB, 0);
        double exptimeB = moo_pfits_get_exptime(listb);
        if (fabs(exptimeA - exptimeB) < 1E-12) {
            res = 1;
        }
        cpl_propertylist_delete(listb);
    }
    cpl_propertylist_delete(lista);
    return res;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Interpret the command line options and execute the data processing
  @param    frameset   the frames list
  @param    parlist    the parameters list
  @return   0 if everything is ok
 */
/*----------------------------------------------------------------------------*/
static int
_moons_sort_exptime(const cpl_frame *a, const cpl_frame *b)
{
    int res = 0;

    const char *filenameA = cpl_frame_get_filename(a);
    const char *filenameB = cpl_frame_get_filename(b);

    cpl_propertylist *lista = cpl_propertylist_load(filenameA, 0);
    if (cpl_propertylist_has(lista, MOO_PFITS_EXPTIME)) {
        double exptimeA = moo_pfits_get_exptime(lista);
        cpl_propertylist *listb = cpl_propertylist_load(filenameB, 0);
        double exptimeB = moo_pfits_get_exptime(listb);
        res = exptimeA - exptimeB;
        cpl_propertylist_delete(listb);
    }
    cpl_propertylist_delete(lista);
    return res;
}

static cpl_error_code
_moons_compute_linearity(cpl_frameset *exptime_comb_frames,
                         cpl_frameset *trace_frames,
                         cpl_frameset *ref_frames,
                         moo_linear_params *params,
                         moo_products *products)
{
    moo_loc **loc_tab = NULL;
    moo_bpm **bpm_tab = NULL;
    moo_cube **cube_tab = NULL;
    moo_saturate_map **saturate_map_tab = NULL;
    cpl_frame **refframe_tab = NULL;
    char *cube_name = NULL;
    char *bpm_name = NULL;
    int offsets[2];

    cpl_ensure_code(exptime_comb_frames != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(trace_frames != NULL, CPL_ERROR_NULL_INPUT);

    cpl_msg_info(__func__, "Compute linearity coefficients cube");
    cpl_msg_indent_more();
    cpl_frameset_join(exptime_comb_frames, trace_frames);

    cpl_size nsel = 0;
    cpl_size *selection = NULL;

    cube_tab = cpl_calloc(2, sizeof(moo_cube));
    loc_tab = cpl_calloc(2, sizeof(moo_loc));
    saturate_map_tab = cpl_calloc(2, sizeof(moo_saturate_map));
    bpm_tab = cpl_calloc(2, sizeof(moo_bpm));
    refframe_tab = cpl_calloc(2, sizeof(cpl_frame *));

    moo_try_check(selection =
                      cpl_frameset_labelise(exptime_comb_frames,
                                            &_moons_compare_offset, &nsel),
                  " ");
    int nb_bpm = 0;
    int nb_cube = 0;

    for (int i = 0; i < nsel; i++) {
        cpl_frameset *set =
            cpl_frameset_extract(exptime_comb_frames, selection, i);
        cpl_frameset *refset = cpl_frameset_extract(ref_frames, selection, i);

        cpl_frame *loc_frame = cpl_frameset_find(set, MOONS_TAG_FF_TRACE_GUESS);
        if (loc_frame != NULL) {
            cpl_msg_info("moons_linear", "Use ff_trace: %s",
                         cpl_frame_get_filename(loc_frame));
            loc_tab[i] = moo_loc_load(loc_frame);
            cpl_frameset_erase_frame(set, loc_frame);
        }
        int size = cpl_frameset_get_size(set);

        if (size > 3) {
            const cpl_frame *ref_frame =
                cpl_frameset_get_position_const(refset, 0);
            refframe_tab[i] = cpl_frame_duplicate(ref_frame);
            moo_detlist *detlist = moo_detlist_create(set);
            moo_det *refdet = moo_detlist_get(detlist, 0);

            int offset = moo_pfits_get_slit_offset(refdet->primary_header);
            offsets[i] = offset;
            cpl_msg_info(__func__, "Do offset %d", offset);
            cpl_msg_info(__func__, "Detect saturated pixels");

            saturate_map_tab[i] =
                moo_compute_saturate_pixels(detlist, loc_tab[i], params);
            char *saturate_name =
                cpl_sprintf("%s_OFFSET%d.fits", MOONS_TAG_LINEARITY_SATURATE,
                            offset);
            moo_products_add_saturate_map(products, saturate_map_tab[i],
                                          CPL_FRAME_LEVEL_INTERMEDIATE,
                                          MOONS_TAG_LINEARITY_SATURATE,
                                          saturate_name, ref_frame);
            cpl_free(saturate_name);
            cube_name = cpl_sprintf("%s_OFFSET%d.fits",
                                    MOONS_TAG_LINEARITY_COEFF_CUBE, offset);
            cube_tab[i] = moo_compute_linearity(detlist, loc_tab[i],
                                                saturate_map_tab[i], cube_name);
            nb_cube++;
            moo_products_add_cube(products, cube_tab[i], CPL_FRAME_LEVEL_FINAL,
                                  MOONS_TAG_LINEARITY_COEFF_CUBE, cube_name,
                                  ref_frame);
            cpl_msg_info(__func__, "Producing frame %s", cube_name);
            cpl_free(cube_name);

            moo_detlist_delete(detlist);
        }

        cpl_frameset_delete(refset);
        cpl_frameset_delete(set);
    }

    if (nb_cube == 2) {
        cpl_msg_info("moons_linear", "Merging coeffs cube");
        moo_cube *mcube = moo_cube_merge(cube_tab[0], saturate_map_tab[0],
                                         cube_tab[1], saturate_map_tab[1]);
        cube_name = cpl_sprintf("%s.fits", MOONS_TAG_LINEARITY_COEFF_CUBE);
        moo_products_add_cube(products, mcube, CPL_FRAME_LEVEL_FINAL,
                              MOONS_TAG_LINEARITY_COEFF_CUBE, cube_name,
                              refframe_tab[0]);
        cpl_msg_info(__func__, "Producing frame %s", cube_name);
        cpl_free(cube_name);
        moo_cube_delete(mcube);
    }

    for (int i = 0; i < 2; i++) {
        if (cube_tab[i] != NULL) {
            int offset = offsets[i];
            cpl_frame *ref_frame = refframe_tab[i];
            moo_cube_normalise(cube_tab[i], saturate_map_tab[i]);
            cube_name =
                cpl_sprintf("%s_OFFSET%d.fits",
                            MOONS_TAG_LINEARITY_NORM_COEFF_CUBE, offset);
            moo_products_add_cube(products, cube_tab[i],
                                  CPL_FRAME_LEVEL_INTERMEDIATE,
                                  MOONS_TAG_LINEARITY_NORM_COEFF_CUBE,
                                  cube_name, ref_frame);
            cpl_msg_info(__func__, "Producing frame %s", cube_name);
            cpl_free(cube_name);

            cpl_msg_info(__func__, "Compute non linearity bad pixel mask");
            bpm_name =
                cpl_sprintf("%s_OFFSET%d.fits", MOONS_TAG_BP_MAP_NL, offset);
            bpm_tab[i] = moo_compute_bpm_linearity(cube_tab[i], loc_tab[i],
                                                   saturate_map_tab[i], params,
                                                   bpm_name);
            moo_products_add_bpm(products, bpm_tab[i],
                                 CPL_FRAME_LEVEL_INTERMEDIATE,
                                 MOONS_TAG_BP_MAP_NL, bpm_name, ref_frame);
            nb_bpm++;
            cpl_msg_info(__func__, "Producing frame %s", bpm_name);
            cpl_free(bpm_name);
            bpm_name = NULL;
        }
    }
    bpm_name = cpl_sprintf("%s.fits", MOONS_TAG_BP_MAP_NL);

    if (nb_bpm == 2) {
        cpl_msg_info("moons_linear", "Merging bad pixel mask");
        moo_bpm_merge(bpm_tab[0], bpm_tab[1]);
        moo_products_add_bpm(products, bpm_tab[0], CPL_FRAME_LEVEL_FINAL,
                             MOONS_TAG_BP_MAP_NL, bpm_name, refframe_tab[0]);
        cpl_msg_info(__func__, "Producing frame %s", bpm_name);
    }
    else {
        for (int i = 0; i < 2; i++) {
            if (bpm_tab[i] != NULL) {
                moo_products_add_bpm(products, bpm_tab[i],
                                     CPL_FRAME_LEVEL_FINAL, MOONS_TAG_BP_MAP_NL,
                                     bpm_name, refframe_tab[i]);
                cpl_msg_info(__func__, "Producing frame %s", bpm_name);
            }
        }
    }

    for (int i = 0; i < 2; i++) {
        if (saturate_map_tab[i] != NULL) {
            moo_saturate_map_delete(saturate_map_tab[i]);
        }

        if (loc_tab[i] != NULL) {
            moo_loc_delete(loc_tab[i]);
        }

        if (cube_tab[i] != NULL) {
            moo_cube_delete(cube_tab[i]);
        }

        if (bpm_tab[i] != NULL) {
            moo_bpm_delete(bpm_tab[i]);
        }

        if (refframe_tab[i] != NULL) {
            cpl_frame_delete(refframe_tab[i]);
        }
    }

moo_try_cleanup:
    cpl_msg_info(__func__, "cleanup");
    cpl_msg_indent_less();
    cpl_free(bpm_name);
    cpl_free(loc_tab);
    cpl_free(bpm_tab);
    cpl_free(saturate_map_tab);
    cpl_free(cube_tab);
    cpl_free(refframe_tab);
    cpl_free(selection);
    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
  @brief Combine LINEARITY files for each EXPTIME
  @param inputset the input LIBNEARITY  frameset
  @param tag the inputset tag
  @param products the products object
  @return the combine LINEARITY set of files or NULL
 *
 * Possible #_cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if a required input pointer is NULL
  - CPL_ERROR_DATA_NOT_FOUND in case of missing files
 */
/*----------------------------------------------------------------------------*/

static cpl_frameset *
_moons_combine_exptime(cpl_frameset *inputset,
                       const char *tag,
                       const cpl_frameset *ref_frameset,
                       moo_products *products,
                       cpl_frameset **ref_comb_frameset)
{
    cpl_errorstate prestate = cpl_errorstate_get();

    cpl_ensure(inputset != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(products != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(tag != NULL, CPL_ERROR_NULL_INPUT, NULL);

    cpl_msg_info(__func__, "Combine frames by offset with same exptime");
    cpl_msg_indent_more();
    cpl_frameset *result = cpl_frameset_new();

    cpl_size nsel = 0;
    cpl_size *selection = NULL;
    cpl_size *selection2 = NULL;
    cpl_frameset *raw_expset = NULL;
    cpl_frameset *expset = NULL;
    cpl_frameset *raw_timeset = NULL;
    cpl_frameset *timeset = NULL;
    moo_crh_params *crh_params = NULL;
    moo_det *median = NULL;
    moo_detlist *detlist = NULL;
    char *resname = NULL;
    char *restag = NULL;
    moo_try_check(crh_params = moo_crh_params_new(), " ");
    crh_params->method = MOO_CRH_METHOD_MEDIAN;

    moo_try_check(selection =
                      cpl_frameset_labelise(inputset, &_moons_compare_offset,
                                            &nsel),
                  " ");
    int i, j;
    *ref_comb_frameset = cpl_frameset_new();
    for (i = 0; i < nsel; i++) {
        expset = cpl_frameset_extract(inputset, selection, i);
        cpl_frameset_sort(expset, _moons_sort_exptime);
        raw_expset = cpl_frameset_extract(ref_frameset, selection, i);
        cpl_frameset_sort(raw_expset, _moons_sort_exptime);
        cpl_size nsel2 = 0;
        moo_try_check(selection2 =
                          cpl_frameset_labelise(expset, &_moons_compare_exptime,
                                                &nsel2),
                      " ");
        moo_try_assure(
            nsel2 >= 3, CPL_ERROR_ILLEGAL_INPUT,
            "Invalid number of different exptime by offset (%lld) expected >=3",
            nsel2);

        for (j = 0; j < nsel2; j++) {
            timeset = cpl_frameset_extract(expset, selection2, j);
            raw_timeset = cpl_frameset_extract(raw_expset, selection2, j);

            int size = cpl_frameset_get_size(timeset);
            moo_try_assure(size >= 1, CPL_ERROR_ILLEGAL_INPUT,
                           "Invalid number of frames for a given exptime (%d) "
                           "expected >=1",
                           size);
            const cpl_frame *ref_frame =
                cpl_frameset_get_position_const(raw_timeset, 0);
            moo_try_check(detlist = moo_detlist_create(timeset), " ");
            moo_try_check(median = moo_remove_CRH(detlist, NULL, crh_params),
                          " ");
            moo_detlist_delete(detlist);
            detlist = NULL;

            restag = cpl_sprintf("%s_EXPTIME_COMB", tag);
            resname = cpl_sprintf("%s_%d_%d.fits", restag, i, j);

            cpl_frame *resframe =
                moo_products_add(products, median, CPL_FRAME_LEVEL_INTERMEDIATE,
                                 restag, resname, ref_frame);
            cpl_frameset_insert(*ref_comb_frameset,
                                cpl_frame_duplicate(ref_frame));
            cpl_frameset_insert(result, cpl_frame_duplicate(resframe));
            cpl_msg_info(__func__, "Producing frame %s", resname);
            moo_det_delete(median);
            median = NULL;
            cpl_free(resname);
            resname = NULL;
            cpl_free(restag);
            restag = NULL;
            cpl_frameset_delete(raw_timeset);
            raw_timeset = NULL;
            cpl_frameset_delete(timeset);
            timeset = NULL;
        }
        cpl_free(selection2);
        selection2 = NULL;
        cpl_frameset_delete(expset);
        expset = NULL;
        cpl_frameset_delete(raw_expset);
        raw_expset = NULL;
    }
moo_try_cleanup:
    cpl_msg_indent_less();
    if (!cpl_errorstate_is_equal(prestate)) {
        cpl_frameset_delete(result);
        cpl_frameset_delete(*ref_comb_frameset);
        cpl_frameset_delete(expset);
        cpl_frameset_delete(raw_expset);
        cpl_frameset_delete(timeset);
        cpl_frameset_delete(raw_timeset);
        cpl_free(selection2);
        moo_det_delete(median);
        moo_detlist_delete(detlist);
        cpl_free(restag);
        cpl_free(resname);
        *ref_comb_frameset = NULL;
        result = NULL;
    }
    moo_crh_params_delete(crh_params);
    cpl_free(selection);
    return result;
}

/*----------------------------------------------------------------------------*/
/**
  @brief  Extract from SOF the input data
  @param frameset the input SOF frameset
  @param rawlin_frames [output] the LINEARITY set of files or NULL
  @param rawlinoff_frames [output] the LINEARITY_OFF set of files or NULL  
  @return relevant error code
 *
 * Possible #_cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if a required input pointer is NULL
  - CPL_ERROR_DATA_NOT_FOUND in case of missing files
 */
/*----------------------------------------------------------------------------*/
static cpl_error_code
_moons_linear_check_sof(cpl_frameset *frameset,
                        cpl_frameset **rawlin_frames,
                        const char **bpmap_rp_name,
                        const char **bpmap_nl_name,
                        cpl_frameset **trace_frames)
{
    cpl_error_code status = CPL_ERROR_NONE;
    cpl_errorstate prestate = cpl_errorstate_get();
    cpl_size *selection = NULL;
    cpl_size nsel;

    cpl_ensure_code(frameset != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(rawlin_frames != NULL, CPL_ERROR_NULL_INPUT);

    /* SOF file */
    cpl_ensure_code(moo_dfs_set_groups(frameset) == CPL_ERROR_NONE,
                    cpl_error_get_code());


    int nrawlin = 0;
    int ntrace = 0;

    *rawlin_frames = cpl_frameset_new();
    *trace_frames = cpl_frameset_new();

    for (int i = 0; i < cpl_frameset_get_size(frameset); ++i) {
        const cpl_frame *current_frame =
            cpl_frameset_get_position_const(frameset, i);
        if (!strcmp(cpl_frame_get_tag(current_frame), MOONS_TAG_LINEARITY)) {
            cpl_frame *new_frame = cpl_frame_duplicate(current_frame);
            cpl_frameset_insert(*rawlin_frames, new_frame);
            ++nrawlin;
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_BP_MAP_RP)) {
            *bpmap_rp_name = cpl_frame_get_filename(current_frame);
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_BP_MAP_NL)) {
            *bpmap_nl_name = cpl_frame_get_filename(current_frame);
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_FF_TRACE_GUESS)) {
            cpl_frame *new_frame = cpl_frame_duplicate(current_frame);
            cpl_frameset_insert(*trace_frames, new_frame);
            ntrace++;
        }
    }

    moo_try_assure(nrawlin >= 3, CPL_ERROR_DATA_NOT_FOUND,
                   "SOF does not have enough files (%d >= %d) tagged with %s",
                   nrawlin, 3, MOONS_TAG_LINEARITY);
    moo_try_check(selection =
                      cpl_frameset_labelise(*rawlin_frames,
                                            &_moons_compare_offset, &nsel),
                  " ");
/*moo_try_assure(ntrace>=nsel,CPL_ERROR_DATA_NOT_FOUND,
      "SOF does not have enough files (%d>=%lld) with tag %s", ntrace,nsel,
      MOONS_TAG_FF_TRACE_GUESS);
    */
moo_try_cleanup:
    cpl_free(selection);

    if (!cpl_errorstate_is_equal(prestate)) {
        cpl_frameset_delete(*rawlin_frames);
        cpl_frameset_delete(*trace_frames);
        *rawlin_frames = NULL;
        *trace_frames = NULL;
        status =
            cpl_error_get_code();  // Recover from the error(s) (Reset to prestate))
    }
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief Prepare one RAW DARK file
  @param products the products object
  @param frame the input RAW FLAT frame
  @param bpmap_rp_name the BPMAP_RP file name or NULL
  @param i the index of frames in set
  @return   a new DET_DARK frame
 *
 * Possible #_cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if a required input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
static cpl_frame *
_moons_prepare(moo_products *products,
               const cpl_frame *frame,
               const char *bpmap_rp_name,
               int i,
               int j,
               moo_prepare_params *prepare_params)
{
    cpl_frame *result = NULL;
    cpl_frame *pframe = NULL;
    char *detname1 = NULL;
    moo_det *det = NULL;
    cpl_ensure(frame != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_errorstate prestate = cpl_errorstate_get();

    moo_try_check(det = moo_prepare_adu(frame, bpmap_rp_name, prepare_params),
                  " ");
    moo_try_check(detname1 = cpl_sprintf("%s_%d_%d.fits",
                                         MOONS_TAG_LINEARITY_PREPAREADU, i, j),
                  " ");
    moo_try_check(pframe = moo_products_add(products, det,
                                            CPL_FRAME_LEVEL_INTERMEDIATE,
                                            MOONS_TAG_LINEARITY_PREPAREADU,
                                            detname1, frame),
                  " ");
    moo_try_check(result = cpl_frame_duplicate(pframe), " ");

moo_try_cleanup:
    if (!cpl_errorstate_is_equal(prestate)) {
        cpl_frame_delete(result);
        result = NULL;
    }
    moo_det_delete(det);
    cpl_free(detname1);
    return result;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Prepare a set of RAW DARK files in DET format
  @param    raw_frames the input set of RAW frames
  @param bpmap_rp_name the BPMAP_RP file name or NULL
  @param bpmap_nl_name the BPMAP_NL file name or NULL  
  @param products the products object
  @return   a new set of DET_DARK frames
 *
 * Possible #_cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if a required input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
static cpl_frameset *
_moons_prepare_set(cpl_frameset *raw_frames,
                   const char *bpmap_rp_name,
                   cpl_frameset **rawout_frames,
                   moo_prepare_params *prepare_params,
                   moo_products *products)
{
    cpl_frameset *detframes = NULL;
    cpl_size nsel = 0;
    cpl_size *selection = NULL;
    cpl_frameset *timeset = NULL;

    cpl_ensure(raw_frames, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(products, CPL_ERROR_NULL_INPUT, NULL);

    cpl_errorstate prestate = cpl_errorstate_get();

    moo_try_check(detframes = cpl_frameset_new(), " ");
    moo_try_check(*rawout_frames = cpl_frameset_new(), " ");
    moo_try_check(selection =
                      cpl_frameset_labelise(raw_frames, &_moons_compare_exptime,
                                            &nsel),
                  " ");

    for (int i = 0; i < nsel; i++) {
        timeset = cpl_frameset_extract(raw_frames, selection, i);
        int size = cpl_frameset_get_size(timeset);

        for (int j = 0; j < size; j++) {
            cpl_frame *frame = NULL;
            const cpl_frame *on_frame =
                cpl_frameset_get_position_const(timeset, j);
            moo_try_check(frame =
                              _moons_prepare(products, on_frame, bpmap_rp_name,
                                             i, j, prepare_params),
                          " ");
            moo_try_check(cpl_frameset_insert(detframes, frame), " ");
            moo_try_check(cpl_frameset_insert(*rawout_frames,
                                              cpl_frame_duplicate(on_frame)),
                          " ");
        }
        cpl_frameset_delete(timeset);
        timeset = NULL;
    }
moo_try_cleanup:
    cpl_free(selection);
    if (!cpl_errorstate_is_equal(prestate)) {
        cpl_frameset_delete(timeset);
        cpl_frameset_delete(detframes);
        detframes = NULL;
        *rawout_frames = NULL;
    }
    return detframes;
}

static int
_moons_linear(cpl_frameset *frameset, const cpl_parameterlist *parlist)
{
    moo_prepare_params *prepare_params = NULL;
    moo_linear_params *linear_params = NULL;
    cpl_frameset *rawlin_frames = NULL;
    cpl_frameset *ref_frames = NULL;
    cpl_frameset *ref_comb_frames = NULL;
    cpl_frameset *trace_frames = NULL;
    const char *bpmap_rp_name = NULL;
    const char *bpmap_nl_name = NULL;
    cpl_frameset *detlin_frames = NULL;
    cpl_frameset *exptimecomb_frames = NULL;
    cpl_frameset *exptimenorm_frames = NULL;
    cpl_frameset *snr_frames = NULL;

    moo_products *products = moo_products_new(frameset, parlist, "moons_linear",
                                              PACKAGE "/" PACKAGE_VERSION);
    /* parameters */
    const moo_params *params = moo_products_get_params(products);
    moo_try_check(prepare_params = moo_params_get_prepare(params, parlist),
                  " ");

    moo_try_check(linear_params = moo_params_get_linear(params, parlist), " ");

    moo_try_check(_moons_linear_check_sof(frameset, &rawlin_frames,
                                          &bpmap_rp_name, &bpmap_nl_name,
                                          &trace_frames),
                  " ");

    detlin_frames = _moons_prepare_set(rawlin_frames, bpmap_rp_name,
                                       &ref_frames, prepare_params, products);

    exptimecomb_frames =
        _moons_combine_exptime(detlin_frames, MOONS_TAG_LINEARITY, ref_frames,
                               products, &ref_comb_frames);

    _moons_compute_linearity(exptimecomb_frames, trace_frames, ref_comb_frames,
                             linear_params, products);
moo_try_cleanup:
    cpl_frameset_delete(ref_comb_frames);
    cpl_frameset_delete(ref_frames);
    cpl_frameset_delete(snr_frames);
    cpl_frameset_delete(trace_frames);
    cpl_frameset_delete(exptimecomb_frames);
    cpl_frameset_delete(exptimenorm_frames);
    cpl_frameset_delete(detlin_frames);
    cpl_frameset_delete(rawlin_frames);
    moo_linear_params_delete(linear_params);
    moo_prepare_params_delete(prepare_params);
    moo_products_delete(products);
    return (int)cpl_error_get_code();
}
