/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#include "moo_utils.h"
#include "moo_pfits.h"
#include "moo_dfs.h"
#include "moo_drl.h"
#include "moo_params.h"
#include "moo_products.h"
#include <cpl.h>

#include <time.h>
#include <string.h>

/*-----------------------------------------------------------------------------
                              Plugin registration
 -----------------------------------------------------------------------------*/

int cpl_plugin_get_info(cpl_pluginlist *list);

/*-----------------------------------------------------------------------------
                            Private function prototypes
 -----------------------------------------------------------------------------*/

static int _moons_mbias_create(cpl_plugin *plugin);
static int _moons_mbias_exec(cpl_plugin *plugin);
static int _moons_mbias_destroy(cpl_plugin *plugin);
static int
_moons_mbias(cpl_frameset *frameset, const cpl_parameterlist *parlist);

/*-----------------------------------------------------------------------------
                            Static variables
 -----------------------------------------------------------------------------*/

static const char *const _moons_mbias_description =
    "INPUT FRAMES\n"
    "  * RawList n>=3 files                    (RAW) with tag BIAS : "
    "Bias files\n"
    "  * [OPTIONAL] ReferenceBadPixMask 1 file (QUA) with tag BP_MAP_RP : "
    "cosmetic bad pixel map\n"
    "PRODUCTS\n"
    "  * MASTER_BIAS.fits                      (DET) with tag MASTER_BIAS : "
    "the master bias file\n";

/*-----------------------------------------------------------------------------
                                Function code
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Build the list of available plugins, for this module.
  @param    list    the plugin list
  @return   0 if everything is ok, 1 otherwise
  @note     Only this function is exported

  Create the recipe instance and make it available to the application using the
  interface.
 */
/*----------------------------------------------------------------------------*/

int
cpl_plugin_get_info(cpl_pluginlist *list)
{
    cpl_recipe *recipe = cpl_calloc(1, sizeof *recipe);
    cpl_plugin *plugin = &recipe->interface;

    if (cpl_plugin_init(plugin, CPL_PLUGIN_API, MOONS_BINARY_VERSION,
                        CPL_PLUGIN_TYPE_RECIPE, "moons_mbias",
                        "Create a master bias product",
                        _moons_mbias_description, "Regis Haigron",
                        PACKAGE_BUGREPORT, moo_get_license(),
                        _moons_mbias_create, _moons_mbias_exec,
                        _moons_mbias_destroy)) {
        cpl_msg_error(cpl_func, "Plugin initialization failed");
        (void)cpl_error_set_where(cpl_func);
        return 1;
    }

    if (cpl_pluginlist_append(list, plugin)) {
        cpl_msg_error(cpl_func, "Error adding plugin to list");
        (void)cpl_error_set_where(cpl_func);
        return 1;
    }

    return 0;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Setup the recipe options
  @param    plugin  the plugin
  @return   0 if everything is ok

  Defining the command-line/configuration parameters for the recipe.
 */
/*----------------------------------------------------------------------------*/

static int
_moons_mbias_create(cpl_plugin *plugin)
{
    cpl_recipe *recipe;

    /* Do not create the recipe if an error code is already set */
    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        cpl_msg_error(cpl_func, "%s():%d: An error is already set: %s",
                      cpl_func, __LINE__, cpl_error_get_where());
        return (int)cpl_error_get_code();
    }

    if (plugin == NULL) {
        cpl_msg_error(cpl_func, "Null plugin");
        cpl_ensure_code(0, (int)CPL_ERROR_NULL_INPUT);
    }

    /* Verify plugin type */
    if (cpl_plugin_get_type(plugin) != CPL_PLUGIN_TYPE_RECIPE) {
        cpl_msg_error(cpl_func, "Plugin is not a recipe");
        cpl_ensure_code(0, (int)CPL_ERROR_TYPE_MISMATCH);
    }

    /* Get the recipe */
    recipe = (cpl_recipe *)plugin;

    /* Create the parameters list in the cpl_recipe object */
    recipe->parameters = cpl_parameterlist_new();
    if (recipe->parameters == NULL) {
        cpl_msg_error(cpl_func, "Parameter list allocation failed");
        cpl_ensure_code(0, (int)CPL_ERROR_ILLEGAL_OUTPUT);
    }
    moo_params *params = moo_params_new("moons", "moons_mbias");
    moo_params_add_keep_temp(params, recipe->parameters);
    /* Fill the parameters list */
    moo_params_add_prepare(params, recipe->parameters);
    moo_params_add_crh(params, recipe->parameters, MOO_CRH_METHOD_MEDIAN);
    moo_params_add_bias(params, recipe->parameters);
    moo_params_delete(params);
    return 0;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Execute the plugin instance given by the interface
  @param    plugin  the plugin
  @return   0 if everything is ok
 */
/*----------------------------------------------------------------------------*/

static int
_moons_mbias_exec(cpl_plugin *plugin)
{
    cpl_recipe *recipe;
    int recipe_status;
    cpl_errorstate initial_errorstate = cpl_errorstate_get();

    /* Return immediately if an error code is already set */
    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        cpl_msg_error(cpl_func, "%s():%d: An error is already set: %s",
                      cpl_func, __LINE__, cpl_error_get_where());
        return (int)cpl_error_get_code();
    }

    if (plugin == NULL) {
        cpl_msg_error(cpl_func, "Null plugin");
        cpl_ensure_code(0, (int)CPL_ERROR_NULL_INPUT);
    }

    /* Verify plugin type */
    if (cpl_plugin_get_type(plugin) != CPL_PLUGIN_TYPE_RECIPE) {
        cpl_msg_error(cpl_func, "Plugin is not a recipe");
        cpl_ensure_code(0, (int)CPL_ERROR_TYPE_MISMATCH);
    }

    /* Get the recipe */
    recipe = (cpl_recipe *)plugin;

    /* Verify parameter and frame lists */
    if (recipe->parameters == NULL) {
        cpl_msg_error(cpl_func, "Recipe invoked with NULL parameter list");
        cpl_ensure_code(0, (int)CPL_ERROR_NULL_INPUT);
    }
    if (recipe->frames == NULL) {
        cpl_msg_error(cpl_func, "Recipe invoked with NULL frame set");
        cpl_ensure_code(0, (int)CPL_ERROR_NULL_INPUT);
    }

    /* Invoke the recipe */
    recipe_status = _moons_mbias(recipe->frames, recipe->parameters);

    /* Ensure DFS-compliance of the products */
    if (cpl_dfs_update_product_header(recipe->frames)) {
        if (!recipe_status)
            recipe_status = (int)cpl_error_get_code();
    }

    if (!cpl_errorstate_is_equal(initial_errorstate)) {
        /* Dump the error history since recipe execution start.
           At this point the recipe cannot recover from the error */
        cpl_errorstate_dump(initial_errorstate, CPL_FALSE, NULL);
    }

    return recipe_status;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Destroy what has been created by the 'create' function
  @param    plugin  the plugin
  @return   0 if everything is ok
 */
/*----------------------------------------------------------------------------*/

static int
_moons_mbias_destroy(cpl_plugin *plugin)
{
    cpl_recipe *recipe;

    if (plugin == NULL) {
        cpl_msg_error(cpl_func, "Null plugin");
        cpl_ensure_code(0, (int)CPL_ERROR_NULL_INPUT);
    }

    /* Verify plugin type */
    if (cpl_plugin_get_type(plugin) != CPL_PLUGIN_TYPE_RECIPE) {
        cpl_msg_error(cpl_func, "Plugin is not a recipe");
        cpl_ensure_code(0, (int)CPL_ERROR_TYPE_MISMATCH);
    }

    /* Get the recipe */
    recipe = (cpl_recipe *)plugin;

    cpl_parameterlist_delete(recipe->parameters);

    return 0;
}

/*----------------------------------------------------------------------------*/
/**
  @brief Prepare one RAW file in DET format
  @param products the products object
  @param frame the input RAW frame
  @param bpmap_name the BPMAP_RP file name or NULL
  @param i the index of frames in set
  @param params prepare parameters
  @return   a new DET frame
 *
 * Possible #_cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if a required input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
static cpl_frame *
_moons_prepare(moo_products *products,
               const cpl_frame *frame,
               const char *bpmap_name,
               int i,
               moo_prepare_params *params)
{
    cpl_frame *result = NULL;
    cpl_ensure(frame != NULL, CPL_ERROR_NULL_INPUT, NULL);

    char *detname = NULL;
    moo_det *det = NULL;

    detname = cpl_sprintf("%s_%d.fits", MOONS_TAG_BIAS_PREPARE, i);
    moo_try_check(det =
                      moo_prepare(frame, bpmap_name, NULL, NULL, NULL, params),
                  " ");
    moo_try_check(result =
                      moo_products_add(products, det,
                                       CPL_FRAME_LEVEL_INTERMEDIATE,
                                       MOONS_TAG_BIAS_PREPARE, detname, frame),
                  " ");

moo_try_cleanup:
    moo_det_delete(det);
    cpl_free(detname);
    return result;
}

/*----------------------------------------------------------------------------*/
/**
  @brief Prepare a set of RAW files in DET format
  @param in the input set of RAW frames
  @param bpmap_name the BPMAP_RP file name or NULL
  @param products the products object
  @param params prepare parameters
  @return   a new set of DET frames
 *
 * Possible #_cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if a required input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
static cpl_frameset *
_moons_prepare_set(cpl_frameset *in,
                   const char *bpmap_name,
                   moo_products *products,
                   moo_prepare_params *params)
{
    cpl_ensure(in != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(products != NULL, CPL_ERROR_NULL_INPUT, NULL);

    cpl_frameset *result = cpl_frameset_new();
    int i;

    cpl_errorstate prestate = cpl_errorstate_get();

    for (i = 0; i < cpl_frameset_get_size(in); ++i) {
        const cpl_frame *current_frame = NULL;
        cpl_frame *pframe = NULL;
        moo_try_check(current_frame = cpl_frameset_get_position_const(in, i),
                      " ");
        moo_try_check(pframe = _moons_prepare(products, current_frame,
                                              bpmap_name, i, params),
                      " ");
        moo_try_check(cpl_frameset_insert(result, cpl_frame_duplicate(pframe)),
                      " ");
    }
moo_try_cleanup:
    if (!cpl_errorstate_is_equal(prestate)) {
        cpl_msg_error(__func__, "dump error ");
        cpl_errorstate_dump(prestate, CPL_FALSE, cpl_errorstate_dump_one);
        cpl_frameset_delete(result);
        result = NULL;
    }
    return result;
}

/*----------------------------------------------------------------------------*/
/**
  @brief  Create a MASTER BIAS and add it to recipe products from a DET_BIAS
 * frameset
  @param in the input DET_BIAS frameset
  @param layout_frame LAYOUT frame
  @param products the products object
  @param crh_params the remove crh parameters
  @param bias_params the master bias parameters
  @return relevant error code
 *
 * Possible #_cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if a required input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
static cpl_error_code
_moons_create_masterbias(cpl_frameset *in,
                         moo_products *products,
                         moo_crh_params *crh_params,
                         moo_bias_params *bias_params)
{
    cpl_error_code status = CPL_ERROR_NONE;
    moo_det *cleanBias = NULL;
    moo_detlist *detlist = NULL;

    cpl_errorstate prestate = cpl_errorstate_get();
    cpl_ensure_code(in != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(products != NULL, CPL_ERROR_NULL_INPUT);

    moo_try_check(detlist = moo_detlist_create(in), " ");
    moo_try_check(cleanBias = moo_remove_CRH(detlist, NULL, crh_params), " ");
    moo_try_check(status =
                      moo_masterbias(cleanBias, detlist, bias_params, products),
                  " ");

moo_try_cleanup:
    moo_detlist_delete(detlist);
    moo_det_delete(cleanBias);

    if (!cpl_errorstate_is_equal(prestate)) {
        cpl_msg_error(__func__, "Can't create master bias");
        status = cpl_error_get_code();
        cpl_errorstate_dump(prestate, CPL_FALSE, cpl_errorstate_dump_one);
    }
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief  Extract from SOF the input data
  @param frameset the input SOF frameset
  @param rawframes [output] the BIAS set of files
  @param bpmap_rp_name [output] the BPMAP_RP file name or NULL
  @param layout_frame [output] the LAYOUT_FRAME or NULL in case of error
  @return relevant error code
 *
 * Possible #_cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if a required input pointer is NULL
  - CPL_ERROR_DATA_NOT_FOUND in case of missing files
 */
/*----------------------------------------------------------------------------*/
static cpl_error_code
_moons_mbias_check_sof(cpl_frameset *frameset,
                       cpl_frameset **rawframes,
                       const char **bpmap_rp_name)
{
    cpl_ensure_code(rawframes != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(frameset != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(bpmap_rp_name != NULL, CPL_ERROR_NULL_INPUT);

    /* set group */
    cpl_ensure_code(moo_dfs_set_groups(frameset) == CPL_ERROR_NONE,
                    cpl_error_get_code());


    moo_try_check(*rawframes = cpl_frameset_new(), " ");
    int nraw = 0;
    int i;
    for (i = 0; i < cpl_frameset_get_size(frameset); ++i) {
        const cpl_frame *current_frame = NULL;
        moo_try_check(current_frame =
                          cpl_frameset_get_position_const(frameset, i),
                      " ");

        if (!strcmp(cpl_frame_get_tag(current_frame), MOONS_TAG_BIAS)) {
            cpl_frame *new_frame = cpl_frame_duplicate(current_frame);
            cpl_frameset_insert(*rawframes, new_frame);
            ++nraw;
        }
        if (!strcmp(cpl_frame_get_tag(current_frame), MOONS_TAG_BP_MAP_RP)) {
            *bpmap_rp_name = cpl_frame_get_filename(current_frame);
        }
    }
    moo_try_assure(nraw >= 3, CPL_ERROR_DATA_NOT_FOUND,
                   "SOF does not have enough files (%d<3) tagged with %s", nraw,
                   MOONS_TAG_BIAS);

moo_try_cleanup:
    return cpl_error_get_code();
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Interpret the command line options and execute the data processing
  @param    frameset   the frames list
  @param    parlist    the parameters list
  @return   0 if everything is ok
 */
/*----------------------------------------------------------------------------*/
static int
_moons_mbias(cpl_frameset *frameset, const cpl_parameterlist *parlist)
{
    /* parameters */
    moo_prepare_params *prepare_params = NULL;
    moo_crh_params *crh_params = NULL;
    moo_bias_params *bias_params = NULL;
    /* input files */
    cpl_frameset *rawframes = NULL;
    const char *bpmap_name = NULL;
    /* others */
    cpl_frameset *detset = NULL;

    srand(time(NULL));

    moo_products *products = moo_products_new(frameset, parlist, "moons_mbias",
                                              PACKAGE "/" PACKAGE_VERSION);

    /* parameters */
    const moo_params *params = moo_products_get_params(products);
    moo_try_check(prepare_params = moo_params_get_prepare(params, parlist),
                  " ");
    /* force ignore detectors for YJ_1 YJ_2 H_1 H_2 */
    prepare_params->ignore_detector[1] = 1;
    prepare_params->ignore_detector[2] = 1;
    prepare_params->ignore_detector[4] = 1;
    prepare_params->ignore_detector[5] = 1;

    moo_try_check(crh_params = moo_params_get_crh(params, parlist), " ");
    moo_try_check(bias_params = moo_params_get_bias(params, parlist), " ");
    /* input files */
    moo_try_check(_moons_mbias_check_sof(frameset, &rawframes, &bpmap_name),
                  " ");

    /* main */
    moo_try_check(detset = _moons_prepare_set(rawframes, bpmap_name, products,
                                              prepare_params),
                  " ");
    moo_try_check(_moons_create_masterbias(detset, products, crh_params,
                                           bias_params),
                  " ");

moo_try_cleanup:
    cpl_frameset_delete(detset);
    cpl_frameset_delete(rawframes);
    moo_bias_params_delete(bias_params);
    moo_prepare_params_delete(prepare_params);
    moo_crh_params_delete(crh_params);
    moo_products_delete(products);
    return (int)cpl_error_get_code();
}
