/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include "cpl_error.h"
#include "cpl_frameset.h"
#define ALTERNATIVE_DRIVER

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#include <string.h>

#include <cpl.h>

#include "moo_utils.h"
#include "moo_pfits.h"
#include "moo_dfs.h"
#include "moo_params.h"
#include "moo_drl.h"
#include "moo_products.h"
#include "moo_flat_shift_compute.h"
#include "moo_compute_slitoffset.h"

/*-----------------------------------------------------------------------------
                              Plugin registration
 -----------------------------------------------------------------------------*/

int cpl_plugin_get_info(cpl_pluginlist *list);

/*-----------------------------------------------------------------------------
                            Private function prototypes
 -----------------------------------------------------------------------------*/

static int _moons_mflat_create(cpl_plugin *plugin);
static int _moons_mflat_exec(cpl_plugin *plugin);
static int _moons_mflat_destroy(cpl_plugin *plugin);

static int
_moons_mflat(cpl_frameset *frameset, const cpl_parameterlist *parlist);

/*-----------------------------------------------------------------------------
                            Static variables
 -----------------------------------------------------------------------------*/

static const char *const _moons_mflat_description[2] = {
    "This recipe aims at creating the products necessary to locate and extract "
    "1D"
    " spectra and correct from flat-field signatures. It can be used in three"
    " different flows\n\n"
    "Workflow 1: Localises the spectra from scratch and creates the guess"
    " localisation traces and widths\n"
    "INPUT FRAMES\n"
    "  * RawList n1>=3 files                      (RAW) with tag FLAT : "
    "flat files\n"
    "  * RawOffList n2==n1 files                  (RAW) with tag FLAT_OFF : "
    "flat off files\n"
    "  * [OPTIONAL] ReferenceBadPixMask 1 file    (QUA) with tag BP_MAP_RP : "
    "cosmetic bad pixel map\n"
    "  * MasterBias 1 file                        (DET) with tag MASTER_BIAS : "
    "master bias file\n"
    "  * [OPTIONAL] MasterDarkVis 1 file          (DET) with tag "
    "MASTER_DARK_VIS : "
    "master dark vis file\n"
    "  * [OPTIONAL] MasterDarkNir 1 file          (DET) with tag "
    "MASTER_DARK_NIR : "
    "master dark nir file\n"
    "PRODUCTS\n"
    "  * FF_TRACE_GUESS_OFFSET[offset]_[insmode].fits (LOC) with tag "
    "FF_TRACE_GUESS : "
    "the localisation trace stores the Y centroid values of the spectra traces"
    " and the width of the spectra in Y direction at each centroid position "
    "\n\n"
    "Workflow 2: Localises the spectra and creates the localisation traces"
    " and widths. Creates the master and extracted flat-fields, and "
    "the fibre-to-fibre relative response. Updates the bad-pixel mask.\n"
    "INPUT FRAMES\n"
    "  * RawList n1>=3 files                      (RAW) with tag FLAT : "
    "flat files\n"
    "  * RawOffList n2==n1 files                  (RAW) with tag FLAT_OFF : "
    "flat off files\n"
    "  * [OPTIONAL] ReferenceBadPixMask 1 file    (QUA) with tag BP_MAP_RP : "
    "cosmetic bad pixel map\n"
    "  * [OPTIONAL] NonLinearityBadPixMask 1 file (QUA) with tag BP_MAP_NL : "
    "non linearity bad pixel map coming from linearity recipe\n"
    "  * MasterBias 1 file                        (DET) with tag MASTER_BIAS : "
    "master bias file\n"
    "  * [OPTIONAL] MasterDarkVis 1 file          (DET) with tag "
    "MASTER_DARK_VIS : "
    "master dark vis file\n"
    "  * [OPTIONAL] MasterDarkNir 1 file          (DET) with tag "
    "MASTER_DARK_NIR : "
    "master dark nir file\n"
    "  * [OPTIONAL] LinCoeffsCube          1 file (3D) with tag "
    "LINEARITY_COEFF_CUBE : "
    "linearity coefficients coming from linearity recipe\n"
    "  * [OPTIONAL] P2pMap                 1 file (DET) with tag P2P_MAP : "
    "the pixel to pixel variation map \n"
    "  * FfTraceGuess                      1 file (LOC) with tag "
    "FF_TRACE_GUESS : "
    "localisation trace produced in workflow 1\n"
    "PRODUCTS\n"
    "  * FF_TRACE_OFFSET[offset]_[insmode].fits      (LOC) with tag FF_TRACE : "
    "the localisation trace stores the Y centroid values of the spectra traces"
    " and the width of the spectra in Y direction at each centroid position\n"
    "  * FF_EXTSPECTRA_OFFSET[offset]_[insmode].fits (EXT) with tag "
    "FF_FF_EXTSPECTRA : "
    "extracted flat-fields\n"
    "  * F2F_OFFSET[offset]_[insmode].fits           (F2F) with tag F2F_TABLE "
    ": "
    "the fibre-to-fibre relative response\n"
    "[if --extract-method==OPTIMAL]\n"
    "  * MASTER_FLAT_OFFSET[offset]_[insmode].fits   (PSF) with tag "
    "MASTER_FLAT : "
    "the master normalized flat product for given position\n"
    "[endif]\n\n",
    "Workflow 3: Localises the spectra and creates the localisation traces"
    " and widths. Creates the master and extracted flat-fields,"
    " the pixel-to-pixel variation map and the fibre-to-fibre relative "
    "response."
    " Updates the bad-pixel mask\n"
    "INPUT FRAMES\n"
    "  * RawList n1>=3 files                      (RAW) with tag FLAT : "
    "flat files at offset 0\n"
    "  * RawList n2>=3 files                      (RAW) with tag FLAT : "
    "flat files at offset 1\n"
    "  * RawOffList n2+n1 files                   (RAW) with tag FLAT_OFF : "
    "flat off files\n"
    "  * [OPTIONAL] ReferenceBadPixMask 1 file    (QUA) with tag BP_MAP_RP : "
    "cosmetic bad pixel map\n"
    "  * [OPTIONAL] NonLinearityBadPixMask 1 file (QUA) with tag BP_MAP_NL : "
    "non linearity bad pixel map coming from linearity recipe\n"
    "  * MasterBias 1 file                        (DET) with tag MASTER_BIAS : "
    "master bias file\n"
    "  * [OPTIONAL] MasterDarkVis 1 file          (DET) with tag "
    "MASTER_DARK_VIS : "
    "master dark vis file\n"
    "  * [OPTIONAL] MasterDarkNir 1 file          (DET) with tag "
    "MASTER_DARK_NIR : "
    "master dark nir file\n"
    "  * [OPTIONAL] LinCoeffsCube1         1 file (3D) with tag "
    "LINEARITY_COEFF_CUBE : "
    "linearity coefficients coming from linearity recipe at offset 0\n"
    "  * [OPTIONAL] LinCoeffsCube2         1 file (3D) with tag "
    "LINEARITY_COEFF_CUBE : "
    "linearity coefficients coming from linearity recipe at offset 1\n"
    "  * FfTraceGuess1                     1 file (LOC) with tag "
    "FF_TRACE_GUESS : "
    "localisation trace produced in workflow 1 at offset 0\n"
    "  * FfTraceGuess2                     1 file (LOC) with tag "
    "FF_TRACE_GUESS : "
    "localisation trace produced in workflow 1 at offset 1\n"
    "PRODUCTS\n"
    "  * P2P_[insmode].fits      (DET) with tag P2P_MAP : "
    "the pixel to pixel variation map \n\n"
    "Workflow 4: Compute shift in fibre-to-fibre relative response."
    "INPUT FRAMES\n"
    "  * FlatAttached                             (RAW) with tag FLAT_ATTACHED "
    ": "
    "flat file\n"
    "  * RawOff                                   (RAW) with tag FLAT_OFF : "
    "flat off files\n"
    "  * [OPTIONAL] ReferenceBadPixMask 1 file    (QUA) with tag BP_MAP_RP : "
    "cosmetic bad pixel map\n"
    "  * [OPTIONAL] NonLinearityBadPixMask 1 file (QUA) with tag BP_MAP_NL : "
    "non linearity bad pixel map coming from linearity recipe\n"
    "  * MasterBias 1 file                        (DET) with tag MASTER_BIAS : "
    "master bias file\n"
    "  * [OPTIONAL] MasterDarkVis 1 file          (DET) with tag "
    "MASTER_DARK_VIS : "
    "master dark vis file\n"
    "  * [OPTIONAL] MasterDarkNir 1 file          (DET) with tag "
    "MASTER_DARK_NIR : "
    "master dark nir file\n"
    "  * [OPTIONAL] LinCoeffsCube          1 file (3D) with tag "
    "LINEARITY_COEFF_CUBE : "
    "linearity coefficients coming from linearity recipe\n"
    "  * FfTrace                           1 file (LOC): "
    "localisation trace produced in workflow 2\n"
    "PRODUCTS\n"
    "  * F2F_OFFSET[offset]_[insmode].fits           (F2F) with tag F2F_TABLE "
    ": "
    "the fibre-to-fibre relative response\n"};

/*-----------------------------------------------------------------------------
                                Function code
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Build the list of available plugins, for this module.
  @param    list    the plugin list
  @return   0 if everything is ok, 1 otherwise
  @note     Only this function is exported

  Create the recipe instance and make it available to the application using the
  interface.
 */
/*----------------------------------------------------------------------------*/

int
cpl_plugin_get_info(cpl_pluginlist *list)
{
    cpl_recipe *recipe = cpl_calloc(1, sizeof *recipe);
    cpl_plugin *plugin = &recipe->interface;

    cpl_size sz[2] = {strlen(_moons_mflat_description[0]),
                      strlen(_moons_mflat_description[1])};
    char *description = cpl_calloc(sizeof(char), sz[0] + sz[1] + 1);

    cpl_size idx;
    cpl_size offset = 0;
    for (idx = 0; idx < 2; ++idx) {
        strncpy(&description[offset], _moons_mflat_description[idx], sz[idx]);
        offset += sz[idx];
    }

#ifdef ALTERNATIVE_DRIVER
    if (cpl_plugin_init(plugin, CPL_PLUGIN_API, MOONS_BINARY_VERSION,
                        CPL_PLUGIN_TYPE_RECIPE, "moons_mflat2",
                        "Create a master flat product", description,
                        "Regis Haigron", PACKAGE_BUGREPORT, moo_get_license(),
                        _moons_mflat_create, _moons_mflat_exec,
                        _moons_mflat_destroy)) {
        cpl_free(description);
        cpl_msg_error(cpl_func, "Plugin initialization failed");
        (void)cpl_error_set_where(cpl_func);
        return 1;
    }
#else
    if (cpl_plugin_init(plugin, CPL_PLUGIN_API, MOONS_BINARY_VERSION,
                        CPL_PLUGIN_TYPE_RECIPE, "moons_mflat",
                        "Create a master flat product", description,
                        "Regis Haigron", PACKAGE_BUGREPORT, moo_get_license(),
                        _moons_mflat_create, _moons_mflat_exec,
                        _moons_mflat_destroy)) {
        cpl_free(description);
        cpl_msg_error(cpl_func, "Plugin initialization failed");
        (void)cpl_error_set_where(cpl_func);
        return 1;
    }
#endif
    cpl_free(description);

    if (cpl_pluginlist_append(list, plugin)) {
        cpl_msg_error(cpl_func, "Error adding plugin to list");
        (void)cpl_error_set_where(cpl_func);
        return 1;
    }

    return 0;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Setup the recipe options
  @param    plugin  the plugin
  @return   0 if everything is ok

  Defining the command-line/configuration parameters for the recipe.
 */
/*----------------------------------------------------------------------------*/

static int
_moons_mflat_create(cpl_plugin *plugin)
{
    cpl_recipe *recipe;

    /* Do not create the recipe if an error code is already set */
    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        cpl_msg_error(cpl_func, "%s():%d: An error is already set: %s",
                      cpl_func, __LINE__, cpl_error_get_where());
        return (int)cpl_error_get_code();
    }

    if (plugin == NULL) {
        cpl_msg_error(cpl_func, "Null plugin");
        cpl_ensure_code(0, (int)CPL_ERROR_NULL_INPUT);
    }

    /* Verify plugin type */
    if (cpl_plugin_get_type(plugin) != CPL_PLUGIN_TYPE_RECIPE) {
        cpl_msg_error(cpl_func, "Plugin is not a recipe");
        cpl_ensure_code(0, (int)CPL_ERROR_TYPE_MISMATCH);
    }

    /* Get the recipe */
    recipe = (cpl_recipe *)plugin;

    /* Create the parameters list in the cpl_recipe object */
    recipe->parameters = cpl_parameterlist_new();
    if (recipe->parameters == NULL) {
        cpl_msg_error(cpl_func, "Parameter list allocation failed");
        cpl_ensure_code(0, (int)CPL_ERROR_ILLEGAL_OUTPUT);
    }

    moo_params *params = moo_params_new("moons", "moons_mflat");

    /* Fill the parameters list */
    moo_params_add_keep_temp(params, recipe->parameters);
    moo_params_add_prepare(params, recipe->parameters);
    moo_params_add_correct_bias(params, recipe->parameters,
                                MOO_CORRECT_BIAS_METHOD_MASTER);
    moo_params_add_crh(params, recipe->parameters, MOO_CRH_METHOD_SIGCLIP);
    moo_params_add_localise(params, recipe->parameters);
    moo_params_add_compute_slitoffset(params, recipe->parameters);
    moo_params_add_extract(params, recipe->parameters);
    moo_params_add_model_flat(params, recipe->parameters);
    moo_params_add_compute_fibtrans(params, recipe->parameters);
    moo_params_delete(params);

    return 0;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Execute the plugin instance given by the interface
  @param    plugin  the plugin
  @return   0 if everything is ok
 */
/*----------------------------------------------------------------------------*/

static int
_moons_mflat_exec(cpl_plugin *plugin)
{
    cpl_recipe *recipe;
    int recipe_status;
    cpl_errorstate initial_errorstate = cpl_errorstate_get();

    /* Return immediately if an error code is already set */
    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        cpl_msg_error(cpl_func, "%s():%d: An error is already set: %s",
                      cpl_func, __LINE__, cpl_error_get_where());
        return (int)cpl_error_get_code();
    }

    if (plugin == NULL) {
        cpl_msg_error(cpl_func, "Null plugin");
        cpl_ensure_code(0, (int)CPL_ERROR_NULL_INPUT);
    }

    /* Verify plugin type */
    if (cpl_plugin_get_type(plugin) != CPL_PLUGIN_TYPE_RECIPE) {
        cpl_msg_error(cpl_func, "Plugin is not a recipe");
        cpl_ensure_code(0, (int)CPL_ERROR_TYPE_MISMATCH);
    }

    /* Get the recipe */
    recipe = (cpl_recipe *)plugin;

    /* Verify parameter and frame lists */
    if (recipe->parameters == NULL) {
        cpl_msg_error(cpl_func, "Recipe invoked with NULL parameter list");
        cpl_ensure_code(0, (int)CPL_ERROR_NULL_INPUT);
    }
    if (recipe->frames == NULL) {
        cpl_msg_error(cpl_func, "Recipe invoked with NULL frame set");
        cpl_ensure_code(0, (int)CPL_ERROR_NULL_INPUT);
    }

    /* Invoke the recipe */
    recipe_status = _moons_mflat(recipe->frames, recipe->parameters);

    /* Ensure DFS-compliance of the products */
    if (cpl_dfs_update_product_header(recipe->frames)) {
        if (!recipe_status)
            recipe_status = (int)cpl_error_get_code();
    }

    if (!cpl_errorstate_is_equal(initial_errorstate)) {
        /* Dump the error history since recipe execution start.
           At this point the recipe cannot recover from the error */
        cpl_errorstate_dump(initial_errorstate, CPL_FALSE, NULL);
    }

    return recipe_status;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Destroy what has been created by the 'create' function
  @param    plugin  the plugin
  @return   0 if everything is ok
 */
/*----------------------------------------------------------------------------*/

static int
_moons_mflat_destroy(cpl_plugin *plugin)
{
    cpl_recipe *recipe;

    if (plugin == NULL) {
        cpl_msg_error(cpl_func, "Null plugin");
        cpl_ensure_code(0, (int)CPL_ERROR_NULL_INPUT);
    }

    /* Verify plugin type */
    if (cpl_plugin_get_type(plugin) != CPL_PLUGIN_TYPE_RECIPE) {
        cpl_msg_error(cpl_func, "Plugin is not a recipe");
        cpl_ensure_code(0, (int)CPL_ERROR_TYPE_MISMATCH);
    }

    /* Get the recipe */
    recipe = (cpl_recipe *)plugin;

    cpl_parameterlist_delete(recipe->parameters);

    return 0;
}

/*----------------------------------------------------------------------------*/
/**
  @brief Prepare one RAW FLAT file and corresponding FLAT_OFF and
 * after correct it from the BIAS and DARK
  @param products the products object
  @param frame the input RAW FLAT frame
  @param frame_off the input RAW FLAT_OFF frame
  @param bpmap_rp_name the BPMAP_RP file name or NULL
  @param bpmap_nl_name the BPMAP_NL file name or NULL
  @param masterbias the MASTER_BIAS frame or NULL
  @param masterdark_vis the MASTER_DARK_VIS frame or NULL
  @param masterdark_nir the MASTER_DARK_NIR frame or NULL
  @param i the index of frames in set
  @return   a new DET_FLAT_DARKCORR frame
 *
 * Possible #_cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if a required input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
static cpl_frame *
_moons_prepare(moo_products *products,
               const cpl_frame *frame,
               const cpl_frame *frame_off,
               const char *bpmap_rp_name,
               const char *bpmap_nl_name,
               const cpl_frame *masterbias,
               const cpl_frame *masterdark_vis,
               const cpl_frame *masterdark_nir,
               const cpl_frame *coeffs_cube,
               moo_prepare_params *prepare_params,
               moo_correct_bias_params *params,
               int i,
               int offset,
               moo_mode_type mode)
{
    cpl_frame *result = NULL;
    cpl_frame *pframe = NULL;
    char *detname1 = NULL;
    char *detname2 = NULL;
    char *detname3 = NULL;
    moo_det *det = NULL;
    moo_det *detoff = NULL;

    cpl_ensure(frame != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(frame_off != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_errorstate prestate = cpl_errorstate_get();

    moo_try_check(det = moo_prepare(frame, bpmap_rp_name, bpmap_nl_name,
                                    masterbias, coeffs_cube, prepare_params),
                  " ");

    moo_try_check(moo_correct_bias(det, masterbias, params), " ");

    moo_try_check(detname1 = cpl_sprintf("%s_OFFSET%d_%s_%d.fits",
                                         MOONS_TAG_FLAT_CORRECTBIAS, offset,
                                         moo_mode_get_name(mode), i),
                  " ");
    moo_try_check(moo_products_add(products, det, CPL_FRAME_LEVEL_INTERMEDIATE,
                                   MOONS_TAG_FLAT_CORRECTBIAS, detname1, frame),
                  " ");

    moo_try_check(detname2 = cpl_sprintf("%s_OFFSET%d_%s_%d.fits",
                                         MOONS_TAG_FLATOFF_PREPARE, offset,
                                         moo_mode_get_name(mode), i),
                  " ");

    moo_try_check(detoff = moo_prepare(frame_off, bpmap_rp_name, bpmap_nl_name,
                                       masterbias, coeffs_cube, prepare_params),
                  " ");

    moo_try_check(pframe = moo_products_add(products, detoff,
                                            CPL_FRAME_LEVEL_INTERMEDIATE,
                                            MOONS_TAG_FLATOFF_PREPARE, detname2,
                                            frame_off),
                  " ");

    moo_try_check(moo_correct_dark(det, detoff, masterdark_vis, masterdark_nir),
                  " ");
    moo_try_check(detname3 = cpl_sprintf("%s_OFFSET%d_%s_%d.fits",
                                         MOONS_TAG_FLAT_CORRECTDARK, offset,
                                         moo_mode_get_name(mode), i),
                  " ");
    moo_try_check(pframe = moo_products_add(products, det,
                                            CPL_FRAME_LEVEL_INTERMEDIATE,
                                            MOONS_TAG_FLAT_CORRECTDARK,
                                            detname3, frame),
                  " ");

    moo_try_check(result = cpl_frame_duplicate(pframe), " ");

moo_try_cleanup:
    if (!cpl_errorstate_is_equal(prestate)) {
        cpl_frame_delete(result);
        result = NULL;
    }
    moo_det_delete(det);
    moo_det_delete(detoff);
    cpl_free(detname1);
    cpl_free(detname2);
    cpl_free(detname3);
    return result;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Prepare set of RAW FLAT and RAW FLAT_OFF files in DET format
  @param raw_frames the input set of RAW FLAT frames
  @param rawoff_frames the input set of RAW FLAT_OFF frames
  @param bpmap_rp_name the BPMAP_RP file name or NULL
  @param bpmap_nl_name the BPMAP_NL file name or NULL
  @param masterbias the MASTER_BIAS frame or NULL
  @param masterdark_vis the MASTER_DARK_VIS frame or NULL
  @param masterdark_nir the MASTER_DARK_NIR frame or NULL
  @param products the products object
  @return   a new set of DET_FLAT, DET_FLAT_OFF frames
 *
 * Possible #_cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if a required input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
static cpl_frameset *
_moons_prepare_set(cpl_frameset *raw_frames,
                   cpl_frameset *rawoff_frames,
                   const char *bpmap_rp_name,
                   const char *bpmap_nl_name,
                   const cpl_frame *masterbias,
                   const cpl_frame *masterdark_vis,
                   const cpl_frame *masterdark_nir,
                   const cpl_frame *coeffs_cube,
                   moo_prepare_params *prepare_params,
                   moo_correct_bias_params *params,
                   moo_products *products,
                   int offset,
                   moo_mode_type mode)
{
    cpl_frameset *detframes = NULL;
    moo_det *det = NULL;
    moo_det *detoff = NULL;

    cpl_ensure(raw_frames, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(rawoff_frames, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(products, CPL_ERROR_NULL_INPUT, NULL);

    cpl_errorstate prestate = cpl_errorstate_get();

    moo_try_check(detframes = cpl_frameset_new(), " ");

    int i;

    for (i = 0; i < cpl_frameset_get_size(raw_frames); ++i) {
        const cpl_frame *current_frame = NULL;
        const cpl_frame *raw_off = NULL;
        cpl_frame *pframe = NULL;

        moo_try_check(current_frame =
                          cpl_frameset_get_position_const(raw_frames, i),
                      " ");
        moo_try_check(raw_off =
                          cpl_frameset_get_position_const(rawoff_frames, i),
                      " ");

        moo_try_check(pframe = _moons_prepare(products, current_frame, raw_off,
                                              bpmap_rp_name, bpmap_nl_name,
                                              masterbias, masterdark_vis,
                                              masterdark_nir, coeffs_cube,
                                              prepare_params, params, i, mode,
                                              offset),
                      " ");
        moo_try_check(cpl_frameset_insert(detframes, pframe), " ");
    }

moo_try_cleanup:
    if (!cpl_errorstate_is_equal(prestate)) {
        moo_det_delete(det);
        moo_det_delete(detoff);
        cpl_frameset_delete(detframes);
        detframes = NULL;
    }
    return detframes;
}

/*----------------------------------------------------------------------------*/
/**
  @brief check that RAW FLAT frame and FF_TRACE match
  @param raw the input FLAT frame
  @param ff_trace the FF_TRACE frame
  @return   a new set of DET_FLAT, DET_FLAT_OFF frames
 *
 * Possible #_cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if a required input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
static cpl_error_code
_moons_flat_check_header(const cpl_frame *raw, const cpl_frame *ff_trace)
{
    cpl_ensure_code(raw != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(ff_trace != NULL, CPL_ERROR_NULL_INPUT);
    cpl_error_code status = CPL_ERROR_NONE;
    const char *filename = NULL;
    const char *trace_filename = NULL;
    cpl_propertylist *raw_pheader = NULL;
    cpl_propertylist *trace_pheader = NULL;
    int raw_offset, trace_offset;

    cpl_errorstate prestate = cpl_errorstate_get();
    moo_try_check(filename = cpl_frame_get_filename(raw), " ");
    moo_try_check(trace_filename = cpl_frame_get_filename(ff_trace), " ");

    moo_try_check(raw_pheader = cpl_propertylist_load(filename, 0), " ");
    moo_try_check(trace_pheader = cpl_propertylist_load(trace_filename, 0),
                  " ");

    moo_try_check(raw_offset = moo_pfits_get_slit_offset(raw_pheader), " ");
    moo_try_check(trace_offset = moo_pfits_get_slit_offset(trace_pheader), " ");

    if (raw_offset != trace_offset) {
        cpl_msg_error("check_header",
                      "OFFSET keyword mismatch between FLAT file %s value %d "
                      "and FF_TRACE file %s value %d",
                      filename, raw_offset, trace_filename, trace_offset);
        status = CPL_ERROR_ILLEGAL_INPUT;
        cpl_error_set("check_header", status);
    }

moo_try_cleanup:
    if (!cpl_errorstate_is_equal(prestate)) {
        status = cpl_error_get_code();
    }
    cpl_propertylist_delete(raw_pheader);
    cpl_propertylist_delete(trace_pheader);
    return status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief operate the localization and the create model workflow
  @param raw_frames the input FLAT frame
  @param rawoff_frames the input FLAT_OFF frame
  @param bpmap_rp_name the BPMAP_RP file name or NULL
  @param bpmap_nl_name the BPMAP_NL file name or NULL
  @param masterbias the MASTER_BIAS frame or NULL
  @param masterdark_vis the MASTER_DARK_VIS frame or NULL
  @param masterdark_nir the MASTER_DARK_NIR frame or NULL
  @param ff_trace_guess the FF_TRACE_GUESS frame or NULL

  @return   a new set of DET_FLAT, DET_FLAT_OFF frames
 *
 * Possible #_cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if a required input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
static moo_loc *
_moons_mflat_localise(cpl_frameset *raw_frames,
                      cpl_frameset *rawoff_frames,
                      const char *bpmap_rp_name,
                      const char *bpmap_nl_name,
                      const cpl_frame *masterbias,
                      const cpl_frame *masterdark_vis,
                      const cpl_frame *masterdark_nir,
                      const cpl_frame *coeffs_cube,
                      const cpl_frame *p2pmap,
                      const cpl_frame *ff_trace_guess,
                      moo_prepare_params *prepare_params,
                      moo_crh_params *crh_params,
                      moo_compute_slitoffset_params *compute_slitoffset_params,
                      moo_localise_params *localise_params,
                      moo_model_flat_params *modelflat_params,
                      moo_correct_bias_params *cbias_params,
                      int domodel,
                      const char *loctag,
                      moo_products *products,
                      moo_det **medflat,
                      moo_psf **psf,
                      const cpl_frame **ref_frame,
                      double *offset,
                      moo_mode_type *mode,
                      cpl_frame_level level)
{
    moo_loc *loc = NULL;
    cpl_frameset *detframes = NULL;
    moo_detlist *detList = NULL;
    char *locname = NULL;
    char *flatname = NULL;
    char *flatp2p_name = NULL;
    moo_det *p2p = NULL;

    cpl_ensure(loctag != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(raw_frames != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(rawoff_frames != NULL, CPL_ERROR_NULL_INPUT, NULL);

    cpl_errorstate prestate = cpl_errorstate_get();

    int nraw, nraw_off;

    moo_try_check(nraw = cpl_frameset_get_size(raw_frames), " ");
    moo_try_check(nraw_off = cpl_frameset_get_size(rawoff_frames), " ");

    if (nraw == 0) {
        cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                              "SOF does not have any file tagged "
                              "with %s",
                              MOONS_TAG_FLAT);
        return NULL;
    }

    if (nraw < 3) {
        cpl_msg_warning(cpl_func,
                        "SOF does not have enough files (%d<3) tagged "
                        "with %s",
                        nraw, MOONS_TAG_FLAT);
    }

    if (nraw_off != nraw) {
        cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                              "SOF does not have %d files tagged "
                              "with %s",
                              nraw, MOONS_TAG_FLAT_OFF);
        return NULL;
    }

    if (masterbias == NULL) {
        cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                              "SOF does not have any file tagged "
                              "with %s",
                              MOONS_TAG_MASTER_BIAS);
        return NULL;
    }

    if (ff_trace_guess != NULL) {
        moo_try_check(_moons_flat_check_header(
                          cpl_frameset_get_position_const(raw_frames, 0),
                          ff_trace_guess),
                      " Mismatch header");
    }

    moo_try_check(*ref_frame = cpl_frameset_get_position_const(raw_frames, 0),
                  " ");
    moo_try_check(*mode = moo_mode_get(*ref_frame), " ");
    moo_try_check(*offset = moo_offset_get(*ref_frame), " ");


    moo_try_check(detframes = _moons_prepare_set(raw_frames, rawoff_frames,
                                                 bpmap_rp_name, bpmap_nl_name,
                                                 masterbias, masterdark_vis,
                                                 masterdark_nir, coeffs_cube,
                                                 prepare_params, cbias_params,
                                                 products, *mode, *offset),
                  " ");

    moo_try_check(detList = moo_detlist_create(detframes), " ");
    moo_try_check(*medflat = moo_remove_CRH(detList, NULL, crh_params), " ");
    moo_try_check(flatname = cpl_sprintf("%s_OFFSET%d_%s.fits",
                                         MOONS_TAG_FLAT_REMOVECRH, (int)*offset,
                                         moo_mode_get_name(*mode)),
                  " ");

    moo_try_check(moo_products_add(products, *medflat,
                                   CPL_FRAME_LEVEL_INTERMEDIATE,
                                   MOONS_TAG_FLAT_REMOVECRH, flatname,
                                   *ref_frame),
                  " ");

    if (p2pmap != NULL) {
        moo_try_check(p2p = moo_det_create(p2pmap), " ");
        moo_try_check(moo_apply_p2p(*medflat, p2p), " ");
        flatp2p_name =
            cpl_sprintf("%s_OFFSET%d_%s.fits", MOONS_TAG_FLAT_APPLYP2P,
                        (int)*offset, moo_mode_get_name(*mode));
        moo_products_add(products, *medflat, CPL_FRAME_LEVEL_INTERMEDIATE,
                         MOONS_TAG_FLAT_APPLYP2P, flatp2p_name, *ref_frame);
        moo_det_delete(p2p);
        p2p = NULL;
    }
    if (ff_trace_guess != NULL) {
        moo_loc *gloc = moo_loc_load(ff_trace_guess);
        float *offsets =
            moo_compute_slitoffset(*medflat, gloc, compute_slitoffset_params);
        for (int i = 0; i < 6; i++) {
            cpl_msg_info("test", "Offsets[%d]=%f", i, offsets[i]);
        }
        moo_loc_delete(gloc);
        cpl_free(offsets);
    }
    moo_try_check(locname = cpl_sprintf("%s_OFFSET%d_%s.fits", loctag,
                                        (int)*offset, moo_mode_get_name(*mode)),
                  " ");
    moo_try_check(loc = moo_localise(*medflat, ff_trace_guess, localise_params,
                                     locname),
                  " ");

    moo_try_check(moo_products_add_loc(products, loc,
                                       localise_params->keep_points, level,
                                       loctag, locname, *ref_frame),
                  " ");

    if (domodel) {
        char *psfname =
            cpl_sprintf("%s_OFFSET%d_%s.fits", MOONS_TAG_MASTER_FLAT,
                        (int)*offset, moo_mode_get_name(*mode));
        *psf = moo_model_flat(*medflat, loc, modelflat_params, psfname);
        moo_products_add_psf(products, *psf, level, MOONS_TAG_MASTER_FLAT,
                             psfname, *ref_frame);
        cpl_free(psfname);
    }

moo_try_cleanup:
    cpl_frameset_delete(detframes);
    moo_detlist_delete(detList);
    cpl_free(flatname);
    cpl_free(flatp2p_name);
    cpl_free(locname);
    if (!cpl_errorstate_is_equal(prestate)) {
        moo_loc_delete(loc);
        loc = NULL;
        moo_det_delete(p2p);
        p2p = NULL;
    }
    return loc;
}

static cpl_error_code
_moons_mflat_fibtrans(moo_det *medflat,
                      moo_loc *ff_trace,
                      moo_psf *psf,
                      moo_extract_params *extract_params,
                      moo_compute_fibtrans_params *fibtrans_params,
                      moo_products *products,
                      const cpl_frame *ref_frame,
                      double offset,
                      moo_mode_type mode)
{
    moo_f2f *f2f = NULL;
    moo_ext *ext = NULL;
    char *flatname = NULL;
    char *extname = NULL;
    char *ffname = NULL;
    char *f2fname = NULL;

    cpl_ensure_code(medflat != NULL, CPL_ERROR_NULL_INPUT);
    cpl_msg_info("moons_mflat", "Work on offset %d mode %s", (int)offset,
                 moo_mode_get_name(mode));

    extname = cpl_sprintf("%s_OFFSET%d_%s.fits", MOONS_TAG_FLAT_EXTSPECTRA,
                          (int)offset, moo_mode_get_name(mode));

    moo_try_check(ext = moo_extract(medflat, ff_trace, psf, extract_params,
                                    extname),
                  " ");
    moo_products_add_ext(products, ext, CPL_FRAME_LEVEL_INTERMEDIATE,
                         MOONS_TAG_FLAT_EXTSPECTRA, extname, ref_frame);


    ffname = cpl_sprintf("%s_OFFSET%d_%s.fits", MOONS_TAG_FF_EXTSPECTRA,
                         (int)offset, moo_mode_get_name(mode));
    f2fname = cpl_sprintf("%s_OFFSET%d_%s.fits", MOONS_TAG_F2F_TABLE,
                          (int)offset, moo_mode_get_name(mode));

    moo_try_check(f2f = moo_compute_fibtrans(ext, fibtrans_params, f2fname),
                  " ");
    moo_products_add_ext(products, ext, CPL_FRAME_LEVEL_FINAL,
                         MOONS_TAG_FF_EXTSPECTRA, ffname, ref_frame);
    moo_products_add_f2f(products, f2f, CPL_FRAME_LEVEL_FINAL,
                         MOONS_TAG_F2F_TABLE, f2fname, ref_frame);

moo_try_cleanup:
    cpl_free(flatname);
    cpl_free(extname);
    cpl_free(f2fname);
    cpl_free(ffname);

    moo_ext_delete(ext);
    moo_f2f_delete(f2f);

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
  @brief Execute the mflat workflow 1
  @param raw_byoffset_set array of FLAT files organize by offset keyword
  @param rawoff_frames the FLAT_OFF set of files
  @param bpmap_rp_name the BPMAP_RP file name or NULL
  @param bpmap_nl_name the BPMAP_NL file name or NULL
  @param master_bias the MASTER_BIAS frame or NULL
  @param master_dark_vis the MASTER_DARK_VIS frame or NULL
  @param master_dark_nir the MASTER_DARK_NIR frame or NULL
  @param crh_params the crh parameters
  @param localise_params the localisation parameters
  @param products the product object
  @return relevant error code
 *
 * Possible #_cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if a required input pointer is NULL
  - CPL_ERROR_ILLEGAL_INPUT if there is more of one OFFSET in the FLAT set
  - - CPL_ERROR_ILLEGAL_INPUT if there empty FLAT set
 */
/*----------------------------------------------------------------------------*/
#ifdef ALTERNATIVE_DRIVER
static int
_moons_mflat_wf1(cpl_frameset *raw_frames,
                 cpl_frameset *rawoff_frames,
                 const char *bpmap_rp_name,
                 const char *bpmap_nl_name,
                 const cpl_frame *masterbias,
                 const cpl_frame *masterdark_vis,
                 const cpl_frame *masterdark_nir,
                 const cpl_frame *coeffs_cube,
                 moo_prepare_params *prepare_params,
                 moo_crh_params *crh_params,
                 moo_localise_params *localise_params,
                 moo_correct_bias_params *cbias_params,
                 moo_products *products)
#else
static int
_moons_mflat_wf1(cpl_frameset **raw_byoffset_set,
                 cpl_frameset *rawoff_frames,
                 const char *bpmap_rp_name,
                 const char *bpmap_nl_name,
                 const cpl_frame *masterbias,
                 const cpl_frame *masterdark_vis,
                 const cpl_frame *masterdark_nir,
                 const cpl_frame *coeffs_cube,
                 moo_prepare_params *prepare_params,
                 moo_crh_params *crh_params,
                 moo_localise_params *localise_params,
                 moo_correct_bias_params *cbias_params,
                 moo_products *products)
#endif
{
#ifdef ALTERNATIVE_DRIVER
    cpl_ensure_code(raw_frames != NULL, CPL_ERROR_NULL_INPUT);
#else
    cpl_ensure_code(raw_byoffset_set != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(raw_byoffset_set[0] != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(raw_byoffset_set[1] != NULL, CPL_ERROR_NULL_INPUT);

    int size1 = 0;
    int size2 = 0;
    cpl_frameset *raw_frames = NULL;
#endif
    const cpl_frame *ref_frame = NULL;
    moo_det *medflat = NULL;
    moo_loc *guess_loc = NULL;
    double offset;
    moo_mode_type mode;

#ifndef ALTERNATIVE_DRIVER
    moo_try_check(size1 = cpl_frameset_get_size(raw_byoffset_set[0]), " ");
    moo_try_check(size2 = cpl_frameset_get_size(raw_byoffset_set[1]), " ");


    if (size1 > 0) {
        if (size2 > 0) {
            return (int)
                cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT,
                                      "SOF have two different offset in input");
        }
        raw_frames = raw_byoffset_set[0];
    }
    else {
        raw_frames = raw_byoffset_set[1];
    }
#endif
    moo_try_assure(cpl_frameset_get_size(raw_frames) > 0,
                   CPL_ERROR_ILLEGAL_INPUT, "No FLAT frames in input");
    moo_try_check(guess_loc = _moons_mflat_localise(
                      raw_frames, rawoff_frames, bpmap_rp_name, bpmap_nl_name,
                      masterbias, masterdark_vis, masterdark_nir, coeffs_cube,
                      NULL, NULL, prepare_params, crh_params, NULL,
                      localise_params, NULL, cbias_params, 0,
                      MOONS_TAG_FF_TRACE_GUESS, products, &medflat, NULL,
                      &ref_frame, &offset, &mode, CPL_FRAME_LEVEL_FINAL),
                  " ");

moo_try_cleanup:
    moo_loc_delete(guess_loc);
    moo_det_delete(medflat);
    return CPL_ERROR_NONE;
}
/*----------------------------------------------------------------------------*/
/**
  @brief Execute the mflat workflow 4
  @param flat_attached_frame FLAT_ATTACHED file
  @param flat_off_frame the FLAT_OFF file or NULL
  @param bpmap_rp_name the BPMAP_RP file name or NULL
  @param bpmap_nl_name the BPMAP_NL file name or NULL
  @param master_bias the MASTER_BIAS frame or NULL
  @param master_dark_vis the MASTER_DARK_VIS frame or NULL
  @param master_dark_nir the MASTER_DARK_NIR frame or NULL
  @param ff_trace the FF_TRACE frame
  @param p2pmap the P2P_MAP frame or NULL
  @param fibtrans_params the compute fibres transmission parameters
  @param products the product object
  @return relevant error code
 *
 * Possible #_cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if a required input pointer is NULL
  - CPL_ERROR_ILLEGAL_INPUT if there is more of one OFFSET in the FLAT set
  - - CPL_ERROR_ILLEGAL_INPUT if there empty FLAT set
 */
/*----------------------------------------------------------------------------*/
static int
_moons_mflat_wf4(const cpl_frame *flat_attached_frame,
                 const cpl_frame *flat_off_frame,
                 const char *bpmap_rp_name,
                 const char *bpmap_nl_name,
                 const cpl_frame *masterbias,
                 const cpl_frame *masterdark_vis,
                 const cpl_frame *masterdark_nir,
                 const cpl_frame *coeffs_cube,
                 const cpl_frame *ff_trace_frame,
                 const cpl_frame *p2pmap,
                 const cpl_frame *master_flat,
                 moo_prepare_params *prepare_params,
                 moo_extract_params *extract_params,
                 moo_compute_fibtrans_params *fibtrans_params,
                 moo_correct_bias_params *cbias_params,
                 moo_products *products)
{
    cpl_ensure_code(flat_attached_frame != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(flat_off_frame != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(ff_trace_frame != NULL, CPL_ERROR_NULL_INPUT);

    moo_loc *ff_trace = NULL;
    int offset;
    moo_mode_type mode;
    char *detname1 = NULL;
    char *detname2 = NULL;
    char *detname3 = NULL;
    moo_det *det = NULL;
    moo_det *detoff = NULL;
    moo_f2f *f2f = NULL;
    moo_ext *ext = NULL;
    char *flatname = NULL;
    char *extname = NULL;
    char *f2fname = NULL;
    moo_det *p2p = NULL;
    moo_psf *psf = NULL;

    cpl_errorstate prestate = cpl_errorstate_get();

    if (masterbias == NULL) {
        cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                              "SOF does not have any file tagged "
                              "with %s",
                              MOONS_TAG_MASTER_BIAS);
        return CPL_ERROR_NONE;
    }

    if (ff_trace_frame != NULL) {
        moo_try_check(_moons_flat_check_header(flat_attached_frame,
                                               ff_trace_frame),
                      " Mismatch header");
    }

    moo_try_check(mode = moo_mode_get(flat_attached_frame), " ");
    moo_try_check(offset = moo_offset_get(flat_attached_frame), " ");
    moo_try_check(det = moo_prepare(flat_attached_frame, bpmap_rp_name,
                                    bpmap_nl_name, masterbias, coeffs_cube,
                                    prepare_params),
                  " ");

    moo_try_check(detoff =
                      moo_prepare(flat_off_frame, bpmap_rp_name, bpmap_nl_name,
                                  masterbias, coeffs_cube, prepare_params),
                  " ");
    moo_try_check(moo_correct_bias(det, masterbias, cbias_params), " ");

    moo_try_check(detname1 = cpl_sprintf("%s_OFFSET%d_%s.fits",
                                         MOONS_TAG_FLAT_CORRECTBIAS, offset,
                                         moo_mode_get_name(mode)),
                  " ");
    moo_try_check(moo_products_add(products, det, CPL_FRAME_LEVEL_INTERMEDIATE,
                                   MOONS_TAG_FLAT_CORRECTBIAS, detname1,
                                   flat_attached_frame),
                  " ");

    moo_try_check(detname2 = cpl_sprintf("%s_OFFSET%d_%s.fits",
                                         MOONS_TAG_FLATOFF_PREPARE, offset,
                                         moo_mode_get_name(mode)),
                  " ");

    moo_try_check(moo_products_add(products, detoff,
                                   CPL_FRAME_LEVEL_INTERMEDIATE,
                                   MOONS_TAG_FLATOFF_PREPARE, detname2,
                                   flat_off_frame),
                  " ");

    moo_try_check(moo_correct_dark(det, detoff, masterdark_vis, masterdark_nir),
                  " ");
    moo_try_check(detname3 = cpl_sprintf("%s_OFFSET%d_%s.fits",
                                         MOONS_TAG_FLAT_CORRECTDARK, offset,
                                         moo_mode_get_name(mode)),
                  " ");
    moo_try_check(moo_products_add(products, det, CPL_FRAME_LEVEL_INTERMEDIATE,
                                   MOONS_TAG_FLAT_CORRECTDARK, detname3,
                                   flat_attached_frame),
                  " ");

    if (p2pmap != NULL) {
        moo_try_check(p2p = moo_det_create(p2pmap), " ");
        moo_try_check(moo_apply_p2p(det, p2p), " ");
        flatname = cpl_sprintf("%s_OFFSET%d_%s.fits", MOONS_TAG_FLAT_APPLYP2P,
                               offset, moo_mode_get_name(mode));
        moo_products_add(products, det, CPL_FRAME_LEVEL_INTERMEDIATE,
                         MOONS_TAG_FLAT_APPLYP2P, flatname,
                         flat_attached_frame);

        moo_det_delete(p2p);
        p2p = NULL;
    }

    extname =
        cpl_sprintf("EXT_OFFSET%d_%s.fits", offset, moo_mode_get_name(mode));

    moo_try_check(ff_trace = moo_loc_load(ff_trace_frame), " ");

    if (master_flat != NULL) {
        psf = moo_psf_load(master_flat);
    }
    moo_try_check(ext =
                      moo_extract(det, ff_trace, psf, extract_params, extname),
                  " ");

    moo_products_add_ext(products, ext, CPL_FRAME_LEVEL_INTERMEDIATE, "EXT",
                         extname, flat_attached_frame);


    f2fname =
        cpl_sprintf("F2F_OFFSET%d_%s.fits", offset, moo_mode_get_name(mode));
    f2f = moo_compute_fibtrans(ext, fibtrans_params, f2fname);
    moo_flat_shift_compute(det, ff_trace);

    moo_products_add_f2f(products, f2f, CPL_FRAME_LEVEL_FINAL,
                         MOONS_TAG_F2F_TABLE, f2fname, flat_attached_frame);

moo_try_cleanup:
    cpl_free(flatname);
    cpl_free(extname);
    cpl_free(f2fname);

    moo_ext_delete(ext);
    moo_f2f_delete(f2f);
    if (!cpl_errorstate_is_equal(prestate)) {
        moo_det_delete(p2p);
    }
    moo_det_delete(det);
    moo_det_delete(detoff);
    cpl_free(detname1);
    cpl_free(detname2);
    cpl_free(detname3);
    moo_loc_delete(ff_trace);
    moo_psf_delete(psf);

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
  @brief Execute the mflat workflow 2
  @param raw_byoffset_set array of FLAT files organize by offset keyword
  @param rawoff_frames the FLAT_OFF set of files
  @param bpmap_rp_name the BPMAP_RP file name or NULL
  @param bpmap_nl_name the BPMAP_NL file name or NULL
  @param master_bias the MASTER_BIAS frame or NULL
  @param master_dark_vis the MASTER_DARK_VIS frame or NULL
  @param master_dark_nir the MASTER_DARK_NIR frame or NULL
  @param ff_trace_guess the FF_TRACE_GUESS frame
  @param p2pmap the P2P_MAP frame or NULL
  @param crh_params the crh parameters
  @param localise_params the localisation parameters
  @param modelflat_params the model flat parameters
  @param fibtrans_params the compute fibres transmission parameters
  @param products the product object
  @return relevant error code
 *
 * Possible #_cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if a required input pointer is NULL
  - CPL_ERROR_ILLEGAL_INPUT if there is more of one OFFSET in the FLAT set
  - - CPL_ERROR_ILLEGAL_INPUT if there empty FLAT set
 */
/*----------------------------------------------------------------------------*/
#ifdef ALTERNATIVE_DRIVER
static int
_moons_mflat_wf2(cpl_frameset *raw_frames,
                 cpl_frameset *rawoff_frames,
                 const char *bpmap_rp_name,
                 const char *bpmap_nl_name,
                 const cpl_frame *masterbias,
                 const cpl_frame *masterdark_vis,
                 const cpl_frame *masterdark_nir,
                 const cpl_frame *coeffs_cube,
                 const cpl_frame *ff_trace_guess,
                 const cpl_frame *p2pmap,
                 moo_prepare_params *prepare_params,
                 moo_crh_params *crh_params,
                 moo_localise_params *localise_params,
                 moo_compute_slitoffset_params *compute_slitoffset_params,
                 moo_extract_params *extract_params,
                 moo_model_flat_params *modelflat_params,
                 moo_compute_fibtrans_params *fibtrans_params,
                 moo_correct_bias_params *cbias_params,
                 moo_products *products)
#else
static int
_moons_mflat_wf2(cpl_frameset **raw_byoffset_set,
                 cpl_frameset *rawoff_frames,
                 const char *bpmap_rp_name,
                 const char *bpmap_nl_name,
                 const cpl_frame *masterbias,
                 const cpl_frame *masterdark_vis,
                 const cpl_frame *masterdark_nir,
                 const cpl_frame *coeffs_cube,
                 const cpl_frame *ff_trace_guess,
                 const cpl_frame *p2pmap,
                 moo_prepare_params *prepare_params,
                 moo_crh_params *crh_params,
                 moo_localise_params *localise_params,
                 moo_compute_slitoffset_params *compute_slitoffset_params,
                 moo_extract_params *extract_params,
                 moo_model_flat_params *modelflat_params,
                 moo_compute_fibtrans_params *fibtrans_params,
                 moo_correct_bias_params *cbias_params,
                 moo_products *products)
#endif
{
#ifdef ALTERNATIVE_DRIVER
    cpl_ensure_code(raw_frames != NULL, CPL_ERROR_NULL_INPUT);
#else
    cpl_ensure_code(raw_byoffset_set != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(raw_byoffset_set[0] != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(raw_byoffset_set[1] != NULL, CPL_ERROR_NULL_INPUT);
#endif
    int domodel =
        (strcmp(extract_params->method, MOO_EXTRACT_METHOD_OPTIMAL) == 0);

    const cpl_frame *ref_frame = NULL;
    moo_det *medflat = NULL;
    moo_loc *ff_trace = NULL;
    moo_psf *psf = NULL;
    double offset;
    moo_mode_type mode;
#ifndef ALTERNATIVE_DRIVER
    cpl_frameset *raw_frames = NULL;

    int size1 = 0;
    int size2 = 0;

    moo_try_check(size1 = cpl_frameset_get_size(raw_byoffset_set[0]), " ");
    moo_try_check(size2 = cpl_frameset_get_size(raw_byoffset_set[1]), " ");

    if (size1 > 0) {
        if (size2 > 0) {
            return (int)cpl_error_set_message(
                cpl_func, CPL_ERROR_ILLEGAL_INPUT,
                "SOF have two different offset in input ");
        }
        raw_frames = raw_byoffset_set[0];
    }
    else {
        raw_frames = raw_byoffset_set[1];
    }
#endif

    moo_try_assure(cpl_frameset_get_size(raw_frames) > 0,
                   CPL_ERROR_ILLEGAL_INPUT, "No FLAT frames in input");


    moo_try_check(ff_trace = _moons_mflat_localise(
                      raw_frames, rawoff_frames, bpmap_rp_name, bpmap_nl_name,
                      masterbias, masterdark_vis, masterdark_nir, coeffs_cube,
                      p2pmap, ff_trace_guess, prepare_params, crh_params,
                      compute_slitoffset_params, localise_params,
                      modelflat_params, cbias_params, domodel,
                      MOONS_TAG_FF_TRACE, products, &medflat, &psf, &ref_frame,
                      &offset, &mode, CPL_FRAME_LEVEL_FINAL),
                  " ");

    moo_try_check(_moons_mflat_fibtrans(medflat, ff_trace, psf, extract_params,
                                        fibtrans_params, products, ref_frame,
                                        offset, mode),
                  " ");

moo_try_cleanup:
    moo_loc_delete(ff_trace);
    moo_det_delete(medflat);
    moo_psf_delete(psf);

    return CPL_ERROR_NONE;
}

#ifdef ALTERNATIVE_DRIVER
static int
_moons_mflat_wf3(cpl_frameset *frames_offset1,
                 cpl_frameset *frames_offset2,
                 cpl_frameset *frames_other,
                 moo_prepare_params *prepare_params,
                 moo_crh_params *crh_params,
                 moo_compute_slitoffset_params *compute_slitoffset_params,
                 moo_localise_params *localise_params,
                 moo_model_flat_params *modelflat_params,
                 moo_correct_bias_params *cbias_params,
                 moo_products *products)
#else
static int
_moons_mflat_wf3(cpl_frameset **raw_byoffset_set,
                 cpl_frameset *rawoff_frames,
                 const char *bpmap_rp_name,
                 const char *bpmap_nl_name,
                 const cpl_frame *masterbias,
                 const cpl_frame *masterdark_vis,
                 const cpl_frame *masterdark_nir,
                 cpl_frameset **cube_set,
                 cpl_frameset **ff_trace_set,
                 moo_prepare_params *prepare_params,
                 moo_crh_params *crh_params,
                 moo_compute_slitoffset_params *compute_slitoffset_params,
                 moo_localise_params *localise_params,
                 moo_model_flat_params *modelflat_params,
                 moo_correct_bias_params *cbias_params,
                 moo_products *products)
#endif
{
    const cpl_frame *coeffs_cube1 = NULL;
    const cpl_frame *coeffs_cube2 = NULL;
    const cpl_frame *ref_frame1 = NULL;
    const cpl_frame *ref_frame2 = NULL;

    moo_det *medflat1 = NULL;
    moo_loc *ff_trace1 = NULL;
    double offset1;
    moo_psf *psf1 = NULL;

    moo_det *medflat2 = NULL;
    moo_loc *ff_trace2 = NULL;
    double offset2;
    moo_psf *psf2 = NULL;

    moo_mode_type mode;
    moo_det *p2p = NULL;
    char *p2pname = NULL;

#ifdef ALTERNATIVE_DRIVER
    const int min_nflats_on = 3;

    cpl_frameset *raw_frames1 =
        moo_dfs_extract_tag(frames_offset1, MOONS_TAG_FLAT);
    cpl_frameset *raw_frames2 =
        moo_dfs_extract_tag(frames_offset2, MOONS_TAG_FLAT);

    if ((raw_frames1 == NULL) &&
        (cpl_frameset_get_size(raw_frames1) < min_nflats_on)) {
        cpl_error_code ecode = CPL_ERROR_DATA_NOT_FOUND;
        cpl_frameset_delete(raw_frames2);
        cpl_frameset_delete(raw_frames1);
        cpl_error_set_message(cpl_func, ecode,
                              "Too few input `%s' frames are available for "
                              "slit offset position `%d'. Minimum required "
                              "input frames is `%d'!",
                              MOONS_TAG_FLAT, 1, min_nflats_on);
        return ecode;
    }
    if ((raw_frames2 == NULL) &&
        (cpl_frameset_get_size(raw_frames2) < min_nflats_on)) {
        cpl_error_code ecode = CPL_ERROR_DATA_NOT_FOUND;
        cpl_frameset_delete(raw_frames2);
        cpl_frameset_delete(raw_frames1);
        cpl_error_set_message(cpl_func, ecode,
                              "Too few input `%s' frames are available for "
                              "slit offset position `%d'. Minimum required "
                              "input frames is `%d'!",
                              MOONS_TAG_FLAT, 2, min_nflats_on);
        return ecode;
    }
    if (cpl_frameset_get_size(raw_frames1) !=
        cpl_frameset_get_size(raw_frames2)) {
        cpl_error_code ecode = CPL_ERROR_INCOMPATIBLE_INPUT;
        cpl_frameset_delete(raw_frames2);
        cpl_frameset_delete(raw_frames1);
        cpl_error_set_message(
            cpl_func, ecode,
            "The number of provided `%s' frames for slit "
            "offset position `%d' does not match "
            "the number of provided `%s' frames for slit offset position `%d'!",
            MOONS_TAG_FLAT, 1, MOONS_TAG_FLAT, 2);
        return ecode;
    }

    cpl_frameset *rawoff_frames1 =
        moo_dfs_extract_tag(frames_offset1, MOONS_TAG_FLAT_OFF);
    cpl_frameset *rawoff_frames2 =
        moo_dfs_extract_tag(frames_offset2, MOONS_TAG_FLAT_OFF);

    if (cpl_frameset_get_size(rawoff_frames1) !=
        cpl_frameset_get_size(raw_frames1)) {
        cpl_error_code ecode = CPL_ERROR_INCOMPATIBLE_INPUT;
        cpl_frameset_delete(rawoff_frames2);
        cpl_frameset_delete(rawoff_frames1);
        cpl_frameset_delete(raw_frames2);
        cpl_frameset_delete(raw_frames1);
        cpl_error_set_message(cpl_func, ecode,
                              "The number of provided `%s' frames for slit "
                              "offset position `%d' does not match "
                              "the number of provided `%s' frames!",
                              MOONS_TAG_FLAT_OFF, 1, MOONS_TAG_FLAT);
        return ecode;
    }
    if (cpl_frameset_get_size(rawoff_frames2) !=
        cpl_frameset_get_size(raw_frames2)) {
        cpl_error_code ecode = CPL_ERROR_INCOMPATIBLE_INPUT;
        cpl_frameset_delete(rawoff_frames2);
        cpl_frameset_delete(rawoff_frames1);
        cpl_frameset_delete(raw_frames2);
        cpl_frameset_delete(raw_frames1);
        cpl_error_set_message(cpl_func, ecode,
                              "The number of provided `%s' frames for slit "
                              "offset position `%d' does not match "
                              "the number of provided `%s'frames!",
                              MOONS_TAG_FLAT_OFF, 2, MOONS_TAG_FLAT);
        return ecode;
    }

    const cpl_frame *bp_map_nl[] = {
        cpl_frameset_find_const(frames_offset1, MOONS_TAG_BP_MAP_NL),
        cpl_frameset_find_const(frames_offset2, MOONS_TAG_BP_MAP_NL)};
    if (((bp_map_nl[0] == NULL) && (bp_map_nl[1] != NULL)) ||
        ((bp_map_nl[1] == NULL) && (bp_map_nl[0] != NULL))) {
        cpl_error_code ecode = CPL_ERROR_INCOMPATIBLE_INPUT;
        cpl_frameset_delete(rawoff_frames2);
        cpl_frameset_delete(rawoff_frames1);
        cpl_frameset_delete(raw_frames2);
        cpl_frameset_delete(raw_frames1);
        cpl_error_set_message(
            cpl_func, ecode,
            "Inconsistent number of optional `%s' frames! The number of "
            "provided input frames must be the same for both slit offset "
            "positions!",
            MOONS_TAG_BP_MAP_NL);
        return ecode;
    }

    const cpl_frame *coeffs_cube[] = {
        cpl_frameset_find_const(frames_offset1, MOONS_TAG_LINEARITY_COEFF_CUBE),
        cpl_frameset_find_const(frames_offset2,
                                MOONS_TAG_LINEARITY_COEFF_CUBE)};
    if (((coeffs_cube[0] == NULL) && (coeffs_cube[1] != NULL)) ||
        ((coeffs_cube[1] == NULL) && (coeffs_cube[0] != NULL))) {
        cpl_error_code ecode = CPL_ERROR_INCOMPATIBLE_INPUT;
        cpl_frameset_delete(rawoff_frames2);
        cpl_frameset_delete(rawoff_frames1);
        cpl_frameset_delete(raw_frames2);
        cpl_frameset_delete(raw_frames1);
        cpl_error_set_message(
            cpl_func, ecode,
            "Inconsistent number of optional `%s' frames! The number of "
            "provided input frames must be the same for both slit offset "
            "positions!",
            MOONS_TAG_BP_MAP_NL);
        return ecode;
    }

    // FIXME: If the guess fftrace frame is not found this results in a different error code
    //        compared to the original code. This may have to be adjusted for compatibility
    //        reasons, but the returned error codes are anyway inconsistent in the original
    //        code when comparing the same error condition across the 4 workflows. So for
    //        now stick to the error code model used in the other functions of the new code.
    const cpl_frame *fftrace_guess[] = {
        cpl_frameset_find_const(frames_offset1, MOONS_TAG_FF_TRACE_GUESS),
        cpl_frameset_find_const(frames_offset2, MOONS_TAG_FF_TRACE_GUESS)};
    if (((fftrace_guess[0] == NULL) || (fftrace_guess[1] == NULL))) {
        cpl_error_code ecode = CPL_ERROR_DATA_NOT_FOUND;
        cpl_frameset_delete(rawoff_frames2);
        cpl_frameset_delete(rawoff_frames1);
        cpl_frameset_delete(raw_frames2);
        cpl_frameset_delete(raw_frames1);
        cpl_error_set_message(cpl_func, ecode,
                              "Required input frame `%s' is missing for at "
                              "least one slit offset position!",
                              MOONS_TAG_FF_TRACE_GUESS);
        return ecode;
    }

    const cpl_frame *bp_map_rp =
        cpl_frameset_find_const(frames_other, MOONS_TAG_BP_MAP_RP);
    const cpl_frame *masterbias =
        cpl_frameset_find_const(frames_other, MOONS_TAG_MASTER_BIAS);
    cpl_msg_info(cpl_func, "masterbias: %p", masterbias);
    cpl_frameset_dump(frames_other, NULL);
    cpl_msg_info(cpl_func, "master bias frame:");
    cpl_frame_dump(masterbias, NULL);

    const cpl_frame *masterdark_vis =
        cpl_frameset_find_const(frames_other, MOONS_TAG_MASTER_DARK_VIS);
    const cpl_frame *masterdark_nir =
        cpl_frameset_find_const(frames_other, MOONS_TAG_MASTER_DARK_NIR);

    if (masterbias == NULL) {
        cpl_error_code ecode = CPL_ERROR_DATA_NOT_FOUND;
        cpl_frameset_delete(rawoff_frames2);
        cpl_frameset_delete(rawoff_frames1);
        cpl_frameset_delete(raw_frames2);
        cpl_frameset_delete(raw_frames1);
        cpl_error_set_message(cpl_func, ecode,
                              "Required `%s' input frame is missing!",
                              MOONS_TAG_MASTER_BIAS);
        return ecode;
    }

    const char *bpmap_rp_name =
        (bp_map_rp != NULL) ? cpl_frame_get_filename(bp_map_rp) : NULL;

    // FIXME: Adapt to original function interface. Remove if new driver code
    //        is accepted!
    const cpl_frame *ff_trace_guess1 = fftrace_guess[0];
    const cpl_frame *ff_trace_guess2 = fftrace_guess[1];

    const char *bpmap_nl_name1 =
        (bp_map_nl[0] != NULL) ? cpl_frame_get_filename(bp_map_nl[0]) : NULL;
    const char *bpmap_nl_name2 =
        (bp_map_nl[1] != NULL) ? cpl_frame_get_filename(bp_map_nl[1]) : NULL;
#else
    int sff_trace1 = cpl_frameset_get_size(ff_trace_set[0]);
    int sff_trace2 = cpl_frameset_get_size(ff_trace_set[1]);

    cpl_ensure_code(sff_trace1 == 1, CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(sff_trace2 == 1, CPL_ERROR_ILLEGAL_INPUT);

    cpl_frameset *raw_frames1 = raw_byoffset_set[0];
    cpl_frameset *raw_frames2 = raw_byoffset_set[1];
    cpl_frameset *rawoff_frames1 = rawoff_frames;
    cpl_frameset *rawoff_frames2 = rawoff_frames;

    const cpl_frame *ff_trace_guess1 =
        cpl_frameset_get_position_const(ff_trace_set[0], 0);
    const cpl_frame *ff_trace_guess2 =
        cpl_frameset_get_position_const(ff_trace_set[1], 0);

    int csize1 = cpl_frameset_get_size(cube_set[0]);
    int csize2 = cpl_frameset_get_size(cube_set[1]);
    if (csize1 > 0) {
        coeffs_cube1 = cpl_frameset_get_position_const(cube_set[0], 0);
    }
    if (csize2 > 0) {
        coeffs_cube2 = cpl_frameset_get_position_const(cube_set[1], 0);
    }

    const char *bpmap_nl_name1 = bpmap_nl_name;
    const char *bpmap_nl_name2 = bpmap_nl_name;
#endif

    moo_try_check(ff_trace1 = _moons_mflat_localise(
                      raw_frames1, rawoff_frames1, bpmap_rp_name,
                      bpmap_nl_name1, masterbias, masterdark_vis,
                      masterdark_nir, coeffs_cube1, NULL, ff_trace_guess1,
                      prepare_params, crh_params, compute_slitoffset_params,
                      localise_params, modelflat_params, cbias_params, 1,
                      MOONS_TAG_FF_TRACE, products, &medflat1, &psf1,
                      &ref_frame1, &offset1, &mode,
                      CPL_FRAME_LEVEL_INTERMEDIATE),
                  "Fibre localisation failed for slit offset position 1!");

    moo_try_check(ff_trace2 = _moons_mflat_localise(
                      raw_frames2, rawoff_frames2, bpmap_rp_name,
                      bpmap_nl_name2, masterbias, masterdark_vis,
                      masterdark_nir, coeffs_cube2, NULL, ff_trace_guess2,
                      prepare_params, crh_params, compute_slitoffset_params,
                      localise_params, modelflat_params, cbias_params, 1,
                      MOONS_TAG_FF_TRACE, products, &medflat2, &psf2,
                      &ref_frame2, &offset2, &mode,
                      CPL_FRAME_LEVEL_INTERMEDIATE),
                  "Fibre localisation failed for slit offset position 2!");

    moo_try_check(p2p = moo_compute_p2p(medflat1, ff_trace1, psf1, medflat2,
                                        ff_trace2, psf2),
                  "Pixel-to=pixel variation map computation failed!");

    moo_try_check(p2pname = cpl_sprintf("%s_%s.fits", MOONS_TAG_P2P_MAP,
                                        moo_mode_get_name(mode)),
                  " ");

    moo_products_add_det(products, p2p, CPL_FRAME_LEVEL_FINAL,
                         MOONS_TAG_P2P_MAP, p2pname, ref_frame1,
                         "ESO INS SLIT .*");

moo_try_cleanup:
    moo_det_delete(medflat1);
    moo_loc_delete(ff_trace1);
    moo_det_delete(medflat2);
    moo_loc_delete(ff_trace2);
    moo_psf_delete(psf1);
    moo_psf_delete(psf2);
    moo_det_delete(p2p);
    cpl_free(p2pname);

    return CPL_ERROR_NONE;
}

#ifndef ALTERNATIVE_DRIVER
/*----------------------------------------------------------------------------*/
/**
  @brief  Extract from SOF the input data
  @param frameset the input SOF frameset
  @param ff_trace_guess_frames [output] the FF_TRACE_GUESS set of files
  @param raw_frames [output] the FLAT set of files
  @param raw_off_frames [output] the FLAT_OFF set of files
  @param bpmap_rp_name [output] the BPMAP_RP file name or NULL
  @param bpmap_nl_name [output] the BPMAP_NL file name or NULL
  @param master_bias [output] the MASTER_BIAS frame or NULL
  @param master_dark_vis [output] the MASTER_DARK_VIS frame or NULL
  @param p2pmap [output] the P2P_MAP frame or NULL
  @return relevant error code
 *
 * Possible #_cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if a required input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
static cpl_error_code
_moons_mflat_check_sof(cpl_frameset *frameset,
                       cpl_frameset **ff_trace_guess_frames,
                       cpl_frameset **raw_frames,
                       const cpl_frame **flat_attached,
                       const cpl_frame **ff_trace,
                       const cpl_frame **master_flat,
                       cpl_frameset **rawoff_frames,
                       const char **bpmap_rp_name,
                       const char **bpmap_nl_name,
                       const cpl_frame **master_bias,
                       const cpl_frame **master_dark_vis,
                       const cpl_frame **master_dark_nir,
                       cpl_frameset **coeffs_cube_frames,
                       const cpl_frame **p2pmap)
{
    cpl_errorstate prestate = cpl_errorstate_get();

    cpl_ensure_code(frameset != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(ff_trace_guess_frames != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(raw_frames != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(rawoff_frames != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(moo_dfs_set_groups(frameset) == CPL_ERROR_NONE,
                    cpl_error_get_code());

    moo_try_check(*ff_trace_guess_frames = cpl_frameset_new(), " ");
    moo_try_check(*coeffs_cube_frames = cpl_frameset_new(), " ");
    moo_try_check(*raw_frames = cpl_frameset_new(), " ");
    moo_try_check(*rawoff_frames = cpl_frameset_new(), " ");

    int size = 0;
    moo_try_check(size = cpl_frameset_get_size(frameset), " ");

    for (int i = 0; i < size; ++i) {
        const cpl_frame *current_frame = NULL;

        moo_try_check(current_frame =
                          cpl_frameset_get_position_const(frameset, i),
                      " ");

        if (!strcmp(cpl_frame_get_tag(current_frame), MOONS_TAG_FLAT)) {
            moo_try_check(cpl_frameset_insert(*raw_frames, cpl_frame_duplicate(
                                                               current_frame)),
                          " ");
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_FLAT_ATTACHED)) {
            *flat_attached = current_frame;
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_FLAT_OFF)) {
            moo_try_check(cpl_frameset_insert(*rawoff_frames,
                                              cpl_frame_duplicate(
                                                  current_frame)),
                          " ");
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_BP_MAP_RP)) {
            moo_try_check(*bpmap_rp_name =
                              cpl_frame_get_filename(current_frame),
                          " ");
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_BP_MAP_NL)) {
            moo_try_check(*bpmap_nl_name =
                              cpl_frame_get_filename(current_frame),
                          " ");
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_MASTER_BIAS)) {
            *master_bias = current_frame;
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_MASTER_DARK_VIS)) {
            *master_dark_vis = current_frame;
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_MASTER_DARK_NIR)) {
            *master_dark_nir = current_frame;
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame), MOONS_TAG_P2P_MAP)) {
            *p2pmap = current_frame;
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_FF_TRACE_GUESS)) {
            moo_try_check(cpl_frameset_insert(*ff_trace_guess_frames,
                                              cpl_frame_duplicate(
                                                  current_frame)),
                          " ");
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_FF_TRACE)) {
            *ff_trace = current_frame;
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_MASTER_FLAT)) {
            *master_flat = current_frame;
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_LINEARITY_COEFF_CUBE)) {
            moo_try_check(cpl_frameset_insert(*coeffs_cube_frames,
                                              cpl_frame_duplicate(
                                                  current_frame)),
                          " ");
        }
    }

moo_try_cleanup:
    if (!cpl_errorstate_is_equal(prestate)) {
        cpl_frameset_delete(*ff_trace_guess_frames);
        cpl_frameset_delete(*raw_frames);
        cpl_frameset_delete(*rawoff_frames);
        *ff_trace_guess_frames = NULL;
        *raw_frames = NULL;
        *rawoff_frames = NULL;
    }
    return cpl_error_get_code();
}
#endif

/*----------------------------------------------------------------------------*/
/**
  @brief    Interpret the command line options and execute the data processing
  @param    frameset   the frames list
  @param    parlist    the parameters list
  @return   0 if everything is ok
 */
/*----------------------------------------------------------------------------*/
static int
_moons_mflat(cpl_frameset *frameset, const cpl_parameterlist *parlist)
{
#ifdef ALTERNATIVE_DRIVER
    const char *const msg_id = "moons_mflat";

    // moons_mflat is essentially four recipes, with related but different
    // purposes and thus 4 different "workflows" with different input data
    // requirements. Three of the 4 supported workflows work on a single
    // slit offset position, but "workflow 3" works on both slit offset
    // positions simultaneously. To keep the workflow interfaces simple
    // with respect to the input data, the incoming frameset is first
    // split by slit offset position, each of these framesets is then
    // checked for completeness depending on the requested workflow.

    // Tag input frames with their respective DFS 'RAW' and 'CALIB' group labels
    cpl_error_code ecode = moo_dfs_set_groups(frameset);
    cpl_ensure((ecode == CPL_ERROR_NONE), ecode, ecode);

    // Split the input set-of-frames. The input frames are sorted into groups according to
    // their slit offset position

    cpl_frameset *frames_offset1 = cpl_frameset_new();
    cpl_frameset *frames_offset2 = cpl_frameset_new();
    cpl_frameset *frames_other = cpl_frameset_new();

    ecode = moo_dfs_group_offsets(frameset, frames_offset1, frames_offset2,
                                  frames_other);
    if (ecode != CPL_ERROR_NONE) {
        cpl_frameset_delete(frames_other);
        cpl_frameset_delete(frames_offset2);
        cpl_frameset_delete(frames_offset1);
        cpl_error_set_message(cpl_func, ecode,
                              "Cannot group input frames according to their "
                              "slit offset position property");
        return ecode;
    }

    typedef enum
    {
        UNDEFINED = -1,
        UNKNOWN = 0,
        GUESS = 1,
        TRACE,
        P2P,
        ATTACHED
    } mflat_mode;

    // Determine the operating mode of moons_mflat. Check the grouped framesets for
    // the input data required for each mode.

    mflat_mode mode = UNDEFINED;
    cpl_frameset *frames = NULL;
    if ((!cpl_frameset_is_empty(frames_offset1) &&
         (cpl_frameset_count_tags(frames_offset1, MOONS_TAG_FLAT) > 0)) &&
        (!cpl_frameset_is_empty(frames_offset2) &&
         (cpl_frameset_count_tags(frames_offset2, MOONS_TAG_FLAT) > 0))) {
        mode = P2P;
        cpl_msg_debug(msg_id,
                      "Input data set contains `%s' for 2 slit offset "
                      "positions, running in mode `%d'",
                      MOONS_TAG_FLAT, (int)mode);
    }
    else {
        frames = (!cpl_frameset_is_empty(frames_offset1))
                     ? cpl_frameset_duplicate(frames_offset1)
                     : cpl_frameset_duplicate(frames_offset2);
        cpl_frameset_join(frames, frames_other);

        if (cpl_frameset_count_tags(frames, MOONS_TAG_FLAT_ATTACHED) > 0) {
            mode = ATTACHED;
            cpl_msg_debug(
                msg_id,
                "Found frame with tag `%s` in input, running in mode `%d'",
                MOONS_TAG_FLAT_ATTACHED, (int)mode);
        }
        else if ((cpl_frameset_count_tags(frames_offset1, MOONS_TAG_FLAT) >
                  0) &&
                 (cpl_frameset_count_tags(frames, MOONS_TAG_FF_TRACE_GUESS) >
                  0)) {
            mode = TRACE;
            cpl_msg_debug(
                msg_id,
                "Found frame with tag `%s` in input, running in mode `%d'",
                MOONS_TAG_FF_TRACE_GUESS, (int)mode);
        }
        else if (cpl_frameset_count_tags(frames_offset1, MOONS_TAG_FLAT) > 0) {
            mode = GUESS;
            cpl_msg_debug(
                msg_id,
                "Got data without fibre tracing solution, running in mode `%d'",
                (int)mode);
        }
        else {
            mode = UNKNOWN;
        }
    }
    if ((mode == UNDEFINED) || (mode == UNKNOWN)) {
        cpl_frameset_delete(frames);
        cpl_frameset_delete(frames_other);
        cpl_frameset_delete(frames_offset2);
        cpl_frameset_delete(frames_offset1);
        ecode = CPL_ERROR_ILLEGAL_INPUT;
        cpl_error_set_message(
            cpl_func, ecode,
            "Unable to determine the `%s' operating mode! Received input data "
            "does not contain a valid set of frames to run any `%s' workflow!",
            msg_id, msg_id);
        return ecode;
    }

    moo_products *products = moo_products_new(frameset, parlist, "moons_mflat",
                                              PACKAGE "/" PACKAGE_VERSION);

    // Setup parameter configuration
    const moo_params *params = moo_products_get_params(products);

    // Declare parameter variables prior to the first call to moo_try_check
    // otherwise they will be used uninitialized on cleanup!
    moo_prepare_params *prepare_params = NULL;
    moo_correct_bias_params *correct_bias_params = NULL;
    moo_localise_params *localise_params = NULL;
    moo_crh_params *crh_params = NULL;
    moo_compute_slitoffset_params *compute_slitoffset_params = NULL;
    moo_extract_params *extract_params = NULL;
    moo_model_flat_params *model_flat_params = NULL;
    moo_compute_fibtrans_params *fibtrans_params = NULL;

    moo_try_check(prepare_params = moo_params_get_prepare(params, parlist),
                  " ");
    moo_try_check(correct_bias_params =
                      moo_params_get_correct_bias(params, parlist),
                  " ");
    moo_try_check(localise_params = moo_params_get_localise(params, parlist),
                  " ");
    moo_try_check(crh_params = moo_params_get_crh(params, parlist), " ");
    moo_try_check(compute_slitoffset_params =
                      moo_params_get_compute_slitoffset(params, parlist),
                  " ");
    moo_try_check(extract_params = moo_params_get_extract(params, parlist),
                  " ");
    moo_try_check(model_flat_params =
                      moo_params_get_model_flat(params, parlist),
                  " ");
    moo_try_check(fibtrans_params =
                      moo_params_get_compute_fibtrans(params, parlist),
                  " ");


    // FIXME: All checks preceding the actual workflow function call should be moved
    //        into that function. Benefits are a unified interface just passing
    //        framesets and it simplifies the error handling. For now we keep
    //        the workflow function interface as much as possible and try to
    //        stick to the original error handling scheme using work-arounds
    //        when necessary!
    switch (mode) {
        case GUESS: {
            cpl_msg_info(msg_id,
                         "Workflow 1 : create a guess fibre "
                         "tracing solution `%s' (single slit offset)",
                         MOONS_TAG_FF_TRACE_GUESS);

            const int min_nflats_on = 3;

            cpl_frameset *flats_on =
                moo_dfs_extract_tag(frames, MOONS_TAG_FLAT);
            cpl_frameset *flats_off =
                moo_dfs_extract_tag(frames, MOONS_TAG_FLAT_OFF);
            const cpl_frame *bp_map_rp =
                cpl_frameset_find_const(frames, MOONS_TAG_BP_MAP_RP);
            const cpl_frame *bp_map_nl =
                cpl_frameset_find_const(frames, MOONS_TAG_BP_MAP_NL);
            const cpl_frame *masterbias =
                cpl_frameset_find_const(frames, MOONS_TAG_MASTER_BIAS);
            const cpl_frame *masterdark_vis =
                cpl_frameset_find_const(frames, MOONS_TAG_MASTER_DARK_VIS);
            const cpl_frame *masterdark_nir =
                cpl_frameset_find_const(frames, MOONS_TAG_MASTER_DARK_NIR);

            // Check for required input frames
            if ((flats_on == NULL) ||
                (cpl_frameset_get_size(flats_on) < min_nflats_on)) {
                cpl_frameset_delete(flats_off);
                ecode = CPL_ERROR_DATA_NOT_FOUND;
                cpl_error_set_message(
                    cpl_func, ecode,
                    "Too few input `%s' frames are available. Minimum required "
                    "input frames is `%d'!",
                    MOONS_TAG_FLAT, min_nflats_on);
                break;
            }
            if (cpl_frameset_get_size(flats_on) !=
                cpl_frameset_get_size(flats_off)) {
                cpl_frameset_delete(flats_off);
                cpl_frameset_delete(flats_on);
                ecode = CPL_ERROR_INCOMPATIBLE_INPUT;
                cpl_error_set_message(
                    cpl_func, ecode,
                    "The number of provided `%s' frames does not match "
                    "the number of provided `%s'frames!",
                    MOONS_TAG_FLAT_OFF, MOONS_TAG_FLAT);
                break;
            }
            if (masterbias == NULL) {
                cpl_frameset_delete(flats_off);
                cpl_frameset_delete(flats_on);
                ecode = CPL_ERROR_DATA_NOT_FOUND;
                cpl_error_set_message(cpl_func, ecode,
                                      "Required `%s' input frame is missing!",
                                      MOONS_TAG_MASTER_BIAS);
                break;
            }

            const char *bpmap_rp_name =
                (bp_map_rp != NULL) ? cpl_frame_get_filename(bp_map_rp) : NULL;
            const char *bpmap_nl_name =
                (bp_map_nl != NULL) ? cpl_frame_get_filename(bp_map_nl) : NULL;

            _moons_mflat_wf1(flats_on, flats_off, bpmap_rp_name, bpmap_nl_name,
                             masterbias, masterdark_vis, masterdark_nir, NULL,
                             prepare_params, crh_params, localise_params,
                             correct_bias_params, products);
            break;
        }
        case TRACE: {
            cpl_msg_info(msg_id,
                         "Workflow 2 : create a fibre "
                         "tracing solution `%s' (single slit offset)",
                         MOONS_TAG_FF_TRACE);

            const int min_nflats_on = 3;

            cpl_frameset *flats_on =
                moo_dfs_extract_tag(frames, MOONS_TAG_FLAT);
            cpl_frameset *flats_off =
                moo_dfs_extract_tag(frames, MOONS_TAG_FLAT_OFF);
            const cpl_frame *bp_map_rp =
                cpl_frameset_find_const(frames, MOONS_TAG_BP_MAP_RP);
            const cpl_frame *bp_map_nl =
                cpl_frameset_find_const(frames, MOONS_TAG_BP_MAP_NL);
            const cpl_frame *masterbias =
                cpl_frameset_find_const(frames, MOONS_TAG_MASTER_BIAS);
            const cpl_frame *masterdark_vis =
                cpl_frameset_find_const(frames, MOONS_TAG_MASTER_DARK_VIS);
            const cpl_frame *masterdark_nir =
                cpl_frameset_find_const(frames, MOONS_TAG_MASTER_DARK_NIR);
            const cpl_frame *fftrace_guess =
                cpl_frameset_find_const(frames, MOONS_TAG_FF_TRACE_GUESS);
            const cpl_frame *p2pmap =
                cpl_frameset_find_const(frames, MOONS_TAG_P2P_MAP);
            const cpl_frame *coeffs_cube =
                cpl_frameset_find_const(frames, MOONS_TAG_LINEARITY_COEFF_CUBE);


            // Check for required input frames
            if ((flats_on == NULL) ||
                (cpl_frameset_get_size(flats_on) < min_nflats_on)) {
                cpl_frameset_delete(flats_off);
                ecode = CPL_ERROR_DATA_NOT_FOUND;
                cpl_error_set_message(
                    cpl_func, ecode,
                    "Too few input `%s' frames are available. Minimum required "
                    "input frames is `%d'!",
                    MOONS_TAG_FLAT, min_nflats_on);
                break;
            }
            if (cpl_frameset_get_size(flats_on) !=
                cpl_frameset_get_size(flats_off)) {
                cpl_frameset_delete(flats_off);
                cpl_frameset_delete(flats_on);
                ecode = CPL_ERROR_INCOMPATIBLE_INPUT;
                cpl_error_set_message(
                    cpl_func, ecode,
                    "The number of provided `%s' frames does not match "
                    "the number of provided `%s'frames!",
                    MOONS_TAG_FLAT_OFF, MOONS_TAG_FLAT);
                break;
            }
            if (masterbias == NULL) {
                cpl_frameset_delete(flats_off);
                cpl_frameset_delete(flats_on);
                ecode = CPL_ERROR_DATA_NOT_FOUND;
                cpl_error_set_message(cpl_func, ecode,
                                      "Required `%s' input frame is missing!",
                                      MOONS_TAG_MASTER_BIAS);
                break;
            }
            if (fftrace_guess == NULL) {
                cpl_frameset_delete(flats_off);
                cpl_frameset_delete(flats_on);
                ecode = CPL_ERROR_DATA_NOT_FOUND;
                cpl_error_set_message(cpl_func, ecode,
                                      "Required `%s' input frame is missing!",
                                      MOONS_TAG_FF_TRACE_GUESS);
                break;
            }

            const char *bpmap_rp_name =
                (bp_map_rp != NULL) ? cpl_frame_get_filename(bp_map_rp) : NULL;
            const char *bpmap_nl_name =
                (bp_map_nl != NULL) ? cpl_frame_get_filename(bp_map_nl) : NULL;

            _moons_mflat_wf2(flats_on, flats_off, bpmap_rp_name, bpmap_nl_name,
                             masterbias, masterdark_vis, masterdark_nir,
                             coeffs_cube, fftrace_guess, p2pmap, prepare_params,
                             crh_params, localise_params,
                             compute_slitoffset_params, extract_params,
                             model_flat_params, fibtrans_params,
                             correct_bias_params, products);
            break;
        }
        case P2P: {
            cpl_msg_info(msg_id,
                         "Workflow 3 : create a pixel-to-pixel variation map "
                         "`%s' (dual slit offset)",
                         MOONS_TAG_P2P_MAP);

            _moons_mflat_wf3(frames_offset1, frames_offset2, frames_other,
                             prepare_params, crh_params,
                             compute_slitoffset_params, localise_params,
                             model_flat_params, correct_bias_params, products);

            break;
        }
        case ATTACHED: {
            cpl_msg_info(msg_id,
                         "Workflow 4 : create a fibre-to-fibre relative "
                         "response `%s' (single slit offset)",
                         MOONS_TAG_F2F_TABLE);

            const cpl_frame *flat_attached =
                cpl_frameset_find_const(frames, MOONS_TAG_FLAT_ATTACHED);
            const cpl_frame *flat_off =
                cpl_frameset_find_const(frames, MOONS_TAG_FLAT_OFF);
            const cpl_frame *bp_map_rp =
                cpl_frameset_find_const(frames, MOONS_TAG_BP_MAP_RP);
            const cpl_frame *bp_map_nl =
                cpl_frameset_find_const(frames, MOONS_TAG_BP_MAP_NL);
            const cpl_frame *masterbias =
                cpl_frameset_find_const(frames, MOONS_TAG_MASTER_BIAS);
            const cpl_frame *masterdark_vis =
                cpl_frameset_find_const(frames, MOONS_TAG_MASTER_DARK_VIS);
            const cpl_frame *masterdark_nir =
                cpl_frameset_find_const(frames, MOONS_TAG_MASTER_DARK_NIR);
            const cpl_frame *masterflat =
                cpl_frameset_find_const(frames, MOONS_TAG_MASTER_FLAT);
            const cpl_frame *fftrace =
                cpl_frameset_find_const(frames, MOONS_TAG_FF_TRACE);
            const cpl_frame *p2pmap =
                cpl_frameset_find_const(frames, MOONS_TAG_P2P_MAP);
            const cpl_frame *coeffs_cube =
                cpl_frameset_find_const(frames, MOONS_TAG_LINEARITY_COEFF_CUBE);

            // Check for required input frames
            moo_try_assure((flat_attached != NULL), CPL_ERROR_DATA_NOT_FOUND,
                           "Required `%s' input frame is missing!",
                           MOONS_TAG_FLAT_ATTACHED);
            moo_try_assure((flat_off != NULL), CPL_ERROR_DATA_NOT_FOUND,
                           "Required `%s' input frame is missing!",
                           MOONS_TAG_FLAT_OFF);
            moo_try_assure((fftrace != NULL), CPL_ERROR_DATA_NOT_FOUND,
                           "Required `%s' input frame is missing!",
                           MOONS_TAG_FF_TRACE);

            const char *bpmap_rp_name =
                (bp_map_rp != NULL) ? cpl_frame_get_filename(bp_map_rp) : NULL;
            const char *bpmap_nl_name =
                (bp_map_nl != NULL) ? cpl_frame_get_filename(bp_map_nl) : NULL;

            _moons_mflat_wf4(flat_attached, flat_off, bpmap_rp_name,
                             bpmap_nl_name, masterbias, masterdark_vis,
                             masterdark_nir, coeffs_cube, fftrace, p2pmap,
                             masterflat, prepare_params, extract_params,
                             fibtrans_params, correct_bias_params, products);
            break;
        }
        default: {
            // This point should never be reached!
            cpl_error_set_message(
                cpl_func, CPL_ERROR_UNSUPPORTED_MODE,
                "Got unsupported mflat operating mode '%d'! Aborting!", mode);
            break;
        }
    }

moo_try_cleanup:
    moo_compute_fibtrans_params_delete(fibtrans_params);
    moo_model_flat_params_delete(model_flat_params);
    moo_extract_params_delete(extract_params);
    moo_compute_slitoffset_params_delete(compute_slitoffset_params);
    moo_crh_params_delete(crh_params);
    moo_localise_params_delete(localise_params);
    moo_correct_bias_params_delete(correct_bias_params);
    moo_prepare_params_delete(prepare_params);
    moo_products_delete(products);
    cpl_frameset_delete(frames);
    cpl_frameset_delete(frames_other);
    cpl_frameset_delete(frames_offset2);
    cpl_frameset_delete(frames_offset1);

#else

    /* parameters */
    moo_prepare_params *prepare_params = NULL;
    moo_correct_bias_params *correct_bias_params = NULL;
    moo_localise_params *localise_params = NULL;
    moo_crh_params *crh_params = NULL;
    moo_compute_slitoffset_params *compute_slitoffset_params = NULL;
    moo_extract_params *extract_params = NULL;
    moo_model_flat_params *model_flat_params = NULL;
    moo_compute_fibtrans_params *fibtrans_params = NULL;

    /* input files */
    const char *bpmap_rp_name = NULL;
    const char *bpmap_nl_name = NULL;
    const cpl_frame *masterbias = NULL;
    const cpl_frame *masterdark_vis = NULL;
    const cpl_frame *masterdark_nir = NULL;
    const cpl_frame *p2pmap = NULL;
    cpl_frameset *coeffs_cube_frames = NULL;
    cpl_frameset *ff_trace_guess_frames = NULL;
    cpl_frameset *raw_frames = NULL;
    const cpl_frame *flat_attached = NULL;
    const cpl_frame *ff_trace = NULL;
    const cpl_frame *master_flat = NULL;
    cpl_frameset *rawoff_frames = NULL;
    cpl_frameset **raw_byoffset_set = NULL;

    moo_products *products = moo_products_new(frameset, parlist, "moons_mflat",
                                              PACKAGE "/" PACKAGE_VERSION);

    /* parameters */
    const moo_params *params = moo_products_get_params(products);
    moo_try_check(prepare_params = moo_params_get_prepare(params, parlist),
                  " ");

    moo_try_check(correct_bias_params =
                      moo_params_get_correct_bias(params, parlist),
                  " ");
    moo_try_check(localise_params = moo_params_get_localise(params, parlist),
                  " ");
    moo_try_check(crh_params = moo_params_get_crh(params, parlist), " ");

    moo_try_check(compute_slitoffset_params =
                      moo_params_get_compute_slitoffset(params, parlist),
                  " ");

    moo_try_check(extract_params = moo_params_get_extract(params, parlist),
                  " ");

    moo_try_check(model_flat_params =
                      moo_params_get_model_flat(params, parlist),
                  " ");

    moo_try_check(fibtrans_params =
                      moo_params_get_compute_fibtrans(params, parlist),
                  " ");

    /* SOF file */
    moo_try_check(_moons_mflat_check_sof(frameset, &ff_trace_guess_frames,
                                         &raw_frames, &flat_attached, &ff_trace,
                                         &master_flat, &rawoff_frames,
                                         &bpmap_rp_name, &bpmap_nl_name,
                                         &masterbias, &masterdark_vis,
                                         &masterdark_nir, &coeffs_cube_frames,
                                         &p2pmap),
                  " ");

    int size = 0;
    int csize = 0;

    moo_try_check(csize = cpl_frameset_get_size(coeffs_cube_frames), " ");
    moo_try_check(size = cpl_frameset_get_size(ff_trace_guess_frames), " ");
    moo_try_check(raw_byoffset_set = moo_dfs_split_by_offset(raw_frames), " ");

    if (flat_attached != NULL) {
        cpl_msg_info("moons_mflat", "Workflow 4 : create a F2F_TABLE");
        const cpl_frame *coeffs_cube = NULL;
        const cpl_frame *flat_off = NULL;
        int off_size = 0;
        moo_try_check(off_size = cpl_frameset_get_size(rawoff_frames), " ");
        if (off_size > 0) {
            flat_off = cpl_frameset_get_position_const(rawoff_frames, 0);
        }
        if (csize > 0) {
            coeffs_cube =
                cpl_frameset_get_position_const(coeffs_cube_frames, 0);
        }

        _moons_mflat_wf4(flat_attached, flat_off, bpmap_rp_name, bpmap_nl_name,
                         masterbias, masterdark_vis, masterdark_nir,
                         coeffs_cube, ff_trace, p2pmap, master_flat,
                         prepare_params, extract_params, fibtrans_params,
                         correct_bias_params, products);
    }
    else if (size == 0) {
        cpl_msg_info("moons_mflat", "Workflow 1 : create a FF_TRACE_GUESS");
        _moons_mflat_wf1(raw_byoffset_set, rawoff_frames, bpmap_rp_name,
                         bpmap_nl_name, masterbias, masterdark_vis,
                         masterdark_nir, NULL, prepare_params, crh_params,
                         localise_params, correct_bias_params, products);
    }
    else if (size == 1) {
        const cpl_frame *coeffs_cube = NULL;

        if (csize > 0) {
            coeffs_cube =
                cpl_frameset_get_position_const(coeffs_cube_frames, 0);
        }
        cpl_msg_info("moons_mflat", "Workflow 2 : one offset reduction");
        const cpl_frame *ff_trace_guess =
            cpl_frameset_get_position_const(ff_trace_guess_frames, 0);
        _moons_mflat_wf2(raw_byoffset_set, rawoff_frames, bpmap_rp_name,
                         bpmap_nl_name, masterbias, masterdark_vis,
                         masterdark_nir, coeffs_cube, ff_trace_guess, p2pmap,
                         prepare_params, crh_params, localise_params,
                         compute_slitoffset_params, extract_params,
                         model_flat_params, fibtrans_params,
                         correct_bias_params, products);
    }
    else {
        cpl_msg_info("moons_mflat", "Workflow 3 : two offsets reduction");
        cpl_frameset **fftrace_byoffset_set =
            moo_dfs_split_by_offset(ff_trace_guess_frames);
        cpl_frameset **cube_byoffset_set =
            moo_dfs_split_by_offset(coeffs_cube_frames);

        _moons_mflat_wf3(raw_byoffset_set, rawoff_frames, bpmap_rp_name,
                         bpmap_nl_name, masterbias, masterdark_vis,
                         masterdark_nir, cube_byoffset_set,
                         fftrace_byoffset_set, prepare_params, crh_params,
                         compute_slitoffset_params, localise_params,
                         model_flat_params, correct_bias_params, products);
        for (int i = 0; i < 2; i++) {
            cpl_frameset_delete(fftrace_byoffset_set[i]);
        }
        cpl_free(fftrace_byoffset_set);

        for (int i = 0; i < 2; i++) {
            cpl_frameset_delete(cube_byoffset_set[i]);
        }
        cpl_free(cube_byoffset_set);
    }

moo_try_cleanup:
    if (raw_byoffset_set != NULL) {
        for (int i = 0; i < 2; i++) {
            cpl_frameset_delete(raw_byoffset_set[i]);
        }
        cpl_free(raw_byoffset_set);
    }
    cpl_frameset_delete(raw_frames);
    cpl_frameset_delete(rawoff_frames);
    cpl_frameset_delete(coeffs_cube_frames);
    cpl_frameset_delete(ff_trace_guess_frames);
    moo_crh_params_delete(crh_params);
    moo_localise_params_delete(localise_params);
    moo_compute_slitoffset_params_delete(compute_slitoffset_params);
    moo_extract_params_delete(extract_params);
    moo_model_flat_params_delete(model_flat_params);
    moo_compute_fibtrans_params_delete(fibtrans_params);
    moo_prepare_params_delete(prepare_params);
    moo_correct_bias_params_delete(correct_bias_params);
    moo_products_delete(products);
#endif

    return (int)cpl_error_get_code();
}
