/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#include "moo_utils.h"
#include "moo_pfits.h"
#include "moo_dfs.h"
#include "moo_params.h"
#include "moo_drl.h"
#include "moo_products.h"
#include <cpl.h>
#include "hdrl.h"
#include <string.h>
#include "moo_correct_tell.h"
#include "moo_molecfit.h"
/*-----------------------------------------------------------------------------
                              Plugin registration
 -----------------------------------------------------------------------------*/

int cpl_plugin_get_info(cpl_pluginlist *list);

/*-----------------------------------------------------------------------------
                            Private function prototypes
 -----------------------------------------------------------------------------*/

static int _moons_molecfit_correct_create(cpl_plugin *plugin);
static int _moons_molecfit_correct_exec(cpl_plugin *plugin);
static int _moons_molecfit_correct_destroy(cpl_plugin *plugin);
static int _moons_molecfit_correct(cpl_frameset *frameset,
                                   const cpl_parameterlist *parlist);

/*-----------------------------------------------------------------------------
                            Static variables
 -----------------------------------------------------------------------------*/

static const char *const _moons_molecfit_correct_description =
    "Apply telluric correction on science files\n"
    "INPUT FRAMES\n"
    "  * file (SCI) with tag SCIENCE 1 to n files : "
    " star file to correct \n"
    "  * file (TELLURIC_CORR) with tag TELLURIC_CORR 1 file : "
    "telluric correction file\n"
    "PRODUCTS\n"
    "  * SCIENCE_TELLCORSPECTRA_i.fits (SCI) with tag SCIENCE_TELLCORSPECTRA : "
    "science with telluric correction apply"
    "\n";

/*-----------------------------------------------------------------------------
                                Function code
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Build the list of available plugins, for this module.
  @param    list    the plugin list
  @return   0 if everything is ok, 1 otherwise
  @note     Only this function is exported

  Create the recipe instance and make it available to the application using the
  interface.
 */
/*----------------------------------------------------------------------------*/

int
cpl_plugin_get_info(cpl_pluginlist *list)
{
    cpl_recipe *recipe = cpl_calloc(1, sizeof *recipe);
    cpl_plugin *plugin = &recipe->interface;

    if (cpl_plugin_init(plugin, CPL_PLUGIN_API, MOONS_BINARY_VERSION,
                        CPL_PLUGIN_TYPE_RECIPE, "moons_molecfit_correct",
                        "Apply telluric correction to the science",
                        _moons_molecfit_correct_description, "Regis Haigron",
                        PACKAGE_BUGREPORT, moo_get_license(),
                        _moons_molecfit_correct_create,
                        _moons_molecfit_correct_exec,
                        _moons_molecfit_correct_destroy)) {
        cpl_msg_error(cpl_func, "Plugin initialization failed");
        (void)cpl_error_set_where(cpl_func);
        return 1;
    }

    if (cpl_pluginlist_append(list, plugin)) {
        cpl_msg_error(cpl_func, "Error adding plugin to list");
        (void)cpl_error_set_where(cpl_func);
        return 1;
    }

    return 0;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Setup the recipe options
  @param    plugin  the plugin
  @return   0 if everything is ok

  Defining the command-line/configuration parameters for the recipe.
 */
/*----------------------------------------------------------------------------*/

static int
_moons_molecfit_correct_create(cpl_plugin *plugin)
{
    cpl_recipe *recipe;

    /* Do not create the recipe if an error code is already set */
    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        cpl_msg_error(cpl_func, "%s():%d: An error is already set: %s",
                      cpl_func, __LINE__, cpl_error_get_where());
        return (int)cpl_error_get_code();
    }

    if (plugin == NULL) {
        cpl_msg_error(cpl_func, "Null plugin");
        cpl_ensure_code(0, (int)CPL_ERROR_NULL_INPUT);
    }

    /* Verify plugin type */
    if (cpl_plugin_get_type(plugin) != CPL_PLUGIN_TYPE_RECIPE) {
        cpl_msg_error(cpl_func, "Plugin is not a recipe");
        cpl_ensure_code(0, (int)CPL_ERROR_TYPE_MISMATCH);
    }

    /* Get the recipe */
    recipe = (cpl_recipe *)plugin;

    /* Create the parameters list in the cpl_recipe object */
    recipe->parameters = cpl_parameterlist_new();
    if (recipe->parameters == NULL) {
        cpl_msg_error(cpl_func, "Parameter list allocation failed");
        cpl_ensure_code(0, (int)CPL_ERROR_ILLEGAL_OUTPUT);
    }

    moo_params *params = moo_params_new("moons", "moons_molecfit_correct");

    /* Fill the parameters list */
    moo_params_add_keep_temp(params, recipe->parameters);
    moo_params_delete(params);

    return 0;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Execute the plugin instance given by the interface
  @param    plugin  the plugin
  @return   0 if everything is ok
 */
/*----------------------------------------------------------------------------*/

static int
_moons_molecfit_correct_exec(cpl_plugin *plugin)
{
    cpl_recipe *recipe;
    int recipe_status;
    cpl_errorstate initial_errorstate = cpl_errorstate_get();

    /* Return immediately if an error code is already set */
    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        cpl_msg_error(cpl_func, "%s():%d: An error is already set: %s",
                      cpl_func, __LINE__, cpl_error_get_where());
        return (int)cpl_error_get_code();
    }

    if (plugin == NULL) {
        cpl_msg_error(cpl_func, "Null plugin");
        cpl_ensure_code(0, (int)CPL_ERROR_NULL_INPUT);
    }

    /* Verify plugin type */
    if (cpl_plugin_get_type(plugin) != CPL_PLUGIN_TYPE_RECIPE) {
        cpl_msg_error(cpl_func, "Plugin is not a recipe");
        cpl_ensure_code(0, (int)CPL_ERROR_TYPE_MISMATCH);
    }

    /* Get the recipe */
    recipe = (cpl_recipe *)plugin;

    /* Verify parameter and frame lists */
    if (recipe->parameters == NULL) {
        cpl_msg_error(cpl_func, "Recipe invoked with NULL parameter list");
        cpl_ensure_code(0, (int)CPL_ERROR_NULL_INPUT);
    }
    if (recipe->frames == NULL) {
        cpl_msg_error(cpl_func, "Recipe invoked with NULL frame set");
        cpl_ensure_code(0, (int)CPL_ERROR_NULL_INPUT);
    }

    /* Invoke the recipe */
    recipe_status = _moons_molecfit_correct(recipe->frames, recipe->parameters);

    /* Ensure DFS-compliance of the products */
    if (cpl_dfs_update_product_header(recipe->frames)) {
        if (!recipe_status)
            recipe_status = (int)cpl_error_get_code();
    }

    if (!cpl_errorstate_is_equal(initial_errorstate)) {
        /* Dump the error history since recipe execution start.
           At this point the recipe cannot recover from the error */
        cpl_errorstate_dump(initial_errorstate, CPL_FALSE, NULL);
    }

    return recipe_status;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Destroy what has been created by the 'create' function
  @param    plugin  the plugin
  @return   0 if everything is ok
 */
/*----------------------------------------------------------------------------*/

static int
_moons_molecfit_correct_destroy(cpl_plugin *plugin)
{
    cpl_recipe *recipe;

    if (plugin == NULL) {
        cpl_msg_error(cpl_func, "Null plugin");
        cpl_ensure_code(0, (int)CPL_ERROR_NULL_INPUT);
    }

    /* Verify plugin type */
    if (cpl_plugin_get_type(plugin) != CPL_PLUGIN_TYPE_RECIPE) {
        cpl_msg_error(cpl_func, "Plugin is not a recipe");
        cpl_ensure_code(0, (int)CPL_ERROR_TYPE_MISMATCH);
    }

    /* Get the recipe */
    recipe = (cpl_recipe *)plugin;

    cpl_parameterlist_delete(recipe->parameters);

    return 0;
}

static cpl_error_code
_moons_molecfit_correct_check_sof(cpl_frameset *frameset,
                                  cpl_frameset **sci_frameset,
                                  const cpl_frame **tell_frame)
{
    cpl_ensure_code(moo_dfs_set_groups(frameset) == CPL_ERROR_NONE,
                    cpl_error_get_code());
    int i;

    moo_try_check(*sci_frameset = cpl_frameset_new(), " ");

    for (i = 0; i < cpl_frameset_get_size(frameset); ++i) {
        cpl_frame *current_frame = cpl_frameset_get_position(frameset, i);
        if (!strcmp(cpl_frame_get_tag(current_frame), MOONS_TAG_MOLECFIT_SCI)) {
            cpl_frame *new_frame = cpl_frame_duplicate(current_frame);
            cpl_frameset_insert(*sci_frameset, new_frame);
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_TELLURIC_CORR)) {
            *tell_frame = current_frame;
        }
    }
    if (*tell_frame == NULL) {
        return (int)cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                          "SOF does not have any file tagged "
                                          "with %s",
                                          MOONS_TAG_TELLURIC_CORR);
    }

moo_try_cleanup:
    return CPL_ERROR_NONE;
}


static cpl_frameset *
_moons_correct_set(cpl_frameset *set,
                   moo_telluric *tell,
                   moo_products *products)
{
    moo_sci *sci = NULL;
    char *sci_name = NULL;
    cpl_ensure(set != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(tell != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(products != NULL, CPL_ERROR_NULL_INPUT, NULL);

    cpl_frameset *result = cpl_frameset_new();
    int i;

    cpl_errorstate prestate = cpl_errorstate_get();

    for (i = 0; i < cpl_frameset_get_size(set); ++i) {
        const cpl_frame *current_frame = NULL;
        moo_try_check(current_frame = cpl_frameset_get_position_const(set, i),
                      " ");
        sci = moo_sci_create(current_frame);
        moo_try_check(moo_correct_tell_science(sci, tell), " ");
        sci_name =
            cpl_sprintf("%s_%d.fits", MOONS_TAG_SCIENCE_TELLCORRSPECTRA, i);
        cpl_frame *product =
            moo_products_add_sci(products, sci, CPL_FRAME_LEVEL_FINAL,
                                 MOONS_TAG_SCIENCE_TELLCORRSPECTRA, sci_name,
                                 current_frame);
        cpl_free(sci_name);
        moo_try_check(cpl_frameset_insert(result, cpl_frame_duplicate(product)),
                      " ");
        moo_sci_delete(sci);
        sci = NULL;
    }
moo_try_cleanup:
    if (!cpl_errorstate_is_equal(prestate)) {
        cpl_msg_error(__func__, "dump error ");
        cpl_errorstate_dump(prestate, CPL_FALSE, cpl_errorstate_dump_one);
        moo_sci_delete(sci);
        cpl_frameset_delete(result);
        result = NULL;
        cpl_free(sci_name);
    }
    return result;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Interpret the command line options and execute the data processing
  @param    frameset   the frames list
  @param    parlist    the parameters list
  @return   0 if everything is ok  @return   0 if everything is ok
 */
/*----------------------------------------------------------------------------*/
static int
_moons_molecfit_correct(cpl_frameset *frameset,
                        const cpl_parameterlist *parlist)
{
    /* parameters */
    cpl_frameset *science_set = NULL;
    const cpl_frame *tell_frame = NULL;
    moo_telluric *tell = NULL;
    cpl_frameset *result_set = NULL;

    moo_products *products =
        moo_products_new(frameset, parlist, "moons_molecfit_correct",
                         PACKAGE "/" PACKAGE_VERSION);


    /* SOF file */
    moo_try_check(_moons_molecfit_correct_check_sof(frameset, &science_set,
                                                    &tell_frame),
                  " ");

    moo_try_check(tell = moo_telluric_load(tell_frame), " ");
    moo_try_check(result_set = _moons_correct_set(science_set, tell, products),
                  " ");

moo_try_cleanup:
    moo_products_delete(products);
    cpl_frameset_delete(science_set);
    moo_telluric_delete(tell);
    cpl_frameset_delete(result_set);

    return (int)cpl_error_get_code();
}
