/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#include "hdrl.h"
#include "moo_dfs.h"
#include "moo_drl.h"
#include "moo_molecfit.h"
#include "moo_molectable.h"
#include "moo_params.h"
#include "moo_pfits.h"
#include "moo_products.h"
#include "moo_utils.h"
#include <cpl.h>
#include <string.h>
/*-----------------------------------------------------------------------------
                              Plugin registration
 -----------------------------------------------------------------------------*/

int cpl_plugin_get_info(cpl_pluginlist *list);

/*-----------------------------------------------------------------------------
                            Private function prototypes
 -----------------------------------------------------------------------------*/

static int _moons_molecfit_model_create(cpl_plugin *plugin);
static int _moons_molecfit_model_exec(cpl_plugin *plugin);
static int _moons_molecfit_model_destroy(cpl_plugin *plugin);
static int
_moons_molecfit_model(cpl_frameset *frameset, const cpl_parameterlist *parlist);

/*-----------------------------------------------------------------------------
                            Static variables
 -----------------------------------------------------------------------------*/

static const char *const _moons_molecfit_model_description =
    "Produces the model using molecfit\n"
    "INPUT FRAMES\n"
    "  * file (SCI) with tag SCIENCE 1 file : "
    "molecfit_model star file\n"
    "  * file (MOLECULES) with tag MOLECULES 1 file : "
    "molecules list file\n"
    "PRODUCTS\n"
    "  * GDAS User GDAS profile"
    "  * GDAS_BEFORE If ESO DB GDAS is used, file before the MJD-OBS"
    "  * GDAS_AFTER If ESO DB GDAS is used, file after the MJD-OBS"
    "\n";

/*-----------------------------------------------------------------------------
                                Function code
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Build the list of available plugins, for this module.
  @param    list    the plugin list
  @return   0 if everything is ok, 1 otherwise
  @note     Only this function is exported

  Create the recipe instance and make it available to the application using the
  interface.
 */
/*----------------------------------------------------------------------------*/

int
cpl_plugin_get_info(cpl_pluginlist *list)
{
    cpl_recipe *recipe = cpl_calloc(1, sizeof *recipe);
    cpl_plugin *plugin = &recipe->interface;

    if (cpl_plugin_init(
            plugin, CPL_PLUGIN_API, MOONS_BINARY_VERSION,
            CPL_PLUGIN_TYPE_RECIPE, "moons_molecfit_model",
            "Produces the instrumental response for relative flux calibration, "
            "and the telluric absorption spectrum.",
            _moons_molecfit_model_description, "Regis Haigron",
            PACKAGE_BUGREPORT, moo_get_license(), _moons_molecfit_model_create,
            _moons_molecfit_model_exec, _moons_molecfit_model_destroy)) {
        cpl_msg_error(cpl_func, "Plugin initialization failed");
        (void)cpl_error_set_where(cpl_func);
        return 1;
    }

    if (cpl_pluginlist_append(list, plugin)) {
        cpl_msg_error(cpl_func, "Error adding plugin to list");
        (void)cpl_error_set_where(cpl_func);
        return 1;
    }

    return 0;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Setup the recipe options
  @param    plugin  the plugin
  @return   0 if everything is ok

  Defining the command-line/configuration parameters for the recipe.
 */
/*----------------------------------------------------------------------------*/

static int
_moons_molecfit_model_create(cpl_plugin *plugin)
{
    cpl_recipe *recipe;

    /* Do not create the recipe if an error code is already set */
    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        cpl_msg_error(cpl_func, "%s():%d: An error is already set: %s",
                      cpl_func, __LINE__, cpl_error_get_where());
        return (int)cpl_error_get_code();
    }

    if (plugin == NULL) {
        cpl_msg_error(cpl_func, "Null plugin");
        cpl_ensure_code(0, (int)CPL_ERROR_NULL_INPUT);
    }

    /* Verify plugin type */
    if (cpl_plugin_get_type(plugin) != CPL_PLUGIN_TYPE_RECIPE) {
        cpl_msg_error(cpl_func, "Plugin is not a recipe");
        cpl_ensure_code(0, (int)CPL_ERROR_TYPE_MISMATCH);
    }

    /* Get the recipe */
    recipe = (cpl_recipe *)plugin;

    /* Create the parameters list in the cpl_recipe object */
    recipe->parameters = cpl_parameterlist_new();
    if (recipe->parameters == NULL) {
        cpl_msg_error(cpl_func, "Parameter list allocation failed");
        cpl_ensure_code(0, (int)CPL_ERROR_ILLEGAL_OUTPUT);
    }

    moo_params *params = moo_params_new("moons", "moons_molecfit_model");

    /* Fill the parameters list */
    moo_params_add_keep_temp(params, recipe->parameters);
    moo_params_add_molecfit_model(params, recipe->parameters);
    moo_params_delete(params);

    return 0;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Execute the plugin instance given by the interface
  @param    plugin  the plugin
  @return   0 if everything is ok
 */
/*----------------------------------------------------------------------------*/

static int
_moons_molecfit_model_exec(cpl_plugin *plugin)
{
    cpl_recipe *recipe;
    int recipe_status;
    cpl_errorstate initial_errorstate = cpl_errorstate_get();

    /* Return immediately if an error code is already set */
    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        cpl_msg_error(cpl_func, "%s():%d: An error is already set: %s",
                      cpl_func, __LINE__, cpl_error_get_where());
        return (int)cpl_error_get_code();
    }

    if (plugin == NULL) {
        cpl_msg_error(cpl_func, "Null plugin");
        cpl_ensure_code(0, (int)CPL_ERROR_NULL_INPUT);
    }

    /* Verify plugin type */
    if (cpl_plugin_get_type(plugin) != CPL_PLUGIN_TYPE_RECIPE) {
        cpl_msg_error(cpl_func, "Plugin is not a recipe");
        cpl_ensure_code(0, (int)CPL_ERROR_TYPE_MISMATCH);
    }

    /* Get the recipe */
    recipe = (cpl_recipe *)plugin;

    /* Verify parameter and frame lists */
    if (recipe->parameters == NULL) {
        cpl_msg_error(cpl_func, "Recipe invoked with NULL parameter list");
        cpl_ensure_code(0, (int)CPL_ERROR_NULL_INPUT);
    }
    if (recipe->frames == NULL) {
        cpl_msg_error(cpl_func, "Recipe invoked with NULL frame set");
        cpl_ensure_code(0, (int)CPL_ERROR_NULL_INPUT);
    }

    /* Invoke the recipe */
    recipe_status = _moons_molecfit_model(recipe->frames, recipe->parameters);

    /* Ensure DFS-compliance of the products */
    if (cpl_dfs_update_product_header(recipe->frames)) {
        if (!recipe_status)
            recipe_status = (int)cpl_error_get_code();
    }

    if (!cpl_errorstate_is_equal(initial_errorstate)) {
        /* Dump the error history since recipe execution start.
           At this point the recipe cannot recover from the error */
        cpl_errorstate_dump(initial_errorstate, CPL_FALSE, NULL);
    }

    return recipe_status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Destroy what has been created by the 'create' function
  @param    plugin  the plugin
  @return   0 if everything is ok
 */
/*----------------------------------------------------------------------------*/

static int
_moons_molecfit_model_destroy(cpl_plugin *plugin)
{
    cpl_recipe *recipe;

    if (plugin == NULL) {
        cpl_msg_error(cpl_func, "Null plugin");
        cpl_ensure_code(0, (int)CPL_ERROR_NULL_INPUT);
    }

    /* Verify plugin type */
    if (cpl_plugin_get_type(plugin) != CPL_PLUGIN_TYPE_RECIPE) {
        cpl_msg_error(cpl_func, "Plugin is not a recipe");
        cpl_ensure_code(0, (int)CPL_ERROR_TYPE_MISMATCH);
    }

    /* Get the recipe */
    recipe = (cpl_recipe *)plugin;

    cpl_parameterlist_delete(recipe->parameters);

    return 0;
}

static cpl_error_code
_moons_molecfit_model_check_sof(cpl_frameset *frameset,
                                const cpl_frame **sci_frame,
                                const cpl_frame **molecules_frame,
                                const cpl_frame **winc_frame)
{
    cpl_ensure_code(moo_dfs_set_groups(frameset) == CPL_ERROR_NONE,
                    cpl_error_get_code());
    int i;

    for (i = 0; i < cpl_frameset_get_size(frameset); ++i) {
        cpl_frame *current_frame = cpl_frameset_get_position(frameset, i);
        if (!strcmp(cpl_frame_get_tag(current_frame), MOONS_TAG_MOLECFIT_SCI)) {
            *sci_frame = current_frame;
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_MOLECFIT_MOLECULES)) {
            *molecules_frame = current_frame;
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_MOLECFIT_WINCLUDE)) {
            *winc_frame = current_frame;
        }
    }
    if (*sci_frame == NULL) {
        return (int)cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                          "SOF does not have any file tagged "
                                          "with %s",
                                          MOONS_TAG_MOLECFIT_SCI);
    }

    if (*molecules_frame == NULL) {
        return (int)cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                          "SOF does not have any file tagged "
                                          "with %s",
                                          MOONS_TAG_MOLECFIT_MOLECULES);
    }

    if (*winc_frame == NULL) {
        return (int)cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                          "SOF does not have any file tagged "
                                          "with %s",
                                          MOONS_TAG_MOLECFIT_WINCLUDE);
    }
    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Interpret the command line options and execute the data processing
  @param    frameset   the frames list
  @param    parlist    the parameters list
  @return   0 if everything is ok  @return   0 if everything is ok
 */
/*----------------------------------------------------------------------------*/
static int
_moons_molecfit_model(cpl_frameset *frameset, const cpl_parameterlist *parlist)
{
    /* parameters */
    moo_mode_type mode;
    const char *modename = NULL;
    const cpl_frame *sci_frame = NULL;
    const cpl_frame *molecule_frame = NULL;
    const cpl_frame *winc_frame = NULL;
    moo_molecfit_model_params *molecfit_model_params = NULL;

    moo_molectable *atm_fitted = NULL;
    moo_molectable *best_fitted_params = NULL;
    moo_molectable *best_fitted_model = NULL;

    char *best_fitted_params_name = NULL;
    char *best_fitted_model_name = NULL;
    char *atm_fitted_name = NULL;

    moo_products *products =
        moo_products_new(frameset, parlist, "moons_molecfit_model",
                         PACKAGE "/" PACKAGE_VERSION);

    const moo_params *params = moo_products_get_params(products);
    moo_try_check(molecfit_model_params =
                      moo_params_get_molecfit_model(params, parlist),
                  " ");

    /* SOF file */
    moo_try_check(_moons_molecfit_model_check_sof(frameset, &sci_frame,
                                                  &molecule_frame, &winc_frame),
                  " ");

    moo_try_check(mode = moo_mode_get(sci_frame), " ");

    modename = moo_mode_get_name(mode);

    moo_try_check(best_fitted_params =
                      moo_molecfit_model(sci_frame, modename, molecule_frame,
                                         winc_frame, &atm_fitted,
                                         &best_fitted_model,
                                         molecfit_model_params),
                  " ");
    atm_fitted_name = cpl_sprintf("%s_%s.fits", MOONS_TAG_MOLECFIT_ATM_PARAMS,
                                  moo_mode_get_name(mode));
    best_fitted_params_name =
        cpl_sprintf("%s_%s.fits", MOONS_TAG_MOLECFIT_BEST_FIT_PARAMS,
                    moo_mode_get_name(mode));
    best_fitted_model_name =
        cpl_sprintf("%s_%s.fits", MOONS_TAG_MOLECFIT_BEST_FIT_MODEL,
                    moo_mode_get_name(mode));

    moo_try_check(moo_products_add_molectable(
                      products, best_fitted_params, CPL_FRAME_LEVEL_FINAL,
                      MOONS_TAG_MOLECFIT_BEST_FIT_PARAMS,
                      best_fitted_params_name, sci_frame),
                  " ");
    moo_try_check(moo_products_add_molectable(products, best_fitted_model,
                                              CPL_FRAME_LEVEL_INTERMEDIATE,
                                              MOONS_TAG_MOLECFIT_BEST_FIT_MODEL,
                                              best_fitted_model_name,
                                              sci_frame),
                  " ");
    moo_try_check(moo_products_add_molectable(products, atm_fitted,
                                              CPL_FRAME_LEVEL_FINAL,
                                              MOONS_TAG_MOLECFIT_ATM_PARAMS,
                                              atm_fitted_name, sci_frame),
                  " ");

moo_try_cleanup:
    cpl_free(best_fitted_params_name);
    cpl_free(best_fitted_model_name);
    cpl_free(atm_fitted_name);
    moo_molectable_delete(best_fitted_model);
    moo_molectable_delete(atm_fitted);
    moo_molectable_delete(best_fitted_params);
    moo_molecfit_model_params_delete(molecfit_model_params);
    moo_products_delete(products);
    return (int)cpl_error_get_code();
}
