/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#include "moo_dfs.h"
#include "moo_drl.h"
#include "moo_extlist.h"
#include "moo_params.h"
#include "moo_pfits.h"
#include "moo_products.h"
#include "moo_utils.h"
#include <cpl.h>

#include <string.h>

/*-----------------------------------------------------------------------------
                              Plugin registration
 -----------------------------------------------------------------------------*/

int cpl_plugin_get_info(cpl_pluginlist *list);

/*-----------------------------------------------------------------------------
                            Private function prototypes
 -----------------------------------------------------------------------------*/

static int _moons_quicklook_create(cpl_plugin *plugin);
static int _moons_quicklook_exec(cpl_plugin *plugin);
static int _moons_quicklook_destroy(cpl_plugin *plugin);
static int
_moons_quicklook(cpl_frameset *frameset, const cpl_parameterlist *parlist);

/*-----------------------------------------------------------------------------
                            Static variables
 -----------------------------------------------------------------------------*/

static const char *const _moons_quicklook_description =
    "quickly produce reduced spectra for a subset of the observed objects and "
    "assess the quality of the data\n"
    "INPUT FRAMES\n"
    "  * science exposure frame 1 file            (RAW) with tag OBJECT_STARE "
    ": "
    "science frame in stare mode file\n"
    "  * [OPTIONAL] ReferenceBadPixMask 1 file    (QUA) with tag BP_MAP_RP : "
    "cosmetic bad pixel map\n"
    "  * [OPTIONAL] NonLinearityBadPixMask 1 file (QUA) with tag BP_MAP_NL : "
    "cosmetic bad pixel map coming from linearity recipe\n"
    "  * MasterBias 1 file                        (DET) with tag MASTER_BIAS : "
    "master bias file\n"
    "  * MasterDarkNir 1 file                     (DET) with tag "
    "MASTER_DARK_NIR : "
    "master dark nir file\n"
    "  * [OPTIONAL]MasterDarkVis 1 file           (DET) with tag "
    "MASTER_DARK_VIS : "
    "master dark vis file\n"
    "  * [OPTIONAL] CoeffsCube 1 file             (3D) with tag "
    "LINEARITY_COEFF_CUBE : "
    "coefficients to correct pixels detector linearity\n"
    "  * [OPTIONAL] P2pMap 1 file                 (DET) with tag P2P_MAP : "
    "pixel to pixel map\n"
    "  * LocTab 1 file                            (LOC) with tag FF_TRACE : "
    "the localisation table\n"
    "  * Ffext 1 file                             (EXT) with tag FF_EXTSPECTRA "
    ": "
    "the extracted flat field\n"
    "  * F2f 1 file                               (F2F) with tag F2F_TABLE : "
    "the fibre-to-fibre relative response\n"
    "  * SFormat 1 file                           (FMT) with tag "
    "SPECTRAL_FORMAT : "
    "the spectral format table\n"
    "  * WaveMap 1 file                           (WMAP) with tag WAVE_MAP : "
    "the wavelength map\n"
    "  * SkyLines 1 file                          (CAT) with tag SKY_LINE_LIST "
    ": "
    "the sky lines list table\n"
    "PRODUCTS\n"
    "  * QUICKLOOK_SCI.fits (SCI) with tag SCIENCE_QUICKLOOK_SKSSPECTRA : "
    "Science product coadd calibrated frame\n"
    "\n";

/*-----------------------------------------------------------------------------
                                Function code
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Build the list of available plugins, for this module.
  @param    list    the plugin list
  @return   0 if everything is ok, 1 otherwise
  @note     Only this function is exported

  Create the recipe instance and make it available to the application using the
  interface.
 */
/*----------------------------------------------------------------------------*/

int
cpl_plugin_get_info(cpl_pluginlist *list)
{
    cpl_recipe *recipe = cpl_calloc(1, sizeof *recipe);
    cpl_plugin *plugin = &recipe->interface;

    if (cpl_plugin_init(plugin, CPL_PLUGIN_API, MOONS_BINARY_VERSION,
                        CPL_PLUGIN_TYPE_RECIPE, "moons_quicklook",
                        "quickly produce reduced spectra for a subset of the "
                        "observed objects and assess the quality of the data",
                        _moons_quicklook_description, "Regis Haigron",
                        PACKAGE_BUGREPORT, moo_get_license(),
                        _moons_quicklook_create, _moons_quicklook_exec,
                        _moons_quicklook_destroy)) {
        cpl_msg_error(cpl_func, "Plugin initialization failed");
        (void)cpl_error_set_where(cpl_func);
        return 1;
    }

    if (cpl_pluginlist_append(list, plugin)) {
        cpl_msg_error(cpl_func, "Error adding plugin to list");
        (void)cpl_error_set_where(cpl_func);
        return 1;
    }

    return 0;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Setup the recipe options
  @param    plugin  the plugin
  @return   0 if everything is ok

  Defining the command-line/configuration parameters for the recipe.
 */
/*----------------------------------------------------------------------------*/

static int
_moons_quicklook_create(cpl_plugin *plugin)
{
    cpl_recipe *recipe;

    /* Do not create the recipe if an error code is already set */
    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        cpl_msg_error(cpl_func, "%s():%d: An error is already set: %s",
                      cpl_func, __LINE__, cpl_error_get_where());
        return (int)cpl_error_get_code();
    }

    if (plugin == NULL) {
        cpl_msg_error(cpl_func, "Null plugin");
        cpl_ensure_code(0, (int)CPL_ERROR_NULL_INPUT);
    }

    /* Verify plugin type */
    if (cpl_plugin_get_type(plugin) != CPL_PLUGIN_TYPE_RECIPE) {
        cpl_msg_error(cpl_func, "Plugin is not a recipe");
        cpl_ensure_code(0, (int)CPL_ERROR_TYPE_MISMATCH);
    }

    /* Get the recipe */
    recipe = (cpl_recipe *)plugin;

    /* Create the parameters list in the cpl_recipe object */
    recipe->parameters = cpl_parameterlist_new();
    if (recipe->parameters == NULL) {
        cpl_msg_error(cpl_func, "Parameter list allocation failed");
        cpl_ensure_code(0, (int)CPL_ERROR_ILLEGAL_OUTPUT);
    }

    moo_params *params = moo_params_new("moons", "moons_quicklook");

    /* Fill the parameters list */
    moo_params_add_keep_temp(params, recipe->parameters);
    moo_params_add_prepare(params, recipe->parameters);
    moo_params_add_quicklook_fibre_list(params, recipe->parameters);
    moo_params_add_correct_bias(params, recipe->parameters,
                                MOO_CORRECT_BIAS_METHOD_MASTER);
    moo_params_add_rebin(params, recipe->parameters);
    moo_params_add_sub_sky_stare_simple(params, recipe->parameters);
    moo_params_delete(params);

    return 0;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Execute the plugin instance given by the interface
  @param    plugin  the plugin
  @return   0 if everything is ok
 */
/*----------------------------------------------------------------------------*/

static int
_moons_quicklook_exec(cpl_plugin *plugin)
{
    cpl_recipe *recipe;
    int recipe_status;
    cpl_errorstate initial_errorstate = cpl_errorstate_get();

    /* Return immediately if an error code is already set */
    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        cpl_msg_error(cpl_func, "%s():%d: An error is already set: %s",
                      cpl_func, __LINE__, cpl_error_get_where());
        return (int)cpl_error_get_code();
    }

    if (plugin == NULL) {
        cpl_msg_error(cpl_func, "Null plugin");
        cpl_ensure_code(0, (int)CPL_ERROR_NULL_INPUT);
    }

    /* Verify plugin type */
    if (cpl_plugin_get_type(plugin) != CPL_PLUGIN_TYPE_RECIPE) {
        cpl_msg_error(cpl_func, "Plugin is not a recipe");
        cpl_ensure_code(0, (int)CPL_ERROR_TYPE_MISMATCH);
    }

    /* Get the recipe */
    recipe = (cpl_recipe *)plugin;

    /* Verify parameter and frame lists */
    if (recipe->parameters == NULL) {
        cpl_msg_error(cpl_func, "Recipe invoked with NULL parameter list");
        cpl_ensure_code(0, (int)CPL_ERROR_NULL_INPUT);
    }
    if (recipe->frames == NULL) {
        cpl_msg_error(cpl_func, "Recipe invoked with NULL frame set");
        cpl_ensure_code(0, (int)CPL_ERROR_NULL_INPUT);
    }

    /* Invoke the recipe */
    recipe_status = _moons_quicklook(recipe->frames, recipe->parameters);

    /* Ensure DFS-compliance of the products */
    if (cpl_dfs_update_product_header(recipe->frames)) {
        if (!recipe_status)
            recipe_status = (int)cpl_error_get_code();
    }

    if (!cpl_errorstate_is_equal(initial_errorstate)) {
        /* Dump the error history since recipe execution start.
           At this point the recipe cannot recover from the error */
        cpl_errorstate_dump(initial_errorstate, CPL_FALSE, NULL);
    }

    return recipe_status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Destroy what has been created by the 'create' function
  @param    plugin  the plugin
  @return   0 if everything is ok
 */
/*----------------------------------------------------------------------------*/

static int
_moons_quicklook_destroy(cpl_plugin *plugin)
{
    cpl_recipe *recipe;

    if (plugin == NULL) {
        cpl_msg_error(cpl_func, "Null plugin");
        cpl_ensure_code(0, (int)CPL_ERROR_NULL_INPUT);
    }

    /* Verify plugin type */
    if (cpl_plugin_get_type(plugin) != CPL_PLUGIN_TYPE_RECIPE) {
        cpl_msg_error(cpl_func, "Plugin is not a recipe");
        cpl_ensure_code(0, (int)CPL_ERROR_TYPE_MISMATCH);
    }

    /* Get the recipe */
    recipe = (cpl_recipe *)plugin;

    cpl_parameterlist_delete(recipe->parameters);

    return 0;
}

static int
_moons_select_quicklook_targets(moo_det *det,
                                moo_loc *loc,
                                const cpl_frame *f2f_frame,
                                moo_sub_sky_stare_params *params,
                                cpl_array *index_list)
{
    moo_f2f *f2f = NULL;
    cpl_array *sel = NULL;
    cpl_array *sky_sel = NULL;
    cpl_array *obj_sel = NULL;
    cpl_table **sky_stare_tables = NULL;
    int nb_targets = 0;

    cpl_ensure(det != NULL, CPL_ERROR_NULL_INPUT, 0);
    cpl_ensure(loc != NULL, CPL_ERROR_NULL_INPUT, 0);
    cpl_ensure(f2f_frame != NULL, CPL_ERROR_NULL_INPUT, 0);
    cpl_ensure(params != NULL, CPL_ERROR_NULL_INPUT, 0);

    moo_try_check(f2f = moo_f2f_load(f2f_frame), " ");
    cpl_table *det_fibre_table = moo_det_get_fibre_table(det);
    cpl_table *loc_fibre_table = moo_loc_get_fibre_table(loc);

    moo_try_assure(det_fibre_table != NULL, CPL_ERROR_ILLEGAL_INPUT, " ");
    moo_try_assure(loc_fibre_table != NULL, CPL_ERROR_ILLEGAL_INPUT, " ");


    /* sort all table by index */
    moo_fibres_table_by_index(det_fibre_table);
    moo_fibres_table_by_index(loc_fibre_table);
    moo_f2f_order_by_index(f2f);
    cpl_table_new_column(det_fibre_table, MOO_TARGET_TABLE_INDEXRBN,
                         CPL_TYPE_INT);
    int tsize = cpl_table_get_nrow(det_fibre_table);
    int *indexextt =
        cpl_table_get_data_int(loc_fibre_table, MOO_FIBRES_TABLE_INDEXEXT);
    for (int i = 1; i <= tsize; i++) {
        int indexext = indexextt[i];
        if (indexext > 0) {
            cpl_table_set_int(det_fibre_table, MOO_TARGET_TABLE_INDEXRBN, i - 1,
                              i);
        }
    }
    int index_size = cpl_array_get_size(index_list);
    /* remove all CHECK_OBJECT */
    if (index_size > 0) {
        cpl_table_unselect_all(det_fibre_table);
        int size =
            cpl_table_or_selected_string(det_fibre_table, MOO_FIBRES_TABLE_TYPE,
                                         CPL_EQUAL_TO,
                                         MOO_FIBRES_TABLE_TYPE_CHECKOBJ);
        sel = cpl_table_where_selected(det_fibre_table);
        for (int i = 0; i < size; i++) {
            int fidx = cpl_array_get_cplsize(sel, i, NULL);
            cpl_table_set_string(det_fibre_table, MOO_FIBRES_TABLE_TYPE, fidx,
                                 "");
        }
        cpl_array_delete(sel);
        sel = NULL;

        for (int i = 0; i < index_size; i++) {
            const char *fibname = cpl_array_get_string(index_list, i);
            cpl_table_unselect_all(det_fibre_table);
            size = cpl_table_or_selected_string(det_fibre_table,
                                                MOO_FIBRES_TABLE_FIBRE,
                                                CPL_EQUAL_TO, fibname);
            sel = cpl_table_where_selected(det_fibre_table);
            int fidx = cpl_array_get_cplsize(sel, 0, NULL);
            cpl_table_set_string(det_fibre_table, MOO_FIBRES_TABLE_TYPE, fidx,
                                 MOO_FIBRES_TABLE_TYPE_CHECKOBJ);
            cpl_array_delete(sel);
            sel = NULL;
        }
    }
    /* propagate broken fibre from loc */
    cpl_table_unselect_all(loc_fibre_table);
    int size =
        cpl_table_or_selected_int(loc_fibre_table, MOO_FIBRES_TABLE_HEALTH,
                                  CPL_EQUAL_TO, 0);
    sel = cpl_table_where_selected(loc_fibre_table);

    for (int i = 0; i < size; i++) {
        int fidx = cpl_array_get_cplsize(sel, i, NULL);
        cpl_table_set_int(det_fibre_table, MOO_FIBRES_TABLE_HEALTH, fidx, 0);
    }
    cpl_array_delete(sel);
    sel = NULL;

    cpl_table_select_all(det_fibre_table);
    cpl_table_and_selected_int(det_fibre_table, MOO_FIBRES_TABLE_INDEXRBN,
                               CPL_NOT_EQUAL_TO, 0);
    cpl_table_and_selected_int(det_fibre_table, MOO_FIBRES_TABLE_HEALTH,
                               CPL_NOT_EQUAL_TO, 0);
    nb_targets =
        cpl_table_and_selected_string(det_fibre_table, MOO_FIBRES_TABLE_TYPE,
                                      CPL_EQUAL_TO,
                                      MOO_FIBRES_TABLE_TYPE_CHECKOBJ);

    if (nb_targets > 0) {
        size = cpl_table_or_selected_string(det_fibre_table,
                                            MOO_FIBRES_TABLE_TYPE, CPL_EQUAL_TO,
                                            MOO_FIBRES_TABLE_TYPE_SKY);
        cpl_table_and_selected_int(det_fibre_table, MOO_FIBRES_TABLE_INDEXRBN,
                                   CPL_NOT_EQUAL_TO, 0);

        size = cpl_table_not_selected(det_fibre_table);
        sel = cpl_table_where_selected(det_fibre_table);

        for (int i = 0; i < size; i++) {
            int fidx = cpl_array_get_cplsize(sel, i, NULL);
            int h = cpl_table_get_int(det_fibre_table, MOO_FIBRES_TABLE_HEALTH,
                                      fidx, NULL);
            if (h != 0) {
                cpl_table_set_int(det_fibre_table, MOO_FIBRES_TABLE_HEALTH,
                                  fidx, -1);
            }
        }
        cpl_table_select_all(det_fibre_table);
        int sky_size =
            cpl_table_and_selected_string(det_fibre_table,
                                          MOO_FIBRES_TABLE_TYPE, CPL_EQUAL_TO,
                                          MOO_FIBRES_TABLE_TYPE_SKY);
        sky_size =
            cpl_table_and_selected_int(det_fibre_table, MOO_FIBRES_TABLE_HEALTH,
                                       CPL_EQUAL_TO, 1);
        sky_size = cpl_table_and_selected_int(det_fibre_table,
                                              MOO_FIBRES_TABLE_INDEXRBN,
                                              CPL_NOT_EQUAL_TO, 0);

        sky_sel = cpl_table_where_selected(det_fibre_table);

        cpl_table_select_all(det_fibre_table);
        cpl_table_and_selected_string(det_fibre_table, MOO_FIBRES_TABLE_TYPE,
                                      CPL_EQUAL_TO,
                                      MOO_FIBRES_TABLE_TYPE_CHECKOBJ);
        size =
            cpl_table_and_selected_int(det_fibre_table, MOO_FIBRES_TABLE_HEALTH,
                                       CPL_NOT_EQUAL_TO, 0);
        size = cpl_table_and_selected_int(det_fibre_table,
                                          MOO_FIBRES_TABLE_INDEXRBN,
                                          CPL_NOT_EQUAL_TO, 0);

        obj_sel = cpl_table_where_selected(det_fibre_table);

        int obj_idx = 1;

        sky_stare_tables = cpl_calloc(3, sizeof(cpl_table *));
        for (int t = 0; t < 3; t++) {
            cpl_table *sky_stare_table = cpl_table_new(0);
            cpl_table_new_column(sky_stare_table, MOO_SKY_STARE_INDEXTARG,
                                 CPL_TYPE_INT);
            cpl_table_new_column(sky_stare_table, MOO_SKY_STARE_TARGNAME,
                                 CPL_TYPE_STRING);
            cpl_table_new_column(sky_stare_table, MOO_SKY_STARE_INDEXRBN,
                                 CPL_TYPE_INT);
            cpl_table_new_column(sky_stare_table, MOO_SKY_STARE_SKYINDEXRBN,
                                 CPL_TYPE_INT);
            cpl_table_new_column(sky_stare_table, MOO_SKY_STARE_SKYDSLIT,
                                 CPL_TYPE_INT);
            cpl_table_new_column(sky_stare_table, MOO_SKY_STARE_SKYTRANS,
                                 CPL_TYPE_DOUBLE);
            cpl_table_new_column(sky_stare_table, MOO_SKY_STARE_RADIUS,
                                 CPL_TYPE_DOUBLE);
            cpl_table_new_column(sky_stare_table, MOO_SKY_STARE_SKYDIST,
                                 CPL_TYPE_DOUBLE);
            for (int i = 0; i < size; i++) {
                moo_detector_type type = t;
                int fidx = cpl_array_get_cplsize(obj_sel, i, NULL);
                const char *tname =
                    cpl_table_get_string(det_fibre_table,
                                         MOO_FIBRES_TABLE_TARGNAME, fidx);
                int indexrbn =
                    cpl_table_get_int(det_fibre_table,
                                      MOO_TARGET_TABLE_INDEXRBN, fidx, NULL);
                double obj_targalpha =
                    cpl_table_get_double(det_fibre_table,
                                         MOO_FIBRES_TABLE_TARGALPHA, fidx,
                                         NULL);
                double obj_targdelta =
                    cpl_table_get_double(det_fibre_table,
                                         MOO_FIBRES_TABLE_TARGDELTA, fidx,
                                         NULL);

                cpl_array *sky_indexes =
                    moo_select_sky(tname, obj_idx, indexrbn, obj_targalpha,
                                   obj_targdelta, det_fibre_table,
                                   MOO_TARGET_TABLE_INDEXRBN, sky_stare_table,
                                   type, params->maxdistslit, params->min_trans,
                                   params->min_sky, params->radius_sky,
                                   params->step_r, fidx, f2f, sky_sel);

                obj_idx++;
                cpl_array_delete(sky_indexes);
            }
            sky_stare_tables[t] = sky_stare_table;
        }

        for (int i = 0; i < sky_size; i++) {
            int fidx = cpl_array_get_cplsize(sky_sel, i, NULL);
            cpl_table_set_int(det_fibre_table, MOO_FIBRES_TABLE_HEALTH, fidx,
                              -1);
        }

        for (int i = 0; i < 3; i++) {
            cpl_table *sky_stare_table = sky_stare_tables[i];
            int nrow = cpl_table_get_nrow(sky_stare_table);

            for (int r = 0; r < nrow; r++) {
                int indexrbn =
                    cpl_table_get_int(sky_stare_table,
                                      MOO_SKY_STARE_SKYINDEXRBN, r, NULL);
                cpl_table_select_all(det_fibre_table);
                cpl_table_and_selected_int(det_fibre_table,
                                           MOO_FIBRES_TABLE_INDEXRBN,
                                           CPL_EQUAL_TO, indexrbn);
                cpl_array *detsel = cpl_table_where_selected(det_fibre_table);
                int fidx = cpl_array_get_cplsize(detsel, 0, NULL);
                cpl_table_set_int(det_fibre_table, MOO_FIBRES_TABLE_HEALTH,
                                  fidx, 1);
                cpl_array_delete(detsel);
            }
            cpl_table_delete(sky_stare_table);
        }

        cpl_table_select_all(det_fibre_table);
        size =
            cpl_table_and_selected_int(det_fibre_table, MOO_FIBRES_TABLE_HEALTH,
                                       CPL_EQUAL_TO, -1);
        cpl_array_delete(sel);
        sel = cpl_table_where_selected(det_fibre_table);
        for (int i = 0; i < size; i++) {
            int fidx = cpl_array_get_cplsize(sel, i, NULL);
            cpl_table_set_int(loc_fibre_table, MOO_FIBRES_TABLE_HEALTH, fidx,
                              -1);
        }
    }
    cpl_table_erase_column(det_fibre_table, MOO_TARGET_TABLE_INDEXRBN);

moo_try_cleanup:
    cpl_free(sky_stare_tables);
    cpl_array_delete(sky_sel);
    cpl_array_delete(obj_sel);
    cpl_array_delete(sel);
    moo_f2f_delete(f2f);
    return nb_targets;
}

static cpl_error_code
_moons_quicklook_check_sof(cpl_frameset *frameset,
                           cpl_frame **frame,
                           const char **bpmap_rp_name,
                           const char **bpmap_nl_name,
                           const cpl_frame **masterbias,
                           const cpl_frame **masterdark_vis,
                           const cpl_frame **masterdark_nir,
                           const cpl_frame **fftrace,
                           const cpl_frame **coeffs_cube,
                           const cpl_frame **wmap,
                           const cpl_frame **skylines,
                           const cpl_frame **flat,
                           const cpl_frame **f2f,
                           const cpl_frame **p2pmap,
                           const cpl_frame **sformat,
                           const cpl_frame **atmo,
                           const cpl_frame **resp)
{
    cpl_ensure_code(moo_dfs_set_groups(frameset) == CPL_ERROR_NONE,
                    cpl_error_get_code());
    int i;
    int nb_object = 0;

    for (i = 0; i < cpl_frameset_get_size(frameset); ++i) {
        cpl_frame *current_frame = cpl_frameset_get_position(frameset, i);
        if (!strcmp(cpl_frame_get_tag(current_frame), MOONS_TAG_OBJECT_STARE)) {
            *frame = current_frame;
            nb_object++;
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_OBJECT_STARENOD)) {
            *frame = current_frame;
            nb_object++;
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_OBJECT_XSWITCH)) {
            *frame = current_frame;
            nb_object++;
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_BP_MAP_RP)) {
            *bpmap_rp_name = cpl_frame_get_filename(current_frame);
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_BP_MAP_NL)) {
            *bpmap_nl_name = cpl_frame_get_filename(current_frame);
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_MASTER_BIAS)) {
            *masterbias = current_frame;
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_MASTER_DARK_VIS)) {
            *masterdark_vis = current_frame;
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_MASTER_DARK_NIR)) {
            *masterdark_nir = current_frame;
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame), MOONS_TAG_P2P_MAP)) {
            *p2pmap = current_frame;
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_SPECTRAL_FORMAT)) {
            *sformat = current_frame;
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_ATMOS_EXT)) {
            *atmo = current_frame;
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_RESPONSE)) {
            *resp = current_frame;
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_FF_TRACE)) {
            *fftrace = current_frame;
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_LINEARITY_COEFF_CUBE)) {
            *coeffs_cube = current_frame;
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame), MOONS_TAG_WAVEMAP)) {
            *wmap = current_frame;
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_SKY_LINE_LIST)) {
            *skylines = current_frame;
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_FF_EXTSPECTRA)) {
            *flat = current_frame;
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_F2F_TABLE)) {
            *f2f = current_frame;
        }
    }

    if (*sformat == NULL) {
        return (int)cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                          "SOF does not have any file tagged "
                                          "with %s",
                                          MOONS_TAG_SPECTRAL_FORMAT);
    }

    if (nb_object != 1) {
        return (int)cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                          "SOF does exactly one file tagged "
                                          "with %s or %s or %s",
                                          MOONS_TAG_OBJECT_STARE,
                                          MOONS_TAG_OBJECT_STARENOD,
                                          MOONS_TAG_OBJECT_XSWITCH);
    }

    if (*fftrace == NULL) {
        return (int)cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                          "SOF does not have any file tagged "
                                          "with %s",
                                          MOONS_TAG_FF_TRACE);
    }

    if (*wmap == NULL) {
        return (int)cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                          "SOF does not have any file tagged "
                                          "with %s",
                                          MOONS_TAG_WAVEMAP);
    }

    if (*skylines == NULL) {
        return (int)cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                          "SOF does not have any file tagged "
                                          "with %s",
                                          MOONS_TAG_SKY_LINE_LIST);
    }
    if (*flat == NULL) {
        return (int)cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                          "SOF does not have any file tagged "
                                          "with %s",
                                          MOONS_TAG_FF_EXTSPECTRA);
    }
    return CPL_ERROR_NONE;
}

static cpl_error_code
_moons_p2pmap(moo_det *arc, const cpl_frame *p2pmap)
{
    cpl_error_code status = CPL_ERROR_NONE;
    moo_det *p2p = NULL;

    if (p2pmap != NULL) {
        cpl_msg_info("test", "apply_p2P");
        moo_try_check(p2p = moo_det_create(p2pmap), " ");
        moo_try_check(moo_apply_p2p(arc, p2p), " ");
    }
moo_try_cleanup:
    moo_det_delete(p2p);
    return status;
}

static cpl_error_code
_moons_apply_flat(moo_ext *ext,
                  const cpl_frame *flat_frame,
                  const cpl_frame *f2f_frame)
{
    cpl_error_code status = CPL_ERROR_NONE;
    moo_ext *flat = NULL;
    moo_f2f *f2f = NULL;

    moo_try_check(flat = moo_ext_create(flat_frame), " ");
    if (f2f_frame != NULL) {
        f2f = moo_f2f_load(f2f_frame);
    }
    moo_try_check(moo_apply_flat(ext, flat, f2f), " ");

moo_try_cleanup:
    moo_ext_delete(flat);
    moo_f2f_delete(f2f);
    return status;
}

static moo_sci *
_moons_sub_sky_stare(moo_rbn *rbn,
                     moo_target_table *target_table,
                     const cpl_frame *f2f_frame,
                     const cpl_frame *solflux_frame,
                     const cpl_frame *airglow_group_frame,
                     const cpl_frame *airglow_var_frame,
                     moo_sub_sky_stare_params *sky_params,
                     const char *sci_filename)
{
    moo_sci *result = NULL;
    moo_f2f *f2f = NULL;

    if (f2f_frame != NULL) {
        f2f = moo_f2f_load(f2f_frame);
    }
    moo_try_check(result =
                      moo_sub_sky_stare(rbn, target_table, rbn, f2f,
                                        solflux_frame, airglow_group_frame,
                                        airglow_var_frame, sky_params,
                                        sci_filename, MOO_SCI1D_NOT_PAIRED),
                  " ");

moo_try_cleanup:
    moo_f2f_delete(f2f);
    return result;
}

static moo_sci *
_moons_quicklook_notarget_sci(moo_det *det, const char *filename)
{
    moo_sci *res = NULL;

    res = moo_sci_new();
    res->filename = filename;
    res->primary_header = cpl_propertylist_duplicate(det->primary_header);
    moo_target_table *ttable = moo_create_empty_target_table();

    res->target_table = ttable;
    moo_sci_save(res, filename);
    return res;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Interpret the command line options and execute the data processing
  @param    frameset   the frames list
  @param    parlist    the parameters list
  @return   0 if everything is ok  @return   0 if everything is ok
 */
/*----------------------------------------------------------------------------*/

static int
_moons_quicklook(cpl_frameset *frameset, const cpl_parameterlist *parlist)
{
    /* parameters */
    moo_prepare_params *prepare_params = NULL;
    moo_correct_bias_params *correct_bias_params = NULL;
    moo_extract_params *extract_params = NULL;
    moo_rebin_params *rbn_params = NULL;
    moo_target_table_params *target_table_params = NULL;
    moo_sub_sky_stare_params *sky_params = NULL;
    cpl_array *index_list = NULL;
    moo_loc *loc = NULL;
    moo_spectral_format *sformat = NULL;
    moo_map *wmap = NULL;
    moo_sky_lines_list *skylines = NULL;
    moo_det *det = NULL;
    moo_ext *ext = NULL;
    moo_rbn *rbn = NULL;
    moo_sci *sci = NULL;
    moo_target_table *target_table = NULL;

    moo_atm *atm = NULL;
    moo_resp *resp = NULL;

    cpl_frame *frame = NULL;

    const char *bpmap_rp_name = NULL;
    const char *bpmap_nl_name = NULL;
    const cpl_frame *masterbias = NULL;
    const cpl_frame *masterdark_vis = NULL;
    const cpl_frame *masterdark_nir = NULL;
    const cpl_frame *p2pmap = NULL;
    const cpl_frame *sformat_frame = NULL;
    const cpl_frame *atmo_frame = NULL;
    const cpl_frame *resp_frame = NULL;
    const cpl_frame *fftrace = NULL;
    const cpl_frame *coeffs_cube = NULL;
    const cpl_frame *wmap_frame = NULL;
    const cpl_frame *skylines_frame = NULL;
    const cpl_frame *flat_frame = NULL;
    const cpl_frame *f2f_frame = NULL;
    const char *ref_filename = NULL;
    cpl_propertylist *ref_header = NULL;

    moo_mode_type mode;
    const char *mode_name = NULL;
    int offset;
    int nb_targets = 0;
    char *extname = NULL;
    char *rbnname = NULL;
    char *sciname = NULL;

    cpl_errorstate prestate = cpl_errorstate_get();

    moo_products *products =
        moo_products_new(frameset, parlist, "moons_quicklook",
                         PACKAGE "/" PACKAGE_VERSION);

    const moo_params *params = moo_products_get_params(products);
    moo_try_check(prepare_params = moo_params_get_prepare(params, parlist),
                  " ");
    moo_try_check(correct_bias_params =
                      moo_params_get_correct_bias(params, parlist),
                  " ");
    extract_params = moo_extract_params_new();
    extract_params->method = MOO_EXTRACT_METHOD_SUM;
    moo_try_check(rbn_params = moo_params_get_rebin(params, parlist), " ");
    target_table_params = moo_target_table_params_new();
    target_table_params->nosky = CPL_TRUE;
    moo_try_check(sky_params =
                      moo_params_get_sub_sky_stare_simple(params, parlist),
                  " ");
    moo_try_check(index_list =
                      moo_params_get_quicklook_fibre_list(params, parlist),
                  " ");
    moo_try_check(_moons_quicklook_check_sof(frameset, &frame, &bpmap_rp_name,
                                             &bpmap_nl_name, &masterbias,
                                             &masterdark_vis, &masterdark_nir,
                                             &fftrace, &coeffs_cube,
                                             &wmap_frame, &skylines_frame,
                                             &flat_frame, &f2f_frame, &p2pmap,
                                             &sformat_frame, &atmo_frame,
                                             &resp_frame),
                  " ");

    ref_filename = cpl_frame_get_filename(frame);
    moo_try_check(mode = moo_mode_get(frame), " ");
    mode_name = moo_mode_get_name(mode);
    moo_try_check(ref_header = cpl_propertylist_load(ref_filename, 0), " ");
    moo_try_check(offset = moo_pfits_get_slit_offset(ref_header), " ");
    cpl_propertylist_delete(ref_header);
    ref_header = NULL;
    cpl_msg_info(__func__, "Do offset %d Use mode %s", offset, mode_name);

    moo_try_check(sformat = moo_spectral_format_load(sformat_frame, mode), " ");
    moo_try_check(wmap = moo_map_load(wmap_frame), " ");
    moo_try_check(loc = moo_loc_load(fftrace), " ");

    moo_try_check(det = moo_prepare(frame, bpmap_rp_name, bpmap_nl_name,
                                    masterbias, NULL, prepare_params),
                  " ");
    moo_try_check(moo_products_add(products, det, CPL_FRAME_LEVEL_INTERMEDIATE,
                                   MOONS_TAG_QUICKLOOK_PREPARE, NULL, frame),
                  " ");

    moo_try_check(nb_targets =
                      _moons_select_quicklook_targets(det, loc, f2f_frame,
                                                      sky_params, index_list),
                  " ");

    if (nb_targets == 0) {
        cpl_msg_info("test", "No target");
        sci = _moons_quicklook_notarget_sci(det, "QLOOK_SCI.fits");
    }
    else {
        moo_try_check(moo_correct_bias(det, masterbias, correct_bias_params),
                      " ");
        moo_try_check(moo_correct_dark(det, NULL, masterdark_vis,
                                       masterdark_nir),
                      " ");
        moo_try_check(moo_products_add(products, det,
                                       CPL_FRAME_LEVEL_INTERMEDIATE,
                                       MOONS_TAG_QUICKLOOK_CORRECTDARK, NULL,
                                       frame),
                      " ");

        moo_try_check(_moons_p2pmap(det, p2pmap), " ");

        moo_try_check(moo_products_add(products, det,
                                       CPL_FRAME_LEVEL_INTERMEDIATE,
                                       MOONS_TAG_QUICKLOOK_APPLYP2P, NULL,
                                       frame),
                      " ");
        extname =
            cpl_sprintf("%s.fits", MOONS_TAG_SCIENCE_QUICKLOOK_EXTSPECTRA);
        moo_try_check(ext =
                          moo_extract(det, loc, NULL, extract_params, extname),
                      " ");
        moo_try_check(moo_products_add_ext(
                          products, ext, CPL_FRAME_LEVEL_INTERMEDIATE,
                          MOONS_TAG_SCIENCE_QUICKLOOK_EXTSPECTRA, NULL, frame),
                      " ");

        moo_try_check(_moons_apply_flat(ext, flat_frame, f2f_frame), " ");
        moo_try_check(skylines = moo_sky_lines_list_load(skylines_frame), " ");
        moo_try_check(moo_ext_compute_snr(ext, wmap, sformat, skylines), " ");
        moo_try_check(
            moo_products_add_ext(products, ext, CPL_FRAME_LEVEL_INTERMEDIATE,
                                 MOONS_TAG_SCIENCE_QUICKLOOK_FFEXTSPECTRA, NULL,
                                 frame),
            " ");
        rbnname =
            cpl_sprintf("%s.fits", MOONS_TAG_SCIENCE_QUICKLOOK_RBNSPECTRA);
        moo_try_check(rbn = moo_rebin(ext, wmap, sformat, rbn_params, rbnname),
                      " ");
        moo_try_check(moo_rbn_compute_snr(rbn, skylines), " ");
        moo_try_check(moo_products_add_rbn(
                          products, rbn, CPL_FRAME_LEVEL_INTERMEDIATE,
                          MOONS_TAG_SCIENCE_QUICKLOOK_RBNSPECTRA, NULL, frame),
                      " ");

        moo_try_check(target_table =
                          moo_create_target_table(rbn, NULL, MOO_MODE_STARE,
                                                  target_table_params),
                      " ");

        moo_try_check(moo_products_add_target_table(
                          products, target_table, CPL_FRAME_LEVEL_INTERMEDIATE,
                          MOONS_TAG_SCIENCE_QUICKLOOK_TARGET_TABLE, NULL,
                          frame),
                      " ");

        sciname =
            cpl_sprintf("%s.fits", MOONS_TAG_SCIENCE_QUICKLOOK_SKSSPECTRA);
        moo_try_check(sci = _moons_sub_sky_stare(rbn, target_table, f2f_frame,
                                                 NULL, NULL, NULL, sky_params,
                                                 sciname),
                      " ");
        moo_try_check(moo_sci_compute_snr(sci, skylines), " ");
    }
    moo_try_check(moo_products_add_sci(products, sci, CPL_FRAME_LEVEL_FINAL,
                                       MOONS_TAG_SCIENCE_QUICKLOOK_SKSSPECTRA,
                                       NULL, frame),
                  " ");
moo_try_cleanup:
    if (!cpl_errorstate_is_equal(prestate)) {
        cpl_propertylist_delete(ref_header);
    }

    moo_ext_delete(ext);
    moo_det_delete(det);

    moo_atm_delete(atm);
    moo_resp_delete(resp);
    moo_loc_delete(loc);
    moo_sky_lines_list_delete(skylines);
    moo_map_delete(wmap);
    moo_spectral_format_delete(sformat);
    moo_target_table_delete(target_table);
    moo_sci_delete(sci);
    moo_rbn_delete(rbn);
    moo_target_table_params_delete(target_table_params);
    moo_rebin_params_delete(rbn_params);
    moo_sub_sky_stare_params_delete(sky_params);
    moo_extract_params_delete(extract_params);
    moo_correct_bias_params_delete(correct_bias_params);
    moo_prepare_params_delete(prepare_params);
    cpl_array_delete(index_list);
    moo_products_delete(products);
    cpl_free(extname);
    cpl_free(rbnname);
    cpl_free(sciname);
    return (int)cpl_error_get_code();
}
