/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#include "moo_utils.h"
#include "moo_pfits.h"
#include "moo_dfs.h"
#include "moo_params.h"
#include "moo_drl.h"
#include "moo_products.h"
#include "moo_correct_tell.h"
#include "moo_molecfit.h"
#include "moo_molectable.h"
#include <cpl.h>
#include <string.h>

/*-----------------------------------------------------------------------------
                              Plugin registration
 -----------------------------------------------------------------------------*/

int cpl_plugin_get_info(cpl_pluginlist *list);

/*-----------------------------------------------------------------------------
                            Private function prototypes
 -----------------------------------------------------------------------------*/

static int _moons_standard_create(cpl_plugin *plugin);
static int _moons_standard_exec(cpl_plugin *plugin);
static int _moons_standard_destroy(cpl_plugin *plugin);
static int
_moons_standard(cpl_frameset *frameset, const cpl_parameterlist *parlist);

/*-----------------------------------------------------------------------------
                            Static variables
 -----------------------------------------------------------------------------*/

static const char *const _moons_standard_description =
    "Produces the instrumental response for relative flux calibration, and "
    "the telluric absorption spectrum.\n"
    "INPUT FRAMES\n"
    "  * flux or telluric standard star n>=3 file(s) (RAW) with tag STD_FLUX "
    "or STD_TELL : "
    "standard star file\n"
    "  * [OPTIONAL] ReferenceBadPixMask 1 file       (QUA) with tag BP_MAP_RP "
    ": "
    "cosmetic bad pixel map\n"
    "  * [OPTIONAL] NonLinearityBadPixMask 1 file    (QUA) with tag BP_MAP_NL "
    ": "
    "cosmetic bad pixel map coming from linearity recipe\n"
    "  * MasterBias 1 file                           (DET) with tag "
    "MASTER_BIAS : "
    "master bias file\n"
    "  * MasterDarkNir 1 file                        (DET) with tag "
    "MASTER_DARK_NIR : "
    "master dark nir file\n"
    "  * [OPTIONAL]MasterDarkVis 1 file              (DET) with tag "
    "MASTER_DARK_VIS : "
    "master dark vis file\n"
    "  * [OPTIONAL] CoeffsCube 1 file                (3D) with tag "
    "LINEARITY_COEFF_CUBE : "
    "coefficients to correct pixels detector linearity\n"
    "  * [OPTIONAL] P2pMap 1 file                    (DET) with tag P2P_MAP : "
    "pixel to pixel map\n"
    "  * LocTab 1 file                               (LOC) with tag FF_TRACE : "
    "the localisation table\n"
    "  * Ffext 1 file                                (EXT) with tag "
    "FF_EXTSPECTRA : "
    "the extracted flat field\n"
    "  * F2f 1 file                                  (F2F) with tag F2F_TABLE "
    ": "
    "the fibre-to-fibre relative response\n"
    "  * SFormat 1 file                              (FMT) with tag "
    "SPECTRAL_FORMAT : "
    "the spectral format table\n"
    "  * [OPTIONAL] MasterFlat 1 file                (PSF) with tag "
    "MASTER_FLAT : "
    "the master flat used for optimal extraction\n"
    "  * WaveMap 1 file                              (WMAP) with tag WAVE_MAP "
    ": "
    "the wavelength map\n"
    "  * fluxStdCat 1 file                           (FLX) with tag "
    "FLUX_STD_CAT : "
    "the standard star flux table\n"
    "  * atmosExt 1 file                             (ATMO) with tag ATMOS_EXT "
    ": "
    "the atmospheric extinction table\n"
    "  * SkyLines 1 file                             (CAT) with tag "
    "SKY_LINE_LIST : "
    "the sky lines list table\n"
    "  * [STDTELL only] file (MOLECULES) with tag MOLECULES 1 file : "
    "molecules list file\n"
    "PRODUCTS\n"
    "  * STD_EXTSPECTRA_i_[insmode].fits (EXT) with tag STD_EXTSPECTRA : "
    "Science product individual extracted frames\n"
    "  * STD_RBNSPECTRA_i_[insmode].fits (RBN) with tag STD_RBNSPECTRA : "
    "Science product individual calibrated frames\n"
    "  * STD_SKSSPECTRA_[insmode]_i.fits (SCI) with tag STD_SKSSPECTRA : "
    "Science product individual sky subtracted frames\n"
    "  * STD_SKSSPECTRA_[insmode].fits   (SCI) with tag STD_SKSSPECTRA : "
    "Science product flux calibrated frame\n"
    "  * RESPONSE_[insmode].fits         (RESP) with tag RESPONSE : "
    "Instrumental response product\n"
    "  * TELLURIC_[insmode].fits         (TELLURIC) with tag TELLURIC : "
    "Telluric product\n"
    "  * GDAS User GDAS profile"
    "  * GDAS_BEFORE If ESO DB GDAS is used, file before the MJD-OBS"
    "  * GDAS_AFTER If ESO DB GDAS is used, file after the MJD-OBS"
    "\n";

/*-----------------------------------------------------------------------------
                                Function code
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Build the list of available plugins, for this module.
  @param    list    the plugin list
  @return   0 if everything is ok, 1 otherwise
  @note     Only this function is exported

  Create the recipe instance and make it available to the application using the
  interface.
 */
/*----------------------------------------------------------------------------*/

int
cpl_plugin_get_info(cpl_pluginlist *list)
{
    cpl_recipe *recipe = cpl_calloc(1, sizeof *recipe);
    cpl_plugin *plugin = &recipe->interface;

    if (cpl_plugin_init(plugin, CPL_PLUGIN_API, MOONS_BINARY_VERSION,
                        CPL_PLUGIN_TYPE_RECIPE, "moons_standard",
                        "Produces the instrumental response for relative flux "
                        "calibration, and the telluric absorption spectrum.",
                        _moons_standard_description, "Regis Haigron",
                        PACKAGE_BUGREPORT, moo_get_license(),
                        _moons_standard_create, _moons_standard_exec,
                        _moons_standard_destroy)) {
        cpl_msg_error(cpl_func, "Plugin initialization failed");
        (void)cpl_error_set_where(cpl_func);
        return 1;
    }

    if (cpl_pluginlist_append(list, plugin)) {
        cpl_msg_error(cpl_func, "Error adding plugin to list");
        (void)cpl_error_set_where(cpl_func);
        return 1;
    }

    return 0;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Setup the recipe options
  @param    plugin  the plugin
  @return   0 if everything is ok

  Defining the command-line/configuration parameters for the recipe.
 */
/*----------------------------------------------------------------------------*/

static int
_moons_standard_create(cpl_plugin *plugin)
{
    cpl_recipe *recipe;

    /* Do not create the recipe if an error code is already set */
    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        cpl_msg_error(cpl_func, "%s():%d: An error is already set: %s",
                      cpl_func, __LINE__, cpl_error_get_where());
        return (int)cpl_error_get_code();
    }

    if (plugin == NULL) {
        cpl_msg_error(cpl_func, "Null plugin");
        cpl_ensure_code(0, (int)CPL_ERROR_NULL_INPUT);
    }

    /* Verify plugin type */
    if (cpl_plugin_get_type(plugin) != CPL_PLUGIN_TYPE_RECIPE) {
        cpl_msg_error(cpl_func, "Plugin is not a recipe");
        cpl_ensure_code(0, (int)CPL_ERROR_TYPE_MISMATCH);
    }

    /* Get the recipe */
    recipe = (cpl_recipe *)plugin;

    /* Create the parameters list in the cpl_recipe object */
    recipe->parameters = cpl_parameterlist_new();
    if (recipe->parameters == NULL) {
        cpl_msg_error(cpl_func, "Parameter list allocation failed");
        cpl_ensure_code(0, (int)CPL_ERROR_ILLEGAL_OUTPUT);
    }

    moo_params *params = moo_params_new("moons", "moons_standard");

    /* Fill the parameters list */
    moo_params_add_keep_temp(params, recipe->parameters);
    moo_params_add_prepare(params, recipe->parameters);
    moo_params_add_correct_bias(params, recipe->parameters,
                                MOO_CORRECT_BIAS_METHOD_MASTER);
    moo_params_add_crh(params, recipe->parameters, MOO_CRH_METHOD_MEDIAN);
    moo_params_add_extract(params, recipe->parameters);
    moo_params_add_rebin(params, recipe->parameters);
    moo_params_add_target_table(params, recipe->parameters, CPL_TRUE);
    moo_params_add_sub_sky_stare(params, recipe->parameters);
    moo_params_add_compute_resp(params, recipe->parameters);
    moo_params_add_coadd(params, recipe->parameters);
    moo_params_add_molecfit_model(params, recipe->parameters);
    moo_params_delete(params);

    return 0;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Execute the plugin instance given by the interface
  @param    plugin  the plugin
  @return   0 if everything is ok
 */
/*----------------------------------------------------------------------------*/

static int
_moons_standard_exec(cpl_plugin *plugin)
{
    cpl_recipe *recipe;
    int recipe_status;
    cpl_errorstate initial_errorstate = cpl_errorstate_get();

    /* Return immediately if an error code is already set */
    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        cpl_msg_error(cpl_func, "%s():%d: An error is already set: %s",
                      cpl_func, __LINE__, cpl_error_get_where());
        return (int)cpl_error_get_code();
    }

    if (plugin == NULL) {
        cpl_msg_error(cpl_func, "Null plugin");
        cpl_ensure_code(0, (int)CPL_ERROR_NULL_INPUT);
    }

    /* Verify plugin type */
    if (cpl_plugin_get_type(plugin) != CPL_PLUGIN_TYPE_RECIPE) {
        cpl_msg_error(cpl_func, "Plugin is not a recipe");
        cpl_ensure_code(0, (int)CPL_ERROR_TYPE_MISMATCH);
    }

    /* Get the recipe */
    recipe = (cpl_recipe *)plugin;

    /* Verify parameter and frame lists */
    if (recipe->parameters == NULL) {
        cpl_msg_error(cpl_func, "Recipe invoked with NULL parameter list");
        cpl_ensure_code(0, (int)CPL_ERROR_NULL_INPUT);
    }
    if (recipe->frames == NULL) {
        cpl_msg_error(cpl_func, "Recipe invoked with NULL frame set");
        cpl_ensure_code(0, (int)CPL_ERROR_NULL_INPUT);
    }

    /* Invoke the recipe */
    recipe_status = _moons_standard(recipe->frames, recipe->parameters);

    /* Ensure DFS-compliance of the products */
    if (cpl_dfs_update_product_header(recipe->frames)) {
        if (!recipe_status)
            recipe_status = (int)cpl_error_get_code();
    }

    if (!cpl_errorstate_is_equal(initial_errorstate)) {
        /* Dump the error history since recipe execution start.
           At this point the recipe cannot recover from the error */
        cpl_errorstate_dump(initial_errorstate, CPL_FALSE, NULL);
    }

    return recipe_status;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Destroy what has been created by the 'create' function
  @param    plugin  the plugin
  @return   0 if everything is ok
 */
/*----------------------------------------------------------------------------*/

static int
_moons_standard_destroy(cpl_plugin *plugin)
{
    cpl_recipe *recipe;

    if (plugin == NULL) {
        cpl_msg_error(cpl_func, "Null plugin");
        cpl_ensure_code(0, (int)CPL_ERROR_NULL_INPUT);
    }

    /* Verify plugin type */
    if (cpl_plugin_get_type(plugin) != CPL_PLUGIN_TYPE_RECIPE) {
        cpl_msg_error(cpl_func, "Plugin is not a recipe");
        cpl_ensure_code(0, (int)CPL_ERROR_TYPE_MISMATCH);
    }

    /* Get the recipe */
    recipe = (cpl_recipe *)plugin;

    cpl_parameterlist_delete(recipe->parameters);

    return 0;
}

static cpl_error_code
_moons_standard_check_sof(cpl_frameset *frameset,
                          cpl_frameset *std_frameset,
                          const char **bpmap_rp_name,
                          const char **bpmap_nl_name,
                          const cpl_frame **masterbias,
                          const cpl_frame **masterdark_vis,
                          const cpl_frame **masterdark_nir,
                          const cpl_frame **coeffs_cube,
                          const cpl_frame **p2pmap,
                          const cpl_frame **fftrace,
                          const cpl_frame **master_flat,
                          const cpl_frame **sformat,
                          const cpl_frame **wmap,
                          const cpl_frame **skylines,
                          const cpl_frame **flat,
                          const cpl_frame **tell,
                          const cpl_frame **f2f,
                          const cpl_frame **flx,
                          const cpl_frame **atmo,
                          const cpl_frame **molecules_frame,
                          const cpl_frame **winc_frame,
                          const moo_extract_params *ext_params,
                          int *mode)
{
    cpl_ensure_code(moo_dfs_set_groups(frameset) == CPL_ERROR_NONE,
                    cpl_error_get_code());
    cpl_ensure_code(mode != NULL, CPL_ERROR_NULL_INPUT);

    int nb_std_flux = 0;
    int nb_std_tell = 0;

    int i;
    for (i = 0; i < cpl_frameset_get_size(frameset); ++i) {
        cpl_frame *current_frame = cpl_frameset_get_position(frameset, i);
        if (!strcmp(cpl_frame_get_tag(current_frame), MOONS_TAG_STD_FLUX)) {
            cpl_frameset_insert(std_frameset,
                                cpl_frame_duplicate(current_frame));
            nb_std_flux++;
            *mode = 0;
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_STD_TELL)) {
            cpl_frameset_insert(std_frameset,
                                cpl_frame_duplicate(current_frame));
            nb_std_tell++;
            *mode = 1;
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_BP_MAP_RP)) {
            *bpmap_rp_name = cpl_frame_get_filename(current_frame);
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_BP_MAP_NL)) {
            *bpmap_nl_name = cpl_frame_get_filename(current_frame);
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_MASTER_BIAS)) {
            *masterbias = current_frame;
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_MASTER_DARK_VIS)) {
            *masterdark_vis = current_frame;
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_MASTER_DARK_NIR)) {
            *masterdark_nir = current_frame;
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_LINEARITY_COEFF_CUBE)) {
            *coeffs_cube = current_frame;
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame), MOONS_TAG_P2P_MAP)) {
            *p2pmap = current_frame;
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_FF_TRACE)) {
            *fftrace = current_frame;
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_MASTER_FLAT)) {
            *master_flat = current_frame;
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_SPECTRAL_FORMAT)) {
            *sformat = current_frame;
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame), MOONS_TAG_WAVEMAP)) {
            *wmap = current_frame;
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_SKY_LINE_LIST)) {
            *skylines = current_frame;
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_FF_EXTSPECTRA)) {
            *flat = current_frame;
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_F2F_TABLE)) {
            *f2f = current_frame;
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_FLUX_STD_CATALOG)) {
            *flx = current_frame;
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_ATMOS_EXT)) {
            *atmo = current_frame;
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_TELLURIC_CORR)) {
            *tell = current_frame;
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_MOLECFIT_MOLECULES)) {
            *molecules_frame = current_frame;
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_MOLECFIT_WINCLUDE)) {
            *winc_frame = current_frame;
        }
    }


    if (cpl_frameset_get_size(std_frameset) == 0) {
        return (int)cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                          "SOF does not have any file tagged "
                                          "with %s or %s",
                                          MOONS_TAG_STD_FLUX,
                                          MOONS_TAG_STD_TELL);
    }
    if (nb_std_flux > 0 && nb_std_tell > 0) {
        return (int)cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                          "SOF have mixed STD input files"
                                          "%d with %s and  %d with %s",
                                          nb_std_flux, MOONS_TAG_STD_FLUX,
                                          nb_std_tell, MOONS_TAG_STD_TELL);
    }
    if (*fftrace == NULL) {
        return (int)cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                          "SOF does not have any file tagged "
                                          "with %s",
                                          MOONS_TAG_FF_TRACE);
    }

    if (*f2f == NULL) {
        return (int)cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                          "SOF does not have any file tagged "
                                          "with %s",
                                          MOONS_TAG_F2F_TABLE);
    }

    if (*master_flat == NULL &&
        strcmp(ext_params->method, MOO_EXTRACT_METHOD_OPTIMAL) == 0) {
        return (int)cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                          "SOF does not have any file tagged "
                                          "with %s",
                                          MOONS_TAG_MASTER_FLAT);
    }

    if (*sformat == NULL) {
        return (int)cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                          "SOF does not have any file tagged "
                                          "with %s",
                                          MOONS_TAG_SPECTRAL_FORMAT);
    }

    if (*wmap == NULL) {
        return (int)cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                          "SOF does not have any file tagged "
                                          "with %s",
                                          MOONS_TAG_WAVEMAP);
    }

    if (*skylines == NULL) {
        return (int)cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                          "SOF does not have any file tagged "
                                          "with %s",
                                          MOONS_TAG_SKY_LINE_LIST);
    }

    if (*flat == NULL) {
        return (int)cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                          "SOF does not have any file tagged "
                                          "with %s",
                                          MOONS_TAG_FF_EXTSPECTRA);
    }

    if (nb_std_flux > 0 && *flx == NULL) {
        return (int)cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                          "SOF does not have any file tagged "
                                          "with %s",
                                          MOONS_TAG_FLUX_STD_CATALOG);
    }
    if (nb_std_flux > 0 && *atmo == NULL) {
        return (int)cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                          "SOF does not have any file tagged "
                                          "with %s",
                                          MOONS_TAG_ATMOS_EXT);
    }
    if (nb_std_tell > 0) {
        *tell = NULL;
        if (*molecules_frame == NULL) {
            return (
                int)cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                          "SOF does not have any file tagged "
                                          "with %s",
                                          MOONS_TAG_MOLECFIT_MOLECULES);
        }

        if (*winc_frame == NULL) {
            return (
                int)cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                          "SOF does not have any file tagged "
                                          "with %s",
                                          MOONS_TAG_MOLECFIT_WINCLUDE);
        }
    }
    return CPL_ERROR_NONE;
}


static cpl_error_code
_moons_p2pmap(moo_det *arc, const cpl_frame *p2pmap)
{
    cpl_error_code status = CPL_ERROR_NONE;
    moo_det *p2p = NULL;

    if (p2pmap != NULL) {
        moo_try_check(p2p = moo_det_create(p2pmap), " ");
        moo_try_check(moo_apply_p2p(arc, p2p), " ");
    }
moo_try_cleanup:
    moo_det_delete(p2p);
    return status;
}

static cpl_error_code
_moons_apply_flat(moo_ext *ext,
                  const cpl_frame *flat_frame,
                  const cpl_frame *f2f_frame)
{
    cpl_error_code status = CPL_ERROR_NONE;
    moo_ext *flat = NULL;
    moo_f2f *f2f = NULL;

    moo_try_check(flat = moo_ext_create(flat_frame), " ");
    if (f2f_frame != NULL) {
        f2f = moo_f2f_load(f2f_frame);
    }
    moo_try_check(moo_apply_flat(ext, flat, f2f), " ");

moo_try_cleanup:
    moo_ext_delete(flat);
    moo_f2f_delete(f2f);
    return status;
}

static moo_sci *
_moons_sub_sky_stare(moo_rbn *rbn,
                     moo_target_table *target_table,
                     const cpl_frame *f2f_frame,
                     moo_sub_sky_stare_params *sky_params,
                     const char *sci_filename)
{
    moo_sci *result = NULL;
    moo_f2f *f2f = NULL;

    if (f2f_frame != NULL) {
        f2f = moo_f2f_load(f2f_frame);
    }
    moo_try_check(result =
                      moo_sub_sky_stare(rbn, target_table, rbn, f2f, NULL, NULL,
                                        NULL, sky_params, sci_filename,
                                        MOO_SCI1D_NOT_PAIRED),
                  " ");

moo_try_cleanup:
    moo_f2f_delete(f2f);
    return result;
}

static cpl_frame *
_moons_standard_frame(int inum,
                      const cpl_frame *frame,
                      const cpl_frame *refframe,
                      const cpl_frame *p2pmap,
                      const cpl_frame *fftrace,
                      const cpl_frame *master_flat,
                      moo_spectral_format *sformat,
                      moo_map *wmap,
                      moo_sky_lines_list *skylines,
                      const cpl_frame *flat_frame,
                      const cpl_frame *f2f_frame,
                      const cpl_frame *tell_frame,
                      moo_extract_params *extract_params,
                      moo_rebin_params *rbn_params,
                      moo_target_table_params *target_table_params,
                      moo_sub_sky_stare_params *sky_params,
                      moo_mode_type mode,
                      moo_products *products,
                      int rmode,
                      int offset)
{
    const char *stdp2p_tags[] = { MOONS_TAG_STDFLUX_APPLYP2P,
                                  MOONS_TAG_STDTELL_APPLYP2P };

    const char *stdext_tags[] = { MOONS_TAG_STDFLUX_EXTSPECTRA,
                                  MOONS_TAG_STDTELL_EXTSPECTRA };

    const char *stdffext_tags[] = { MOONS_TAG_STDFLUX_FFEXTSPECTRA,
                                    MOONS_TAG_STDTELL_FFEXTSPECTRA };

    const char *stdrbn_tags[] = { MOONS_TAG_STDFLUX_RBNSPECTRA,
                                  MOONS_TAG_STDTELL_RBNSPECTRA };
    const char *stdttable_tags[] = { MOONS_TAG_STDFLUX_TARGET_TABLE,
                                     MOONS_TAG_STDTELL_TARGET_TABLE };

    const char *stdsks_tags[] = { MOONS_TAG_STDFLUX_SKSSPECTRA,
                                  MOONS_TAG_STDTELL_SKSSPECTRA };

    cpl_frame *sci_frame = NULL;
    moo_det *det = NULL;
    moo_loc *loc = NULL;
    moo_psf *psf = NULL;
    moo_ext *ext = NULL;
    moo_rbn *rbn = NULL;
    moo_telluric *telluric = NULL;
    moo_sci *sci = NULL;

    char *det_filename = NULL;
    char *ext_filename = NULL;
    char *ffext_filename = NULL;
    char *rbn_filename = NULL;
    char *tellcorr_filename = NULL;
    char *ttable_filename = NULL;
    char *sci_filename = NULL;

    moo_target_table *target_table = NULL;
    cpl_errorstate prestate = cpl_errorstate_get();

    moo_try_check(det = moo_det_create(frame), " ");
    moo_try_check(_moons_p2pmap(det, p2pmap), " ");

    det_filename = cpl_sprintf("%s_OFFSET%d_%s_%d.fits", stdp2p_tags[rmode],
                               offset, moo_mode_get_name(mode), inum);
    moo_try_check(moo_products_add(products, det, CPL_FRAME_LEVEL_INTERMEDIATE,
                                   stdp2p_tags[rmode], det_filename, refframe),
                  " ");
    moo_try_check(loc = moo_loc_load(fftrace), " ");
    if (master_flat != NULL) {
        psf = moo_psf_load(master_flat);
    }
    ext_filename = cpl_sprintf("%s_OFFSET%d_%s_%d.fits", stdext_tags[rmode],
                               offset, moo_mode_get_name(mode), inum);
    moo_try_check(ext =
                      moo_extract(det, loc, psf, extract_params, ext_filename),
                  " ");
    moo_try_check(moo_products_add_ext(products, ext,
                                       CPL_FRAME_LEVEL_INTERMEDIATE,
                                       stdext_tags[rmode], ext_filename,
                                       refframe),
                  " ");

    moo_try_check(_moons_apply_flat(ext, flat_frame, f2f_frame), " ");
    ffext_filename = cpl_sprintf("%s_OFFSET%d_%s_%d.fits", stdffext_tags[rmode],
                                 offset, moo_mode_get_name(mode), inum);

    moo_try_check(moo_ext_compute_snr(ext, wmap, sformat, skylines), " ");
    moo_try_check(moo_products_add_ext(products, ext, CPL_FRAME_LEVEL_FINAL,
                                       stdffext_tags[rmode], ffext_filename,
                                       refframe),
                  " ");

    rbn_filename = cpl_sprintf("%s_OFFSET%d_%s_%d.fits", stdrbn_tags[rmode],
                               offset, moo_mode_get_name(mode), inum);
    moo_try_check(rbn = moo_rebin(ext, wmap, sformat, rbn_params, rbn_filename),
                  " ");

    moo_try_check(moo_rbn_compute_snr(rbn, skylines), " ");

    moo_try_check(moo_products_add_rbn(products, rbn, CPL_FRAME_LEVEL_FINAL,
                                       stdrbn_tags[rmode], rbn_filename,
                                       refframe),
                  " ");
    if (tell_frame != NULL) {
        moo_try_check(telluric = moo_telluric_load(tell_frame), " ");

        tellcorr_filename = cpl_sprintf("%s_OFFSET%d_%s_%d.fits",
                                        MOONS_TAG_STDFLUX_RBNTELLCORRSPECTRA,
                                        offset, moo_mode_get_name(mode), inum);
        moo_try_check(moo_correct_tell(rbn, telluric), " ");
        moo_try_check(moo_products_add_rbn(products, rbn, CPL_FRAME_LEVEL_FINAL,
                                           MOONS_TAG_STDFLUX_RBNTELLCORRSPECTRA,
                                           tellcorr_filename, refframe),
                      " ");
    }
    else {
        cpl_msg_info(__func__, "No telluric correction");
    }

    moo_try_check(target_table =
                      moo_create_target_table(rbn, NULL, MOO_MODE_STARE,
                                              target_table_params),
                  " ");
    ttable_filename =
        cpl_sprintf("%s_OFFSET%d_%s_%d.fits", stdttable_tags[rmode], offset,
                    moo_mode_get_name(mode), inum);
    moo_try_check(moo_products_add_target_table(products, target_table,
                                                CPL_FRAME_LEVEL_INTERMEDIATE,
                                                stdttable_tags[rmode],
                                                ttable_filename, refframe),
                  " ");
    sci_filename = cpl_sprintf("%s_OFFSET%d_%s_%d.fits", stdsks_tags[rmode],
                               offset, moo_mode_get_name(mode), inum);
    moo_try_check(sci = _moons_sub_sky_stare(rbn, target_table, f2f_frame,
                                             sky_params, sci_filename),
                  " ");
    moo_try_check(moo_sci_compute_snr(sci, skylines), " ");

    moo_try_check(sci_frame = cpl_frame_duplicate(
                      moo_products_add_sci(products, sci, CPL_FRAME_LEVEL_FINAL,
                                           stdsks_tags[rmode], sci_filename,
                                           refframe)),
                  " ");

moo_try_cleanup:
    // dump error from the state
    if (!cpl_errorstate_is_equal(prestate)) {
        cpl_frame_delete(sci_frame);
        sci_frame = NULL;
    }
    cpl_free(det_filename);
    cpl_free(ext_filename);
    cpl_free(ffext_filename);
    cpl_free(rbn_filename);
    cpl_free(ttable_filename);
    cpl_free(tellcorr_filename);
    cpl_free(sci_filename);
    moo_target_table_delete(target_table);
    moo_sci_delete(sci);
    moo_telluric_delete(telluric);
    moo_rbn_delete(rbn);
    moo_ext_delete(ext);
    moo_loc_delete(loc);
    moo_psf_delete(psf);
    moo_det_delete(det);

    return sci_frame;
}


static cpl_frame *
_moons_prepare(int inum,
               const cpl_frame *frame,
               const char *bpmap_rp_name,
               const char *bpmap_nl_name,
               const cpl_frame *masterbias,
               const cpl_frame *masterdark_vis,
               const cpl_frame *masterdark_nir,
               const cpl_frame *coeffs_cube,
               moo_prepare_params *prepare_params,
               moo_correct_bias_params *cbias_params,
               moo_mode_type mode,
               moo_products *products,
               int rmode,
               int offset)
{
    cpl_frame *result = NULL;
    moo_det *det = NULL;

    const char *det_tag[] = { MOONS_TAG_STDFLUX_CORRECTDARK,
                              MOONS_TAG_STDTELL_CORRECTDARK };
    char *det_filename = NULL;

    cpl_errorstate prestate = cpl_errorstate_get();

    moo_try_check(det = moo_prepare(frame, bpmap_rp_name, bpmap_nl_name,
                                    masterbias, coeffs_cube, prepare_params),
                  " ");

    moo_try_check(moo_correct_bias(det, masterbias, cbias_params), " ");
    moo_try_check(moo_correct_dark(det, NULL, masterdark_vis, masterdark_nir),
                  " ");

    det_filename = cpl_sprintf("%s_OFFSET%d_%s_%d.fits", det_tag[rmode], offset,
                               moo_mode_get_name(mode), inum);
    moo_try_check(result = cpl_frame_duplicate(
                      moo_products_add(products, det,
                                       CPL_FRAME_LEVEL_INTERMEDIATE,
                                       det_tag[rmode], det_filename, frame)),
                  " ");

moo_try_cleanup:
    if (!cpl_errorstate_is_equal(prestate)) {
        cpl_frame_delete(result);
        result = NULL;
    }
    moo_det_delete(det);
    cpl_free(det_filename);
    return result;
}

static cpl_frameset *
_moons_prepare_set(const cpl_frameset *std_frameset,
                   const char *bpmap_rp_name,
                   const char *bpmap_nl_name,
                   const cpl_frame *masterbias,
                   const cpl_frame *masterdark_vis,
                   const cpl_frame *masterdark_nir,
                   const cpl_frame *coeffs_cube,
                   moo_prepare_params *prepare_params,
                   moo_correct_bias_params *cbias_params,
                   moo_mode_type mode,
                   moo_products *products,
                   int rmode,
                   int offset)
{
    cpl_frameset *detframes = NULL;
    moo_det *det = NULL;

    cpl_ensure(std_frameset, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(products, CPL_ERROR_NULL_INPUT, NULL);

    cpl_errorstate prestate = cpl_errorstate_get();

    moo_try_check(detframes = cpl_frameset_new(), " ");

    for (int i = 0; i < cpl_frameset_get_size(std_frameset); ++i) {
        const cpl_frame *current_frame = NULL;
        cpl_frame *frame = NULL;

        moo_try_check(current_frame =
                          cpl_frameset_get_position_const(std_frameset, i),
                      " ");

        moo_try_check(frame = _moons_prepare(i, current_frame, bpmap_rp_name,
                                             bpmap_nl_name, masterbias,
                                             masterdark_vis, masterdark_nir,
                                             coeffs_cube, prepare_params,
                                             cbias_params, mode, products,
                                             rmode, offset),
                      " ");

        moo_try_check(cpl_frameset_insert(detframes, frame), " ");
    }
moo_try_cleanup:
    if (!cpl_errorstate_is_equal(prestate)) {
        moo_det_delete(det);
        cpl_frameset_delete(detframes);
        detframes = NULL;
    }
    return detframes;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Interpret the command line options and execute the data processing
  @param    frameset   the frames list
  @param    parlist    the parameters list
  @return   0 if everything is ok  @return   0 if everything is ok
 */
/*----------------------------------------------------------------------------*/

static int
_moons_standard(cpl_frameset *frameset, const cpl_parameterlist *parlist)
{
    /* parameters */
    moo_mode_type mode;
    int offset;
    moo_prepare_params *prepare_params = NULL;
    moo_correct_bias_params *correct_bias_params = NULL;
    moo_crh_params *crh_params = NULL;
    moo_extract_params *extract_params = NULL;
    moo_wavesol_params *wavesol_params = NULL;
    moo_rebin_params *rbn_params = NULL;
    moo_target_table_params *target_table_params = NULL;
    moo_sub_sky_stare_params *sky_params = NULL;
    moo_coadd_params *coadd_params = NULL;
    moo_compute_resp_params *resp_params = NULL;
    moo_spectral_format *sformat = NULL;
    moo_map *wmap = NULL;
    moo_sky_lines_list *skylines = NULL;
    moo_flx *flx = NULL;
    moo_atm *atm = NULL;
    moo_detlist *det_list = NULL;
    moo_det *std_med = NULL;
    moo_scilist *sci_list = NULL;
    moo_sci *sci_coadd = NULL;
    moo_resp *resp = NULL;

    /* SOF file */
    cpl_frameset *std_frameset = cpl_frameset_new();
    cpl_frameset *sci_set = cpl_frameset_new();

    const char *bpmap_rp_name = NULL;
    const char *bpmap_nl_name = NULL;
    const cpl_frame *masterbias = NULL;
    const cpl_frame *masterdark_vis = NULL;
    const cpl_frame *masterdark_nir = NULL;
    const cpl_frame *coeffs_cube = NULL;
    const cpl_frame *p2pmap = NULL;
    const cpl_frame *fftrace = NULL;
    const cpl_frame *master_flat = NULL;
    const cpl_frame *sformat_frame = NULL;
    const cpl_frame *wmap_frame = NULL;
    const cpl_frame *skylines_frame = NULL;
    const cpl_frame *tell_frame = NULL;
    const cpl_frame *flat_frame = NULL;
    const cpl_frame *f2f_frame = NULL;
    const cpl_frame *flx_frame = NULL;
    const cpl_frame *atmo_frame = NULL;
    cpl_frameset *stddet_frameset = NULL;
    char *med_filename = NULL;
    char *resp_filename = NULL;
    char *calibflux_filename = NULL;
    /* FLUX ==> 0, TELL ==> 1 */
    int rmode;
    const char *rmodenames[] = { "STDFLUX", "STDTELL" };

    const char *modename = NULL;
    const cpl_frame *molecule_frame = NULL;
    const cpl_frame *winc_frame = NULL;
    moo_molecfit_model_params *molecfit_model_params = NULL;

    moo_molectable *atm_fitted = NULL;
    moo_molectable *best_fitted_params = NULL;
    moo_molectable *best_fitted_model = NULL;

    char *best_fitted_params_name = NULL;
    char *best_fitted_model_name = NULL;
    char *atm_fitted_name = NULL;

    moo_products *products =
        moo_products_new(frameset, parlist, "moons_standard",
                         PACKAGE "/" PACKAGE_VERSION);


    const moo_params *params = moo_products_get_params(products);
    moo_try_check(prepare_params = moo_params_get_prepare(params, parlist),
                  " ");
    moo_try_check(correct_bias_params =
                      moo_params_get_correct_bias(params, parlist),
                  " ");
    moo_try_check(crh_params = moo_params_get_crh(params, parlist), " ");
    moo_try_check(extract_params = moo_params_get_extract(params, parlist),
                  " ");
    moo_try_check(rbn_params = moo_params_get_rebin(params, parlist), " ");
    moo_try_check(target_table_params =
                      moo_params_get_target_table(params, parlist),
                  " ");
    moo_try_check(sky_params = moo_params_get_sub_sky_stare(params, parlist),
                  " ");
    moo_try_check(coadd_params = moo_params_get_coadd(params, parlist), " ");
    moo_try_check(resp_params = moo_params_get_compute_resp(params, parlist),
                  " ");
    moo_try_check(molecfit_model_params =
                      moo_params_get_molecfit_model(params, parlist),
                  " ");
    moo_try_check(_moons_standard_check_sof(
                      frameset, std_frameset, &bpmap_rp_name, &bpmap_nl_name,
                      &masterbias, &masterdark_vis, &masterdark_nir,
                      &coeffs_cube, &p2pmap, &fftrace, &master_flat,
                      &sformat_frame, &wmap_frame, &skylines_frame, &flat_frame,
                      &tell_frame, &f2f_frame, &flx_frame, &atmo_frame,
                      &molecule_frame, &winc_frame, extract_params, &rmode),
                  " ");

    const char *rmodename = rmodenames[rmode];
    int i;


    cpl_frame *frame = cpl_frameset_get_position(std_frameset, 0);

    moo_try_check(mode = moo_mode_get(frame), " ");
    moo_try_check(offset = moo_offset_get(frame), " ");

    cpl_msg_info("moons_standard",
                 "Use standard in mode %s frame mode %s offset %d", rmodename,
                 moo_mode_get_name(mode), offset);

    moo_try_check(sformat = moo_spectral_format_load(sformat_frame, mode), " ");
    moo_try_check(wmap = moo_map_load(wmap_frame), " ");
    moo_try_check(skylines = moo_sky_lines_list_load(skylines_frame), " ");

    moo_try_check(stddet_frameset =
                      _moons_prepare_set(std_frameset, bpmap_rp_name,
                                         bpmap_nl_name, masterbias,
                                         masterdark_vis, masterdark_nir,
                                         coeffs_cube, prepare_params,
                                         correct_bias_params, mode, products,
                                         rmode, offset),
                  " ");

    if (rmode == 1) {
        moo_try_check(det_list = moo_detlist_create(stddet_frameset), " ");
        moo_try_check(std_med = moo_remove_CRH(det_list, NULL, crh_params),
                      " ");
        med_filename =
            cpl_sprintf("%s_OFFSET%d_%s.fits", MOONS_TAG_STDTELL_REMOVECRH,
                        offset, moo_mode_get_name(mode));
        cpl_frameset_delete(stddet_frameset);
        stddet_frameset = cpl_frameset_new();
        moo_try_check(cpl_frameset_insert(stddet_frameset,
                                          cpl_frame_duplicate(moo_products_add(
                                              products, std_med,
                                              CPL_FRAME_LEVEL_INTERMEDIATE,
                                              MOONS_TAG_STDTELL_REMOVECRH,
                                              med_filename, frame))),
                      " ");
    }
    int size = cpl_frameset_get_size(stddet_frameset);

    for (i = 0; i < size; ++i) {
        const cpl_frame *current_frame = NULL;
        const cpl_frame *ref_frame = NULL;
        cpl_frame *sci_frame = NULL;
        moo_try_check(current_frame =
                          cpl_frameset_get_position_const(stddet_frameset, i),
                      " ");
        moo_try_check(ref_frame =
                          cpl_frameset_get_position_const(std_frameset, i),
                      " ");

        moo_try_check(sci_frame = _moons_standard_frame(
                          i, current_frame, ref_frame, p2pmap, fftrace,
                          master_flat, sformat, wmap, skylines, flat_frame,
                          f2f_frame, tell_frame, extract_params, rbn_params,
                          target_table_params, sky_params, mode, products,
                          rmode, offset),
                      " ");
        cpl_frameset_insert(sci_set, sci_frame);
    }

    if (rmode == 0) {
        sci_list = moo_scilist_create_clean(sci_set);

        moo_try_check(flx = moo_flx_load(flx_frame), " ");
        moo_try_check(atm = moo_atm_load(atmo_frame), " ");

        moo_try_check(resp = moo_compute_resp(sci_list, atm, flx, resp_params),
                      " ");

        resp_filename = cpl_sprintf("%s_%s.fits", MOONS_TAG_RESPONSE,
                                    moo_mode_get_name(mode));
        moo_try_check(moo_products_add_resp(products, resp,
                                            CPL_FRAME_LEVEL_FINAL,
                                            MOONS_TAG_RESPONSE, resp_filename,
                                            frame),
                      " ");
    }
    else {
        modename = moo_mode_get_name(mode);
        const cpl_frame *sci_frame = NULL;
        moo_try_check(sci_frame = cpl_frameset_get_position_const(sci_set, 0),
                      " ");
        moo_try_check(best_fitted_params =
                          moo_molecfit_model(sci_frame, modename,
                                             molecule_frame, winc_frame,
                                             &atm_fitted, &best_fitted_model,
                                             molecfit_model_params),
                      " ");
        atm_fitted_name =
            cpl_sprintf("%s_%s.fits", MOONS_TAG_MOLECFIT_ATM_PARAMS,
                        moo_mode_get_name(mode));
        best_fitted_params_name =
            cpl_sprintf("%s_%s.fits", MOONS_TAG_MOLECFIT_BEST_FIT_PARAMS,
                        moo_mode_get_name(mode));
        best_fitted_model_name =
            cpl_sprintf("%s_%s.fits", MOONS_TAG_MOLECFIT_BEST_FIT_MODEL,
                        moo_mode_get_name(mode));

        moo_try_check(moo_products_add_molectable(
                          products, best_fitted_params, CPL_FRAME_LEVEL_FINAL,
                          MOONS_TAG_MOLECFIT_BEST_FIT_PARAMS,
                          best_fitted_params_name, sci_frame),
                      " ");
        moo_try_check(
            moo_products_add_molectable(products, best_fitted_model,
                                        CPL_FRAME_LEVEL_INTERMEDIATE,
                                        MOONS_TAG_MOLECFIT_BEST_FIT_MODEL,
                                        best_fitted_model_name, sci_frame),
            " ");
        moo_try_check(moo_products_add_molectable(products, atm_fitted,
                                                  CPL_FRAME_LEVEL_FINAL,
                                                  MOONS_TAG_MOLECFIT_ATM_PARAMS,
                                                  atm_fitted_name, sci_frame),
                      " ");
    }

moo_try_cleanup:
    cpl_free(resp_filename);
    cpl_free(med_filename);
    cpl_free(calibflux_filename);
    moo_sci_delete(sci_coadd);
    moo_scilist_delete(sci_list);
    moo_sky_lines_list_delete(skylines);
    moo_map_delete(wmap);
    moo_spectral_format_delete(sformat);
    moo_flx_delete(flx);
    moo_atm_delete(atm);
    moo_resp_delete(resp);
    moo_det_delete(std_med);
    moo_detlist_delete(det_list);
    cpl_frameset_delete(std_frameset);
    cpl_frameset_delete(stddet_frameset);
    cpl_frameset_delete(sci_set);
    moo_extract_params_delete(extract_params);
    moo_wavesol_params_delete(wavesol_params);
    moo_target_table_params_delete(target_table_params);
    moo_rebin_params_delete(rbn_params);
    moo_sub_sky_stare_params_delete(sky_params);
    moo_coadd_params_delete(coadd_params);
    moo_compute_resp_params_delete(resp_params);
    moo_crh_params_delete(crh_params);
    moo_correct_bias_params_delete(correct_bias_params);
    moo_prepare_params_delete(prepare_params);
    cpl_free(best_fitted_params_name);
    cpl_free(best_fitted_model_name);
    cpl_free(atm_fitted_name);
    moo_molectable_delete(best_fitted_model);
    moo_molectable_delete(atm_fitted);
    moo_molectable_delete(best_fitted_params);
    moo_molecfit_model_params_delete(molecfit_model_params);
    moo_products_delete(products);
    return (int)cpl_error_get_code();
}
