/*
 * This file is part of the MOONS Pipeline
 * Copyright (C) 2002-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#include "moo_dfs.h"
#include "moo_drl.h"
#include "moo_params.h"
#include "moo_pfits.h"
#include "moo_products.h"
#include "moo_utils.h"
#include <cpl.h>

#include <string.h>

/*-----------------------------------------------------------------------------
                              Plugin registration
 -----------------------------------------------------------------------------*/

int cpl_plugin_get_info(cpl_pluginlist *list);

/*-----------------------------------------------------------------------------
                            Private function prototypes
 -----------------------------------------------------------------------------*/

static int _moons_wavecal_create(cpl_plugin *plugin);
static int _moons_wavecal_exec(cpl_plugin *plugin);
static int _moons_wavecal_destroy(cpl_plugin *plugin);
static int
_moons_wavecal(cpl_frameset *frameset, const cpl_parameterlist *parlist);

/*-----------------------------------------------------------------------------
                            Static variables
 -----------------------------------------------------------------------------*/

static const char *const _moons_wavecal_description =
    "This recipe aims at creating the products necessary to calibrate in"
    " wavelength the extracted 1D spectra and rebin them. It can be used to"
    " produce first the guess wavelength map, or to produce the final one,"
    " from the guess map.\n"
    "INPUT FRAMES\n"
    "  * Arc 1 file                               (RAW) with tag ARC : "
    "arc file\n"
    "  * Arc_off 1 file                           (RAW) with tag ARC_OFF : "
    "arc off file\n"
    "  * [OPTIONAL] ReferenceBadPixMask 1 file    (QUA) with tag BP_MAP_RP : "
    "cosmetic bad pixel map\n"
    "  * [OPTIONAL] NonLinearityBadPixMask 1 file (QUA) with tag BP_MAP_NL : "
    "cosmetic bad pixel map coming from linearity recipe\n"
    "  * MasterBias 1 file                        (DET) with tag MASTER_BIAS : "
    "master bias file\n"
    "  * [OPTIONAL] P2pMap 1 file                 (DET) with tag P2P_MAP : "
    "pixel to pixel map\n"
    "  * LocTab 1 file                            (LOC) with tag FF_TRACE : "
    "the localisation table\n"
    "FF_EXTSPECTRA : "
    "the extracted flat field\n"
    "  * F2f 1 file                               (F2F) with tag F2F_TABLE "
    ": "
    "the fibre-to-fibre relative response\n"
    "  * LineCat 1 file                           (CAT) with tag  "
    "ARC_LINE_LIST : "
    "the reference line list\n"
    "  * SFormat 1 file                           (FMT) with tag "
    "SPECTRAL_FORMAT : "
    "the spectral format table\n"
    "  * [OPTIONAL] WaveMapGuess 1 file           (EXT) with tag "
    "WAVE_MAP_GUESS : "
    "the first guess wavelength map\n"
    "PRODUCTS\n"
    "  * ARC_EXTSPECTRA[_GUESS]_OFFSET[offset]_[insmode].fits (EXT) with tag "
    "ARC_EXTSPECTRA : "
    "Arc extracted spectra\n"
    "  * WAVE_MAP[_GUESS]_OFFSET[offset]_[insmode].fits       (EXT) with tag "
    "WAVE_MAP[_GUESS] : "
    "Wavelength map\n"
    "  * ARC_RBNSPECTRA[_GUESS]_OFFSET[offset]_[insmode].fits (RBN) with tag "
    "RBN_EXTSPECTRA : "
    "Arc extracted spectra\n"
    "\n";

/*-----------------------------------------------------------------------------
                                Function code
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Build the list of available plugins, for this module.
  @param    list    the plugin list
  @return   0 if everything is ok, 1 otherwise
  @note     Only this function is exported

  Create the recipe instance and make it available to the application using the
  interface.
 */
/*----------------------------------------------------------------------------*/

int
cpl_plugin_get_info(cpl_pluginlist *list)
{
    cpl_recipe *recipe = cpl_calloc(1, sizeof *recipe);
    cpl_plugin *plugin = &recipe->interface;

    if (cpl_plugin_init(plugin, CPL_PLUGIN_API, MOONS_BINARY_VERSION,
                        CPL_PLUGIN_TYPE_RECIPE, "moons_wavecal",
                        "Produces the wavelength solution",
                        _moons_wavecal_description, "Regis Haigron",
                        PACKAGE_BUGREPORT, moo_get_license(),
                        _moons_wavecal_create, _moons_wavecal_exec,
                        _moons_wavecal_destroy)) {
        cpl_msg_error(cpl_func, "Plugin initialization failed");
        (void)cpl_error_set_where(cpl_func);
        return 1;
    }

    if (cpl_pluginlist_append(list, plugin)) {
        cpl_msg_error(cpl_func, "Error adding plugin to list");
        (void)cpl_error_set_where(cpl_func);
        return 1;
    }

    return 0;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Setup the recipe options
  @param    plugin  the plugin
  @return   0 if everything is ok

  Defining the command-line/configuration parameters for the recipe.
 */
/*----------------------------------------------------------------------------*/

static int
_moons_wavecal_create(cpl_plugin *plugin)
{
    cpl_recipe *recipe;

    /* Do not create the recipe if an error code is already set */
    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        cpl_msg_error(cpl_func, "%s():%d: An error is already set: %s",
                      cpl_func, __LINE__, cpl_error_get_where());
        return (int)cpl_error_get_code();
    }

    if (plugin == NULL) {
        cpl_msg_error(cpl_func, "Null plugin");
        cpl_ensure_code(0, (int)CPL_ERROR_NULL_INPUT);
    }

    /* Verify plugin type */
    if (cpl_plugin_get_type(plugin) != CPL_PLUGIN_TYPE_RECIPE) {
        cpl_msg_error(cpl_func, "Plugin is not a recipe");
        cpl_ensure_code(0, (int)CPL_ERROR_TYPE_MISMATCH);
    }

    /* Get the recipe */
    recipe = (cpl_recipe *)plugin;

    /* Create the parameters list in the cpl_recipe object */
    recipe->parameters = cpl_parameterlist_new();
    if (recipe->parameters == NULL) {
        cpl_msg_error(cpl_func, "Parameter list allocation failed");
        cpl_ensure_code(0, (int)CPL_ERROR_ILLEGAL_OUTPUT);
    }

    moo_params *params = moo_params_new("moons", "moons_wavecal");

    /* Fill the parameters list */
    moo_params_add_keep_temp(params, recipe->parameters);
    moo_params_add_prepare(params, recipe->parameters);
    moo_params_add_correct_bias(params, recipe->parameters,
                                MOO_CORRECT_BIAS_METHOD_MASTER);
    moo_params_add_crh(params, recipe->parameters, MOO_CRH_METHOD_MEDIAN);
    moo_params_add_extract(params, recipe->parameters);
    moo_params_add_wavesol(params, recipe->parameters);
    moo_params_add_rebin(params, recipe->parameters);

    moo_params_delete(params);

    return 0;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Execute the plugin instance given by the interface
  @param    plugin  the plugin
  @return   0 if everything is ok
 */
/*----------------------------------------------------------------------------*/

static int
_moons_wavecal_exec(cpl_plugin *plugin)
{
    cpl_recipe *recipe;
    int recipe_status;
    cpl_errorstate initial_errorstate = cpl_errorstate_get();

    /* Return immediately if an error code is already set */
    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        cpl_msg_error(cpl_func, "%s():%d: An error is already set: %s",
                      cpl_func, __LINE__, cpl_error_get_where());
        return (int)cpl_error_get_code();
    }

    if (plugin == NULL) {
        cpl_msg_error(cpl_func, "Null plugin");
        cpl_ensure_code(0, (int)CPL_ERROR_NULL_INPUT);
    }

    /* Verify plugin type */
    if (cpl_plugin_get_type(plugin) != CPL_PLUGIN_TYPE_RECIPE) {
        cpl_msg_error(cpl_func, "Plugin is not a recipe");
        cpl_ensure_code(0, (int)CPL_ERROR_TYPE_MISMATCH);
    }

    /* Get the recipe */
    recipe = (cpl_recipe *)plugin;

    /* Verify parameter and frame lists */
    if (recipe->parameters == NULL) {
        cpl_msg_error(cpl_func, "Recipe invoked with NULL parameter list");
        cpl_ensure_code(0, (int)CPL_ERROR_NULL_INPUT);
    }
    if (recipe->frames == NULL) {
        cpl_msg_error(cpl_func, "Recipe invoked with NULL frame set");
        cpl_ensure_code(0, (int)CPL_ERROR_NULL_INPUT);
    }

    /* Invoke the recipe */
    recipe_status = _moons_wavecal(recipe->frames, recipe->parameters);

    /* Ensure DFS-compliance of the products */
    if (cpl_dfs_update_product_header(recipe->frames)) {
        if (!recipe_status)
            recipe_status = (int)cpl_error_get_code();
    }

    if (!cpl_errorstate_is_equal(initial_errorstate)) {
        /* Dump the error history since recipe execution start.
           At this point the recipe cannot recover from the error */
        cpl_errorstate_dump(initial_errorstate, CPL_FALSE, NULL);
    }

    return recipe_status;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Destroy what has been created by the 'create' function
  @param    plugin  the plugin
  @return   0 if everything is ok
 */
/*----------------------------------------------------------------------------*/

static int
_moons_wavecal_destroy(cpl_plugin *plugin)
{
    cpl_recipe *recipe;

    if (plugin == NULL) {
        cpl_msg_error(cpl_func, "Null plugin");
        cpl_ensure_code(0, (int)CPL_ERROR_NULL_INPUT);
    }

    /* Verify plugin type */
    if (cpl_plugin_get_type(plugin) != CPL_PLUGIN_TYPE_RECIPE) {
        cpl_msg_error(cpl_func, "Plugin is not a recipe");
        cpl_ensure_code(0, (int)CPL_ERROR_TYPE_MISMATCH);
    }

    /* Get the recipe */
    recipe = (cpl_recipe *)plugin;

    cpl_parameterlist_delete(recipe->parameters);

    return 0;
}

static cpl_error_code
_moons_wavecal_check_sof(cpl_frameset *frameset,
                         cpl_frameset **arc_frameset,
                         cpl_frameset **arc_off_frameset,
                         const char **bpmap_rp_name,
                         const char **bpmap_nl_name,
                         const cpl_frame **masterbias,
                         const cpl_frame **masterdark_vis,
                         const cpl_frame **masterdark_nir,
                         const cpl_frame **p2pmap,
                         const cpl_frame **fftrace,
                         const cpl_frame **arc_line_list,
                         const cpl_frame **sformat,
                         const cpl_frame **wmap_guess,
                         const cpl_frame **flat_frame,
                         const cpl_frame **f2f_frame)
{
    cpl_ensure_code(moo_dfs_set_groups(frameset) == CPL_ERROR_NONE,
                    cpl_error_get_code());
    int nraw = 0;
    int nraw_off = 0;

    *arc_frameset = cpl_frameset_new();
    *arc_off_frameset = cpl_frameset_new();

    int i;
    for (i = 0; i < cpl_frameset_get_size(frameset); ++i) {
        cpl_frame *current_frame = cpl_frameset_get_position(frameset, i);
        if (!strcmp(cpl_frame_get_tag(current_frame), MOONS_TAG_ARC)) {
            cpl_frame *new_frame = cpl_frame_duplicate(current_frame);
            cpl_frameset_insert(*arc_frameset, new_frame);
            ++nraw;
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame), MOONS_TAG_ARC_OFF)) {
            cpl_frame *new_frame = cpl_frame_duplicate(current_frame);
            cpl_frameset_insert(*arc_off_frameset, new_frame);
            ++nraw_off;
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_BP_MAP_RP)) {
            *bpmap_rp_name = cpl_frame_get_filename(current_frame);
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_BP_MAP_NL)) {
            *bpmap_nl_name = cpl_frame_get_filename(current_frame);
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_MASTER_BIAS)) {
            *masterbias = current_frame;
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_MASTER_DARK_VIS)) {
            *masterdark_vis = current_frame;
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_MASTER_DARK_NIR)) {
            *masterdark_nir = current_frame;
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame), MOONS_TAG_P2P_MAP)) {
            *p2pmap = current_frame;
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_FF_TRACE)) {
            *fftrace = current_frame;
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_ARC_LINE_LIST)) {
            *arc_line_list = current_frame;
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_SPECTRAL_FORMAT)) {
            *sformat = current_frame;
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_WAVEMAP_GUESS)) {
            *wmap_guess = current_frame;
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_FF_EXTSPECTRA)) {
            *flat_frame = current_frame;
        }
        else if (!strcmp(cpl_frame_get_tag(current_frame),
                         MOONS_TAG_F2F_TABLE)) {
            *f2f_frame = current_frame;
        }
    }

    if (nraw == 0) {
        return (int)cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                          "SOF does not have any file tagged "
                                          "with %s",
                                          MOONS_TAG_ARC);
    }
    if (nraw_off == 0) {
        return (int)cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                          "SOF does not have any file tagged "
                                          "with %s",
                                          MOONS_TAG_ARC_OFF);
    }
    if (nraw != nraw_off) {
        return (int)cpl_error_set_message(
            cpl_func, CPL_ERROR_DATA_NOT_FOUND,
            "SOF does not have same number of  file with %s (%d) and %s (%d)",
            MOONS_TAG_ARC, nraw, MOONS_TAG_ARC_OFF, nraw_off);
    }
    if (*fftrace == NULL) {
        return (int)cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                          "SOF does not have any file tagged "
                                          "with %s",
                                          MOONS_TAG_FF_TRACE);
    }
    if (*arc_line_list == NULL) {
        return (int)cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                          "SOF does not have any file tagged "
                                          "with %s",
                                          MOONS_TAG_ARC_LINE_LIST);
    }

    if (*sformat == NULL) {
        return (int)cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                          "SOF does not have any file tagged "
                                          "with %s",
                                          MOONS_TAG_SPECTRAL_FORMAT);
    }
    return CPL_ERROR_NONE;
}

static cpl_error_code
_moons_p2pmap(moo_det *arc, const cpl_frame *p2pmap)
{
    cpl_error_code status = CPL_ERROR_NONE;
    moo_det *p2p = NULL;

    if (p2pmap != NULL) {
        moo_try_check(p2p = moo_det_create(p2pmap), " ");
        moo_try_check(moo_apply_p2p(arc, p2p), " ");
    }
moo_try_cleanup:
    moo_det_delete(p2p);
    return status;
}

static cpl_error_code
_moons_apply_flat(moo_ext *ext,
                  const cpl_frame *flat_frame,
                  const cpl_frame *f2f_frame)
{
    cpl_error_code status = CPL_ERROR_NONE;
    moo_ext *flat = NULL;
    moo_f2f *f2f = NULL;

    if (flat_frame != NULL) {
        moo_try_check(flat = moo_ext_create(flat_frame), " ");
        if (f2f_frame != NULL) {
            f2f = moo_f2f_load(f2f_frame);
        }
        moo_try_check(moo_apply_flat(ext, flat, f2f), " ");
    }

moo_try_cleanup:
    moo_ext_delete(flat);
    moo_f2f_delete(f2f);
    return status;
}

static int
_moons_get_offset(const cpl_frame *frame)
{
    const char *filename = NULL;
    cpl_propertylist *header = NULL;
    int result = 0;

    cpl_ensure(frame != NULL, CPL_ERROR_NULL_INPUT, 0);

    moo_try_check(filename = cpl_frame_get_filename(frame), " ");
    moo_try_check(header = cpl_propertylist_load(filename, 0), " ");
    moo_try_check(result = moo_pfits_get_slit_offset(header), " ");

moo_try_cleanup:
    cpl_propertylist_delete(header);
    return result;
}

/*----------------------------------------------------------------------------*/
/**
  @brief Prepare one RAW ARC file and correct it from the BIAS,DARK
  @param products the products object
  @param frame the input RAW FLAT frame
  @param bpmap_rp_name the BPMAP_RP file name or NULL
  @param bpmap_nl_name the BPMAP_NL file name or NULL
  @param masterbias the MASTER_BIAS frame or NULL
  @param i the index of frames in set
  @return   a new DET_DARK frame
 *
 * Possible #_cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if a required input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
static cpl_frame *
_moons_prepare(moo_products *products,
               const cpl_frame *arc_frame,
               const cpl_frame *arc_off_frame,
               const char *bpmap_rp_name,
               const char *bpmap_nl_name,
               const cpl_frame *masterbias,
               const cpl_frame *masterdark_vis,
               const cpl_frame *masterdark_nir,
               moo_prepare_params *prepare_params,
               moo_correct_bias_params *correct_bias_params,
               int i)
{
    cpl_frame *result = NULL;
    cpl_frame *pframe = NULL;
    char *detname1 = NULL;
    moo_det *det_arc = NULL;
    moo_det *det_arc_off = NULL;

    cpl_ensure(arc_frame != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_errorstate prestate = cpl_errorstate_get();

    moo_try_check(det_arc = moo_prepare(arc_frame, bpmap_rp_name, bpmap_nl_name,
                                        masterbias, NULL, prepare_params),
                  " ");

    moo_try_check(det_arc_off =
                      moo_prepare(arc_off_frame, bpmap_rp_name, bpmap_nl_name,
                                  masterbias, NULL, prepare_params),
                  " ");

    moo_try_check(moo_correct_bias(det_arc, masterbias, correct_bias_params),
                  " ");
    moo_try_check(detname1 =
                      cpl_sprintf("%s_%d.fits", MOONS_TAG_ARC_CORRECTBIAS, i),
                  " ");

    moo_try_check(moo_products_add(products, det_arc,
                                   CPL_FRAME_LEVEL_INTERMEDIATE,
                                   MOONS_TAG_ARC_CORRECTBIAS, detname1,
                                   arc_frame),
                  " ");
    cpl_free(detname1);
    detname1 = NULL;
    moo_try_check(detname1 =
                      cpl_sprintf("%s_%d.fits", MOONS_TAG_ARCOFF_PREPARE, i),
                  " ");
    moo_try_check(moo_products_add(products, det_arc_off,
                                   CPL_FRAME_LEVEL_INTERMEDIATE,
                                   MOONS_TAG_ARCOFF_PREPARE, detname1,
                                   arc_off_frame),
                  " ");
    cpl_free(detname1);
    detname1 = NULL;
    moo_try_check(moo_correct_dark(det_arc, det_arc_off, masterdark_vis,
                                   masterdark_nir),
                  " ");
    moo_try_check(detname1 =
                      cpl_sprintf("%s_%d.fits", MOONS_TAG_ARC_CORRECTDARK, i),
                  " ");
    moo_try_check(pframe = moo_products_add(products, det_arc,
                                            CPL_FRAME_LEVEL_INTERMEDIATE,
                                            MOONS_TAG_ARC_CORRECTDARK, detname1,
                                            arc_frame),
                  " ");

    moo_try_check(result = cpl_frame_duplicate(pframe), " ");

moo_try_cleanup:
    if (!cpl_errorstate_is_equal(prestate)) {
        cpl_frame_delete(result);
        result = NULL;
    }
    moo_det_delete(det_arc);
    moo_det_delete(det_arc_off);
    cpl_free(detname1);
    return result;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Prepare a set of RAW DARK files in DET format
  @param    raw_frames the input set of RAW frames
  @param bpmap_rp_name the BPMAP_RP file name or NULL
  @param bpmap_nl_name the BPMAP_NL file name or NULL
  @param masterbias the MASTER_BIAS frame or NULL
  @param do_correct_bias boolean to apply bias correction
  @param products the products object
  @return   a new set of DET_DARK frames
 *
 * Possible #_cpl_error_code_ set in this function:
  - CPL_ERROR_NULL_INPUT if a required input pointer is NULL
 */
/*----------------------------------------------------------------------------*/
static cpl_frameset *
_moons_prepare_set(cpl_frameset *raw_frames,
                   cpl_frameset *raw_off_frames,
                   const char *bpmap_rp_name,
                   const char *bpmap_nl_name,
                   const cpl_frame *masterbias,
                   const cpl_frame *masterdark_vis,
                   const cpl_frame *masterdark_nir,
                   moo_prepare_params *prepare_params,
                   moo_correct_bias_params *params,
                   moo_products *products)
{
    cpl_frameset *detframes = NULL;

    cpl_ensure(raw_frames, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(raw_off_frames, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(products, CPL_ERROR_NULL_INPUT, NULL);

    cpl_errorstate prestate = cpl_errorstate_get();

    moo_try_check(detframes = cpl_frameset_new(), " ");

    for (int i = 0; i < cpl_frameset_get_size(raw_frames); ++i) {
        const cpl_frame *current_frame = NULL;
        const cpl_frame *current_off_frame = NULL;
        cpl_frame *frame = NULL;

        moo_try_check(current_frame =
                          cpl_frameset_get_position_const(raw_frames, i),
                      " ");
        moo_try_check(current_off_frame =
                          cpl_frameset_get_position_const(raw_off_frames, i),
                      " ");

        moo_try_check(frame = _moons_prepare(products, current_frame,
                                             current_off_frame, bpmap_rp_name,
                                             bpmap_nl_name, masterbias,
                                             masterdark_vis, masterdark_nir,
                                             prepare_params, params, i),
                      " ");

        moo_try_check(cpl_frameset_insert(detframes, frame), " ");
    }
moo_try_cleanup:
    if (!cpl_errorstate_is_equal(prestate)) {
        cpl_frameset_delete(detframes);
        detframes = NULL;
    }
    return detframes;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Interpret the command line options and execute the data processing
  @param    frameset   the frames list
  @param    parlist    the parameters list
  @return   0 if everything is ok
 */
/*----------------------------------------------------------------------------*/

static int
_moons_wavecal(cpl_frameset *frameset, const cpl_parameterlist *parlist)
{
    /* parameters */
    moo_prepare_params *prepare_params = NULL;
    moo_correct_bias_params *correct_bias_params = NULL;
    moo_crh_params *crh_params = NULL;
    moo_extract_params *extract_params = NULL;
    moo_wavesol_params *wavesol_params = NULL;
    moo_rebin_params *rbn_params = NULL;
    moo_detlist *arc_det_list = NULL;
    moo_det *arc_det_med = NULL;
    moo_spectral_format *sformat = NULL;
    moo_loc *loc = NULL;
    moo_ext *ext = NULL;
    moo_map *wmap_guess = NULL;
    moo_map *wmap_refit = NULL;
    moo_map *wmap = NULL;
    moo_rbn *rbn = NULL;
    moo_rbn *refit_rbn = NULL;
    moo_rbn *ppm_rbn = NULL;
    moo_mode_type mode;
    int offset = 0;
    char *ext_filename = NULL;
    char *ffext_filename = NULL;
    char *ppm_rbn_filename = NULL;
    char *ppm_wave_filename = NULL;
    char *refit_rbn_filename = NULL;
    char *refit_wave_filename = NULL;
    char *rbn_filename = NULL;
    char *wave_filename = NULL;

    /* SOF file */
    cpl_frameset *arc_frameset = NULL;
    cpl_frameset *arc_off_frameset = NULL;
    cpl_frameset *arc_det_frameset = NULL;
    const char *bpmap_rp_name = NULL;
    const char *bpmap_nl_name = NULL;
    const cpl_frame *masterbias = NULL;
    const cpl_frame *masterdark_vis = NULL;
    const cpl_frame *masterdark_nir = NULL;
    const cpl_frame *p2pmap = NULL;
    const cpl_frame *fftrace = NULL;
    const cpl_frame *flat_frame = NULL;
    const cpl_frame *f2f_frame = NULL;

    const cpl_frame *arc_line_list = NULL;
    const cpl_frame *sformat_frame = NULL;
    const cpl_frame *wmap_frame = NULL;

    moo_products *products =
        moo_products_new(frameset, parlist, "moons_wavecal",
                         PACKAGE "/" PACKAGE_VERSION);

    const moo_params *params = moo_products_get_params(products);
    moo_try_check(prepare_params = moo_params_get_prepare(params, parlist),
                  " ");
    moo_try_check(correct_bias_params =
                      moo_params_get_correct_bias(params, parlist),
                  " ");
    moo_try_check(crh_params = moo_params_get_crh(params, parlist), " ");
    moo_try_check(extract_params = moo_params_get_extract(params, parlist),
                  " ");
    if (strcmp(extract_params->method, MOO_EXTRACT_METHOD_SUM) != 0) {
        cpl_msg_info(__func__,
                     "Extract method %s not supported : use %s instead",
                     extract_params->method, MOO_EXTRACT_METHOD_SUM);
        extract_params->method = MOO_EXTRACT_METHOD_SUM;
    }
    moo_try_check(wavesol_params = moo_params_get_wavesol(params, parlist),
                  " ");
    moo_try_check(rbn_params = moo_params_get_rebin(params, parlist), " ");

    moo_try_check(_moons_wavecal_check_sof(frameset, &arc_frameset,
                                           &arc_off_frameset, &bpmap_rp_name,
                                           &bpmap_nl_name, &masterbias,
                                           &masterdark_vis, &masterdark_nir,
                                           &p2pmap, &fftrace, &arc_line_list,
                                           &sformat_frame, &wmap_frame,
                                           &flat_frame, &f2f_frame),
                  " ");

    cpl_frame *arc_frame = cpl_frameset_get_position(arc_frameset, 0);

    moo_try_check(mode = moo_mode_get(arc_frame), " ");
    moo_try_check(offset = _moons_get_offset(arc_frame), " ");

    cpl_msg_info(__func__, "Use mode %s", moo_mode_get_name(mode));

    moo_try_check(arc_det_frameset =
                      _moons_prepare_set(arc_frameset, arc_off_frameset,
                                         bpmap_rp_name, bpmap_nl_name,
                                         masterbias, masterdark_vis,
                                         masterdark_nir, prepare_params,
                                         correct_bias_params, products),
                  " ");

    moo_try_check(arc_det_list = moo_detlist_create(arc_det_frameset), " ");

    moo_try_check(arc_det_med = moo_remove_CRH(arc_det_list, NULL, crh_params),
                  " ");

    const char *arc_filename = MOONS_TAG_ARC_REMOVECRH ".fits";
    moo_try_check(moo_products_add(products, arc_det_med,
                                   CPL_FRAME_LEVEL_INTERMEDIATE,
                                   MOONS_TAG_ARC_REMOVECRH, arc_filename,
                                   arc_frame),
                  " ");

    moo_try_check(_moons_p2pmap(arc_det_med, p2pmap), " ");

    moo_try_check(loc = moo_loc_load(fftrace), " ");
    moo_try_check(sformat = moo_spectral_format_load(sformat_frame, mode), " ");


    const char *wtag = MOONS_TAG_WAVEMAP_GUESS;
    const char *arc_line_list_name = cpl_frame_get_filename(arc_line_list);

    if (wmap_frame != NULL) {
        ext_filename =
            cpl_sprintf("%s_OFFSET%d_%s.fits", MOONS_TAG_ARC_EXTSPECTRA, offset,
                        moo_mode_get_name(mode));

        moo_try_check(ext = moo_extract(arc_det_med, loc, NULL, extract_params,
                                        ext_filename),
                      " ");
        moo_try_check(moo_products_add_ext(products, ext, CPL_FRAME_LEVEL_FINAL,
                                           MOONS_TAG_ARC_EXTSPECTRA,
                                           ext_filename, arc_frame),
                      " ");
        moo_try_check(_moons_apply_flat(ext, flat_frame, f2f_frame), " ");
        if (flat_frame != NULL) {
            ffext_filename =
                cpl_sprintf("%s_OFFSET%d_%s.fits", MOONS_TAG_ARC_FFEXTSPECTRA,
                            offset, moo_mode_get_name(mode));

            moo_try_check(moo_products_add_ext(products, ext,
                                               CPL_FRAME_LEVEL_FINAL,
                                               MOONS_TAG_ARC_FFEXTSPECTRA,
                                               ffext_filename, arc_frame),
                          " ");
        }
        moo_try_check(wmap_guess = moo_map_load(wmap_frame), " ");
        wtag = MOONS_TAG_WAVEMAP;
        rbn_filename =
            cpl_sprintf("%s_OFFSET%d_%s.fits", MOONS_TAG_ARC_RBNSPECTRA, offset,
                        moo_mode_get_name(mode));
        wave_filename = cpl_sprintf("%s_OFFSET%d_%s.fits", wtag, offset,
                                    moo_mode_get_name(mode));
        moo_try_check(wmap = moo_wavesol(ext, arc_line_list_name, sformat, loc,
                                         wmap_guess, wavesol_params),
                      " ");

        moo_try_check(rbn = moo_rebin(ext, wmap, sformat, rbn_params,
                                      rbn_filename),
                      " ");

        moo_try_check(moo_products_add_rbn(products, rbn, CPL_FRAME_LEVEL_FINAL,
                                           MOONS_TAG_ARC_RBNSPECTRA,
                                           rbn_filename, arc_frame),
                      " ");

        moo_try_check(moo_products_add_map(products, wmap,
                                           CPL_FRAME_LEVEL_FINAL, wtag,
                                           wave_filename, arc_frame, rbn),
                      "can't create wave map product");
    }
    else {
        ext_filename =
            cpl_sprintf("%s_OFFSET%d_%s.fits", MOONS_TAG_ARC_EXTSPECTRA_GUESS,
                        offset, moo_mode_get_name(mode));

        moo_try_check(ext = moo_extract(arc_det_med, loc, NULL, extract_params,
                                        ext_filename),
                      " ");
        moo_try_check(moo_products_add_ext(products, ext, CPL_FRAME_LEVEL_FINAL,
                                           MOONS_TAG_ARC_EXTSPECTRA_GUESS,
                                           ext_filename, arc_frame),
                      " ");
        moo_try_check(_moons_apply_flat(ext, flat_frame, f2f_frame), " ");
        if (flat_frame != NULL) {
            ffext_filename =
                cpl_sprintf("%s_OFFSET%d_%s.fits", MOONS_TAG_ARC_FFEXTSPECTRA,
                            offset, moo_mode_get_name(mode));

            moo_try_check(moo_products_add_ext(products, ext,
                                               CPL_FRAME_LEVEL_FINAL,
                                               MOONS_TAG_ARC_FFEXTSPECTRA,
                                               ffext_filename, arc_frame),
                          " ");
        }
        ppm_wave_filename =
            cpl_sprintf("%s_OFFSET%d_%s.fits", MOONS_TAG_WAVEMAP_PPM, offset,
                        moo_mode_get_name(mode));
        ppm_rbn_filename =
            cpl_sprintf("%s_OFFSET%d_%s.fits", MOONS_TAG_ARC_RBNSPECTRA_PPM,
                        offset, moo_mode_get_name(mode));
        refit_wave_filename =
            cpl_sprintf("%s_OFFSET%d_%s.fits", MOONS_TAG_WAVEMAP_REFIT, offset,
                        moo_mode_get_name(mode));
        refit_rbn_filename =
            cpl_sprintf("%s_OFFSET%d_%s.fits", MOONS_TAG_ARC_RBNSPECTRA_REFIT,
                        offset, moo_mode_get_name(mode));
        wave_filename =
            cpl_sprintf("%s_OFFSET%d_%s.fits", MOONS_TAG_WAVEMAP_GUESS, offset,
                        moo_mode_get_name(mode));
        rbn_filename =
            cpl_sprintf("%s_OFFSET%d_%s.fits", MOONS_TAG_ARC_RBNSPECTRA_GUESS,
                        offset, moo_mode_get_name(mode));

        moo_try_check(wmap_guess = moo_wavesol(ext, arc_line_list_name, sformat,
                                               loc, NULL, wavesol_params),
                      " ");
        moo_try_check(wmap_refit = moo_wavesol(ext, arc_line_list_name, sformat,
                                               loc, wmap_guess, wavesol_params),
                      " ");

        if (wmap_refit != NULL) {
            moo_try_check(ppm_rbn = moo_rebin(ext, wmap_guess, sformat,
                                              rbn_params, ppm_rbn_filename),
                          " ");
            moo_try_check(refit_rbn = moo_rebin(ext, wmap_refit, sformat,
                                                rbn_params, refit_rbn_filename),
                          " ");

            moo_try_check(moo_products_add_rbn(products, ppm_rbn,
                                               CPL_FRAME_LEVEL_INTERMEDIATE,
                                               MOONS_TAG_ARC_RBNSPECTRA_PPM,
                                               ppm_rbn_filename, arc_frame),
                          " ");

            moo_try_check(moo_products_add_map(products, wmap_guess,
                                               CPL_FRAME_LEVEL_INTERMEDIATE,
                                               MOONS_TAG_WAVEMAP_PPM,
                                               ppm_wave_filename, arc_frame,
                                               ppm_rbn),
                          "can't create wave map product");

            moo_try_check(moo_products_add_rbn(products, refit_rbn,
                                               CPL_FRAME_LEVEL_INTERMEDIATE,
                                               MOONS_TAG_ARC_RBNSPECTRA_REFIT,
                                               refit_rbn_filename, arc_frame),
                          " ");

            moo_try_check(moo_products_add_map(products, wmap_refit,
                                               CPL_FRAME_LEVEL_INTERMEDIATE,
                                               MOONS_TAG_WAVEMAP_REFIT,
                                               refit_wave_filename, arc_frame,
                                               refit_rbn),
                          "can't create wave map product");

            for (int i = 0; i < 6; i++) {
                wavesol_params->linefit_winhsize[i] =
                    wavesol_params->linedetect_winhsize[i];
            }
            wavesol_params->isrefit = 2;

            moo_try_check(wmap = moo_wavesol(ext, arc_line_list_name, sformat,
                                             loc, wmap_refit, wavesol_params),
                          " ");

            moo_try_check(rbn = moo_rebin(ext, wmap, sformat, rbn_params,
                                          rbn_filename),
                          " ");
            moo_try_check(moo_products_add_rbn(products, rbn,
                                               CPL_FRAME_LEVEL_FINAL,
                                               MOONS_TAG_ARC_RBNSPECTRA_GUESS,
                                               rbn_filename, arc_frame),
                          " ");

            moo_try_check(moo_products_add_map(products, wmap,
                                               CPL_FRAME_LEVEL_FINAL, wtag,
                                               wave_filename, arc_frame, rbn),
                          "can't create wave map product");
        }
        else {
            moo_try_check(rbn = moo_rebin(ext, wmap_guess, sformat, rbn_params,
                                          rbn_filename),
                          " ");

            moo_try_check(moo_products_add_map(products, wmap_guess,
                                               CPL_FRAME_LEVEL_FINAL, wtag,
                                               wave_filename, arc_frame, rbn),
                          "can't create wave map product");

            moo_try_check(moo_products_add_rbn(products, rbn,
                                               CPL_FRAME_LEVEL_FINAL,
                                               MOONS_TAG_ARC_RBNSPECTRA,
                                               rbn_filename, arc_frame),
                          " ");
        }
    }

moo_try_cleanup:
    cpl_free(ext_filename);
    cpl_free(ffext_filename);
    cpl_free(rbn_filename);
    cpl_free(wave_filename);
    cpl_free(refit_rbn_filename);
    cpl_free(refit_wave_filename);
    cpl_free(ppm_rbn_filename);
    cpl_free(ppm_wave_filename);
    moo_rbn_delete(rbn);
    moo_rbn_delete(refit_rbn);
    moo_rbn_delete(ppm_rbn);
    moo_map_delete(wmap);
    moo_map_delete(wmap_guess);
    moo_map_delete(wmap_refit);
    moo_ext_delete(ext);
    moo_loc_delete(loc);
    moo_spectral_format_delete(sformat);
    moo_det_delete(arc_det_med);
    moo_detlist_delete(arc_det_list);
    cpl_frameset_delete(arc_frameset);
    cpl_frameset_delete(arc_off_frameset);

    cpl_frameset_delete(arc_det_frameset);
    moo_extract_params_delete(extract_params);
    moo_wavesol_params_delete(wavesol_params);
    moo_rebin_params_delete(rbn_params);
    moo_crh_params_delete(crh_params);
    moo_correct_bias_params_delete(correct_bias_params);
    moo_prepare_params_delete(prepare_params);
    moo_products_delete(products);
    return (int)cpl_error_get_code();
}
