import logging
from pathlib import Path

from adari_core.data_libs.master_dark_bias import MasterDarkBiasReport
from adari_core.plots.text import TextPlot
from adari_core.utils.esodet import strip_prescan_overscan

from .moons_utils import MoonsReportMixin

logger = logging.getLogger(__name__)


class MoonsMasterBiasReport(MoonsReportMixin, MasterDarkBiasReport):
    # Attributes of the multi-extension report

    def __init__(self):
        super().__init__("moons_master_bias")
        self.data_extensions = ["RI_1", "RI_2"]

        self.center_size = 200
        self.image_category = "master_im"
        self.fov_per_panel = 2
        self.plot_zoom_fov = False
        self.panel_kwargs = {"x": 2, "y": 1, "height_ratios": [1, 1]}
        self.im_clipping = "percentile"
        self.im_clipping_kwargs = {"percentile": 50}

    def remove_raw_scan(self, im_hdu, **kwargs):
        return strip_prescan_overscan(im_hdu, debug=True, **kwargs)

    def parse_sof(self):
        raw_bias = None
        master_bias = None

        for name, catg in self.inputs:
            if catg == "BIAS" and raw_bias is None:
                raw_bias = name
            elif catg == "MASTER_BIAS" and master_bias is None:
                master_bias = name

        files = []
        if raw_bias is not None and master_bias is not None:
            files.append(
                {
                    "raw_im": raw_bias,
                    "master_im": master_bias,
                }
            )
        return files

    def generate_panels(self, **kwargs):
        panels = {}
        for ext in self.data_extensions:
            _panels = super().generate_raw_cuts_panels(
                master_im_ext=ext,
                master_title="Master Bias",
                raw_im_ext=ext,
                **kwargs,
            )
            for i, (panel, panel_description) in enumerate(_panels.items()):
                panel_description["report_description"] = (
                    f"MOONS bias panel - {Path(panel_description['master_im']).name}, "
                    "{panel_description['master_im_ext']}"
                )
                master_im = self.hdus[i]["master_im"]

                # Text Plot
                px = 0
                py = 0
                vspace = 0.3
                t1 = TextPlot(columns=1, v_space=vspace)
                col1 = (
                    str(master_im["PRIMARY"].header.get("INSTRUME")),
                    "EXTNAME: " + str(master_im[ext].header.get("EXTNAME", "N/A")),
                    "PRO CATG: " + str(master_im["PRIMARY"].header.get("HIERARCH ESO PRO CATG")),
                    "FILE NAME: " + Path(master_im.filename()).name,
                    "RAW1 NAME: " + str(master_im["PRIMARY"].header.get("HIERARCH ESO PRO REC1 RAW1 NAME")),
                )
                t1.add_data(col1)
                panel.assign_plot(t1, px, py, xext=2)

                px = px + 2
                t2 = TextPlot(columns=1, v_space=vspace, xext=1)
                col2 = "DATANCOM: " + str(master_im["PRIMARY"].header.get("HIERARCH ESO PRO DATANCOM", "N/A"))
                t2.add_data(col2)
                panel.assign_plot(t2, px, py, xext=1)

            panels = {**panels, **_panels}

        # Generate the multi-extension panels
        img_kw_list = [
            {
                "title": channel,
                "v_clip": self.im_clipping,
                "v_clip_kwargs": self.im_clipping_kwargs,
            }
            for channel in self.data_extensions
        ]

        _panels = self.generate_multi_extension_panels(img_kw_list=img_kw_list)
        for i, (panel, panel_description) in enumerate(_panels.items()):
            hdul = self.hdus[i][self.image_category]
            setup = str(hdul["PRIMARY"].header.get("HIERARCH ESO INS TAS MODE", "N/A"))
            panel_description["report_name"] = f"{self.name}_{setup.lower()}_multi"
            panel_description["report_description"] = f"{self.name} Panel - {Path(hdul.filename()).name}, "

            text_col = (f"DATANCOM: {hdul['PRIMARY'].header.get('HIERARCH ESO PRO DATANCOM', 'N/A')!s}",)
            text = TextPlot(columns=1, v_space=0.3)
            text.add_data(text_col)
            panel.assign_plot(text, 1, 0, xext=1)

        return {**panels, **_panels}


rep = MoonsMasterBiasReport()
