from pathlib import Path

from adari_core.data_libs.master_rawdisp import MasterRawdispReport

from .moons_utils import MoonsReportMixin, MoonsSetupInfo


class MoonsRawdispReport(MoonsReportMixin, MasterRawdispReport):
    def __init__(self):
        super().__init__("moons_rawdisp")

        self.raw_extensions = {
            "RI_1": "RI_1",
            "YJ_1": "YJ_1",
            "H_1": "H_1",
            "RI_2": "RI_2",
            "YJ_2": "YJ_2",
            "H_2": "H_2",
        }
        self.tasks = {
            "BIAS": "bias",
        }
        self.select_raw_files = {}
        self.task_scaling = {}
        self.setup_info = MoonsSetupInfo()
        self.extensions = []

    def parse_sof(self):
        tags = list(self.tasks.keys())
        added = {}
        file_lists = []
        for filename, catg in self.inputs:
            if catg in tags and (filename is not None and catg not in added):
                file_lists.append({"filename": filename})
                added[catg] = self.tasks[catg]
                self.sof_tag.append(catg)
        return file_lists

    def get_extensions(self):
        """
        Find the data extensions required for each MOONS file.

        Description
        -----------
        After the SOF has been parsed, this method iterates over the different
        HDUS to find which extension(s) contains the data.
        """

        new_hdu_list = []
        new_sof_tag = []
        for i, filedict in enumerate(self.hdus):
            hdul = filedict["filename"]
            all_hdu_names = [hdu.name for hdu in hdul]

            # Only the RI extensions contain data for BIAS
            # and DARK_VIS frames
            if self.sof_tag[i] == "BIAS":
                for extension in ["RI_1", "RI_2"]:
                    if extension in all_hdu_names:
                        new_hdu_list.append(filedict)
                        new_sof_tag.append(self.sof_tag[i])
                        self.extensions.append(extension)
                continue

            for extension in self.raw_extensions:
                if extension in all_hdu_names:
                    new_hdu_list.append(filedict)
                    new_sof_tag.append(self.sof_tag[i])
                    self.extensions.append(extension)

        self.hdus = new_hdu_list
        self.sof_tag = new_sof_tag

    def generate_panels(self, **kwargs):
        panels = {}
        self.get_extensions()

        _panels = super().generate_panels(ext=self.extensions, **kwargs)
        for i, (_panel, panel_description) in enumerate(_panels.items()):
            try:
                task_name = panel_description["task_name"]
            except KeyError as err:
                msg = "A report has been created by MasterRawdispReport that "
                "did not come back with a task name attached!"
                raise RuntimeError(msg) from err

            panel_description["report_name"] = "moons_rawdisp_{}_{}_{}_{}".format(
                task_name,
                self.raw_extensions[self.extensions[i]],
                self.sof_tag[i].lower(),
                Path(panel_description["filename"]).name,
            )
            panel_description["report_description"] = (
                f"MOONS rawdisp panel - "
                f"{panel_description['task_name']}, "
                f"{panel_description['tag']}, "
                f"{Path(panel_description['filename']).name}, "
                f"{panel_description['ext']}"
            )
        return {**panels, **_panels}


rep = MoonsRawdispReport()
