/*
 * This file is part of the ESO Telluric Correction Library
 * Copyright (C) 2001-2018 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

/*----------------------------------------------------------------------------*/
/**
 *                              Includes
 */
/*----------------------------------------------------------------------------*/

#include "mf_kernel_user.h"

/*----------------------------------------------------------------------------*/
/**
 *                 Typedefs: Enumeration types
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Defines
 */
/*----------------------------------------------------------------------------*/

/* cpl_propertylist KEYWORDS in FITS files */
#define MF_KEYWORD_CRVAL1 "CRVAL1" /*  */
#define MF_KEYWORD_CD1_1  "CD1_1"  /*  */
#define MF_KEYWORD_CD2_2  "CD2_2"  /*  */
#define MF_KEYWORD_CDELT1 "CDELT1" /*  */

/*----------------------------------------------------------------------------*/
/**
 *                 Global variables
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Macros
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Typedefs: Structured types
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Functions prototypes
 */
/*----------------------------------------------------------------------------*/

/* */
static cpl_matrix *mf_kernel_user_resample(
    const cpl_size          spec_n_lambdas,
    const cpl_propertylist *header_spec,
    const cpl_propertylist *header_kernel,
    const cpl_matrix       *kernel
);

/*  */
static cpl_error_code mf_kernel_user_normalize(cpl_matrix *kernel);

/* Convolution of a flux array with given kernel for a fixed range of pixels */
static cpl_error_code
mf_kernel_convolve(cpl_array *convflux, const cpl_array *flux, const int range[2], const cpl_array *kernel);

/*----------------------------------------------------------------------------*/
/**
 *                 Functions
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 * @defgroup mf_kernel_user       Tools to generate the input user kernel.
 *
 * @brief
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/* ---------------------------------------------------------------------------*/
/**
 * @brief   Resample and normalize the input kernel to the input spec.
 *
 * @param spec_n_lambdas       .
 * @param header_spec          .
 * @param header_kernel        .
 * @param kernel               .
 *
 * @return cpl_error_code    CPL_ERROR_NONE is everything is OK.
 *                           If not, these are the errors:
 *                           - .
 *                           - Error in subroutine (see subroutines).
 *
 * @note .
 *
 */
/* ---------------------------------------------------------------------------*/
cpl_matrix *mf_kernel_user_create(
    const cpl_size          spec_n_lambdas,
    const cpl_propertylist *header_spec,
    const cpl_propertylist *header_kernel,
    const cpl_matrix       *kernel
)
{
    /* Resample kernel */
    cpl_matrix *kernel_new = mf_kernel_user_resample(spec_n_lambdas, header_spec, header_kernel, kernel);
    if (kernel_new) {
        /* Get error code */
        cpl_error_code err = cpl_error_get_code();

        /* Normalize kernel */
        if (!err) {
            err = mf_kernel_user_normalize(kernel_new);
        }

        /* Check errors */
        if (err != CPL_ERROR_NONE) {
            cpl_matrix_delete(kernel_new);
            return NULL;
        }
    }

    return kernel_new;
}

/* ---------------------------------------------------------------------------*/
/**
 * @brief Apply user kernel
 *
 * @param spec                            in/out: convolved spectrum (CPL table)
 * @param kernel_resample_normalize       .
 * @param spec_rows_selected_to_convolve  CPL array with spectrum rows to be convolved
 *
 * @return cpl_error_code    CPL_ERROR_NONE is everything is OK.
 *                           If not, these are the errors:
 *                           - No data
 *                           - Invalid object structure
 *
 * @note Applies kernels read from a file to a model spectrum as provided by a
 *          CPL table with columns MF_COL_LAMBDA and MF_COL_FLUX. For each spectrum row, a
 *          kernel is expected. They have to be provided in matrix form via the
 *          mf_parameters parameter structure. All kernels have to have the same size and
 *          the kernel center should always be at the expected median position.
 *
 */
/* ---------------------------------------------------------------------------*/
cpl_error_code mf_kernel_user(
    cpl_table        *spec,
    const cpl_matrix *kernel_resample_normalize,
    const cpl_array  *spec_rows_selected_to_convolve
)
{
    /* Check number of data points in spectrum */
    cpl_size m = cpl_table_get_nrow(spec);
    if (m <= 0) {
        return cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND, "No data: cpl_table *spec");
    }

    /* Check existence of 'flux' column in spectrum table */
    if (cpl_table_has_column(spec, MF_COL_IN_FLUX) != 1) {
        return cpl_error_set_message(
            cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT, "Invalid object structure: cpl_table *spec (no 'flux' column)"
        );
    }

    /* Check number of selected spectrum pixels */
    cpl_size nsel = cpl_array_get_size(spec_rows_selected_to_convolve);
    if (nsel != m) {
        return cpl_error_set_message(
            cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
            "Invalid object structure: cpl_array *specrows (inconsistent with size of cpl_table *spec)"
        );
    }

    /* Get pointer to array of row numbers */
    const int *rows = cpl_array_get_data_int_const(spec_rows_selected_to_convolve);

    /* Check validity of row numbers */
    if (rows[0] < 0 || rows[nsel - 1] < rows[0] || rows[nsel - 1] - rows[0] + 1 != nsel) {
        return cpl_error_set_message(
            cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
            "Invalid object structure: cpl_array *specrows (no list of increasing row numbers)"
        );
    }

    /* Get pointer to matrix of kernel elements, and kernel size */
    const double *matkern = cpl_matrix_get_data_const(kernel_resample_normalize);
    cpl_size      npmax   = cpl_matrix_get_ncol(kernel_resample_normalize);
    cpl_size      nrow    = cpl_matrix_get_nrow(kernel_resample_normalize);

    /* Check number of rows in kernel matrix */
    if (nrow < nsel || nrow - 1 < rows[nsel - 1]) {
        return cpl_error_set_message(
            cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
            "Invalid object structure: cpl_matrix *kernel_resample_normalize (inconsistent with size of cpl_array "
            "*specrows)"
        );
    }

    /* Initialize CPL array for kernel and get pointer */
    cpl_array *kernel = cpl_array_new(npmax, CPL_TYPE_DOUBLE);
    cpl_array_fill_window_double(kernel, 0, npmax, 0.);
    double *kern = cpl_array_get_data_double(kernel);

    /* Copy flux column into CPL array */
    double    *vec  = cpl_table_get_data_double(spec, MF_COL_IN_FLUX);
    cpl_array *flux = cpl_array_new(m, CPL_TYPE_DOUBLE);
    cpl_array_copy_data_double(flux, vec);

    /* Create CPL array for convolved flux */
    cpl_array *convflux = cpl_array_new(m, CPL_TYPE_DOUBLE);
    cpl_array_fill_window_double(convflux, 0, m, 0.);

    /* Convolve spectrum with kernel(s) (pixel by pixel) */
    for (cpl_size i = 0; i < m; i++) {
        /* Set pixel range */
        int range[2] = { i, i };

        /* Copy kernel elements into array of correct length */
        for (cpl_size j = 0; j < npmax; j++) {
            cpl_size idx = j + npmax * rows[i];
            kern[j]      = matkern[idx];
        }

        /* Convolve pixel with kernel */
        mf_kernel_convolve(convflux, flux, range, kernel);
    }

    /* Copy resulting flux array into MF_COL_FLUX column of input CPL table */
    vec = cpl_array_get_data_double(convflux);
    cpl_table_copy_data_double(spec, MF_COL_IN_FLUX, vec);

    /* Cleanup */
    cpl_array_delete(kernel);
    cpl_array_delete(flux);
    cpl_array_delete(convflux);


    return CPL_ERROR_NONE;
}


/** @cond PRIVATE */

/* ---------------------------------------------------------------------------*/
/**
 * @brief .
 *
 * @param .                  .
 * @param                    .
 *
 * @return cpl_error_code    CPL_ERROR_NONE is everything is OK.
 *                           If not, these are the errors:
 *                           - .
 *                           - Error in subroutine (see subroutines).
 *
 * @description .
 *
 * @note .
 *
 */
/* ---------------------------------------------------------------------------*/
static cpl_matrix *mf_kernel_user_resample(
    const cpl_size          spec_n_lambdas,
    const cpl_propertylist *header_spec,
    const cpl_propertylist *header_kernel,
    const cpl_matrix       *kernel
)
{
    /* Check inputs */
    cpl_error_ensure(
        header_spec && header_kernel && kernel, CPL_ERROR_NULL_INPUT, return NULL, "Not provide lambdas in the spectrum"
    );

    /* Size specturm and kernel : Number of wavelengths  */
    cpl_size kernel_n_lambdas = cpl_matrix_get_nrow(kernel);
    cpl_size kernel_n_points  = cpl_matrix_get_ncol(kernel);
    cpl_error_ensure(
        spec_n_lambdas > 0 && kernel_n_lambdas > 0 && kernel_n_points > 0, CPL_ERROR_NULL_INPUT, return NULL,
        "Not provide lambdas in the spectrum"
    );

    /* Get Wavelength values: CRVAL1 (initial wavelength) and CD1_1 (step wavelength) */
    double spec_lambda_ini = cpl_propertylist_get_double(header_spec, MF_KEYWORD_CRVAL1);
    cpl_error_ensure(
        spec_lambda_ini >= MF_WAVELENGTH_MIN_MICRONS && spec_lambda_ini <= MF_WAVELENGTH_MAX_MICRONS,
        CPL_ERROR_INCOMPATIBLE_INPUT, return NULL, MF_KEYWORD_CRVAL1 "=%g invalid spectrum data in telluriccorr",
        spec_lambda_ini
    );

    /*  double spec_lambda_delta = cpl_propertylist_get_double(header_spec, MF_KEYWORD_CD1_1);*/
    /* NOTE: Old test data may still be using the DEPRECATED CDELT1 instead of CD_1_1 so until test data */
    /*       is upgraded we add a search for CDELT1 if CD_1_1 is missing and assert a warning */
    double spec_lambda_delta;
    if (cpl_propertylist_has(header_spec, MF_KEYWORD_CD1_1)) {
        spec_lambda_delta = cpl_propertylist_get_double(header_spec, MF_KEYWORD_CD1_1);
    }
    else {
        cpl_msg_warning(cpl_func, "Cannot find CD1_1 keyword. Will look for the DEPRECATED CDELT1 keyword instead!");
        spec_lambda_delta = cpl_propertylist_get_double(header_spec, MF_KEYWORD_CDELT1);
    }

    cpl_error_ensure(
        spec_lambda_delta > 0 && spec_lambda_ini + (spec_n_lambdas * spec_lambda_delta) <= MF_WAVELENGTH_MAX_MICRONS,
        CPL_ERROR_INCOMPATIBLE_INPUT, return NULL,
        MF_KEYWORD_CD1_1 "=%g invalid spectrum data in telluriccorr (wave_ini=%g , n_spec_wave=%lld)",
        spec_lambda_delta, spec_lambda_ini, spec_n_lambdas
    );

    /* Get Wavelength values in kernel : CRVAL1 (initial wavelength) and CD2_2 (step wavelength) */
    double kernel_lambda_ini = cpl_propertylist_get_double(header_kernel, MF_KEYWORD_CRVAL1);
    cpl_error_ensure(
        kernel_lambda_ini >= MF_WAVELENGTH_MIN_MICRONS && kernel_lambda_ini <= MF_WAVELENGTH_MAX_MICRONS,
        CPL_ERROR_INCOMPATIBLE_INPUT, return NULL, MF_KEYWORD_CRVAL1 "=%g invalid kernel data in telluriccorr",
        kernel_lambda_ini
    );

    double kernel_lambda_delta = cpl_propertylist_get_double(header_kernel, MF_KEYWORD_CD2_2);
    cpl_error_ensure(
        kernel_lambda_delta > 0 &&
            kernel_lambda_ini + (kernel_n_lambdas * kernel_lambda_delta) <= MF_WAVELENGTH_MAX_MICRONS,
        CPL_ERROR_INCOMPATIBLE_INPUT, return NULL,
        MF_KEYWORD_CD2_2 "=%g invalid kernel data in telluriccorr (kernel_ini=%g , n_kernel_wave=%lld)",
        kernel_lambda_delta, kernel_lambda_ini, kernel_n_lambdas
    );

    /* Get and check variables */

    double diff_lambda_ini   = fabs(spec_lambda_ini - kernel_lambda_ini);
    double diff_lambda_delta = fabs(spec_lambda_delta - kernel_lambda_delta);

    cpl_error_ensure(
        spec_n_lambdas > 0 && spec_lambda_ini > 0 && spec_lambda_delta > 0 && kernel_n_lambdas > 0 &&
            kernel_lambda_ini > 0 && kernel_lambda_delta > 0 && kernel_n_points > 0,
        CPL_ERROR_ILLEGAL_INPUT, return NULL, "Illegal value in the inputs headers"
    );

    /* Check if it's necessary resample the kernel */
    cpl_matrix *kernel_resample;
    if (spec_n_lambdas == kernel_n_lambdas && diff_lambda_ini <= MF_KERNEL_USER_DIFF_LAMBDA_INI_TOL &&
        diff_lambda_delta <= MF_KERNEL_USER_DIFF_LAMBDA_DELTA_TOL) {
        /*** Not resampling kernel ***/
        cpl_msg_info(
            cpl_func,
            "(mf_kernel    ) Not need kernel RESAMPLE : n_lambdas(SPEC=%lld, KERNEL=%lld), diff_lambda_ini=%g, "
            "diff_lambda_delta=%g",
            spec_n_lambdas, kernel_n_lambdas, diff_lambda_ini, diff_lambda_delta
        );

        kernel_resample = cpl_matrix_duplicate(kernel);
    }
    else {
        /*** Resampling kernel ***/
        cpl_msg_info(
            cpl_func,
            "(mf_kernel    ) Kernel RESAMPLING ... n_lambdas(SPEC=%lld, KERNEL=%lld), diff_lambda_ini=%g, "
            "diff_lambda_delta=%g ",
            spec_n_lambdas, kernel_n_lambdas, diff_lambda_ini, diff_lambda_delta
        );

        /* Save original kernel and create the new kernel resampled */
        kernel_resample = cpl_matrix_new(spec_n_lambdas, kernel_n_points);

        /* Get access to rows in the matrix */
        const double *kernel_old = cpl_matrix_get_data_const(kernel);
        double       *kernel_new = cpl_matrix_get_data(kernel_resample);

        /* Loop about lambdas */
        double   lambdaSpec     = spec_lambda_ini;
        cpl_size kernel_old_idx = 0;
        for (cpl_size i = 0; i < spec_n_lambdas; i++) {
            /* Initialize index */
            cpl_size kernel_old_idx_i = kernel_old_idx;
            cpl_size kernel_old_idx_j = kernel_old_idx;

            /* Get index kernels: top/down */
            double lambdaKernel_ini = kernel_lambda_ini + kernel_old_idx_i * kernel_lambda_delta;
            double lambdaKernel_end = lambdaKernel_ini;
            while (lambdaKernel_end < lambdaSpec) {
                if (lambdaKernel_end + kernel_lambda_delta < lambdaSpec) {
                    kernel_old_idx_i++;
                }
                kernel_old_idx_j++;
                lambdaKernel_end += kernel_lambda_delta;
            }

            /* The last position doesn't match between spectrum and kernel, Not resampling for avoid access beyond the boundaries */
            if (kernel_old_idx_j + 1 > kernel_n_lambdas) {
                kernel_old_idx_j = kernel_old_idx_i;
                lambdaKernel_end = lambdaKernel_ini;
            }

            /* Update index */
            kernel_old_idx = kernel_old_idx_i;

            /* Get Row and high resolution signal */
            cpl_size kernel_new_index   = i * kernel_n_points;
            cpl_size kernel_old_index_1 = kernel_old_idx_i * kernel_n_points;
            cpl_size kernel_old_index_2 = kernel_old_idx_j * kernel_n_points;

            /* Check if (i==j) --> (init/end cases) */
            if (kernel_old_idx_i == kernel_old_idx_j) {
                /* Copy values related with k_old_index_1 */
                for (cpl_size j = 0; j < kernel_n_points; j++) {
                    kernel_new[j + kernel_new_index] = kernel_old[j + kernel_old_index_1];
                }
            }
            else {
                /* Apply linear interpolation */
                double x  = lambdaSpec;
                double x1 = lambdaKernel_ini;
                double x2 = lambdaKernel_end;
                for (cpl_size j = 0; j < kernel_n_points; j++) {
                    double y1 = kernel_old[j + kernel_old_index_1];
                    double y2 = kernel_old[j + kernel_old_index_2];

                    kernel_new[j + kernel_new_index] = y1 + (x - x1) * (y2 - y1) / (x2 - x1);
                }
            }

            /* Next lambda */
            lambdaSpec += spec_lambda_delta;
        }
    }

    return kernel_resample;
}

/* ---------------------------------------------------------------------------*/
/**
 * @brief .
 *
 * @param .                  .
 * @param                    .
 *
 * @return cpl_error_code    CPL_ERROR_NONE is everything is OK.
 *                           If not, these are the errors:
 *                           - .
 *                           - Error in subroutine (see subroutines).
 *
 * @description .
 *
 * @note .
 *
 */
/* ---------------------------------------------------------------------------*/
static cpl_error_code mf_kernel_user_normalize(cpl_matrix *kernel)
{
    /* Check inputs */
    cpl_error_ensure(kernel, CPL_ERROR_NULL_INPUT, return CPL_ERROR_NULL_INPUT, "Not provide kernel");

    /*** Normalize kernel ***/
    cpl_msg_info(cpl_func, "(mf_kernel    ) Kernel NORMALIZING ... ");
    for (cpl_size row = 0; row < cpl_matrix_get_nrow(kernel); row++) {
        /* Calculate sum of the row */
        double sum_row = 0.;
        for (cpl_size col = 0; col < cpl_matrix_get_ncol(kernel); col++) {
            sum_row += cpl_matrix_get(kernel, row, col);
        }

        /* Divided by sum of the row */
        for (cpl_size col = 0; col < cpl_matrix_get_ncol(kernel); col++) {
            cpl_matrix_set(kernel, row, col, cpl_matrix_get(kernel, row, col) / sum_row);
        }
    }

    return cpl_error_get_code();
}

/* ---------------------------------------------------------------------------*/
/**
 * @brief .
 *
 * @param .                  .
 * @param                    .
 *
 * @return cpl_error_code    CPL_ERROR_NONE is everything is OK.
 *                           If not, these are the errors:
 *                           - .
 *                           - Error in subroutine (see subroutines).
 *
 * @description .
 *
 * @note .
 *
 */
/* ---------------------------------------------------------------------------*/
static cpl_error_code
mf_kernel_convolve(cpl_array *convflux, const cpl_array *flux, const int range[2], const cpl_array *kernel)
{
    /*!
     * Convolution of a flux array with given kernel for a fixed range of
     * pixels. The output array has to exist and have the same number of
     * pixels as the input array. Technically, the convolution is carried out
     * by adding the convolved fluxes for each pixel in the selected window.
     *
     * \note The center of the convolution function is shifted by -0.5 pixels
     *       for an even number of kernel pixels.
     *
     * \b INPUT:
     * \param convflux  CPL array of convolved flux values
     * \param flux      CPL array of unconvolved input flux
     * \param range     range of pixels to be considered (minimum and maximum)
     * \param kernel    kernel as CPL array
     *                  (required: sum of all values = 1)
     *
     * \b OUTPUT:
     * \param convflux  output array with added convolved flux from given
     *                  range
     *
     * \b ERRORS:
     * - Invalid object structure
     * - Invalid input parameter(s)
     * - Invalid object value(s)
     */

    /* No data points -> no convolution */
    cpl_size n = cpl_array_get_size(flux);
    if (n <= 0) {
        return CPL_ERROR_NONE;
    }

    /* Check correspondence of data points */
    if (cpl_array_get_size(convflux) != n) {
        return cpl_error_set_message(
            cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
            "Invalid object structure: cpl_array *convflux != cpl_array *flux (size)"
        );
    }

    /* Check range values */
    if (range[0] > range[1]) {
        return cpl_error_set_message(
            cpl_func, CPL_ERROR_ILLEGAL_INPUT, "Invalid input parameter(s): range[2] (min. > max.)"
        );
    }

    if (range[0] < 0 || range[1] >= n) {
        return cpl_error_set_message(
            cpl_func, CPL_ERROR_ILLEGAL_INPUT, "Invalid input parameter(s): range[2] (invalid pixel)"
        );
    }

    /* Initialize output array if not done so far */

    if (cpl_array_has_invalid(convflux) == 1) {
        cpl_array_fill_window_double(convflux, 0, n, 0.);
    }

    /* Get pointers to CPL arrays */
    const double *influx  = cpl_array_get_data_double_const(flux);
    double       *outflux = cpl_array_get_data_double(convflux);

    /* No kernel data -> no convolution */
    cpl_size nkpix = cpl_array_get_size(kernel);
    if (nkpix == 0) {
        for (cpl_size i = range[0]; i <= range[1]; i++) {
            outflux[i] += influx[i];
        }
        return CPL_ERROR_NONE;
    }

    /* Get pointer to CPL array */
    const double *kern = cpl_array_get_data_double_const(kernel);

    /* Check kernel */
    double sum = 0.;
    for (cpl_size k = 0; k < nkpix; k++) {
        if (kern[k] < 0 || kern[k] > 1) {
            for (cpl_size i = range[0]; i <= range[1]; i++) {
                outflux[i] += influx[i];
            }
            return cpl_error_set_message(
                cpl_func, CPL_ERROR_ILLEGAL_INPUT,
                "Invalid object value(s): cpl_array *kernel (kernel element(s) < 0 or > 1)"
            );
        }
        sum += kern[k];
    }

    if (sum < 1 - MF_TOL || sum > 1 + MF_TOL) {
        for (cpl_size i = range[0]; i <= range[1]; i++) {
            outflux[i] += influx[i];
        }
        return cpl_error_set_message(
            cpl_func, CPL_ERROR_ILLEGAL_INPUT,
            "Invalid object value(s): cpl_array *kernel (sum of kernel elements != 1)"
        );
    }

    /* Skip convolution if number of kernel pixels is one */
    if (nkpix == 1) {
        for (cpl_size i = range[0]; i <= range[1]; i++) {
            outflux[i] += influx[i];
        }
        return CPL_ERROR_NONE;
    }

    /* Kernel with even or odd pixel number? Note: center of kernel at -0.5 pixels for even pixel number */
    cpl_size kmin = (nkpix % 2 == 0) ? -nkpix / 2 : -(nkpix - 1) / 2;
    cpl_size kmax = kmin + nkpix - 1;

    /* Set pixel range (add virtual pixels for marginal ranges) */
    cpl_size jmin = (range[0] == 0) ? -kmax : range[0];
    cpl_size jmax = (range[1] == n - 1) ? n - 1 - kmin : range[1];

    /* Set flux of virtual input pixels */
    double in0   = influx[0];
    double innm1 = influx[n - 1];

    /* Convolve array with kernel */
    for (cpl_size j = jmin; j <= jmax; j++) {
        /* Flux of real and virtual input pixels */
        double in;
        if (j < 0) {
            in = in0;
        }
        else if (j >= n) {
            in = innm1;
        }
        else {
            in = influx[j];
        }

        /* Calculate output flux for each kernel element and add it to the corresponding output pixel */
        for (cpl_size k = CPL_MAX(kmin, -j); k <= CPL_MIN(kmax, n - 1 - j); k++) {
            cpl_size i = j + k;
            outflux[i] += in * kern[k - kmin];
        }
    }

    return CPL_ERROR_NONE;
}

/** @endcond */


/**@}*/
