/*
 * This file is part of the ESO Telluric Correction Library
 * Copyright (C) 2001-2018 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

/*----------------------------------------------------------------------------*/
/**
 *                              Includes
 */
/*----------------------------------------------------------------------------*/

#include "mf_wrap.h"
struct mf_wrap_model_parameter;

/*----------------------------------------------------------------------------*/
/**
 *                 Typedefs: Enumeration types
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Defines
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Global variables
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Macros
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Typedefs: Structured types
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Functions prototypes
 */
/*----------------------------------------------------------------------------*/

cpl_error_code set_output_filename_suffix(const char *suffix, cpl_parameterlist *parlist)
{
    /* TODO: This function is never used; remove it? */

    // for xshooter / multiobject stuff
    // create the tag to be checked ?
    cpl_parameter *file_suffix = cpl_parameter_new_value("tag", CPL_TYPE_STRING, "ARM", "string context", suffix);
    // file_suffix = suffix;  // HOWEVER THE PARAMETERS ARE SET
    cpl_error_code err = cpl_parameterlist_append(parlist, file_suffix);

    return err;
}

// WRITE THIS PART !!!
/*cpl_error_code map_tags_to_mf_wrap(cpl_frameset * frameset, const char * tags){
    // iterate through the framset names to find any in the tags list, and if found, set that value to MOLECFIT_SCIENCE etc

    cpl_error_code err = CPL_ERROR_NONE;
    framset
    //checks for multiple type files can be added in later.


    return err;
}*/

/* TODO: mf_wrap_model_data is never used. Why is it here? */
__attribute__((unused)) cpl_error_code mf_wrap_model_data(const char *fname, mf_wrap_fits *data)
{
    //mf_wrap_fits * mf_wrap_data(cpl_frame *input_frame, cpl_parameterlist *parlist, cpl_boolean *use_only_input_pri_ext){

    cpl_error_code err;

    //cpl_boolean science_data_is_table= CPL_FALSE;
    /* Checking the SCIENCE file data type eg is it in table form */
    //science_data_is_table= mf_wrap_config_chk_science_is_table(input_frame);
    //if ( science_data_is_table) cpl_msg_info(cpl_func,"Science frame is in table data format");
    //if (!science_data_is_table) cpl_msg_info(cpl_func,"Science frame is not in table data format");

    //mf_wrap_fits *data = NULL;

    //const char *data_file = cpl_frame_get_filename(input_frame);
    cpl_msg_info(cpl_func, "Reading in spectrum from %s", fname);

    data = mf_wrap_fits_load(fname, CPL_FALSE);
    /*if(data == NULL){
            cpl_msg_info(cpl_func,"mf_wrap_fits* data == NULL");
        } else {
            cpl_msg_info(cpl_func,"mf_wrap_fits: filename %s",data->filename);
            cpl_msg_info(cpl_func,"mf_wrap_fits: n_ext %d",data->n_ext);
            //cpl_vector
            cpl_msg_info(cpl_func,"mf_wrap_fits: v_ext[0].vector size %lld",cpl_vector_get_size(data->v_ext[0].vector));

            //cpl_msg_info(cpl_func,"mf_wrap_fits: format %s",data->format);
            cpl_msg_info(cpl_func,"mf_wrap_fits_load returned results");
        }*/
    /*cpl_size nrows = cpl_table_get_nrow(data->v_ext[1].spectrum_data);
        cpl_msg_info(cpl_func,"nrows: %d",nrows); 
        cpl_table_dump(data->v_ext[1].spectrum_data,0,nrows,stdout);
        */


    if (!data) {
        err = CPL_ERROR_ILLEGAL_INPUT;
    }
    else {
        err = cpl_error_get_code();
    }
    return err;
}

cpl_error_code mf_wrap_data_check(
    mf_wrap_fits            *data,
    const cpl_parameterlist *parlist,
    mf_wrap_model_parameter *parameters,
    const cpl_boolean       *use_only_input_pri_ext
)
{
    /* If the data file has been loaded successfully check the header keywords */

    cpl_error_code err = CPL_ERROR_NONE;
    /* Check that the following keywords (if user specified) do exists in the fits data header */
    cpl_error_code kwd_err = CPL_ERROR_NONE;
    kwd_err += mf_wrap_config_check_keyword(data, parlist, MF_PARAMETERS_OBSERVATORY_ERF_RV_KEYWORD);
    kwd_err += mf_wrap_config_check_keyword(data, parlist, MF_PARAMETERS_OBSERVING_DATE_KEYWORD);
    kwd_err += mf_wrap_config_check_keyword(data, parlist, MF_PARAMETERS_UTC_KEYWORD);
    kwd_err += mf_wrap_config_check_keyword(data, parlist, MF_PARAMETERS_TELESCOPE_ANGLE_KEYWORD);
    kwd_err += mf_wrap_config_check_keyword(data, parlist, MF_PARAMETERS_RELATIVE_HUMIDITY_KEYWORD);
    kwd_err += mf_wrap_config_check_keyword(data, parlist, MF_PARAMETERS_PRESSURE_KEYWORD);
    kwd_err += mf_wrap_config_check_keyword(data, parlist, MF_PARAMETERS_TEMPERATURE_KEYWORD);
    kwd_err += mf_wrap_config_check_keyword(data, parlist, MF_PARAMETERS_MIRROR_TEMPERATURE_KEYWORD);
    kwd_err += mf_wrap_config_check_keyword(data, parlist, MF_PARAMETERS_ELEVATION_KEYWORD);
    kwd_err += mf_wrap_config_check_keyword(data, parlist, MF_PARAMETERS_LONGITUDE_KEYWORD);
    kwd_err += mf_wrap_config_check_keyword(data, parlist, MF_PARAMETERS_LATITUDE_KEYWORD);
    kwd_err += mf_wrap_config_check_keyword(data, parlist, MF_PARAMETERS_SLIT_WIDTH_KEYWORD);
    kwd_err += mf_wrap_config_check_keyword(data, parlist, MF_PARAMETERS_PIXEL_SCALE_KEYWORD);

    if (kwd_err) {
        cpl_msg_error(cpl_func, "Header Keyword Names specifed in the .rc file not found in the target science frame!");
        err = CPL_ERROR_ILLEGAL_INPUT;
    }

    /* ----------------------------------------------------------------------------------------------------*/
    /* Convert the imported fits file data which could be in one of many different formats into cpl tables */
    /* that molecfit/telluriccorr can process                                                              */
    /* ----------------------------------------------------------------------------------------------------*/
    /* Note: the imported fits data is already contained in the data structure *data    */
    /* Note: the new cpl_tables will be added to the structure *data                    */
    /* Note in the event of chip_extension=true, all extension data in the fits file is merged into a single cpl_table  */

    /* First check that all required column names exists (This test is only if USE_ONLY_PRIMARY_DATA=FALSE)*/

    //cpl_boolean science_data_is_table= CPL_FALSE;
    //science_data_is_table= mf_wrap_config_chk_science_is_table(data->filename);  // <- this check should already be done..
    //we might not need to do this check....

    if (!use_only_input_pri_ext) {  // && science_data_is_table) {
        cpl_error_code col_err = CPL_ERROR_NONE;

        col_err |= mf_wrap_config_chk_column(data, parameters->mf_config->parameters->inputs.column_lambda);
        col_err |= mf_wrap_config_chk_column(data, parameters->mf_config->parameters->inputs.column_flux);
        col_err |= mf_wrap_config_chk_column(data, parameters->mf_config->parameters->inputs.column_dflux);
        col_err |= mf_wrap_config_chk_column(data, parameters->mf_config->parameters->inputs.column_mask);
        if (col_err) {
            err = CPL_ERROR_ILLEGAL_INPUT;
        }
    }

    return err;
}


/*----------------------------------------------------------------------------*/
/**
 * @brief    Load the molecules table required to call mf_model 
 *
 * @param    frameset   the frames list
 * @param    parlist    the parameters list
 * @param    parameters the molecfit parameters list
 *
 * @return   the molecules cpl_table 
 *
 */
/*----------------------------------------------------------------------------*/
cpl_table *mf_wrap_molecules(
    cpl_frameset            *frameset,
    mf_wrap_fits            *data,
    const cpl_parameterlist *parlist,
    mf_wrap_model_parameter *parameters,
    const char              *suffix
)
{
    //cpl_errorstate initial_errorstate = cpl_errorstate_get();
    cpl_error_code err;

    const char *input_tag    = mf_wrap_tag_suffix(MOLECFIT_MOLECULES, suffix, CPL_FALSE);
    const char *output_fname = mf_wrap_tag_suffix(MOLECFIT_MODEL_MOLECULES, suffix, CPL_TRUE);

    /* Load TAG = MOLECULES */
    cpl_table *molecules = NULL;
    /* Get MOLECUTES table */
    cpl_msg_info(cpl_func, "Loading %s cpl_table", input_tag);
    if (parameters->molecules_table) {
        //cpl_msg_info(cpl_func,"parameters->molecules_table is NOT NULL");
        molecules = cpl_table_duplicate(parameters->molecules_table);
    }
    else {
        //cpl_msg_info(cpl_func,"parameters->molecules_table is NULL");
        cpl_msg_info(cpl_func, "Loading %s cpl_table", input_tag);
        molecules = mf_wrap_load_unique_table(frameset, input_tag);
    }
    //cpl_msg_info(cpl_func,"DUMP MOLECULES TABLE AFTER 'Get MOLECUTES table'");
    //cpl_table_dump(molecules,0,cpl_table_get_nrow(molecules),stdout);
    //OK to here...

    if (!molecules) {
        //jerr = CPL_ERROR_ILLEGAL_INPUT;
        // cpl_error_set_message(cpl_func, err, "Bad input");
        return NULL;
    }
    if ((int)cpl_table_get_column_max(molecules, "FIT_MOLEC") < 1) {
        cpl_msg_warning(cpl_func, "No molecules to fit !");
    }
    err = cpl_error_get_code();
    if (err) {
        return NULL;
    }

    /* Now check that selected Reference Atmosphere file has the required molecules */
    err = mf_config_chk_ref_atmos(
        parameters->mf_config->parameters->directories.telluriccorr_data_path,
        parameters->mf_config->parameters->atmospheric.ref_atm, molecules
    );
    if (err) {
        cpl_msg_info(cpl_func, "Bad Specifications of Molecules and Reference Atmosphere");
        // err = CPL_ERROR_ILLEGAL_INPUT;
        // cpl_error_set_message(cpl_func, err, "Bad Specifications of Molecules and Reference Atmosphere");
        return NULL;
    }

    //cpl_msg_info(cpl_func,"after 'Reference Atmosphere' check: %s",cpl_error_get_message());

    /* Save MOLECULES table */
    const char *tag_name = mf_wrap_tag_suffix(MOLECFIT_MODEL_MOLECULES, suffix, CPL_FALSE);
    err = mf_wrap_save(frameset, frameset, parlist, RECIPE_NAME, parameters->pl, tag_name, output_fname);
    //if (!err) err = mf_wrap_save_mf_results(data->v_ext[0].header, "MOLECFIT_MOLECULES_brent.fits", CPL_TRUE, NULL, molecules, NULL);
    if (err) {
        return NULL;
    }
    err = mf_wrap_save_mf_results(data->v_ext[0].header, output_fname, CPL_FALSE, NULL, molecules, NULL);
    if (err) {
        return NULL;
    }
    //if (!err) err = mf_wrap_save_mf_results(data->v_ext[0].header, MOLECFIT_MODEL_MOLECULES, CPL_TRUE, NULL, molecules, NULL);

    cpl_msg_info(cpl_func, "after 'Save MOLECULES table': %s", cpl_error_get_message());

    return molecules;
}


/*load up inc_wranges/exc_wranges/exc_pixranges*/
cpl_error_code mf_wrap_wranges(
    cpl_frameset            *frameset,
    mf_wrap_fits            *data,
    const cpl_parameterlist *parlist,
    mf_wrap_model_parameter *parameters,
    cpl_table              **inc_wranges,
    cpl_table              **exc_wranges,
    cpl_table              **exc_pranges,
    const char              *suffix
)
{
    /* TODO: ADD mf_wrap_model_amalgamate_ranges_table to mf_wrap_utils... 
       return the three cpl_tables by reference? */
    /* ------------------------------------------------------------*/
    /* Sort the wavlength ranges and any range specific parameters */
    /* ------------------------------------------------------------*/

    cpl_error_code err = CPL_ERROR_NONE;
    cpl_msg_info(cpl_func, "before 'Obtain all relevant parameters': %s", cpl_error_get_message());

    /* Obtain all relevant parameters as defined in the .rc file */
    const char *wave_include_str = cpl_parameter_get_string(cpl_parameterlist_find_const(parlist, "WAVE_INCLUDE"));
    const char *map_regions_str =
        cpl_parameter_get_string(cpl_parameterlist_find_const(parlist, "MAP_REGIONS_TO_CHIP"));
    const char *fit_continuum_str = cpl_parameter_get_string(cpl_parameterlist_find_const(parlist, "FIT_CONTINUUM"));
    const char *continuum_n_str   = cpl_parameter_get_string(cpl_parameterlist_find_const(parlist, "CONTINUUM_N"));
    const char *fit_wlc_str       = cpl_parameter_get_string(cpl_parameterlist_find_const(parlist, "FIT_WLC"));
    double      wlg2mn            = cpl_parameter_get_double(cpl_parameterlist_find_const(parlist, "WLG_TO_MICRON"));
    cpl_msg_info(
        cpl_func,
        "mf_wrap_wranges WAVE_INCLUDE %s MAP_REGIONS_TO_CHIP %s FIT_CONTINUUM %s CONTINUUM_N %s FIT_WLC %s "
        "WLG_TO_MICRON %.3f",
        wave_include_str, map_regions_str, fit_continuum_str, continuum_n_str, fit_wlc_str, wlg2mn
    );
    cpl_table *inc_wranges_ext_table = NULL;
    //setup internal variables for inc_wranges, exc_wranges and exc_pranges
    //at the end, we duplicate them to set their cpl_table** params
    cpl_table *inc_wranges_int = NULL;
    cpl_table *exc_wranges_int = NULL;
    cpl_table *exc_pranges_int = NULL;
    if (inc_wranges != NULL) {
        *inc_wranges = NULL;
    }
    if (exc_wranges != NULL) {
        *exc_wranges = NULL;
    }
    if (exc_pranges != NULL) {
        *exc_pranges = NULL;
    }

    int n_chip_subranges = -1; /* -1 Implies that chip subranges are not being used */

    cpl_msg_info(cpl_func, "before 'Determine if using CHIPS mechanism': %s", cpl_error_get_message());
    /* ---------------------------------------------------------------------*/
    /* Determine if using CHIPS mechanism and if so how many chips are there*/
    /* -------------------------------------------------------------------- */
    if (!err) {
        /* If USECHIPS flag is true then extensions are to be treated as "CHIP" sub ranges of
          a full spectrum and chip specfic parameters must be checked.        */
        /* Use chips mechanism only if chip_extension flag is true and use_oly_pri is false*/
        if (!parameters->use_only_input_pri_ext && parameters->chip_extensions) {
            /* No of chip subregions is the number of extensions (not counting the primary) */
            n_chip_subranges = data->n_ext - 1;
        }

        if (n_chip_subranges > 0) {
            cpl_msg_info(cpl_func, "Use CHIPS flag set to true");
            cpl_msg_info(cpl_func, "Use only primary data flag set to false.");
            cpl_msg_info(
                cpl_func, "Will treat the %d extensions of the data file as subregions of full spectrum",
                n_chip_subranges
            );

        } /* end if n_chip_subranges>0*/
    }

    if (!err) {
        cpl_msg_info(cpl_func, "before 'Load up on any external fits table': %s", cpl_error_get_message());
        /* Load up on any external fits table or previously defined table which may have some or even all required data*/
        if (parameters->wave_ranges_include_table) {
            inc_wranges_ext_table = cpl_table_duplicate(parameters->wave_ranges_include_table);
        }
        else {
            const char *input_tag = mf_wrap_tag_suffix(MOLECFIT_WAVE_INCLUDE, suffix, CPL_FALSE);
            cpl_msg_info(cpl_func, "Loading %s cpl_table as external", input_tag);
            inc_wranges_ext_table = mf_wrap_load_unique_table(frameset, input_tag);
        }

        /* Merge all the .rc string parameters and table data into a single inc_wranges table*/

        //CHIPS ISSUE TEMPORARY FIX FOR XSHOOTER -
        int nchips = (1 > n_chip_subranges) ? 1 : n_chip_subranges;
        // CHIPS TEMPORARY FIX END ---

        cpl_msg_info(cpl_func, "before 'mf_wrap_model_amalgamate_ranges_table': %s", cpl_error_get_message());
        inc_wranges_int = mf_wrap_model_amalgamate_ranges_table(
            inc_wranges_ext_table, wave_include_str, map_regions_str, fit_continuum_str, continuum_n_str, fit_wlc_str,
            nchips, wlg2mn
        );
        cpl_msg_info(cpl_func, "after 'mf_wrap_model_amalgamate_ranges_table': %s", cpl_error_get_message());
        cpl_msg_info(cpl_func, "dump inc_wranges table");
        cpl_table_dump(inc_wranges_int, 0, cpl_table_get_nrow(inc_wranges_int), stdout);

        cpl_msg_info(cpl_func, "before 'Free up the external table': %s", cpl_error_get_message());
        /* Free up the external table*/
        if (inc_wranges_ext_table != NULL) {
            cpl_free(inc_wranges_ext_table);
        }

        if (inc_wranges_int == NULL) {
            cpl_msg_error(cpl_func, "Unable to construct full wavelength range table!");
            err = CPL_ERROR_NULL_INPUT;
        }
        else {
            err = cpl_error_get_code();
        }
        cpl_msg_info(cpl_func, "dump inc_wranges table2");
        cpl_table_dump(inc_wranges_int, 0, cpl_table_get_nrow(inc_wranges_int), stdout);
    }

    cpl_msg_info(cpl_func, "before next block %s", cpl_error_get_message());

    /* -----------------------------------------------------------------------------*/
    /* Define the initial coefficient values to use for the WLC and Continuum models*/
    /* NOTE: May be overriden by EXPERT MODE lower down */
    /* -----------------------------------------------------------------------------*/

    if (!err) {
        /* The number of ranges is now defined in the inc_wranges table */
        cpl_size n_ranges = cpl_table_get_nrow(inc_wranges_int);

        /* Default initial shift value shuld be define in the parameters structure */
        double const_value = parameters->mf_config->parameters->fitting.fit_continuum.const_val;

        /* Go through each range and define the initial poly as a shift by the const_value*/
        int max_porder = 0;
        for (int i = 0; i < n_ranges; i++) {
            /* Get the polynomial order for this range and set all coeffs to zero*/
            int porder = cpl_table_get_int(inc_wranges_int, MF_COL_WAVE_RANGE_CONT_ORDER, i, NULL);
            if (porder > max_porder) {
                max_porder = porder;
            }
            for (int j = 0; j <= porder; j++) {
                parameters->mf_config->parameters->fitting.cont_coeffs[i][j] = 0.0;
            }

            /* Now set as y=x+shift*/
            parameters->mf_config->parameters->fitting.cont_coeffs[i][0] = const_value;
            parameters->mf_config->parameters->fitting.cont_coeffs[i][1] = 1.0;
        }

        /* Originally there was only one poly order for all ranges and it was defined in: */
        /*       parameters->mf_config->parameters->fitting.fit_continuum.n               */
        /* This parameter is still used to allocate array sizes for the coeffs in the     */
        /* the range tables passed to molecfit execution. It would appear that array      */
        /* columns in tables must have the same dimension so we change the meaning of     */
        /* this parameter to be the maximum poly order of all the ranges                  */
        parameters->mf_config->parameters->fitting.fit_continuum.n = max_porder;


        /* Default initial shift value shuld be define in the parameters structure */
        int      n_chips     = (1 > n_chip_subranges) ? 1 : n_chip_subranges;
        cpl_size porder      = parameters->mf_config->parameters->fitting.fit_wavelenght.n;
        double   const_shift = parameters->mf_config->parameters->fitting.fit_wavelenght.const_val;

        cpl_msg_info(
            cpl_func, "N chips=%d, porder = %" CPL_SIZE_FORMAT " const_shift=%f", n_chips, porder, const_shift
        );

        /* Go through each chip and define the initial poly as a shift by shft)value*/
        for (int i = 0; i < n_chips; i++) {
            for (int j = 0; j <= porder; j++) {
                parameters->mf_config->parameters->fitting.wlc_coeffs[i][j] = 0.0;
            }

            /* Now set as y = x+shift */
            parameters->mf_config->parameters->fitting.wlc_coeffs[i][0] = const_shift;
            parameters->mf_config->parameters->fitting.wlc_coeffs[i][1] = 1.0;
        }
    }


    cpl_msg_info(
        cpl_func, "before 'Temporaray solution to fit chips and fit ranges flag mechanism' %s", cpl_error_get_message()
    );


    /* ------------------------------------------------------------*/
    /* Temporary solution to fit chips and fit ranges flag mechanism */
    /* ------------------------------------------------------------*/

    if (!err) {
        /* 
         We are still using the halfway house code which overrides wlc fit flags and 
         continuum fit flags in subroutine:
                                 mf_spectrum_replace_coefficientstine
         with parameters:
                          params->config->fitting.fit_ranges[i]
                          params->config->fitting.fit_chips[i]
      */

        /* Sort out the fit_ranges parameter */
        cpl_size n_ranges = cpl_table_get_nrow(inc_wranges_int);
        for (int i = 0; i < n_ranges; i++) {
            cpl_boolean flag = CPL_TRUE;
            int         ival = cpl_table_get_int(inc_wranges_int, MF_COL_WAVE_RANGE_CONT_FIT, i, NULL);
            if (ival == 0) {
                flag = CPL_FALSE;
            }
            parameters->mf_config->parameters->fitting.fit_ranges[i] = flag;
            int order = cpl_table_get_int(inc_wranges_int, MF_COL_WAVE_RANGE_CONT_ORDER, i, NULL);
            parameters->mf_config->parameters->fitting.cont_poly_order[i] = order;
        }

        /* Sort out the fit chips parameter */
        int n_chips = (1 > n_chip_subranges) ? 1 : n_chip_subranges;

        cpl_boolean fitv[n_chips];
        for (int i = 0; i < n_chips; i++) {
            fitv[i] = CPL_FALSE;
        }

        for (int i = 0; i < n_ranges; i++) {
            int ival = cpl_table_get_int(inc_wranges_int, MF_COL_WAVE_RANGE_WLC_FIT, i, NULL);
            if (ival == 0) {
                continue;
            }
            int chip_no  = cpl_table_get_int(inc_wranges_int, MF_COL_WAVE_RANGE_MAP2CHIP, i, NULL);
            int chip_idx = chip_no - 1;
            if (chip_idx < 0 || chip_idx > n_chips) {
                continue;
            }
            fitv[chip_idx] = CPL_TRUE;
        }

        for (int i = 0; i < n_chips; i++) {
            parameters->mf_config->parameters->fitting.fit_chips[i] = fitv[i];
            cpl_msg_info(cpl_func, "Sorting chip fits chip %d fit flag = %d", i + 1, fitv[i]);
        }
    }


    /* ------------------------------------------------- ------------*/
    /* END Sort the wavelength ranges and any associated RANGE flags */
    /* ------------------------------------------------- ------------*/

    /* ------------------------------------------------- ------------*/
    /* Sort any conflicts between the RANGES and the CHIPS           */
    /* ------------------------------------------------- ------------*/
    // REMOVED - FILL IN IF NEEDED.

    /* ------------------------------------------------- ------------*/
    /* END Sort any conflicts between the RANGES and the CHIPS       */
    /* ------------------------------------------------- ------------*/

    cpl_msg_info(cpl_func, "after 'lots of end comments' / before 'load tables'%s", cpl_error_get_message());

    if (!err) {
        if (parameters->wave_ranges_exclude_table) {
            cpl_msg_info(cpl_func, "parameters->wave_ranges_exclude_table is NOT NULL %s", cpl_error_get_message());
            exc_wranges_int = cpl_table_duplicate(parameters->wave_ranges_exclude_table);
        }
        else {
            const char *input_tag = mf_wrap_tag_suffix(MOLECFIT_WAVE_EXCLUDE, suffix, CPL_FALSE);
            cpl_msg_info(cpl_func, "parameters->wave_ranges_exclude_table is NULL %s", cpl_error_get_message());
            cpl_msg_info(cpl_func, "Loading %s cpl_table", input_tag);
            exc_wranges_int = mf_wrap_load_unique_table(frameset, input_tag);
        }

        if (parameters->pixel_ranges_exclude_table) {
            cpl_msg_info(cpl_func, "parameters->pixel_ranges_exclude_table is NOT NULL %s", cpl_error_get_message());
            exc_pranges_int = cpl_table_duplicate(parameters->pixel_ranges_exclude_table);
        }
        else {
            const char *input_tag = mf_wrap_tag_suffix(MOLECFIT_PIXEL_EXCLUDE, suffix, CPL_FALSE);
            cpl_msg_info(cpl_func, "parameters->pixel_ranges_exclude_table is NULL %s", cpl_error_get_message());
            cpl_msg_info(cpl_func, "Loading %s cpl_table", input_tag);
            exc_pranges_int = mf_wrap_load_unique_table(frameset, input_tag);
        }

        err = cpl_error_get_code();
    }

    cpl_msg_info(cpl_func, "before 'Save WV_INCLUDE RANGE's table' %s", cpl_error_get_message());
    /* Save WV_INCLUDE RANGE's table */
    if (!err && inc_wranges_int) {
        const char *output_fname = mf_wrap_tag_suffix(MOLECFIT_WAVE_INCLUDE, suffix, CPL_TRUE);
        const char *tag_name     = mf_wrap_tag_suffix(MOLECFIT_WAVE_INCLUDE, suffix, CPL_FALSE);
        err = mf_wrap_save(frameset, frameset, parlist, RECIPE_NAME, parameters->pl, tag_name, output_fname);
        //err = mf_wrap_save(frameset, frameset, parlist, RECIPE_NAME, cpl_propertylist_duplicate(parameters->pl), MOLECFIT_WAVE_INCLUDE, "WAVE_INCLUDE.fits");
        if (!err) {
            err = mf_wrap_save_mf_results(data->v_ext[0].header, output_fname, CPL_FALSE, NULL, inc_wranges_int, NULL);
        }
        /*
      if (!err) err = mf_wrap_save_mf_results(data->v_ext[0].header, MOLECFIT_WAVE_INCLUDE, CPL_TRUE, NULL, inc_wranges_new_table, NULL);
*/
        if (inc_wranges) {
            *inc_wranges = cpl_table_duplicate(inc_wranges_int);
        }
    }

    cpl_msg_info(cpl_func, "before 'Save WV_EXCLUDE RANGE's table' %s", cpl_error_get_message());
    /* Save WV_EXCLUDE RANGE's table */
    if (!err && exc_wranges_int) {
        cpl_msg_info(cpl_func, "TRYING TO SAVE exc_wranges");
        const char *output_fname = mf_wrap_tag_suffix(MOLECFIT_WAVE_EXCLUDE, suffix, CPL_TRUE);
        const char *tag_name     = mf_wrap_tag_suffix(MOLECFIT_WAVE_EXCLUDE, suffix, CPL_FALSE);
        err = mf_wrap_save(frameset, frameset, parlist, RECIPE_NAME, parameters->pl, tag_name, output_fname);
        if (!err) {
            err = mf_wrap_save_mf_results(data->v_ext[0].header, output_fname, CPL_FALSE, NULL, exc_wranges_int, NULL);
        }
        if (exc_wranges) {
            *exc_wranges = cpl_table_duplicate(exc_wranges_int);
        }
    }

    cpl_msg_info(cpl_func, "before 'Save PIX_INCLUDE RANGE's table' %s", cpl_error_get_message());
    /* Save PIX_INCLUDE RANGE's table */
    if (!err && exc_pranges_int) {
        const char *output_fname = mf_wrap_tag_suffix(MOLECFIT_PIXEL_EXCLUDE, suffix, CPL_TRUE);
        const char *tag_name     = mf_wrap_tag_suffix(MOLECFIT_PIXEL_EXCLUDE, suffix, CPL_FALSE);
        err = mf_wrap_save(frameset, frameset, parlist, RECIPE_NAME, parameters->pl, tag_name, output_fname);
        if (!err) {
            err = mf_wrap_save_mf_results(data->v_ext[0].header, output_fname, CPL_FALSE, NULL, exc_pranges_int, NULL);
        }
        if (exc_pranges) {
            *exc_pranges = cpl_table_duplicate(exc_pranges_int);
        }
    }

    return err;
}


cpl_error_code mf_wrap_kernel(
    cpl_frameset            *frameset,
    mf_wrap_model_parameter *parameters,
    mf_wrap_fits           **kernel_data,
    cpl_table              **mapping_kernel,
    const char              *suffix
)
{
    cpl_error_code err = CPL_ERROR_NONE;
    //const char* input_tag = mf_wrap_tag_suffix(MOLECFIT_MODEL_MOLECULES,suffix,CPL_FALSE);
    //const char* output_fname = mf_wrap_tag_suffix(MOLECFIT_MODEL_MOLECULES,suffix,CPL_TRUE);

    if (!err) {
        /* mf_wrap_load_kernel_tag is in molecfit.c */
        const char *load_tag = mf_wrap_tag_suffix(MOLECFIT_MODEL_KERNEL_LIBRARY, suffix, CPL_FALSE);

        *kernel_data = mf_wrap_load_kernel_tag(
            frameset, load_tag, parameters->use_input_kernel, parameters->mf_config->parameters, CPL_TRUE
        );
        err = cpl_error_get_code();

        if (*kernel_data && !err) {
            if (parameters->mapping_kernel_table) {
                /* Mapping in the recipe parameters */
                *mapping_kernel = cpl_table_duplicate(parameters->mapping_kernel_table);
            }
            else {
                /* Mapping in the static_calib input FITS files */
                cpl_errorstate pre_state = cpl_errorstate_get();
                const char    *input_tag = mf_wrap_tag_suffix(MOLECFIT_MODEL_MAPPING_KERNEL, suffix, CPL_FALSE);

                const cpl_frame *input_mapping_kernel = cpl_frameset_find(frameset, input_tag);
                if (input_mapping_kernel) {
                    cpl_msg_info(cpl_func, "Loading %s cpl_table", input_tag);
                    *mapping_kernel = mf_wrap_load_unique_table(frameset, input_tag);
                }
                else {
                    cpl_errorstate_set(pre_state);
                    input_tag            = mf_wrap_tag_suffix(MOLECFIT_MAPPING_KERNEL, suffix, CPL_FALSE);
                    input_mapping_kernel = cpl_frameset_find(frameset, input_tag);
                    if (input_mapping_kernel) {
                        cpl_msg_info(cpl_func, "Loading %s cpl_table", input_tag);
                        *mapping_kernel = mf_wrap_load_unique_table(frameset, input_tag);
                    }
                    else {
                        err = cpl_error_set_message(
                            cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                            "Mapping kernel data not found in frameset and neither in the recipe parameter!"
                        );
                    }
                }
            }

            if (!*mapping_kernel && !err) {
                err = CPL_ERROR_ILLEGAL_INPUT;
            }
            else {
                err = cpl_error_get_code();
            }
        }

        if (!err) {
            cpl_msg_info(cpl_func, "Using the default molecfit kernels -> With the values inside BEST_FIT_PARMS!");
        }
    }
    /* Save table */
    /* NOTE: May not want to save it here ...only save after instrument specific changes are made?*/
    /*  if(!err && *kernel_data){
          err += mf_wrap_save(frameset, frameset, parlist, RECIPE_NAME, parameters->pl, MOLECFIT_MODEL_KERNEL_LIBRARY, NULL);
  }
  */

    return err;
}


cpl_error_code
mf_wrap_gdas(cpl_frameset *frameset, mf_wrap_model_parameter *parameters, cpl_table **gdas_user, const char *suffix)
{
    cpl_error_code err       = CPL_ERROR_NONE;
    const char    *input_tag = mf_wrap_tag_suffix(MOLECFIT_GDAS, suffix, CPL_FALSE);


    if (!err) {
        char          *gdas_prof = parameters->mf_config->parameters->atmospheric.gdas_prof;
        cpl_error_code gdas_err;
        gdas_err = mf_config_chk_gdas_prof(gdas_prof);
        if (gdas_err) {
            cpl_msg_info(cpl_func, "Invalid value for GDAS_PROF: %s", gdas_prof);
        }

        //only search the SOF if GDAS_PROF = NULL (default is auto)
        if ((strcmp(gdas_prof, "null") == 0 || strcmp(gdas_prof, "NULL") == 0)) {
            if (cpl_frameset_find_const(frameset, input_tag) && !gdas_err) {
                *gdas_user = mf_wrap_load_unique_table(frameset, input_tag);
                if (*gdas_user) {
                    cpl_msg_info(cpl_func, "Loaded %s cpl_table", input_tag);
                }
                err = cpl_error_get_code();
            }
            else {
                /* Use the average if possible */
                cpl_msg_info(cpl_func, "GDAS profile missing from SOF: Reverting to GDAS_PROF=auto behaviour");
                //parameters->mf_config->parameters->atmospheric.gdas_prof = "auto";
                cpl_free(parameters->mf_config->parameters->atmospheric.gdas_prof);
                parameters->mf_config->parameters->atmospheric.gdas_prof = cpl_strdup("auto");
            }
        }
        //if gdas_prof is a file...
        else if (!(!strcmp(gdas_prof, "auto") || !strcmp(gdas_prof, "AUTO") || !strcmp(gdas_prof, "none") ||
                   !strcmp(gdas_prof, "NONE") || !strcmp(gdas_prof, "null") || !strcmp(gdas_prof, "NULL"))) {
            if (!gdas_err) {
                cpl_msg_info(cpl_func, "Loading GDAS_PROF from file: %s", gdas_prof);
                *gdas_user = cpl_table_load(gdas_prof, 1, 0);
                if (!*gdas_user) {
                    /* Error loading file, even if gdas_prof suspects it is OK*/
                    err = CPL_ERROR_ILLEGAL_INPUT;
                    cpl_msg_error(cpl_func, "Error loading file specified by GDAS_PROF: %s", gdas_prof);
                }
            }
            else {
                /* Error parsing gdas_prof */
                err = CPL_ERROR_ILLEGAL_INPUT;
                cpl_msg_error(cpl_func, "Error loading file specified by GDAS_PROF: %s", gdas_prof);
            }
        }
    }

    /* Save GDAS table */
    /* Note this is done *after* calling mf_model() */

    return err;
}


cpl_error_code mf_wrap_atm(
    cpl_frameset            *frameset,
    mf_wrap_model_parameter *parameters,
    cpl_table              **atm_profile_standard,
    const char              *suffix
)
{
    //cpl_table** atm_profile_combined -- is not used !

    cpl_error_code err       = CPL_ERROR_NONE;
    const char    *input_tag = mf_wrap_tag_suffix(MOLECFIT_ATM_PROFILE_STANDARD, suffix, CPL_FALSE);
    //const char* output_fname = mf_wrap_tag_suffix(MOLECFIT_ATM_PARAMETERS,suffix,CPL_TRUE);
    cpl_table *atm_profile_standard_int = NULL;
    if (atm_profile_standard != NULL) {
        *atm_profile_standard = NULL;
    }
    if (!err) {
        //cpl_msg_info(cpl_func,"DUMP of frameset in mf_wrap_atm");
        //cpl_frameset_dump(frameset,stdout);

        atm_profile_standard_int = mf_wrap_load_unique_table(frameset, input_tag);

        if (atm_profile_standard_int) {
            cpl_msg_info(cpl_func, "Loaded %s cpl_table", input_tag);
        }
        err = cpl_error_get_code();
    }

    /* Check Request LNFL DB is valid */
    if (!err) {
        cpl_error_code lnfl_db_err = CPL_ERROR_NONE;
        lnfl_db_err                = mf_config_chk_lnfl_line_db(
            parameters->mf_config->parameters->directories.telluriccorr_data_path, parameters->mf_config->lnfl->line_db
        );
        cpl_msg_error(
            cpl_func, "after mf_config_chk_lnfl_line_db: %d %s (path %s)", err, cpl_error_get_message(),
            parameters->mf_config->parameters->directories.telluriccorr_data_path
        );
        if (lnfl_db_err) {
            cpl_msg_error(
                cpl_func, "Invalid parameter value for LNFL_LINE_DB: %s", parameters->mf_config->lnfl->line_db
            );
            err = CPL_ERROR_ILLEGAL_INPUT;
            cpl_error_set_message(cpl_func, err, "molecfit_model .rc parameter checks failed!");
        }
    }
    /*** Save generic output files */
    if (!err) {
        //const char* tag_name = mf_wrap_tag_suffix(MOLECFIT_ATM_PARAMETERS,suffix,CPL_FALSE);
        //err     += mf_wrap_save(frameset, frameset, parlist, RECIPE_NAME, parameters->pl, tag_name, output_fname);
        if (atm_profile_standard_int && atm_profile_standard) {
            *atm_profile_standard = cpl_table_duplicate(atm_profile_standard_int);
        }
    }

    return err;
}

cpl_error_code mf_gdas_write(
    mf_model_results        *results,
    const cpl_parameterlist *parlist,
    const cpl_table         *gdas_user,
    cpl_frameset            *frameset,
    mf_wrap_model_parameter *parameters,
    mf_wrap_fits            *data,
    const char              *suffix
)
{  //FLESH THIS OUT


    cpl_error_code err = CPL_ERROR_NONE;

    if (!(results->gdas_interpolate)) {
        err = CPL_ERROR_ILLEGAL_OUTPUT;
    }
    else {
        if (gdas_user) {
            cpl_msg_info(
                cpl_func, "Saving Molecfit output user provide fits files ('%s') ... [only first call to Molecfit!]",
                MOLECFIT_GDAS
            );
        }
        else if (results->gdas_before && results->gdas_after && results->gdas_interpolate) {
            cpl_msg_info(
                cpl_func,
                "Save Molecfit output GDAS automatic fits files ('%s','%s','%s') ... [only first call to Molecfit!]",
                MOLECFIT_GDAS_BEFORE, MOLECFIT_GDAS_AFTER, MOLECFIT_GDAS
            );

            /* Keep time, no access to ESO GDAS DB : Same gdas_interpolate for all the IFUs because all the arms have the same 'date' -> MDJ_OBS in the header */
            gdas_user = cpl_table_duplicate(results->gdas_interpolate);
        }

        const char *output_fname = mf_wrap_tag_suffix(MOLECFIT_GDAS, suffix, CPL_TRUE);
        const char *tag_name     = mf_wrap_tag_suffix(MOLECFIT_GDAS, suffix, CPL_FALSE);

        err = mf_wrap_save(frameset, frameset, parlist, RECIPE_NAME, parameters->pl, tag_name, output_fname);
        if (!err) {
            err = mf_wrap_save_mf_results(
                data->v_ext[0].header, output_fname, CPL_FALSE, NULL, results->gdas_interpolate, NULL
            );
        }

        /* If not default GDAS profiles, save the GDAS files (before/after) from the ESO GDAS DB */

        if (results->gdas_before) {
            output_fname = mf_wrap_tag_suffix(MOLECFIT_GDAS_BEFORE, suffix, CPL_TRUE);
            tag_name     = mf_wrap_tag_suffix(MOLECFIT_GDAS_BEFORE, suffix, CPL_FALSE);
            err = mf_wrap_save(frameset, frameset, parlist, RECIPE_NAME, parameters->pl, tag_name, output_fname);
            if (!err) {
                err = mf_wrap_save_mf_results(
                    data->v_ext[0].header, output_fname, CPL_FALSE, NULL, results->gdas_before, NULL
                );
            }
        }

        if (results->gdas_after) {
            output_fname = mf_wrap_tag_suffix(MOLECFIT_GDAS_AFTER, suffix, CPL_TRUE);
            tag_name     = mf_wrap_tag_suffix(MOLECFIT_GDAS_AFTER, suffix, CPL_FALSE);
            err = mf_wrap_save(frameset, frameset, parlist, RECIPE_NAME, parameters->pl, tag_name, output_fname);
            if (!err) {
                err = mf_wrap_save_mf_results(
                    data->v_ext[0].header, output_fname, CPL_FALSE, NULL, results->gdas_after, NULL
                );
            }
        }

        //gdas_write = CPL_TRUE; -> check should move to the instrument recipe.
    }

    return err;
}

cpl_error_code mf_atm_write(
    mf_model_results        *results,
    const cpl_parameterlist *parlist,
    cpl_frameset            *frameset,
    mf_wrap_model_parameter *parameters,
    mf_wrap_fits            *data,
    cpl_table               *atm_profile_standard,
    const char              *suffix
)
{
    //cpl_table *atm_profile_combined){ // FLESH THIS OUT

    cpl_error_code err;

    if (!(results->atm_profile_standard) || !(results->atm_profile_combined)) {
        err = CPL_ERROR_ILLEGAL_OUTPUT;
        return err;
    }
    if (atm_profile_standard) {
        cpl_msg_info(
            cpl_func, "Save Molecfit output user provide fits files ('%s') ... [only first call to Molecfit!]",
            MOLECFIT_ATM_PROFILE_STANDARD
        );
    }
    else {
        cpl_msg_info(
            cpl_func,
            "Saving Molecfit output automatic ATM_PROFILE fits files ('%s','%s') ... [only first call to "
            "Molecfit!]",
            MOLECFIT_ATM_PROFILE_STANDARD, MOLECFIT_ATM_PROFILE_COMBINED
        );

        /* Keep time, no access to disk : Same atm_profile_standard for all the extensions because they have the same 'date' -> MDJ_OBS in the header */
        atm_profile_standard = cpl_table_duplicate(results->atm_profile_standard);
    }

    /* Keep time, no access to disk : Same atm_profile_standard for all the extensions because they have the same 'date' -> MDJ_OBS in the header */
    //atm_profile_combined = cpl_table_duplicate(results->atm_profile_combined); -- IS THIS NEEDED ??

    const char *output_fname = mf_wrap_tag_suffix(MOLECFIT_ATM_PROFILE_STANDARD, suffix, CPL_TRUE);
    const char *tag_name     = mf_wrap_tag_suffix(MOLECFIT_ATM_PROFILE_STANDARD, suffix, CPL_FALSE);

    err = mf_wrap_save(frameset, frameset, parlist, RECIPE_NAME, parameters->pl, tag_name, output_fname);
    if (err) {
        return err;
    }
    err = mf_wrap_save_mf_results(data->v_ext[0].header, output_fname, CPL_FALSE, NULL, atm_profile_standard, NULL);
    if (err) {
        return err;
    }

    output_fname = mf_wrap_tag_suffix(MOLECFIT_ATM_PROFILE_COMBINED, suffix, CPL_TRUE);
    tag_name     = mf_wrap_tag_suffix(MOLECFIT_ATM_PROFILE_COMBINED, suffix, CPL_FALSE);
    err          = mf_wrap_save(frameset, frameset, parlist, RECIPE_NAME, parameters->pl, tag_name, output_fname);
    if (!err) {
        err = mf_wrap_save_mf_results(
            data->v_ext[0].header, output_fname, CPL_FALSE, NULL, results->atm_profile_combined, NULL
        );
    }

    //atm_profile_standard_write = CPL_TRUE; -> check should move to the instrument recipe.


    return err;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief    Checks and merges all wavelength range specific data into a single
 *           table.
 *
 * @param  inc_wranges_new_table  The new wavelength range table to be built.
 * @param  inc_wranges_ext_table  The wavelength range table loaded from external fits file.
 * @param  wave_include_str       Parameter string from .rc file that defines wave_include
 * @param  map_regions_str        Parameter string from .rc file that defines map_regions
 * @param  fit_continuum_str      Parameter string from .rc file that defines fit_continuum
 * @param  continuum_n_str        Parameter string from .rc file that defines continuum_n
 * @param  fit_wlc_str            Parameter string from .rc file that defines fit_wlc
 * @param  nchips                 No of chips declared.
*/
/*----------------------------------------------------------------------------*/
cpl_table *mf_wrap_model_amalgamate_ranges_table(
    cpl_table  *inc_wranges_ext_table,
    const char *wave_include_str,
    const char *map_regions_str,
    const char *inp_fit_continuum_str,
    const char *continuum_n_str,
    const char *inp_fit_wlc_str,
    int         nchips,
    double      wlg2mn
)


{
    /* Temporary Fix from old regression tests that sets continuum_str = TRUE or FALSE */
    const char *fit_continuum_str;
    if (strcmp(inp_fit_continuum_str, "TRUE") == 0) {
        fit_continuum_str = "1";
    }
    else if (strcmp(inp_fit_continuum_str, "FALSE") == 0) {
        fit_continuum_str = "0";
    }
    else {
        fit_continuum_str = inp_fit_continuum_str;
    }

    const char *fit_wlc_str;
    if (strcmp(inp_fit_wlc_str, "TRUE") == 0) {
        fit_wlc_str = "1";
    }
    else if (strcmp(inp_fit_wlc_str, "FALSE") == 0) {
        fit_wlc_str = "0";
    }
    else {
        fit_wlc_str = inp_fit_wlc_str;
    }


    cpl_table *inc_wranges_new_table = NULL; /* The return value if succesful */

    double      rangev[2 * MF_PARAMETERS_MAXIMUM_NO_OF_RANGES];
    int         mapv[MF_PARAMETERS_MAXIMUM_NO_OF_RANGES];
    int         fit_contv[MF_PARAMETERS_MAXIMUM_NO_OF_RANGES];
    int         cont_nv[MF_PARAMETERS_MAXIMUM_NO_OF_RANGES];
    int         fit_wlcv[MF_PARAMETERS_MAXIMUM_NO_OF_RANGES];
    cpl_boolean R2C_MAP;


    cpl_msg_info(cpl_func, "Amalgamating all region related parameters into single table");


    /* ------------------------------------------------------------------------------
  If there is no external wave_include fits file then check that none of the string
     parameters is declared as "NULL"
     ------------------------------------------------------------------------------*/
    if (inc_wranges_ext_table == NULL) {
        cpl_msg_info(cpl_func, "There is no external table");

        if (strcmp(wave_include_str, MF_PARAMETERS_NULL) == 0) {
            cpl_msg_error(cpl_func, "No range values defined");
            /*return NULL;*/
        }

        if (strcmp(map_regions_str, MF_PARAMETERS_NULL) == 0) {
            cpl_msg_info(cpl_func, "No map regions to chips defined. Assuming all are mapped to chip 1");
            /* The map_regions and the setting of R2C_MAP is handled further below. */
        }

        if (strcmp(fit_continuum_str, MF_PARAMETERS_NULL) == 0) {
            cpl_msg_error(cpl_func, "No range specific continuum fit flags defined");
            return NULL;
        }

        if (strcmp(continuum_n_str, MF_PARAMETERS_NULL) == 0) {
            cpl_msg_error(cpl_func, "No range specific poly order of continuum modelling defined");
            return NULL;
        }

        if (strcmp(fit_wlc_str, MF_PARAMETERS_NULL)) {
            cpl_msg_error(cpl_func, "No range specific wavlength correction flags set");
            return NULL;
        }
    }
    else {
        cpl_msg_info(cpl_func, "There is an external table");
    }

    /* At this stage we know that there is either an external table or range values defined in a string */


    /* ------------------------------- */
    /* WAVE_INCLUDE RANGE LIMIT VALUES */
    /* ------------------------------- */

    cpl_size nranges;
    double   llimv[MF_PARAMETERS_MAXIMUM_NO_OF_RANGES];
    double   ulimv[MF_PARAMETERS_MAXIMUM_NO_OF_RANGES];

    if (strcmp(wave_include_str, MF_PARAMETERS_NULL) == 0) {
        /* No values defined in a string so get them from the external table */
        cpl_msg_info(cpl_func, "Wave Inlude string=NULL. Reading data from external table.");

        /* Check that we have the lower limit data column */
        if (!cpl_table_has_column(inc_wranges_ext_table, MF_COL_WAVE_RANGE_LOWER)) {
            cpl_msg_error(cpl_func, "External table does not have column %s", MF_COL_WAVE_RANGE_LOWER);
            return NULL;
        }

        /* Check that we have the upper limit data column */
        if (!cpl_table_has_column(inc_wranges_ext_table, MF_COL_WAVE_RANGE_UPPER)) {
            cpl_msg_error(cpl_func, "External table does not have column %s", MF_COL_WAVE_RANGE_UPPER);
            return NULL;
        }

        nranges = cpl_table_get_nrow(inc_wranges_ext_table);
        for (int i = 0; i < nranges; i++) {
            llimv[i] = cpl_table_get(inc_wranges_ext_table, MF_COL_WAVE_RANGE_LOWER, i, NULL);
            ulimv[i] = cpl_table_get(inc_wranges_ext_table, MF_COL_WAVE_RANGE_UPPER, i, NULL);
        }
    }
    else {
        /* Values are defined in a string as an even list of doubles so parse the string into a vector of doubles */
        cpl_msg_info(cpl_func, "Wave Include string=%s", wave_include_str);

        int full_string_size = (int)strlen(wave_include_str);
        int ndvals           = mf_wrap_parsReadDValsFromString(
            wave_include_str, rangev, 2 * MF_PARAMETERS_MAXIMUM_NO_OF_RANGES, full_string_size
        );

        /* Check that parsing string was successful */
        if (ndvals < 1) {
            cpl_msg_error(cpl_func, "Invalid list of wavelength ranges : %s", wave_include_str);
            return NULL;
        }

        /* Check that we have an even no of doubles */
        if (ndvals % 2 != 0) {
            cpl_msg_error(cpl_func, "Odd number of range boundaries defined: %s", wave_include_str);
            return NULL;
        }
        else {
            cpl_msg_info(cpl_func, "Even number of range values defined");
        }

        nranges = ndvals / 2;
        for (int i = 0; i < nranges; i++) {
            llimv[i] = rangev[2 * i];
            ulimv[i] = rangev[2 * i + 1];
        }
    }

    /* Now check that lower values < upper values */
    cpl_msg_info(cpl_func, "%" CPL_SIZE_FORMAT " Ranges defined:", nranges);
    for (int i = 0; i < nranges; i++) {
        cpl_msg_info(cpl_func, "Range %d: [%f, %f]um", i + 1, llimv[i] * wlg2mn, ulimv[i] * wlg2mn);
        if (llimv[i] >= ulimv[i]) {
            cpl_msg_error(cpl_func, "Invalid range boundaries [%f, %f]um", llimv[i] * wlg2mn, ulimv[i] * wlg2mn);
            return NULL;
        }
    }


    /* ------------------------------- */
    /* RANGES MAPPED TO CHIP  DETAILS  */
    /* ------------------------------- */

    /* Assume there is no valid range to chip map column unless proven otherwise */
    R2C_MAP = CPL_FALSE;
    cpl_msg_info(cpl_func, "Sorting Range Mappings To %d Chips.", nchips);

    if (strcmp(map_regions_str, MF_PARAMETERS_NULL) == 0) {
        /* No values defined in a string so get them from the external table */
        cpl_msg_info(cpl_func, "Map Regions string=NULL. Reading data from external table");

        /* Check that there is the required data column */
        if (!cpl_table_has_column(inc_wranges_ext_table, MF_COL_WAVE_RANGE_MAP2CHIP)) {
            /* No valid column so report and set R2C_MAP to false */
            cpl_msg_info(
                cpl_func, "External table does not have column %s. Assuming that all regions are mapped to chip 1",
                MF_COL_WAVE_RANGE_MAP2CHIP
            );
        }
        else {
            /* Load the column values into a vector */
            for (int i = 0; i < nranges; i++) {
                mapv[i] = cpl_table_get_int(inc_wranges_ext_table, MF_COL_WAVE_RANGE_MAP2CHIP, i, NULL);
            }
            R2C_MAP = CPL_TRUE;
        }
    }
    else {
        /* Values are defined in a string as a list of integers so parse the string into a vector of integers */
        cpl_msg_info(cpl_func, "Map Regions string=%s", map_regions_str);

        int full_string_size = (int)strlen(map_regions_str);
        int nivals           = mf_wrap_parsReadIValsFromString(
            map_regions_str, mapv, MF_PARAMETERS_MAXIMUM_NO_OF_RANGES, full_string_size
        );

        /* Check that we have the correct number of integers */
        if (nivals != nranges && nivals != 1) {
            cpl_msg_error(cpl_func, "Range mapping parameter has incorrect list size");
            return NULL;
        }
        else if (nivals == 1) {
            /* The exception to the rule is if there is only one value then that implies that this is the smae for all */
            cpl_msg_info(cpl_func, "Assuming that all regions are mapped to Chip %d", mapv[0]);
            for (int i = 1; i < nranges; i++) {
                mapv[i] = mapv[0];
            }
        }
        /* The R2C MAP has now been derived from a string definition into the vector form */
        R2C_MAP = CPL_TRUE;
    }

    /* In the event that there are no maps defined, (i.e. the map string is set to "NULL" and either the external   */
    /* table does not exists or the required column does not exist in the external table) then the requirements are */
    /* that all ranges should be mapped to chip  1                                                                  */
    if (R2C_MAP == CPL_FALSE) {
        for (int i = 0; i < nranges; i++) {
            mapv[i] = 1;
        }
    }


    /* ------------------------------- */
    /* RANGES FIT CONTINUUM FLAGS      */
    /* ------------------------------- */
    if (strcmp(fit_continuum_str, MF_PARAMETERS_NULL) == 0) {
        /* No values defined in a string so get them from the external table */

        /* Check that there is the required data column */
        if (!cpl_table_has_column(inc_wranges_ext_table, MF_COL_WAVE_RANGE_CONT_FIT)) {
            cpl_msg_error(cpl_func, "External table does not have column %s", MF_COL_WAVE_RANGE_CONT_FIT);
            return NULL;
        }

        /* Load the values into a vector */
        for (int i = 0; i < nranges; i++) {
            fit_contv[i] = cpl_table_get_int(inc_wranges_ext_table, MF_COL_WAVE_RANGE_CONT_FIT, i, NULL);
        }
    }
    else {
        /* Values are defined in a string as a list of integers so parse the string into a vector of integers */
        cpl_msg_info(cpl_func, "Fit Continuum string=%s", fit_continuum_str);

        int full_string_size = (int)strlen(fit_continuum_str);
        int nivals           = mf_wrap_parsReadIValsFromString(
            fit_continuum_str, fit_contv, MF_PARAMETERS_MAXIMUM_NO_OF_RANGES, full_string_size
        );

        /* Check that we have the correct number of integers */
        if (nivals != nranges && nivals != 1) {
            cpl_msg_error(cpl_func, "Range fit continuum parameter has incorrect list size");
            return NULL;
        }
        else if (nivals == 1) {
            /* Special case for nivals=1 implies that this value should be used for all */
            cpl_msg_info(cpl_func, "Assuming that all ranges have fit continuum flag %d", fit_contv[0]);
            for (int i = 1; i < nranges; i++) {
                fit_contv[i] = fit_contv[0];
            }
        }
    }

    /* ------------------------------------------- */
    /* RANGES POLYNOMIAL ORDER FOR CONTINUUM MODEL */
    /* ------------------------------------------- */

    if (strcmp(continuum_n_str, MF_PARAMETERS_NULL) == 0) {
        /* No values defined in a string so get them from the external table */

        /* Check that there is the required data column */
        if (!cpl_table_has_column(inc_wranges_ext_table, MF_COL_WAVE_RANGE_CONT_ORDER)) {
            cpl_msg_error(cpl_func, "External table does not have column %s", MF_COL_WAVE_RANGE_CONT_ORDER);
            return NULL;
        }

        /* Load the values into a vector */
        for (int i = 0; i < nranges; i++) {
            cont_nv[i] = cpl_table_get_int(inc_wranges_ext_table, MF_COL_WAVE_RANGE_CONT_ORDER, i, NULL);
        }
    }
    else {
        /* Values are defined in a string as a list of integers so parse the string into a vector of integers */
        cpl_msg_info(cpl_func, "Continuum n string=%s", continuum_n_str);

        int full_string_size = (int)strlen(continuum_n_str);
        int nivals           = mf_wrap_parsReadIValsFromString(
            continuum_n_str, cont_nv, MF_PARAMETERS_MAXIMUM_NO_OF_RANGES, full_string_size
        );

        /* Check that we have the correct number of integers */
        if (nivals != nranges && nivals != 1) {
            cpl_msg_error(cpl_func, "Range continuum polynomial order parameter has incorrect list size");
            return NULL;
        }
        else if (nivals == 1) {
            /* Special case for nivals=1 implies that this value should be used for all ranges */
            cpl_msg_info(cpl_func, "Assuming that all ranges have polynomial order %d to model continuum", cont_nv[0]);
            for (int i = 1; i < nranges; i++) {
                cont_nv[i] = cont_nv[0];
            }
        }
    }


    /* ------------------------------------------- */
    /* RANGES WAVLENGTH CORRECTION FIT FLAGS       */
    /* ------------------------------------------- */

    if (strcmp(fit_wlc_str, MF_PARAMETERS_NULL) == 0) {
        cpl_msg_info(cpl_func, "Fit WLC string is NULL!");

        /* No values defined in a string so get them from the external table */

        /* Check that there is the required data column */
        if (!cpl_table_has_column(inc_wranges_ext_table, MF_COL_WAVE_RANGE_WLC_FIT)) {
            cpl_msg_error(cpl_func, "External table does not have column %s", MF_COL_WAVE_RANGE_WLC_FIT);
            return NULL;
        }

        /* Load the values into a vector */
        cpl_msg_info(cpl_func, "Loading WLC Fit flags from fits table");
        for (int i = 0; i < nranges; i++) {
            fit_wlcv[i] = cpl_table_get_int(inc_wranges_ext_table, MF_COL_WAVE_RANGE_WLC_FIT, i, NULL);
        }
    }
    else {
        /* Values are defined in a string as a list of integers so parse the string into a vector of integers */
        cpl_msg_info(cpl_func, "Fit WLC string=%s", fit_wlc_str);

        int full_string_size = (int)strlen(fit_wlc_str);
        int nivals           = mf_wrap_parsReadIValsFromString(
            fit_wlc_str, fit_wlcv, MF_PARAMETERS_MAXIMUM_NO_OF_RANGES, full_string_size
        );

        /* Check that we have the correct number of integers */
        if (nivals != nranges && nivals != 1) {
            cpl_msg_error(cpl_func, "Range wavlength correction flags parameter has incorrect list size");
            return NULL;
        }
        else if (nivals == 1) {
            /* Special case for nivals=1 implies that this value should be used for all ranges */
            for (int i = 1; i < nranges; i++) {
                fit_wlcv[i] = fit_wlcv[0];
            }
        }
    }
    for (int i = 0; i < nranges; i++) {
        cpl_msg_info(cpl_func, "WLC Fit flag for range %d =%d", i + 1, fit_wlcv[i]);
    }


    /* ------------------------------------------- */
    /* ALL DATA OBTAINED SO NOW CREATE NEW TABLE   */
    /* ------------------------------------------- */

    cpl_error_code err = CPL_ERROR_NONE;

    cpl_msg_info(cpl_func, "About to create a new table with %" CPL_SIZE_FORMAT " rows", nranges);

    /* Create a new table with same number of rows as there are ranges*/
    inc_wranges_new_table = cpl_table_new(nranges);

    if (inc_wranges_new_table == NULL) {
        err = CPL_ERROR_NULL_INPUT;
        cpl_msg_error(cpl_func, "Failed to create a new table");
    }
    else {
        cpl_size tmpi = cpl_table_get_nrow(inc_wranges_new_table);
        cpl_msg_info(cpl_func, "Have created a new table with %" CPL_SIZE_FORMAT " rows", tmpi);
    }

    /* Define the table columns  */
    if (!err) {
        err = cpl_table_new_column(inc_wranges_new_table, MF_COL_WAVE_RANGE_LOWER, CPL_TYPE_DOUBLE);
    }
    if (!err) {
        err = cpl_table_new_column(inc_wranges_new_table, MF_COL_WAVE_RANGE_UPPER, CPL_TYPE_DOUBLE);
    }
    if (!err) {
        err = cpl_table_new_column(inc_wranges_new_table, MF_COL_WAVE_RANGE_MAP2CHIP, CPL_TYPE_INT);
    }
    if (!err) {
        err = cpl_table_new_column(inc_wranges_new_table, MF_COL_WAVE_RANGE_CONT_FIT, CPL_TYPE_INT);
    }
    if (!err) {
        err = cpl_table_new_column(inc_wranges_new_table, MF_COL_WAVE_RANGE_CONT_ORDER, CPL_TYPE_INT);
    }
    if (!err) {
        err = cpl_table_new_column(inc_wranges_new_table, MF_COL_WAVE_RANGE_WLC_FIT, CPL_TYPE_INT);
    }

    /* Populate the table */
    for (int i = 0; i < nranges; i++) {
        if (!err) {
            err = cpl_table_set_double(inc_wranges_new_table, MF_COL_WAVE_RANGE_LOWER, i, llimv[i]);
        }
        if (!err) {
            err = cpl_table_set_double(inc_wranges_new_table, MF_COL_WAVE_RANGE_UPPER, i, ulimv[i]);
        }
        if (!err) {
            err = cpl_table_set_int(inc_wranges_new_table, MF_COL_WAVE_RANGE_CONT_FIT, i, fit_contv[i]);
        }
        if (!err) {
            err = cpl_table_set_int(inc_wranges_new_table, MF_COL_WAVE_RANGE_CONT_ORDER, i, cont_nv[i]);
        }
        if (!err) {
            err = cpl_table_set_int(inc_wranges_new_table, MF_COL_WAVE_RANGE_MAP2CHIP, i, mapv[i]);
        }
        if (!err) {
            err = cpl_table_set_int(inc_wranges_new_table, MF_COL_WAVE_RANGE_WLC_FIT, i, fit_wlcv[i]);
        }
    }

    if (err) {
        cpl_msg_error(cpl_func, "Unable to create new wavelength range table");
        if (inc_wranges_new_table != NULL) {
            cpl_free(inc_wranges_new_table);
        }
        return NULL;
    }

    return inc_wranges_new_table;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief    Parses a string defining a ',' deliminated list of doublevalues
 *           and populates values onto supplied vector
 *           table. Returns no of integer values parsed or -1 if failure.
 *
 * @param  str           The string with the list of values
 * @param  vec           The vector to populate
 * @param  vec           The size of the vector
 * @param  max_strlen    Maximum size of the string and deliminated substrings
*/
/*----------------------------------------------------------------------------*/


int mf_wrap_parsReadDValsFromString(const char *str, double *vec, int vsize, int max_strlen)
{
    char bufferA[vsize][max_strlen];

    /*Parse string lists into individual string elements and place in bufferA*/
    int str_size = (int)strlen(str);
    int ndelim   = 0;
    int i        = 0;
    for (int idx = 0; idx < str_size; idx++) {
        if (str[idx] == ',') {
            bufferA[ndelim][i] = '\0';
            ndelim++;
            i = 0;
        }
        else {
            bufferA[ndelim][i] = str[idx];
            i++;
            bufferA[ndelim][i] = '\0';
        }
    }

    /* Iterate each string stored in the buffer array
       convert to an integer and store in the return array */
    int    nelems = ndelim + 1;
    double dval;

    for (i = 0; i < nelems; i++) {
        int rn = sscanf(bufferA[i], "%lf", &dval);

        if (rn == 1) {
            vec[i] = dval;
        }
        else {
            return -1;
        }
    }

    return nelems;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief    Parses a string defining a ',' deliminated list of integer values
 *           and populates values onto supplied vector
 *           table. Returns no of integer values parsed or -1 if failure.
 *
 * @param  str           The string with the list of values
 * @param  vec           The vector to populate
 * @param  vec           The size of the vector
 * @param  max_strlen    Maximum size of the string and deliminated substrings
*/
/*----------------------------------------------------------------------------*/

int mf_wrap_parsReadIValsFromString(const char *str, int *vec, int vsize, int max_strlen)
{
    char bufferA[vsize][max_strlen];

    /*Parse string lists into individual string elements and place in bufferA*/
    int str_size = (int)strlen(str);
    int ndelim   = 0;
    int i        = 0;
    for (int idx = 0; idx < str_size; idx++) {
        if (str[idx] == ',') {
            bufferA[ndelim][i] = '\0';
            ndelim++;
            i = 0;
        }
        else {
            bufferA[ndelim][i] = str[idx];
            i++;
            bufferA[ndelim][i] = '\0';
        }
    }

    /* Iterate each string stored in the buffer array
       convert to an integer and store in the return array */
    int nelems = ndelim + 1;
    int ival;

    for (i = 0; i < nelems; i++) {
        int rn = sscanf(bufferA[i], "%d", &ival);
        if (rn == 1) {
            vec[i] = ival;
        }
        else {
            return -1;
        }
    }

    return nelems;
}
