/*
 * This file is part of the ESO Telluric Correction Library
 * Copyright (C) 2001-2018 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */


/*----------------------------------------------------------------------------*/
/*   NOTES                                                                    */
/*----------------------------------------------------------------------------*/

/*
   List of mf_io functions that could be tested:
          mf_io_pwd                               -> Within tests 
          mf_io_getenv                            -> Within tests 
          mf_io_sync
          mf_io_access                            -> Within tests 
          mf_io_system                            -> Within tests 
          mf_io_systemOpenMP
          mf_io_rm                                -> Within tests 
          mf_io_mv                                -> Within tests  
          mf_io_rm_rf
          mf_io_mkstemp
          mf_io_mkdir                             -> Within tests 
          mf_io_curl                              -> In mf_io_curl-test.c
          mf_io_tarball_nfiles
          mf_io_read_gdas_file_and_create_table
          mf_io_write_lnfl_configuration          -> Within tests
          mf_io_write_lblrtm_configuration
          mf_io_find_klim
          mf_io_read_lblrtm_and_update_spec
          mf_io_symlink                           -> Within tests
*/


/*----------------------------------------------------------------------------*/
/**
 *                              Includes
 */
/*----------------------------------------------------------------------------*/
#include "mf_constants.h"

#include "mf_parameters.h"
#include "mf_model.h"
#include "mf_io.h"

#include <cpl_test.h>


/*----------------------------------------------------------------------------*/
/**
 *                 Typedefs: Enumeration types
 */
/*----------------------------------------------------------------------------*/
/* Options for running the LNFL test */
enum LNFL_TEST_RUN_CODE
{
    RUN_OK,
    BAD_INPUT,
    SYS_CALL,
    CORE_DUMP
};

/*----------------------------------------------------------------------------*/
/**
 *                 Defines
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Global variables
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Macros
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Typedefs: Structured types
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Functions prototypes
 */
/*----------------------------------------------------------------------------*/
static cpl_error_code
test_mf_io_write_lnfl_configuration(mf_parameters_config *pars, enum LNFL_TEST_RUN_CODE option_flag);

/*----------------------------------------------------------------------------*/
/**
 *                 Program
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 * @defgroup test_mf_model   .
 *
 * @brief
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/* ---------------------------------------------------------------------------*/
/**
 * @brief .
 *
 * @param .                  .
 * @param                    .
 *
 * @return cpl_error_code    CPL_ERROR_NONE is everything is OK.
 *                           If not, these are the errors:
 *                           - .
 *                           - Error in subroutine (see subroutines).
 *
 * @description .
 *
 * @note .
 *
 */

/* ---------------------------------------------------------------------------*/
/* ---------------------------------------------------------------------------*/

int main(int argc, char *argv[], char *envp[])
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_DEBUG);

    /* Initialize */
    cpl_errorstate errstate = cpl_errorstate_get();

    /* Get time */
    double cs = cpl_test_get_walltime();

    /* Create configuration */
    mf_parameters_config *config_parameters = mf_parameters_config_create();

    /* Initialize parameters */
    mf_parameters *params = mf_parameters_initialize(config_parameters, NULL);

    /* Set the current error as NONE */
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull(params);


    //TODO
    if (argc > 0) {
        printf("%s", argv[0]);
    }
    int i;
    for (i = 0; envp[i] != NULL; i++) {
        cpl_msg_info(cpl_func, "ENV: %s", envp[i]);
    }

    /* ===================================================*/
    /*                TEST BASIC FILE COMMANDS            */
    /* ===================================================*/
    const char *TEMP_DIR = "test_mf_io_TEMPORARY_unit_test_directory";

    char          *pwd_str;
    char          *file_str;
    char          *none_str;
    cpl_error_code err_code;

    /* ------------------------------------------ */
    /*            Simple file IO tests            */
    /* ------------------------------------------ */
    cpl_msg_info(cpl_func, "-----------------------------------");
    cpl_msg_info(cpl_func, "Testing mf_io file functions      :");
    cpl_msg_info(cpl_func, "-----------------------------------");


    /* Get Working Directory */
    pwd_str = mf_io_pwd();
    cpl_msg_info(cpl_func, "test_mf_io() -> PWD: %s", pwd_str);

    /* Define a real file */
    file_str = mf_io_pwd();
    strcat(file_str, "/input/empty.file");
    cpl_msg_info(cpl_func, "test_mf_io() -> REAL FILE: %s", file_str);
    err_code = mf_io_access(file_str);
    cpl_msg_info(cpl_func, "test_mf_io() -> ACCESS err_code : %d", err_code);

    /* Define a non existant file */
    none_str = mf_io_pwd();
    none_str = strcat(none_str, "/input/none.file");
    cpl_msg_info(cpl_func, "test_mf_io() -> NONE FILE: %s", none_str);
    err_code = mf_io_access(none_str);
    cpl_msg_info(cpl_func, "test_mf_io() -> ACCESS err_code : %d", err_code);

    /* Test getenv() */
    const char *env_str = mf_io_getenv("HOME");
    cpl_msg_info(cpl_func, "mf_io_getenv(\"HOME\") -> %s", env_str);
    const char *xxxx_env_str = mf_io_getenv("XXXX");
    cpl_msg_info(cpl_func, "mf_io_getenv(\"XXXX\") -> %s", xxxx_env_str);

    /* Check that we can find executables and data files correctly */
    char *exec_str = cpl_sprintf("%s/%s", config_parameters->directories.telluric_path, MF_BIN_LNFL);
    if (!mf_io_access(exec_str)) {
        cpl_msg_info(cpl_func, "lnfl executable found -> : %s", exec_str);
    }
    else {
        cpl_msg_error(cpl_func, "cannot find executable   : %s", exec_str);
    }
    cpl_free(exec_str);

    char *data_str = cpl_sprintf("%s/%s", config_parameters->directories.telluriccorr_data_path, MF_HITRAN_PATH);
    if (!mf_io_access(data_str)) {
        cpl_msg_info(cpl_func, "data path found -> : %s", data_str);
    }
    else {
        cpl_msg_info(cpl_func, "cannot find data path   : %s", data_str);
    }
    cpl_free(data_str);

    if (!mf_io_access(config_parameters->directories.telluriccorr_data_path)) {
        cpl_msg_info(cpl_func, "data path found -> : %s", config_parameters->directories.telluriccorr_data_path);
    }
    else {
        cpl_msg_info(cpl_func, "cannot find data path   : %s", config_parameters->directories.telluriccorr_data_path);
    }

    char *data_str2 = cpl_sprintf(
        "%s/%s/%s", config_parameters->directories.telluriccorr_data_path, MF_HITRAN_PATH, MF_LNFL_LINE_DB_AER_VERSION
    );
    if (!mf_io_access(data_str2)) {
        cpl_msg_info(cpl_func, "data file found -> : %s", data_str2);
    }
    else {
        cpl_msg_info(cpl_func, "cannot find data file   : %s", data_str2);
    }
    cpl_free(data_str2);

    char *data_path = cpl_sprintf("%s/%s", config_parameters->directories.telluric_path, "share");
    if (!mf_io_access(data_path)) {
        cpl_msg_info(cpl_func, "data path found -> : %s", data_path);
    }
    else {
        cpl_msg_info(cpl_func, "cannot find data path   : %s", data_path);
    }
    cpl_free(data_path);

    char *data_path2 = cpl_sprintf("%s/%s", config_parameters->directories.telluric_path, "share/molecfit");
    if (!mf_io_access(data_path2)) {
        cpl_msg_info(cpl_func, "data path found -> : %s", data_path2);
    }
    else {
        cpl_msg_info(cpl_func, "cannot find data path   : %s", data_path2);
    }
    cpl_free(data_path2);

    char *data_path3 = cpl_sprintf("%s/%s", config_parameters->directories.telluric_path, "share/molecfit/data");
    if (!mf_io_access(data_path3)) {
        cpl_msg_info(cpl_func, "data path found -> : %s", data_path3);
    }
    else {
        cpl_msg_info(cpl_func, "cannot find data path   : %s", data_path3);
    }
    cpl_free(data_path3);

    char *data_path4 = cpl_sprintf("%s/%s", config_parameters->directories.telluric_path, "share/molecfit/data/hitran");
    if (!mf_io_access(data_path4)) {
        cpl_msg_info(cpl_func, "data path found -> : %s", data_path4);
    }
    else {
        cpl_msg_info(cpl_func, "cannot find data path   : %s", data_path4);
    }
    cpl_free(data_path4);

    char *data_file4 =
        cpl_sprintf("%s/%s", config_parameters->directories.telluric_path, "share/molecfit/data/hitran/aer_v_3.6");
    if (!mf_io_access(data_file4)) {
        cpl_msg_info(cpl_func, "data path found -> : %s", data_file4);
    }
    else {
        cpl_msg_info(cpl_func, "cannot find data path   : %s", data_file4);
    }
    cpl_free(data_file4);


    /* Test mkdir() and rm() */
    char *tmpdir_str = cpl_sprintf("%s/%s", pwd_str, TEMP_DIR);
    err_code         = mf_io_mkdir(tmpdir_str);
    if (!err_code) {
        cpl_msg_info(cpl_func, "mf_io_mkdir -> created : %s", tmpdir_str);
    }
    if (err_code) {
        cpl_msg_error(cpl_func, "mf_io_mkdir -> failed to create : %s", tmpdir_str);
    }
    if (!err_code) {
        err_code = mf_io_rm(tmpdir_str);
    }
    if (!err_code) {
        cpl_msg_info(cpl_func, "mf_io_rm -> removed : %s", tmpdir_str);
    }
    if (err_code) {
        cpl_msg_error(cpl_func, "mf_io_rm -> failed to remove : %s", tmpdir_str);
    }

    /* Test that we can find an output directory for the following tests */
    char *output_dir = cpl_sprintf("%s/%s", pwd_str, "output");
    err_code         = mf_io_access(output_dir);
    if (err_code) {
        cpl_msg_info(cpl_func, "Cannot access output directory: %s", output_dir);
    }
    else {
        cpl_msg_info(cpl_func, "Found Output directory: %s", output_dir);
    }
    cpl_free(output_dir);

    /* If there were any failures in the above file operations then quit now as the
   following tests will also fail */
    if (err_code) {
        /* Show errors and return */
        cpl_errorstate_dump(errstate, CPL_TRUE, cpl_errorstate_dump_one);
        /* Cleanup */
        cpl_free(pwd_str);
        cpl_free(file_str);
        cpl_free(none_str);
        cpl_free(tmpdir_str);
        mf_parameters_delete(params);
        mf_parameters_config_delete(config_parameters);
        return cpl_test_end(0);
    }

    /* Test mv() and symlink() */
    char *tmpfile_str1 = cpl_sprintf("%s/%s/%s", pwd_str, TEMP_DIR, "empty_file1.txt");
    char *tmpfile_str2 = cpl_sprintf("%s/%s/%s", pwd_str, TEMP_DIR, "empty_file2.txt");
    mf_io_mkdir(tmpdir_str);

    FILE *fh = fopen(tmpfile_str1, "w");
    if (fh != NULL) {
        fclose(fh);
    }
    err_code = mf_io_access(tmpfile_str1);
    cpl_msg_info(cpl_func, "test_mf_io() -> ACCESS to empty_file err_code : %d", err_code);
    err_code = mf_io_mv(tmpfile_str1, tmpfile_str2);
    if (!err_code) {
        cpl_msg_info(cpl_func, "mf_io_mv -> %s", tmpfile_str2);
    }
    if (err_code) {
        cpl_msg_error(cpl_func, "mf_io_mv -> failed to move  %s", tmpfile_str1);
    }
    mf_io_rm(tmpfile_str2);
    mf_io_rm(tmpdir_str);
    cpl_free(tmpfile_str1);
    cpl_free(tmpfile_str2);
    cpl_free(tmpdir_str);

    /*
  err_code=mf_io_rm_rf(tmpdir_str,1);
  if ( err_code) cpl_msg_error(cpl_func, "mf_io_rm_rf -> failed to remove : %s",tmpdir_str);
*/
    /* Cleanup */
    cpl_free(pwd_str);
    cpl_free(file_str);
    cpl_free(none_str);


    /* ======================================================= */
    /*          TEST THE mf_io_write_lnfl_configurations()     */
    /* ======================================================= */

    err_code = CPL_ERROR_NONE;

    /* ----------------------------------------------*/
    /* Standard simple test with no errors expected  */
    /* ----------------------------------------------*/
    if (err_code == CPL_ERROR_NONE) {
        cpl_msg_info(cpl_func, "-----------------------------------");
        cpl_msg_info(cpl_func, "Testing standard lnfl configuration:");
        cpl_msg_info(cpl_func, "-----------------------------------");
        err_code = test_mf_io_write_lnfl_configuration(config_parameters, RUN_OK);
        if (err_code == CPL_ERROR_NONE) {
            cpl_msg_info(cpl_func, "Standard lnfl execution test OK\n");
        }
        else {
            cpl_msg_info(cpl_func, "Standard lnfl execution test FAILED\n");
        }
    }


    /* -----------------------------------------------------*/
    /* Standard simple test with invalid path to data files */
    /* -----------------------------------------------------*/
    if (err_code == CPL_ERROR_NONE) {
        cpl_msg_info(cpl_func, "--------------------------------------------------------------------------");
        cpl_msg_info(cpl_func, "Testing error handling of standard lnfl execution with invalid data paths:");
        cpl_msg_info(cpl_func, "--------------------------------------------------------------------------");
        cpl_errorstate old_err_state = cpl_errorstate_get();
        err_code                     = test_mf_io_write_lnfl_configuration(config_parameters, BAD_INPUT);
        if (err_code == CPL_ERROR_NONE) {
            cpl_msg_error(cpl_func, "Standard lnfl execution test passed when it should have failed");
            cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_OUTPUT, "cpl_error failed to catch standard lnfl error!");
        }
        else {
            cpl_msg_info(cpl_func, "Standard lnfl execution FAILED as expected. Error dump as follows:");
            cpl_errorstate_dump(errstate, CPL_TRUE, cpl_errorstate_dump_one);
            cpl_msg_info(cpl_func, "End of error test. Reseting the error stack.");
            cpl_errorstate_set(old_err_state);
            err_code = CPL_ERROR_NONE;
            errstate = cpl_errorstate_get();
        }
    }


    /* ---------------------------------------------------------------*/
    /* Standard simple test with syscall to lnfl forcing a core dump  */
    /* ---------------------------------------------------------------*/
    if (err_code == CPL_ERROR_NONE) {
        cpl_msg_info(cpl_func, "--------------------------------------------------------");
        cpl_msg_info(cpl_func, "Rerun standard with lnfl execution producing core dump:");
        cpl_msg_info(cpl_func, "-------------------------------------------------------");
        cpl_errorstate old_err_state = cpl_errorstate_get();
        err_code                     = test_mf_io_write_lnfl_configuration(config_parameters, CORE_DUMP);
        if (err_code == CPL_ERROR_NONE) {
            cpl_msg_info(cpl_func, "Core Dump Error Capture of lnfl segmentation fault FAILED");
            cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_OUTPUT, "cpl_error failed to catch lnfl core dump!");
        }
        else {
            cpl_msg_info(cpl_func, "Core Dump Error Captured as expected Error dump as follows\n");
            cpl_errorstate_dump(errstate, CPL_TRUE, cpl_errorstate_dump_one);
            cpl_msg_info(cpl_func, "End of error test. Reseting the error stack.\n");
            cpl_errorstate_set(old_err_state);
            err_code = CPL_ERROR_NONE;
        }
    }

    /* ---------------------------------------------------------------*/
    /* Standard simple test with syscall to lnfl leaving data intact  */
    /* ---------------------------------------------------------------*/
    if (err_code == CPL_ERROR_NONE) {
        cpl_msg_info(cpl_func, "-----------------------------------");
        cpl_msg_info(cpl_func, "Rerun standard with lnfl execution:");
        cpl_msg_info(cpl_func, "-----------------------------------");
        err_code = test_mf_io_write_lnfl_configuration(config_parameters, SYS_CALL);
        if (err_code == CPL_ERROR_NONE) {
            cpl_msg_info(cpl_func, "Standard lnfl execution test OK\n");
        }
        else {
            cpl_msg_info(cpl_func, "Standard lnfl execution test FAILED\n");
        }
    }


    /* =========================================== */
    /*                    CLEANUP                  */
    /* =========================================== */

    mf_parameters_delete(params);
    mf_parameters_config_delete(config_parameters);
    /*cpl_test_error(CPL_ERROR_NONE);*/

    /* Show time */
    double ce = cpl_test_get_walltime();
    cpl_msg_info(cpl_func, "test_mf_io() -> Run time: %g min\n", (ce - cs) / 60.);

    /* Show errors and return */
    cpl_errorstate_dump(errstate, CPL_TRUE, cpl_errorstate_dump_one);
    return cpl_test_end(0);
}


/** @cond PRIVATE */

/** @endcond */

/**@}*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Benchmark Tests for mf_io_write_lnfl_configuration
  @param option_flag  Test option, i.e. standard, lnfl execution or forced errors
  @return   cpl_error_code
 */
/*----------------------------------------------------------------------------*/

static cpl_error_code
test_mf_io_write_lnfl_configuration(mf_parameters_config *pars, enum LNFL_TEST_RUN_CODE option_flag)
{
    char              *data_path;
    char              *w_dir;
    double             wn_start;
    double             wn_end;
    const char        *lbl_molecs = "111111111111111111111111111111111111111";
    mf_io_lnfl_config *config;
    cpl_error_code     err;

    cpl_msg_info(cpl_func, "test_mf_io() -> TEST write_lnfl_configuration. Option:%d", option_flag);


    /* Create a parameter config as this will generate the default paths to be used*/
    /* mf_parameters_config *pars = mf_parameters_config_create(); */
    cpl_msg_info(cpl_func, "mf_parameters() telluric_path -> %s", pars->directories.telluric_path);

    /* Define wavelength range */
    wn_start = 3000.0;
    wn_end   = 3500.0;

    /* Define the working directory as the output subdirectory of the current directory*/
    char *cd_str = mf_io_pwd();
    w_dir        = cpl_sprintf("%s/%s", cd_str, "output");

    /* Declare where the HITRAN data files are kept that lnfl needs to read */
    /*  NOTE: We declare where it should be (data_path1) and where it maybe (data_path2)*/
    char *data_path1;
    char *data_path2;
    data_path1 = cpl_sprintf("%s", pars->directories.telluriccorr_data_path);
    data_path2 = cpl_sprintf("%s/%s", pars->directories.telluric_path, "/share/molecfit/data");

    /* Define the expected output files of mf_io_write_lnfl_configuration()*/
    char *tape1 = cpl_sprintf("%s/%s", w_dir, MF_AER_TAPE1_FILE);
    char *tape5 = cpl_sprintf("%s/%s", w_dir, MF_AER_TAPE5_FILE);

    /* Define the expected output files of syscall to lnfl*/
    char *tape3  = cpl_sprintf("%s/%s", w_dir, MF_AER_TAPE3_FILE);
    char *tape6  = cpl_sprintf("%s/%s", w_dir, MF_AER_TAPE6_FILE);
    char *tape7  = cpl_sprintf("%s/%s", w_dir, MF_AER_TAPE7_FILE);
    char *tape10 = cpl_sprintf("%s/%s", w_dir, MF_AER_TAPE10_FILE);

    /* Cleanout any left over junk */
    if (mf_io_access(tape1) == CPL_ERROR_NONE) {
        mf_io_rm(tape1);
    }
    if (mf_io_access(tape5) == CPL_ERROR_NONE) {
        mf_io_rm(tape5);
    }


    /* Create a default config object for the lnfl input (and check it)*/
    config = mf_lnfl_config_create();
    err    = mf_lnfl_config_check(config);
    if (err) {
        cpl_msg_error(cpl_func, "Error in config check");
    }

    /* Using the config object determine which data path holds the required data file*/
    char *data_file;
    data_file = cpl_sprintf("%s/%s/%s", data_path1, MF_HITRAN_PATH, config->line_db);
    if (!mf_io_access(data_file)) {
        cpl_msg_info(cpl_func, "Found data file -> %s", data_file);
        data_path = cpl_strdup(data_path1);
    }
    else {
        cpl_msg_info(cpl_func, "Couldnt find data file -> %s", data_file);
        char *data_file2;
        data_file2 = cpl_sprintf("%s/%s/%s", data_path2, MF_HITRAN_PATH, config->line_db);
        if (!mf_io_access(data_file2)) {
            cpl_msg_info(cpl_func, "Found data file -> %s", data_file2);
            data_path = cpl_strdup(data_path2);
        }
        else {
            cpl_msg_info(cpl_func, "Cannot find data file -> %s", data_file2);
            data_path = NULL;
        }
        cpl_free(data_file2);
    }
    cpl_free(data_file);

    /* Check that we have a valid data path and can continue else return error*/
    if (data_path == NULL) {
        cpl_msg_error(cpl_func, "Error uable to find HITRAN data files");
        return cpl_error_set_message(cpl_func, CPL_ERROR_FILE_NOT_FOUND, "Error uable to find HITRAN data files");
    }
    else {
        cpl_msg_info(cpl_func, "Using data path location: %s", data_path);
    }

    /* If we are running BAD_INPUT then redefine the data_path to the output directory*/
    if (option_flag == BAD_INPUT) {
        /* Use an invalid data path so as to trigger an error that will be tested */
        cpl_free(data_path);
        data_path = cpl_sprintf("%s/%s", cd_str, "output");
    }


    /* Execute the most basic lnfl configuration*/
    if (!err) {
        cpl_msg_info(cpl_func, "Calling mf_io_write_lnfl_configuration");
        err = mf_io_write_lnfl_configuration(data_path, w_dir, wn_start, wn_end, lbl_molecs, config);

        /* Check for a lnfl execution error*/
        if (err) {
            cpl_msg_error(cpl_func, "Error detected in lnfl configuration.");
        }
    }


    /* Check that correct outputs exists */
    if (!err) {
        /* Check that TAPE1  exists */
        if (mf_io_access(tape1) == CPL_ERROR_NONE) {
            cpl_msg_info(cpl_func, "Output file %s exists. OK", tape1);
        }
        else {
            cpl_msg_error(cpl_func, "Output file %s does not exist. FAIL!", tape1);
        }

        /* Check that TAPE5  exists */
        if (mf_io_access(tape5) == CPL_ERROR_NONE) {
            cpl_msg_info(cpl_func, "Output file %s exists. OK", tape5);
        }
        else {
            cpl_msg_error(cpl_func, "Output file %s does not exist. FAIL!", tape5);
            cpl_error_set_message(cpl_func, CPL_ERROR_FILE_NOT_CREATED, MF_AER_TAPE5_FILE);
            err = CPL_ERROR_FILE_NOT_CREATED;
        }
    }

    /* If requested make a system call test on lnfl using these configuration outputs as inputs*/
    if (!err && (option_flag == SYS_CALL || option_flag == CORE_DUMP)) {
        cpl_msg_info(cpl_func, "Testing lnfl system call on configuration");
        double runtime = 0.0;
        char  *lnfl_syscall;
        lnfl_syscall = cpl_sprintf("%s/%s", pars->directories.telluric_path, MF_BIN_LNFL);

        if (option_flag == CORE_DUMP) {
            cpl_msg_info(cpl_func, "Removing input files TAPE1 and TAPE5 to force error");

            /* Remove input tape files before executing lnfl this will cause a core dump*/
            mf_io_rm(tape1);
            mf_io_rm(tape5);
        }

        err = mf_io_system(lnfl_syscall, w_dir, &runtime);

        /* If execution is ok then check for the outputs*/
        if (!err) {
            if (mf_io_access(tape3) == CPL_ERROR_NONE && mf_io_access(tape6) == CPL_ERROR_NONE &&
                mf_io_access(tape7) == CPL_ERROR_NONE && mf_io_access(tape10) == CPL_ERROR_NONE) {
                cpl_msg_info(cpl_func, "All lnfl output files exists. OK");
            }
            else {
                cpl_msg_error(cpl_func, "System Call %s failure!", lnfl_syscall);
                err = cpl_error_set_message(cpl_func, CPL_ERROR_FILE_NOT_CREATED, "lnfl output tapes");
                /*err=CPL_ERROR_FILE_NOT_CREATED;*/
            }
        }

        cpl_free(lnfl_syscall);

    } /* end SYS_CALL checks*/


    /* Cleanup */
    if (!err && option_flag != SYS_CALL) {
        cpl_msg_info(cpl_func, "Cleaning out output directory\n");
        mf_io_rm(tape1);
        mf_io_rm(tape5);
    }
    mf_lnfl_config_delete(config);
    cpl_free(cd_str);
    cpl_free(w_dir);
    cpl_free(data_path);
    cpl_free(data_path1);
    cpl_free(data_path2);
    cpl_free(tape1);
    cpl_free(tape5);
    cpl_free(tape3);
    cpl_free(tape6);
    cpl_free(tape7);
    cpl_free(tape10);

    if (err) {
        cpl_msg_error(cpl_func, "Error in mf_io_write_lnfl_configuration test. FAIL!");
    }

    return err;
}
