/*                                                                            *
 *   This file is part of the ESPRESSO Pipeline                               *
 *   Copyright (C) 2006 European Southern Observatory                         *
 *                                                                            *
 *   This library is free software; you can redistribute it and/or modify     *
 *   it under the terms of the GNU General Public License as published by     *
 *   the Free Software Foundation; either version 2 of the License, or        *
 *   (at your option) any later version.                                      *
 *                                                                            *
 *   This program is distributed in the hope that it will be useful,          *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 *   GNU General Public License for more details.                             *
 *                                                                            *
 *   You should have received a copy of the GNU General Public License        *
 *   along with this program; if not, write to the Free Software              *
 *   Foundation, 51 Franklin St, Fifth Floor, Boston, MA  02111-1307  USA     *
 *                                                                            */

/*
 * $Author: asegovia $
 * $Date: 2015-05-01 15:13:41 $
 * $Revision:     $
 * $Name: not supported by cvs2svn $
 */


#include <espdr_contam.h>
#include <espdr_overscan.h>
#include <espdr_instrument.h>
#include <espdr_background.h>
#include <espdr_blaze.h>
#include <espdr_wave_cal.h>
#include <espdr_orders.h>
#include <espdr_pixels.h>
#include <stdlib.h>

/*----------------------------------------------------------------------------
 Functions code
 ----------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/**
 @brief     Add extra QC keywords
 @param     keywords    property list to add the extra KWs
 @param     ext_nb      number of extensions
 @return    0 if everything is ok
 */
/*---------------------------------------------------------------------------*/

static cpl_error_code espdr_add_extra_qc(cpl_propertylist** keywords,
                                         const int ext_nb) {

    const char* ksuff[2] = {" BLUE", " RED"};
    cpl_propertylist_append_int(*keywords,"ESO QC PROC VERSION", 2);
    int* ord_ref = espdr_get_ord_ref(*keywords, "espdr_cal_contam");

    espdr_add_qc_key_stat_ord_pri(keywords, ext_nb, "MAX FLUX", "DUMMY", CPL_TRUE, CPL_TRUE,
                                  CPL_FALSE, CPL_FALSE, CPL_FALSE);

    espdr_add_qc_key_contam_ord_ref(keywords, ext_nb, ord_ref, "MAX FLUX",ksuff,
                                    CPL_TRUE, CPL_TRUE, CPL_FALSE, CPL_FALSE,
                                    CPL_FALSE);
    cpl_free(ord_ref);
    return cpl_error_get_code();
}



/*---------------------------------------------------------------------------*/
/**
 @brief    Interpret the command line options and execute the data processing
 @param    parameters     the parameters list
 @param    frameset   the frames list

 In case of failure the cpl_error_code is set.
 */
/*---------------------------------------------------------------------------*/

int espdr_cal_contam(cpl_parameterlist *parameters, cpl_frameset *frameset,
		const char* recipe_id)
{
	cpl_frameset *used_frames = NULL;
	cpl_error_code my_error = CPL_ERROR_NONE;

    const int rec_ntags = 11;
    const char* rec_tags[11] = {
        ESPDR_CCD_GEOM, ESPDR_PRO_CATG_MBIAS_RES,
        ESPDR_PRO_CATG_HOT_PIXELS, ESPDR_PRO_CATG_MDARK,
        ESPDR_PRO_CATG_BAD_PIXELS,
        ESPDR_PRO_CATG_ORDERS_A, ESPDR_PRO_CATG_ORDERS_B,
        ESPDR_CONTAM_RAW_FP, ESPDR_CONTAM_RAW_THAR, ESPDR_CONTAM_RAW_LFC,
        ESPDR_ORDERS_MASK
    };
    int is_required[11] = {1, 0, 1, 0, 1, 1, 1, 0, 0, 0, 0};

	cpl_msg_set_level(CPL_MSG_INFO);
	//cpl_msg_set_level(CPL_MSG_DEBUG);

    espdr_CCD_geometry *CCD_geom = NULL;
    espdr_inst_config *inst_config = NULL;
    espdr_qc_keywords *qc_kws = NULL;
    espdr_OVSC_param *OVSC_param = NULL;
    espdr_CAL_CONTAM_param *CAL_CONTAM_param = NULL;
    cpl_frame* CCD_geom_frame = NULL;
    cpl_frame* inst_config_frame = NULL;

    espdr_msg("Starting cal_contam");

    espdr_ensure(espdr_check_input_tags(frameset,
                                        rec_tags, is_required,
                                        rec_ntags) != CPL_ERROR_NONE,
                 cpl_error_get_code(), "Wrong input tag!");
    espdr_ensure(espdr_check_input_inst_config(frameset) != CPL_ERROR_NONE,
                 cpl_error_get_code(), "Wrong input tag!");
    espdr_ensure(espdr_check_input_flat(frameset, 0) != CPL_ERROR_NONE,
                 cpl_error_get_code(), "Wrong flat input tag!");

    /* Identify the RAW and CALIB frames in the input frameset */
    espdr_ensure(espdr_dfs_set_groups(frameset) != CPL_ERROR_NONE,
                 cpl_error_get_code(),
                 "DFS setting groups failed. Expected inputs are:\n"
                 "raw contamination image, tagged as CONTAM_FP or CONTAM_LFC or CONTAM_THAR\n"
                 "CCD geometry table, tagged as CCD_GEOM\n"
                 "instrument configuration table, tagged as MASTER_INST_CONFIG or INST_CONFIG\n"
                 "ThAr lines reference table, tagged as REF_LINE_TABLEA or REF_LINE_TABLE_B\n"
                 "master bias residuals image, tagged as MASTER_BIAS_RES\n"
                 "hot pixel mask image, tagged as HOT_PIXEL_MASK\n"
                 "bad pixel mask image, tagged as BAD_PIXEL_MASK\n"
                 "orders definition images for fibres A and B, tagged as ORDER_TABLE_A & ORDER_TABLE_B\n"
                 "orders profile images for fibres A and B, tagged as ORDER_PROFILE_A & ORDER_PROFILE_B\n"
                 "flat images for fibres A and B, tagged as FLAT_A & FLAT_B\n");

    /* AMO Changed way to read CCD geom frame and to init relevant recipe
     * internal parameters structures to simplify this calling function
     */
    OVSC_param        = espdr_parameters_OVSC_init(recipe_id, parameters);
    CCD_geom_frame    = espdr_frame_find(frameset, ESPDR_CCD_GEOM);
    espdr_ensure(CCD_geom_frame == NULL,cpl_error_get_code(),
                 "CCD geometry frame not found!");
    CCD_geom          = espdr_CCD_geom_init(parameters, CCD_geom_frame);
    inst_config_frame = espdr_get_inst_config(frameset);
    espdr_ensure(inst_config_frame == NULL,cpl_error_get_code(),
                 "Instrument config frame not found!");
    inst_config       = espdr_inst_config_init(parameters,
                                               CCD_geom->ext_nb,
                                               inst_config_frame);
    CAL_CONTAM_param  = espdr_CAL_CONTAM_param_init(recipe_id, parameters);

    used_frames = cpl_frameset_new();
    my_error = cpl_frameset_insert(used_frames,
                                   cpl_frame_duplicate(CCD_geom_frame));
    my_error = cpl_frameset_insert(used_frames,
                                   cpl_frame_duplicate(inst_config_frame));

    /* Filling up the DRS QC KWs structure */
    qc_kws = (espdr_qc_keywords *)cpl_malloc( sizeof(espdr_qc_keywords));
    my_error = espdr_fill_qc_keywords(inst_config, qc_kws);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Filling QC KWs failed: %s",
                 cpl_error_get_message_default(my_error));
    espdr_ensure(qc_kws == NULL, CPL_ERROR_ILLEGAL_OUTPUT,
                 "QC KWs structure is NULL");

    cpl_frame *contam_frame = NULL;
    if (espdr_frameset_has_frame(frameset, ESPDR_CONTAM_RAW_FP)) {
        contam_frame = espdr_frame_find(frameset, ESPDR_CONTAM_RAW_FP);
    } else {
        if (espdr_frameset_has_frame(frameset, ESPDR_CONTAM_RAW_THAR)) {
            contam_frame = espdr_frame_find(frameset, ESPDR_CONTAM_RAW_THAR);
        } else {
            if (espdr_frameset_has_frame(frameset, ESPDR_CONTAM_RAW_LFC)) {
                contam_frame = espdr_frame_find(frameset, ESPDR_CONTAM_RAW_LFC);
            } else {
                espdr_msg_error("No CONTAM raw frame, exiting");
                return (CPL_ERROR_NULL_INPUT);
            }
        }
    }
    espdr_msg("CONTAM TAG: %s", cpl_frame_get_tag(contam_frame));
    my_error = cpl_frameset_insert(used_frames, cpl_frame_duplicate(contam_frame));

    cpl_propertylist *keywords = NULL;
    /* Load contam keywords from the first image (primary header) */
	const char *input_filename = cpl_frame_get_filename(contam_frame);
    espdr_msg("KEYWORDS for CAL_CONTAM input filename: %s", input_filename);
    keywords = cpl_propertylist_load(input_filename, 0);
    espdr_ensure(keywords == NULL, CPL_ERROR_ILLEGAL_OUTPUT, "keywords are NULL");

#if SAVE_DEBUG_PRODUCT_PREPROCESSING
    // Adding artificially the ARCFILE if not present - final filename - to be removed when using ESO DFS
    if (cpl_propertylist_has(keywords, "ARCFILE") == 0) {
        char *arc_filename = strrchr(input_filename, '/')+1;
        espdr_msg("ARCFILE not existing - updating the filename: %s", arc_filename);
        my_error = cpl_propertylist_update_string(keywords, "ARCFILE", arc_filename);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "Updating the ARCFILE KW (%s) failed: %s",
                     arc_filename, cpl_error_get_message_default(my_error));
    }
#endif

    /* Parsing contam source */
    const char *contam_src_kw = cpl_propertylist_get_string(keywords, inst_config->dpr_type_kw);
    espdr_msg( "%s = %s", inst_config->dpr_type_kw, contam_src_kw);
    char contam_src[128];
    char *separator;
    char to_split[64];
    strcpy(to_split, contam_src_kw);
    separator = strtok(to_split, ",");
    while(separator != NULL) {
        strcpy(contam_src, separator);
        separator = strtok(NULL, ",");
    }

    int contam_src_len = strlen(contam_src);
    if (isdigit(contam_src[contam_src_len-1])) {
        contam_src[contam_src_len-1] = '\0';
    }
    espdr_msg("contam_src: %s", contam_src);

    // Background subtraction parameter managemenet
    char background_sw[10];
    sprintf(background_sw, "%s", "on");

    cpl_parameter *par_background_sw =
    cpl_parameterlist_find(parameters,"espdr.espdr_cal_contam.background_sw");
    int background_sw_set = cpl_parameter_get_default_flag(par_background_sw);
    if (background_sw_set == 1) {
        sprintf(background_sw, "%s", CAL_CONTAM_param->background_sw);
        cpl_parameter_set_string(par_background_sw, CAL_CONTAM_param->background_sw);
    }
    espdr_msg("Background SW is %s", background_sw);

    /* Produce the contam frame */
    int remove_bias_res = 0;
    if (inst_config->raw_bias_limit_nb > 0) {
        remove_bias_res = 1;
    }

    int *orders_nb_per_fibre = (int *)cpl_calloc(inst_config->fibres_nb, sizeof(int));
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        orders_nb_per_fibre[i] = 0;
        for (int j = 0; j < CCD_geom->ext_nb; j++) {
            orders_nb_per_fibre[i] += inst_config->orders_nb[i*CCD_geom->ext_nb+j];
        }
    }

    double *RON = (double *)cpl_calloc(CCD_geom->total_output_nb, sizeof(double));
    int **cosmics_nb = (int **)cpl_calloc(inst_config->fibres_nb, sizeof(int *));
    double **snr_per_fibre = (double **) cpl_calloc(inst_config->fibres_nb,sizeof(int *));
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        snr_per_fibre[i] = (double *) cpl_calloc(orders_nb_per_fibre[i],sizeof(double));
    }
    cpl_imagelist *contam_bkgr_subs = NULL;
    cpl_image **flat_corr_flux_table = (cpl_image **)cpl_malloc(inst_config->fibres_nb * sizeof(cpl_image *));
    cpl_image **flat_corr_err_table = (cpl_image **)cpl_malloc(inst_config->fibres_nb * sizeof(cpl_image *));
    cpl_image **flat_corr_qual_table = (cpl_image **)cpl_malloc(inst_config->fibres_nb * sizeof(cpl_image *));
    my_error = espdr_process_calib_till_extraction(frameset, parameters, recipe_id,
                                                   contam_frame, used_frames,
                                                   inst_config, CCD_geom,
                                                   qc_kws, keywords,
                                                   orders_nb_per_fibre,
                                                   remove_bias_res, contam_src,
                                                   background_sw,
                                                   RON, cosmics_nb, snr_per_fibre,
                                                   &contam_bkgr_subs,
                                                   flat_corr_flux_table,
                                                   flat_corr_err_table,
                                                   flat_corr_qual_table);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_process_calib_till_extraction failed: %s",
                 cpl_error_get_message_default(my_error));

	/* QC check */
    espdr_msg("QC check");
    my_error = espdr_cal_contam_QC(inst_config, CCD_geom, qc_kws,
                                   contam_src, RON,
                                   flat_corr_flux_table[0], // S2D flat corrected fibre A (science fibre)
                                   flat_corr_qual_table[0], // S2D flat corrected fibre A (quality map)
                                   flat_corr_flux_table[1], // S2D flat corrected fibre B (calib fibre)
                                   flat_corr_qual_table[1], // S2D flat corrected fibre B (quality map)
                                   &keywords);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_cal_contam_QC failed: %s",
                 cpl_error_get_message_default(my_error));

	/* saving products */
    char pro_catg_string[64];
    cpl_frame* product_frame;
    cpl_frameset *frameset_orig = cpl_frameset_duplicate(frameset);
	for (int fibre = 0; fibre < inst_config->fibres_nb; fibre++) {
        char s2d_contam_name[64];
        sprintf(s2d_contam_name, "%s_CONTAM_S2D_%c.fits", inst_config->instrument, fibre_name[fibre]);
        sprintf(pro_catg_string, "%s_%c", "CONTAM_S2D", fibre_name[fibre]);
        my_error = espdr_dfs_save_S2D(flat_corr_flux_table[fibre],
                                      flat_corr_err_table[fibre],
                                      flat_corr_qual_table[fibre],
                                      frameset_orig, used_frames,
                                      parameters, keywords,
                                      s2d_contam_name,
                                      pro_catg_string, recipe_id);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_dfs_save_S2D failed: %s for file %s",
                     cpl_error_get_message_default(my_error),
                     s2d_contam_name);

        espdr_frame_new(&product_frame,
                        s2d_contam_name,
                        CPL_FRAME_GROUP_PRODUCT,
                        CPL_FRAME_LEVEL_FINAL,
                        CPL_FRAME_TYPE_IMAGE,
                        pro_catg_string);
        cpl_frame_set_filename(product_frame, s2d_contam_name);
        cpl_frameset_insert(frameset, product_frame);
	}

    char bkgr_subs_name[64];
    sprintf(bkgr_subs_name, "%s_CONTAM_%s_B.fits", inst_config->instrument,contam_src);
    sprintf(pro_catg_string, "CONTAM_%s", contam_src);
    my_error = cpl_propertylist_update_string(keywords, PRO_CATG_KW, pro_catg_string);

    cpl_propertylist **keywords_ext = (cpl_propertylist**)cpl_malloc(
                                CCD_geom->ext_nb * sizeof(cpl_propertylist*));
    const char *input_filename_ext = cpl_frame_get_filename(contam_frame);
    for (int j = 0; j < CCD_geom->ext_nb; j++) {
        keywords_ext[j] = cpl_propertylist_load(input_filename_ext,j+1);
    }

    if ((strcmp(inst_config->instrument, "HARPN") != 0) &&
        (strcmp(inst_config->instrument, "CORALIE") != 0)) {
        my_error = espdr_add_extra_qc(&keywords, CCD_geom->ext_nb);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_add_extra_qc failed: %s ",
                     cpl_error_get_message_default(my_error));
    }

	my_error = espdr_dfs_image_save(frameset_orig, parameters,
                                    used_frames, recipe_id,
                                    keywords, keywords_ext,
                                    bkgr_subs_name,
                                    contam_bkgr_subs,
                                    CPL_TYPE_FLOAT, CCD_geom);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "contam_bkgr_subs image saving has failed: %s",
                 cpl_error_get_message_default(my_error));

	espdr_frame_new(&product_frame,
                    bkgr_subs_name,
                    CPL_FRAME_GROUP_PRODUCT,
                    CPL_FRAME_LEVEL_FINAL,
                    CPL_FRAME_TYPE_IMAGE,
                    pro_catg_string);
    cpl_frame_set_filename(product_frame,bkgr_subs_name);
    cpl_frameset_insert(frameset, product_frame);

    /* cleaning memory */
    espdr_msg("Cleaning memory");
	for (int i = 0; i < inst_config->fibres_nb; i++) {
		cpl_image_delete(flat_corr_flux_table[i]);
		cpl_image_delete(flat_corr_err_table[i]);
		cpl_image_delete(flat_corr_qual_table[i]);
        cpl_free(cosmics_nb[i]);
        cpl_free(snr_per_fibre[i]);
	}
    cpl_free(flat_corr_flux_table);
    cpl_free(flat_corr_err_table);
    cpl_free(flat_corr_qual_table);
    cpl_imagelist_delete(contam_bkgr_subs);
    cpl_free(cosmics_nb);
    cpl_free(snr_per_fibre);
    cpl_free(orders_nb_per_fibre);

    for (int i = 0; i < CCD_geom->ext_nb; i++) {
        cpl_propertylist_delete(keywords_ext[i]);
    }
    cpl_free(keywords_ext);
    cpl_propertylist_delete(keywords);
    cpl_free(qc_kws);

    cpl_frameset_delete(used_frames);
    cpl_frameset_delete(frameset_orig);
    cpl_free(RON);

    espdr_parameters_OVSC_delete(OVSC_param);
    espdr_parameters_CAL_CONTAM_delete(CAL_CONTAM_param);
    espdr_parameters_CCD_geometry_delete(CCD_geom);
    espdr_parameters_inst_config_delete(inst_config);

	return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Create the structure for CAL_CONTAM input parameters
 @param    recipe_id	recipe identifier
 @param    list			list of input parameters
 @param    p			input CAL_CONTAM parameters structure
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_parameters_CAL_CONTAM_create(const char* recipe_id,
                                                  cpl_parameterlist *list,
                                                  espdr_CAL_CONTAM_param *p) {
    espdr_ensure(recipe_id == NULL,
                 CPL_ERROR_NULL_INPUT, "The recipe ID is NULL");
    
	/* check parameters */
	espdr_ensure(list == NULL, CPL_ERROR_NULL_INPUT, 
		     "The parameters list NULL");
    
	cpl_error_code error_got;
    char comment[COMMENT_LENGTH];
    int bkgr_x_min = BKGR_X_MIN;
    int bkgr_x_max = BKGR_X_MAX;
    int bkgr_y_min = BKGR_Y_MIN;
    int bkgr_y_max = BKGR_Y_MAX;

	/* Fill the parameter list */
    error_got = espdr_parameters_new_string(recipe_id, list,
                                            "background_sw",
                                            p->background_sw,
                                            "Background subtraction activation (on/off)");
	espdr_ensure(error_got != CPL_ERROR_NONE, error_got,
                 "Error adding parameter background_sw to the list");
    
	//AMO: instrument mode dependent */
    sprintf(comment,
            "Grid size in x used to calculate the background, between: %d and %d",
            bkgr_x_min,bkgr_x_max);
    error_got = espdr_parameters_new_range_int(recipe_id, list,
                                               "contam_bkgr_grid_size_x",
                                               p->bkgr_grid_size_x,
                                               bkgr_x_min, bkgr_x_max,
                                               comment);
    espdr_ensure(error_got != CPL_ERROR_NONE, error_got,
                 "Error adding parameter bkgr_grid_size_x to the list");

    sprintf(comment,
            "Grid size in y used to calculate the background, between: %d and %d",
            bkgr_y_min,bkgr_y_max);
    error_got = espdr_parameters_new_range_int(recipe_id, list,
                                               "contam_bkgr_grid_size_y",
                                               p->bkgr_grid_size_y,
                                               bkgr_y_min, bkgr_y_max,
                                               comment);
    espdr_ensure(error_got != CPL_ERROR_NONE, error_got,
                 "Error adding parameter bkgr_grid_size_y to the list");
    
    error_got = espdr_parameters_new_string(recipe_id, list,
                                            "contam_extraction_method",
                                            p->extraction_method,
                                            "Method used to extract orders");
    espdr_ensure(error_got != CPL_ERROR_NONE, error_got,
                 "Error adding parameter extraction_method to the list");
    
    error_got = espdr_parameters_new_double(recipe_id, list,
                                            "contam_extraction_ksigma",
                                            p->extraction_ksigma,
                                            "ksigma used to extract orders");
    espdr_ensure(error_got != CPL_ERROR_NONE, error_got,
                 "Error adding parameter extraction_method to the list");

	return (CPL_ERROR_NONE);
}


/*----------------------------------------------------------------------------*/
/**
 @brief    Delete the structure for CAL_CONTAM input parameters
 @param    p	input CAL_CONTAM parameters structure
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_parameters_CAL_CONTAM_delete(espdr_CAL_CONTAM_param* p) {
	
    cpl_free((void *)p->background_sw);
    cpl_free((void *)p->extraction_method);
    cpl_free(p);
    p = NULL;
    
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief     Get the cal_contam recipe parameters
 @param     recipe_id   recipe ID
 @param     param_list  parameters list
 @param     CAL_CONTAM_param  CAL_CONTAM parameters structure
 @return    CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_parameters_CAL_CONTAM_get(const char* recipe_id,
                                         cpl_parameterlist* param_list,
                                         espdr_CAL_CONTAM_param *CAL_CONTAM_param) {
	
    espdr_ensure(recipe_id == NULL,
                 CPL_ERROR_NULL_INPUT, "The recipe ID is NULL");
    
	/* check parameters */
	espdr_ensure(param_list == NULL, CPL_ERROR_NULL_INPUT,
				 "Parameters list is NULL");
	
	/* Fill the structure */
    CAL_CONTAM_param->background_sw = espdr_parameters_get_string(recipe_id,
                                                                  param_list,
                                                        "background_sw");

    CAL_CONTAM_param->bkgr_grid_size_x = espdr_parameters_get_int(recipe_id,
                                                                  param_list,
                                                        "contam_bkgr_grid_size_x");
    
    CAL_CONTAM_param->bkgr_grid_size_y = espdr_parameters_get_int(recipe_id,
                                                                  param_list,
                                                        "contam_bkgr_grid_size_y");
    
    CAL_CONTAM_param->extraction_method = espdr_parameters_get_string(recipe_id,
                                                                      param_list,
                                                        "contam_extraction_method");
    
    CAL_CONTAM_param->extraction_ksigma = espdr_parameters_get_double(recipe_id,
                                                                      param_list,
                                                        "contam_extraction_ksigma");
    
	return cpl_error_get_code();
}


/*---------------------------------------------------------------------------*/
/**
 @brief     print the CAL_CONTAM parameters
 @param     CAL_CONTAM_param  CAL_CONTAM parameters structure
 @return    CPL_ERROR_NONE iff OK
 */
/*---------------------------------------------------------------------------*/

cpl_error_code espdr_parameters_CAL_CONTAM_print(
                                espdr_CAL_CONTAM_param *CAL_CONTAM_param) {
	
	espdr_msg("\tCAL_CONTAM parameters: ");
    espdr_msg("\t\tCAL_CONTAM background subtraction = %s",
              CAL_CONTAM_param->background_sw);
    espdr_msg("\t\tCAL_CONTAM background grid size x = %d",
              CAL_CONTAM_param->bkgr_grid_size_x);
    espdr_msg("\t\tCAL_CONTAM background grid size y = %d",
              CAL_CONTAM_param->bkgr_grid_size_y);
    espdr_msg("\t\tCAL_CONTAM extraction method = %s",
              CAL_CONTAM_param->extraction_method);
    espdr_msg("\t\tCAL_CONTAM extraction ksigma = %.2f",
              CAL_CONTAM_param->extraction_ksigma);
    
	return (CPL_ERROR_NONE);
}


/*----------------------------------------------------------------------------*/
/**
 @brief    Init CAL_CONTAM_param structure
 
 @param    recipe_id    recipe name
 @param    parameters   recipe input parameters
 
 @return   allocated structure iff OK
 */
/*----------------------------------------------------------------------------*/

espdr_CAL_CONTAM_param * espdr_CAL_CONTAM_param_init(const char *recipe_id, 
					 cpl_parameterlist *parameters) {
    
    espdr_CAL_CONTAM_param *CAL_CONTAM_param =
    (espdr_CAL_CONTAM_param *)cpl_malloc(sizeof(espdr_CAL_CONTAM_param));
    
    /* Read the cal_contam parameters */
	espdr_parameters_CAL_CONTAM_get(recipe_id, parameters, CAL_CONTAM_param);
	espdr_parameters_CAL_CONTAM_print(CAL_CONTAM_param);
    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        return NULL;
    } else {
        return CAL_CONTAM_param;
    }
}


/*----------------------------------------------------------------------------*/
/**
 @brief
 @param
 @param
 @param
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_process_calib_till_extraction(cpl_frameset *frameset,
                                                   cpl_parameterlist *parameters,
                                                   const char *RECIPE_ID,
                                                   cpl_frame *raw_frame,
                                                   cpl_frameset *used_frames,
                                                   espdr_inst_config *inst_config,
                                                   espdr_CCD_geometry *CCD_geom,
                                                   espdr_qc_keywords *qc_kws,
                                                   cpl_propertylist *keywords,
                                                   int *orders_nb_per_fibre,
                                                   int remove_bias_res_flag,
                                                   char *fibre_b,
                                                   char *background_sw,
                                                   double *ron,
                                                   int **cosmics_nb,
                                                   double **snr_per_fibre,
                                                   cpl_imagelist **raw_bkgr_subs,
                                                   cpl_image **flat_corr_flux_RE,
                                                   cpl_image **flat_corr_err_RE,
                                                   cpl_image **flat_corr_qual_RE) {
    
    cpl_error_code my_error = CPL_ERROR_NONE;
    
    espdr_msg("Start processing %s", RECIPE_ID);
    
    int recipes[10] = {1, 1, 1, 1, 1, 0, 0, 0, 0, 0};
    char recipe_name[32];
    if (strcmp(RECIPE_ID, "espdr_cal_contam") == 0) {
        strcpy(recipe_name, "CONTAM");
    } else {
        strcpy(recipe_name, "EFF_AB");
    }
    my_error = espdr_compute_calibrations_intervals(frameset, inst_config, keywords,
                                                    NULL, NULL, NULL, recipe_name, recipes);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_compute_calibrations_intervals failed: %s",
                 cpl_error_get_message_default(my_error));

    /* Extract master BIAS residuals frame if it was generated and is needed */
    cpl_frame* master_bias_frame;
    if ((inst_config->raw_bias_limit_nb > 0) && (remove_bias_res_flag == 1)) {
        master_bias_frame = espdr_get_mbias_from_set(frameset, used_frames);
    }
    
    /* Extract hot pixels mask frame */
    cpl_frame* hot_pixels_frame = espdr_get_hpixmap_from_set(frameset, used_frames);
    
    /* Extract master dark frame */
    cpl_frame* master_dark_frame = espdr_get_mdark_from_set(frameset, used_frames);
    
    /* Extract bad pixels mask frame */
    cpl_frame* bad_pixels_frame = espdr_get_bpixmap_from_set(frameset, used_frames);
    
    /* Extract orders mask frame */
    cpl_frame* orders_mask_frame = espdr_get_orders_mask_from_set(frameset, used_frames);
    
    /* Extract orders maps frame */
    cpl_table ***orders_coeffs = NULL;
    espdr_get_orders_coeff_from_set(frameset, used_frames, inst_config->fibres_nb,
                                    CCD_geom->ext_nb, &orders_coeffs);
    
    /* Extract order profile frame per fibre */
    cpl_frame **order_profile_fibre_frame = NULL;
    espdr_get_order_profile_from_set(frameset, used_frames, inst_config->fibres_nb,
                                     CCD_geom->ext_nb, &order_profile_fibre_frame);
    
    /* Extract flat frame per fibre */
    cpl_frame** flat_fibre_frame = NULL;
    espdr_get_flat_fibre_from_set(frameset, used_frames, inst_config->fibres_nb,
                                  CCD_geom->ext_nb, &flat_fibre_frame);
    
    cpl_imagelist *master_bias_list = NULL;
    if ((inst_config->raw_bias_limit_nb > 0) && (remove_bias_res_flag == 1)) {
        /* Extract all the outputs from the master bias frame */
        master_bias_list = cpl_imagelist_new();
        my_error = espdr_extract_real_outputs(master_bias_frame, CPL_TYPE_DOUBLE,
                                              CCD_geom, 0, &master_bias_list);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_extract_real_outputs failed: %s",
                     cpl_error_get_message_default(my_error));
    }
    
    /* Extract all the outputs from the hot pixels frame */
    cpl_imagelist *hot_pixels_list = cpl_imagelist_new();
    my_error = espdr_extract_real_outputs(hot_pixels_frame, CPL_TYPE_INT,
                                          CCD_geom, 0, &hot_pixels_list);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_extract_real_outputs failed: %s",
                 cpl_error_get_message_default(my_error));
    
    /* Extract all the outputs from the master dark frame if provided */
    cpl_imagelist *master_dark_list = NULL;
    cpl_imagelist *master_dark_pxl_nb_list = NULL;
    if (master_dark_frame != NULL) {
        master_dark_list = cpl_imagelist_new();
        my_error = espdr_extract_real_outputs(master_dark_frame, CPL_TYPE_DOUBLE,
                                              CCD_geom, 0, &master_dark_list);
        master_dark_pxl_nb_list = cpl_imagelist_new();
        my_error = espdr_extract_real_outputs(master_dark_frame, CPL_TYPE_INT,
                                              CCD_geom, CCD_geom->ext_nb,
                                              &master_dark_pxl_nb_list);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_extract_real_outputs failed: %s",
                     cpl_error_get_message_default(my_error));
    }
    
    /* Extract all the outputs from the bad pixels frame */
    cpl_imagelist *bad_pixels_list = cpl_imagelist_new();
    my_error = espdr_extract_real_outputs(bad_pixels_frame, CPL_TYPE_INT,
                                          CCD_geom, 0, &bad_pixels_list);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_extract_raw_outputs failed: %s",
                 cpl_error_get_message_default(my_error));
    
    /* Extract all the outputs from the orders mask frame if provided */
    cpl_imagelist *orders_mask_list = NULL;
    if (orders_mask_frame != NULL) {
        orders_mask_list = cpl_imagelist_new();
        my_error = espdr_extract_extensions(orders_mask_frame, CPL_TYPE_INT,
                                            CCD_geom->ext_nb, &orders_mask_list);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_extract_real_outputs failed: %s",
                     cpl_error_get_message_default(my_error));
    }
    
    /* Extract extensions from order profile frames */
    cpl_imagelist **order_profile_fibre_imagelist = (cpl_imagelist **)cpl_malloc
                                    (inst_config->fibres_nb * sizeof(cpl_imagelist *));
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        order_profile_fibre_imagelist[i] = cpl_imagelist_new();
        my_error = espdr_extract_extensions(order_profile_fibre_frame[i], CPL_TYPE_DOUBLE,
                                            CCD_geom->ext_nb, &order_profile_fibre_imagelist[i]);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_extract_extensions failed: %s",
                     cpl_error_get_message_default(my_error));
    }
    
    /* Extract extensions from flat frames */
    cpl_image **flat_table = (cpl_image **)cpl_malloc(inst_config->fibres_nb*sizeof(cpl_image *));
    cpl_image **flat_error_table = (cpl_image **)cpl_malloc(inst_config->fibres_nb*sizeof(cpl_image *));
    cpl_image **flat_quality_table = (cpl_image **)cpl_malloc(inst_config->fibres_nb*sizeof(cpl_image *));
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        const char *flat_filename = cpl_frame_get_filename(flat_fibre_frame[i]);
        flat_table[i] = cpl_image_load(flat_filename, CPL_TYPE_DOUBLE, 0, 1);
        flat_error_table[i] = cpl_image_load(flat_filename, CPL_TYPE_DOUBLE, 0, 2);
        flat_quality_table[i] = cpl_image_load(flat_filename, CPL_TYPE_INT, 0, 3);
    }
    my_error = cpl_error_get_code();
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Extracting FLAT image failed: %s",
                 cpl_error_get_message_default(my_error));
    
    /* Extract raw and real images from the raw frame */
    cpl_imagelist *raw_imagelist = cpl_imagelist_new();
    cpl_imagelist *real_imagelist = cpl_imagelist_new();
    if (inst_config->inst_type == NIR) {
        my_error = espdr_extract_extensions(raw_frame, CPL_TYPE_DOUBLE,
                                            CCD_geom->ext_nb, &raw_imagelist);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_extract_extensions failed: %s",
                     cpl_error_get_message_default(my_error));
    } else {
        my_error = espdr_extract_raw_outputs_one_frame(raw_frame, CPL_TYPE_DOUBLE,
                                                       CCD_geom, &raw_imagelist);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_extract_raw_outputs_one_frame failed: %s",
                     cpl_error_get_message_default(my_error));
    }
    my_error = espdr_extract_real_outputs_one_frame(raw_frame, CPL_TYPE_DOUBLE,
                                                    CCD_geom, &real_imagelist);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_extract_real_outputs_one_frame failed: %s",
                 cpl_error_get_message_default(my_error));
    
    /* hot & bad pixel mask creation */
    espdr_msg("Creating hot & bad pixels mask");
    // Update bad pixel mask with all the nans from raw frames
    if (inst_config->inst_type == NIR) {
        cpl_frameset *raw_frames_to_reduce = cpl_frameset_new();
        cpl_frameset_insert(raw_frames_to_reduce, cpl_frame_duplicate(raw_frame));
        my_error = espdr_update_bad_pixel_mask(raw_frames_to_reduce, CCD_geom,
                                               cpl_imagelist_get(bad_pixels_list, 0));
        cpl_frameset_delete(raw_frames_to_reduce);
    }
    
    cpl_imagelist *pixels_mask = cpl_imagelist_new();
    my_error = espdr_create_hot_bad_pixels_mask(hot_pixels_list, bad_pixels_list,
                                                &pixels_mask);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_create_hot_bad_pixels_mask failed: %s",
                 cpl_error_get_message_default(my_error));
    
    cpl_imagelist *merged_pixels_mask = cpl_imagelist_new();
    my_error = espdr_image_merge_2_dim(pixels_mask, CCD_geom, CPL_TYPE_INT,
                                       &merged_pixels_mask);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_image_merge_2_dim failed: %s",
                 cpl_error_get_message_default(my_error));
    
    cpl_imagelist *geometry_corrected_pixels_mask = cpl_imagelist_new();
    my_error = espdr_correct_geometry(merged_pixels_mask, CCD_geom,
                                      geometry_corrected_pixels_mask);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_correct_geometry failed: %s",
                 cpl_error_get_message_default(my_error));
    
    cpl_imagelist_delete(merged_pixels_mask);
    
    /* Checking for saturation, creating saturation mask */
    /* Checking the saturation outside hot and bad pixels */
    double cosmics_part = inst_config->image_cosmics_part;
    cpl_imagelist *geometry_corrected_saturation_mask = NULL;
    double *max_flux = (double *)cpl_calloc(cpl_imagelist_get_size(real_imagelist),
                                            sizeof(double));
    int print_max_flux = 1;
    if (strcmp(fibre_b, "THAR") == 0) {
        cosmics_part = inst_config->image_thar_part;
        print_max_flux = 0;
    }
    
    // For NIR we have to multiply the input images by Texp, to get ADUs instead of ADUs/s
    if (inst_config->inst_type == NIR) {
        double texp = cpl_propertylist_get_double(keywords, inst_config->Texp_kw);
        //espdr_msg("--->>>>>  exposure time: %f", texp);
        my_error = cpl_imagelist_multiply_scalar(real_imagelist, texp);
    }
    my_error = espdr_check_saturation(real_imagelist,
                                      CCD_geom, inst_config,
                                      pixels_mask, cosmics_part,
                                      max_flux, print_max_flux,
                                      &geometry_corrected_saturation_mask);
    cpl_imagelist_delete(real_imagelist);
    
    /* Add QC KWs to the header */
    if (strcmp(fibre_b, "THAR") != 0) {
        char *check_kw = NULL;
        if (strcmp(RECIPE_ID, "espdr_cal_contam") == 0) {
            check_kw = qc_kws->qc_cal_contam_check_kw;
        } else {
            check_kw = qc_kws->qc_cal_eff_ab_check_kw;
        }
        my_error = espdr_max_flux_QC(inst_config, CCD_geom, qc_kws,
                                     max_flux, check_kw, keywords);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_max_flux_QC failed: %s",
                     cpl_error_get_message_default(my_error));
    }
    
    /* keywords contains all the keywords read from
     the first input frame header */
    cpl_propertylist *keywords_HP = NULL;
    cpl_propertylist *keywords_BP = NULL;
    
    /* Load primary header keywords from the hot pixels image */
    const char *input_filename_HP = NULL;
    if (master_dark_frame != NULL) {
        input_filename_HP = cpl_frame_get_filename(master_dark_frame);
        espdr_msg("KEYWORDS for MASTER DARK input filename: %s", input_filename_HP);
    } else {
        input_filename_HP = cpl_frame_get_filename(hot_pixels_frame);
        espdr_msg("KEYWORDS for HOT PIXELS input filename: %s", input_filename_HP);
    }
    keywords_HP = cpl_propertylist_load(input_filename_HP, 0);
    
    /* Load primary header keywords from the bad pixels image */
    const char *input_filename_BP = cpl_frame_get_filename(bad_pixels_frame);
    espdr_msg("KEYWORDS for BAD PIXELS input filename: %s", input_filename_BP);
    keywords_BP = cpl_propertylist_load(input_filename_BP, 0);
    
    double *dark = (double *)cpl_calloc(CCD_geom->total_output_nb, sizeof(double));
    double *dark_ron = (double *)cpl_calloc(CCD_geom->total_output_nb, sizeof(double));
    double *conad = (double *)cpl_calloc(CCD_geom->total_output_nb, sizeof(double));
    cpl_imagelist *CCD_corrected_raw = cpl_imagelist_new();
    
    // Detector signature removal for the contam frame process ...
    my_error = espdr_remove_det_signature(raw_imagelist,
                                          orders_mask_list,
                                          master_bias_list,
                                          master_dark_list,
                                          keywords, keywords_HP,
                                          keywords_BP, qc_kws,
                                          inst_config, remove_bias_res_flag,
                                          CCD_geom, CPL_TYPE_DOUBLE,
                                          ron, dark, dark_ron, conad,
                                          CCD_corrected_raw);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_remove_detector_signature failed: %s",
                 cpl_error_get_message_default(my_error));
    cpl_imagelist_delete(raw_imagelist);
    
    /* Background calculation process ... */
    cpl_imagelist *raw_bkgr_measured = NULL;
    if (strcmp(background_sw, "on") == 0) {
        *raw_bkgr_subs = cpl_imagelist_new();
        raw_bkgr_measured = cpl_imagelist_new();
        espdr_msg("----> Backgroud measurement process ...");
        int bkgr_grid_size_x = inst_config->contam_bkgr_grid_size_x;
        int bkgr_grid_size_y = inst_config->contam_bkgr_grid_size_y;
        if (strcmp(RECIPE_ID, "espdr_cal_eff_ab") == 0) {
            bkgr_grid_size_x = inst_config->eff_ab_bkgr_grid_size_x;
            bkgr_grid_size_y = inst_config->eff_ab_bkgr_grid_size_y;
        }
        my_error = espdr_measure_background_science_by_ext(CCD_corrected_raw,
                                                           order_profile_fibre_imagelist[1], /* fibre B */
                                                           conad, ron,
                                                           bkgr_grid_size_x,
                                                           bkgr_grid_size_y,
                                                           CCD_geom,
                                                           raw_bkgr_measured,
                                                           *raw_bkgr_subs);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_measure_inter_order_background failed");
        espdr_msg("----> Backgroud measurement process finished !");
        if (raw_bkgr_measured == NULL) {
            espdr_msg("raw bkgr measured is NULL");
        } else {
            espdr_msg("raw bkgr measured is not NULL");
        }
    } else {
        *raw_bkgr_subs = cpl_imagelist_duplicate(CCD_corrected_raw);
    }
    
    //espdr_msg("Saving BKGR KWs");
    char *new_keyword = NULL;
    double bkgr_mean = -9999.9, bkgr_max = -9999.9, bkgr_min = -9999.9;
    for (int i = 0; i < CCD_geom->ext_nb; i++) {
        if (strcmp(background_sw, "on") == 0) {
            bkgr_mean = cpl_image_get_mean(cpl_imagelist_get(raw_bkgr_measured,i));
            bkgr_min = cpl_image_get_min(cpl_imagelist_get(raw_bkgr_measured,i));
            bkgr_max = cpl_image_get_max(cpl_imagelist_get(raw_bkgr_measured,i));
        }
        
        new_keyword = espdr_add_output_index_to_keyword(qc_kws->qc_cal_eff_ab_bkgr_mean_kw_first,
                                                        qc_kws->qc_cal_eff_ab_bkgr_mean_kw_last, i);
        my_error = espdr_keyword_add_double(new_keyword, bkgr_mean, "Bkgr mean [e-] in the extension",
                                            &keywords);
        cpl_free(new_keyword);
        
        new_keyword = espdr_add_output_index_to_keyword(qc_kws->qc_cal_eff_ab_bkgr_min_kw_first,
                                                        qc_kws->qc_cal_eff_ab_bkgr_min_kw_last, i);
        my_error = espdr_keyword_add_double(new_keyword, bkgr_min, "Bkgr min [e-] in the extension",
                                            &keywords);
        cpl_free(new_keyword);
        
        new_keyword = espdr_add_output_index_to_keyword(qc_kws->qc_cal_eff_ab_bkgr_max_kw_first,
                                                        qc_kws->qc_cal_eff_ab_bkgr_max_kw_last, i);
        my_error = espdr_keyword_add_double(new_keyword, bkgr_max, "Bkgr max [e-] in the extension",
                                            &keywords);
        cpl_free(new_keyword);
    }
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Add of one of BKGR keywords to the propertylist failed: %s",
                 cpl_error_get_message_default(my_error));

    double exp_time_hour = cpl_propertylist_get_double(keywords,inst_config->Texp_kw) / 3600.0;
    double exp_time_hour_mdark = cpl_propertylist_get_double(keywords_HP, inst_config->Texp_kw) / 3600.0;
    
    cpl_image **s2d_flux = (cpl_image **)cpl_malloc(inst_config->fibres_nb * sizeof(cpl_image *));
    cpl_image **s2d_error = (cpl_image **)cpl_malloc(inst_config->fibres_nb * sizeof(cpl_image *));
    cpl_image **s2d_quality = (cpl_image **)cpl_malloc(inst_config->fibres_nb * sizeof(cpl_image *));
    
    for (int t = 0; t < inst_config->fibres_nb; t++) {
        /* extraction process */
        espdr_msg("----> Horne extraction process fibre %c ...",fibre_name[t]);
        cosmics_nb[t] = (int *) cpl_calloc(orders_nb_per_fibre[t], sizeof(int));
        cpl_imagelist *input_iml = NULL;
        if (strcmp(background_sw, "off") == 0) {
            input_iml = CCD_corrected_raw;
        } else {
            input_iml = *raw_bkgr_subs;
        }
        
        double tolerance_rejection = inst_config->tolerance_rejection;
        int extraction_window_size = inst_config->extraction_window_size[t];
        double extraction_ksigma[2];
        extraction_ksigma[0] = inst_config->extraction_ksigma_dark;
        if (strcmp(fibre_b, "FP") == 0) {
            extraction_ksigma[1] = inst_config->extraction_ksigma_fp;
        } else {
            if (strcmp(fibre_b, "THAR") == 0) {
                extraction_ksigma[1] = inst_config->extraction_ksigma_thar;
            } else {
                if (strcmp(fibre_b, "LFC") == 0) {
                    extraction_ksigma[1] = inst_config->extraction_ksigma_lfc;
                } else {
                    extraction_ksigma[1] = inst_config->extraction_ksigma_dark;
                }
            }
        }
        if (strcmp(RECIPE_ID, "espdr_cal_eff_ab") == 0) {
            extraction_ksigma[0] = inst_config->extraction_ksigma_sky;
            extraction_ksigma[1] = inst_config->extraction_ksigma_sky;
        }
        
        if(strcmp(inst_config->extraction_method, "horne") == 0) {
            my_error = espdr_horne_extraction_one_fibre(input_iml,
                                                        geometry_corrected_pixels_mask,
                                                        geometry_corrected_saturation_mask,
                                                        NULL, // no CRH mask
                                                        raw_bkgr_measured,
                                                        order_profile_fibre_imagelist[t], NULL,
                                                        master_dark_list, master_dark_pxl_nb_list,
                                                        orders_coeffs[t], t,
                                                        inst_config, CCD_geom,
                                                        ron, dark, dark_ron, conad,
                                                        exp_time_hour, exp_time_hour_mdark,
                                                        extraction_window_size,
                                                        tolerance_rejection,
                                                        extraction_ksigma[t],
                                                        cosmics_nb[t],
                                                        &s2d_flux[t],
                                                        &s2d_error[t],
                                                        &s2d_quality[t]);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "espdr_horne_extraction failed");
            espdr_msg("----> Horne extraction process fibre %c finished !", fibre_name[t]);
            
        } else if (strcmp(inst_config->extraction_method, "simple") == 0) {
            espdr_msg(ANSI_COLOR_RED"Warning : Simple extraction method ..."ANSI_COLOR_RESET);
            my_error = espdr_simple_extraction_one_fibre(input_iml,
                                                         geometry_corrected_pixels_mask,
                                                         geometry_corrected_saturation_mask,
                                                         NULL, // no CRH mask
                                                         raw_bkgr_measured,
                                                         master_dark_list, master_dark_pxl_nb_list,
                                                         orders_coeffs[t], t,
                                                         inst_config, CCD_geom,
                                                         ron, dark, dark_ron, conad,
                                                         exp_time_hour, exp_time_hour_mdark,
                                                         extraction_window_size,
                                                         &s2d_flux[t],
                                                         &s2d_error[t],
                                                         &s2d_quality[t]);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "espdr_simple_extraction failed: %s",
                         cpl_error_get_message_default(my_error));
            espdr_msg("Simple extraction done");
        } else {
            espdr_msg("Error: %s is an unknown extraction method. Allowed only 'horne' and 'simple'",
                      inst_config->extraction_method);
            return (CPL_ERROR_ILLEGAL_INPUT);
        }
        
        /* Flat correction */
        espdr_msg("----> Flat correction process fibre %c ...", fibre_name[t]);
        my_error = espdr_correct_flat(s2d_flux[t],
                                      s2d_error[t],
                                      s2d_quality[t],
                                      flat_table[t],
                                      flat_error_table[t],
                                      flat_quality_table[t],
                                      &flat_corr_flux_RE[t],
                                      &flat_corr_err_RE[t],
                                      &flat_corr_qual_RE[t]);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_correct_flat failed for fiber %c: %s",
                     fibre_name[t], cpl_error_get_message_default(my_error));
        espdr_msg("---->Flat correction process fibre %c finished !", fibre_name[t]);
        
        if (strcmp(RECIPE_ID, "espdr_cal_eff_ab") == 0) {
            espdr_msg("Compute SNR for fiber %c on extracted spectrum", fibre_name[t]);
            my_error = espdr_compute_snr(s2d_flux[t], s2d_error[t],
                                         inst_config->snr_averaging_window,
                                         snr_per_fibre[t]);
        }
        
    } // end loop fibre.
    
    /* Cleaning some memory */
    cpl_imagelist_delete(CCD_corrected_raw);
    cpl_imagelist_delete(geometry_corrected_saturation_mask);
    cpl_imagelist_delete(pixels_mask);
    cpl_imagelist_delete(geometry_corrected_pixels_mask);
    
    cpl_propertylist_delete(keywords_HP);
    cpl_propertylist_delete(keywords_BP);
    
    cpl_free(conad);
    cpl_free(dark);
    cpl_free(max_flux);
    
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        for (int j = 0; j < CCD_geom->ext_nb; j++) {
            cpl_table_delete(orders_coeffs[i][j]);
        }
        cpl_free(orders_coeffs[i]);
        cpl_image_delete(s2d_flux[i]);
        cpl_image_delete(s2d_error[i]);
        cpl_image_delete(s2d_quality[i]);
    }
    cpl_free(orders_coeffs);
    cpl_free(s2d_flux);
    cpl_free(s2d_error);
    cpl_free(s2d_quality);
    
    if ((inst_config->raw_bias_limit_nb > 0) && (remove_bias_res_flag == 1)) {
        cpl_imagelist_delete(master_bias_list);
    }
    cpl_imagelist_delete(hot_pixels_list);
    if (master_dark_frame != NULL) {
        cpl_imagelist_delete(master_dark_list);
        cpl_imagelist_delete(master_dark_pxl_nb_list);
    }
    cpl_imagelist_delete(bad_pixels_list);
    if (orders_mask_frame != NULL) {
        cpl_imagelist_delete(orders_mask_list);
    }
    cpl_imagelist_delete(raw_bkgr_measured);
    
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        cpl_imagelist_delete(order_profile_fibre_imagelist[i]);
        cpl_image_delete(flat_table[i]);
        cpl_image_delete(flat_error_table[i]);
        cpl_image_delete(flat_quality_table[i]);
    }
    cpl_free(order_profile_fibre_imagelist);
    cpl_free(flat_table);
    cpl_free(flat_error_table);
    cpl_free(flat_quality_table);
    
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief    Perform CONTAM QC
 @param     inst_config instrument config
 @param     qc_kw       QC KWs names
 @param     contam      resulting image (?)
 @param     keywords_RE completed KWs list
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_cal_contam_QC(espdr_inst_config *inst_config,
                                   espdr_CCD_geometry *CCD_geom,
                                   espdr_qc_keywords *qc_kws,
                                   char *src,
                                   double *RON,
                                   cpl_image *contam_a,
                                   cpl_image *contam_qual_a,
                                   cpl_image *contam_b,
                                   cpl_image *contam_qual_b,
                                   cpl_propertylist **keywords_RE) {
    
    cpl_error_code my_error = CPL_ERROR_NONE;
    
    int i, j, index;
    char *new_keyword;
    int flux_QC = 1;
    int saturation_QC = 1;
    int global_QC = 1;
    
    double flux;
    cpl_image *contam_order;
    cpl_image *qual_order;
    double *contam_order_data = NULL;
    int *qual_order_data = NULL;
    
    int orders_nr = cpl_image_get_size_y(contam_a);
    int contam_size_x = cpl_image_get_size_x(contam_a);
    
    if (cpl_propertylist_has(*keywords_RE, qc_kws->qc_saturation_check_kw)) {
        saturation_QC = cpl_propertylist_get_int(*keywords_RE, qc_kws->qc_saturation_check_kw);
        if ((saturation_QC == 0) && (strcmp(src, "THAR") != 0)) {
            global_QC = 0;
        }
    }
    
    for (i = 0; i < orders_nr; i++) {
        
        contam_order = cpl_image_extract(contam_a, 1, i+1,
                                         cpl_image_get_size_x(contam_a), i+1);
        
        qual_order = cpl_image_extract(contam_qual_a, 1, i+1,
                                       cpl_image_get_size_x(contam_a), i+1);
        
        new_keyword = espdr_add_output_index_to_keyword(
                                    qc_kws->qc_cal_contam_max_flux_kw_first,
                                    qc_kws->qc_cal_contam_max_flux_kw_last,
                                    i+1);
        
        contam_order_data = cpl_image_get_data_double(contam_order);
        qual_order_data = cpl_image_get_data_int(qual_order);
        
        flux = 0.0;
        for (j = 0; j < contam_size_x; j++) {
            if (qual_order_data[j] == 0) {
                if (contam_order_data[j] > flux) {
                    flux = contam_order_data[j];
                }
            }
        }
        
        my_error = espdr_keyword_add_double(new_keyword,
                                            flux,
                                            "S2D flux max, fibre A",
                                            keywords_RE);
        
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "Add keyword QC_ORDER_MAX_FLUX to the propertylist failed: %s",
                     cpl_error_get_message_default(my_error));
        
        cpl_free(new_keyword);
        
        cpl_image_delete(contam_order);
        cpl_image_delete(qual_order);
        
    }
    
    espdr_msg("Flux check done");
    
    contam_order_data = cpl_image_get_data_double(contam_b);
    qual_order_data = cpl_image_get_data_int(contam_qual_b);
    int size_x = cpl_image_get_size_x(contam_b);
    int size_y = cpl_image_get_size_y(contam_b);
    
    double contam_b_mean = 0.0;
    int valid_pxl_nb = 0;
    
    espdr_msg("Computing mean on %d x %d pixels",
              size_x, size_y);
    
    for (int i = 0; i < size_x*size_y; i++) {
        if (qual_order_data[i] == 0) {
            contam_b_mean = contam_b_mean + contam_order_data[i];
            valid_pxl_nb++;
        }
    }
    
    espdr_msg("Valid pixels: %d", valid_pxl_nb);
    
    contam_b_mean = contam_b_mean / valid_pxl_nb;
    
    espdr_msg("Mean on B computed");
    
    double RON_max = 0.0;
    for (int i = 0; i < CCD_geom->total_output_nb; i++) {
        espdr_msg("RON[%d] = %f", i, RON[i]);
        if (RON[i] > RON_max) {
            RON_max = RON[i];
        }
    }
    espdr_msg("RON_max = %f", RON_max);
    
    if (contam_b_mean < 5.0 * RON_max) {
        flux_QC = 0;
        espdr_msg_warning(ANSI_COLOR_RED
                          "Flux level on simultaneous reference is abnormally low"
                          ANSI_COLOR_RESET);
    }
    
    my_error = espdr_keyword_add_int(qc_kws->qc_cal_contam_min_flux_check_kw,
                                     flux_QC,
                                     "QC on minimum flux in the S2D B",
                                     keywords_RE);
    
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Add keyword QC_CONTAM_FLUX_CHECK to the propertylist failed: %s",
                 cpl_error_get_message_default(my_error));
    
    if (flux_QC == 0) {
        global_QC = 0;
    }
    
    my_error = espdr_keyword_add_int(qc_kws->qc_cal_contam_check_kw,
                                     global_QC,
                                     "CONTAM global QC",
                                     keywords_RE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Add keyword QC_CONTAM_CHECK to the propertylist failed: %s",
                 cpl_error_get_message_default(my_error));
    
    return cpl_error_get_code();
}


