/*                                                                            *
 *   This file is part of the ESPRESSO Pipeline                               *
 *   Copyright (C) 2006 European Southern Observatory                         *
 *                                                                            *
 *   This library is free software; you can redistribute it and/or modify     *
 *   it under the terms of the GNU General Public License as published by     *
 *   the Free Software Foundation; either version 2 of the License, or        *
 *   (at your option) any later version.                                      *
 *                                                                            *
 *   This program is distributed in the hope that it will be useful,          *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 *   GNU General Public License for more details.                             *
 *                                                                            *
 *   You should have received a copy of the GNU General Public License        *
 *   along with this program; if not, write to the Free Software              *
 *   Foundation, 51 Franklin St, Fifth Floor, Boston, MA  02111-1307  USA     *
 *                                                                            */

/*
 * $Author: dsosnows $
 * $Date: 2014-10-28 15:13:41 $
 * $Revision: 1.3 $
 * $Name: not supported by cvs2svn $
 */


#include <espdr_fit.h>
#include <gsl/gsl_version.h>
/*----------------------------------------------------------------------------
 Functions code
 ----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 @brief set the gls vector for the n gaussian fit
 @param         v       initial guess of the fit
 @param         data    data to put into gsl vector
 @param[out]    f       gsl vector set
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

static int n_gauss_f (const gsl_vector *v, void *data, gsl_vector *f) {
    
    double *x = ((espdr_ngauss_data *) data)->x;
    double *y = ((espdr_ngauss_data *) data)->y;
    double *err = ((espdr_ngauss_data *) data)->err;
    double *fit = ((espdr_ngauss_data *) data)->fit;
    int n = ((espdr_ngauss_data *) data)->n;
    int m = ((espdr_ngauss_data *) data)->m;
    int i, j;
    
    double c = gsl_vector_get(v, 0);
    double *k = (double *)cpl_calloc(m, sizeof(double));
    double *x0 = (double *)cpl_calloc(m, sizeof(double));
    double *fwhm = (double *)cpl_calloc(m, sizeof(double));
    double *sigma = (double *)cpl_calloc(m, sizeof(double));
    
    for (j = 0; j < m; j++) {
        
        k[j] = gsl_vector_get(v, j+1);
        x0[j] = gsl_vector_get(v, m+j+1);
        fwhm[j] = gsl_vector_get(v, 2*m+j+1);
        sigma[j] = fwhm[j]/2/sqrt(2*log(2));
    }
    
    for (i = 0; i < n; i++) {
        
        fit[i] = c;
        
        for (j = 0; j < m; j++)
            fit[i] += k[j]*exp(-(x[i]-x0[j])*(x[i]-x0[j])/2/sigma[j]/sigma[j]);
        
        gsl_vector_set (f, i, (fit[i]-y[i])/err[i]);
    }
    
    cpl_free(k);
    cpl_free(x0);
    cpl_free(fwhm);
    cpl_free(sigma);
    
    return GSL_SUCCESS;
}



/*----------------------------------------------------------------------------*/
/**
 @brief     set the gsl matrix for the n gaussian fit
 @param         v       initial guess of the fit
 @param         data    data to be put into the gsl matrix
 @param[out]    J       gsl matrix set
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

static int n_gauss_df (const gsl_vector *v, void *data, gsl_matrix *J) {
    
    double *x = ((espdr_ngauss_data *) data)->x;
    //double *y = ((espdr_ngauss_data *) data)->y;
    double *err = ((espdr_ngauss_data *) data)->err;
    //double *fit = ((espdr_ngauss_data *) data)->fit;
    int n = ((espdr_ngauss_data *) data)->n;
    int m = ((espdr_ngauss_data *) data)->m;
    int i, j;
    
    //double c = gsl_vector_get(v, 0);
    double *k = (double *)cpl_calloc(m, sizeof(double));
    double *x0 = (double *)cpl_calloc(m, sizeof(double));
    double *fwhm = (double *)cpl_calloc(m, sizeof(double));
    double *sigma = (double *)cpl_calloc(m, sizeof(double));
    double *e = (double *)cpl_calloc(m, sizeof(double));
    
    for (j = 0; j < m; j++) {

        k[j] = gsl_vector_get(v, j+1);
        x0[j] = gsl_vector_get(v, m+j+1);
        fwhm[j] = gsl_vector_get(v, 2*m+j+1);
        sigma[j] = fwhm[j]/2/sqrt(2*log(2));
    }
    
    for (i = 0; i < n; i++) {
        
        gsl_matrix_set (J, i, 0, 1/err[i]);
        
        for (j = 0; j < m; j++) {
            
            e[j] = exp(-(x[i]-x0[j])*(x[i]-x0[j])/2/sigma[j]/sigma[j]);
            gsl_matrix_set (J, i, j+1, e[j]/err[i]);
            gsl_matrix_set (J, i, m+j+1,
                            k[j]/err[i]*e[j]*(x[i]-x0[j])/sigma[j]/sigma[j]);
            gsl_matrix_set (J, i, 2*m+j+1,
                            k[j]/err[i]*e[j]*(x[i]-x0[j])*(x[i]-x0[j])/sigma[j]/sigma[j]/sigma[j]/2/sqrt(2*log(2)));
        }
    }
    
    cpl_free(k);
    cpl_free(x0);
    cpl_free(fwhm);
    cpl_free(sigma);
    cpl_free(e);
    
    return GSL_SUCCESS;
}



/*----------------------------------------------------------------------------*/
/**
 @brief set both the gsl vector and the gsl matrix for the n gaussian fit
 @param             v       initial guess of teh fit
 @param             data    data to be put into the gsl vector and matrix
 @param[out]        f       gsl vector set
 @param[out]        J       gsl matric set
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

static int n_gauss_fdf (const gsl_vector *v, void *data,
                        gsl_vector *f, gsl_matrix *J) {
    
    n_gauss_f (v, data, f);
    n_gauss_df (v, data, J);
    
    return GSL_SUCCESS;
}


/*----------------------------------------------------------------------------*/
/**
 @brief Fit the n-gaussian on n slices order (wrap for gsl functions)
 @param         data                data to fit
 @param         inter_slice_dist    distance netween slices to find peaks
 @param[out]    x0_RE               gaussian peaks returned
 @param[out]    sig_x0_RE           error on gaussian peaks returned
 @param[out]    flux_RE             flux returned
 @param[out]    sig_flux_RE         error on flux returned
 @param[out]    fwhm_RE             FWHM returned
 @param[out]    sig_fwhm_RE         error on FWHM returned
 @param         spectrum_type       1 - emisison, 0 - absorption
 @param         print_flag          1 - print, 0 - don't
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_fit_Ngauss_equi_dist(espdr_ngauss_data *data,
                                          int inter_slice_dist,
                                          int slice_width,
                                          double *x0_RE,
                                          double *sig_x0_RE,
                                          double *flux_RE,
                                          double *sig_flux_RE,
                                          double *fwhm_RE,
                                          double *sig_fwhm_RE,
                                          int spectrum_type,
                                          int print_flag) {
    
    
    const gsl_multifit_fdfsolver_type *T;
    gsl_multifit_fdfsolver *s;
    gsl_multifit_function_fdf f;
    int status;
    int i, j, iter = 0;
    //const size_t n = N;
    //const size_t p = GAUSS_NPAR;
    int npar = GAUSS_NPAR;
    int vector_size = data->m * (npar-1) + 1;
    int n = data->n;
    int m = data->m;
    double c, *k, *x0, *fwhm;
    double *sig_k, *sig_x0, *sig_fwhm;
    int *ind_n;
    double data_max = 0.0;
    int index_max = 0;
    int vector_index = 0;
    double sig_swap;

     if(print_flag == 0) {
	    /* ASE : Do nothing Just to document this parameter is not used*/
     }
    
    gsl_vector *v = gsl_vector_alloc(vector_size);
    gsl_matrix *covar = gsl_matrix_alloc(vector_size, vector_size);
#if defined GSL_MAJOR_VERSION && GSL_MAJOR_VERSION >= 2
    gsl_matrix *J;
#endif
    /* Set the data properly */
    
    double *data_x = (double *)cpl_calloc(n, sizeof(double));
    //double data_x0 = data->x[0];
    for (i = 0; i < n; i++) {
        data_x[i] = data->x[i];
        //data->x[i] = data->x[i] - data_x0;
    }
    
    k = (double *)cpl_calloc(m, sizeof(double));
    ind_n = (int *)cpl_calloc(m, sizeof(int));
    fwhm = (double *)cpl_calloc(m, sizeof(double));
    x0 = (double *)cpl_calloc(m, sizeof(double));
    sig_k = (double *)cpl_calloc(m, sizeof(double));
    sig_x0 = (double *)cpl_calloc(m, sizeof(double));
    sig_fwhm = (double *)cpl_calloc(m, sizeof(double));
   
    /* Set the fit parameters properly */
    c = data->y[0];
    for (i = 0; i < n; i++) {
        if (spectrum_type == 1) { // emission spectrum
            if (data->y[i] < c) {
                c = data->y[i];
            }
        } else {    // absorption spectrum
            if (data->y[i] > c) {
                c = data->y[i];
            }
        }
    }
    gsl_vector_set(v, 0, c);
    
    if (m == 1) {
        x0[0] = (data->x[n-1] + data->x[0])/2.0;
    } else {
        j = 1;
        vector_index = 0;
        for (i = 0; i < m; i++) {
            data_max = 0.0;
            index_max = 0;
            j = vector_index;
            while ((j < vector_index + slice_width) && (j < n)) {
                if (data->y[j] > data_max) {
                    data_max = data->y[j];
                    index_max = j;
                }
                j++;
            }
            x0[i] = data->x[index_max];
            vector_index = index_max + inter_slice_dist - (int)slice_width/2;
        }
    }
    
    for (j = 0; j < data->m; j++) {
        fwhm[j] = (data->x[n-1]-data->x[0])/3.0/data->m;
        //x0[j] = data->x[n*(2*j+1)/2*m];
        ind_n[j] = (x0[j] - data_x[0])/(data_x[n-1] - data_x[0])*n;
        k[j] = data->y[ind_n[j]] - c;
        gsl_vector_set(v, j+1, k[j]);
        //gsl_vector_set(v, m+j+1, x0[j] - data_x[0]);
        gsl_vector_set(v, m+j+1, x0[j]);
        gsl_vector_set(v, 2*m+j+1, fwhm[j]);
    }
    
    f.f = &n_gauss_f;
    f.df = &n_gauss_df;
    f.fdf = &n_gauss_fdf;
    f.n = n;
    f.p = vector_size;
    f.params = data;
    
    T = gsl_multifit_fdfsolver_lmsder;
    s = gsl_multifit_fdfsolver_alloc(T, n, m*(npar-1)+1);
    gsl_multifit_fdfsolver_set(s, &f, v);
    
    //print_state (iter, s);
    
    do
    {
        iter++;
        status = gsl_multifit_fdfsolver_iterate(s);
        
        //printf ("status = %s\n", gsl_strerror (status));
        
        //print_state (iter, s);
        
        if (status)
            break;
        
        status = gsl_multifit_test_delta(s->dx, s->x, 0.0, 1e-5);
    }
    while (status == GSL_CONTINUE && iter < 1000);
    
#if defined GSL_MAJOR_VERSION && GSL_MAJOR_VERSION >= 2
    J = gsl_matrix_alloc(n, vector_size);
    gsl_multifit_fdfsolver_jac(s, J);
    gsl_multifit_covar (J, 0.0, covar);
    gsl_matrix_free (J);
#else
    gsl_multifit_covar (s->J, 0.0, covar);
#endif
    c = gsl_vector_get(s->x, 0);
    for (j = 0; j < m; j++) {
        k[j] = gsl_vector_get(s->x, j+1);
        sig_k[j] = sqrt(gsl_matrix_get(covar,j+1,j+1));
        x0[j] = gsl_vector_get(s->x, m+j+1);
        sig_x0[j] = sqrt(gsl_matrix_get(covar, m+j+1, m+j+1));
        fwhm[j] = gsl_vector_get(s->x, 2*m+j+1);
        sig_fwhm[j] = sqrt(gsl_matrix_get(covar, 2*m+j+1, 2*m+j+1));
    }
    
    /* Compute the x0 */
    *x0_RE = 0.0;
    if (print_flag) {
        espdr_msg("GAUSS FIT X0::::::::::::::::::");
    }
    for (j = 0; j < m; j++) {
        *x0_RE = *x0_RE + x0[j];
        if (print_flag) {
            espdr_msg("x0[%d] = %lf", j, x0[j]);
        }
    }
    *x0_RE = (*x0_RE / m);
    
    *sig_x0_RE = 0.0;
    
    /* m is the number of gaussians, so it is always > 0 */
    if (m == 1) {
        *sig_x0_RE = sig_x0[0];
    } else {
        sig_swap = 0.0;
        for (i = 0; i < m - 1;  i++) {
            for (j = 0; j < m - i - 1; j++) {
                if (sig_x0[j] > sig_x0[j+1]) {
                    sig_swap = sig_x0[j];
                    sig_x0[j] = sig_x0[j+1];
                    sig_x0[j+1] = sig_swap;
                }
            }
        }
        
        if ((m % 2) == 0) {
            *sig_x0_RE = sig_x0[m/2];
        } else {
            *sig_x0_RE = (sig_x0[(m-1)/2] + sig_x0[(m+1)/2]) / 2.0;
        }
    }
    
    
    /* Compute the flux */
    *flux_RE = 0.0;
    for (j = 0; j < m; j++) {
        *flux_RE = *flux_RE + k[j];
    }
    *flux_RE = (*flux_RE / m);
    
    *sig_flux_RE = 0.0;
    
    /* m is the number of gaussians, so it is always > 0 */
    if (m == 1) {
        *sig_flux_RE = sig_k[0];
    } else {
        sig_swap = 0.0;
        for (i = 0; i < m - 1;  i++) {
            for (j = 0; j < m - i - 1; j++) {
                if (sig_k[j] > sig_k[j+1]) {
                    sig_swap = sig_k[j];
                    sig_k[j] = sig_k[j+1];
                    sig_k[j+1] = sig_swap;
                }
            }
        }
        
        if ((m % 2) == 0) {
            *sig_flux_RE = sig_k[m/2];
        } else {
            *sig_flux_RE = (sig_k[(m-1)/2] + sig_k[(m+1)/2]) / 2.0;
        }
    }
    
    
    /* Compute the fwhm */
    *fwhm_RE = 0.0;
    for (j = 0; j < m; j++) {
        *fwhm_RE = *fwhm_RE + fwhm[j];
    }
    *fwhm_RE = (*fwhm_RE / m);
    
    *sig_fwhm_RE = 0.0;
    
    /* m is the number of gaussians, so it is always > 0 */
    if (m == 1) {
        *sig_fwhm_RE = sig_fwhm[0];
    } else {
        sig_swap = 0.0;
        for (i = 0; i < m - 1;  i++) {
            for (j = 0; j < m - i - 1; j++) {
                if (sig_fwhm[j] > sig_fwhm[j+1]) {
                    sig_swap = sig_fwhm[j];
                    sig_fwhm[j] = sig_fwhm[j+1];
                    sig_fwhm[j+1] = sig_swap;
                }
            }
        }
        
        if ((m % 2) == 0) {
            *sig_fwhm_RE = sig_fwhm[m/2];
        } else {
            *sig_fwhm_RE = (sig_fwhm[(m-1)/2] + sig_fwhm[(m+1)/2]) / 2.0;
        }
    }
    
    
    
    gsl_multifit_fdfsolver_free(s);
    gsl_matrix_free(covar);
    gsl_vector_free(v);
    
    cpl_free(data_x);
    cpl_free(k);
    cpl_free(ind_n);
    cpl_free(fwhm);
    cpl_free(x0);
    cpl_free(sig_k);
    cpl_free(sig_x0);
    cpl_free(sig_fwhm);
 
    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 @brief Fit n-gauss on the data (wrap for the gsl functions)
 @param         data            data to be fit
 @param         x0              first guess = lines peaks
 @param[out]    res             fit result
 @param         spectrum_type   1 - emisison, 0 - absorption
 @param         print_flag      1 - print, 0 - don't
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_fit_Ngauss (espdr_ngauss_data *data,
                                 double *x0init,
                                 double fwhm_init_factor,
                                 espdr_ngauss_result *res,
                                 int spectrum_type,
                                 int print_flag) {
    
    
    const gsl_multifit_fdfsolver_type *T;
    gsl_multifit_fdfsolver *s;
    gsl_multifit_function_fdf f;
    int status;
    int i, j, iter = 0;
    int npar = GAUSS_NPAR;
    int vector_size = data->m * (npar-1) + 1;
    int n = data->n;
    int m = data->m;
    double c, *x0, *k, *fwhm;
    double sig_c, *sig_x0, *sig_k, *sig_fwhm;
    int *ind;
#if defined GSL_MAJOR_VERSION && GSL_MAJOR_VERSION >= 2
    gsl_matrix *J;
#endif
    
    gsl_vector *v = gsl_vector_alloc(vector_size);
    gsl_matrix *covar = gsl_matrix_alloc(vector_size, vector_size);
        
    x0 = (double *)cpl_calloc(m, sizeof(double));
    k = (double *)cpl_calloc(m, sizeof(double));
    ind = (int *)cpl_calloc(m, sizeof(int));
    fwhm = (double *)cpl_calloc(m, sizeof(double));
    sig_x0 = (double *)cpl_calloc(m, sizeof(double));
    sig_k = (double *)cpl_calloc(m, sizeof(double));
    sig_fwhm = (double *)cpl_calloc(m, sizeof(double));
    
    /* NOTE: The data->x and x0init vectors are used as provided, without subtracting the mean or first value */
    /* If needed (working with large x values), this should be taken care of outside of this function */
    
    /* Initialize the fit parameters properly */
    
    c = data->y[0];
    for (i = 0; i < n; i++) {
        if (spectrum_type == 1) { // emission spectrum
            if (data->y[i] < c) {
                c = data->y[i];
            }
        } else {    // absorption spectrum
            if (data->y[i] > c) {
                c = data->y[i];
            }
        }
    }
    gsl_vector_set(v, 0, c);
    
    for (j = 0; j < data->m; j++) {
        //fwhm[j] = (data->x[n-1]-data->x[0])/3.0/data->m;
        fwhm[j] = (data->x[n-1]-data->x[0])/fwhm_init_factor/data->m;
        x0[j] = x0init[j];
        ind[j] = (x0[j] - data->x[0])/(data->x[n-1] - data->x[0])*n;
        k[j] = data->y[ind[j]] - c;
        //espdr_msg("Initial conditions for CCF fit (min/max RV = %.2f %.2f) fwhm=%.2f,x0=%.2f,index0=%i,contrast=%.2f (fwhm_factor=%.2f, m=%i)", data->x[0],data->x[n-1],fwhm[j],x0[j],ind[j],k[j],fwhm_init_factor,data->m);
        gsl_vector_set(v, j+1, k[j]);
        gsl_vector_set(v, m+j+1, x0[j]);
        gsl_vector_set(v, 2*m+j+1, fwhm[j]);
    }
    
    if (print_flag) {
        for (i = 0; i < data->m; i++) {
            espdr_msg("NGAUSS: x0[%d] = %lf", i, x0[i]);
        }
    }
    
    f.f = &n_gauss_f;
    f.df = &n_gauss_df;
    f.fdf = &n_gauss_fdf;
    f.n = n;
    f.p = vector_size;
    f.params = data;
    
    T = gsl_multifit_fdfsolver_lmsder;
    s = gsl_multifit_fdfsolver_alloc(T, n, m*(npar-1)+1);
    gsl_multifit_fdfsolver_set(s, &f, v);
    
    //print_state (iter, s);
    
    do
    {
        iter++;
        status = gsl_multifit_fdfsolver_iterate(s);
        
        //printf ("status = %s\n", gsl_strerror (status));
        
        //print_state (iter, s);
        
        if (status)
            break;
        
        status = gsl_multifit_test_delta(s->dx, s->x, 0.0, 1e-7);
    }
    while (status == GSL_CONTINUE && iter < 1000);
    
#if defined GSL_MAJOR_VERSION && GSL_MAJOR_VERSION >= 2
    J = gsl_matrix_alloc(n, vector_size);
    gsl_multifit_fdfsolver_jac(s, J);
    gsl_multifit_covar (J, 0.0, covar);
    gsl_matrix_free (J);
#else
    gsl_multifit_covar (s->J, 0.0, covar);
#endif

    
    c = gsl_vector_get(s->x, 0);
    sig_c = sqrt(gsl_matrix_get(covar,0,0));
    for (j = 0; j < m; j++) {
        k[j] = gsl_vector_get(s->x, j+1);
        sig_k[j] = sqrt(gsl_matrix_get(covar,j+1,j+1));
        x0[j] = gsl_vector_get(s->x, m+j+1);
        sig_x0[j] = sqrt(gsl_matrix_get(covar, m+j+1, m+j+1));
        fwhm[j] = gsl_vector_get(s->x, 2*m+j+1);
        sig_fwhm[j] = sqrt(gsl_matrix_get(covar, 2*m+j+1, 2*m+j+1));
    }
    
    if (print_flag) {
        for (i = 0; i < data->m; i++) {
            espdr_msg("NGAUSS after fit: x0[%d] = %lf", i, x0[i]);
            espdr_msg("NGAUSS after fit: sig_x0[%d] = %lf", i, sig_x0[i]);
            espdr_msg("NGAUSS after fit: fwhm[%d] = %lf", i, fwhm[i]);
            espdr_msg("NGAUSS after fit: sig_fwhm[%d] = %lf", i, sig_fwhm[i]);
            espdr_msg("NGAUSS after fit: k[%d] = %lf", i, k[i]);
            espdr_msg("NGAUSS after fit: sig_k[%d] = %lf", i, sig_k[i]);
        }
    }
    
    res->n = n;
    res->m = m;
    res->c = c;
    res->sig_c = sig_c;
    for (j = 0; j < m; j++) {
        res->k[j] = k[j];
        res->sig_k[j] = sig_k[j];
        res->x0[j] = x0[j];
        res->sig_x0[j] = sig_x0[j];
        res->fwhm[j] = fwhm[j];
        res->sig_fwhm[j] = sig_fwhm[j];
    }
    
    gsl_multifit_fdfsolver_free(s);
    gsl_matrix_free(covar);
    gsl_vector_free(v);
    
    cpl_free(k);
    cpl_free(ind);
    cpl_free(fwhm);
    cpl_free(x0);
    cpl_free(sig_k);
    cpl_free(sig_x0);
    cpl_free(sig_fwhm);
    
    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 @brief     polynomial fit (wrap for the gsl functions)
 @param         data            data to be fitted
 @param         deg             polynomial degree
 @param[out]    fit_RE          fit result
 @param[out]    coeffs_RE       coeffs of the fit
 @param[out]    coeffs_err_RE   error on coeffs of the fit
 @param[out]    chisq_RE        CHI2 of the fit
 @param         print_flag      1 - print, 0 - don't
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_fit_poly (espdr_poly_data *data,
                               int deg,
                               double *fit_RE,
                               double *coeffs_RE,
                               double *coeffs_err_RE,
                               double *chisq_RE,
                               int print_flag) {
    
    int i, j, n;
    double err, chisq;
    gsl_vector *x, *y, *w, *p;
    gsl_matrix *X, *covar;
    //double *fit;
    gsl_multifit_linear_workspace *work;
    
    n = data->n;
    x = gsl_vector_alloc (n);
    y = gsl_vector_alloc (n);
    w = gsl_vector_alloc (n);
    p = gsl_vector_alloc (deg);
    X = gsl_matrix_alloc (n, deg);
    covar = gsl_matrix_alloc (deg, deg);
    
    if (print_flag) {
        espdr_msg("n = %d, deg = %d", n, deg);
    }
    
    for (i = 0; i < n; i++) {
        gsl_vector_set (x, i, data->x[i]);
        gsl_vector_set (y, i, data->y[i]);
        err = data->err[i];
        gsl_vector_set (w, i, 1.0/err/err);
        
        for (j = 0; j < deg; j++)
            gsl_matrix_set (X, i, j, gsl_pow_int(gsl_vector_get(x, i), j));
    }
    
    if (print_flag) {
        espdr_msg("vectors set");
    }
    
    work = gsl_multifit_linear_alloc (n, deg);
    gsl_multifit_wlinear (X, w, y, p, covar, &chisq, work);
    gsl_multifit_linear_free (work);
    
    if (print_flag) {
        espdr_msg("Fit done");
    }
    
    for (i = 0; i < n; i++) {
        fit_RE[i] = 0.;
        for (j = 0; j < deg; j++)
            fit_RE[i] += gsl_matrix_get (X, i, j) * gsl_vector_get (p, j);
    }
    
    chisq = chisq/(n-deg);
    
    for (j = 0; j < deg; j++) {
        coeffs_RE[j] = gsl_vector_get (p, j);
        coeffs_err_RE[j] = sqrt(gsl_matrix_get (covar, j, j));
    }
    
    *chisq_RE = chisq;
    
    gsl_vector_free(x);
    gsl_vector_free(y);
    gsl_vector_free(w);
    gsl_vector_free(p);
    gsl_matrix_free(X);
    gsl_matrix_free(covar);
    
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief     polynomial fit (wrap for the gsl functions)
 @param         data            data to be fitted
 @param         deg             polynomial degree
 @param[out]    fit_RE          fit result
 @param[out]    coeffs_RE       coeffs of the fit
 @param[out]    coeffs_err_RE   error on coeffs of the fit
 @param[out]    chisq_RE        CHI2 of the fit
 @param         print_flag      1 - print, 0 - don't
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_fit_poly_2D(espdr_poly_2D_data *data,
                                 double *fit_RE,
                                 espdr_inst_config *inst_config,
                                 double *coeffs_RE,
                                 double *coeffs_err_RE,
                                 double *chisq_RE,
                                 int print_flag) {
    
    int i, j, n, m;
    double err, chisq;
    gsl_vector *z, *w, *p;
    gsl_matrix *A, *covar;
    //double *fit;
    gsl_multifit_linear_workspace *work;
    
    n = data->n;
    m = (inst_config->th_drift_fit_deg_x+1)*(inst_config->th_drift_fit_deg_y+1);
    z = gsl_vector_alloc (n);
    w = gsl_vector_alloc (n);
    p = gsl_vector_alloc (m);
    A = gsl_matrix_alloc (n, m);
    covar = gsl_matrix_alloc (m, m);
    
    if (print_flag) {
        espdr_msg("n = %d, m = %d", n, m);
    }
    
    for (i = 0; i < n; i++) {
        gsl_vector_set (z, i, data->z[i]);
        err = data->err[i];
        gsl_vector_set (w, i, 1.0/err/err);
        
        int index_poly = 0;
        for (int pow_x = 0; pow_x <= inst_config->th_drift_fit_deg_x; pow_x++){
            for (int pow_y = 0; pow_y <= inst_config->th_drift_fit_deg_y; pow_y++){
                gsl_matrix_set (A, i, index_poly,
                                gsl_pow_int(data->x[i],pow_x)*gsl_pow_int(data->y[i],pow_y));
                index_poly++;
            }
        }
    }
    
    if (print_flag) {
        espdr_msg("vectors set");
    }
    
    work = gsl_multifit_linear_alloc (n, m);
    gsl_multifit_wlinear (A, w, z, p, covar, &chisq, work);
    gsl_multifit_linear_free (work);
    
    if (print_flag) {
        espdr_msg("Fit done");
    }
    
    for (i = 0; i < n; i++) {
        fit_RE[i] = 0.;
        for (j = 0; j < m; j++)
            fit_RE[i] += gsl_matrix_get (A, i, j) * gsl_vector_get (p, j);
    }
    
    chisq = chisq/(n-m);
    
    for (j = 0; j < m; j++) {
        coeffs_RE[j] = gsl_vector_get (p, j);
        coeffs_err_RE[j] = sqrt(gsl_matrix_get (covar, j, j));
    }
    
    *chisq_RE = chisq;
    
    gsl_vector_free(z);
    gsl_vector_free(w);
    gsl_vector_free(p);
    gsl_matrix_free(A);
    gsl_matrix_free(covar);
    
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief
 @param
 @param
 @param
 @return   CPL_ERROR_NONE iff OK
 
 TODO: to be filled by ASE
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_spline_interpolation (double *x1, double *y1, int n1,
                                           double *x2, double *y2, int n2) {
    
    int i;
    
    espdr_ensure(x1 == NULL, CPL_ERROR_NULL_INPUT, "x1 is NULL");
    espdr_ensure(y1 == NULL, CPL_ERROR_NULL_INPUT, "y1 is NULL");
    espdr_ensure(x2 == NULL, CPL_ERROR_NULL_INPUT, "x2 is NULL");
    espdr_ensure(y2 == NULL, CPL_ERROR_NULL_INPUT, "y2 is NULL");
    
    gsl_interp_accel *acc = gsl_interp_accel_alloc ();
    gsl_spline *spline = gsl_spline_alloc (gsl_interp_cspline, n1);
    gsl_spline_init (spline, x1, y1, n1);
    
    for (i = 0; i < n2; i++)
        y2[i] = gsl_spline_eval (spline, x2[i], acc);
    
    gsl_spline_free (spline);
    gsl_interp_accel_free(acc);
    
    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 @brief
 @param
 @param
 @param
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_spline_bounded_interpolation (double *x1, double *y1, int n1,
                                                   double *x2, double *y2, int n2,
                                                   double x_min, double x_max) {

    int i;

    espdr_ensure(x1 == NULL, CPL_ERROR_NULL_INPUT, "x1 is NULL");
    espdr_ensure(y1 == NULL, CPL_ERROR_NULL_INPUT, "y1 is NULL");
    espdr_ensure(x2 == NULL, CPL_ERROR_NULL_INPUT, "x2 is NULL");
    espdr_ensure(y2 == NULL, CPL_ERROR_NULL_INPUT, "y2 is NULL");

    gsl_interp_accel *acc = gsl_interp_accel_alloc ();
    gsl_spline *spline = gsl_spline_alloc (gsl_interp_cspline, n1);
    gsl_spline_init (spline, x1, y1, n1);

        /* ASE : since GSL 1.16, gsl_spline_eval(spline,x,acc) fails   *
         * when x is not strictly between on the min/max of the spline *
         * structure initialized at                                    *
         * gsl_spline_init(spline,xaxis[],yaxis[],size).               *
         * So, interp->xmin and interp->xmax, has been forced to be    *
         * the limits on the xaxis that we need to interpolate.        *
         * To adapt GSL conventions to ESPRD conventions, defining     *
         * limits accordingly.                                         */

    spline->interp->xmin=x_min;
    spline->interp->xmax=x_max;

    for (i = 0; i < n2; i++)
        y2[i] = gsl_spline_eval (spline, x2[i], acc);

    gsl_spline_free (spline);
    gsl_interp_accel_free(acc);

    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief
 @param
 @param
 @param
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_spline_bounded_interp_exterp (double *x1, double *y1, int n1,
                                                   double *x2, double *y2, int n2) {

    int i;

    espdr_ensure(x1 == NULL, CPL_ERROR_NULL_INPUT, "x1 is NULL");
    espdr_ensure(y1 == NULL, CPL_ERROR_NULL_INPUT, "y1 is NULL");
    espdr_ensure(x2 == NULL, CPL_ERROR_NULL_INPUT, "x2 is NULL");
    espdr_ensure(y2 == NULL, CPL_ERROR_NULL_INPUT, "y2 is NULL");
    
    double *x2_short, *y2_short;
    int start_index = 0, end_index = n2, n2_short = n2;
    i = 0;
    while (x2[i] < x1[0]) i++;
    start_index = i;
    while (x2[i] <= x1[n1-1]) i++;
    end_index = i-1;
    n2_short = end_index - start_index;
    x2_short = (double *)cpl_calloc(n2_short, sizeof(double));
    y2_short = (double *)cpl_calloc(n2_short, sizeof(double));
        
    for (i = start_index; i < end_index; i++) {
        x2_short[i-start_index] = x2[i];
    }
    
    //espdr_msg("Interpolation from %f - %f (%d elem) into %f - %f (%d elem), instead of %f - %f (%d elem)",
    //          x1[0], x1[n1-1], n1, x2_short[0], x2_short[n2_short-1], n2_short, x2[0], x2[n2-1], n2);
    
    gsl_interp_accel *acc = gsl_interp_accel_alloc ();
    gsl_spline *spline = gsl_spline_alloc (gsl_interp_cspline, n1);
    gsl_spline_init (spline, x1, y1, n1);

        /* ASE : since GSL 1.16, gsl_spline_eval(spline,x,acc) fails   *
         * when x is not strictly between on the min/max of the spline *
         * structure initialized at                                    *
         * gsl_spline_init(spline,xaxis[],yaxis[],size).               *
         * DSO: So, the interpolation is done on a x2/y2 beeing within *
         * x1/y1 limits, the rest of the y2 is filled with constants:  *
         * y2_short[0] and y2_short[n2_short-1]                        */

    for (i = 0; i < n2_short; i++) {
        y2_short[i] = gsl_spline_eval (spline, x2_short[i], acc);
    }

    gsl_spline_free (spline);
    gsl_interp_accel_free(acc);
    
    for (i = 0; i < start_index; i++) {
        y2[i] = y2_short[0];
    }
    for (i = start_index; i < end_index; i++) {
        y2[i] = y2_short[i-start_index];
    }
    for (i = end_index; i < n2; i++) {
        y2[i] = y2_short[n2_short-1];
    }
    
    cpl_free(x2_short);
    cpl_free(y2_short);
    
    return cpl_error_get_code();
}


