/*-----------------------------------------------------------------*/
/*!
  \file itrf_icrf.c
  \brief transformations between International Terrestrial Reference Frame
         and International Celestial Reference Frame

  \author  H. Manche
           Astronomie et Systemes Dynamiques, IMCCE, CNRS, Observatoire de Paris.

   Copyright 2014, 2015, 2016, CNRS
   email of the author : herve.manche@obspm.fr

  last revision: 05/06/2014

  History:
  \note H. Manche 05/06/2014 : creation
 */
/*-----------------------------------------------------------------*/

/*-----------------------------------------------------------------*/
/* License  of this file :
 This file is "triple-licensed", you have to choose one  of the three licenses
 below to apply on this file.

    CeCILL-C
    	The CeCILL-C license is close to the GNU LGPL.
    	( http://www.cecill.info/licences/Licence_CeCILL-C_V1-en.html )

 or CeCILL-B
        The CeCILL-B license is close to the BSD.
        (http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.txt)

 or CeCILL v2.0
      The CeCILL license is compatible with the GNU GPL.
      ( http://www.cecill.info/licences/Licence_CeCILL_V2-en.html )


This library is governed by the CeCILL-C, CeCILL-B or the CeCILL license under
French law and abiding by the rules of distribution of free software.
You can  use, modify and/ or redistribute the software under the terms
of the CeCILL-C,CeCILL-B or CeCILL license as circulated by CEA, CNRS and INRIA
at the following URL "http://www.cecill.info".

As a counterpart to the access to the source code and  rights to copy,
modify and redistribute granted by the license, users are provided only
with a limited warranty  and the software's author,  the holder of the
economic rights,  and the successive licensors  have only  limited
liability.

In this respect, the user's attention is drawn to the risks associated
with loading,  using,  modifying and/or developing or reproducing the
software by the user in light of its specific status of free software,
that may mean  that it is complicated to manipulate,  and  that  also
therefore means  that it is reserved for developers  and  experienced
professionals having in-depth computer knowledge. Users are therefore
encouraged to load and test the software's suitability as regards their
requirements in conditions enabling the security of their systems and/or
data to be ensured and,  more generally, to use and operate it in the
same conditions as regards security.

The fact that you are presently reading this means that you have had
knowledge of the CeCILL-C,CeCILL-B or CeCILL license and that you accept its terms.
*/
/*-----------------------------------------------------------------*/


#include <stdio.h>
#include <string.h>


#include "itrf_icrf.h"
#include "earth_orientation_parameters.h"
#include "erfa.h"
#include "erfam.h"
#include "errors.h"

/*!
  Initialize the routines that compute the CIP coordinates

  @param ModelName (in): name of the model ("CIP/2006/2006A", "CIP/2000/2000B", ...)
  @param parameters (inout): parameters defining the model

 */
void init_CIPmodel(const char *ModelName,t_ITCRF_Model *parameters)
{

  parameters->ptrF_XYS = NULL ;
  parameters->ptrF_S = NULL ;

  if ( strcmp(ModelName,"CIP/2000/2000A" )==0)
  {
    strcpy(parameters->modelCIP,ModelName);
    parameters->ptrF_XYS= &eraXys00a ;
    parameters->ptrF_S= &eraS00 ;
  }
  else if ( strcmp(ModelName,"CIP/2000/2000B" )==0)
  {
    strcpy(parameters->modelCIP,ModelName);
    parameters->ptrF_XYS= &eraXys00b ;
    parameters->ptrF_S= &eraS00 ;
  }
  else if ( strcmp(ModelName,"CIP/2006/2006A" )==0)
  {
    strcpy(parameters->modelCIP,ModelName);
    parameters->ptrF_XYS= &eraXys06a ;
    parameters->ptrF_S= &eraS06 ;
  }
  else
  {
    FatalError("Unknown model for CIP coordinates",__FILE__,__LINE__);
  }

};





/*!
  Computes the transformation matrix between ICRF and ITRF

  @param JulianDay_TT (in): julian day in TT timescale
  @param EOPvalues (in): values of EOPs (xp,yp,UT1-TT,dX,dY) at JulianDay_TT
  @param matrix (out): transformation matrix from ICRF to ITRF ( X_ITRF = M * X_ICRF)
  @param dmatrix (out): time derivative of the matrix (approximation)
  @param parameters (in): parameters defining the model

 */
void icrf_to_itrf_matrix(const double JulianDay_TT[2],
                         const EarthOrientationParameters_t EOPvalues,
                         double matrix[3][3], double dmatrix[3][3],
                         const t_ITCRF_Model parameters)
{
  double X_CIP,Y_CIP,s ;
  double Matrix_ICRF2Intermediate[3][3] ;
  double Matrix_PolarMotion[3][3] ;
  double JulianDay_UT1[2] ;
  double ERA ; /* Earth Rotation Angle */
  const double dERA=ERFA_D2PI*1.00273781191135448/86400.0 ; /* time derivative (second TT) of ERA */
  double dR[3][3] ; /* time derivative of R matrix (see IERS Conventions 2010, eq 5.1*/
  double xp,yp, sp ; /* quantities for polar motion */
  int i,j ;


  /* Checks if pointers to functions are initialized */
  if ( (parameters.ptrF_XYS==NULL) || (parameters.ptrF_XYS==NULL) )
  {
    FatalError("Pointers to functions not initialized",__FILE__,__LINE__);
  }

  /* Celestial Intermediate Pole and s parameter  */
  (*parameters.ptrF_XYS) ( JulianDay_TT[0], JulianDay_TT[1], &X_CIP, &Y_CIP, &s );

  /* correction of CIP given by the Earth Orientation Parameters */
  if ( parameters.active_CorrectionCIP==1 )
  {
    X_CIP += EOPvalues.dX ;
    Y_CIP += EOPvalues.dY ;
    s = (*parameters.ptrF_S)    ( JulianDay_TT[0], JulianDay_TT[1], X_CIP, Y_CIP );
  }
  eraC2ixys ( X_CIP, Y_CIP, s, Matrix_ICRF2Intermediate );

  /* correction of UT1 given by the Earth Orientation Parameters */
  if ( parameters.active_UT1mUTC==1 )
  {
    (void)eraTtut1(JulianDay_TT[0], JulianDay_TT[1], -EOPvalues.UT1mTT,
                   &JulianDay_UT1[0], &JulianDay_UT1[1]);
  }
  else /* UT1=UTC*/
  {
    double JD_TAI[2] ;
    eraTttai(JulianDay_TT[0], JulianDay_TT[1],&JD_TAI[0],&JD_TAI[1]);
    eraTaiutc(JD_TAI[0], JD_TAI[1],&JulianDay_UT1[0],&JulianDay_UT1[1]);
  }
  ERA = eraEra00 ( JulianDay_UT1[0], JulianDay_UT1[1]) ;

  /* correction of polar motion given by the Earth Orientation Parameters */
  if ( parameters.active_PolarMotion==1 )
  {
    xp = EOPvalues.xp ; yp = EOPvalues.yp ;
  }
  else
  {
    xp = 0.0 ; yp = 0.0 ;
  }
  sp=eraSp00 ( JulianDay_TT[0],JulianDay_TT[1] ) ;
  eraPom00 ( xp, yp, sp, Matrix_PolarMotion ) ;

  /* construction of the matrix from icrf to itrf */
  eraC2tcio ( Matrix_ICRF2Intermediate, ERA, Matrix_PolarMotion, matrix) ;

  /* approximated value of the time derivative of the matrix */
  dR[0][0] = sin(ERA) ; dR[0][1] = -cos(ERA) ; dR[0][2] = 0.0 ;
  dR[1][0] = cos(ERA) ; dR[1][1] =  sin(ERA) ; dR[1][2] = 0.0 ;
  dR[2][0] =    0.0   ; dR[2][1] =    0.0    ; dR[2][2] = 0.0 ;
  for (i=0 ; i<2 ; i++)
  {
    for (j=0 ; j<2 ; j++)
    {
      dR[i][j] *=-dERA ;
    }
  }
 eraRxr(dR, Matrix_ICRF2Intermediate, dmatrix);
 eraRxr(Matrix_PolarMotion, dmatrix, dmatrix);
};


/*!
  Computes the transformation matrix between ITRF and ICRF. Calls icrf_to_itrf_matrix
  and the result is then transposed.

  @param JulianDay_TT (in): julian day in TT timescale
  @param EOPvalues (in): values of EOPs (xp,yp,UT1-TT,dX,dY) at JulianDay_TT
  @param matrix (out): transformation matrix from ITRF to ICRF ( X_ICRF = M * X_ITRF)
  @param dmatrix (out): time derivative of the matrix (approximation)
  @param parameters (in): parameters defining the model

 */
void itrf_to_icrf_matrix(const double JulianDay_TT[2],
                         const EarthOrientationParameters_t EOPvalues,
                         double matrix[3][3], double dmatrix[3][3],
                         const t_ITCRF_Model parameters)
{
  double matrix2[3][3] ;
  double dmatrix2[3][3] ;
  int i,j ;

  /* transformation matrix from ICRF to ITRF */
  icrf_to_itrf_matrix(JulianDay_TT, EOPvalues,matrix2, dmatrix2, parameters ) ;

  for (i=0 ; i<3 ; i++)
  {
    for (j=0 ; j<3 ; j++)
    {
      matrix[i][j] = matrix2[j][i] ;
      dmatrix[i][j] = dmatrix2[j][i] ;
    }
  }
};




/*!
  Transformation of a vector from mean equator J2000 to mean ecliptic J2000
  @param vecEQ2000[3] (in): coordinates in mean equator mean equinox J2000
  @param vecECL2000[3] (out): coordinates in mean ecliptic mean equinox J2000
 */
void eq2000_to_ecl2000(const double vecEQ2000[3], double vecECL2000[3])
{

  /* Mean obliquity at J2000, set to  IAU2006 value (in arcsecond, then converted into radians) */
  const double obliquity_J2000 = 84381.406*ERFA_DAS2R ; /*ERFA_DAS2R is defined in erfam.h (arcsecond to radian) */

  vecECL2000[0] =  vecEQ2000[0] ;
  vecECL2000[1] =  cos(obliquity_J2000)*vecEQ2000[1]+ sin(obliquity_J2000)*vecEQ2000[2];
  vecECL2000[2] = -sin(obliquity_J2000)*vecEQ2000[1]+ cos(obliquity_J2000)*vecEQ2000[2];

};
