/*                                                                            *
 *   This file is part of the ESPRESSO Pipeline                               *
 *   Copyright (C) 2006 European Southern Observatory                         *
 *                                                                            *
 *   This library is free software; you can redistribute it and/or modify     *
 *   it under the terms of the GNU General Public License as published by     *
 *   the Free Software Foundation; either version 2 of the License, or        *
 *   (at your option) any later version.                                      *
 *                                                                            *
 *   This program is distributed in the hope that it will be useful,          *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 *   GNU General Public License for more details.                             *
 *                                                                            *
 *   You should have received a copy of the GNU General Public License        *
 *   along with this program; if not, write to the Free Software              *
 *   Foundation, 51 Franklin St, Fifth Floor, Boston, MA  02111-1307  USA     *
 *                                                                            */

/*
 * $Author: dsosnows $
 * $Date: $
 * $Revision: $
 * $Name: not supported by cvs2svn $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*----------------------------------------------------------------------------
 Includes
 ----------------------------------------------------------------------------*/

/* Utility fonctions */
#include <espdr_utils.h>
#include <espdr_msg.h>

/* DFS functions */
#include <espdr_dfs.h>
#include <espdr_instrument.h>
#include <espdr_keywords.h>

/* DRL functions */
#include <espdr_bias.h>
#include <espdr_dark.h>
#include <espdr_led_flat.h>
#include <espdr_orders.h>
#include <espdr_parameters.h>
#include <espdr_CCD.h>
#include <espdr_wave_cal.h>
#include <espdr_background.h>

/* Library */
#include <cpl.h>
#include <assert.h>

/*----------------------------------------------------------------------------
 Defines
 ----------------------------------------------------------------------------*/

#define RECIPE_ID "espdr_orderdef"
#define RECIPE_AUTHOR "P. Figueira, Ch. Lovis, D. Sosnowska"
#define RECIPE_CONTACT "nirps_support@listes.umontreal.ca"

/*-----------------------------------------------------------------------------
 Plugin registration
 -----------------------------------------------------------------------------*/

int cpl_plugin_get_info(cpl_pluginlist * list);

/*----------------------------------------------------------------------------
 Private functions prototypes
 ----------------------------------------------------------------------------*/

/*
 *   Plugin initalization, execute and cleanup handlers
 */

static int espdr_orderdef_create(cpl_plugin *);
static int espdr_orderdef_exec(cpl_plugin *);
static int espdr_orderdef_destroy(cpl_plugin *);

/*----------------------------------------------------------------------------
 Static variables
 ----------------------------------------------------------------------------*/
static char espdr_orderdef_description_short[] =
                            "Defines the orders on the CCD";

static char espdr_orderdef_description[] =
"The recipe computes the position of orders on the CCD\n\
Input Frames : \n\
- A RAW frame tracing the object fiber (Format=RAW, Tag = ORDERDEF_A)\n\
- A RAW frame tracing the calibration fiber (Format=RAW, Tag = ORDERDEF_B)\n\
- A static table indicating the detector geometry (Tag = CCD_GEOM)\n\
- A static table to set instrument config. parameters (Tag = INST_CONFIG)\n\
- The hot pixel map frame (Format=PRE, Tag = HOT_PIXEL_MASK)\n\
- The non-linear pixels map frame (Format=PRE, Tag = BAD_PIXEL_MASK)\n\
Products : \n\
- An order table for each fiber (Format=PRE, PRO.CATG = ORDER_TABLE_A/B)\n";

/*----------------------------------------------------------------------------
 Functions code
 ----------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*/
/**
 @brief    Build the list of available plugins, for this module.
 @param    list    the plugin list
 @return   0 if everything is ok, -1 otherwise
 
 Create the recipe instance and make it available to the application using
 the interface. This function is exported.
 */
/*---------------------------------------------------------------------------*/

int cpl_plugin_get_info(cpl_pluginlist * list)
{
    cpl_recipe *recipe = NULL;
    cpl_plugin *plugin = NULL;
	
    recipe = cpl_calloc(1, sizeof(*recipe));
    if (recipe == NULL) {
		return 1;
    }
	
    plugin = &recipe->interface;
	
    if (cpl_plugin_init(plugin, 
						CPL_PLUGIN_API,	/* Plugin API */
						ESPDR_BINARY_VERSION,	/* Plugin version */
						CPL_PLUGIN_TYPE_RECIPE,	/* Plugin type */
						RECIPE_ID,	/* Plugin name */
						espdr_orderdef_description_short,	/* Short help */
						espdr_orderdef_description,	/* Detailed help */
						RECIPE_AUTHOR,	/* Author name */
						RECIPE_CONTACT,	/* Contact address as in xsh, PACKAGE_BUGREPORT in the tpl */
						espdr_get_license(),	/* Copyright */
						espdr_orderdef_create,
						espdr_orderdef_exec,
						espdr_orderdef_destroy)) {
		cpl_msg_error(cpl_func, "Plugin initialiazation failed");
		(void) cpl_error_set_where(cpl_func);
		return 1;
	}
	
	if (cpl_pluginlist_append(list, plugin)) {
		cpl_msg_error(cpl_func, "Error adding plugin to list");
		(void) cpl_error_set_where(cpl_func);
		return 1;
	}
	
	
    /*return (cpl_error_get_code() != CPL_ERROR_NONE); as in xsh*/
	return 0;
}


/*---------------------------------------------------------------------------*/
/**
 @brief    Setup the recipe options
 @param    plugin  the plugin
 @return   0 if everything is ok
 
 Create the recipe instance and make it available to the application using
 the interface.
 
 */
/*--------------------------------------------------------------------------*/

static int espdr_orderdef_create(cpl_plugin * plugin)
{
	cpl_recipe *recipe = NULL;
    cpl_error_code my_error = cpl_error_get_code();
	const char* init_bias_correction_method = "overscan";

	const char* init_ovsc_method = "mean";
	double init_ovsc_ksigma = 4.0;
	int init_ovsc_max_iter = 10;

	const char* init_res_method = "mean";
	double init_res_ksigma = 6.0; // Was 4.0
	int init_res_max_iter = 10;
	
    /* Do not create the recipe if an error code is already set */
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "An error is alreadu set: %s",
                 cpl_error_get_message_default(my_error));
    
	espdr_ensure(plugin == NULL, CPL_ERROR_NULL_INPUT, "Null plugin");
	
    /* Verify plugin type */
	espdr_ensure(cpl_plugin_get_type(plugin) != CPL_PLUGIN_TYPE_RECIPE, 
                 CPL_ERROR_TYPE_MISMATCH, "Plugin is not a recipe");
	
    /* Get the recipe */
    recipe = (cpl_recipe *)plugin;
	
    /* Create the parameters list in the cpl_recipe object */
    recipe->parameters = cpl_parameterlist_new();
	espdr_ensure(recipe->parameters == NULL, 
                 CPL_ERROR_ILLEGAL_OUTPUT,
                 "Parameter list allocation failed");
	
	/* Set led_ff parameters */
	//espdr_OVSC_param OVSC_param = {};
	espdr_OVSC_param OVSC_param = {init_ovsc_method, init_ovsc_ksigma,
        init_ovsc_max_iter, init_bias_correction_method} ;
	espdr_ORDERDEF_param ORDERDEF_param = {};
	
	my_error = espdr_parameters_OVSC_create(RECIPE_ID,
                                            recipe->parameters,
                                            &OVSC_param);
	espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Error creating the OVSC parameters data structure");
	
	my_error = espdr_parameters_ORDERDEF_create(RECIPE_ID,
                                                recipe->parameters,
                                                &ORDERDEF_param);
	espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Error creating the LED_FF parameters data structure");
		
	return (my_error);
}


/*---------------------------------------------------------------------------*/
/**
 @brief    Execute the plugin instance given by the interface
 @param    plugin  the plugin
 @return   0 if everything is ok
 */
/*---------------------------------------------------------------------------*/

static int espdr_orderdef_exec(cpl_plugin * plugin)
{
	cpl_recipe *recipe = NULL;
    int recipe_status;
    cpl_errorstate initial_errorstate = cpl_errorstate_get();
    cpl_error_code my_error = cpl_error_get_code();
	
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "An error is alreadu set: %s",
                 cpl_error_get_message_default(my_error));
    
	espdr_ensure(plugin == NULL, CPL_ERROR_NULL_INPUT, "Null plugin");
	
    /* Verify plugin type */
	espdr_ensure(cpl_plugin_get_type(plugin) != CPL_PLUGIN_TYPE_RECIPE, 
                 CPL_ERROR_TYPE_MISMATCH, "Plugin is not a recipe");
	
    /* Get the recipe */
	recipe = (cpl_recipe *) plugin;
	
    /* Verify parameter and frame lists */
	espdr_ensure(recipe->parameters == NULL, 
                 CPL_ERROR_NULL_INPUT,
                 "Recipe invoked with NULL parameter list");
	
	espdr_ensure(recipe->frames == NULL, CPL_ERROR_NULL_INPUT, 
                 "Recipe invoked with NULL frame set");
	
    /* Set the time display for logs */
    cpl_msg_set_time_on();
    
    /* Invoke the recipe */
    recipe_status = espdr_orderdef(recipe->parameters, recipe->frames, RECIPE_ID);
	
    /* Ensure DFS-compliance of the products */
    if (cpl_dfs_update_product_header(recipe->frames)) {
        if (!recipe_status) recipe_status = (int)cpl_error_get_code();
    }
	
    if (!cpl_errorstate_is_equal(initial_errorstate)) {
        /* Dump the error history since recipe execution start.
		 At this point the recipe cannot recover from the error */
        cpl_errorstate_dump(initial_errorstate, CPL_FALSE, NULL);
    }
	
	/*printf("recipe status: %d\n", recipe_status);*/
    
    /* Cancel the time display for logs */
    cpl_msg_set_time_off();
    
	return recipe_status;
}


/*---------------------------------------------------------------------------*/
/**
 @brief    Destroy what has been created by the 'create' function
 @param    plugin  the plugin
 @return   0 if everything is ok
 */
/*---------------------------------------------------------------------------*/
static int espdr_orderdef_destroy(cpl_plugin * plugin)
{
	cpl_recipe *recipe = NULL;
	
	cpl_error_reset();
	
	/* Verify the plugin existance */
	espdr_ensure(plugin == NULL, CPL_ERROR_NULL_INPUT, "Null plugin");
	
    /* Verify plugin type */
	espdr_ensure(cpl_plugin_get_type(plugin) != CPL_PLUGIN_TYPE_RECIPE, 
                 CPL_ERROR_TYPE_MISMATCH, "Plugin is not a recipe");
	
    /* Get the recipe */
	recipe = (cpl_recipe *) plugin;
	
    cpl_parameterlist_delete(recipe->parameters);
	
	return (cpl_error_get_code());
}


	
