/*
 * espdr_bias_hdrl.c
 *
 *  Created on: Aug 7, 2014
 *      Author: Andrea Modigliani
 */

/*----------------------------------------------------------------------------
                                Includes
 ----------------------------------------------------------------------------*/

#include <espdr_hdrl_func.h>
#include <espdr_pixels.h>
#include <espdr_utils.h>
#include <hdrl.h>
#include <hdrl_imagelist_view.h>
#include <hdrl_imagelist_io.h>
#include <hdrl_imagelist_basic.h>
/*----------------------------------------------------------------------------
                              Functions code
 ----------------------------------------------------------------------------*/

#define ESPDR_CREA_ERROR 0

/*----------------------------------------------------------------------------*/
/**

  @brief    Computes error associated to input image
  @param    data    the input data
  @return   The associated error computed using MAD

  Possible cpl_error_code set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
/*----------------------------------------------------------------------------*/

static cpl_image* espdr_hdrl_compute_error(cpl_image* data)
{
  cpl_image* error;
  double dmad;

  cpl_ensure(data != NULL, CPL_ERROR_NULL_INPUT, NULL);

  cpl_image_get_mad(data, &dmad);
  error = cpl_image_duplicate(data);
  cpl_image_multiply_scalar(error, 0);
  cpl_image_add_scalar(error, (dmad * CPL_MATH_STD_MAD));

  return error;
}

cpl_error_code espdr_hdrl_master_port(cpl_imagelist *raw_data,
                                      double klow,
                                      double khigh,
                                      int max_iter,
                                      cpl_image **master_RE,
                                      double *RON_RE,
                                      int *totalCosmics_RE)
{

   /* convert input cpl_imagelist to hdrl_imagelist */
   hdrl_imagelist* hdrl_ilist=NULL;
   //hdrl_image* himg=NULL;
   //hdrl_parameter      *   lacosmic_params = NULL;
   hdrl_image* master=NULL;

   cpl_imagelist* raw_errs=NULL;
   cpl_image* err=NULL;
   cpl_image* img=NULL;
   //cpl_mask* mask=NULL;
   cpl_image* contrib_map=NULL;
   cpl_image* reject_low=NULL;
   cpl_image* reject_high=NULL;

   //double dmad;
   double stddev=0;

   //int total_cosmics;
   int cosmics;
   int frames_nb=0;
   int i=0;

   //cpl_ensure(raw_data != NULL, CPL_ERROR_NULL_INPUT, NULL);

   if (ESPDR_CREA_ERROR) {
       // generate error imagelist
       frames_nb=cpl_imagelist_get_size(raw_data);
       raw_errs=cpl_imagelist_new();
       for(i=0;i<frames_nb;i++) {
           img=cpl_imagelist_get(raw_data,i);
           err=espdr_hdrl_compute_error(img);
           cpl_imagelist_set(raw_errs,err,i);
       }

       // To use less RAM:
       //TODO: clarify the 1 parameter.
       hdrl_ilist=(hdrl_imagelist* )hdrl_imagelist_const_cpl_row_view(raw_data,raw_errs,(cpl_size)1,cpl_image_get_size_y(img));
       //hdrl_ilist=hdrl_imagelist_create(raw_data,raw_errs);
   } else {
       // No error: use less RAM
       //hdrl_ilist=hdrl_imagelist_create(raw_data,NULL);
       img=cpl_imagelist_get(raw_data,0);
       hdrl_ilist=(hdrl_imagelist* )hdrl_imagelist_const_cpl_row_view(raw_data,NULL,(cpl_size)1,cpl_image_get_size_y(img));
       /* NOTE: we cannot use a wrapper function without errors in HDRL
        * hdrl_ilist=espdr_hdrl_imagelist_to_cplwrap(raw_data,NULL);
        */
   }

   /* compute master frame :
    * Note: HDRL offer only median method, and uses IQR instead of MAD
    * TODO: try to test also test with MAD
    */
     hdrl_imagelist_collapse_sigclip(hdrl_ilist, klow, khigh, max_iter,
                                     &master, &contrib_map, &reject_low,
                                     &reject_high);

     /* TODO: why hard-coded kappa, maxiter? */
     /* TODO: Danuta says that this is not needed as we do not need to flag
      * cosmics we just need the master.
      */
    double kappa_low=5;
    double kappa_high=5;
    cpl_mask* out_mask;
    /* master may contain bad/traps pixels: we flag them before computing stdev
     * BUT THIS METHOD IS 2 TIMES SLOWER.
     * TODO: remove hard coded parameters
     * Method 2D
     */
    hdrl_parameter* bpm_2d_params;

    int maxiter=3;
    int smooth_x=5;
    int smooth_y=5;
    bpm_2d_params = hdrl_bpm_2d_parameter_create_filtersmooth(kappa_low, kappa_high, maxiter,
                CPL_FILTER_MEDIAN, CPL_BORDER_FILTER, smooth_x, smooth_y) ;
    out_mask = hdrl_bpm_2d_compute(master, bpm_2d_params);
    hdrl_parameter_delete(bpm_2d_params);
    *master_RE=cpl_image_duplicate(hdrl_image_get_image(master));
    cpl_image_reject_from_mask(*master_RE, out_mask);
    //*totalCosmics_RE=cpl_mask_count(out_mask);
    //espdr_msg("cosmics 2d=%d",*totalCosmics_RE);
    cpl_mask_delete(out_mask);
    stddev=cpl_image_get_stdev(*master_RE);
     //cpl_image_delete(*master_RE);

    /*
     * Search for cosmics on each input frame: Method 3D
     */
     hdrl_parameter * bpm_3d_params = hdrl_bpm_3d_parameter_create(kappa_low,kappa_high,
                                                  HDRL_BPM_3D_THRESHOLD_RELATIVE);

     cpl_imagelist* out_imlist = hdrl_bpm_3d_compute(hdrl_ilist, bpm_3d_params);
     hdrl_parameter_delete(bpm_3d_params);
     //cpl_imagelist_save(out_imlist,"pippo1.fits",CPL_TYPE_FLOAT,NULL,CPL_IO_DEFAULT);
     int nimg=cpl_imagelist_get_size(out_imlist);
     int tot_cosmics=0;

     for(i=0;i<nimg;i++) {
         img=cpl_imagelist_get(out_imlist,i);
         cosmics=cpl_image_get_flux(img);
         //espdr_msg("cosmics 2d=%d",cosmics);
         tot_cosmics +=cosmics;
     }
     //espdr_msg("tot cosmics 3d=%d",tot_cosmics);
     //cpl_image* out_mask_ima=cpl_imagelist_collapse_create(out_imlist);
     cpl_imagelist_delete(out_imlist);
     //cpl_image_save(out_mask_ima,"pippo2.fits",CPL_TYPE_FLOAT,NULL,CPL_IO_DEFAULT);

     //out_mask=cpl_mask_threshold_image_create(out_mask_ima,0,nimg);
     *totalCosmics_RE=tot_cosmics;
     espdr_msg("cosmics 3d=%d",*totalCosmics_RE);

     //cpl_image_threshold(out_mask_ima,0,nimg/2,0,1);
     //espdr_msg("Real bad pixels=%g",cpl_image_get_flux(out_mask_ima));

     //*master_RE=cpl_image_duplicate(hdrl_image_get_image(master));
     //cpl_image_reject_from_mask(*master_RE, out_mask);

     //cpl_mask_delete(out_mask);
     //cpl_image_delete(out_mask_ima);

   /* make robust stdev: get mad, flag major outliers as NAN, reject NANs
      * FASTER (but simpler) METHOD
     *master_RE=cpl_image_duplicate(hdrl_image_get_image(master));
     cpl_image_get_mad(*master_RE, &dmad);
     stddev=dmad;
     //for(i=0;i<stats_param->stats_max_iter;i++) {
         double median=cpl_image_get_median(*master_RE);
         cpl_image_threshold(*master_RE,
                             median-4*stddev,
                             median+4*stddev,
                             median,median);
         cpl_mask_threshold_image()
         cpl_image_reject_value(*master_RE,CPL_VALUE_NAN);
         stddev=cpl_image_get_stdev(*master_RE);
         espdr_msg("stdev=%g mad=%g",stddev,dmad);
     //}
   */
    //cpl_image_save(*master_RE,"master.fits",CPL_TYPE_FLOAT,NULL,CPL_IO_CREATE);

   /* TODO: RON comes slightly different */
   *RON_RE = stddev * sqrt(frames_nb);
   espdr_msg("stdev=%g frames_nb=%d RON=%g",stddev,frames_nb,*RON_RE);

   /*free memory */
   hdrl_image_delete(master);
   cpl_imagelist_delete(raw_errs);
   hdrl_imagelist_delete(hdrl_ilist);
   //hdrl_imagelist_unwrap(hdrl_ilist);
   cpl_image_delete(contrib_map);
   cpl_image_delete(reject_low);
   cpl_image_delete(reject_high);
   return cpl_error_get_code();
}


cpl_error_code espdr_hdrl_hot_pixels(cpl_imagelist *master_dark_list,
                                     espdr_CCD_geometry *CCD_geom,
                                     espdr_inst_config *inst_config,
                                     double *mean_RE,
                                     int *hot_pixels_RE,
                                     double *sigma_RE,
                                     cpl_imagelist **hot_pixel_mask_list_RE) {

    int i, j, k, index;
    cpl_image *curr_image;
    //cpl_image *curr_output;
    cpl_image *hot_pixel_image;
    //cpl_vector *curr_image_vector = NULL;
    //int real_llx, real_lly, real_urx, real_ury;

    //int *hot_pixel_mask = NULL;
    //cpl_error_code my_error = CPL_ERROR_NONE;
    //double CoD = 0.0;
    //double sigma = 0.0;
    double mean = 0.0, stddev = 0.0;
    //double low_limit = 0.0, high_limit = 0.0;
    int hot_pixels;
    /* forced MAD value in case MAD == 0.0 */
    //double forced_MAD = 0.01;

    hdrl_image* curr_himg;
    cpl_image* curr_error;
    cpl_mask* out_mask;
    index = 0;
    int smooth_x=7;
    int smooth_y=7;
    cpl_filter_mode filter_mode = CPL_FILTER_MEDIAN ;
    cpl_border_mode border = CPL_BORDER_FILTER ;
    hdrl_parameter * bpm_params = hdrl_bpm_2d_parameter_create_filtersmooth(inst_config->hotpixels_ksigma,
                                                                            inst_config->hotpixels_ksigma,
                                                                            inst_config->hotpixels_max_iter,
                                                                            filter_mode, border,
                                                                            smooth_x, smooth_y) ;
    for (i = 0; i < CCD_geom->ext_nb; i++) {
        for (j = 0; j < CCD_geom->exts[i].out_nb_x; j++) {
            for (k = 0; k < CCD_geom->exts[i].out_nb_y; k++) {
                curr_image = cpl_imagelist_get(master_dark_list, index);
                curr_error=espdr_hdrl_compute_error(curr_image);
                curr_himg=hdrl_image_create(curr_image,curr_error);
                out_mask = hdrl_bpm_2d_compute(curr_himg, bpm_params);

                /* AMO: are the following position needed if we use HDRL?
                 * do one do the computation only on a sub-region of the image?
                 */
                /* AMO: commented out as not used in HDRL
                int real_llx = CCD_geom->exts[i].outputs[j][k].real_llx;
                int real_lly = CCD_geom->exts[i].outputs[j][k].real_lly;
                int real_urx = CCD_geom->exts[i].outputs[j][k].real_urx;
                int real_ury = CCD_geom->exts[i].outputs[j][k].real_ury;
                */


                /* Not needed using HDRL
                int used_real_nx = real_urx - real_llx + 1;
                int used_real_ny = real_ury - real_lly + 1;
                curr_image_vector = cpl_vector_wrap(used_real_nx * used_real_ny,
                                                    cpl_image_get_data_double(curr_image));



                my_error = espdr_sig_clip(curr_image_vector,
                                          DARK_param->hotpixels_ksigma,
                                          DARK_param->hotpixels_ksigma,
                                          DARK_param->hotpixels_sigma_clipping_method,
                                          DARK_param->hotpixels_max_iter, 1, forced_MAD,
                                          &CoD, &sigma, &hot_pixels,
                                          &mean, &stddev,
                                          &low_limit, &high_limit);
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "espdr_sig_clip() failed");
                cpl_vector_unwrap(curr_image_vector);
                */

                hot_pixel_image = cpl_image_new_from_mask(out_mask);
                cpl_image_multiply_scalar(hot_pixel_image,HOT_PIXEL);
                mean=cpl_image_get_mean(hdrl_image_get_image(curr_himg));
                stddev=cpl_image_get_stdev(hdrl_image_get_image(curr_himg));
                hot_pixels=(int)cpl_mask_count(out_mask);
                cpl_mask_delete(out_mask);
                cpl_image_delete(curr_error);
                hdrl_image_delete(curr_himg);

                /*
                hot_pixel_image = cpl_image_new(used_real_nx, used_real_ny,
                                                CPL_TYPE_INT);

                my_error = espdr_create_mask(curr_image,
                                             used_real_nx * used_real_ny,
                                             low_limit, high_limit, HOT_PIXEL,
                                             &hot_pixel_image);

                espdr_ensure(my_error != CPL_ERROR_NONE, my_error, "espdr_create_mask() failed");
                */
                mean_RE[index] = mean;
                hot_pixels_RE[index] = hot_pixels;
                sigma_RE[index] = stddev;

                cpl_imagelist_set(*hot_pixel_mask_list_RE,
                                  cpl_image_duplicate(hot_pixel_image), index);
                cpl_image_delete(hot_pixel_image);
                index++;
            }
        }
    }
    hdrl_parameter_delete(bpm_params);
    return cpl_error_get_code();

}

/* AMO added to simplify */
static cpl_error_code
espdr_oscan_get_real_region(const espdr_CCD_geometry *CCD_geom,
                            const int ext_no,
                            const int out_x, const int out_y,
                            int* real_llx, int* real_lly,
                            int* real_urx, int* real_ury,
                            int* nx, int* ny) {
    
    *real_llx = CCD_geom->exts[ext_no].outputs[out_x][out_y].real_llx;
    *real_lly = CCD_geom->exts[ext_no].outputs[out_x][out_y].real_lly;
    *real_urx = CCD_geom->exts[ext_no].outputs[out_x][out_y].real_urx;
    *real_ury = CCD_geom->exts[ext_no].outputs[out_x][out_y].real_ury;
    *nx = CCD_geom->exts[ext_no].outputs[out_x][out_y].real_nx;
    *ny = CCD_geom->exts[ext_no].outputs[out_x][out_y].real_ny;
    
    return cpl_error_get_code();
    
}
/* AMO added to simplify */
static cpl_error_code
espdr_oscan_get_pscan_region(const espdr_CCD_geometry *CCD_geom,
                             const int ext_no,
                             const int out_x, const int out_y,
                             int* pscan_llx, int* pscan_lly,
                             int* pscan_urx, int* pscan_ury,
                             int* pscan_nx, int* pscan_ny) {
    
    *pscan_llx = CCD_geom->exts[ext_no].outputs[out_x][out_y].pscan_llx;
    *pscan_lly = CCD_geom->exts[ext_no].outputs[out_x][out_y].pscan_lly;
    *pscan_urx = CCD_geom->exts[ext_no].outputs[out_x][out_y].pscan_urx;
    *pscan_ury = CCD_geom->exts[ext_no].outputs[out_x][out_y].pscan_ury;
    
    *pscan_nx = CCD_geom->exts[ext_no].outputs[out_x][out_y].pscan_nx;
    *pscan_ny = CCD_geom->exts[ext_no].outputs[out_x][out_y].pscan_ny;
    
    return cpl_error_get_code();
    
}

/* AMO added to simplify */
static cpl_error_code
espdr_oscan_get_oscan_region(const espdr_CCD_geometry *CCD_geom,
                             const int ext_no,
                             const int out_x, const int out_y,
                             int* oscan_llx, int* oscan_lly,
                             int* oscan_urx, int* oscan_ury,
                             int* oscan_nx, int* oscan_ny) {
    
    *oscan_llx = CCD_geom->exts[ext_no].outputs[out_x][out_y].oscan_llx;
    *oscan_lly = CCD_geom->exts[ext_no].outputs[out_x][out_y].oscan_lly;
    *oscan_urx = CCD_geom->exts[ext_no].outputs[out_x][out_y].oscan_urx;
    *oscan_ury = CCD_geom->exts[ext_no].outputs[out_x][out_y].oscan_ury;
    
    *oscan_nx = CCD_geom->exts[ext_no].outputs[out_x][out_y].oscan_nx;
    *oscan_ny = CCD_geom->exts[ext_no].outputs[out_x][out_y].oscan_ny;
    
    return cpl_error_get_code();
    
}

cpl_error_code espdr_hdrl_overscan(const cpl_image *master_bias_image,
                                   int ext_no,
                                   int out_x,
                                   int out_y,
                                   const espdr_CCD_geometry *CCD_geom,
                                   espdr_inst_config *inst_config,
                                   cpl_image **overscan_image_RE,
                                   double *RON_RE,
                                   int *total_cosmics_RE) {

    if(total_cosmics_RE == NULL) {
           /* ASE : Do nothing Just to document this paramater is not used */
    }	

    /* collumn_orientation: 0 = horizontal, 1 = vertical */
    /* correction_type: 0 = line_by_line, 1 = collumn_by_collumn */

    /* loop indices */
    //int i = 0;
    //int line = 0;
    //int ovsc_point = 0;
    /* CCD orientation */
    int horizontal = 0;
    /* size of the prescan + overscan */
    int data_size_x = 0;
    int data_size_y = 0;
    /* vector of concatenated prescan and overscan */
    //double *data_vector = NULL;
    /* standard deviation of the (prescan,overscan) line or column */
    /* mean of the (prescan,overscan) line */
    //double line_overscan = 0.0;
    /* mean of all the (prescan,overscan) standard deviations */

    /* flag indictaing to reject 0 when calculating MAD */
    /* not used anymore */
    //int reject_zeroMAD = 1;
    /* value for MAD if MAD == 0.0 */
    //double forced_MAD = 0.1;
    /* mea, stddev and limits of sig_clip*/
    //double mean, stddev;
    //double reject_low, reject_high;
    /* error code */
    cpl_error_code my_error = CPL_ERROR_NONE;
    /* precsan and overscan image */
    cpl_image *prescan, *overscan, *pre_overscan;
    //cpl_image *pre_overscan_line;
    //double overscan_value = 0.0;

    /* AMO Simplified */
    int real_llx,real_lly,real_urx,real_ury,nx,ny;
    espdr_oscan_get_real_region(CCD_geom,ext_no,out_x,out_y,
                                         &real_llx, &real_lly,
                                         &real_urx, &real_ury,
                                         &nx,&ny);


    int used_real_nx = real_urx - real_llx + 1;
    int used_real_ny = real_ury - real_lly + 1;
    int collumn_orientation = CCD_geom->exts[ext_no].coll_or;
    int correction_type = CCD_geom->exts[ext_no].corr_type;


    int pscan_llx,pscan_lly,pscan_urx,pscan_ury,pscan_nx,pscan_ny;
    espdr_oscan_get_pscan_region(CCD_geom,ext_no,out_x,out_y,
                                     &pscan_llx, &pscan_lly,
                                     &pscan_urx, &pscan_ury,
                                     &pscan_nx,  &pscan_ny);


    int oscan_llx,oscan_lly,oscan_urx,oscan_ury,oscan_nx,oscan_ny;

    espdr_oscan_get_oscan_region(CCD_geom,ext_no,out_x,out_y,
                                 &oscan_llx, &oscan_lly,
                                 &oscan_urx, &oscan_ury,
                                 &oscan_nx,  &oscan_ny);


    espdr_msg_debug("Starting espdr_overscan with");
    espdr_msg_debug("\tREAL llx = %d, lly = %d, urx = %d, ury = %d",
                       real_llx, real_lly, real_urx, real_ury);

    espdr_msg_debug("\tREAL nx = %d, ny = %d, coll = %d, corr = %d",
                    nx,ny,
                    CCD_geom->exts[ext_no].coll_or,
                    CCD_geom->exts[ext_no].corr_type);
    espdr_msg_debug("\tPSCAN llx = %d, lly = %d, urx = %d, ury = %d",
                    pscan_llx, pscan_lly, pscan_urx, pscan_ury);

    espdr_msg_debug("\tPSCAN nx = %d, ny = %d",
                    pscan_nx,pscan_ny);

    espdr_msg_debug("\tOSCAN llx = %d, lly = %d, urx = %d, ury = %d",
                    oscan_llx, oscan_lly, oscan_urx, oscan_ury);
    espdr_msg_debug("\tOSCAN nx = %d, ny = %d",
                    oscan_nx, oscan_ny);

    /* the direction is set by collumn_orientation and correction_type */
    if (((collumn_orientation == 0) && (correction_type == 1)) ||
        ((collumn_orientation == 1) && (correction_type == 0))) {
        /*AMO: this seems to be useful only to determine if horizontal is 1 :
         * It actually determines also data_size_x and data_size_y
         */
        horizontal = 1;
        /* average value calculated along the horizontal direction */
        /* data_size = length of the prescan + overscan */
        data_size_x = (pscan_urx - pscan_llx + 1) + (oscan_urx - oscan_llx + 1);
        data_size_y = used_real_ny;
    } else {
        if (((collumn_orientation == 1) && (correction_type == 1)) ||
            ((collumn_orientation == 0) && (correction_type == 0))) {
            horizontal = 0;
            /* average value calculated along the vertical direction */
            /* data_size = length of the prescan + overscan */
            data_size_x = used_real_nx;
            data_size_y = (pscan_ury - pscan_lly + 1) + (oscan_ury - oscan_lly + 1);
        } else {
            espdr_ensure(0, CPL_ERROR_INCOMPATIBLE_INPUT,
                         "Wrong combination of collumn_orientation and correction_type, exiting");
        }
    }

    /* Merge the pre and overscan images into one */
    pre_overscan = cpl_image_new(data_size_x, data_size_y, CPL_TYPE_DOUBLE);


    prescan = cpl_image_extract(master_bias_image, pscan_llx, pscan_lly,
                                pscan_urx, pscan_ury);
    espdr_ensure(prescan == NULL, CPL_ERROR_ILLEGAL_OUTPUT,
                 "Prescan image resulting from extract is NULL");

    if (horizontal) {
        overscan = cpl_image_extract(master_bias_image,
                                     pscan_nx + nx + oscan_llx, oscan_lly,
                                     pscan_nx + nx + oscan_urx, oscan_ury);
    } else {
        overscan = cpl_image_extract(master_bias_image,
                                     oscan_llx, pscan_ny + ny + oscan_lly,
                                     oscan_urx, pscan_ny + ny + oscan_ury);
    }

    espdr_ensure(overscan == NULL, CPL_ERROR_ILLEGAL_OUTPUT,
                 "Overscan image resulting from extract is NULL");

    /*
    espdr_msg("whole pre-overscan data_size_x = %d, data_size_y = %d",
              data_size_x, data_size_y);
    espdr_msg("pscan: %d %d %d %d", pscan_llx, pscan_lly, pscan_urx, pscan_ury);
    */
    my_error = cpl_image_copy(pre_overscan, prescan, 1, 1);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "cpl_image_copy with prescan failed: %s",
                 cpl_error_get_message_default(my_error));

    if (horizontal) {
        my_error = cpl_image_copy(pre_overscan, overscan,
                                  (pscan_urx-pscan_llx+2), 1);
    } else {
        my_error = cpl_image_copy(pre_overscan, overscan,
                                  1, (pscan_ury-pscan_lly+2));
    }
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "cpl_image_copy with overscan failed: %s",
                 cpl_error_get_message_default(my_error));

    /* Overscan computation */

    /* overscan_vector is the whole real pixels vector */


    /* AMO HDRL part
     *
     */
    hdrl_parameter * os_params = NULL;
    /* TODO: retrieve RON value, define box_hsize */
    double ccd_ron=1;
    int box_hsize = 0;
    hdrl_direction      correction_direction;
    hdrl_parameter  *   collapse;
    hdrl_parameter  *   rect_region;

    if(horizontal) {

        correction_direction = HDRL_X_AXIS;
        rect_region=hdrl_rect_region_parameter_create(pscan_nx + nx + oscan_llx, oscan_lly,
                                          pscan_nx + nx + oscan_urx, oscan_ury);
        /*
        espdr_msg("pscan llx=%d lly=%d urx=%d ury=%d",
                  pscan_nx + nx + oscan_llx, oscan_lly,
                  pscan_nx + nx + oscan_urx, oscan_ury);
                  */
    } else {
        correction_direction = HDRL_Y_AXIS;
        rect_region=hdrl_rect_region_parameter_create(oscan_llx, pscan_ny + ny + oscan_lly,
                                          oscan_urx, pscan_ny + ny + oscan_ury);
    }
    if ( strcmp(inst_config->ovsc_sigma_clipping_method,"mean") == 0 ) {
        collapse = HDRL_COLLAPSE_MEAN;
    } else {
        collapse = HDRL_COLLAPSE_MEDIAN;
    }

    os_params = hdrl_overscan_parameter_create(correction_direction,ccd_ron,
                    box_hsize,collapse,rect_region);
    /*
    espdr_msg("ima sx=%d sy=%d",
              cpl_image_get_size_x(master_bias_image),
              cpl_image_get_size_y(master_bias_image));
              */
    /*
    cpl_image* error = espdr_hdrl_compute_error(master_bias_image);
    cpl_image_save(master_bias_image, "hdrl_data.fits",CPL_BPP_IEEE_FLOAT,NULL,CPL_IO_DEFAULT);
    cpl_image_save(error, "hdrl_error.fits",CPL_BPP_IEEE_FLOAT,NULL,CPL_IO_DEFAULT);
    hdrl_image* himg = hdrl_image_create(master_bias_image,error);
    cpl_type type = cpl_image_get_type(hdrl_image_get_image(himg));
    */
    hdrl_overscan_compute_result * os_computation;
    os_computation = hdrl_overscan_compute(master_bias_image, os_params);
    hdrl_parameter_delete(os_params);
    hdrl_parameter_delete(rect_region);

    cpl_image* hoscan;
    hdrl_image* correction= hdrl_overscan_compute_result_get_correction(os_computation);

    //cpl_image* contribution=hdrl_overscan_compute_result_get_contribution(os_computation);
    hoscan = hdrl_image_get_image(correction);
    cpl_image_save(hoscan, "oscan.fits",CPL_BPP_IEEE_FLOAT,NULL,CPL_IO_DEFAULT);
    *overscan_image_RE = cpl_image_new(used_real_nx, used_real_ny,CPL_TYPE_DOUBLE);
    double* pima=cpl_image_get_data_double(*overscan_image_RE);
    double* poscan=cpl_image_get_data_double(hoscan);
    //espdr_msg("nx=%d ny=%d",cpl_image_get_size_x(hoscan),cpl_image_get_size_y(hoscan));
    //espdr_msg("nx=%d ny=%d",used_real_nx,used_real_ny);
    double val=0;
    if(horizontal) {

        for(int j=0;j<used_real_ny;j++){
            val=poscan[j];
            for(int i=0;i<used_real_nx;i++){
                pima[j*used_real_nx+i]=val;
            }
        }

    } else {

        for(int j=0;j<used_real_ny;j++){
            for(int i=0;i<used_real_nx;i++){
                pima[j*used_real_nx+i]=poscan[i];
            }
        }
    }
    /* end HDRL specific part the rest from espresso code can be removed */

    my_error = cpl_error_get_code();
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "cpl_image_wrap_double failed");

    /* copy the output variables */
    espdr_msg_debug("RON_RE = %lf", *RON_RE);
    hdrl_overscan_compute_result_delete(os_computation);
    cpl_image_delete(prescan);
    cpl_image_delete(overscan);
    cpl_image_delete(pre_overscan);
    return cpl_error_get_code();
}



