/*                                                                            *
 *   This file is part of the ESPRESSO Pipeline                               *
 *   Copyright (C) 2006 European Southern Observatory                         *
 *                                                                            *
 *   This library is free software; you can redistribute it and/or modify     *
 *   it under the terms of the GNU General Public License as published by     *
 *   the Free Software Foundation; either version 2 of the License, or        *
 *   (at your option) any later version.                                      *
 *                                                                            *
 *   This program is distributed in the hope that it will be useful,          *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 *   GNU General Public License for more details.                             *
 *                                                                            *
 *   You should have received a copy of the GNU General Public License        *
 *   along with this program; if not, write to the Free Software              *
 *   Foundation, 51 Franklin St, Fifth Floor, Boston, MA  02111-1307  USA     *
 *                                                                            */

/*
 * $Author: dsosnows $
 * $Date: 2013-10-24 15:16:17 $
 * $Revision: 1.5 $
 * $Name: not supported by cvs2svn $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*----------------------------------------------------------------------------
 Includes
 ----------------------------------------------------------------------------*/

#include <espdr_led_flat.h>
#include <espdr_detector_signature.h>
/*----------------------------------------------------------------------------
 Defines
 ----------------------------------------------------------------------------*/

#define RECIPE_ID "espdr_led_ff"
#define RECIPE_AUTHOR "P. Figueira, Ch. Lovis, D. Sosnowska, A. Modigliani"
#define RECIPE_CONTACT "nirps_support@listes.umontreal.ca"

/*-----------------------------------------------------------------------------
 Plugin registration
 -----------------------------------------------------------------------------*/

int cpl_plugin_get_info(cpl_pluginlist * list);

/*----------------------------------------------------------------------------
 Private functions prototypes
 ----------------------------------------------------------------------------*/

/*
 *   Plugin initalization, execute and cleanup handlers
 */

static int espdr_led_ff_create(cpl_plugin *);
static int espdr_led_ff_exec(cpl_plugin *);
static int espdr_led_ff_destroy(cpl_plugin *);


/*----------------------------------------------------------------------------
 Static variables
 ----------------------------------------------------------------------------*/
static char espdr_led_ff_description_short[] =
                            "Computes the mean gain and detect the bad pixels";

static char espdr_led_ff_description[] =
"The recipe computes the mean gain and detect the bad pixels\n\
The user should provide at least 2 sets of frames with the same exposure\n\
time, with 5 frames at least in each set\n\
Input Frames : \n\
- A set of n RAW frames (Format=RAW, n >=10, Tag = LED_FF)\n\
- A static table indicating the detector geometry (Tag = CCD_GEOM)\n\
- A static table to set instrument config. parameters (Tag = INST_CONFIG)\n\
- A static table to set the windows for gain comput.(Tag = LED_FF_GAIN_WINDOWS)\n\
- The hot pixel map frame (Format=PRE, Tag = HOT_PIXEL_MASK)\n\
Products : \n\
- A map of the non linear pixels (Format=PRE, PRO.CATG = BAD_PIXEL_MASK)\n";


/*----------------------------------------------------------------------------
 Functions code
 ----------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*/
/**
 @brief    Build the list of available plugins, for this module.
 @param    list    the plugin list
 @return   0 if everything is ok, -1 otherwise
 
 Create the recipe instance and make it available to the application using
 the interface. This function is exported.
 */
/*---------------------------------------------------------------------------*/

int cpl_plugin_get_info(cpl_pluginlist * list)
{
    cpl_recipe *recipe = NULL;
    cpl_plugin *plugin = NULL;
	
    recipe = cpl_calloc(1, sizeof(*recipe));
    if (recipe == NULL) {
		return 1;
    }
	
    plugin = &recipe->interface;
	
    if (cpl_plugin_init(plugin, 
						CPL_PLUGIN_API,	/* Plugin API */
						ESPDR_BINARY_VERSION,	/* Plugin version */
						CPL_PLUGIN_TYPE_RECIPE,	/* Plugin type */
						RECIPE_ID,	/* Plugin name */
						espdr_led_ff_description_short,	/* Short help */
						espdr_led_ff_description,	/* Detailed help */
						RECIPE_AUTHOR,	/* Author name */
						RECIPE_CONTACT,	/* Contact address as in xsh, PACKAGE_BUGREPORT in the tpl */
						espdr_get_license(),	/* Copyright */
						espdr_led_ff_create,
						espdr_led_ff_exec,
						espdr_led_ff_destroy)) {
		cpl_msg_error(cpl_func, "Plugin initialiazation failed");
		(void) cpl_error_set_where(cpl_func);
		return 1;
	}
	
	if (cpl_pluginlist_append(list, plugin)) {
		cpl_msg_error(cpl_func, "Error adding plugin to list");
		(void) cpl_error_set_where(cpl_func);
		return 1;
	}
	
	
    /*return (cpl_error_get_code() != CPL_ERROR_NONE); as in xsh*/
	return 0;
}


/*---------------------------------------------------------------------------*/
/**
 @brief    Setup the recipe options
 @param    plugin  the plugin
 @return   0 if everything is ok
 
 Create the recipe instance and make it available to the application using
 the interface.
 
 TODO: why gain parameter method has so many components? Is it really needed
 to specify different regions for each of the input detectors chips?
 If that is the case would not be better to store than in a reference static
 table? The problem is that in this way the recipe has many input parameters
 and this makes it prone to robustness problems (any mistake in setting any
 of those regions input parameter may lead to a recipe failure), and make
 more time consuming the recipe verifications by regression testing.
 The user manual that has to define each input parameter would be in some way
 more complex.
 --> DSO:   Yes, it is really needed to specify different regions for each
            detector output. These parameters should be easily changeable
            by the user, at least during commissionings. FITS tables are 
            not so easy to update.
 --> DSO:   Windows definition moved to the static fits table.
 
 */
/*--------------------------------------------------------------------------*/

static int espdr_led_ff_create(cpl_plugin * plugin)
{
	cpl_recipe *recipe = NULL;
	cpl_error_code error_got = CPL_ERROR_NONE;
	const char* init_bias_correction_method = "overscan";

	const char* init_ovsc_method = "mean";
	double init_ovsc_ksigma = 4.0;
	int init_ovsc_max_iter = 10;

	const char* init_bad_pixels_method = "mean";
	double init_bad_pixels_ksigma = 10.0;
	int init_bad_pixels_max_iter = 10000; // Was 10;

	const char* init_led_ff_method = "median";
	double init_led_ff_ksigma = 10.;// Was 4.0;

    /* Do not create the recipe if an error code is already set */
    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        cpl_msg_error(cpl_func, "%s():%d: An error is already set: %s",
                      cpl_func, __LINE__, cpl_error_get_where());
        return (int)cpl_error_get_code();
    }
	
	espdr_ensure(plugin == NULL, CPL_ERROR_NULL_INPUT, "Null plugin");
	
    /* Verify plugin type */
	espdr_ensure(cpl_plugin_get_type(plugin) != CPL_PLUGIN_TYPE_RECIPE, 
                 CPL_ERROR_TYPE_MISMATCH, "Plugin is not a recipe");
	
    /* Get the recipe */
    recipe = (cpl_recipe *)plugin;
	
    /* Create the parameters list in the cpl_recipe object */
    recipe->parameters = cpl_parameterlist_new();
	espdr_ensure(recipe->parameters == NULL, 
                 CPL_ERROR_ILLEGAL_OUTPUT,
                 "Parameter list allocation failed");
	
	/* Set led_ff parameters */
	espdr_OVSC_param OVSC_param = {init_ovsc_method, init_ovsc_ksigma,
        init_ovsc_max_iter, init_bias_correction_method} ;
	espdr_LED_FF_param LED_FF_param = {init_led_ff_method, init_led_ff_ksigma,
        init_bad_pixels_method, init_bad_pixels_ksigma, init_bad_pixels_max_iter};
	
	error_got = espdr_parameters_OVSC_create(RECIPE_ID,
                                             recipe->parameters,
                                             &OVSC_param);
	espdr_ensure(error_got != CPL_ERROR_NONE, error_got, 
                 "Error creating the OVSC parameters data structure");
	
	error_got = espdr_parameters_LED_FF_create(RECIPE_ID,
                                               recipe->parameters,
                                               &LED_FF_param);
	espdr_ensure(error_got != CPL_ERROR_NONE, error_got, 
			   "Error creating the LED_FF parameters data structure");
    
	return (error_got);
}


/*---------------------------------------------------------------------------*/
/**
 @brief    Execute the plugin instance given by the interface
 @param    plugin  the plugin
 @return   0 if everything is ok
 */
/*---------------------------------------------------------------------------*/

static int espdr_led_ff_exec(cpl_plugin * plugin)
{
	cpl_recipe *recipe = NULL;
    int recipe_status;
    cpl_errorstate initial_errorstate = cpl_errorstate_get();
    cpl_error_code my_error = cpl_error_get_code();
	
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "An error is alreadu set: %s",
                 cpl_error_get_message_default(my_error));
    
	espdr_ensure(plugin == NULL, CPL_ERROR_NULL_INPUT, "Null plugin");
	
    /* Verify plugin type */
	espdr_ensure(cpl_plugin_get_type(plugin) != CPL_PLUGIN_TYPE_RECIPE, 
                 CPL_ERROR_TYPE_MISMATCH, "Plugin is not a recipe");
	
    /* Get the recipe */
	recipe = (cpl_recipe *) plugin;
	
    /* Verify parameter and frame lists */
	espdr_ensure(recipe->parameters == NULL, 
                 CPL_ERROR_NULL_INPUT,
                 "Recipe invoked with NULL parameter list");
	
	espdr_ensure(recipe->frames == NULL, CPL_ERROR_NULL_INPUT, 
                 "Recipe invoked with NULL frame set");
	
    /* Set the time display for logs */
    cpl_msg_set_time_on();
    
    /* Invoke the recipe */
    recipe_status = espdr_led_ff(recipe->parameters, recipe->frames, RECIPE_ID);
	
    /* Ensure DFS-compliance of the products */
    if (cpl_dfs_update_product_header(recipe->frames)) {
        if (!recipe_status) recipe_status = (int)cpl_error_get_code();
    }
	
    if (!cpl_errorstate_is_equal(initial_errorstate)) {
        /* Dump the error history since recipe execution start.
		 At this point the recipe cannot recover from the error */
        cpl_errorstate_dump(initial_errorstate, CPL_FALSE, NULL);
    }
	
	/*printf("recipe status: %d\n", recipe_status);*/
    
    /* Cancel the time display for logs */
    cpl_msg_set_time_off();
    
	return recipe_status;
}


/*---------------------------------------------------------------------------*/
/**
 @brief    Destroy what has been created by the 'create' function
 @param    plugin  the plugin
 @return   0 if everything is ok
 */
/*---------------------------------------------------------------------------*/
static int espdr_led_ff_destroy(cpl_plugin * plugin)
{
	cpl_recipe *recipe = NULL;
	
	cpl_error_reset();
	
	/* Verify the plugin existance */
	espdr_ensure(plugin == NULL, CPL_ERROR_NULL_INPUT, "Null plugin");
	
    /* Verify plugin type */
	espdr_ensure(cpl_plugin_get_type(plugin) != CPL_PLUGIN_TYPE_RECIPE, 
			   CPL_ERROR_TYPE_MISMATCH, "Plugin is not a recipe");
	
    /* Get the recipe */
	recipe = (cpl_recipe *) plugin;
	
    cpl_parameterlist_delete(recipe->parameters);
	
	return (cpl_error_get_code());
}


	
