/*
 * This file is part of the PIONIER pipeline
 * Copyright (C) 2013 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                  Includes
 -----------------------------------------------------------------------------*/

#include "pioni_utils.h"
#include <cpl.h>

/*-----------------------------------------------------------------------------
                            Static Prototypes
 -----------------------------------------------------------------------------*/

static cpl_image * pioni_mef_get(cpl_frame *, cpl_size, hdrl_parameter *);

/*----------------------------------------------------------------------------*/
/**
 * @defgroup pioni_utils  recipe related utilities
 *
 * TBD
 */
/*----------------------------------------------------------------------------*/

/**@{*/



/*----------------------------------------------------------------------------*/
/**
  @brief Load an hdrl_image from some frames. 
         If missing, an error is determined using a simple detector model.
  @param in             Input frame
  @param ext_num        Extension to load the image from
  @param in_err         Input frame containing errors (optional)
  @param ext_num_err    Extension to load the error from
  @param in_bpm         Input frame containing BPMs (optional)
  @param ext_num_bpm    Extension to load the BPM from
  @param region_params  Region to Load (if NULL, load all)
  @param ron            Read out noise
  @param gain           Detector gain
  @param out_hdrl_ima   Output hdrl_image object
  @return   0 if everything is ok
 */
/*----------------------------------------------------------------------------*/
cpl_error_code pioni_hdrl_image_load(
        cpl_frame       *   in,
        cpl_size            ext_num,
        cpl_frame       *   in_err,
        cpl_size            ext_num_err,
        cpl_frame       *   in_bpm,
        cpl_size            ext_num_bpm,
        hdrl_parameter  *   region_params,
        double              ron,
        double              gain,
        hdrl_image      **  out_hdrl_ima)
{
    cpl_image       *   img ;

    /* Check Entries */
    cpl_ensure_code(in, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(out_hdrl_ima, CPL_ERROR_NULL_INPUT);

    /* Load in image */
    img = pioni_mef_get(in, ext_num, region_params);
    if (img == NULL) return CPL_ERROR_ILLEGAL_INPUT ;

    /* Load BPM */
    if (in_bpm != NULL) {
        cpl_image * bpm = pioni_mef_get(in_bpm, ext_num_bpm, region_params);
        cpl_mask * bpm_mask = cpl_mask_threshold_image_create(bpm,
                -0.5, 0.5) ;
        cpl_image_delete(bpm) ;
        cpl_mask_not(bpm_mask) ;
        cpl_image_reject_from_mask(img, bpm_mask);
        cpl_mask_delete(bpm_mask) ;
    }

    /* Load error */
    cpl_image * err;
    if (in_err != NULL) {
        err = pioni_mef_get(in_err, ext_num_err, region_params);
    } else if (ron < 0. && gain < 0.) {
        /* no error */
        err = NULL;
    } else {
        /* No passed error -> Create uniform error */
        pioni_detector_shotnoise_model(img, gain, ron, &err);
    }

    /* Create out himage */
    *out_hdrl_ima = hdrl_image_create(img, err);

    /* Cleanup */
    cpl_image_delete(img);
    cpl_image_delete(err);
    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
  @brief   compute photon count error in [ADU]
  @param   ima_data in [ADU]
  @param   gain detector's gain in [e- / ADU]
  @param   ron  detector's read out noise in [ADU]
  @param   ima_errs output error image in [ADU]
  @return  cpl_error_code
  @note ima_errs need to be deallocated
        ima_data must contain the photon counts with no offsets
        this usually means the image must be overscan and bias corrected
        Then the shot noise can be calculated from the poissonian distribution
        as sqrt(electron-counts). To this (transformed back into ADUs) the
        readout noise is added in quadrature.
  @doc
  error is computed with standard formula

  \f$ err_{ADU} = \sqrt{ \frac{ counts }{ gain } + ron^{ 2 } } \f$

  If an image value is negative the associated error is set to RON
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
pioni_detector_shotnoise_model(const cpl_image* ima_data, const double gain,
                              const double ron, cpl_image ** ima_errs)
{
    cpl_ensure_code(ima_data, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(ima_errs, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(gain > 0., CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(ron > 0., CPL_ERROR_ILLEGAL_INPUT);

    *ima_errs = cpl_image_duplicate(ima_data);
    /* set negative values (= zero measurable electrons) to read out noise */
    cpl_image_threshold(*ima_errs, 0., INFINITY, ron, ron);

    /* err_ADU = sqrt(counts/gain + ron * ron)*/

    cpl_image_divide_scalar(*ima_errs, gain);
    cpl_image_add_scalar(*ima_errs, ron * ron);
    cpl_image_power(*ima_errs, 0.5);

    return cpl_error_get_code();
}



/* ---------------------------------------------------------------------------*/
/**
 * @brief return NAXIS1 entry of plist, uses ZNAXIS1 if present
 */
/* ---------------------------------------------------------------------------*/
int
pioni_get_naxis1(const cpl_propertylist * plist)
{
    if (cpl_propertylist_has(plist, "ZNAXIS1")) {
        return cpl_propertylist_get_int(plist, "ZNAXIS1");
    }
    else {
        return cpl_propertylist_get_int(plist, "NAXIS1");
    }
}

/* ---------------------------------------------------------------------------*/
/**
 * @brief return NAXIS2 entry of plist, uses ZNAXIS2 if present
 */
/* ---------------------------------------------------------------------------*/
int
pioni_get_naxis2(const cpl_propertylist * plist)
{
    if (cpl_propertylist_has(plist, "ZNAXIS2")) {
        return cpl_propertylist_get_int(plist, "ZNAXIS2");
    }
    else {
        return cpl_propertylist_get_int(plist, "NAXIS2");
    }
}

/* ---------------------------------------------------------------------------*/
/**
 * @brief load header of frame and wrap negative region parameters
 *
 * @param frm    frame to load from
 * @param extnum extension number to load from
 * @param par    region parameter whose ranges should be wrapped by the
 *               image size
 */
/* ---------------------------------------------------------------------------*/
static cpl_error_code
pioni_fix_neg_region(const cpl_frame * frm, cpl_size extnum,
                        hdrl_parameter * par)
{
    cpl_propertylist * plist =
        cpl_propertylist_load(cpl_frame_get_filename(frm), extnum);
    if (!plist) {
        return cpl_error_get_code();
    }
    cpl_size nx = pioni_get_naxis1(plist);
    cpl_size ny = pioni_get_naxis2(plist);
    hdrl_rect_region_fix_negatives(par, nx, ny);
    cpl_propertylist_delete(plist);
    return cpl_error_get_code();
}

/* ---------------------------------------------------------------------------*/
/**
 * @brief Erase keywords that should be written by the cpl saving routine
 *
 * @param proplist_yorick_primary    propertylist to clean
 * @return  cpl_error_code
 */
/* ---------------------------------------------------------------------------*/
cpl_error_code
pioni_cleanup_primary(cpl_propertylist* proplist_yorick_primary)
{

  if (proplist_yorick_primary != NULL) {
      cpl_propertylist_erase_regexp(proplist_yorick_primary, "ESO PRO REC.*", 0);
      /* Removing RADECSYS from primary header as renamed to RADESYS by cpl */
      cpl_propertylist_erase_regexp(proplist_yorick_primary, "RADECSYS", 0);

  }

  return cpl_error_get_code();
}


/**@}*/

static cpl_image * pioni_mef_get(
        cpl_frame       *   frm,
        cpl_size            eidx,
        hdrl_parameter  *   region_params)
{
    cpl_image   *   out = NULL ;

    if (region_params == NULL) {
        out = cpl_image_load(cpl_frame_get_filename(frm),
                CPL_TYPE_DOUBLE, 0, eidx);
    } else {
        pioni_fix_neg_region(frm, eidx, region_params);
        out = cpl_image_load_window(cpl_frame_get_filename(frm),
                CPL_TYPE_DOUBLE, 0, eidx,
                hdrl_rect_region_get_llx(region_params),
                hdrl_rect_region_get_lly(region_params),
                hdrl_rect_region_get_urx(region_params),
                hdrl_rect_region_get_ury(region_params)) ;
    }
    return out ;
}


/*----------------------------------------------------------------------------*/
/**
 * @brief    Save a cpl table oifits compliant
 * @param    table cpl_table
 * @return
 *   The function returns @c CPL_ERROR_NONE on success or a CPL error
 *   code otherwise.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code savetype_cpl_to_oifit(cpl_table * table){

    if (cpl_table_has_column(table, "STA_INDEX"))
    {
        cpl_table_set_column_savetype(table,"STA_INDEX", CPL_TYPE_SHORT);
    }
    if (cpl_table_has_column(table, "TARGET_ID"))
    {
        cpl_table_set_column_savetype(table,"TARGET_ID", CPL_TYPE_SHORT);
    }
    if (cpl_table_has_column(table, "FLAG"))
    {
        cpl_table_set_column_savetype(table,"FLAG",CPL_TYPE_BOOL);
    }

    return cpl_error_get_code();
}


