/*
 * pioni_wrap_yorick.c
 *
 * This file is part of the PIONIER pipeline
 * Copyright (C) 2013,2014 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 *  Created on: Feb 5, 2015
 *      Author: agabasch
 */


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#define _POSIX_C_SOURCE 200112L // needed by remove() and kill()
#include <string.h>
#include <math.h>
#include <cpl.h>
#include <time.h>
#include <sys/wait.h>
#include <sys/types.h>
#include <signal.h>
#include <stdlib.h>

#include "pioni_wrap_yorick.h"

const int PIONIER_ERROR_TIMEOUT  = 0 + CPL_ERROR_EOL;
const double PIONIER_RECIPE_TIMEOUT  = 36000.; // 10 hours

/*----------------------------------------------------------------------------*/
/**
 * @defgroup pioni_yorick  Yorick wrapper
 *
 * TBD
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Execute the yorick wrapper
  @param    yorick_argv  parameters passed to execv()
  @return   cpl_error_code
 */
/*----------------------------------------------------------------------------*/

cpl_error_code pioni_yorick_exec(char ** yorick_argv)
{
    //cpl_errorstate prestate = cpl_errorstate_get();

    int status;
    char path[1035];
    int pid;                                // No process
    int cf_pipe[2];                         // Child to father pipe
    double timeout;
    char * s;

    timeout = PIONIER_RECIPE_TIMEOUT;


    /* export the environment variable read by yorick */

    setenv("PNDRS_DIR", PNDRS_DIR, 1);

    char * pioni_plugin_path = cpl_sprintf("%s", PIONIER_PRIVATELIB_PATH);
    setenv("PIONIER_PLUGIN_PATH", pioni_plugin_path, 1);
    cpl_free(pioni_plugin_path);

    /*  Create the pipe */
    if (pipe(cf_pipe) != 0)
    {
        return cpl_error_set_message(cpl_func, CPL_ERROR_ASSIGNING_STREAM,
                        "Could not create pipe for Yorick process.");
    }

    /* Create the child process to execute Yorick/Mira */
    switch(pid=fork())
    {
    /* Fork error */
    case (-1):
        return cpl_error_set_message(cpl_func, CPL_ERROR_UNSPECIFIED, "Could "
                        "not create the fork Yorick process.");
    /* Child process (Yorick wrapper) : stdout to pipe*/
    case 0:
        cpl_msg_info(cpl_func, "Start Yorick process (pid: %u)",  getpid());

        /* Close the unused side */
        close(cf_pipe[0]);

        /* redirect stdout to the pipe */
        dup2(cf_pipe[1], STDOUT_FILENO);

        /* Execute the yorick script */
        if (execv(yorick_argv[0], yorick_argv) == -1 ){
            return cpl_error_set_message(cpl_func, CPL_ERROR_UNSPECIFIED,
                            "Error in Yorick call (%s)", YORICK_BIN);
        }

        // Child process stop
        printf("Child process (%d) - Stop\n", getpid());
        exit(0);

        /* Parent process : stdin from pipe */
    default:

        /* Close the unused side of the pipe */
        close(cf_pipe[1]);

        /* Redirect the pipe output to STDIN */
        //dup2(cf_pipe[0], STDIN_FILENO);
        FILE * rpipe = fdopen(cf_pipe[0], "r");

        /* Read the output a line at a time - output it. */
        time_t ticks1, ticks2;


        ticks1=time(NULL);
        while (fgets(path, sizeof(path)-1, rpipe) != NULL) {
            /* IMAGE                 */

            /* delete the trailing \n                */
            if ((s = strrchr(path, '\n'))) s[0]='\0';
            /* PNDRS INFO */
            cpl_msg_info("PNDRS/Yorick", "%s", path);

            ticks2=time(NULL);
            if (difftime(ticks2,ticks1) > timeout)
            {
                cpl_error_set_message(cpl_func, CPL_ERROR_UNSPECIFIED,
                                "Timeout (%g s) PNDRS/Yorick process "
                                "killed.", timeout);
                cpl_msg_error(cpl_func, "Timeout PNDRS/Yorick process Killed");
                kill((pid_t) pid, SIGKILL);
            }
        }
        /* wait end of child process */
        wait(&status);
        fclose(rpipe);
        cpl_msg_info(cpl_func, "End of Yorick process (pid: %u with satus %d)",
                     pid, status);
    }

    if (status != 0) {
        return cpl_error_set_message(cpl_func, CPL_ERROR_UNSPECIFIED,
                        "An error occured in the Yorick datareduction part");
    }
    else {
        return cpl_error_get_code();
    }

}

/**@}*/
