/*
 * This file is part of the PIONIER pipeline
 * Copyright (C) 2013,2014 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                    Includes
 -----------------------------------------------------------------------------*/

#include "hdrl.h"

#include "pioni_utils.h"
#include "pioni_dfs.h"
#include "pioni_wrap_yorick.h"
#include <cpl.h>

/*----------------------------------------------------------------------------*/
/**
  @defgroup pioni_dark_calibration     Dark calibration recipe
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*-----------------------------------------------------------------------------
                            Functions prototypes
 -----------------------------------------------------------------------------*/


/*-----------------------------------------------------------------------------
                            Static global variables
 -----------------------------------------------------------------------------*/

#define RECIPE_NAME "pioni_dark_calibration"

static char pioni_dark_calibration_description[] =" TBD \n";



/* Standard CPL recipe definition */
cpl_recipe_define(pioni_dark_calibration, PIONIER_BINARY_VERSION, "HDRL Group", 
        PACKAGE_BUGREPORT, "2015", "Dark Calibration Recipe",
        pioni_dark_calibration_description);                          

/* Function needed by cpl_recipe_define to fill the input parameters */
CPL_DIAG_PRAGMA_PUSH_IGN(-Wunused-parameter);
static cpl_error_code pioni_dark_calibration_fill_parameterlist(
        cpl_parameterlist   *   self)
{

    CPL_UNUSED(self);
    return CPL_ERROR_NONE;
}
CPL_DIAG_PRAGMA_POP;

static int pioni_dark_calibration(
        cpl_frameset            *   frameset, 
        const cpl_parameterlist *   parlist)
{
    cpl_frame               *   dark_cal_frm = NULL;
    char                    *   outfile_recipe = NULL;
    char                    *   outfile_yorick = NULL;
    cpl_propertylist        *   proplist_yorick_primary = NULL;
    cpl_propertylist        *   proplist_yorick_extension = NULL;
    const char              *   dark_cal_file = NULL;
    cpl_errorstate              prestate = cpl_errorstate_get();
    char                    *   yorick_argv[20];
    int                         argv_i = 0;
    int                         i = 0;

    /* Check Entries */
    if (parlist == NULL) {
        return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT,
                "Parameters list not found");
    }

    /* Identify the RAW and CALIB frames in the input frameset */
    if (pioni_dfs_set_groups(frameset)) {
        return cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                "Cannot classify RAW and/or CALIB frames");
    }

    /* Get the input data */
    dark_cal_frm = cpl_frameset_find(frameset, PIONIER_DARK) ;
    if (dark_cal_frm == NULL) {
        return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT,
                        "Frame tagged DARK not found");
    } else {
        dark_cal_file = cpl_frame_get_filename(dark_cal_frm);
        cpl_msg_info(cpl_func, "Input filename %s", dark_cal_file);
        if (dark_cal_file == NULL) {
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                            "Could not retrieve the input filename");
        }
    }

    outfile_yorick = cpl_sprintf("outputFile_darkCalib.fits");
    /* generate the yorick argument table  */
    argv_i=0;
    //yorick_argv[argv_i++] = cpl_sprintf("%s","/usr/bin/echo");
    yorick_argv[argv_i++] = cpl_sprintf("%s",YORICK_BIN);
    yorick_argv[argv_i++] = cpl_sprintf("-batch");
    yorick_argv[argv_i++] = cpl_sprintf("%s%s",PNDRS_DIR,
                    "pioni_dark_calibration.i");
    yorick_argv[argv_i++] = cpl_sprintf("--inputDarkFile=%s",dark_cal_file);
    yorick_argv[argv_i++] = cpl_sprintf("--outputFile=%s", outfile_yorick);
    yorick_argv[argv_i++] = NULL;

    /* Execute the yorick wrapper */
    pioni_yorick_exec(yorick_argv);

    if(!cpl_errorstate_is_equal(prestate)){
        cpl_errorstate_dump(prestate, CPL_FALSE, NULL);
        cpl_free(outfile_yorick);
        for (i=0; i<argv_i-1; i++) cpl_free(yorick_argv[i]);
        return (int)cpl_error_get_code();
    }

    /* Cleanup argv */
    for (i=0; i<argv_i-1; i++) cpl_free(yorick_argv[i]);


    /* Rewrite a dfs conform primary header */

    outfile_recipe = cpl_sprintf("outfile_recipe.fits");
    proplist_yorick_primary = cpl_propertylist_load(outfile_yorick, 0);
    /* Erase keywords that should be written by the cpl routine */
    pioni_cleanup_primary(proplist_yorick_primary);

    if(cpl_propertylist_has(proplist_yorick_primary,  "ESO DET SUBWIN1 GEOMETRY") &&
	cpl_propertylist_has(proplist_yorick_primary, "ESO QC DARK MED") &&
	cpl_propertylist_has(proplist_yorick_primary, "ESO DET DIT") &&
	cpl_propertylist_has(proplist_yorick_primary, "ESO DET NDIT") &&
	cpl_propertylist_has(proplist_yorick_primary, "ESO DET SUBWINS")) {

	double dark_med = cpl_propertylist_get_double(proplist_yorick_primary,
						      "ESO QC DARK MED");
	double det_did = cpl_propertylist_get_double(proplist_yorick_primary,
						     "ESO DET DIT");
	int    det_ndit = cpl_propertylist_get_int(proplist_yorick_primary,
						   "ESO DET NDIT");
	int    det_subwins = cpl_propertylist_get_int(proplist_yorick_primary,
						      "ESO DET SUBWINS");

	const char * subwin1 = cpl_propertylist_get_string(proplist_yorick_primary,
							   "ESO DET SUBWIN1 GEOMETRY");
	int rows = 0;
	sscanf(subwin1, "%dx*", &rows);

	double qc = dark_med / (det_did * det_ndit * det_subwins * rows);
	cpl_propertylist_append_double(proplist_yorick_primary,
				       "ESO QC DARK MED NORM", qc);
    }


    cpl_dfs_save_propertylist(frameset, NULL, parlist, frameset,
                              NULL, "pioni_dark_calibration",
                              proplist_yorick_primary, NULL,
                              PACKAGE "/" PACKAGE_VERSION, outfile_recipe);
    cpl_propertylist_delete(proplist_yorick_primary);

    /* Rewrite all extension */

    cpl_size next = cpl_fits_count_extensions(outfile_yorick);
    for (i=1; i < next + 1 ; i++) {
        proplist_yorick_extension = cpl_propertylist_load(outfile_yorick, i);
        cpl_table * table = cpl_table_load(outfile_yorick, i, 1);
        cpl_table_save(table, NULL, proplist_yorick_extension, outfile_recipe,
                       CPL_IO_EXTEND);
        cpl_table_delete(table);
        cpl_propertylist_delete(proplist_yorick_extension);
    }

    /* cleaning up the working directory */
     remove(outfile_yorick);

    cpl_free(outfile_yorick);
    cpl_free(outfile_recipe);

    return (int)cpl_error_get_code();
}


/**@}*/

