/*
 * This file is part of the PIONIER pipeline
 * Copyright (C) 2013,2014 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                    Includes
 -----------------------------------------------------------------------------*/

#include "hdrl.h"

#include "pioni_utils.h"
#include "pioni_dfs.h"
#include "pioni_wrap_yorick.h"
#include <cpl.h>

/*----------------------------------------------------------------------------*/
/**
  @defgroup pioni_kappa_matrix     Kappa Matrix recipe
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*-----------------------------------------------------------------------------
                            Functions prototypes
 -----------------------------------------------------------------------------*/


/*-----------------------------------------------------------------------------
                            Static global variables
 -----------------------------------------------------------------------------*/

#define RECIPE_NAME "pioni_kappa_matrix"

static char pioni_kappa_matrix_description[] =" TBD \n";



/* Standard CPL recipe definition */
cpl_recipe_define(pioni_kappa_matrix, PIONIER_BINARY_VERSION, "HDRL Group", 
        PACKAGE_BUGREPORT, "2015", "Kappa Matrix Recipe",
        pioni_kappa_matrix_description);                          

/* Function needed by cpl_recipe_define to fill the input parameters */
CPL_DIAG_PRAGMA_PUSH_IGN(-Wunused-parameter);
static cpl_error_code pioni_kappa_matrix_fill_parameterlist(
        cpl_parameterlist   *   self)
{

    CPL_UNUSED(self);
    return CPL_ERROR_NONE;
}
CPL_DIAG_PRAGMA_POP;

static int pioni_kappa_matrix(
        cpl_frameset            *   frameset, 
        const cpl_parameterlist *   parlist)
{
    char                    *   outfile_recipe = NULL;
    char                    *   outfile_yorick = NULL;
    cpl_propertylist        *   proplist_yorick_primary = NULL;
    cpl_propertylist        *   proplist_yorick_extension = NULL;
    char                    *   kappa_mat_file = NULL;
    const char              *   dark_file = NULL;
    cpl_errorstate              prestate = cpl_errorstate_get();
    char                    *   yorick_argv[20];
    int                         argv_i = 0;
    cpl_frame               *   cur_frame = NULL;
    const char              *   tag = NULL;
    int                         nframes = 0;
    int                         i = 0;
    int                         cdark = 0; /*counter for dark files */
    int                         ckappa = 0; /*counter for kappa files */
    /* Check Entries */
    if (parlist == NULL) {
        return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT,
                "Parameters list not found");
    }

    /* Identify the RAW and CALIB frames in the input frameset */
    if (pioni_dfs_set_groups(frameset)) {
        return cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                "Cannot classify RAW and/or CALIB frames");
    }

    /* Get the input data */

    nframes = cpl_frameset_get_size(frameset);

    /* Loop on frames */
    for (i = 0; i < nframes; i++) {
        cur_frame = cpl_frameset_get_position(frameset, i);
        tag = cpl_frame_get_tag(cur_frame);

        /* Kappa Matrix Files */
        if (!strcmp(tag, PIONIER_KAPPA)) {
            if(ckappa == 0 ) {
                kappa_mat_file = cpl_sprintf("%s",
                                cpl_frame_get_filename(cur_frame));
            }
            else {
                char * tmp_file = cpl_sprintf("%s", kappa_mat_file);
                cpl_free(kappa_mat_file);
                kappa_mat_file = cpl_sprintf("%s,%s", tmp_file,
                                cpl_frame_get_filename(cur_frame));
                cpl_free(tmp_file);
            }
            ckappa++;
        }

        /* Dark File */
        if (!strcmp(tag, PIONIER_DARK_CALIBRATION)) {
            dark_file = cpl_frame_get_filename(cur_frame);
            cdark++;
        }
    }

    if (dark_file == NULL) {
        if (kappa_mat_file != NULL) {
            cpl_free(kappa_mat_file);
        }
        return cpl_error_set_message(cpl_func, CPL_ERROR_FILE_NOT_FOUND,
                        "Could not retrieve the input filename for the "
                        "Dark frame");
    }

    if (kappa_mat_file == NULL || ckappa < 4) {
        return cpl_error_set_message(cpl_func, CPL_ERROR_FILE_NOT_FOUND,
                        "Could not retrieve all input filenames for the Kappa "
                        "matrix frames");
    }

    outfile_yorick = cpl_sprintf("outputFile.fits");
    /* generate the yorick argument table  */
    argv_i=0;
    //yorick_argv[argv_i++] = cpl_sprintf("%s","/usr/bin/echo");
    yorick_argv[argv_i++] = cpl_sprintf("%s",YORICK_BIN);
    yorick_argv[argv_i++] = cpl_sprintf("-batch");
    yorick_argv[argv_i++] = cpl_sprintf("%s%s",PNDRS_DIR,
                    "pioni_kappa_matrix.i");
    yorick_argv[argv_i++] = cpl_sprintf("--inputDarkFile=%s", dark_file);
    yorick_argv[argv_i++] = cpl_sprintf("--inputMatrixFiles=%s",kappa_mat_file);
    yorick_argv[argv_i++] = cpl_sprintf("--outputFile=%s", outfile_yorick);
    yorick_argv[argv_i++] = NULL;

    /* Execute the yorick wrapper */
    pioni_yorick_exec(yorick_argv);

    if(!cpl_errorstate_is_equal(prestate)){
        cpl_errorstate_dump(prestate, CPL_FALSE, NULL);
        cpl_free(outfile_yorick);
        cpl_free(kappa_mat_file);
        for (i=0; i<argv_i-1; i++) cpl_free(yorick_argv[i]);
        return (int)cpl_error_get_code();
    }

    /* Cleanup argv */
    for (i=0; i<argv_i-1; i++) cpl_free(yorick_argv[i]);


    /* Rewrite a dfs conform primary header */

    outfile_recipe = cpl_sprintf("outfile_recipe.fits");
    proplist_yorick_primary = cpl_propertylist_load(outfile_yorick, 0);
    /* Erase keywords that should be written by the cpl routine */
    pioni_cleanup_primary(proplist_yorick_primary);

    if(cpl_propertylist_has(proplist_yorick_primary, "ESO QC KAPPARAW1 AVG") &&
	cpl_propertylist_has(proplist_yorick_primary, "ESO QC KAPPARAW2 AVG") &&
	cpl_propertylist_has(proplist_yorick_primary, "ESO QC KAPPARAW3 AVG") &&
	cpl_propertylist_has(proplist_yorick_primary, "ESO QC KAPPARAW4 AVG")) {

	double val1 = cpl_propertylist_get_double(proplist_yorick_primary,
						  "ESO QC KAPPARAW1 AVG");
	double val2 = cpl_propertylist_get_double(proplist_yorick_primary,
						  "ESO QC KAPPARAW2 AVG");
	double val3 = cpl_propertylist_get_double(proplist_yorick_primary,
						  "ESO QC KAPPARAW3 AVG");
	double val4 = cpl_propertylist_get_double(proplist_yorick_primary,
						  "ESO QC KAPPARAW4 AVG");

	double valtot = val1 + val2 + val3 + val4;

	cpl_propertylist_append_double(proplist_yorick_primary,
				       "ESO QC KAPPARAW1 RATIO", val1/valtot);
	cpl_propertylist_append_double(proplist_yorick_primary,
				       "ESO QC KAPPARAW2 RATIO", val2/valtot);
	cpl_propertylist_append_double(proplist_yorick_primary,
				       "ESO QC KAPPARAW3 RATIO", val3/valtot);
	cpl_propertylist_append_double(proplist_yorick_primary,
				       "ESO QC KAPPARAW4 RATIO", val4/valtot);
	cpl_propertylist_append_double(proplist_yorick_primary,
				       "ESO QC KAPPARAW TOT", valtot);
    }

    cpl_dfs_save_propertylist(frameset, NULL, parlist, frameset,
                              NULL, "pioni_kappa_matrix",
                              proplist_yorick_primary, NULL,
                              PACKAGE "/" PACKAGE_VERSION, outfile_recipe);
    cpl_propertylist_delete(proplist_yorick_primary);


    /* Rewrite all extension */

    cpl_size next = cpl_fits_count_extensions(outfile_yorick);
    for (i=1; i < next+1 ; i++) {
        proplist_yorick_extension = cpl_propertylist_load(outfile_yorick, i);
        cpl_imagelist * imagelist = cpl_imagelist_load(outfile_yorick,
                        CPL_TYPE_DOUBLE, i);
        cpl_imagelist_save(imagelist, outfile_recipe, CPL_TYPE_DOUBLE,
                           proplist_yorick_extension, CPL_IO_EXTEND);
        cpl_imagelist_delete(imagelist);
        cpl_propertylist_delete(proplist_yorick_extension);
    }

    /* cleaning up the working directory */
    remove(outfile_yorick);

    cpl_free(outfile_yorick);
    cpl_free(outfile_recipe);
    cpl_free(kappa_mat_file);
    return (int)cpl_error_get_code();
}


/**@}*/

