import numpy as np
import os
from adari_core.plots.panel import Panel
from adari_core.plots.points import LinePlot
from adari_core.plots.text import TextPlot
from adari_core.data_libs.master_interfer_calib import MasterInterferCalibReport
from adari_core.utils.utils import fetch_kw_or_default
from .pionier_utils import PionierSetupInfo, PionierReportMixin
from .pionier_utils import find_ext_index


class PionierFringesReport(PionierReportMixin, MasterInterferCalibReport):
    def __init__(self):
        super().__init__("pionier_fringes")

    def parse_sof(self):

        raw_file = None
        calib_file = None

        for filename, catg in self.inputs:
            if catg == "FRINGE":
                raw_file = filename

        for filename, catg in self.inputs:
            if catg == "CALIB_OIDATA_RAW":
                calib_file = filename

        file_lists = []
        if raw_file is not None:
            file_lists.append(
                {
                    "raw": raw_file,
                    "oi_data": calib_file,
                }
            )
        return file_lists

    def generate_panels(self, **kwargs):
        vspace = 0.4

        # Visibility
        oi_data_ext = "OI_VIS2"
        calib = self.hdus[0]["oi_data"]
        oi_data_iext = find_ext_index(calib, oi_data_ext)

        panels = super().generate_panels(
            oi_ext_list=oi_data_iext, y_label="VIS2", y_lim="lim", **kwargs
        )

        for panel, panel_descr in panels.items():
            i_ext = panel_descr["ext"]
            mdata_hdul = calib
            mdata_ext = 0
            t0 = TextPlot(columns=1, v_space=vspace)
            col0 = (
                "INSTRUME: " + str(mdata_hdul[mdata_ext].header.get("INSTRUME")),
                "EXTNAME: " + oi_data_ext,
                "INSNAME: "
                + str(
                    fetch_kw_or_default(mdata_hdul[i_ext], "INSNAME", default="N/A")
                ).split("(")[0],
                "PRO.CATG: "
                + str(
                    fetch_kw_or_default(
                        mdata_hdul[mdata_ext], "HIERARCH ESO PRO CATG", default="N/A"
                    )
                ),
                "FILE NAME: "
                + str(
                    fetch_kw_or_default(
                        mdata_hdul[mdata_ext],
                        "HIERARCH ESO PRO REC1 CAL1 NAME",
                        default="N/A",
                    )
                ),
                "RAW1 NAME: "
                + str(
                    fetch_kw_or_default(
                        mdata_hdul[mdata_ext],
                        "HIERARCH ESO PRO REC1 RAW1 NAME",
                        default="N/A",
                    )
                ),
            )
            t0.add_data(col0)
            panel.assign_plot(t0, 0, 0, xext=1)

            raw = self.hdus[0]["raw"]
            metadata1 = PionierSetupInfo.fringes(raw)
            t1 = TextPlot(columns=1, v_space=vspace, xext=1)
            col1 = metadata1

            t1.add_data(col1)
            panel.assign_plot(t1, 2, 0, xext=1)

            metadata2 = PionierSetupInfo.calibrator_fringes(mdata_hdul)
            t2 = TextPlot(columns=1, v_space=vspace, xext=1)
            col2 = metadata2[-2:]

            t2.add_data(col2)
            panel.assign_plot(t2, 3, 0, xext=1)

        panels_raw = {}

        # Raw fringes

        ext = "IMAGING_DATA"

        dat_to_plt = ["DATA2", "DATA8", "DATA12", "DATA6", "DATA16", "DATA22"]
        bas_to_plt = ["Bas12", "Bas13", "Bas14", "Bas23", "Bas24", "Bas34"]

        raw = self.hdus[0]["raw"]
        data = raw[ext].data

        p = Panel(
            6, 7, height_ratios=[1, 4, 4, 4, 4, 4, 4], x_stretch=0.75, y_stretch=0.75
        )

        # Text Plot

        fname = os.path.basename(str(raw.filename()))
        mdata_hdul = raw
        mdata_ext = "PRIMARY"
        t0 = TextPlot(columns=1, v_space=vspace, h_space=0)
        col0 = (
            "INSTRUME: " + str(mdata_hdul[mdata_ext].header.get("INSTRUME")),
            "EXTNAME: " + ext,
            "DPR.TYPE: "
            + str(mdata_hdul[mdata_ext].header.get("HIERARCH ESO DPR TYPE")),
            "FILE NAME: " + fname,
            "RAW1 NAME: N/A",
        )
        t0.add_data(col0)
        p.assign_plot(t0, 0, 0, xext=2)

        metadata1 = PionierSetupInfo.fringes(raw)
        t1 = TextPlot(columns=1, v_space=vspace, xext=1)
        col1 = metadata1

        t1.add_data(col1)
        p.assign_plot(t1, 2, 0, xext=1)

        metadata2 = PionierSetupInfo.calibrator_fringes(raw)
        t2 = TextPlot(columns=1, v_space=vspace, xext=1)
        col2 = metadata2[:4]

        t2.add_data(col2)
        p.assign_plot(t2, 3, 0, xext=1)

        t3 = TextPlot(columns=1, v_space=vspace, xext=1)
        col3 = metadata2[4:]

        t3.add_data(col3)
        p.assign_plot(t3, 4, 0, xext=1)

        # Raw fringes plot

        for i, it in enumerate(dat_to_plt):
            dat1 = data[it]
            mean1 = np.mean(dat1, axis=3).flatten()
            frin_plot = LinePlot(
                title="",
                x_label="Frame number",
                y_label=bas_to_plt[i] + ": " + it,
                legend=False,
            )
            frin_plot.set_xlim(0, 1550)
            frin_plot.add_data((np.arange(mean1.size), mean1), label="", color="black")
            p.assign_plot(frin_plot, 0, i + 1, xext=6)

        addme = {
            "ext": ext,
            "report_name": f"PIONIER_{str(fname).removeprefix('PIONI.').removesuffix('.fits')}_{str(ext).lower()}",
            "report_description": f"Fringes panel" f"{ext})",
            "report_tags": [],
        }

        panels_raw[p] = addme

        # Closure

        ext = "OI_T3"
        ext_list = find_ext_index(calib, ext)

        panels_cp = super().generate_panels(
            oi_ext_list=ext_list,
            n_panels=4,
            y_label="Closure phase",
            y_column="T3PHI",
            **kwargs,
        )

        for panel, panel_descr in panels_cp.items():
            i_ext = panel_descr["ext"]
            mdata_hdul = calib
            mdata_ext = "PRIMARY"
            t0 = TextPlot(columns=1, v_space=vspace)
            col0 = (
                "INSTRUME:" + str(mdata_hdul[mdata_ext].header.get("INSTRUME")),
                "EXTNAME: " + ext,
                "INSNAME: "
                + str(
                    fetch_kw_or_default(mdata_hdul[i_ext], "INSNAME", default="N/A")
                ).split("(")[0],
                "PRO.CATG: "
                + str(
                    fetch_kw_or_default(
                        mdata_hdul[mdata_ext], "HIERARCH ESO PRO CATG", default="N/A"
                    )
                ),
                "FILE NAME: "
                + str(
                    fetch_kw_or_default(
                        mdata_hdul[mdata_ext],
                        "HIERARCH ESO PRO REC1 CAL1 NAME",
                        default="N/A",
                    )
                ),
                "RAW1 NAME: "
                + str(
                    fetch_kw_or_default(
                        mdata_hdul[mdata_ext],
                        "HIERARCH ESO PRO REC1 RAW1 NAME",
                        default="N/A",
                    )
                ),
            )
            t0.add_data(col0)
            panel.assign_plot(t0, 0, 0, xext=1)

            metadata1 = PionierSetupInfo.fringes(raw)
            t1 = TextPlot(columns=1, v_space=vspace, xext=1)
            col1 = metadata1

            t1.add_data(col1)
            panel.assign_plot(t1, 2, 0, xext=1)

            metadata2 = PionierSetupInfo.calibrator_fringes(calib)
            t2 = TextPlot(columns=1, v_space=vspace, xext=1)
            col2 = metadata2[-2:]

            t2.add_data(col2)
            panel.assign_plot(t2, 3, 0, xext=1)

        panels = {**panels, **panels_raw, **panels_cp}

        return panels


rep = PionierFringesReport()
