/*
 * This file is part of the PIONIER pipeline
 * Copyright (C) 2014 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                    Includes
 -----------------------------------------------------------------------------*/
#include <string.h>
#include "hdrl.h"

#include "pioni_utils.h"
#include "pioni_dfs.h"
#include <cpl.h>
#include <math.h>

/*----------------------------------------------------------------------------*/
/**
  @defgroup pioni_bpm_fit    BPM
  @par Synopsis: TBD
  @par Input frames: TBD
  @par Output frames: TBD
  @code
  @endcode
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*-----------------------------------------------------------------------------
                            Functions prototypes
 -----------------------------------------------------------------------------*/
static cpl_error_code
pioni_adjust_data_value_sig2(int ndit, double saturation, hdrl_image* input);

static cpl_error_code
pioni_adjust_data_sig2_sig4(int ndit, hdrl_image* input);

static hdrl_image*
pioni_read_bias(int extnum_raw, double saturation, cpl_frame* biasframe,
                hdrl_parameter* region_params);

static hdrl_imagelist*
pioni_read_value_sig2(int extnum_raw, double saturation,
                      cpl_vector** exptime, cpl_frameset* fs_data,
                      cpl_vector** mediancounts, hdrl_parameter* region_params,
                      hdrl_image* bias);

static hdrl_imagelist*
pioni_read_sig2_sig4(cpl_frameset* fs_data, hdrl_parameter* region_params);

static cpl_error_code
pioni_save_bpmfit(cpl_image* out_chi2, cpl_image* out_dof,
                  hdrl_imagelist* out_coef, int degree, cpl_frameset* frameset,
                  const cpl_parameterlist* parlist);

static cpl_error_code
pioni_save_gainfit(cpl_image* out_chi2, cpl_image* out_dof,
                  hdrl_imagelist* out_coef, int degree, cpl_frameset* frameset,
                  const cpl_parameterlist* parlist);

static cpl_error_code pioni_detmon_save(
                const char              *   procatg,
                const char              *   filename,
                const cpl_type              savetype,
                const cpl_image         *   image,
                const cpl_parameterlist *   parlist,
                const cpl_propertylist  *   qclist,
                cpl_frameset            *   frameset);
/*-----------------------------------------------------------------------------
                            Static global variables
 -----------------------------------------------------------------------------*/

#define RECIPE_NAME "pioni_detmon"

static char pioni_detmon_description[] =
"The recipe derives bad pixels and the pixel-gain on a sequence of          \n"
"images by fitting a polynomial to the data.                                \n"
"                                                                           \n"
"Input files:                                                               \n"
"                                                                           \n"
"  DO category:                  Explanation:           Required:           \n"
"  DETMON                        Data                   Yes                 \n"
"  BIAS                          Bias Frame             NO                  \n"
"  STATIC_MASK                   Static mask            NO                  \n"
"                                                                           \n"
"Output files:                                                              \n"
"                                                                           \n"
"  DO category:                  Explanation:                               \n"
"  PIONI_MASTER_BPM              Master BPM                                 \n"
"  PIONI_DETMON_BPMFIT_CHI2      Chi-squared of the bpm-fit                 \n"
"  PIONI_DETMON_BPMFIT_COEFX     Fit Coefficients of the bpm-fit            \n"
"  PIONI_DETMON_BPMFIT_DOF       Degree of Freedom of the bpm-fit           \n"
"  PIONI_DETMON_BPMFIT_RED_CHI2  Reduced Chi-squared of the bpm-fit         \n"
"  PIONI_DETMON_GAIN             Gain and Gain error per pixel              \n"
"  PIONI_DETMON_GAINFIT_CHI2     Chi-squared of the gain-fit                \n"
"  PIONI_DETMON_GAINFIT_COEFX    Fit Coefficients of the gain-fit           \n"
"  PIONI_DETMON_GAINFIT_DOF      Degree of Freedom of the gain-fit          \n"
"  PIONI_DETMON_GAINFIT_RED_CHI2 Reduced Chi-squared of the gain-fit        \n"
"                                                                           \n"
"Usage of the recipe:                                                       \n"
"                                                                           \n"
"BPM:                                                                       \n" 
"                                                                           \n"
"There are currently three methods implemented to determine the             \n"
"bpm. The switch between the different methods is done by giving the        \n"
"related parameters meaningful values (currently they have a value of       \n"
"-1 (default) if they are not used). Moreover, the degree of the fit        \n"
"can be controlled with the (--degree) recipe parameter. The different      \n"
"methods are                                                                \n"
"                                                                           \n"
"                                                                           \n"
"(--rel-chi-low/-high):  This method marks pixels as bad by using the       \n"
"                        relative chi (low/high) threshold:                 \n"
"                        pixels with values below/above                     \n"
"                        this threshold times measured-rms are marked       \n"
"                        as bad.                                            \n"
"                                                                           \n"
"(--rel-coef-low/-high): This method marks pixels as bad by using the       \n"
"                        relative coefficient (low/high) threshold:         \n"
"                        pixels with values below/above this threshold      \n"
"                        times measured-rms are marked as bad. Please       \n"
"                        note that the rms is derived on the full           \n"
"                        coefficient-image.                                 \n"
"                        The output image encodes which coefficient         \n"
"                        was not in the threshold as a power of two.        \n"
"                        E.g. a value of 5 means coefficient 0 and 2        \n"
"                        were not within the relative thresholds.           \n"
"                                                                           \n"
"( --pval):              This method uses the p-value (between 0% and       \n"
"                        100%) to discriminate between good and bad         \n"
"                        pixels. The p-value is the integral of the chi2    \n"
"                        distributions probability density function         \n"
"                        from the fit-chi2 to infinity. Fits with a         \n"
"                        p-value below the threshold are considered to      \n"
"                        be bad pixels.                                     \n"
"                                                                           \n"
"                                                                           \n"
"If a static mask is provided, the number of bad pixels are additionally    \n"
"derived on the detector regions not masked out by this mask. A bad pixel   \n"
"on the detector which is set to unity (1) in the static mask gets excluded \n"
"from the general bad pixel counts.                                         \n"
"                                                                           \n"
"Gain:                                                                      \n"
"                                                                           \n"
"In order to determine the detector gain the variance as a function of      \n"
"the counts is fitted by a first order polynomial. A weighted fit is        \n"
"used and the error of the variance is derived from the fourth-order        \n"
"moment. The gain is then the inverse of the first-order coefficient of     \n"
"the fit.                                                                   \n"
"                                                                           \n"
"Please note, that the DETMON files should contain the header keyword       \n"
"EXPTIME which is used as the sampling position of the fit. The bad         \n"
"pixel map creation parameters must be given in mutually exclusive          \n"
"groups, e.g. abs-rchi2-low AND abs-rchi2-high OR rel-rchi2-low AND         \n"
"rel-rchi2-high OR pval, etc.                                               \n"
"                                                                           \n"
"Please note that part of the code is parallelised. In order to optimise    \n"
"use of computing resources you should set the environment variable         \n"
"OMP_NUM_THREADS to a proper value, like (for bash), to use 4 cores         \n"
"export OMP_NUM_THREADS=4                                                   \n";


/* Standard CPL recipe definition */
cpl_recipe_define(pioni_detmon, PIONIER_BINARY_VERSION, "HDRL Group",
                  PACKAGE_BUGREPORT, "2014", "PIONIER - BPM FIT",
                  pioni_detmon_description);
/* Function needed by cpl_recipe_define to fill the input parameters */
static cpl_error_code pioni_detmon_fill_parameterlist(
        cpl_parameterlist   *   self) 
{                                  
    cpl_parameter   *   par ;

    /* --pioni_detmon.ext-nb-raw */
    par = cpl_parameter_new_value(RECIPE_NAME".ext-nb-raw", CPL_TYPE_INT,
            "FITS extension of the DETMON", RECIPE_NAME, 0);
    cpl_parameter_set_alias(par, CPL_PARAMETER_MODE_CLI, "ext-r");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);
    
    /* --pioni_detmon.region-llx/lly/urx/ury */
    hdrl_parameter * deflts = hdrl_rect_region_parameter_create(1, 1, 0, 0);
    cpl_parameterlist * reg_param = hdrl_rect_region_parameter_create_parlist(
                RECIPE_NAME, "", "region-", deflts);
    hdrl_parameter_delete(deflts);
    for (cpl_parameter * p = cpl_parameterlist_get_first(reg_param);
            p != NULL; p = cpl_parameterlist_get_next(reg_param))
        cpl_parameterlist_append(self, cpl_parameter_duplicate(p));
    cpl_parameterlist_delete(reg_param);


    deflts = hdrl_bpm_fit_parameter_create_rel_coef(1, 4., 4.);
    cpl_parameterlist * fpar =
        hdrl_bpm_fit_parameter_create_parlist(RECIPE_NAME, "", deflts) ;
    hdrl_parameter_delete(deflts);
    for (cpl_parameter * p = cpl_parameterlist_get_first(fpar) ;
            p != NULL; p = cpl_parameterlist_get_next(fpar))
        cpl_parameterlist_append(self, cpl_parameter_duplicate(p));
    cpl_parameterlist_delete(fpar);


    /* --pioni_detmon.saturation */
    par = cpl_parameter_new_value(RECIPE_NAME".saturation", CPL_TYPE_DOUBLE,
            "Saturation [ADU]", RECIPE_NAME, 4000.);
    cpl_parameter_set_alias(par, CPL_PARAMETER_MODE_CLI, "saturation");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);


    return CPL_ERROR_NONE;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Saves the pioni_detmon product in a fitsfile
  @param    procatg    PRO.CATG of the product
  @param    filename   The filename on the product
  @param    savetype   Type of the product
  @param    image      The image to be saved
  @param    parlist    The recipe parameter list
  @param    qclist     QC parameters to be saved with the product
  @param    frameset   The recipe frameset
  @return   CPL_ERROR_NONE if everything is ok, an error code otherwise
 */
/*----------------------------------------------------------------------------*/
static cpl_error_code pioni_detmon_save(
                const char              *   procatg,
                const char              *   filename,
                const cpl_type              savetype,
                const cpl_image         *   image,
                const cpl_parameterlist *   parlist,
                const cpl_propertylist  *   qclist,
                cpl_frameset            *   frameset)
{
    /* Add passed QC parameter  */
    cpl_propertylist  * qclist_loc = cpl_propertylist_new();

    if (qclist != NULL) {
        cpl_propertylist_append(qclist_loc, qclist);
    }

    /* Add the product category and save image */
    cpl_propertylist_update_string(qclist_loc, CPL_DFS_PRO_CATG, procatg);

    cpl_dfs_save_image(frameset, NULL, parlist, frameset, NULL, image,
                       savetype, RECIPE_NAME, qclist_loc, NULL,
                       PACKAGE "/" PACKAGE_VERSION, filename);
    cpl_propertylist_delete(qclist_loc);

    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Convert  (counts, variance) into (counts, error) 
  @param    ndit       Number of detector (sub-)integrations
  @param    saturation The saturation limit of the detector
  @param    input      The image where the apply the conversion
  @return   CPL_ERROR_NONE if everything is ok, an error code otherwise

  The image structure of the DETCHAR files are such that the first
  extension contains the variance, i.e. var = 1/n *
  (Sum((x-x_mean)^2)) and not the error thus in order to get the error
  of the values in extension 0 one has to do: SQRT(var)/sqrt(n)
 */
/*----------------------------------------------------------------------------*/
static cpl_error_code
pioni_adjust_data_value_sig2(int ndit, double saturation, hdrl_image* input)
{

    /* The image structure of the DETCHAR files are such that the first
     * extension contains the variance, i.e. var = 1/n * (Sum((x-x_mean)^2))
     * and not the error thus in order to get the error of the values in
     * extension 0 one has to do: SQRT(var)/sqrt(n) */

    cpl_image_power(hdrl_image_get_error(input), 0.5);
    cpl_image_divide_scalar(hdrl_image_get_error(input), sqrt((double) ndit));
    /* It can happen that the error derived in the instrument is Zero
     * In this case the pixel is set to be bad a priory*/
    /* Assumption: there is no mask in the datasection yet! */
    cpl_mask * mask_from_error = cpl_mask_threshold_image_create(
                    hdrl_image_get_error(input), 0, DBL_MAX);
    cpl_mask * mask_from_image = cpl_mask_threshold_image_create(
                    hdrl_image_get_image(input), 0, saturation);
    cpl_mask_and(mask_from_error, mask_from_image);


    cpl_mask_not(mask_from_error);
    hdrl_image_reject_from_mask(input, mask_from_error);
    cpl_mask_delete(mask_from_error);
    cpl_mask_delete(mask_from_image);

    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Convert  (variance, 4th moment) into (variance, error) 
  @param    ndit       Number of detector (sub-)integrations
  @param    input      The image where the apply the conversion
  @return   CPL_ERROR_NONE if everything is ok, an error code otherwise

 The image structure of the DETCHAR files are such that the first
 extension contains the variance, i.e. mue2 = 1/n *
 (Sum((x-x_mean)^2)) and the third extension contains the forth order
 moment (kurtosis).

 Derive the variance of the variance by using the 4th order moment
 (mue4)

 coeff1 = (n - 1) * (n - 1) / n^3

 coeff2 = (n - 1) * (n - 3) / n^3

 -> Var(mue2^2) = coeff1 * mue4 - coeff2*mue2^2

 -> error = SQRT(Var) See
 http://mathworld.wolfram.com/SampleVarianceDistribution.html and
 (Kenney and Keeping 1951, p. 164; Rose and Smith 2002, p. 264).
  */
/*----------------------------------------------------------------------------*/
static cpl_error_code
pioni_adjust_data_sig2_sig4(int ndit, hdrl_image* input)
{
    cpl_image * subtractima = NULL;
    /* The image structure of the DETCHAR files are such that the first
     * extension contains the variance, i.e. mue2 = 1/n * (Sum((x-x_mean)^2))
     * and the third extension contains the forth order moment (kurtosis).
     *
     * Derive the variance of the variance by using the 4th order moment (mue4)
     *
     * coeff1 = (N - 1) * (N - 1) / N^3
     * coeff2 = (N - 1) * (N - 3) / N^3
     *
     * -> Var(s^2) = coeff1 * mue4 - coeff2*mue2^2
     *
     * -> error = SQRT(Var)
     * See http://mathworld.wolfram.com/SampleVarianceDistribution.html
     * and (Kenney and Keeping 1951, p. 164; Rose and Smith 2002, p. 264).
     * */


    double dndit = (double)ndit;
    double coeff1 = (dndit-1.) * (dndit-1.) / (dndit * dndit * dndit);
    double coeff2 = (dndit-1.) * (dndit-3.) / (dndit * dndit * dndit);

    /* A) coeff1 * mue4 */
    cpl_image_multiply_scalar(hdrl_image_get_error(input), coeff1);

    /* B) coeff2 * mue2^2 */
    subtractima = cpl_image_duplicate(hdrl_image_get_image_const(input));
    cpl_image_power(subtractima, 2.);
    cpl_image_multiply_scalar(subtractima, coeff2);

    /* A) - B) */
    cpl_image_subtract(hdrl_image_get_error(input), subtractima);

    /* Get the error from the variance */
    cpl_image_power(hdrl_image_get_error(input), 0.5);

    /* It can happen that the error derived in the instrument is Zero
     * In this case the pixel is set to be bad a priory*/
    /* Assumption: there is no mask in the datasection yet! */
    cpl_mask * mask_from_error = cpl_mask_threshold_image_create(
                    hdrl_image_get_error(input), 0, DBL_MAX);

    cpl_mask_not(mask_from_error);
    hdrl_image_reject_from_mask(input, mask_from_error);
    cpl_mask_delete(mask_from_error);
    cpl_image_delete(subtractima);

    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Read the bias frame into a hdrl image
  @param    extnum_raw     The extension of the bias frame data section
  @param    saturation     The saturation limit of the detector
  @param    biasframe      The bias frame to be read
  @param    region_params  Region where to read the frame
  @return   The hdrl image containing the bias data and bias errors
 */
/*----------------------------------------------------------------------------*/
static hdrl_image*
pioni_read_bias(int extnum_raw, double saturation, cpl_frame* biasframe,
                hdrl_parameter* region_params)
{
    int ndit = 1;
    hdrl_image * bias = NULL;
    if (pioni_hdrl_image_load(biasframe, extnum_raw, biasframe, 1,
    NULL, 0, region_params, -1., -1., &bias) != CPL_ERROR_NONE) {
        //biasframe = NULL;
        //if(bias != NULL) hdrl_image_delete(bias);
        return NULL;
    }
    else {
        cpl_msg_info(cpl_func, "Bias %s loaded ",
                        cpl_frame_get_filename(biasframe));

        cpl_propertylist * plist = cpl_propertylist_load(
                        cpl_frame_get_filename(biasframe), 0);
        if (cpl_propertylist_has(plist, "ESO DET NDIT")) {
            ndit = cpl_propertylist_get_int(plist, "ESO DET NDIT");
        }
        else {
            cpl_msg_warning(cpl_func, "No ESO DET NDIT, using %d", ndit);
            ndit = 1000;
        }
        cpl_propertylist_delete(plist);

        pioni_adjust_data_value_sig2(ndit, saturation, bias);
    }
    return bias;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Read data and associated error into a hdrl imagelist
  @param    extnum_raw[in]    The extension of the bias frame data section
  @param    saturation[in]    The saturation limit of the detector
  @param    exptime[out]      The exposure time
  @param    fs_data[in]       The frameset with the images to be read
  @param    mediancount[out]  The median of the read image after bias-subtract
  @param    region_params[in] Region where to read the frame
  @param    bias[in]          The bias to subtract or NULL
  @return   The hdrl imagelist containing the the data and associated error
 */
/*----------------------------------------------------------------------------*/
static hdrl_imagelist*
pioni_read_value_sig2(int extnum_raw, double saturation,
                      cpl_vector** exptime, cpl_frameset* fs_data,
                      cpl_vector** mediancounts, hdrl_parameter* region_params,
                      hdrl_image* bias)
{
    hdrl_imagelist * input = hdrl_imagelist_new();
    *exptime = cpl_vector_new(cpl_frameset_get_size(fs_data));
    *mediancounts = cpl_vector_new(cpl_frameset_get_size(fs_data));
    for (cpl_size i = 0; i < cpl_frameset_get_size(fs_data); i++) {
        cpl_frame * frm_d = cpl_frameset_get_position(fs_data, i);
        hdrl_image * hima;
        int ndit = 1;
        cpl_msg_info(cpl_func, "Loading frameset %d", (int) i);

        /* Loading the data and the variance(!) stored in the detmon data in
         * extension 1 - not the error(!) */
        if (pioni_hdrl_image_load(frm_d, extnum_raw, frm_d, 1,
        NULL, 0, region_params, -1., -1., &hima) != CPL_ERROR_NONE) {
             cpl_error_set_message(cpl_func, CPL_ERROR_FILE_NOT_FOUND,
             "Cannot load frame %s", cpl_frame_get_filename(frm_d));
             return NULL;
        }

        cpl_propertylist * plist = cpl_propertylist_load(
                        cpl_frame_get_filename(frm_d), 0);
        if (cpl_propertylist_has(plist, "EXPTIME")) {
            cpl_vector_set(*exptime, i,
                            cpl_propertylist_get_double(plist, "EXPTIME"));
        }
        else {
            cpl_msg_warning(cpl_func, "No EXPTIME keyword, using %d", (int) i);
            cpl_vector_set(*exptime, i, i);
        }

        if (cpl_propertylist_has(plist, "ESO DET NDIT")) {
            ndit = cpl_propertylist_get_int(plist, "ESO DET NDIT");
        }
        else {
            cpl_msg_warning(cpl_func, "No ESO DET NDIT, using %d", ndit);
            ndit = 1000;
        }
        cpl_propertylist_delete(plist);

        pioni_adjust_data_value_sig2(ndit, saturation, hima);

        if (bias) {
            hdrl_image_sub_image(hima, bias);
        }

        hdrl_value median = hdrl_image_get_median(hima);
        cpl_vector_set(*mediancounts, i, median.data);

        hdrl_imagelist_set(input, hima, i);
    }
    return input;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Read variance and associated error into a hdrl imagelist
  @param    fs_data[in]       The frameset with the images to be read
  @param    region_params[in] Region where to read the frame
  @return   The hdrl imagelist containing the the variance and associated error
 */
/*----------------------------------------------------------------------------*/
static hdrl_imagelist*
pioni_read_sig2_sig4(cpl_frameset* fs_data, hdrl_parameter* region_params)
{
    hdrl_imagelist * input = hdrl_imagelist_new();
    for (cpl_size i = 0; i < cpl_frameset_get_size(fs_data); i++) {
        cpl_frame * frm_d = cpl_frameset_get_position(fs_data, i);
        hdrl_image * hima;
        int ndit = 1;
        cpl_msg_info(cpl_func, "Loading frameset %d", (int) i);

        /* Loading the second order moment (extension 1) into the data section
         * and the forth order moment  (kurtosis, extension 3 )into the error
         * section
         */
        if (pioni_hdrl_image_load(frm_d, 1, frm_d, 3,
        NULL, 0, region_params, -1., -1., &hima) != CPL_ERROR_NONE) {
             cpl_error_set_message(cpl_func, CPL_ERROR_FILE_NOT_FOUND,
             "Cannot load frame %s", cpl_frame_get_filename(frm_d));
             return NULL;
        }

        cpl_propertylist * plist = cpl_propertylist_load(
                        cpl_frame_get_filename(frm_d), 0);

        if (cpl_propertylist_has(plist, "ESO DET NDIT")) {
            ndit = cpl_propertylist_get_int(plist, "ESO DET NDIT");
        }
        else {
            cpl_msg_warning(cpl_func, "No ESO DET NDIT, using %d", ndit);
            ndit = 1000;
        }
        cpl_propertylist_delete(plist);

        pioni_adjust_data_sig2_sig4(ndit, hima);
        hdrl_imagelist_set(input, hima, i);
    }
    return input;
}

static cpl_error_code
pioni_save_bpmfit(cpl_image* out_chi2, cpl_image* out_dof,
                  hdrl_imagelist* out_coef, int degree, cpl_frameset* frameset,
                  const cpl_parameterlist* parlist)
{
    cpl_propertylist* qclist = NULL;
   cpl_msg_info(cpl_func, "Storing results");
    pioni_detmon_save("PIONI_DETMON_BPMFIT_CHI2", "pioni_detmon_bpmfit_chi2.fits",
                      CPL_TYPE_FLOAT, out_chi2, parlist, NULL, frameset);
    pioni_detmon_save("PIONI_DETMON_BPMFIT_DOF", "pioni_detmon_bpmfit_dof.fits",
                      CPL_TYPE_FLOAT, out_dof, parlist, NULL, frameset);
    cpl_image_divide(out_chi2, out_dof);
    qclist = cpl_propertylist_new();
    double chi2_mean = cpl_image_get_mean(out_chi2);
    double chi2_median = cpl_image_get_median(out_chi2);
    double chi2_stdev = cpl_image_get_stdev(out_chi2);
    double chi2_stdev_mad = 0.;
    cpl_image_get_mad(out_chi2, &chi2_stdev_mad);
    chi2_stdev_mad *= CPL_MATH_STD_MAD;
    cpl_msg_info(cpl_func, "Mean of reduced chi2: %g", chi2_mean);
    cpl_msg_info(cpl_func, "Median of reduced chi2: %g", chi2_median);
    cpl_msg_info(cpl_func, "Standard deviation of reduced chi2: %g",
                 chi2_stdev);
    cpl_msg_info(cpl_func, "MAD based STD of reduced chi2: %g", chi2_stdev_mad);
    cpl_propertylist_update_double(qclist, "ESO QC REDCHI2 MEAN", chi2_mean);
    cpl_propertylist_update_double(qclist, "ESO QC REDCHI2 MEDIAN",
                                   chi2_median);
    cpl_propertylist_update_double(qclist, "ESO QC REDCHI2 STDEV", chi2_stdev);
    cpl_propertylist_update_double(qclist, "ESO QC REDCHI2 STDEVMAD",
                                   chi2_stdev_mad);
    pioni_detmon_save("PIONI_DETMON_BPMFIT_RED_CHI2", "pioni_detmon_bpmfit_redchi2.fits",
                      CPL_TYPE_FLOAT, out_chi2, parlist, qclist, frameset);
    cpl_propertylist_delete(qclist);
    if (cpl_image_count_rejected(out_chi2)
                    == (cpl_image_get_size_x(out_chi2)
                                    * cpl_image_get_size_y(out_chi2))) {
        cpl_msg_error(cpl_func, "Too few good pixels to fit polynomial of "
                        "degree %d in all pixels", degree);
        return CPL_ERROR_UNSPECIFIED;
    }

    for (cpl_size i = 0; i < hdrl_imagelist_get_size(out_coef); i++) {
        hdrl_image * hd = hdrl_imagelist_get(out_coef, i);
        cpl_image * d = hdrl_image_get_image(hd);
        cpl_image * e = hdrl_image_get_error(hd);
        char fnbuffer[256];
        char tagbuffer[256];
        cpl_msg_info(cpl_func, "Coefficient %d:", (int) i);
        cpl_msg_info(cpl_func, "  Mean: %g", cpl_image_get_mean(d));
        cpl_msg_info(cpl_func, "  Standard deviation: %g",
                     cpl_image_get_stdev(d));
        cpl_msg_info(cpl_func, "  Mean error of fit: %g",
                     cpl_image_get_mean(e));
        sprintf(fnbuffer, "pioni_detmon_bpmfit_coeff%d.fits", (int) i);
        sprintf(tagbuffer, "PIONI_DETMON_BPMFIT_COEF%d", (int) i);
        pioni_detmon_save(tagbuffer, fnbuffer, CPL_TYPE_FLOAT, d, parlist, NULL,
                          frameset);
        cpl_image_save(e, fnbuffer, CPL_TYPE_DOUBLE, NULL, CPL_IO_EXTEND);
    }
    return cpl_error_get_code();
}

static void
pioni_qcstat(cpl_image* gain, const char * string,  cpl_propertylist* qclist)
{
    char * qcparam = NULL;
    double stdev_mad = 0 ;
    cpl_image_get_mad(gain, &stdev_mad);
    stdev_mad *= CPL_MATH_STD_MAD;
    double mediangain = 0.;
    mediangain = cpl_image_get_median(gain);
    //cpl_msg_info(cpl_func, "Gain (median): %g", mediangain);

    qcparam = cpl_sprintf("ESO QC %s MEDIAN", string);
    cpl_propertylist_update_double(qclist, qcparam, mediangain);
    cpl_free(qcparam);
    qcparam = cpl_sprintf("ESO QC %s STDEVMAD", string);
    cpl_propertylist_update_double(qclist, qcparam, stdev_mad);
    cpl_free(qcparam);
    cpl_size stepsize = 40;
    cpl_size size_x = cpl_image_get_size_x(gain);
    cpl_size size_y = cpl_image_get_size_y(gain);
    int counter = 0;
    for(cpl_size i = 0; (i  + stepsize) <= size_x; i += stepsize){
        counter += 1;
        int xmin = i + 1; /*fits convention - boundary included*/
        int xmax = i + stepsize; /*fits convention - boundary included*/
        //cpl_msg_info(cpl_func, "xmin - xmax: %d - %d", xmin, xmax);
        qcparam = cpl_sprintf("ESO QC %s%d MEDIAN", string, counter);
        cpl_propertylist_update_double(qclist, qcparam,
                                       cpl_image_get_median_window(gain,
                                                       xmin, 1, xmax, size_y));
        cpl_free(qcparam);
        qcparam = cpl_sprintf("ESO QC %s%d STDEVMAD", string, counter);
        cpl_image_get_mad_window(gain, xmin, 1, xmax, size_y, &stdev_mad);
        stdev_mad *= CPL_MATH_STD_MAD;
        cpl_propertylist_update_double(qclist, qcparam, stdev_mad);
        cpl_free(qcparam);
    }

}


static cpl_error_code
pioni_save_gainfit(cpl_image* out_chi2, cpl_image* out_dof,
                   hdrl_imagelist* out_coef, int degree, cpl_frameset* frameset,
                   const cpl_parameterlist* parlist)
{
    cpl_propertylist* qclist = NULL;
    cpl_msg_info(cpl_func, "Storing results");
    pioni_detmon_save("PIONI_DETMON_GAINFIT_CHI2", "pioni_detmon_gainfit_chi2.fits",
                      CPL_TYPE_FLOAT, out_chi2, parlist, NULL, frameset);
    pioni_detmon_save("PIONI_DETMON_GAINFIT_DOF", "pioni_detmon_gainfit_dof.fits",
                      CPL_TYPE_FLOAT, out_dof, parlist, NULL, frameset);
    cpl_image_divide(out_chi2, out_dof);
    qclist = cpl_propertylist_new();
    double chi2_mean = cpl_image_get_mean(out_chi2);
    double chi2_median = cpl_image_get_median(out_chi2);
    double chi2_stdev = cpl_image_get_stdev(out_chi2);
    double stdev_mad = 0.;
    cpl_image_get_mad(out_chi2, &stdev_mad);
    stdev_mad *= CPL_MATH_STD_MAD;
    cpl_msg_info(cpl_func, "Mean of reduced chi2: %g", chi2_mean);
    cpl_msg_info(cpl_func, "Median of reduced chi2: %g", chi2_median);
    cpl_msg_info(cpl_func, "Standard deviation of reduced chi2: %g",
                 chi2_stdev);
    cpl_msg_info(cpl_func, "MAD based STD of reduced chi2: %g", stdev_mad);
    cpl_propertylist_update_double(qclist, "ESO QC REDCHI2 MEAN", chi2_mean);
    cpl_propertylist_update_double(qclist, "ESO QC REDCHI2 MEDIAN",
                                   chi2_median);
    cpl_propertylist_update_double(qclist, "ESO QC REDCHI2 STDEV", chi2_stdev);
    cpl_propertylist_update_double(qclist, "ESO QC REDCHI2 STDEVMAD",
                                   stdev_mad);
    pioni_detmon_save("PIONI_DETMON_GAINFIT_RED_CHI2", "pioni_detmon_gainfit_redchi2.fits",
                      CPL_TYPE_FLOAT, out_chi2, parlist, qclist, frameset);
    cpl_propertylist_delete(qclist);
    if (cpl_image_count_rejected(out_chi2)
                    == (cpl_image_get_size_x(out_chi2)
                                    * cpl_image_get_size_y(out_chi2))) {
        cpl_msg_error(cpl_func, "Too few good pixels to fit polynomial of "
                        "degree %d in all pixels", degree);
        return CPL_ERROR_UNSPECIFIED;
    }
    /* Output statistic for single coefficients */
    for (cpl_size i = 0; i < hdrl_imagelist_get_size(out_coef); i++) {
        hdrl_image * hd = hdrl_imagelist_get(out_coef, i);
        cpl_image * d = hdrl_image_get_image(hd);
        cpl_image * e = hdrl_image_get_error(hd);
        char fnbuffer[256];
        char tagbuffer[256];
        cpl_msg_info(cpl_func, "Coefficient %d:", (int) i);
        cpl_msg_info(cpl_func, "  Mean: %g", cpl_image_get_mean(d));
        cpl_msg_info(cpl_func, "  Standard deviation: %g",
                     cpl_image_get_stdev(d));
        cpl_msg_info(cpl_func, "  Mean error of fit: %g",
                     cpl_image_get_mean(e));
        sprintf(fnbuffer, "pioni_detmon_gainfit_coeff%d.fits", (int) i);
        sprintf(tagbuffer, "PIONI_DETMON_GAINFIT_COEF%d", (int) i);
        pioni_detmon_save(tagbuffer, fnbuffer, CPL_TYPE_FLOAT, d, parlist, NULL,
                          frameset);
        cpl_image_save(e, fnbuffer, CPL_TYPE_DOUBLE, NULL, CPL_IO_EXTEND);
    }

    /* Output statistic for the gain iteslf. Please note that the gain is the
     * inverse of the first coefficient for a y=ax+b fit, i.e. a=1/gain */
    {
        /* The gain is the inverse of the first coefficient for a y=ax+b fit */
        cpl_size i = 1;
        hdrl_image * hd = hdrl_image_duplicate(hdrl_imagelist_get(out_coef, i));
        hdrl_image_pow_scalar(hd, (hdrl_value){-1., 0.});
        cpl_image * gain = hdrl_image_get_image(hd);
        cpl_image * gain_error = hdrl_image_get_error(hd);
        /*Save gain*/
        qclist = cpl_propertylist_new();

        pioni_qcstat(gain, "GAIN", qclist);
        pioni_qcstat(gain_error, "GAIN ERROR", qclist);
        pioni_detmon_save("PIONI_DETMON_GAIN", "pioni_detmon_gain.fits",
                          CPL_TYPE_FLOAT, gain, parlist, qclist, frameset);
        cpl_image_save(gain_error, "pioni_detmon_gain.fits", CPL_TYPE_FLOAT,
                       NULL, CPL_IO_EXTEND);
        cpl_propertylist_delete(qclist);
        hdrl_image_delete(hd);
    }

    return cpl_error_get_code();
}

static int pioni_detmon(
        cpl_frameset            *   frameset, 
        const cpl_parameterlist *   parlist)
{
    const cpl_parameter     *   par = NULL;
    int                         extnum_raw;
    int                         degree;
    double                      saturation;
    cpl_error_code err = CPL_ERROR_NONE;
    hdrl_parameter * region_params;
    cpl_vector * exptime = NULL;
    cpl_vector * mediancounts = NULL;
    hdrl_imagelist * input_data = NULL;
    hdrl_imagelist * input_sig2 = NULL;
    hdrl_parameter * fpar = NULL;
    cpl_propertylist * qclist = NULL;
    /* Check Entries */
    if (parlist == NULL) {
        return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT,
                "Parameters list not found");
    }

    /* Get parameters*/
    par = cpl_parameterlist_find_const(parlist, RECIPE_NAME".ext-nb-raw");
    extnum_raw = cpl_parameter_get_int(par);

    /* Parse the Region Parameters */
    region_params = hdrl_rect_region_parameter_parse_parlist(parlist,
            RECIPE_NAME, "region-");
    if (region_params == NULL) {
        return cpl_error_set_message(cpl_func, CPL_ERROR_UNSPECIFIED,
                "Parsing of the region parameters failed");
    }

    par = cpl_parameterlist_find_const(parlist, RECIPE_NAME".saturation");
    saturation = cpl_parameter_get_double(par);

    fpar = hdrl_bpm_fit_parameter_parse_parlist(parlist, RECIPE_NAME);
    if (fpar == NULL) {
        return cpl_error_get_code();
    }
    degree = hdrl_bpm_fit_parameter_get_degree(fpar);


    /* Identify the RAW and CALIB frames in the input frameset */
    if (pioni_dfs_set_groups(frameset)) {
        return cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                "Cannot classify RAW and/or CALIB frames");
    }

    cpl_frameset * fs_data = cpl_frameset_new();

    for (cpl_size i = 0; i < cpl_frameset_get_size(frameset); i++) {
        cpl_frame * frm =
                        cpl_frame_duplicate(cpl_frameset_get_position(frameset, i));
        if (!strcmp(cpl_frame_get_tag(frm), PIONIER_DETMON)) {
            cpl_frameset_insert(fs_data, frm);
        } else {
            cpl_frame_delete(frm);
        }
    }

    /* Read bias frame and subtract if present */
    hdrl_image * bias = NULL;
    cpl_frame * biasframe = cpl_frameset_find(frameset, PIONIER_BIAS);

    if(biasframe) {
        bias = pioni_read_bias(extnum_raw, saturation, biasframe, region_params);
    }

    input_data = pioni_read_value_sig2(extnum_raw, saturation, &exptime,
                    fs_data, &mediancounts, region_params, bias);
    input_sig2 = pioni_read_sig2_sig4(fs_data, region_params);

    cpl_frameset_delete(fs_data);
    hdrl_parameter_delete(region_params);
    hdrl_image_delete(bias);

    cpl_msg_info(cpl_func, "polynomial fit for bpm");
    cpl_image * out_chi2, * out_dof;
    hdrl_imagelist * out_coef;
    err = hdrl_fit_polynomial_imagelist(input_data, exptime, degree, &out_coef,
                    &out_chi2, &out_dof);
    if (err) {
        if (mediancounts != NULL) {
            cpl_vector_delete(mediancounts);
        }
        return cpl_error_set_message(cpl_func, CPL_ERROR_FILE_NOT_FOUND,
                        "Fit failed");
    } else {
        /* Store results */
        pioni_save_bpmfit(out_chi2, out_dof, out_coef, degree, frameset, parlist);
        hdrl_imagelist_delete(out_coef);
        cpl_image_delete(out_chi2);
        cpl_image_delete(out_dof);
    }

    cpl_msg_info(cpl_func, "polynomial fit for gain");


    cpl_imagelist * samplepos = cpl_imagelist_new();
    for (cpl_size i = 0; i < hdrl_imagelist_get_size(input_data); i++) {
        hdrl_image * img = hdrl_imagelist_get(input_data, i);
        cpl_imagelist_set(samplepos, hdrl_image_get_image(img), i);
    }

    /* The fit degree must always be 1 as only then the gain makes sense as
     * the inverse of the first coefficient */

    err = hdrl_fit_polynomial_imagelist2(input_sig2, samplepos, 1, &out_coef,
                    &out_chi2, &out_dof);
    cpl_imagelist_unwrap(samplepos);


    if (err) {
        if(mediancounts != NULL) {
            cpl_vector_delete(mediancounts);
        }
        return cpl_error_set_message(cpl_func, CPL_ERROR_FILE_NOT_FOUND,
                        "Fit failed");
    } else {
        /* Store results */
        pioni_save_gainfit(out_chi2, out_dof, out_coef, degree, frameset, parlist);
        hdrl_imagelist_delete(out_coef);
        cpl_image_delete(out_chi2);
        cpl_image_delete(out_dof);
    }


    {
        cpl_image * bpm;
        qclist = cpl_propertylist_new();
        /* fits again but has less output */
        hdrl_bpm_fit_compute(fpar, input_data, exptime, &bpm);

        cpl_mask * m = cpl_mask_threshold_image_create(bpm, -0.5, 0.5);
        cpl_mask_not(m);

        size_t n = cpl_mask_count(m);
        double p = (double)n / (cpl_mask_get_size_x(m) * cpl_mask_get_size_y(m));
        cpl_msg_info(cpl_func, "%zu bad pixels (%g%%)", n, p * 100.);

        int pos1 = 0, pos2 = 0, pos3 = 0;
        if (cpl_mask_get(m, 253, 51)  == CPL_BINARY_1) pos1 = 1;
        if (cpl_mask_get(m, 253, 203) == CPL_BINARY_1) pos2 = 1;
        if (cpl_mask_get(m, 249, 211) == CPL_BINARY_1) pos3 = 1;

        cpl_propertylist_update_int(qclist, "ESO QC BADPIX", n);
        cpl_propertylist_update_double(qclist, "ESO QC MEDIAN MIN",
                                    cpl_vector_get_min(mediancounts));
        cpl_propertylist_update_double(qclist, "ESO QC MEDIAN MAX",
                                    cpl_vector_get_max(mediancounts));

        pioni_detmon_save("PIONI_MASTER_BPM",
                              "pioni_detmon_bpm.fits", CPL_TYPE_INT, bpm,
                              parlist, qclist, frameset);

        /* Read static mask and derive the number of bad pixles taking the
         * static mask into account  */
        cpl_mask * stat_mask = NULL;
        cpl_frame * stat_mask_frame = NULL;

        stat_mask_frame = cpl_frameset_find(frameset, PIONIER_STATIC_MASK);
        if(stat_mask_frame) {
            stat_mask = cpl_mask_load(cpl_frame_get_filename(stat_mask_frame),
                            0, 1);
        }
        if (stat_mask) {
            /* Invert static mask to set all masked pixels as good as we don't
             * want them to be redetected */
            cpl_mask_not(stat_mask);
            /* Get the real bad pixels in the preselected region */
            cpl_mask_and(m, stat_mask);

            n = cpl_mask_count(m);

            /* Invert the mask again in order to fill the image properly with
             * zero*/
            cpl_mask_not(m);
            cpl_image_reject_from_mask(bpm, m);
            cpl_image_fill_rejected(bpm, 0);
            cpl_propertylist_delete(qclist);
            qclist = cpl_propertylist_new();


            cpl_propertylist_update_int(qclist, "ESO QC BADPIX MASKED", n);
            cpl_propertylist_update_int(qclist, "ESO QC BADPIX DETECT1", pos1);
            cpl_propertylist_update_int(qclist, "ESO QC BADPIX DETECT2", pos2);
            cpl_propertylist_update_int(qclist, "ESO QC BADPIX DETECT3", pos3);
            cpl_propertylist_update_int(qclist, "ESO QC BADPIX DETECT TOT",
					pos1 + pos2 + pos3);
            pioni_detmon_save("PIONI_MASTER_BPM_MASKED",
                              "pioni_detmon_bpm_masked.fits", CPL_TYPE_INT,
                              bpm, parlist, qclist, frameset);
            cpl_mask_delete(stat_mask);
        }

        cpl_vector_delete(mediancounts);
        cpl_propertylist_delete(qclist);
        cpl_mask_delete(m);
        cpl_image_delete(bpm);
    }


    //hdrl_imagelist_delete(out_coef);
    //cpl_image_delete(out_chi2);
    //cpl_image_delete(out_dof);
    cpl_vector_delete(exptime);
    hdrl_imagelist_delete(input_data);
    hdrl_imagelist_delete(input_sig2);
    hdrl_parameter_delete(fpar);

    return (int)cpl_error_get_code();
}

/**@}*/

