/*
 * This file is part of the PIONIER pipeline
 * Copyright (C) 2013,2014 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                    Includes
 -----------------------------------------------------------------------------*/

#include "hdrl.h"

#include "pioni_utils.h"
#include "pioni_dfs.h"
#include "pioni_wrap_yorick.h"
#include <cpl.h>

/*----------------------------------------------------------------------------*/
/**
  @defgroup pioni_oidata_calibrated     Oidata Calibrated recipe
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*-----------------------------------------------------------------------------
                            Functions prototypes
 -----------------------------------------------------------------------------*/


/*-----------------------------------------------------------------------------
                            Static global variables
 -----------------------------------------------------------------------------*/

#define RECIPE_NAME "pioni_oidata_calibrated"

static char pioni_oidata_calibrated_description[] =" TBD \n";



/* Standard CPL recipe definition */
cpl_recipe_define(pioni_oidata_calibrated, PIONIER_BINARY_VERSION, "HDRL Group", 
        PACKAGE_BUGREPORT, "2015", "Oidata Calibrated Recipe",
        pioni_oidata_calibrated_description);                          

/* Function needed by cpl_recipe_define to fill the input parameters */
CPL_DIAG_PRAGMA_PUSH_IGN(-Wunused-parameter);
static cpl_error_code pioni_oidata_calibrated_fill_parameterlist(
        cpl_parameterlist   *   self)
{

    CPL_UNUSED(self);
    return CPL_ERROR_NONE;
}
CPL_DIAG_PRAGMA_POP;

static cpl_error_code
rewrite_oifits_from_yorick(const cpl_parameterlist* parlist, cpl_frameset* frameset,
                   char* outfile_yorick, char* outfile_recipe)
{
    cpl_propertylist* proplist_yorick_primary = cpl_propertylist_load(outfile_yorick, 0);
    cpl_propertylist* proplist_yorick_extension = NULL;
    /* Erase keywords that should be written by the cpl routine */
    pioni_cleanup_primary(proplist_yorick_primary);
    cpl_dfs_save_propertylist(frameset, NULL, parlist, frameset, NULL,
                    "pioni_oidata_calibrated", proplist_yorick_primary, NULL,
                    PACKAGE "/" PACKAGE_VERSION, outfile_recipe);
    cpl_propertylist_delete(proplist_yorick_primary);
    /* Rewrite all extension */
    cpl_size next = cpl_fits_count_extensions(outfile_yorick);
    for (int i = 1; i < next + 1; i++) {
        proplist_yorick_extension = cpl_propertylist_load(
                        outfile_yorick, i);
        cpl_table* table = cpl_table_load(outfile_yorick, i, 1);
        savetype_cpl_to_oifit(table);
        cpl_table_save(table, NULL, proplist_yorick_extension,
                        outfile_recipe, CPL_IO_EXTEND);
        cpl_table_delete(table);
        cpl_propertylist_delete(proplist_yorick_extension);
    }
    /* cleaning up the working directory */
    remove(outfile_yorick);
    return cpl_error_get_code();
}

static int pioni_oidata_calibrated(
        cpl_frameset            *   frameset, 
        const cpl_parameterlist *   parlist)
{
    char                    *   outfile_oidatatf_yorick = NULL;
    char                    *   outfile_oidatacal_yorick = NULL;
    char                    *   oidata_file = NULL;
    char                    *   tf_file = NULL;
    cpl_errorstate              prestate = cpl_errorstate_get();
    char                    *   yorick_argv[20];
    int                         argv_i = 0;
    cpl_frame               *   cur_frame = NULL;
    const char              *   tag = NULL;
    int                         nframes = 0;
    int                         i = 0;
    int                         ctf = 0; /*counter for tf files */
    int                         coidata = 0; /*counter for oidata files */
    /* Check Entries */
    if (parlist == NULL) {
        return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT,
                "Parameters list not found");
    }

    /* Identify the RAW and CALIB frames in the input frameset */
    if (pioni_dfs_set_groups(frameset)) {
        return cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                "Cannot classify RAW and/or CALIB frames");
    }

    /* Get the input data */

    nframes = cpl_frameset_get_size(frameset);

    /* Loop on frames */
    for (i = 0; i < nframes; i++) {
        cur_frame = cpl_frameset_get_position(frameset, i);
        tag = cpl_frame_get_tag(cur_frame);

        /* Oidata Files */
        if (!strcmp(tag, PIONIER_TARGET_OIDATA_RAW)) {
            if(coidata == 0 ) {
                oidata_file = cpl_sprintf("%s",
                                cpl_frame_get_filename(cur_frame));
            }
            else {
                char * tmp_file = cpl_sprintf("%s", oidata_file);
                cpl_free(oidata_file);
                oidata_file = cpl_sprintf("%s,%s", tmp_file,
                                cpl_frame_get_filename(cur_frame));
                cpl_free(tmp_file);
            }
            coidata++;
        }

        /* Tf Files */
        if (!strcmp(tag, PIONIER_OIDATA_TF)) {
            if(ctf == 0 ) {
                tf_file = cpl_sprintf("%s",
                                cpl_frame_get_filename(cur_frame));
            }
            else {
                char * tmp_file = cpl_sprintf("%s", tf_file);
                cpl_free(tf_file);
                tf_file = cpl_sprintf("%s,%s", tmp_file,
                                cpl_frame_get_filename(cur_frame));
                cpl_free(tmp_file);
            }
            ctf++;
        }
    }

    /*Check RAW visibility files*/
    if (oidata_file == NULL || coidata < 1) { /*TODO what is the lower limit? */
        if (tf_file != NULL) {
            cpl_free(tf_file);
        }
        cpl_free(oidata_file);
        return cpl_error_set_message(cpl_func, CPL_ERROR_FILE_NOT_FOUND,
                        "Could not retrieve all input filenames for the OIDATA "
                        "frames");
    }

    /*Check transfer function files*/
    if (tf_file == NULL || ctf < 1) { /*TODO what is the lower limit? */
        if (oidata_file != NULL) {
            cpl_free(oidata_file);
        }
        cpl_free(tf_file);
        return cpl_error_set_message(cpl_func, CPL_ERROR_FILE_NOT_FOUND,
                        "Could not retrieve all input filenames for the "
                        "TF files");
    }


    outfile_oidatatf_yorick = cpl_sprintf("outputFileTF.fits");
    outfile_oidatacal_yorick = cpl_sprintf("outputFileCAL.fits");

    /* generate the yorick argument table  */
    argv_i=0;
    //yorick_argv[argv_i++] = cpl_sprintf("%s","/usr/bin/echo");
    yorick_argv[argv_i++] = cpl_sprintf("%s",YORICK_BIN);
    yorick_argv[argv_i++] = cpl_sprintf("-batch");
    yorick_argv[argv_i++] = cpl_sprintf("%s%s",PNDRS_DIR,
                    "pioni_oidata_calibrated.i");
    yorick_argv[argv_i++] = cpl_sprintf("--inputOiDataFiles=%s", oidata_file);
    yorick_argv[argv_i++] = cpl_sprintf("--inputOiDataTfFiles=%s", tf_file);
    yorick_argv[argv_i++] = cpl_sprintf("--outputOiDataCalibratedFile=%s", outfile_oidatacal_yorick);
    yorick_argv[argv_i++] = cpl_sprintf("--outputOiDataTfeFile=%s", outfile_oidatatf_yorick);
    /*TODO put this parameter as recipe parameter*/
    yorick_argv[argv_i++] = cpl_sprintf("--averageFiles=%s", "1");
    yorick_argv[argv_i++] = NULL;

    /* Execute the yorick wrapper */
    pioni_yorick_exec(yorick_argv);

    if(!cpl_errorstate_is_equal(prestate)){
        cpl_errorstate_dump(prestate, CPL_FALSE, NULL);
        cpl_free(outfile_oidatatf_yorick);
        cpl_free(outfile_oidatacal_yorick);
        cpl_free(oidata_file);
        cpl_free(tf_file);
        for (i=0; i<argv_i-1; i++) cpl_free(yorick_argv[i]);
        return (int)cpl_error_get_code();
    }

    /* Cleanup argv */
    for (i=0; i<argv_i-1; i++) cpl_free(yorick_argv[i]);


    /* Rewrite a dfs conform primary header */


    rewrite_oifits_from_yorick(parlist, frameset,
                    outfile_oidatatf_yorick, "outfile_recipe_tf.fits");
    rewrite_oifits_from_yorick(parlist, frameset,
                    outfile_oidatacal_yorick, "outfile_recipe_calibrated.fits");

    cpl_free(oidata_file);
    cpl_free(tf_file);
    cpl_free(outfile_oidatacal_yorick);
    cpl_free(outfile_oidatatf_yorick);

    return (int)cpl_error_get_code();
}


/**@}*/

