/*
 * This file is part of the PIONIER pipeline
 * Copyright (C) 2013,2014 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                    Includes
 -----------------------------------------------------------------------------*/

#include "hdrl.h"

#include "pioni_utils.h"
#include "pioni_dfs.h"
#include "pioni_wrap_yorick.h"
#include <cpl.h>

/*----------------------------------------------------------------------------*/
/**
  @defgroup pioni_oidata_raw     OIDATA Raw recipe
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*-----------------------------------------------------------------------------
                            Functions prototypes
 -----------------------------------------------------------------------------*/


/*-----------------------------------------------------------------------------
                            Static global variables
 -----------------------------------------------------------------------------*/

#define RECIPE_NAME "pioni_oidata_raw"

static char pioni_oidata_raw_description[] =" TBD \n";



/* Standard CPL recipe definition */
cpl_recipe_define(pioni_oidata_raw, PIONIER_BINARY_VERSION, "HDRL Group", 
        PACKAGE_BUGREPORT, "2015", "OIDATA Raw Recipe",
        pioni_oidata_raw_description);                          

/* Function needed by cpl_recipe_define to fill the input parameters */
CPL_DIAG_PRAGMA_PUSH_IGN(-Wunused-parameter);
static cpl_error_code pioni_oidata_raw_fill_parameterlist(
        cpl_parameterlist   *   self)
{

    CPL_UNUSED(self);
    return CPL_ERROR_NONE;
}
CPL_DIAG_PRAGMA_POP;

static int pioni_oidata_raw(
        cpl_frameset            *   frameset, 
        const cpl_parameterlist *   parlist)
{
    cpl_frame               *   spec_cal_frm = NULL;
    cpl_frame               *   dark_cal_frm = NULL;
    cpl_frame               *   kappa_mat_frm = NULL;
    cpl_frame               *   jsdc_cat_frm = NULL;
    cpl_frame               *   oidata_frm = NULL;
    char                    *   outfile_recipe = NULL;
    char                    *   outfile_yorick = NULL;
    cpl_propertylist        *   proplist_yorick_primary = NULL;
    cpl_propertylist        *   proplist_yorick_extension = NULL;
    const char              *   dark_cal_file = NULL;
    const char              *   spec_cal_file = NULL;
    const char              *   kappa_mat_file = NULL;
    const char              *   jsdc_cat_file = NULL;
    const char              *   oidata_file = NULL;
    cpl_errorstate              prestate = cpl_errorstate_get();
    char                    *   yorick_argv[20];
    int                         argv_i = 0;
    int                         i = 0;

    /* Check Entries */
    if (parlist == NULL) {
        return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT,
                "Parameters list not found");
    }

    /* Identify the RAW and CALIB frames in the input frameset */
    if (pioni_dfs_set_groups(frameset)) {
        return cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                "Cannot classify RAW and/or CALIB frames");
    }

    /* Get the input data */

    /* Get the calibration frames */
    spec_cal_frm = cpl_frameset_find(frameset, PIONIER_SPECTRAL_CALIBRATION) ;
    dark_cal_frm = cpl_frameset_find(frameset, PIONIER_DARK_CALIBRATION) ;
    kappa_mat_frm = cpl_frameset_find(frameset, PIONIER_KAPPA_MATRIX) ;
    jsdc_cat_frm = cpl_frameset_find(frameset, PIONIER_JSDC_CAT) ;

    if (dark_cal_frm == NULL || spec_cal_frm == NULL || kappa_mat_frm == NULL ||
                    jsdc_cat_frm == NULL) {
        return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "Not all "
                        "calibration frames found");
    }
    else {
        spec_cal_file  = cpl_frame_get_filename(spec_cal_frm);
        dark_cal_file  = cpl_frame_get_filename(dark_cal_frm);
        kappa_mat_file = cpl_frame_get_filename(kappa_mat_frm);
        jsdc_cat_file  = cpl_frame_get_filename(jsdc_cat_frm);

        if (dark_cal_file == NULL || spec_cal_file == NULL ||
                        kappa_mat_file == NULL || jsdc_cat_file  == NULL) {
            return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT,
                            "Could not retrieve all input filenames");
        }
        else {
            cpl_msg_info(cpl_func, "Using the following Calibration Files");
            cpl_msg_info(cpl_func, "Spectral Calibration: %s", spec_cal_file);
            cpl_msg_info(cpl_func, "Dark Calibration: %s", dark_cal_file);
            cpl_msg_info(cpl_func, "Kappa Matrix: %s", kappa_mat_file);
            cpl_msg_info(cpl_func, "JSDC Catalog: %s", jsdc_cat_file);
        }
    }

    /* Get the raw frame */
    oidata_frm = cpl_frameset_find(frameset, PIONIER_FRINGE) ;
    if (oidata_frm == NULL) {
        return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "No raw "
                        "frame found");
    }
    else {
        oidata_file  = cpl_frame_get_filename(oidata_frm);
        if (oidata_file == NULL) {
            return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT,
                            "Could not retrieve the raw filename");
        }
        else {
            cpl_msg_info(cpl_func, "Using the following Raw File: %s",
                            oidata_file);
        }
    }

    outfile_yorick = cpl_sprintf("outputFile_oidata.fits");
    /* generate the yorick argument table  */
    argv_i=0;
    //yorick_argv[argv_i++] = cpl_sprintf("%s","/usr/bin/echo");
    yorick_argv[argv_i++] = cpl_sprintf("%s",YORICK_BIN);
    yorick_argv[argv_i++] = cpl_sprintf("-batch");
    yorick_argv[argv_i++] = cpl_sprintf("%s%s",PNDRS_DIR, "pioni_oidata_raw.i");
    yorick_argv[argv_i++] = cpl_sprintf("--inputDarkFile=%s", dark_cal_file);
    yorick_argv[argv_i++] = cpl_sprintf("--inputMatrixFile=%s", kappa_mat_file);
    yorick_argv[argv_i++] = cpl_sprintf("--inputSpecCalFile=%s", spec_cal_file);
    yorick_argv[argv_i++] = cpl_sprintf("--inputCatalogFile=%s", jsdc_cat_file);
    yorick_argv[argv_i++] = cpl_sprintf("--inputScienceFile=%s", oidata_file);
    yorick_argv[argv_i++] = cpl_sprintf("--outputFile=%s", outfile_yorick);
    yorick_argv[argv_i++] = NULL;

    /* Execute the yorick wrapper */
    pioni_yorick_exec(yorick_argv);

    if(!cpl_errorstate_is_equal(prestate)){
        cpl_errorstate_dump(prestate, CPL_FALSE, NULL);
        cpl_free(outfile_yorick);
        for (i=0; i<argv_i-1; i++) cpl_free(yorick_argv[i]);
        return (int)cpl_error_get_code();
    }

    /* Cleanup argv */
    for (i=0; i<argv_i-1; i++) cpl_free(yorick_argv[i]);


    /* Rewrite a dfs conform primary header */

    outfile_recipe = cpl_sprintf("outfile_recipe.fits");
    proplist_yorick_primary = cpl_propertylist_load(outfile_yorick, 0);
    /* Erase keywords that should be written by the cpl routine */
    pioni_cleanup_primary(proplist_yorick_primary);

    cpl_dfs_save_propertylist(frameset, NULL, parlist, frameset,
                              NULL, "pioni_oidata_raw",
                              proplist_yorick_primary, NULL,
                              PACKAGE "/" PACKAGE_VERSION, outfile_recipe);
    cpl_propertylist_delete(proplist_yorick_primary);

    /* Rewrite all extension */

    cpl_size next = cpl_fits_count_extensions(outfile_yorick);
    for (i=1; i < next + 1 ; i++) {
        proplist_yorick_extension = cpl_propertylist_load(outfile_yorick, i);
        cpl_table * table = cpl_table_load(outfile_yorick, i, 1);
        savetype_cpl_to_oifit(table);
        cpl_table_save(table, NULL, proplist_yorick_extension, outfile_recipe,
                       CPL_IO_EXTEND);
        cpl_table_delete(table);
        cpl_propertylist_delete(proplist_yorick_extension);
    }

    /* cleaning up the working directory */
     remove(outfile_yorick);

    cpl_free(outfile_yorick);
    cpl_free(outfile_recipe);

    return (int)cpl_error_get_code();
}


/**@}*/

