/*
 * This file is part of the PIONIER pipeline
 * Copyright (C) 2013,2014 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                    Includes
 -----------------------------------------------------------------------------*/

#include "hdrl.h"

#include "pioni_utils.h"
#include "pioni_dfs.h"
#include "pioni_wrap_yorick.h"
#include <cpl.h>

/*----------------------------------------------------------------------------*/
/**
  @defgroup pioni_oidata_tf     Oidata Tf recipe
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*-----------------------------------------------------------------------------
                            Functions prototypes
 -----------------------------------------------------------------------------*/


/*-----------------------------------------------------------------------------
                            Static global variables
 -----------------------------------------------------------------------------*/

#define RECIPE_NAME "pioni_oidata_tf"

static char pioni_oidata_tf_description[] =" TBD \n";



/* Standard CPL recipe definition */
cpl_recipe_define(pioni_oidata_tf, PIONIER_BINARY_VERSION, "HDRL Group", 
        PACKAGE_BUGREPORT, "2015", "Oidata Tf Recipe",
        pioni_oidata_tf_description);                          

/* Function needed by cpl_recipe_define to fill the input parameters */
CPL_DIAG_PRAGMA_PUSH_IGN(-Wunused-parameter);
static cpl_error_code pioni_oidata_tf_fill_parameterlist(
        cpl_parameterlist   *   self)
{

    CPL_UNUSED(self);
    return CPL_ERROR_NONE;
}
CPL_DIAG_PRAGMA_POP;

static int pioni_oidata_tf(
        cpl_frameset            *   frameset, 
        const cpl_parameterlist *   parlist)
{
    char                    *   outfile_recipe = NULL;
    char                    *   outfile_yorick = NULL;
    cpl_propertylist        *   proplist_yorick_primary = NULL;
    cpl_propertylist        *   proplist_yorick_extension = NULL;
    char                    *   oidata_tf_file = NULL;
    const char              *   catalog_file = NULL;
    cpl_errorstate              prestate = cpl_errorstate_get();
    char                    *   yorick_argv[20];
    int                         argv_i = 0;
    cpl_frame               *   cur_frame = NULL;
    const char              *   tag = NULL;
    int                         nframes = 0;
    int                         i = 0;
    int                         ccatalog = 0; /*counter for catalog files */
    int                         coidata = 0; /*counter for oidata files */
    /* Check Entries */
    if (parlist == NULL) {
        return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT,
                "Parameters list not found");
    }

    /* Identify the RAW and CALIB frames in the input frameset */
    if (pioni_dfs_set_groups(frameset)) {
        return cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                "Cannot classify RAW and/or CALIB frames");
    }

    /* Get the input data */

    nframes = cpl_frameset_get_size(frameset);

    /* Loop on frames */
    for (i = 0; i < nframes; i++) {
        cur_frame = cpl_frameset_get_position(frameset, i);
        tag = cpl_frame_get_tag(cur_frame);

        /* Oidata Tf Files */
        if (!strcmp(tag, PIONIER_CALIB_OIDATA_RAW)) {
            if(coidata == 0 ) {
                oidata_tf_file = cpl_sprintf("%s",
                                cpl_frame_get_filename(cur_frame));
            }
            else {
                char * tmp_file = cpl_sprintf("%s", oidata_tf_file);
                cpl_free(oidata_tf_file);
                oidata_tf_file = cpl_sprintf("%s,%s", tmp_file,
                                cpl_frame_get_filename(cur_frame));
                cpl_free(tmp_file);
            }
            coidata++;
        }

        /* Catalog File */
        if (!strcmp(tag, PIONIER_JSDC_CAT)) {
            catalog_file = cpl_frame_get_filename(cur_frame);
            ccatalog++;
        }
    }

    if (catalog_file == NULL) {
        if (oidata_tf_file != NULL) {
            cpl_free(oidata_tf_file);
        }
        return cpl_error_set_message(cpl_func, CPL_ERROR_FILE_NOT_FOUND,
                        "Could not retrieve the input filename for the "
                        "Catalog frame");
    }

    if (oidata_tf_file == NULL || coidata < 1) { /*TODO what is the lower limit? */
        return cpl_error_set_message(cpl_func, CPL_ERROR_FILE_NOT_FOUND,
                        "Could not retrieve all input filenames for the Oidata "
                        "tf frames");
    }

    outfile_yorick = cpl_sprintf("outputFile.fits");
    /* generate the yorick argument table  */
    argv_i=0;
    //yorick_argv[argv_i++] = cpl_sprintf("%s","/usr/bin/echo");
    yorick_argv[argv_i++] = cpl_sprintf("%s",YORICK_BIN);
    yorick_argv[argv_i++] = cpl_sprintf("-batch");
    yorick_argv[argv_i++] = cpl_sprintf("%s%s",PNDRS_DIR,
                    "pioni_oidata_tf.i");
    yorick_argv[argv_i++] = cpl_sprintf("--inputCatalogFile=%s", catalog_file);
    yorick_argv[argv_i++] = cpl_sprintf("--inputOiDataFiles=%s",oidata_tf_file);
    yorick_argv[argv_i++] = cpl_sprintf("--outputOiDataTfFile=%s", outfile_yorick);
    /*TODO put this parameter as recipe parameter*/
    yorick_argv[argv_i++] = cpl_sprintf("--averageFiles=%s", "1");
    yorick_argv[argv_i++] = NULL;

    /* Execute the yorick wrapper */
    pioni_yorick_exec(yorick_argv);

    if(!cpl_errorstate_is_equal(prestate)){
        cpl_errorstate_dump(prestate, CPL_FALSE, NULL);
        cpl_free(outfile_yorick);
        cpl_free(oidata_tf_file);
        for (i=0; i<argv_i-1; i++) cpl_free(yorick_argv[i]);
        return (int)cpl_error_get_code();
    }

    /* Cleanup argv */
    for (i=0; i<argv_i-1; i++) cpl_free(yorick_argv[i]);


    /* Rewrite a dfs conform primary header */

    outfile_recipe = cpl_sprintf("outfile_recipe.fits");
    proplist_yorick_primary = cpl_propertylist_load(outfile_yorick, 0);
    /* Erase keywords that should be written by the cpl routine */
    pioni_cleanup_primary(proplist_yorick_primary);

    cpl_dfs_save_propertylist(frameset, NULL, parlist, frameset,
                              NULL, "pioni_oidata_tf",
                              proplist_yorick_primary, NULL,
                              PACKAGE "/" PACKAGE_VERSION, outfile_recipe);
    cpl_propertylist_delete(proplist_yorick_primary);


    /* Rewrite all extension */

    cpl_size next = cpl_fits_count_extensions(outfile_yorick);
    for (i=1; i < next + 1 ; i++) {
        proplist_yorick_extension = cpl_propertylist_load(outfile_yorick, i);
        cpl_table * table = cpl_table_load(outfile_yorick, i, 1);
        savetype_cpl_to_oifit(table);
        cpl_table_save(table, NULL, proplist_yorick_extension, outfile_recipe,
                       CPL_IO_EXTEND);
        cpl_table_delete(table);
        cpl_propertylist_delete(proplist_yorick_extension);
    }

    /* cleaning up the working directory */
    remove(outfile_yorick);

    cpl_free(outfile_yorick);
    cpl_free(outfile_recipe);
    cpl_free(oidata_tf_file);
    return (int)cpl_error_get_code();
}


/**@}*/

